/*  Copyright (C) 1987, 1988 by Michael J. Roberts.  All Rights Reserved. */

/*
 * OS-dependent module for Unix and Termcap/Termio library.
 *
 * 29-Jan-93
 * Dave Baggett
 */

/*
 * Fri Jan 29 16:14:39 EST 1993
 * dmb@ai.mit.edu	Termcap version works.
 *
 * Thu Apr 22 01:58:07 EDT 1993
 * dmb@ai.mit.edu	Lots of little changes.  Incorporated NeXT, Linux,
 *			and DJGCC changes.
 *
 * Sat Jul 31 02:04:31 EDT 1993
 * dmb@ai.mit.edu	Many changes to improve screen update efficiency.
 *			Added unix_tc flag, set by compiler mainline,
 *			to disable termcap stuff and just run using stdio.
 *
 * Fri Oct 15 15:50:09 EDT 1993
 * dmb@ai.mit.edu       Changed punt() code to save game state.
 *
 * Mon Oct 25 16:39:57 EDT 1993
 * dmb@ai.mit.edu       Now use our own os_defext and os_remext.
 *
 */
#include <stdio.h>
#if !defined(SUN3)
#include <stddef.h>
#endif
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifdef	DJGCC_386
#include <pc.h>
#include <dos.h>
#include "biosvid.h"
#endif

#include "os.h"
#ifndef	USE_STDIO
#include "osdbg.h"
#endif
#include "voc.h"

/*
 * ^C pressed?
 */
static int	break_set = 0;

/*
 * Run-time voc context for save-on-punt
 */
voccxdef   *saved_context;

/*
 * Running as compiler?  If so, we shouldn't do any term handling.
 */
int	unix_tc = 0;

#ifndef	USE_STDIO

#ifndef DJGCC_386
#include <signal.h>
#ifdef	USE_SGTTY
#include <sgtty.h>
#else
#ifdef	TERMIOS_IS_NOT_IN_SYS
#include <termios.h>
#else
#include <sys/termios.h>
#endif	/* TERMIOS_IS_NOT_IN_SYS */
#endif	/* USE_SGTTY */
#endif	/* DJGCC_386 */

#ifdef	HAVE_TPARM
/*
 * Use the standard library routine if we have it.
 */
#define Tparm tparm
#endif

/*
 * The following flag determines whether or not the output routines
 * will attempt to use hardware features of the controlling terminal.
 * If it is NOT on, ALL screen updates will go through t_redraw.
 * This is useful for debugging problems with a termcap entry or
 * getting TADS running on a new environment quickly, but IS INCREDIBLY
 * SLOW and hence should not be used for production releases.
 */
#define T_OPTIMIZE 

/*
 * The following flag determines whether or not ossdsp will just send
 * its string out to t_puts or whether it will redraw the entire affected
 * line.  The latter is slower (perhaps painfully so over slow modem
 * connections), but fixes a problem with the way TADS currently updates
 * the status line.
 *
 * Sat Jul 31 02:06:42 EDT 1993  dmb@ai.mit.edu
 * This problem seems to be fixed now, but I'll leave the flag anyway.
 *
 */
#define FAST_OSSDSP


#define INIT 0
#define RESTORE 1

/*
 * We store an image of the screen in a private array.
 * The maximum dimensions of this array are given here.
 */
#define MAXCOLS (OS_MAXWIDTH-1)
#define MAXLINES 256

/*
 * Running as debugger?  If so, split the screen into the debugging
 * window and the execution window.
 */
int	unix_tdb = 0;
int	breakpoint_color;	/* for debugger */
int	watchpoint_color;	/* for debugger */
int     filename_color;         /* for debugger */
int     usebios = 0;  /* for dbgu.c */

/*
 * Our private notion of what the screen looks like.
 * We keep one array of characters and another array of "colors."
 * Under termcap/termio handling we don't actually do color (at the
 * moment, at least), but can do character attributes like standout,
 * reverse video, blinking, etc.
 *
 * We keep track of the current cursor location and text attributes
 * so that we don't have to send redundant codes out to the terminal.
 */
static int	cw = 0;		/* current window (for debugger) */
static int	COLS, LINES;
static char	COLOR = -1;
static int	cursorX = 0, cursorY = 0;
static int	inputX = 0, inputY = 0;
static char	screen[MAXLINES][MAXCOLS];
static char	colors[MAXLINES][MAXCOLS];

#ifdef	DJGCC_386
static int	plaincolor, revcolor, boldcolor;
#endif

/*
 * Flags indicating whether our controlling terminal has various
 * capabilities.
 */
static int	standout_ok, rev_ok, bold_ok, blink_ok, underscore_ok;
static int	hideshow_ok;

static int	k_erase, k_kill, k_word_erase, k_reprint;

/*
 * "Colors"
 * Note that cNORMAL *must* be zero for other parts of TADS to work.
 */
#define cNORMAL 	(1 << 0)
#define cSTANDOUT	(1 << 1)
#define cREVERSE	(1 << 2)
#define	cBOLD		(1 << 3)
#define	cBLINK		(1 << 4)
#define	cUNDERSCORE	(1 << 5)


/*
 * Terminal capabilities from termcap.
 * Certainly more than you ever wanted to know about.
 */

/*
 * Booleans
 */
static int
	Txs,	/* Standout not erased by overwriting (Hewlett-Packard) */
	Tms,	/* Safe to move in standout modes */
	Tos;	/* Terminal overstrikes on hard-copy terminal */

/*
 * Numbers
 */
static int
	Tsg;	/* Number for characters left by standout */
/*
 * Strings
 */
static char
	*Tcs,	/* Change scroll region to lines #1 through #2 (VT100) */
	*TcS,	/* Change scroll region: #1 = total lines, #2 =  lines
		   above region, #3 = lines below region, #4 = total lines */
	*Tcl,	/* Clear screen and home cursor */
	*Tce,	/* Clear to end of line */
	*Tcd,	/* Clear to end of display */
	*Tcm,	/* Cursor motion to row #1 col #2 */
	*Tdo,	/* Down one line */
	*Tho,	/* Home cursor */
	*Tvi,	/* Make cursor invisible */
	*Tle,   /* Move cursor left one SPACE */
	*Tve,	/* Make cursor appear normal (undo cvvis/civis) */
	*Tnd,	/* Non-destructive space (cursor right) */
	*Tup,	/* Upline (cursor up) */
	*Tvs,	/* Make cursor very visible */
	*Tdl,	/* Delete line */
	*Tmb,	/* Turn on blinking */
	*Tmd,	/* Turn on bold (extra bright) mode */
	*Tme,	/* Turn off attributes (blinking, reverse, bold, etc.) */
	*Tti,	/* String to begin programs that use cursor addresing */
	*Tmr,	/* Turn on reverse video mode */
	*Tso,	/* Begin standout mode */
	*Tus,	/* Start underscore mode */
	*Tte,	/* String to end programs that use cursor addressing */
	*Tse,	/* End standout mode */
	*Tue,	/* End underscore mode */
	*Tal,	/* Add new blank line (insert line) */
	*Tpc,	/* Pad character (rather than null) */
	*TDL,	/* Delete #1 lines */
	*TDO,	/* Move cursor down #1 lines. */
	*TSF,	/* Scroll forward #1 lines. */
	*TAL,	/* Add #1 new blank lines */
	*TLE,	/* Move cursor left #1 spaces */
	*TRI,	/* Move cursor right #1 spaces. */
	*TSR,	/* Scroll backward #1 lines. */
	*TUP,	/* Move cursor up #1 lines. */
	*Tsf,	/* Scroll forward one line. */
	*Tsr,	/* Scroll backward one line. */
	*Twi;	/* Current window is lines #1-#2 cols #3-#4 */

#ifndef	DJGCC_386
/*
 * We must import the following three variables from the termcap
 * library and set the appropriately so that output padding will
 * work properly.
 */
extern char	PC;
extern short	ospeed;
#endif	/* DJGCC_386 */

/*
 * External variables important to TADS but not imported by the
 * standard os.h
 */
extern int	pagelength, linewidth;

/*
 * Declare return types so the compiler doesn't whine.
 */
static void	punt();
static void	susp();
static void	cont();
static void	get_screen_size();
static void	resize();
static void	t_init();
static void	t_tty();
static void	winch();
static void	set_win_size();
static void	t_color_init();
static void	t_term();
static void	t_hide();
static void	t_show();
static void	t_putc();
static void	t_outs();
static void	t_puts();
static void	t_refresh();
static void	t_update();
static void	t_loc();
static void	t_color();
static void	t_redraw();
static void	t_scroll();
static void	t_set_scroll_region();
void		os_term();
void		os_printf();
void		os_flush();
void		os_waitc();
char		*os_gets();
int		os_break();
static void	break_handler();
int		os_paramfile();
void		ossdbgloc(y, x);

/*
 * Set up our controlling terminal and find out what it's capable
 * of doing.  Punt if we have no ability to position the cursor,
 * scroll, etc.
 */
static void
t_init()
{
#ifdef	DJGCC_386
	
	bios_video_init();
	rev_ok = 1;
	bold_ok = 1;
	standout_ok = 0;
	blink_ok = 0;
	underscore_ok = 0;
	hideshow_ok = 0;
	bios_video_get_screen_dimensions(&COLS, &LINES);
	LINES++; COLS++; set_win_size(); LINES--; COLS--;
	t_color_init();
	
#else	/* DJGCC_386*/
	
#define Tgetstr(key) (tgetstr(key, &tbufptr))
	extern char	*tgetstr();

	static char	tbuf[4096];
	register char	*term;
	register char	*tptr;
	char		*tbufptr;
	int		scroll_ok, curs_ok;

	if (unix_tc)
		return;

	/*
	 * Set tty up the way we need it.
	 */
	t_tty(INIT);

	/*
	 * Set all string capabilities to NULL for safety's sake.
	 */
	Tcs = TcS = Tcl = Tce = Tcd = Tcm = Tdo = Tho = Tvi = Tle = Tve =
	Tnd = Tup = Tvs = Tdl = Tme = Tmb = Tmd = Tti = Tmr = Tso = Tus =
	Tte = Tse = Tue = Tal = Tpc = TDL = TDO = TSF = TAL = TLE = TRI =
	TSR = TUP = Twi = NULL;

	/*
	 * Find the name of the user's terminal.
	 */
	term = getenv("TERM");
	if(!term) {
		printf("Can't find TERM, the name of your terminal.\n");
		exit(-1);
	}

	/*
	 * Allocate a buffer for the termcap entry.  Make it big to be safe.
	 */
	tptr = (char *) malloc(4096);
	if (!tptr) {
		printf("Out of memory allocating termcap entry buffer.\n");
		exit(-1);
	}

	/*
	 * Get the termcap entry.
	 */
	tbufptr = tbuf;
	if(tgetent(tptr, term) < 1) {
		printf("Unknown terminal type: %s", term);
		exit(-1);
	}

	/*
	 * Check for fatal conditions:
	 * 
	 *	- Tos (terminal overstrikes)
	 *	- Does not have any of:
	 *		- Add line and delete line (AL/DL, al/dl)
	 *		- Set scrolling region (cs, cS) *and* scroll up and
	 *		  down (sf and sr).
	 *		- Set window (wi) *and* scroll up and scroll down
	 *		  (sf and sr).
	 *	- Does not have cursor addressing *or* home *and* down and right
	 */
	if (tgetflag("os")) {
		printf("Can't run on a hard-copy terminal (termcap os).\n");
		exit(-1);
	}
	TAL = Tgetstr("AL");
	TDL = Tgetstr("DL");
	Tal = Tgetstr("al");
	Tdl = Tgetstr("dl");
	Tcs = Tgetstr("cs");
	TcS = Tgetstr("cS");
	Twi = Tgetstr("wi");
	Tsf = Tgetstr("sf");
	Tsr = Tgetstr("sr");
	TSF = Tgetstr("SF");
	TSR = Tgetstr("SR");

	if (!Tsf)
		Tsf = "\n";

	scroll_ok = 0;
	if ((Tal || TAL) && (Tdl || TDL))
		scroll_ok = 1;
	else if ((Tcs || TcS || Twi) && (Tsf || TSF) && (Tsr || TSR))
		scroll_ok = 1;
	if (!scroll_ok) {
		printf("Can't run without window scrolling capability.\n");
		exit(-1);
	}
	Tcm = Tgetstr("cm");
	Tho = Tgetstr("ho");
	Tle = Tgetstr("le");
	if (!Tle)
		Tle = Tgetstr("bs");	/* obsolete alternative to "le" */
	TLE = Tgetstr("LE");
	Tnd = Tgetstr("nd");
	TRI = Tgetstr("RI");
	Tup = Tgetstr("up");
	TUP = Tgetstr("UP");
	Tdo = Tgetstr("do");
	if (!Tdo)
		Tdo = Tgetstr("nl");	/* obsolete alternative to "do" */
	TDO = Tgetstr("DO");
	Tti = Tgetstr("ti");
	Tte = Tgetstr("te");
	curs_ok = 0;
	if (Tcm)
		curs_ok = 1;
	else if (Tho) {
		if ((Tnd || TRI) && (Tdo || TDO))
			curs_ok = 1;
	}
	if (!curs_ok) {
		printf("Can't run without cursor positioning capability.\n");
		exit(-1);
	}

	/*
	 * Get region clearing capabilities, if any.
	 */
	Tcl = Tgetstr("cl");
	Tce = Tgetstr("ce");
	Tcd = Tgetstr("cd");

	/*
	 * See if we have standout mode.
	 * Don't use it if term has standout erase bug (xs) or it's not
	 * safe to move in standout mode (ms), or standout generates spaces
	 * (sg > 0).
	 */
	standout_ok = 0;
	if (!tgetflag("xs") && !tgetflag("ms") && tgetnum("sg") <= 0) {
		Tso = Tgetstr("so");
		Tse = Tgetstr("se");
		if (Tso && Tse)
			standout_ok = 1;
	}

	/*
	 * See if we can turn off special modes
	 */
	Tme = Tgetstr("me");

	/*
	 * See if we have reverse video mode.
	 */
	rev_ok = 0;
	if (Tme) {
		Tmr = Tgetstr("mr");
		if (Tmr)
			rev_ok = 1;;
	}

	/*
	 * See if we have bold mode.
	 */
	bold_ok = 0;
	if (Tme) {
		Tmd = Tgetstr("md");
		if (Tmd)
			bold_ok = 1;
	}

	/*
	 * See if we have blink mode.
	 */
	blink_ok = 0;
	if (Tme) {
		Tmb = Tgetstr("mb");
		if (Tmb)
			blink_ok = 1;
	}

	/*
	 * See if we have underscore mode.
	 */
	underscore_ok = 0;
	Tus = Tgetstr("us");
	Tue = Tgetstr("ue");
	if (Tus && Tue)
		underscore_ok = 1;

	/*
	 * See if we can hide/show the cursor.
	 */
	hideshow_ok = 0;
	Tvi = Tgetstr("vi");	
	Tve = Tgetstr("ve");
	Tvs = Tgetstr("vs");	
	if (Tvi && Tve)
		hideshow_ok = 1;

	/*
	 * Get padding character (if there is one for this term).
	 */
	Tpc = Tgetstr("pc");
	if (Tpc)
		PC = *Tpc;
	else
		PC = 0;

	free(tptr);

	/*
	 * Do the right thing when window's resized.
	 */
	signal(SIGWINCH, winch);

	/*
	 * Handle ^C.
	 */
	signal(SIGINT, break_handler);
	/* signal(SIGINT, SIG_IGN); */

	/* 
	 * Handle ^Z.
	 */
	signal(SIGTSTP, susp);
	signal(SIGCONT, cont);

	/*
	 * Restore terminal sanity when signals nuke us.
	 */
	signal(SIGHUP, punt);
	signal(SIGQUIT, punt);
	signal(SIGILL, punt);
	signal(SIGTRAP, punt);
	signal(SIGABRT, punt);
#if !defined(LINUX_386)
	signal(SIGEMT, punt);
#endif
	signal(SIGFPE, punt);
#if !defined(LINUX_386)
	signal(SIGBUS, punt);
#endif
	signal(SIGSEGV, punt);
#if !defined(LINUX_386)
	signal(SIGSYS, punt);
#endif

	/*
	 * Determine screen size.  Ensure sanity of text formatter.
	 */
	resize();

	/*
	 * Init "colors"
	 */
	t_color_init();

	/*
	 * Init screen
	 */
	t_puts(Tti);
	t_puts(Tvs);
	
#endif	/* DJGCC_386 */	
}

/*
 * Clean up when we get a request to suspend from the terminal.
 */
static void
susp()
{
#ifndef	DJGCC_386	
	t_term();
	signal(SIGTSTP, SIG_DFL);
	kill(getpid(), SIGTSTP);
#endif	
}

/*
 * Restore term state upon resuming from suspension.
 */
static void
cont()
{
#ifndef	DJGCC_386
	t_init();
	t_update(-1, -1);
	COLOR = -1;
	t_refresh();
#endif	
}

/*
 * Die gracefully when we get a fatal signal.
 * We try to save the game before we exit.
 */
static void
punt()
{
	char s[15];

	if (!unix_tc) {
	        sprintf(s, "fatal%d.sav", getpid());
         	fiosav(saved_context, s);

	        printf("Caught a fatal signal!  Attempting to save game in %s...\n", s);
	}
	else {
	        printf("Caught a fatal signal!\n");
        }

	os_term(-1);
}

/*
 * Get terminal size.
 * Negative and zero values mean the system has no clue.
 */
static void
get_screen_size(w, h)
	int	*w, *h;
{
#ifndef	DJGCC_386
	/*
 	 * Define the 4.3 names in terms of the Sun names
   	 * if the latter exist and the former do not.
	 */
#ifdef TIOCGSIZE
#ifndef TIOCGWINSZ
#define TIOCGWINSZ TIOCGSIZE
#define winsize ttysize
#define ws_row ts_lines
#define ws_col ts_cols
#endif
#endif

	/*
	 * Use the 4.3 names if possible.
	 */
#ifdef TIOCGWINSZ
	struct winsize size;
  	*w = 0;
	*h = 0;
  	if (ioctl(0, TIOCGWINSZ, &size) < 0)
    		return;

  	*w = size.ws_col;
  	*h = size.ws_row;

#else /* not TIOCGWNSIZ */

	/*
	 * System has no clue
	 */
	*w = 0;
	*h = 0;

#endif /* not TIOCGWINSZ */

	/*
 	 * If running as debugger, halve the height
	 * since we want two windows.
	 */
	if (unix_tdb)
		*h /= 2;
	
#endif	/* DJGCC_386 */	
}

static void
resize()
{
#ifndef	DJGCC_386	
	get_screen_size(&COLS, &LINES);

	if (COLS <= 0 || LINES <= 0) {
		COLS = tgetnum("co");
		LINES = tgetnum("li");
	}

	set_win_size();
#endif	
}

static void
winch()
{
#ifndef	DJGCC_386	
	resize();

	/*
	 * Clear screen
	 */
	ossclr(0, 0, LINES - 1, COLS - 1, cNORMAL);
	t_refresh();
#endif	
}

/*
 * When window size changes, we have to notify the rest of the system
 * so text will be formatted properly.
 */
static void
set_win_size()
{
	/*
	 * Ensure output formatter happiness.
	 */
	if (LINES < 5)
		LINES = 5;
	if (COLS < 20)
		COLS = 20;

	/*
	 * Ensure our own personal happiness.
	 */
	if (LINES >= MAXLINES)
		LINES = MAXLINES - 1;
	if (COLS >= MAXCOLS)
		COLS = MAXCOLS - 1;

	/*
	 * Init screen size globals used by the system.
	 * We always use one less than the full width and height to avoid
	 * nasty auto-scrolling problems.  DO NOT CHANGE THIS.  It may
	 * work on *your* terminal but not on some of the others.
	 * Better safe than sorry.
	 */
	max_line = LINES - 2;
 	max_column = COLS - 2;
	pagelength = max_line - text_line - 1;
	linewidth = max_column + 1;
	score_column = max_column - 10;
}

/*
 * Tell the rest of the system what "colors" to use for various things.
 */
static void
t_color_init()
{
	if (rev_ok)
		sdesc_color = cREVERSE;
	else if (standout_ok)
		sdesc_color = cSTANDOUT;
	else if (bold_ok)
		sdesc_color = cBOLD;
	else if (underscore_ok)
		sdesc_color = cUNDERSCORE;
	else
		sdesc_color = cNORMAL;

	ldesc_color = sdesc_color;
	debug_color = sdesc_color;

	text_color  = cNORMAL;
	if (bold_ok)
		text_bold_color = cBOLD;
	else if (standout_ok)
		text_bold_color = cSTANDOUT;
	else if (rev_ok)
		text_bold_color = cREVERSE;
	else if (underscore_ok)
		text_bold_color = cUNDERSCORE;
	else
		text_bold_color = cNORMAL;

	text_normal_color = text_color;
	breakpoint_color = sdesc_color;
	watchpoint_color = sdesc_color;

	if (rev_ok)
		filename_color = cREVERSE;
        else if (underscore_ok)
	        filename_color = cUNDERSCORE;
	else if (bold_ok)
		filename_color = cBOLD;
	else
		filename_color = cNORMAL;
}

static void
t_term()
{
#ifndef	DJGCC_386
	
	if (unix_tc)
		return;

	/*
	 * Restore scroll window if term has that capability.
	 */
	if (Tcs || TcS || Twi)
		if (unix_tdb)
	  		t_set_scroll_region(0, LINES*2 - 1);
		else
	  		t_set_scroll_region(0, LINES - 1);

	/*
	 * Disable all attributes (if possible), including standout.
	 */
	t_puts(Tme);
	t_puts(Tse);
	t_puts(Tue);

	/*
	 * Restore tty
	 */
	t_tty(RESTORE);

	/*
	 * End visual mode
	 */
	t_puts(Tte);

	/*
	 * Make cursor visible
	 */
	t_puts(Tve);
	
#endif	/* DJGCC_386 */	
}

/*
 * Hide cursor
 */
static void
t_hide()
{
#ifdef	DJGCC_386
	void bios_video_cursor_hide();
#else	/* DJGCC_386 */	
	
	if (hideshow_ok)
		t_puts(Tvi);
	
#endif	/* DJGCC_386 */	
}

/*
 * Show cursor
 */
static void
t_show()
{
#ifdef	DJGCC_386
	void bios_video_cursor_hide();
#else	/* DJGCC_386 */	
	
	if (hideshow_ok)
		t_puts(Tve);
	
#endif	/* DJGCC_386 */	
}

/*
 * Write a character to the screen (without padding)
 */
static void
t_putc(c)
	char c;
{
#ifndef	DJGCC_386	
	fputc(c, stdout);
#endif	
}

/*
 * Write a string to the screen (without padding)
 */
static void
t_outs(s)
	char	*s;
{
#ifdef	DJGCC_386	
	
	bios_video_write_string(s);
	
#else	/* DJGCC_386 */	
	if (s) {
		printf("%s", s);
		fflush(stdout);
	}
#endif	
}

/*
 * Write an escape sequence to the string to the screen with padding
 */
static void
t_puts(s)
	char	*s;
{
#ifndef	DJGCC_386	
	
#if 0
	if (s)
		printf("%s", s);
	return;
#endif

	if (s) {
		Tputs(s, 1, (int (*)())t_putc);
		fflush(stdout);
	}
#endif	
}

/*
 * Init or restore terminal.
 */
static void
t_tty(action)
	int	action;
{
#ifndef	DJGCC_386
	
#ifdef	USE_SGTTY
	struct sgttyb		t;
	struct ltchars		t1;
	static struct sgttyb	orig;
#else	/* USE_SGTTY */
	struct termios		t;
	static struct termios	orig;
#endif	/* USE_SGTTY */
	
	if (action == RESTORE) {

#ifdef	USE_IOCTL_INSTEAD_OF_TERMIOS
#ifdef	USE_SGTTY
		ioctl(0, TIOCSETP, &orig);
#else
		ioctl(0, TCSETSW, &orig);
#endif /* USE_SGTTY */
#else
		tcsetattr(0, TCSANOW, &orig);
#endif
	}
	else {

#ifdef	USE_IOCTL_INSTEAD_OF_TERMIOS
#ifdef	USE_SGTTY
		ioctl(0, TIOCGETP, &orig);
		ioctl(0, TIOCGETP, &t);
#else
		ioctl(0, TCGETS, &orig);
		ioctl(0, TCGETS, &t);
#endif /* USE_SGTTY */
#else
		tcgetattr(0, &orig);
		tcgetattr(0, &t);
#endif

#ifdef	USE_SGTTY
		t.sg_flags &= ~(XTABS | ECHO | CRMOD);
		t.sg_flags |= CBREAK;
#else
		t.c_lflag &= ~(ICANON | ECHO | ICRNL);
		t.c_lflag |= INLCR;
		t.c_cc[VMIN] = 1;
		t.c_cc[VTIME] = 0;
#if !defined(SGI_IRIX)
		t.c_oflag &= (~XTABS);
#else
		t.c_oflag &= (~TAB3);
#endif
#endif	/* USE_SGTTY */
		
#ifdef	USE_IOCTL_INSTEAD_OF_TERMIOS
#ifdef	USE_SGTTY
		ioctl(0, TIOCSETP, &t);
#else
		ioctl(0, TCSETSW, &t);
#endif	/* USE_SGTTY */
#else
		tcsetattr(0, TCSANOW, &t);
#endif
	}

	/*
	 * Determine baud rate for Tputs.
	 * Look for it in termios struct; if not found, assume 2400 baud.
	 */
#ifdef	USE_SGTTY
	ospeed = t.sg_ospeed;
#else	
	ospeed = t.c_cflag & CBAUD;
#endif
	if (ospeed == 0)
		ospeed = 11;

	/*
	 * Get special keys
	 */
#ifdef	USE_SGTTY
	k_erase = t.sg_erase;
	k_kill = t.sg_kill;
	ioctl(0, TIOCGLTC, &t1);
	k_word_erase = t1.t_werasc;
	k_reprint = t1.t_rprntc;
#else	/* USE_SGTTY */
	k_erase = t.c_cc[VERASE];
	k_kill = t.c_cc[VKILL];
	k_word_erase = t.c_cc[VWERASE];

#if defined(SGI_IRIX) || defined(ULTRIX_MIPS)
	k_reprint = t.c_cc[VRPRNT];
#else
	k_reprint = t.c_cc[VREPRINT];
#endif
#endif	/* USE_SGTTY */
	
#endif	/* DJGCC_386 */	
}

int
os_init( argc, argv, prompt, buf, bufsiz )
	int	*argc;
	char	*argv[];
	char	*prompt;
	char	*buf;
	int	bufsiz;
{
#ifdef	DJGCC_386	
	FILE	*fp;
	char	nbuf[128];
	
	/*
	 * Monochrome monitor?
	 */
	if (bios_video_monochrome()) {
	    revcolor = 0x70;
	    plaincolor = 7;
	    boldcolor = 0x70;
	}
	else if ( os_locate("trcolor.dat", 11, argv[0], nbuf, sizeof(nbuf))
		 && (fp = fopen(nbuf, "r")) ) {
	    int         i;
		static int *colorptr[] = { &plaincolor, &boldcolor, &revcolor };

		for (i = 0 ; i < sizeof(colorptr)/sizeof(colorptr[0]) ; ++i)
		{
			fgets(nbuf, sizeof(nbuf), fp);
			*colorptr[i] = atoi(nbuf);
		}
		fclose(fp);
	}
	else {
		plaincolor = 7;
		revcolor = 0x70;
		boldcolor = 15;
	}
	text_color = text_normal_color;
#endif	/* DJGCC_386 */
	
	t_init();

	/*
	 * Clear screen
	 */
	if (unix_tdb) {
		cw = 0;
		ossclr(0, 0, LINES - 1, COLS - 1, cNORMAL);
		cw = 1;
		ossclr(0, 0, LINES - 1, COLS - 1, cNORMAL);
	}
	else
		ossclr(0, 0, LINES - 1, COLS - 1, cNORMAL);

#ifdef USE_SCROLLBACK

    osssbini(32767*8);      /* allocate scrollback buffer */

#endif /* USE_SCROLLBACK */
    
	if (!unix_tc) {
		status_mode = 1;
#ifdef	DJGCC_386		
		os_printf( "TADS (GO32 version)" );
#else
		os_printf( "TADS" );
#endif		
		status_mode = 0;
		os_score(0, 0);
	}
	else {
		/*
		 * Set up fake screen size to facilitate stdio output
		 */
		max_line = 1;
		max_column = 135;
		pagelength = max_line - text_line - 1;
		linewidth = max_column + 1;
	}

    return 0;
}

void
os_term(rc)
	int	rc;
{
	if (!unix_tc) {
		os_printf("[strike a key to exit]");
		os_waitc();
	        os_printf("\n");
	}

	t_term();

#ifdef USE_SCROLLBACK
	osssbdel();
#endif /* USE_SCROLLBACK */

	qasclose();

	exit(0); /* always return zero since mainline doesn't set it right */
}

/*
 *   Check for control-break.  Returns status of break flag, and clears
 *   the flag. 
 */
int
os_break()
{
	int	ret;
	
	ret = break_set;
	break_set = 0;
	return ret;
}

/*
 * Handle ^C.
 */
static void
break_handler()
{
	break_set = 1;
}

int
os_paramfile(buf)
{
	return 0;
}

/*
 *   os_exeseek  - opens the given .EXE file and seeks to the end of the
 *   executable part of it, on the presumption that a datafile is to be
 *   found there.
 */
FILE *os_exeseek( exefile )
char *exefile;
{
    return((FILE *)0);
}

/*
 *   os_exfld  - load in an external function from an open file, given
 *   the size of the function (in bytes).  Returns a pointer to the newly
 *   allocated memory block containing the function in memory.
 */
int (*os_exfld( fp, len ))( /*_ void _*/ )
FILE     *fp;
unsigned  len;
{
	return  (int (*)(/*_ void _*/)) 0;
}

/*
 *   Load an external function from a file.  This routine assumes that
 *   the file has the same name as the resource.  
 */
int (*os_exfil( name ))( /*_ void _*/ )
char *name;
{
	return (int (*)(/*_ void _*/)) 0;
}

/*
 *   call an external function, passing it an argument (a string pointer),
 *   and passing back the string pointer returned by the external function
 */
char *os_excall(extfn, arg)
char  *(*extfn)(/*_ char* _*/);
void  *arg;
{
	return NULL;
}

#ifdef	DJGCC_386

int os_getch()
{
	return bios_getchar();
}

void os_waitc()
{
	int c = os_getch();

	if ((c == 0 || c == 0340)) {
	        os_getch();
	}
}

int os_getc()
{
	static char altkeys[] = {
		30, 48, 46, 32, 18, 33, 34, 35, 23, 36, 37, 38, 50,
		49, 24, 25, 16, 19, 31, 20, 22, 47, 17, 45, 21, 44
	};

	int             c;
	static int      cmdval;
    
	if (cmdval) {
		int tmpcmd = cmdval;                    /* save command for a moment */
		cmdval = 0;           /* clear the flag - next call gets another key */
		return tmpcmd;                /* return the command that we saved */
	}

tryAgain:
	c = os_getch();

	if ((c == 0 || c == 0340))  {                       /* Extended keycode? */
		int   i;
		char *p;

		c = os_getch();

		/* check for alt keys */
		cmdval = 0;
		for (i = 0, p = altkeys ; i < 26 ; ++p, ++i)
		{
			if (c == *p)
			{
				cmdval = CMD_ALT + i;
				break;
			}
		}

		if (!cmdval) switch(c) {
			case 0110:      /* Up Arrow */
				cmdval = CMD_UP;
				break;
			case 0120:      /* Down Arrow */
				cmdval = CMD_DOWN;
				break;
			case 0115:      /* Right Arrow */
				cmdval = CMD_RIGHT;
				break;
			case 0113:      /* Left Arrow */
				cmdval = CMD_LEFT;
				break;
			case 0117:      /* End */
				cmdval = CMD_END;
				break;
			case 0107:      /* Home */
				cmdval = CMD_HOME;
				break;
			case 0165:      /* Ctrl-End */
				cmdval = CMD_DEOL;
				break;
			case 0123:      /* Del */
				cmdval = CMD_DEL;
				break;
			case 073:       /* F1 - change this? */
				cmdval = CMD_SCR;
				break;
			case 0111:      /* PgUp */
				cmdval = CMD_PGUP;
				break;
			case 0121:      /* PgDn */
				cmdval = CMD_PGDN;
				break;
			case 132:       /* control-PgUp */
				cmdval = CMD_TOP;
				break;
			case 118:	/* control-PgDn */
				cmdval = CMD_BOT;
				break;
			case 119:       /* control home */
				cmdval = CMD_CHOME;
				break;
			case 115:       /* control left arrow */
				cmdval = CMD_WORD_LEFT;
				break;
			case 116:       /* control right arrow */
				cmdval = CMD_WORD_RIGHT;
				break;

			case 60: cmdval = CMD_F2; break;
			case 61: cmdval = CMD_F3; break;
			case 62: cmdval = CMD_F4; break;
			case 63: cmdval = CMD_F5; break;
			case 64: cmdval = CMD_F6; break;
			case 65: cmdval = CMD_F7; break;
			case 66: cmdval = CMD_F8; break;
			case 67: cmdval = CMD_F9; break;
			case 68: cmdval = CMD_F10; break;

			case 85: cmdval = CMD_SF2; break;                     /* shifted F2 */

			default:        /* Unrecognized function key */
				cmdval = 0;
				break;
		}
	}
	else
	{
		switch( c )
		{
			case 1:     /* ^A */
				cmdval = CMD_HOME;
				break;
			case 2:     /* ^B */
				cmdval = CMD_LEFT;
				break;
			case 4:     /* ^D */
				cmdval = CMD_DEL;
				break;
			case 5:     /* ^E */
				cmdval = CMD_END;
				break;
			case 6:     /* ^F */
				cmdval = CMD_RIGHT;
				break;
			case '\t':
				cmdval = CMD_TAB;
				break;
			case 11:    /* ^K */
				cmdval = CMD_DEOL;
				break;
			case 14:    /* ^N */
				cmdval = CMD_DOWN;
				break;
			case 16:    /* ^P */
				cmdval = CMD_UP;
				break;
			case 21:    /* ^U */
			case 27:    /* Escape */
				cmdval = CMD_KILL;
				break;
		}
	}

	if (cmdval) 
		return 0;
	else if (c & 0xff)
		return c & 0xff;
	else
		goto tryAgain;
}

#else	/* DJGCC_386 */

int
os_getc()
{
    	char		c;
	static char	cbuf;
	static int	buffered = 0;

	/*
	 * If we've got a special key pending, return its code.
	 */
	if (buffered) {
		buffered = 0;
		return cbuf;
	}

	/*
	 * Get the next character from standard input.
	 * (Non-blocking)
	 */
	read(0, &c, 1);

	/*
	 * If we read ^L, refresh the screen.
	 */
	while (c == 12 || c == k_reprint) {
		t_refresh();
		read(0, &c, 1);
	}

	/*
	 * Map NL to CR.
	 */
	if (c == 10)
		c = 13;

#define ctrl(c) (c - 'A' + 1)

	/*
	 * Handle special keys.
	 * os_gets expects certain keys to be certain codes.
	 */
	if (c == k_erase || c == 0x7F) /* erase key or DEL -> backspace */
		c = ctrl('H');
	if (c == k_kill)
		c = ctrl('U');

	/*
	 * Key equivalences for special keys.
	 * Some terminals can send function keys, etc. but we
	 * don't support that right now.
	 */
	switch (c) {
		case 0:
			c = 1;
			break;

		case 10:
			c = 13;
			break;

   	        case ctrl('W'):
			c = 0;
			cbuf = CMD_WORDKILL;
			break;
			
	        case ctrl('D'):
			c = 0;
			cbuf = CMD_DEL;
			break;

		case ctrl('U'):
			c = 0;
			cbuf = CMD_KILL;
			break;

		case ctrl('K'):
			c = 0;
			cbuf = CMD_DEOL;
			break;

		case ctrl('P'):
			c = 0;
			cbuf = CMD_UP;
			break;

		case ctrl('N'):
			c = 0;
			cbuf = CMD_DOWN;
			break;

		case ctrl('F'):
			c = 0;
			cbuf = CMD_RIGHT;
			break;

		case ctrl('B'):
			c = 0;
			cbuf = CMD_LEFT;
			break;

		case ctrl('E'):
			c = 0;
			cbuf = CMD_END;
			break;

		case ctrl('A'):
			c = 0;
			cbuf = CMD_HOME;
			break;

		case 27:
			c = 0;
			cbuf = CMD_SCR;
			break;

		case '<':
			c = 0;
			cbuf = CMD_PGUP;
			break;
			
		case '>':
			c = 0;
			cbuf = CMD_PGDN;
			break;
			
	}

	/*
	 * Special key?  Then note that we should return it on next call.
	 */
	if (c == 0)
		buffered = 1;

	return c;

#undef ctrl
}

void
os_waitc()
{
	char	c;

	/*
	 * Get the next character from standard input.
         * Wait until we get one.
	 */
	while (read(0, &c, 1) < 1);
}
#endif	/* DJGCC_386 */

/*
 * Update the real screen so it matches our idea of what's on the screen.
 */
static void
t_refresh()
{
	if (unix_tdb) {
		t_redraw(0, LINES * 2 - 1);
	}
	else
		t_redraw(0, LINES - 1);
}

/*
 * Update internal understanding of where the physical cursor is.
 */
static void
t_update(y, x)
	int	y, x;
{
	cursorX	= x;
	cursorY = y;
}

/*
 * Move the cursor.  Note that this has nothing to do with the virtual cursor
 * that TADS tells us to move around -- that's updated by ossloc.
 *
 * The force parameter tells us to not use the current value of cursorX
 * and cursorY and to reposition the cursor absolutely.
 *
 */
static void
t_loc(y, x, force)
	int	y, x;
	int	force;
{
#ifdef	DJGCC_386
	
	bios_video_set_cursor_position(x, y);
	
#else	/* DJGCC_386 */	
	
	register int	i;

#if 0
#define X printf("[line %d]\n", __LINE__);
#else
#define X
#endif

	if (!force)
		if (cursorX == x && cursorY == y)
			return;

#if 0
	printf("[*%d, %d*]", x, y);
#endif

	/*
	 * User direct cursor addressing if we have it; otherwise
	 * home the cursor and move right and down from there.
	 *
	 * If we can get to the destiation by moving just one
	 * space, we'll use the shorter single movement commands.
	 */
	if (!force && x == cursorX && y == cursorY - 1 && (Tup || TUP)) {
X
		/*
		 * Up one space
		 */
		if (Tup)
			t_puts(Tup);
		else
			t_puts(Tparm(TUP, 1));
	}	
	else if (!force && x == cursorX && y == cursorY + 1 && (Tdo || TDO)) {
X
		/*
		 * Down one space
		 */
		if (Tdo)
			t_puts(Tdo);
		else
			t_puts(Tparm(TDO, 1));
	}
	else if (!force && y == cursorY && x == cursorX + 1 && (Tnd || TRI)) {
X
		/*
		 * Right one space
		 */
		if (Tnd)
			t_puts(Tnd);
		else
			t_puts(Tparm(TRI, 1));
	}
	else if (!force && y == cursorY && x == cursorX - 1 && (Tle || TLE)) {
X
		/*
		 * Left one space
		 */
		if (Tle)
			t_puts(Tle);
		else
			t_puts(Tparm(TLE, 1));
	}
	else if (Tcm) {
X
		t_puts(Tparm(Tcm, y, x));
	}
	else {
X
		t_puts(Tho);

		if (TRI)
			t_puts(Tparm(TRI, x - 1));
		else
			for (i = 0; i < x; i++)
				t_puts(Tnd);

		if (TDO)
			t_puts(Tparm(TDO, y - 1));
		else 
			for (i = 0; i < y; i++)
				t_puts(Tdo);
	}

#undef X
	t_update(y, x);

#endif	/* DJGCC_386 */
}

/*
 * Change subsequent text to new color.
 */
static void
t_color(c)
	char	c;
{
#ifdef	DJGCC_386

	if (c & cREVERSE)
		bios_video_set_color(revcolor);
	else if (c & cBOLD)
		bios_video_set_color(boldcolor);
	else
		bios_video_set_color(plaincolor);
	
#else	/* DJGCC_386 */
	
	if (c == COLOR)
		return;

	/*
	 * Disable all attributes (if possible), including standout.
	 */
	t_puts(Tme);
	if (standout_ok && !(c & cSTANDOUT))
		t_puts(Tse);	
	if (underscore_ok && !(c & cUNDERSCORE))
		t_puts(Tue);	

	if (c & cSTANDOUT)
		if (standout_ok)
			t_puts(Tso);
	if (c & cREVERSE)
		if (rev_ok)
			t_puts(Tmr);
	if (c & cBOLD)
		if (bold_ok)
			t_puts(Tmd);
	if (c & cBLINK)
		if (blink_ok)
			t_puts(Tmb);
	if (c & cUNDERSCORE)
		if (underscore_ok)
			t_puts(Tus);

	/*
	 * Save color.
	 */
	COLOR = c;
	
#endif	/* DJGCC_386 */	
}

/*
 * Redraw a portion of the screen.
 */
static void
t_redraw(top, bottom)
	int	top, bottom;
{
	static char	ds[MAXCOLS + 1] = {0};

	register int	row, col, pos;
	int		color, newcolor;

	/*
	 * Hide cursor
	 */
	t_hide();

	/*
 	 * Position to top line, column zero.
	 */
	t_loc(top, 0, 1);

	/*
 	 * Redraw each line between top line and bottom line.
	 * We have to build a string from our text and color
	 * information so we can write the output a line at a time
	 * rather than a character at a time.
 	 */
 	color = colors[top][0];
 	t_color(color);
	for (row = top; row <= bottom; row++) {
		/*
		 * Move cursor to beginning of this line
		 */
		t_loc(row, 0, 1);

		/*
		 * Combine color and text information into a single
		 * string for this line.
		 */
		for (col = 0, pos = 0; col < COLS; col++) {
			/*
			 * Do we have to change text color? 
			 * If so, print the text we've buffered for
			 * this line and then print the change string.
			 */
			if ((newcolor = colors[row][col]) != color) {
				ds[pos] = 0;
				t_outs(ds);
				t_color(newcolor);
				t_loc(row, col, 1);
				pos = 0;
				color = newcolor;
			}
			ds[pos++] = screen[row][col];
		}	

		ds[pos] = 0;
		t_outs(ds);		/* blast the string */
	}

	/*
	 * Reposition cursor to its former location
	 */
	t_loc(inputY, inputX, 1);

	/*
	 * Show cursor
	 */
	t_show();
}

/*
 * Scroll region
 * lines < 0 -> scroll up (delete line)
 * lines > 0 -> scroll down (insert line)
 */
static void
t_scroll(top, bot, lines)
	int	top, bot;
	int	lines;
{
#ifdef	DJGCC_386
	
	bios_video_scroll_region(top, bot, 0, COLS - 1, lines);
	
#else	/* DJGCC_386 */

	char	*single = lines > 0 ? Tsr : Tsf;
	char	*multi  = lines > 0 ? TSR : TSF;
	int	labs = lines < 0 ? -lines : lines;
	int	i;

	/*
	 * Make sure new lines have the right background color
	 */
	t_color(text_normal_color);

	/*
	 * If we have scroll region capability, use it.
	 * Otherwise fake it with insert/delete line.
	 */
	if ((single || multi) && (Tcs || TcS || Twi)) {
		t_set_scroll_region(top, bot);

		if (lines < 0)
			t_loc(bot, 0, 1);
		else
			t_loc(top, 0, 1);

		if (labs > 1 && multi || !single)
			t_puts(Tparm(multi, labs));
		else
			for (i = 0; i < labs; i++)
				t_puts(single);

		t_set_scroll_region(0, LINES - 1);
	}
	else {
		/*
		 * Make sure we never eat lines below the window
		 */
		if (labs> bot - top + 1)
			labs = bot - top + 1;

		if (lines < 0) {
			/*
			 * Delete lines
			 */
			t_loc(top, 0, 1);
			if (TDL && labs != 1) {
				t_puts(Tparm(TDL, labs));
			}
			else if (Tdl) {
				for (i = 0; i < labs; i++)
					t_puts(Tdl);
			}
			else {
				/* shouldn't happen */
			}

			/*
			 * Insert lines to keep windows below this
			 * one intact.
			 */
			if (bot < LINES - 1) {
				t_loc(bot + 1 - labs, 0, 1);
				if (TAL && labs != 1) {
					t_puts(Tparm(TAL, labs));
				}
				else if (Tal) {
					for (i = 0; i < labs; i++)
						t_puts(Tal);
				}
				else {
					/* shouldn't happen */
				}
			}	
		}
		else {
			/*
			 * Insert lines
			 */

			/*
			 * Delete lines to keep windows below this
			 * one intact.
			 */
			if (bot < LINES - 1) {
				t_loc(bot + 1 - labs, 0, 1);
				if (TDL && labs != 1) {
					t_puts(Tparm(TDL, labs));
				}
				else if (Tdl) {
					for (i = 0; i < labs; i++)
						t_puts(Tdl);
				}
				else {
					/* shouldn't happen */
				}
			}


			/*
			 * Insert lines at top of window.
			 */
			t_loc(top, 0, 1);
			if (TAL && labs != 1) {
				t_puts(Tparm(TAL, labs));
			}
			else if (Tal) {
				for (i = 0; i < labs; i++)
					t_puts(Tal);
			}
			else {
				/* shouldn't happen */
			}
		}
	}

	/*
	 * After we scroll, we don't know where the cursor is.
	 */
	t_update(-1, -1);
	
#endif	/* DJGCC_386 */
}

static void
t_set_scroll_region(top, bot)
	int	top, bot;
{
#ifndef	DJGCC_386
	static int	topsave = -1, botsave = -1;

	if (top == topsave && bot == botsave)
		return;

	if (Tcs) {
      		t_puts(Tparm(Tcs, top, bot));
	}
  	else if (TcS) {
		t_puts(Tparm(TcS, LINES, top, LINES - (bot + 1), LINES));
	}
  	else {
		t_puts(Tparm(Twi, top, 0, bot, COLS - 1));
	}

	topsave = top;
	botsave = bot;
#endif	
}

/*
 * Scroll region down a line.
 * This is equivalent to inserting a line at the top of the region.
 */
ossscu(top, left, bottom, right, blank_color)
	int	top, left, bottom, right;
	int	blank_color;
{
	register int	r1, r2, col;

        if (unix_tdb) {
		top += LINES * cw;
		bottom += LINES * cw;
	}
		
#ifdef DEBUG_OUTPUT
	printf("[ossscu(%d, %d, %d, %d, %d)]", 
		top, left, bottom, right, blank_color);
#endif

	if (unix_tc)
		return;
	
	/*
	 * Update our internal version
	 */
	for (r1 = bottom - 1, r2 = bottom; r1 >= top; r1--, r2--)
		for (col = left; col <= right; col++) {
			screen[r2][col] = screen[r1][col];
			colors[r2][col] = colors[r1][col];
		}	
	for (col = left; col <= right; col++) {
		screen[r2][col] = ' ';
		colors[r2][col] = blank_color;
	}

	/*
	 * If we can duplicate the effect of this scroll on the screen
	 * with scrolling commands, do so; otherwise, refresh by redrawing
	 * every affected line.
	 */
#ifdef	T_OPTIMIZE
	if (left == 0 && right >= max_column) {
		t_scroll(top, bottom, 1);
	}
	else {
#else
		t_redraw(top, bottom);
#endif

#ifdef	T_OPTIMIZE
	}
#endif
}

/*
 * Scroll region up a line
 * This is equivalent to deleting a line at the top of the region and pulling
 * everything below it up.
 */
ossscr(top, left, bottom, right, blank_color)
	int	top, left, bottom, right;
	int	blank_color;
{
	register int	r1, r2, col;

	if (unix_tdb) {
		top += LINES * cw;
		bottom += LINES * cw;
	}
	
#ifdef DEBUG_OUTPUT
	printf("[ossscr(%d, %d, %d, %d, %d)]", 
		top, left, bottom, right, blank_color);
#endif

	if (unix_tc) {
		putchar('\n');
		return;
	}

	/*
	 * Update our internal version
	 */
	for (r1 = top, r2 = top + 1; r2 <= bottom; r1++, r2++)
		for (col = left; col <= right; col++) {
			screen[r1][col] = screen[r2][col];
			colors[r1][col] = colors[r2][col];
		}	
	for (col = left; col <= right; col++) {
		screen[r1][col] = ' ';
		colors[r1][col] = blank_color;
	}

	/*
	 * If we can duplicate the effect of this scroll on the screen
	 * with a scrolling command, do so; otherwise, refresh by redrawing
	 * every affected line.
	 */
#ifdef	T_OPTIMIZE
	if (left == 0 && right >= max_column) {
		t_scroll(top, bottom, -1);
	}
	else {
#else
		t_redraw(top, bottom);
#endif

#ifdef	T_OPTIMIZE
	}
#endif
}

/*
 * Clear region (fill with spaces)
 */
ossclr(top, left, bottom, right, blank_color)
	int	top, left, bottom, right;
	int	blank_color;
{
	register int	row, col;

	if (unix_tdb) {
		top += LINES * cw;
		bottom += LINES * cw;
	}

#ifdef DEBUG_OUTPUT
	printf("[ossclr(%d, %d, %d, %d, %d)]", 
		top, left, bottom, right, blank_color);
#endif

	if (unix_tc)
		return;

	/*
	 * Update our internal version
	 */
	for (row = top; row <= bottom; row++)
		for (col = left; col <= right; col++) {
			screen[row][col] = ' ';
			colors[row][col] = blank_color;
		}

	/*
	 * If we can duplicate the effect of this clear on the screen
	 * with a clear command, do so; otherwise, refresh by redrawing every
	 * affected line.
	 */
#ifdef	DJGCC_386
	bios_video_set_bkgnd(plaincolor);
	bios_video_clear_region(top, bottom, left, right);
#else	/* DJGCC_386 */	
	
#ifdef	T_OPTIMIZE
	if (Tce && left == 0 && right >= max_column) {
		t_color(blank_color);

		if (Tcl && top == 0 && bottom >= max_line) {
			t_loc(0, 0, 1);
			t_puts(Tcl);
		}
		else if (Tcd && bottom >= max_line) {
			t_loc(top, 0, 1);
			t_puts(Tcd);
		}
		else for (row = top; row <= bottom; row++) {
			t_loc(row, 0, 1);
			t_puts(Tce);
		}

		/*
		 * Don't know where the cursor is after clear.
		 */
		t_update(-1, -1);
	}
	else {
#endif	
		t_redraw(top, bottom);
#ifdef	T_OPTIMIZE
	}
#endif	
	
#endif	/* DJGCC_386 */	
}

/*
 * Locate (input) cursor at given row and column.
 * Nore that this is never used to determine where things are drawn.
 * It's only used for aesthetics; i.e., showing the user where input
 * will be taken from next.
 */ 
ossloc(row, col)
	int	row, col;
{
	if (unix_tc)
		return;

	if (unix_tdb) 
		row += LINES * cw;

	t_loc(row, col, 0);
	
	/*
	 * Update internal cursor position so t_redraw will
	 * be correct.
	 */
	inputX = col;
	inputY = row;	
}

/*
 * Display msg with color at coordinates (y, x).
 * The color must be in the range 0 <= color <= 16, and specifies both
 * foreground and background colors.
 */
ossdsp(y, x, color, msg)
	int		y, x;	
	int		color;
	register char	*msg;
{
	register int	col;
	register char	*s, *m = msg, *c;

#ifdef DEBUG_OUTPUT
	printf("[ossdsp(%d, %d, %d, \"%s\")]", y, x, color, msg);
#endif

	if (unix_tc) {
		printf("%s", msg);
		fflush(stdout);
	}

	if (y >= LINES || x >= COLS)
		return;

	if (unix_tdb)
		y += LINES * cw;
	
	s = &screen[y][x];
	c = &colors[y][x];
	
	/*
	 * Update our own version of the screen.
	 */
	for (col = x; *m && col < COLS; col++) {
		*s++ = *m++;
		*c++ = color;
	}

#ifdef	FAST_OSSDSP
	/*
	 * XXX
	 *
	 * We redraw the whole line if it's the status line.
	 * This is pretty bogus.
	 */
#if 0
	if (y != sdesc_line) {
#else
	if (1) {
#endif
		/*
		 * Update the screen
		 */
		t_loc(y, x, 0);
		t_color(color);
		t_outs(msg);
		t_update(cursorY, cursorX + strlen(msg));
	}
	else
		t_redraw(y, y);
#else
	t_redraw(y, y);
#endif
}


/*
 * Stuff for the debugger
 */
ossgmx(maxline, maxcol)
     int *maxline, *maxcol;
{
    *maxline = LINES - 1;
    *maxcol = COLS - 1;
}

/* clear a window */
void osdbgclr(win)
oswdef *win;
{
	ossclr(win->oswy1, win->oswx1, win->oswy2, win->oswx2, win->oswcolor);
}

int osdbgini(rows, cols)
    int rows;
    int cols;
{
    return(0);
}

int ossvpg(pg)
	char pg;
{
	static int	s_inputX = 0, s_inputY = 0;
	int             ret;

        if (cw == pg)
	        return cw;

	inputX = s_inputX;
	inputY = s_inputY;

	ret = cw;
	cw = pg;
        t_update(-1, -1);
        COLOR = -1;
/*        t_refresh(); */

	return ret;
}

int ossmon()
{
        return 0;
}

/* scroll a window up a line */
static void osdbgsc(win)
oswdef *win;
{
	ossscr(win->oswy1, win->oswx1, win->oswy2, win->oswx2, win->oswcolor);
}

# ifdef USE_STDARG
#if defined(UNIX) && defined(USE_GCC_STDARG)
void osdbgpt(oswdef *win, const char *fmt, ...)
#else   /* USE_GCC_STDARG */
void osdbgpt(win, fmt, ... )
oswdef *win;
char *fmt;
#endif  /* USE_GCC_STDARG */
# else /* USE_STDARG */
void osdbgpt( win, fmt, a1, a2, a3, a4, a5, a6, a7, a8 )
oswdef *win;
char *fmt;
long  a1, a2, a3, a4, a5, a6, a7, a8;
# endif /* USE_STDARG */
{
    char buf[256];
    char *p;

# ifdef USE_STDARG
    va_list argptr;

    va_start( argptr, fmt );
    vsprintf( buf, fmt, argptr );
    va_end( argptr );
# else /* USE_STDARG */
    sprintf( buf, fmt, a1, a2, a3, a4, a5, a6, a7, a8 );
# endif /* USE_STDARG */

    for (p=buf ; *p ; )
    {
	char *p1;

	if ((win->oswflg & OSWFMORE) && win->oswx == win->oswx1 &&
	    win->oswmore+1 >= win->oswy2 - win->oswy1)
	{
	    char c;
	    
	    ossdsp(win->oswy, win->oswx, win->oswcolor, "[More]");
	    ossdbgloc(win->oswy, win->oswx+6);
	    do
	    {
		switch(c = os_getc())
		{
		case '\n':
		case '\r':
		    win->oswmore--;
		    break;

		case ' ':
		    win->oswmore = 0;
		    break;
		}
	    } while (c != ' ' && c != '\n' && c != '\r');
		
	    ossdsp(win->oswy, win->oswx, win->oswcolor, "      ");
	}

	for (p1 = p ; *p1 && *p1 != '\n' && *p1 != '\r' && *p1 != '\t'; p1++);
	if (*p1 == '\n' || *p1 == '\r' || *p1 == '\t')
	{
	    int c = *p1;
	    
	    *p1 = '\0';

	    if (win->oswx + strlen(p) > win->oswx2 &&
		(win->oswflg & OSWFCLIP))
		p[win->oswx2 - win->oswx + 1] = '\0';
	    ossdsp(win->oswy, win->oswx, win->oswcolor, p);

	    if (c == '\n')
	    {
		++(win->oswy);
		win->oswx = win->oswx1;
		if (win->oswy > win->oswy2)
		{
		    win->oswy = win->oswy2;
		    osdbgsc(win);
		}
		win->oswmore++;
	    }
	    else if (c == '\t')
	    {
		win->oswx += strlen(p);
	        do
		{
		    ossdsp(win->oswy, win->oswx, win->oswcolor, " ");
		    ++(win->oswx);
		    if (win->oswx > win->oswx2 && (win->oswflg & OSWFCLIP))
			break;
		} while ((win->oswx - 2) & 7);
	    }
	    p = p1 + 1;
	    if (win->oswx > win->oswx2) return;
	}
	else
	{
	    if (win->oswx + strlen(p) > win->oswx2
		&& (win->oswflg & OSWFCLIP))
		p[win->oswx2 - win->oswx + 1] = '\0';
	    ossdsp(win->oswy, win->oswx, win->oswcolor, p);
	    win->oswx += strlen(p);
	    p = p1;
	}
    }
}

/* open a window - set up location */
void osdbgwop(win, x1, y1, x2, y2, color)
oswdef *win;
int     x1, y1;
int     x2, y2;
int     color;
{
	win->oswx1 = win->oswx = x1;
	win->oswx2 = x2;
	win->oswy1 = win->oswy = y1;
	win->oswy2 = y2;
	win->oswcolor = color;
	win->oswflg = 0;
}

void ossdbgloc(y, x)
char y, x;
{
	ossloc(y, x);
}

/* get some text */
int osdbggts(win, buf, cmdfn, cmdctx)
oswdef *win;
char   *buf;
int   (*cmdfn)(/*_ dvoid *ctx, char cmd _*/);
dvoid  *cmdctx;
{
    char *p = buf;
    char *eol = buf;
    char *eob = buf + 127;
    int   x = win->oswx;
    int   y = win->oswy;
    int   origx = x;
    int   cmd = 0;

    win->oswmore = 0;
    for (buf[0] = '\0' ; ; )
    {
        char c;

        ossdbgloc(y, x);
        switch(c = os_getc())
        {
            case 8:
                if (p > buf)
                {
                    char *q;
                    char  tmpbuf[2];
                    int   thisx, thisy;

                    for ( q=(--p) ; q<eol ; q++ ) *q = *( q+1 );
                    eol--;
                    if ( --x < 0 )
                    {
                        x = win->oswx2;
                        y--;
                    }
                    *eol = ' ';
                    thisx = x;
                    thisy = y;
                    for ( q=p, tmpbuf[1]='\0' ; q<=eol ; q++ )
                    {
                        tmpbuf[0] = *q;
                        ossdsp( thisy, thisx, win->oswcolor, tmpbuf );
                        if ( ++thisx > win->oswx2 )
                        {
                            thisx = 0;
                            thisy++;
                        }
                    }
                    *eol = '\0';
                }
                break;
            case 13:
                /*
                 *   Scroll the screen to account for the carriage return,
                 *   position the cursor at the end of the new line, and
                 *   null-terminate the line.
                 */
                *eol = '\0';
                while( p != eol )
                {
                    p++;
                    if ( ++x > win->oswx2 )
                    {
                        y++;
                        x = 0;
                    }
                }

                if ( y == win->oswy2 ) osdbgsc(win);
		else ++y;
		x = 0;
                ossdbgloc( y, x );

                /*
                 *   Finally, copy the buffer to the screen save buffer
                 *   (if applicable), and return the contents of the buffer.
                 *   Note that we add an extra carriage return if we were
                 *   already on the max_line, since we scrolled the screen
                 *   in this case; otherwise, ossaddsbe will add all the
                 *   blank lines that are necessary.
                 */
		win->oswx = x;
		win->oswy = y;
                return(0);

            case 0:
                switch(c = os_getc())
                {
                    case CMD_LEFT:
                        if ( p>buf )
                        {
                            p--;
                            x--;
                            if ( x < 0 )
                            {
                                x = win->oswx2;
                                y--;
                            }
                        }
                        break;
                    case CMD_RIGHT:
                        if ( p<eol )
                        {
                            p++;
                            x++;
                            if ( x > win->oswx2 )
                            {
                                x = 0;
                                y++;
                            }
                        }
                        break;
                    case CMD_DEL:
                        if ( p<eol )
                        {
                            char *q;
                            char  tmpbuf[2];
                            int   thisx=x, thisy=y;

                            for ( q=p ; q<eol ; q++ ) *q = *(q+1);
                            eol--;
                            *eol = ' ';
                            for ( q=p, tmpbuf[1]='\0' ; q<=eol ; q++ )
                            {
                                tmpbuf[0] = *q;
                                ossdsp( thisy, thisx, win->oswcolor, tmpbuf );
                                if ( ++thisx > win->oswx2 )
                                {
                                    thisx = 0;
                                    thisy++;
                                }
                            }
                            *eol = '\0';
                        }
                        break;
                    case CMD_KILL:
                    case CMD_HOME:
		    do_kill:
                        while( p>buf )
                        {
                            p--;
                            if ( --x < 0 )
                            {
                                x = win->oswx2;
                                y--;
                            }
                        }
                        if ( c == CMD_HOME ) break;
                        /*
                         *   We're at the start of the line now; fall
                         *   through for KILL, UP, and DOWN to the code
                         *   which deletes to the end of the line.
                         */
                    case CMD_DEOL:
                        if ( p<eol )
                        {
                            char *q;
                            int   thisx=x, thisy=y;

                            for ( q=p ; q<eol ; q++ )
                            {
                                ossdsp( thisy, thisx, win->oswcolor, " " );
                                if ( ++thisx > win->oswx2 )
                                {
                                    thisx = 0;
                                    thisy++;
                                }
                            }
                            eol = p;
                            *p = '\0';
                        }
			if (cmd) return(cmd);
                        break;
                    case CMD_END:
                        while ( p<eol )
                        {
                            p++;
                            if ( ++x > win->oswx2 )
                            {
                                x = 0;
                                y++;
                            }
                        }
                        break;
			
		    default:
			if (cmd = (*cmdfn)(cmdctx, c))
			{
			    c = CMD_KILL;
			    goto do_kill;
			}
			break;
                }
                break;
            default:
                if ( c >= ' ' && c < 127 && eol<eob )
                {
                    if ( p != eol )
                    {
                        char *q;
                        int   thisy=y, thisx=x;
                        char  tmpbuf[2];

                        for ( q=(++eol) ; q>p ; q-- ) *q=*(q-1);
                        *p = c;
                        for ( q=p++, tmpbuf[1] = '\0' ; q<eol ; q++ )
                        {
                            tmpbuf[0] = *q;
                            ossdsp( thisy, thisx, win->oswcolor, tmpbuf );
                            thisx++;
                            if ( thisx > win->oswx2 )
                            {
                                thisx = 0;
                                if ( thisy == win->oswy2 )
                                {
                                    y--;
				    osdbgsc(win);
                                }
                                else thisy++;
                            }
                        }
                        if ( ++x > win->oswx2 )
                        {
                            y++;
                            x = 0;
                        }
                    }
                    else
                    {
                        *p++ = c;
                        *p = '\0';
                        eol++;
                        ossdsp( y, x, win->oswcolor, p-1 );
                        if ( ++x > win->oswx2 )
                        {
                            x = 0;
                            if ( y == win->oswy2 )
				osdbgsc(win);
                            else y++;
                        }
                    }
                }
                break;
        }
    }
}


/*
 * End of stuff for debugger
 */

#else	/* USE_STDIO */
	
int
os_init( argc, argv, prompt, buf, bufsiz )
	int	*argc;
	char	*argv[];
	char	*prompt;
	char	*buf;
	int	bufsiz;
{
    return 0;
}

void
os_term(rc)
	int	rc;
{
	exit(rc);
}

int
os_break()
{
	int	ret;
	
	ret = break_set;
	break_set = 0;
	return ret;
}

int
os_paramfile(buf)
{
	return 0;
}

/*
 *   os_exeseek  - opens the given .EXE file and seeks to the end of the
 *   executable part of it, on the presumption that a datafile is to be
 *   found there.
 */
FILE *os_exeseek( exefile )
char *exefile;
{
    return((FILE *)0);
}

/*
 *   os_exfld  - load in an external function from an open file, given
 *   the size of the function (in bytes).  Returns a pointer to the newly
 *   allocated memory block containing the function in memory.
 */
int (*os_exfld( fp, len ))( /*_ void _*/ )
FILE     *fp;
unsigned  len;
{
	return  (int (*)(/*_ void _*/)) 0;
}

/*
 *   Load an external function from a file.  This routine assumes that
 *   the file has the same name as the resource.  
 */
int (*os_exfil( name ))( /*_ void _*/ )
char *name;
{
	return (int (*)(/*_ void _*/)) 0;
}

/*
 *   call an external function, passing it an argument (a string pointer),
 *   and passing back the string pointer returned by the external function
 */
char *os_excall(extfn, arg)
char  *(*extfn)(/*_ char* _*/);
void  *arg;
{
	return NULL;
}

void
os_waitc()
{
	getkey();
}

int
os_getc()
{
	return getkey();
}

#endif	/* USE_STDIO */

/*
 *   Get the temporary file path.  This should fill in the buffer with a
 *   path prefix (suitable for strcat'ing a filename onto) for a good
 *   directory for a temporary file, such as the swap file.  
 */
void
os_get_tmp_path(char *s)
{
  int i;

  strcpy(s, tmpnam(NULL));
  for (i = strlen(s) - 1; i >= 0; i--)
    if (s[i] == '/')
      break;

  s[i + 1] = 0;
}

/* os_defext(fn, ext) should append the default extension ext to the filename
 *  in fn.  It is assumed that the buffer at fn is big enough to hold the added
 *  characters in the extension.  The result should be null-terminated.  When
 *  an extension is already present in the filename at fn, no action should be
 *  taken.  On systems without an analogue of extensions, this routine should
 *  do nothing.
 *
 * For Unix, we extend this to also prepend the default saved game or game file path
 * name.
 * 
 *    - The TADSSAVE environment variable holds the name of the save file directory
 *    - The TADSGAME envirnoment variable holds the name of the game file directory
 *
 * We only prepend  if there are no slashes in the filename already.
 * We don't prepand paths when running as the compiler, because we don't want the output
 * files to go in weird places.
 */
void os_defext( fn, ext )
char *fn;
char *ext;
{
    char *p, *n, tmp[1024];
    char *defpath;

    /*
     * Prepend default path
     */
   if (!memicmp(ext, "sav", strlen(ext)))
	defpath = getenv("TADSSAVE");
   else if (!memicmp(ext, "gam", strlen(ext)))
	defpath = getenv("TADSGAME");
   else
        defpath = NULL;

   if (!unix_tc && defpath) {
   	/*
	 * Look for slashes.  If there are any, don't mess with name.
	 */
	n = fn;
	while (*n) {
		if (*n == '/')
			break;
		n++;
	}
	if (!*n) {
		strcpy(tmp, defpath);
		if (defpath[strlen(defpath)] != '/')
			strcat(tmp, "/");
		strcat(tmp, fn);
		strcpy(fn, tmp);
	}  
   }

    p = fn+strlen(fn);
    while ( p>fn )
      {
        p--;
        if ( *p=='.' ) return;      /* already has an extension */
        if ( *p=='/' || *p=='\\' || *p==':'
           ) break;    /* found a path */
      }
    strcat( fn, "." );              /* add a dot */
    strcat( fn, ext );              /* add the extension */
  }

/* os_remext(fn) removes the extension from fn, if present.  The buffer at
 *  fn should be modified in place.  If no extension is present, no action
 *  should be taken.  For systems without an analogue of extensions, this
 *  routine should do nothing.
 */
void os_remext( fn )
char *fn;
{
    char *p = fn+strlen(fn);
    while ( p>fn )
      {
        p--;
        if ( *p=='.' )
	  {
            *p = '\0';
            return;
	  }
        if ( *p=='/' || *p=='\\' || *p==':'
            ) return;
      }
}

/*
 * Add an extension, even if the filename currently has one 
 */
void os_addext(fn, ext)
char *fn;
char *ext;
{
    strcat(fn, ".");
    strcat(fn, ext);
}


/*
 * Provide memicmp since it's not a standard libc routine.
 */
memicmp(s1, s2, len)
	char	*s1, *s2;
	int	len;
{
	char	*x1, *x2;	
	int	result;
	int	i;

	x1 = malloc(len);
	x2 = malloc(len);

	if (!x1 || !x2) {
		printf("Out of memory!\n");
		exit(-1);
	}

	for (i = 0; i < len; i++) {
		if (isupper(s1[i]))
			x1[i] = tolower(s1[i]);
		else
			x1[i] = s1[i];

		if (isupper(s2[i]))
			x2[i] = tolower(s2[i]);
		else
			x2[i] = s2[i];
	}

	result = memcmp(x1, x2, len);
	free(x1);
	free(x2);
	return result;
}

/*
 * memcpy - copy bytes (handles overlap, so we can equivalence memmove() to it.
 */
void *
our_memcpy(void *dst, const void *src, size_t size)
{
	register char *d;
	register const char *s;
	register size_t n;

	if (size == 0)
		return(dst);

	s = src;
	d = dst;
	if (s <= d && s + (size-1) >= d) {
		/* Overlap, must copy right-to-left. */
		s += size-1;
		d += size-1;
		for (n = size; n > 0; n--)
			*d-- = *s--;
	} else
		for (n = size; n > 0; n--)
			*d++ = *s++;

	return(dst);
}

#if !defined(DJGCC_386)
/*
 * dbgu.c requires strlwr
 */
strlwr(s)
     char *s;
{
	while (*s) {
		if (isupper(*s))
			*s = tolower(*s);

		s++;
	}
}
#endif

#ifndef	DJGCC_386

/*
 * Open file using fopen, stripping of b's from flag string.
 */
#ifdef fopen
#undef fopen
#endif

FILE *
our_fopen(filename, flags)
	char	*filename;
	char	*flags;
{
	static char	f[80];
	int		i = 0;

	while (*flags) {
		if (*flags != 'b')
			f[i++] = *flags;

		flags++;
	}
	f[i] = 0;

	return fopen(filename, f);
}
#endif	/* DJGCC_386 */
