#ifdef RCSID
static char RCSid[] =
"$Header: c:/tads/tads2/RCS/TADSRSC.C 1.2 94/11/06 13:05:34 mroberts Exp $";
#endif

/* Copyright (c) 1992 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  tadsrsc.c - TADS v2 resource manager
Function
  resource manager for TADS v2
Notes
  Somewhat improved over TADS v1 resource manager thanks to new file
  format.  This version can list, add, replace, and remove resources.
Modified
  04/30/92 MJRoberts     - creation
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "os.h"
#include "std.h"
#include "fio.h"

/* printf */
#define rscptf printf

static void usage()
{
    rscptf("usage: tadsrsc resfile [operations]\n");
    rscptf("   or: tadsrsc resfile @opfile\n");
    rscptf("\nIf @opfile is used, operations are read from file.\n");
    rscptf("\nIf no operations are given, the current contents of the\n");
    rscptf("resource file are displayed.\n\n");
    rscptf("\nOperations:\n");
    rscptf("   +file=resname    - add file as resource resname\n");
    rscptf("   -resname         - delete resource resname\n");
    rscptf("   -+file=resname   - replace resource resname with file\n");
    exit(OSEXFAIL);
}

static void errexit(msg, arg)
char *msg;
long  arg;
{
    rscptf(msg, arg);
    rscptf("\n");
    exit(OSEXFAIL);
}


static void listexit(fp, msg)
osfildef *fp;
char     *msg;
{
    osfcls(fp);
    errexit(msg);
}

typedef struct opdef opdef;
struct opdef
{
    opdef  *opnxt;                                /* next operation in list */
    char   *opres;                                         /* resource name */
    char   *opfile;                                          /* file to add */
    int     opflag;               /* indication of what to do with resource */
#   define  OPFADD   0x01                                   /* add resource */
#   define  OPFDEL   0x02                                /* delete resource */
#   define  OPFDONE  0x04                  /* this entry has been processed */
};


/* copy a section of the input file to the output file */
char copybuf[16 * 1024];
void copyout(fpin, fpout, siz, endpos_ofs)
osfildef *fpin;
osfildef *fpout;
ulong     siz;
uint      endpos_ofs;
{
    uint  cursiz;
    ulong curpos;
    ulong endpos;
    uchar buf[4];

    curpos = osfpos(fpout);
    while (siz)
    {
	cursiz = (siz > sizeof(copybuf) ? sizeof(copybuf) : siz);
	siz -= cursiz;
	
	if (osfrb(fpin, copybuf, cursiz)
	    || osfwb(fpout, copybuf, cursiz))
	{
	    osfcls(fpin);
	    osfcls(fpout);
	    errexit("error copying resource");
	}
    }
    
    endpos = osfpos(fpout);
    osfseek(fpout, (ulong)(curpos + endpos_ofs), OSFSK_SET);
    oswp4(buf, endpos);
    if (osfwb(fpout, buf, 4)) errexit("error writing resource");
    osfseek(fpout, endpos, OSFSK_SET);
}

/* process an operation */
static void procop(fpout, op, first_xfcn)
osfildef *fpout;
opdef    *op;
ulong    *first_xfcn;
{
    osfildef *fpin;
    char      buf[128];
    uint      fsiz;
    ulong     sizpos;
    ulong     endpos;

    /* remember location of first resource if necessary */
    if (fpout && *first_xfcn == 0)
	*first_xfcn = osfpos(fpout);

    fpin = osfoprb(op->opfile);
    if (!fpin)
    {
	rscptf("%s: ", op->opfile);
	errexit("unable to open file");
    }

    /* get file size */
    osfseek(fpin, 0L, OSFSK_END);
    fsiz = osfpos(fpin);
    osfseek(fpin, 0L, OSFSK_SET);
    
    /* set up header */
    buf[0] = 4;
    memcpy(buf + 1, "XFCN", 4);
    if (osfwb(fpout, buf, 5)) errexit("error writing resource");
    sizpos = osfpos(fpout);               /* remember where size field goes */
    oswp4(buf, 0);
    if (osfwb(fpout, buf, 4)) errexit("error writing resource");
    
    /* set up XFCN header */
    oswp2(buf, fsiz);
    buf[2] = strlen(op->opres);
    strcpy(buf + 3, op->opres);
    if (osfwb(fpout, buf, (uint)(buf[2] + 3)))
	errexit("error writing resource");
    
    /* copy the resource to the output */
    while (fsiz)
    {
	uint cursiz;
	
	cursiz = (fsiz > sizeof(copybuf) ? sizeof(copybuf) : fsiz);
	fsiz -= cursiz;
	
	if (osfrb(fpin, copybuf, cursiz)
	    || osfwb(fpout, copybuf, cursiz))
	    errexit("error copying resource");
    }

    /* write end position */
    endpos = osfpos(fpout);
    oswp4(buf, endpos);
    osfseek(fpout, sizpos, OSFSK_SET);
    if (osfwb(fpout, buf, 4)) errexit("error writing resource");
    osfseek(fpout, endpos, OSFSK_SET);
    
    osfcls(fpin);
}


/* process changes to resource file (or just list it) */
static void rscproc(fp, fpout, oplist)
osfildef *fp;
osfildef *fpout;
opdef    *oplist;
{
    uchar  buf[128];
    ulong  siz;
    uchar  datebuf[27];
    ulong  endpos;
    uchar  nambuf[40];
    uint   rsiz;
    uint   cnt = 0;
    opdef *op;
    int    copyrsc;
    ulong  startpos;
    uint   endpos_ofs;
    ulong  first_xfcn = 0;
    ulong  extcnt_pos = 0;
    
    /* check file and version headers, and get flags and timestamp */
    if (osfrb(fp, buf, (int)(sizeof(FIOFILHDR) + sizeof(FIOVSNHDR) + 2)))
	listexit(fp, "error reading file header");
    if (memcmp(buf, FIOFILHDR, sizeof(FIOFILHDR)))
	listexit(fp, "invalid resource file header");
    if (memcmp(buf + sizeof(FIOFILHDR), FIOVSNHDR, sizeof(FIOVSNHDR))
	&& memcmp(buf + sizeof(FIOFILHDR), FIOVSNHDR2, sizeof(FIOVSNHDR2))
	&& memcmp(buf + sizeof(FIOFILHDR), FIOVSNHDR3, sizeof(FIOVSNHDR3)))
	listexit(fp, "incompatible resource file version");
    if (osfrb(fp, datebuf, (size_t)26))
	listexit(fp, "error reading file");
    datebuf[26] = '\0';
    
    if (fpout)
    {
	if (osfwb(fpout, buf,
		  (int)(sizeof(FIOFILHDR) + sizeof(FIOVSNHDR) + 2))
	    || osfwb(fpout, datebuf, (size_t)26))
	    listexit(fp, "error writing header");
    }

    /* if listing, show file creation timestamp */
    if (!fpout) rscptf("\nFile compiled: %s\n", datebuf);

    for (;;)
    {
	/* assume this resource will be copied to the output */
	copyrsc = TRUE;

	startpos = osfpos(fp);
	if (osfrb(fp, buf, 1)
	    || osfrb(fp, buf + 1, (int)(buf[0] + 4)))
	    listexit(fp, "error reading file");

	memcpy(nambuf, buf + 1, (size_t)buf[0]);
	nambuf[buf[0]] = '\0';
	
	endpos = osrp4(buf + 1 + buf[0]);
	endpos_ofs = 1 + buf[0];
	siz = endpos - startpos;
	
	/* see what kind of resource we have, and do the right thing */
	if (!strcmp(nambuf, "$EOF"))
	{
	    /* end of file marker - quit here */
	    break;
	}
	else if (!strcmp(nambuf, "EXTCNT"))
	{
	    /* place to write start of XFCN's? */
	    if (siz >= 17 && fpout) extcnt_pos = osfpos(fpout);
	}
	else if (!strcmp(nambuf, "XFCN"))
	{
	    if (osfrb(fp, buf, 3) || osfrb(fp, buf + 3, (int)buf[2]))
		listexit(fp, "error reading file");
	    rsiz = osrp2(buf);
	    buf[3 + buf[2]] = '\0';

	    if (fpout)
	    {
		/* see if this resource is in the list */
		for (op = oplist ; op ; op = op->opnxt)
		{
		    if (!(op->opflag & OPFDONE)
			&& !strcmp(buf + 3, op->opres))
		    {
			op->opflag |= OPFDONE;

			if (!(op->opflag & OPFDEL))
			{
			    rscptf("error: resource \"%s\" already in %s\n",
				   op->opres, "file (use -+ to replace)");
			}
			else
			{
			    if (op->opflag & OPFADD)
				procop(fpout, op, &first_xfcn);
			    copyrsc = FALSE;
			}
			break;
		    }
		}
	    }
	    else
	    {
		/* no output file - just list the resource */
		rscptf("%s (size = %u)\n", buf + 3, rsiz);
	    }
	    
	    ++cnt;
	}
	
	if (fpout && copyrsc)
	{
	    osfseek(fp, startpos, OSFSK_SET);
	    copyout(fp, fpout, siz, endpos_ofs);
	}
	
	/* skip to the next resource */
	osfseek(fp, endpos, OSFSK_SET);
    }
    
    /* now go through what's left, and add new resource */
    if (fpout)
    {
	for (op = oplist ; op ; op = op->opnxt)
	{
	    if (!(op->opflag & OPFDONE) && (op->opflag & OPFADD))
	    {
		procop(fpout, op, &first_xfcn);
		op->opflag |= OPFDONE;
	    }
	}
    }
    
    if (!fpout && cnt == 0) rscptf("No user resources found.\n");
    if (fpout)
    {
	/* write EOF resource */
	if (osfwb(fpout, "\004$EOF\0\0\0\0", 9))
	    errexit("error writing resource");

	/* write first_xfcn value to EXTCNT resource */
	if (extcnt_pos)
	{
	    osfseek(fpout, extcnt_pos + 13, OSFSK_SET);
	    oswp4(buf, first_xfcn);
	    osfwb(fpout, buf, 4);
	}
    }
}

static void os_defext( fn, ext )
char *fn;
char *ext;
{
    char *p = fn+strlen(fn);
    while ( p>fn )
    {
        p--;
        if (*p=='.') return;      /* already has an extension */
        if (*p=='/' || *p=='\\' || *p==':')
	    break;    /* found a path */
    }
    strcat( fn, "." );              /* add a dot */
    strcat( fn, ext );              /* add the extension */
}


static opdef *addop(cur, nam)
opdef *cur;
char  *nam;
{
    int    flag = 0;
    char  *p;
    opdef *newop;

    /* look for leading flags */
    for (;;)
    {
	if (*nam == '-')
	{
	    flag |= OPFDEL;
	    ++nam;
	}
	else if (*nam == '+')
	{
	    flag |= OPFADD;
	    ++nam;
	}
	else if (isspace(*nam))
	    ++nam;
	else
	    break;
    }
    
    /* compatibility:  assume '+' flag */
    if (flag == 0) flag = OPFADD;
    
    /* look for '=' */
    for (p = nam ; *p && *p != '=' ; ++p);
    if (*p == '=')
    {
	*p = '\0';
	++p;
	while (isspace(*p)) ++p;
    }
    
    /* make sure we have a resource name if adding a file */
    if ((flag & OPFADD) && !*p)
    {
	rscptf("%s: ", nam);
	errexit("resource name missing adding file");
    }
    
    /* allocate space and set up new op */
    newop = (opdef *)malloc(sizeof(opdef) + strlen(p) + strlen(nam) + 2);

    newop->opnxt  = cur;
    newop->opflag = flag;
    newop->opres  = (char *)(newop + 1);
    if (flag & OPFADD)
    {
	strcpy(newop->opres, p);
	newop->opfile = newop->opres + strlen(newop->opres) + 1;
	strcpy(newop->opfile, nam);
    }
    else
    {
	strcpy(newop->opres, nam);
	newop->opfile = (char *)0;
    }
    
    return(newop);
}


void main(argc, argv)
int   argc;
char *argv[];
{
    int       i;
    osfildef *fpin;
    osfildef *fpout;
    char      tmpfile[OSFNMAX + 1];
    char      inbuf[OSFNMAX + 1];
    char     *p;
    char     *infile;
    opdef    *op;
    char      buf[128];
    opdef    *oplist = (opdef *)0;
    
    rscptf("TADS Resource Manager version 2.0.15\n");
  rscptf("Copyright (c) 1992 by Michael J. Roberts.  All Rights Reserved.\n");
    if (argc < 2) usage();
    
    /* open input file for reading */
    infile = argv[1];
    strcpy(inbuf, infile);
    os_defext(inbuf, "gam");
    if (!(fpin = osfoprb(inbuf)))
	errexit("unable to open resource file");

    /* if no operations are desired, just list the file */
    if (argc == 2)
    {
	rscproc(fpin, (osfildef *)0);
	osfcls(fpin);
	exit(OSEXSUCC);
    }

    /* we'll be doing some writing; open a temporary file */
    strcpy(tmpfile, inbuf);
    for (p = tmpfile + strlen(tmpfile) ; p > tmpfile &&
         *(p-1) != ':' && *(p-1) != '\\' && *(p-1) != '/' ; --p);
    strcpy(p, "$TADSRSC.TMP");

    if (!(fpout = osfopwb(tmpfile)))
	errexit("unable to create temporary file");

    /* see if we need to read a response file */
    if (argv[2][0] == '@')
    {
	osfildef *argfp;
	int       l;
	char     *p;
	
	if (!(argfp = osfoprt(argv[2]+1)))
	    errexit("unable to open response file");
	
	for (;;)
	{
	    if (!osfgets(buf, sizeof(buf), argfp)) break;
	    l = strlen(buf);
	    if (l && buf[l-1] == '\n') buf[--l] = '\0';
	    for (p = buf ; isspace(*p) ; ++p);
	    if (!*p) continue;
	    oplist = addop(oplist, p);
	}
	osfcls(argfp);
    }
    else
    {
	for (i = 2 ; i < argc ; ++i) oplist = addop(oplist, argv[i]);
    }
    
    rscproc(fpin, fpout, oplist);
    
    for ( ; oplist ; oplist = oplist->opnxt)
    {
	if (!(oplist->opflag & OPFDONE))
	    rscptf("warning: resource \"%s\" not found\n", oplist->opres);
    }
    
    /* close files */
    osfcls(fpin);
    osfcls(fpout);
    
    /* delete original file, and rename temp file */
    if (remove(inbuf)) errexit("error deleting input file");
    if (rename(tmpfile, inbuf)) errexit("error renaming temporary file");
    
    exit(OSEXSUCC);
}



