#ifdef RCSID
static char RCSid[] =
"$Header: c:/tads/tads2/RCS/OBJCOMP.C 1.2 94/11/06 13:06:10 mroberts Exp $";
#endif

/* Copyright (c) 1992 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  objcomp.c - object manipulation routines for compiler
Function
  Provides routines used only by the compiler
Notes
  Split off from obj main module to make run-time smaller
Modified
  12/18/92 MJRoberts     - creation
*/

#include "os.h"
#include "std.h"
#include "err.h"
#include "mcm.h"
#include "obj.h"

/*
 *   Set up for emitting code into an object.  No undo information is
 *   kept for this type of operation, as it is presumed that the object is
 *   being compiled rather than being manipulated at run-time.  Any
 *   previous value for the property is deleted, a property header is set
 *   up, and the offset of the next free byte in the object is returned.
 */
uint objemt(ctx, objn, prop, typ)
mcmcxdef *ctx;
objnum    objn;
prpnum    prop;
dattyp    typ;
{
    objdef *objptr;
    prpdef *p;
    
    objptr = (objdef *)mcmlck(ctx, (mcmon)objn);
    
    ERRBEGIN(ctx->mcmcxgl->mcmcxerr)

    objdelp(ctx, objn, prop);          /* delete old property value, if any */
    p = objpfre(objptr);                        /* get top of property area */
    
    if ((char *)p - (char *)objptr + PRPHDRSIZ >=
	mcmobjsiz(ctx, (mcmon)objn))
    {
	ushort newsiz = 64 + ((objfree(objptr) + PRPHDRSIZ) -
			      mcmobjsiz(ctx, (mcmon)objn));
	objptr = objexp(ctx, objn, &newsiz);
	p = objpfre(objptr);                       /* object may have moved */
    }
    
    /* set up property header as much as we can (don't know size yet) */
    prpsetprop(p, prop);
    prptype(p) = typ;
    prpflg(p) = 0;
    objsnp(objptr, objnprop(objptr) + 1);              /* one more property */

    ERRCLEAN(ctx->mcmcxgl->mcmcxerr)
	mcmunlck(ctx, (mcmon)objn);
    ERRENDCLN(ctx->mcmcxgl->mcmcxerr)

    /* dirty the cache object and release the lock before return */
    mcmtch(ctx, objn);
    mcmunlck(ctx, (mcmon)objn);
    return(((uchar *)prpvalp(p)) - ((uchar *)objptr));
}

/* done emitting code into property - finish setting object information */
void objendemt(ctx, objn, prop, endofs)
mcmcxdef *ctx;
objnum    objn;
prpnum    prop;
uint      endofs;                          /* ending offset of code emitted */
{
    objdef *objptr;
    prpdef *p;
    uint    siz;
    
    objptr = (objdef *)mcmlck(ctx, (mcmon)objn);
    p = objofsp(objptr, objgetp(ctx, objn, prop, (dattyp *)0));
    
    siz = endofs - (((uchar *)prpvalp(p)) - ((uchar *)objptr));
    
    prpsetsize(p, siz);
    objsfree(objptr, objfree(objptr) + siz + PRPHDRSIZ);
    
    /* mark the object as changed, and unlock it */
    mcmtch(ctx, (mcmon)objn);
    mcmunlck(ctx, (mcmon)objn);
}

/* add superclasses to an object */
void objaddsc(mctx, sccnt, objn)
mcmcxdef *mctx;
int       sccnt;
objnum    objn;
{
    objdef *o;
    ushort  siz;
    
    /* get lock on object */
    o = (objdef *)mcmlck(mctx, objn);
    
    /* make sure there's enough space, adding space if needed */
    if (mcmobjsiz(mctx, (mcmon)objn) - objfree(o) < 2 * sccnt)
    {
	siz = 64 + ((2 * sccnt + objfree(o)) -
		    mcmobjsiz(mctx, (mcmon)objn));
	o = objexp(mctx, objn, &siz);                  /* expand the object */
    }

    /* move properties, if any, above added superclasses */
    if (objnprop(o))
	memmove(objprp(o), ((uchar *)objprp(o)) + 2 * sccnt,
		(size_t)(((uchar *)o) + objfree(o) - (uchar *)objprp(o)));
    
    /* set new free pointer */
    objsfree(o, objfree(o) + 2 * sccnt);
    
    /* mark cache object modified and unlock it */
    mcmtch(mctx, objn);
    mcmunlck(mctx, objn);
}

/* delete an object's properties and superclasses */
void objclr(mctx, objn, mindel)
mcmcxdef *mctx;
objnum    objn;
prpnum    mindel;        /* don't delete properties with numbers below this */
{
    objdef *o;
    prpdef *p;
    int     cnt;
    prpnum  prop;
    int     indexed;
    
    /* get a lock on the object */
    o = (objdef *)mcmlck(mctx, objn);
    indexed = objflg(o) & OBJFINDEX;
    
    /* delete superclasses - move properties down over former sc array */
    if (objnprop(o))
	memmove(objsc(o), objprp(o),
		(size_t)(((uchar *)o) + objfree(o) - (uchar *)objprp(o)));
    objsnsc(o, 0);                                 /* zero superclasses now */
    
    /* delete non-"system" properties (propnum < mindel) */
    for (p = objprp(o), cnt = objnprop(o) ; cnt ; --cnt)
    {
	if ((prop = prpprop(p)) >= mindel)
	{
	    prpflg(p) &= ~PRPFIGN;   /* delete even if it was marked ignore */
	    objdelp(mctx, objn, prop);         /* remove prpdef from object */
	    /* p is left pointing at next prop, as it was moved down */
	}
	else
	    p = objpnxt(p);                   /* advance over this property */
    }
    
    /* mark cache object modified and unlock it */
    mcmtch(mctx, objn);
    mcmunlck(mctx, objn);
    if (indexed) objindx(mctx, objn);
}

/* set up just-compiled object:  mark static part and original props */
void objcomp(mctx, objn)
mcmcxdef *mctx;
objnum    objn;
{
    objdef *objptr;
    prpdef *p;
    int     cnt;
    
    /* lock object */
    objptr = (objdef *)mcmlck(mctx, (mcmon)objn);
    p = objprp(objptr);
    
    /* set static entries:  free space pointer, and number of properties */
    objsetst(objptr, objnprop(objptr));
    objsetrst(objptr, objfree(objptr));
    
    /* go through properties, marking each as original */
    for (cnt = objnprop(objptr) ; cnt ; p = objpnxt(p), --cnt)
    {
	assert(p < objptr + mcmobjsiz(mctx, (mcmon)objn));
	prpflg(p) |= PRPFORG;             /* set ORIGINAL flag for property */
    }
    
    /* mark object changed, and unlock it */
    mcmtch(mctx, objn);
    mcmunlck(mctx, objn);
}

