/*
 * unix.c
 *
 * IO Interface for UN*X systems using curses
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef NO_MEMMOVE
#define memmove(a, b, c) bcopy(b, a, c)
#endif

/* Prototypes for getopt */
#ifdef USE_UNISTD_H
#include <unistd.h>
#elif USE_GETOPT_H
#include <getopt.h>
#else
#ifndef USE_NOTHING
extern int getopt(int, char **, char *);
extern int optind, opterr;
extern char *optarg;
#endif
#endif

/* needed for struct timeval, gettimeofday */
#include <sys/time.h>

#ifdef USE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif

#include "frotz.h"

#define INFORMATION "\
\n\
FROTZ V2.01 - an interpreter for all infocom games. Complies with standard 0.2\n\
for all games except V6 (graphic) games. Written by Stefan Jokisch in 1995-96\n\
\n\
\t-f # set the foreground color (if supported)\n\
\t-b # set the background color (if supported)\n\
\t-w # set the screen width\n\
\t-h # set the screen height\n\
\t-l # set the left margin\n\
\t-r # set the right margin\n\n\
\t-u # set the number of undo slots for multiple undo\n\
\t-p   Plain ASCII output only\n\
\t-o   monitor object movement\n\
\t-O   monitor object locating\n\
\t-a   monitor attribute assignment\n\
\t-A   monitor attribute testing\n\n\
\t-t   set the Tandy bit\n"

static int user_foreground_color = -1;
static int user_background_color = -1;
static int user_screen_width = -1;
static int user_screen_height = -1;
static char user_tandy_bit = 0;

static char curses_active = 0; /* true if os_init_screen has been run */
static char dump_message = 0;  /* true if we are discarding text output
				   (set by os_message_start) */
static int current_text_style = 0; /* Since I can't use attr_get, which
                        	      would make things easier, I need to
				      use the same hack the MS-DOS port
				      does...keep the current style in a
				      global variable. */
#ifdef COLOR_SUPPORT
static int current_color = 0;
#endif

#ifdef SOUND_SUPPORT
int unix_init_sound(void);
#endif

#define MAX_HISTORY 20
static char *history_buffer[MAX_HISTORY];
static short history_pointer = 0; /* Pointer to next available slot */
static short history_frame = 0; /* Pointer to what the user's looking at */

/* 
 * The Unix port of frotz can have an arbitrary number of mappings for
 * the European character set defined by the standard.  The mapping is chosen
 * by an appropriate switch on the command line.  The default mapping is
 * ISO-8859-1, a.k.a. Latin-1, as X windows supports this.  A straight ascii
 * mapping is also available, for stupid terminals without accented character
 * support.  
 *
 * Adding new mappings is pretty straightforward.  Just grab a copy of the
 * specification, and create a new array containing single-character
 * mappings for all the european characters (at this time, 155 to 223).
 * If you need two characters to replace a missing ligature with two or more
 * characters, leave it as 0 and make sure os_display_char and os_char_length
 * do the right thing.  (They already catch most of the possible problem
 * characters in order to support the ascii mapping.)
 *
 */
static unsigned char iso_8859_1_map[] = {
  0xe4, 0xf6, 0xfc, 0xc4, 0xd6, 0xdc,    /* aouAOU with umlaut */
  0xdf, 0xab, 0xbb,                      /* sz ligature, quotation marks */
  0xeb, 0xef, 0xff, 0xcb, 0xcf,          /* eiyEI with umlaut */
  0xe1, 0xe9, 0xed, 0xf3, 0xfa, 0xfd,    /* aeiouy with acute */
  0xc1, 0xc9, 0xcd, 0xd3, 0xda, 0xdd,    /* AEIOUY with acute */
  0xe0, 0xe8, 0xec, 0xf2, 0xf9,          /* aeiou with grave */
  0xc0, 0xc8, 0xcc, 0xd2, 0xd9,          /* AEIOU with grave */
  0xe2, 0xea, 0xee, 0xf4, 0xfb,          /* aeiou with circumflex */
  0xc2, 0xca, 0xce, 0xd4, 0xdb,          /* AEIOU with circumflex */
  0xe5, 0xc5, 0xf8, 0xd8,                /* aA with ring, oO with slash */
  0xe3, 0xf1, 0xf5, 0xc3, 0xd1, 0xd5,    /* anoANO with tilde */
  0xe6, 0xc6, 0xe7, 0xc7,                /* ae, AE, cC with cedilla */
  0xfe, 0xf0, 0xde, 0xd0, 0xa3,          /* Icelandic chars and pound sign */
  0x00, 0x00, 0xa1, 0xbf                 /* oe, OE, inverted ! and ? */
};

static unsigned char flatscii_map[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00,    /* aouAOU with umlaut */
  0x00, 0x00, 0x00,                      /* sz ligature, quotation marks */
  'e', 'i', 'y', 'E', 'I',               /* eiyEI with umlaut */
  'a', 'e', 'i', 'o', 'u', 'y',          /* aeiouy with acute */
  'A', 'E', 'I', 'O', 'U', 'Y',          /* AEIOUY with acute */
  'a', 'e', 'i', 'o', 'u',               /* aeiou with grave */
  'A', 'E', 'I', 'O', 'U',               /* AEIOU with grave */
  'a', 'e', 'i', 'o', 'u',               /* aeiou with circumflex */
  'A', 'E', 'I', 'O', 'U',               /* AEIOU with circumflex */
  'a', 'A', 'o', 'O',                    /* aA with ring, oO with slash */
  'a', 'n', 'o', 'A', 'N', 'O',          /* anoANO with tilde */
  0x00, 0x00, 'c', 'C',                  /* ae, AE, cC with cedilla */
  0x00, 0x00, 0x00, 0x00, 'L',           /* Icelandic chars and pound sign */
  0x00, 0x00, '!', '?'                   /* oe, OE, inverted ! and ? */
};

static unsigned char *euro_map = NULL;  /* Pointer to map currently in use */



/* ========================================================================== *\


			  S O U N D   R O U T I N E S


\* ========================================================================== */



/*
 * os_beep
 *
 * Play a beep sound. Ideally, the sound should be high- (number == 1)
 * or low-pitched (number == 2).
 *
 */

void os_beep (int number)
{

    beep();

}/* os_beep */

#ifndef SOUND_SUPPORT
/*
 * os_prepare_sample
 *
 * Load the given sample from the disk.
 *
 */

void os_prepare_sample (int number)
{

    /* Not implemented */

}/* os_prepare_sample */

/*
 * os_start_sample
 *
 * Play the given sample at the given volume (ranging from 1 to 8 and
 * 255 meaning a default volume). The sound is played once or several
 * times in the background (255 meaning forever). In Z-code 3, the
 * repeats value is always 0 and the number of repeats is taken from
 * the sound file itself. The end_of_sound function is called as soon
 * as the sound finishes.
 *
 */

void os_start_sample (int number, int volume, int repeats)
{

    /* Not implemented */

}/* os_start_sample */

/*
 * os_stop_sample
 *
 * Turn off the current sample.
 *
 */

void os_stop_sample (void)
{

    /* Not implemented */

}/* os_stop_sample */

/*
 * os_finish_with_sample
 *
 * Remove the current sample from memory (if any).
 *
 */

void os_finish_with_sample (void)
{

    /* Not implemented */

}/* os_finish_with_sample */

/*
 * os_wait_sample
 *
 * Stop repeating the current sample and wait until it finishes.
 *
 */

void os_wait_sample (void)
{

    /* Not implemented */

}/* os_wait_sample */
#endif /* No SOUND_SUPPORT */


/* ========================================================================== *\


			P I C T U R E   R O U T I N E S


\* ========================================================================== */



/*
 * os_picture_data
 *
 * Return true if the given picture is available. If so, store the
 * picture width and height in the appropriate variables. Picture
 * number 0 is a special case: Write the number of pictures available
 * and the picture file release number into the height and width
 * variables respectively when this picture number is asked for.
 *
 */

int os_picture_data (int picture, int *height, int *width)
{

    /* Not implemented */
    return 0;

}/* os_picture_data */

/*
 * os_draw_picture
 *
 * Display a picture at the given coordinates. Top left is (1,1).
 *
 */

void os_draw_picture (int picture, int y, int x)
{

    /* Not Implemented */

}/* os_draw_picture */

/*
 * os_peek_colour
 *
 * Return the colour of the screen unit below the cursor. (If the
 * interface uses a text mode, it may return the background colour
 * of the character at the cursor position instead.) This is used
 * when text is printed on top of pictures. Note that this coulor
 * need not be in the standard set of Z-machine colours. To handle
 * this situation, Frotz extends the colour scheme: Colours above
 * 15 (and below 256) may be used by the interface to refer to non
 * standard colours. Of course, os_set_colour must be able to deal
 * with these colours.
 *
 */

int os_peek_colour (void)
{

    /* Not Implemented */
    return 0;

}/* os_peek_colour */



/* ========================================================================== *\


			  M O U S E   R O U T I N E S


\* ========================================================================== */



/*
 * os_mouse_area
 *
 * Restrict the movement of the mouse cursor to the given area. Top
 * left coordinates are (1,1).
 *
 */

void os_mouse_area (int top, int left, int bottom, int right)
{

  /* Not Implemented */

}



/* ========================================================================== *\


			   T E X T   R O U T I N E S


\* ========================================================================== */



/*
 * os_font_available
 *
 * Return true if the given font is supported by this interface. The
 * font can be any of
 *
 *    TEXT_FONT
 *    PICTURE_FONT
 *    GRAPHICS_FONT
 *    FIXED_WIDTH_FONT
 *
 * The fonts are defined in the Specification of the Z-machine.
 *
 */

int os_font_available (int font)
{

    if (font == TEXT_FONT) return 1; /* Truth in advertising */
    return 0;

}/* os_font_available */

#ifdef COLOR_SUPPORT
/*
 * unix_convert
 *
 * Converts frotz's (and Infocom's) color values to ncurses color values.
 *
 */

static short unix_convert(int color)
{
  switch(color) {
        case BLACK_COLOUR: return COLOR_BLACK;
        case RED_COLOUR: return COLOR_RED;
        case GREEN_COLOUR: return COLOR_GREEN;
        case YELLOW_COLOUR: return COLOR_YELLOW;
        case BLUE_COLOUR: return COLOR_BLUE;
        case MAGENTA_COLOUR: return COLOR_MAGENTA;
        case CYAN_COLOUR: return COLOR_CYAN;
        case WHITE_COLOUR: return COLOR_WHITE;
  }
  return 0;
}
#endif

/*
 * os_set_colour
 *
 * Set the foreground and background colours which can be:
 *
 *     DEFAULT_COLOUR
 *     BLACK_COLOUR
 *     RED_COLOUR
 *     GREEN_COLOUR
 *     YELLOW_COLOUR
 *     BLUE_COLOUR
 *     MAGENTA_COLOUR
 *     CYAN_COLOUR
 *     WHITE_COLOUR
 *
 *     MS-DOS 320 columns MCGA mode only:
 *
 *     GREY_COLOUR
 *
 *     Amiga only:
 *
 *     LIGHTGREY_COLOUR
 *     MEDIUMGREY_COLOUR
 *     DARKGREY_COLOUR
 *
 * There may be more colours in the range from 16 to 255; see the
 * remarks about os_peek_colour.
 *
 */

#ifdef COLOR_SUPPORT
static short colorspace[10][10];
static short count = 0;
#endif

void os_set_colour (int new_foreground, int new_background)
{
#ifdef COLOR_SUPPORT
    int saved_style;

    saved_style = current_text_style;
    if (new_foreground == 1) new_foreground = h_default_foreground;
    if (new_background == 1) new_background = h_default_background;
    if (!colorspace[new_foreground][new_background]) {
      init_pair(++count, unix_convert(new_foreground), unix_convert(new_background));
      colorspace[new_foreground][new_background] = count;
    }
    current_color = COLOR_PAIR(colorspace[new_foreground][new_background]);
    os_set_text_style(saved_style);

#endif
}/* os_set_colour */

/*
 * os_set_font
 *
 * Set the font for text output. The interpreter takes care not to
 * choose fonts which aren't supported by the interface.
 *
 */

void os_set_font (int new_font, int *height, int *width)
{

    /* Not implemented */
    *height = 1; *width = 1;

}/* os_set_font */

/*
 * os_set_text_style
 *
 * Set the current text style. Following flags can be set:
 *
 *     REVERSE_STYLE
 *     BOLDFACE_STYLE
 *     EMPHASIS_STYLE (aka underline aka italics)
 *     FIXED_WIDTH_STYLE
 *
 */

void os_set_text_style (int new_style)
{
    int temp = 0;

    current_text_style = new_style;
    if (new_style & REVERSE_STYLE) temp |= A_REVERSE;
    if (new_style & BOLDFACE_STYLE) temp |= A_BOLD;
    if (new_style & EMPHASIS_STYLE) temp |= A_UNDERLINE;
#ifdef COLOR_SUPPORT
    attrset(temp | current_color);
#else
    attrset(temp);
#endif

}/* os_set_text_style */


/*
 * os_display_char
 *
 * Display a character of the current font using the current colours
 * and text style. The cursor moves to the next position. Printable
 * codes are all ASCII values from 32 to 126, European characters from
 * EURO_MIN to EURO_MAX (as defined in the Specification of the
 * Z-machine), character 9 (gap between two sentences) and character
 * 11 (paragraph indentation). The screen should not be scrolled after
 * printing to the bottom right corner.
 *
 */

void os_display_char (int c)
{
  
    if (dump_message) return;          /* os_message_XXX says to ignore text */

    if (c >= EURO_MIN && c <= EURO_MAX)
      if (euro_map[c - EURO_MIN])      /* 0 means multi-char substitution */
	c = euro_map[c - EURO_MIN];
      else switch(c) {
	case 155: echochar('a'); c = 'e'; break; /* flatscii */
	case 156: echochar('o'); c = 'e'; break; /* flatscii */
	case 157: echochar('u'); c = 'e'; break; /* flatscii */
        case 158: echochar('A'); c = 'e'; break; /* flatscii */
        case 159: echochar('O'); c = 'e'; break; /* flatscii */
        case 160: echochar('U'); c = 'e'; break; /* flatscii */
        case 161: echochar('s'); c = 's'; break; /* flatscii */
        case 162: echochar('<'); c = '<'; break; /* flatscii */
        case 163: echochar('>'); c = '>'; break; /* flatscii */
        case 211: echochar('a'); c = 'e'; break; /* flatscii */
        case 212: echochar('A'); c = 'E'; break; /* flatscii */
        case 215: case 216: echochar('t'); c = 'h'; break; /* flatscii */
        case 217: case 218: echochar('T'); c = 'H'; break; /* flatscii */
	case 220: echochar('o'); c = 'e'; break; /* iso-8859-1, flatscii */ 
        case 221: echochar('O'); c = 'E'; break; /* iso-8859-1, flatscii */
      }

    echochar(c);

}/* os_display_char */


/*
 * os_display_string
 *
 * Pass a string of characters to os_display_char.
 *
 */

void os_display_string (const char *s)
{
    int i;

    for (i = 0; s[i] != 0; i++)
	os_display_char ((unsigned char) s[i]);

}/* os_display_string */

/*
 * os_char_width
 *
 * Return the length of the character in screen units.
 *
 */

int os_char_width (int c)
{
    /* Certain ligatures may not be represented in the mapping table.  Because
       these change the length of the text, they are represented as zeros.
       They change the length here, and are caught in os_display_char. */
    if ((c >= EURO_MIN) && (c <= EURO_MAX) && !euro_map[c - EURO_MIN])
      return 2;
    else
      return 1;

}/* os_char_width*/

/*
 * os_string_width
 *
 * Calculate the length of a word in screen units. Apart from letters,
 * the word may contain special codes:
 *
 *    NEW_STYLE - next character is a new text style
 *    NEW_FONT  - next character is a new font
 *
 */

int os_string_width (const char *s)
{
    int width;
    int c;
    int i;

    width = 0;
    i = 0;

    while (s[i] != 0) {

	c = (unsigned char) s[i++];

	if (!(c == NEW_STYLE || c == NEW_FONT))
	  width += os_char_width (c);
    }

    return width;

}/* os_string_width */

/*
 * os_set_cursor
 *
 * Place the text cursor at the given coordinates. Top left is (1,1).
 *
 */

void os_set_cursor (int y, int x)
{

    /* Curses thinks the upper left is (0,0) */
    move(--y, --x);

}/* os_set_cursor */



/* ========================================================================== *\


			  I N P U T   R O U T I N E S


\* ========================================================================== */



/*
 * os_cursor_on
 *
 * Make the cursor visible.
 *
 */

void os_cursor_on (void)
{

    curs_set(1);

}/* os_cursor_on */

/*
 * os_cursor_off
 *
 * Turn cursor off.
 *
 */

void os_cursor_off (void)
{

    curs_set(0);

}/* os_cursor_off */

/*
 * unix_set_global_timeout
 *
 * This sets up a time structure to determine when unix_read_char should
 * return zero (representing input timeout).  When current system time
 * equals global_timeout, boom.
 *
 */

static struct timeval global_timeout;

static void unix_set_global_timeout(int timeout)
{
    if (!timeout) global_timeout.tv_sec = 0;
    else {
        gettimeofday(&global_timeout, NULL);
	global_timeout.tv_sec += (timeout/10);
	global_timeout.tv_usec += ((timeout%10)*100000);
	if (global_timeout.tv_usec > 999999) {
	  global_timeout.tv_sec++;
	  global_timeout.tv_usec -= 1000000;
	}
    }
}

/*
 * unix_read_char
 *
 * This uses the curses getch() routine to get the next character typed,
 * and returns it unless the global timeout is reached.  It returns values
 * which the standard considers to be legal input, and also returns editing
 * and frotz hot keys.  If called with a non-zero flag, it will also return
 * line-editing keys like INSERT, 
 *
 */

static int unix_read_char(int flag)
{
    int c;
    struct timeval ctime;

    while(1) {
      /* Timed keyboard input.  Crude but functional. */
      if (global_timeout.tv_sec) {
	  nodelay(stdscr, TRUE);
	  do {
	      c = getch();
	      if (c == ERR) {
		  gettimeofday(&ctime, NULL);
		  if ((ctime.tv_sec >= global_timeout.tv_sec) &&
		      (ctime.tv_usec >= global_timeout.tv_usec)) {
		      nodelay(stdscr, FALSE);
		      return 0;
		  }
	      }
	  } while (c == ERR);
	  nodelay(stdscr, FALSE);
      }
      /* The easy way. */
      else c = getch();

      /* Catch 98% of all input right here... */
      if ( ((c >= 32) && (c <= 127)) || (c == 13)) 
	return c;

      /* ...and the other 2% makes up 98% of the code. :( */
      switch(c) {
	/* Lucian P. Smith reports KEY_ENTER on Irix 5.3.  10 has never
	   been reported, but I'm leaving it in just in case. */
        case 10: case KEY_ENTER: return 13;
	/* I've seen BACKSPACE and DC returned on various terminals
	   for the backspace key.  In addition, ascii 8 has been reported
	   under SunOS and Irix. */
        case KEY_BACKSPACE: case KEY_DC: case 8: return 127;
	/* On seven-bit connections, "Alt-Foo" is returned as an escape 
	   followed by the ASCII value of the letter.  We have to decide
           here whether to return a single escape or a frotz hot key. */
        case 27: nodelay(stdscr, TRUE); c = getch(); nodelay(stdscr, FALSE);
	         if (c == ERR) return 27;
		 switch(c) {
		   case 112: return HOT_KEY_PLAYBACK; /* Alt-P */
		   case 114: return HOT_KEY_RECORDING; /* Alt-R */
		   case 115: return HOT_KEY_SEED; /* Alt-S */
		   case 117: return HOT_KEY_UNDO; /* Alt-U */
		   case 110: return HOT_KEY_RESTART; /* Alt-N */
		   case 120: return HOT_KEY_QUIT; /* Alt-X */
		   default: return 27;
		 }
		 break;
	/* The standard function key block. */
        case KEY_UP: return 129;
        case KEY_DOWN: return 130;
        case KEY_LEFT: return 131;
        case KEY_RIGHT: return 132;
        case KEY_F(1): return 133; case KEY_F(2): return 134;
        case KEY_F(3): return 135; case KEY_F(4): return 136;
        case KEY_F(5): return 137; case KEY_F(6): return 138;
        case KEY_F(7): return 139; case KEY_F(8): return 140;
        case KEY_F(9): return 141; case KEY_F(10): return 142;
        case KEY_F(11): return 143; case KEY_F(12): return 144;
	/* Curses can't differentiate keypad numbers from cursor keys. Damn. */
	/* This will catch the alt-keys on 8-bit clean input streams... */
        case 240: return HOT_KEY_PLAYBACK; /* Alt-P */
        case 242: return HOT_KEY_RECORDING; /* Alt-R */
        case 243: return HOT_KEY_SEED; /* Alt-S */
        case 245: return HOT_KEY_UNDO; /* Alt-U */
        case 238: return HOT_KEY_RESTART; /* Alt-N */
        case 248: return HOT_KEY_QUIT; /* Alt-X */
      }

      /* Finally, if we're in full line mode (os_read), we might return
	 codes which aren't legal Z-machine keys but are used by the editor. */
      if (flag) return c;
    }
}

/*
 * unix_add_to_history
 *
 * Add the given string to the next available history buffer slot.  Commandeer
 * that slot if necessary using realloc.
 *
 */

static void unix_add_to_history(char *str)
{

    if (history_buffer[history_pointer] == NULL)
      history_buffer[history_pointer] = (char *) malloc(strlen(str) + 1);
    else
      history_buffer[history_pointer] =
	(char *) realloc(history_buffer[history_pointer], strlen(str) + 1);
    strcpy(history_buffer[history_pointer], str);
    history_pointer = ((history_pointer + 1) % MAX_HISTORY);
    history_frame = history_pointer; /* Reset user frame after each line */
}

/*
 * unix_history_back
 *
 * Copy last available string to str, if possible.  Return 1 if successful.
 *
 */

static int unix_history_back(char *str)
{

    history_frame--; if (history_frame==-1) history_frame = (MAX_HISTORY - 1);
    if ((history_frame == history_pointer) ||
	(history_buffer[history_frame] == NULL)) {
        beep(); history_frame = (history_frame + 1) % MAX_HISTORY;
	return 0;
    }
    strcpy(str, history_buffer[history_frame]);
    return 1;

}

/*
 * unix_history_forward
 *
 * Opposite of unix_history_back, and works in the same way.
 *
 */

static int unix_history_forward(char *str)
{

    history_frame = (history_frame + 1) % MAX_HISTORY;
    if ((history_frame == history_pointer) ||
	(history_buffer[history_frame] == NULL)) {
        beep(); history_frame--; if (history_frame == -1) history_frame =
							    (MAX_HISTORY - 1);
	return 0;
    }
    strcpy(str, history_buffer[history_frame]);
    return 1;

}

/*
 * os_read
 *
 * Read a line of input from the keyboard into a buffer. The buffer
 * may already be primed with some text. In this case, the "initial"
 * text is already displayed on the screen. After the input action
 * is complete, the function returns with the terminating key value.
 * The length of the input should not exceed "max" characters plus
 * an extra 0 terminator.
 *
 * Terminating keys are the return key (13) and all function keys
 * (see the specification of the Z-machine) which are accepted by
 * the is_terminator function. Mouse clicks behave like function
 * keys except that the mouse position is stored in global variables
 * "mouse_x" and "mouse_y" (top left coordinates are (1,1)).
 *
 * Furthermore, Frotz introduces some special terminating keys:
 *
 *     HOT_KEY_PLAYBACK (Alt-P)
 *     HOT_KEY_RECORD (Alt-R)
 *     HOT_KEY_SEED (Alt-S)
 *     HOT_KEY_UNDO (Alt-U)
 *     HOT_KEY_RESTART (Alt-N, "new game")
 *     HOT_KEY_QUIT (Alt-X, "exit game")
 *
 * If the timeout argument is not zero, the input gets interrupted
 * after timeout/10 seconds (and the return value is 0).
 *
 * The complete input line including the cursor must fit in "width"
 * screen units.
 *
 * The screen is not scrolled after the return key was pressed. The
 * cursor should be placed at the end of the input line when the
 * function returns.
 *
 */

int os_read (int max, char *buffer, int timeout, int width)
{
    int ch, scrpos, pos, y, x;
    char insert_flag = 0;

    scrpos = pos = strlen(buffer);

    unix_set_global_timeout(timeout);

    getyx(stdscr, y, x);

    do {
        refresh(); /* Shouldn't be necessary, but is, to print spaces */

        ch = unix_read_char(1);

	if ((ch == 127) && (scrpos)) {
	    mvdelch(y, --x);
	    pos--; scrpos--;
	    if (scrpos != pos)
	      memmove(&(buffer[scrpos]), &(buffer[scrpos+1]), pos-scrpos);
	}

	/* Left key */
	if ((ch == 131) && (scrpos)) {
	    scrpos--;
	    move(y, --x);
	}
	/* Right key */
	if ((ch == 132) && (scrpos < pos)) {
	    scrpos++;
	    move(y, ++x);
	}

	/* Home */
	if (ch == KEY_HOME) {
	    x -= scrpos; scrpos = 0;
	    move(y, x);
	}
	/* End */
	if (ch == KEY_END) {
	    x += (pos - scrpos); scrpos = pos;
	    move(y,x);
	}

	/* Insert */
	if (ch == KEY_IC)
	  insert_flag = (insert_flag ? 0 : 1);

	/* Up and down keys (history) */
	if (ch == 129) {
	    if (unix_history_back(buffer)) {
	      x -= scrpos;
	      move(y, x);
	      while (scrpos) {addch(' '); scrpos--;}
	      move(y, x);
	      addstr(buffer);
	      scrpos = pos = strlen(buffer);
	      x += scrpos;
	    }
	}
	if (ch == 130) {
	    if (unix_history_forward(buffer)) {
	      x -= scrpos;
	      move(y, x);
	      while(scrpos) {addch(' '); scrpos--;}
	      move(y, x);
	      addstr(buffer);
	      scrpos = pos = strlen(buffer);
	      x += scrpos;
	    }
	}

	/* Page up/down (passthrough as up/down arrows for beyond zork) */
	if (ch == KEY_PPAGE) ch = 129;
	if (ch == KEY_NPAGE) ch = 130;

	/* Escape */
	if (ch == 27) {
	    x -= scrpos;
	    move(y, x);
	    while (scrpos) {addch(' '); scrpos--;}
	    move(y, x); pos = 0;
	}

	/* ASCII printable */
	if ((ch >= 32) && (ch <= 126)) {
	    if (pos == scrpos) {
	        /* Append to end of buffer */
	        if ((pos < max) && (pos < width)) {
		    buffer[pos++] = (char) ch;
		    echochar(ch);
		    scrpos++; x++;
		} else beep();
	    }
	    else {
	        /* Insert/overwrite in middle of buffer */
	        if (insert_flag) {
		    memmove(&buffer[scrpos+1], &buffer[scrpos], pos-scrpos);
		    buffer[scrpos++] = (char) ch;
		    insch(ch);
		    pos++; x++; move(y, x);
		} else {
	            buffer[scrpos++] = (char) ch;
		    echochar(ch);
		    x++;
		}
	    }
	}
    } while (!is_terminator(ch));

    buffer[pos] = '\0';
    if (ch == 13) unix_add_to_history(buffer);
    return ch;

}/* os_read */

/*
 * os_read_char
 *
 * Read a single character from the keyboard (or a mouse click) and
 * return it. Input aborts after timeout/10 seconds.
 *
 */

int os_read_char (int timeout)
{
    int c;

    refresh();

    unix_set_global_timeout(timeout);

    /* Filter out frotz hot keys. */
    do {
      c = unix_read_char(0);
    } while (c >= HOT_KEY_MIN);

    return c;

}/* os_read_char */



/* ========================================================================== *\


	    S C R E E N   M A N I P U L A T I O N   R O U T I N E S


\* ========================================================================== */



/*
 * os_erase_area
 *
 * Fill a rectangular area of the screen with the current background
 * colour. Top left coordinates are (1,1). The cursor does not move.
 *
 */

void os_erase_area (int top, int left, int bottom, int right)
{
    int y, x, i, j;
    int saved_style;

    /* Catch the most common situation and do things the easy way */
#ifdef COLOR_SUPPORT
    if ((top == 1) && (bottom == h_screen_rows) &&
	(left == 1) && (right == h_screen_cols) && !(h_config & CONFIG_COLOUR))
      erase();
#else
    if ((top == 1) && (bottom == h_screen_rows) &&
	(left == 1) && (right == h_screen_cols))
      erase();
#endif
    else {
        /* Sigh... */
        getyx(stdscr, y, x);
	saved_style = current_text_style;
	os_set_text_style(0);
	top--; left--; bottom--; right--;
	for (i = top; i <= bottom; i++) {
	  move(i, left);
	  for (j = left; j <= right; j++)
	    addch(' ');
	}

	os_set_text_style(saved_style);
	move(y, x);
    }

    refresh();

}/* os_erase_area */

/*
 * os_scroll_area
 *
 * Scroll a rectangular area of the screen up (units > 0) or down
 * (units < 0) and fill the empty space with the current background
 * colour. Top left coordinates are (1,1). The cursor stays put.
 *
 */

static int old_scroll_top = 0;
static int old_scroll_bottom = 0;

void os_scroll_area (int top, int left, int bottom, int right, int units)
{
#ifdef COLOR_SUPPORT
    int y, x, i;
    int saved_style;
#endif

    if (units != 1) os_fatal("Can't Happen (os_scroll_area)"); /* FIXME */

    if (!((old_scroll_top == top) && (old_scroll_bottom == bottom))) {
        old_scroll_top = top; old_scroll_bottom = bottom;
        setscrreg(--top, --bottom);
    }
    scrollok(stdscr, TRUE);
    scroll(stdscr);
    scrollok(stdscr, FALSE);

#ifdef COLOR_SUPPORT
    if (h_flags & COLOUR_FLAG) {
        getyx(stdscr, y, x);
	move(old_scroll_bottom, 0);
	saved_style = current_text_style;
	os_set_text_style(0);
	for (i = 0; i <= right; i++) addch(' ');
	os_set_text_style(saved_style);
	move(y, x);
    }
#endif

}/* os_scroll_area */



/* ========================================================================== *\


		 I N I T I A L I S A T I O N   R O U T I N E S


\* ========================================================================== */



/*
 * os_process_arguments
 *
 * Handle command line switches. Some variables may be set to activate
 * special features of Frotz:
 *
 *     option_attribute_assignment
 *     option_attribute_testing
 *     option_context_lines
 *     option_object_locating
 *     option_object_movement
 *     option_left_margin
 *     option_right_margin
 *     option_piracy
 *     option_undo_slots
 *
 * The name of the story file is stored in "story_name".
 *
 */

void os_process_arguments (int argc, char *argv[])
{
    int c;

    /* Parse the options */

    do {

	c = getopt(argc, argv, "f:b:h:w:l:r:u:poOaAt");

	switch(c) {
	  case 'f': user_foreground_color = atoi(optarg);
	            if ((user_foreground_color < 2) || 
			(user_foreground_color > 9)) 
		      user_foreground_color = -1;
		    break;
	  case 'b': user_background_color = atoi(optarg);
	            if ((user_background_color < 2) ||
			(user_background_color > 9))
		      user_background_color = -1;
		    break;
          case 'h': user_screen_height = atoi(optarg); break;
	  case 'w': user_screen_width = atoi(optarg); break;
	  case 'l': option_left_margin = atoi(optarg); break;
	  case 'r': option_right_margin = atoi(optarg); break;
	  case 'u': option_undo_slots = atoi(optarg); break;
	  case 'p': euro_map = flatscii_map; break;
	  case 'o': option_object_movement = 1; break;
	  case 'O': option_object_locating = 1; break;
	  case 'a': option_attribute_assignment = 1; break;
	  case 'A': option_attribute_testing = 1; break;
	  case 't': user_tandy_bit = 1; break;
	}

    } while (c != EOF);

    if (optind != argc - 1) {
	puts (INFORMATION);
	exit (1);
    }

    story_name = argv[optind];

    if (euro_map == NULL) euro_map = iso_8859_1_map; /* Default */

}/* os_process_arguments */

/*
 * os_init_screen
 *
 * Initialise the IO interface. Prepare screen and other devices
 * (mouse, sound card). Set various OS depending story file header
 * entries:
 *
 *     h_config (aka flags 1)
 *     h_flags (aka flags 2)
 *     h_screen_cols (aka screen width in characters)
 *     h_screen_rows (aka screen height in lines)
 *     h_screen_width
 *     h_screen_height
 *     h_font_width
 *     h_font_height
 *     h_default_foreground
 *     h_default_background
 *     h_interpreter_number
 *     h_interpreter_version
 *     h_user_name (optional; not used by any game)
 *
 */

void os_init_screen (void)
{
#ifdef COLOR_SUPPORT
    int c;
#endif

    /*trace(TRACE_CALLS);*/
    curses_active = 1;          /* Let os_fatal know curses is running */
    initscr();                  /* Set up curses */
    cbreak();                   /* Raw input mode, no line processing */
    noecho();                   /* No input echo */
    nonl();                     /* No newline translation */
    intrflush(stdscr, TRUE);    /* Flush output on interrupt */
    keypad(stdscr, TRUE);       /* Enable the keypad and function keys */
    scrollok(stdscr, FALSE);    /* No scrolling unless explicitly asked for */

    if (h_version == V3 && user_tandy_bit != 0)
        h_config |= CONFIG_TANDY;

    if (h_version == V3)
	h_config |= CONFIG_SPLITSCREEN;

    if (h_version >= V4)
	h_config |= CONFIG_BOLDFACE | CONFIG_EMPHASIS | CONFIG_FIXED | CONFIG_TIMEDINPUT;

#ifndef SOUND_SUPPORT
    if (h_version >= V5)
      h_flags &= ~(GRAPHICS_FLAG | SOUND_FLAG | MOUSE_FLAG | MENU_FLAG);

    if (h_version == V3)
	h_flags &= ~OLD_SOUND_FLAG;
#else
    if (h_version >= V5)
      h_flags &= ~(GRAPHICS_FLAG | MOUSE_FLAG | MENU_FLAG);

    if ((h_version >= 5) && (h_flags & SOUND_FLAG)) {
      if (unix_init_sound())
	h_flags |= SOUND_FLAG;
      else
	h_flags &= ~SOUND_FLAG;
    }
    if ((h_version == 3) && (h_flags & OLD_SOUND_FLAG)) {
      if (unix_init_sound())
	h_flags |= OLD_SOUND_FLAG;
      else
	h_flags &= ~OLD_SOUND_FLAG;
    }
#endif

    if (h_version >= V5 && (h_flags & UNDO_FLAG))
        if (option_undo_slots == 0)
            h_flags &= ~UNDO_FLAG;

    getmaxyx(stdscr, h_screen_rows, h_screen_cols);

    if (user_screen_height != -1)
	h_screen_rows = user_screen_height;
    if (user_screen_width != -1)
	h_screen_cols = user_screen_width;

    h_screen_width = h_screen_cols;
    h_screen_height = h_screen_rows;

    h_font_width = 1;
    h_font_height = 1;

    h_interpreter_number = INTERP_DEC_20; /* We is a DECsystem-20 :) */
    h_interpreter_version = 'E';

#ifdef COLOR_SUPPORT
    if (has_colors()) {
        addstr("Do you want color?(Y/N)");
	do {
	    c = getch();
	} while ((c != 'n') && (c != 'N') && (c != 'y') && (c != 'Y'));
	if ((c == 'y') || (c == 'Y')) {
	    h_config |= CONFIG_COLOUR;
	    h_flags |= COLOUR_FLAG; /* FIXME: beyond zork handling? */
	    start_color();
	    if (user_foreground_color != -1)
	      h_default_foreground = user_foreground_color;
	    else
	      h_default_foreground = WHITE_COLOUR;
	    if (user_background_color != -1)
	      h_default_background = user_background_color;
	    else
	      h_default_background = BLUE_COLOUR;
	    os_set_colour(h_default_foreground, h_default_background);
	    os_erase_area(1, 1, h_screen_rows, h_screen_cols);
	}
	else
	    if (h_flags & COLOUR_FLAG) h_flags &= ~COLOUR_FLAG;
    }
    else
        if (h_flags & COLOUR_FLAG) h_flags &= ~COLOUR_FLAG;
#endif

    refresh();

}/* os_init_screen */

/*
 * os_reset_screen
 *
 * Reset the screen before the program ends.
 *
 */

void os_reset_screen (void)
{

    os_set_text_style(0);
    display_string("[Hit any key to exit.]");
    os_read_char(0);
    scrollok(stdscr, TRUE); scroll(stdscr);
    refresh(); endwin();

}/* os_reset_screen */

/*
 * os_fatal
 *
 * Display error message and stop interpreter.
 *
 */

void os_fatal (const char *s)
{

    if (curses_active) {
      display_string("\n\n");
      beep();
      os_set_text_style(BOLDFACE_STYLE);
      display_string("Fatal error: ");
      os_set_text_style(0);
      display_string(s);
      display_string("\n");
      os_reset_screen();
      exit(1);
    }

    fputs ("\nFatal error: ", stderr);
    fputs (s, stderr);
    fputs ("\n\n", stderr);

    exit (1);

}/* os_fatal */



/* ========================================================================== *\


			V A R I O U S   R O U T I N E S


\* ========================================================================== */



/*
 * os_more_prompt
 *
 * Display a MORE prompt, wait for a keypress and remove the MORE
 * prompt from the screen.
 *
 */

void os_more_prompt (void)
{
    int saved_style, saved_x, saved_y;

    /* Save some useful information */
    saved_style = current_text_style;
    getyx(stdscr, saved_y, saved_x);

    os_set_text_style(0);
    addstr("[MORE]");
    os_read_char(0);

    move(saved_y, saved_x);
    addstr("      ");
    move(saved_y, saved_x);
    os_set_text_style(saved_style);

}/* os_more_prompt */

/*
 * os_get_file_name
 *
 * Return the name of a file. Flag can be one of:
 *
 *    FILE_SAVE     - Save file
 *    FILE_RESTORE  - Restore file
 *    FILE_SCRIPT   - Transscript file
 *    FILE_RECORD   - Command file for recording
 *    FILE_PLAYBACK - Command file for playback
 *    FILE_SAVE_AUX - Save auxilary ("preferred settings") file
 *    FILE_LOAD_AUX - Load auxilary ("preferred settings") file
 *
 * The length of the file name is limited by MAX_FILE_NAME. Ideally
 * an interpreter should open a file requester to ask for the file
 * name. If it is unable to do that then it should use diplay_string
 * and display_new_line to prompt for a file name.
 *
 */

int os_get_file_name (char *file_name, char *default_name, int flag)
{

    display_string ("Enter a file name.\nDefault is \"");
    display_string (default_name);
    display_string ("\": ");

    /* Let's just do this the easy way... */
    echo();
    wgetnstr(stdscr, file_name, MAX_FILE_NAME);
    noecho();

    display_string ("\n");

    if (file_name[0] == 0)
	strcpy (file_name, default_name);

    return 1;

}/* os_get_file_name */

/*
 * os_message_start
 *
 * Prepare for printing a "debugging" message. The text of the message
 * will be passed to os_display_char. Possibly, the interface may want
 * to direct the message to an additional "messages" window. Otherwise,
 * it should display the message on the game screen -- or discard it if
 * the upper window is selected (ie. if cwin == UPPER_WINDOW).
 *
 */

static int message_saved_style;

void os_message_start (void)
{

    if (cwin == 0) {
        message_saved_style = current_text_style;
	os_set_text_style(0);

	display_string("\n [");
    } else
      dump_message = 1;

}/* os_message_start */

/*
 * os_message_end
 *
 * Stop printing a debugging message.
 *
 */

void os_message_end (void)
{

    if (cwin == 0) {
        display_string("]\n");
	
	os_set_text_style(message_saved_style);
    } else
      dump_message = 0;

}/* os_message_end */

