/*
 * input.c
 *
 * High level input functions
 *
 */

#include <ctype.h>
#include <string.h>
#include "frotz.h"

#define INPUT_BUFFER_SIZE 120

static unsigned char euro_tolower[] = {
    155, 156, 157, 155, 156, 157, 161, 162, 163, 164,
    165, 166, 164, 165, 169, 170, 171, 172, 173, 174,
    169, 170, 171, 172, 173, 174, 181, 182, 183, 184,
    185, 181, 182, 183, 184, 185, 191, 192, 193, 194,
    195, 191, 192, 193, 194, 195, 201, 201, 203, 203,
    205, 206, 207, 205, 206, 207, 211, 211, 213, 213,
    215, 216, 215, 216, 219, 220, 220, 222, 223
};

/*
 * is_terminator
 *
 * Check if the given key is an input terminator.
 *
 */

int is_terminator (int key)
{
    zword addr;
    zbyte c;

    if (key == 0 || key == 13 || (key >= HOT_KEY_MIN && key <= HOT_KEY_MAX))
	return 1;

    addr = h_terminating_keys;

    if (addr != 0 && ((key >= 129 && key <= 154) || (key >= 252 && key <= 254)))

	do {
	    LOW_BYTE (addr, c)
	    if (c == key || c == 255)
		return 1;
	    addr++;
	} while (c != 0);

    return 0;

}/* is_terminator */

/*
 * z_read
 *
 * Read a line of input with optional timeout.
 *
 */

void z_read (int argc, zword text, zword parse, zword timeout, zword routine)
{
    char buffer[INPUT_BUFFER_SIZE];
    zword addr;
    zbyte max_size;
    zbyte c;
    int continued;
    int read_size;
    int terminator;
    int i;

    continued = 0;

    /* Supply default arguments */

    if (argc < 3)
	timeout = 0;

    /* Set the input flag for a workaround in z_sound_effect */

    input_flag = 1;

    /* Calculate maximum input size */

    addr = text;

    LOW_BYTE (addr, max_size)

    if (h_version <= V4)
	max_size--;

    if (max_size >= INPUT_BUFFER_SIZE)
	max_size = INPUT_BUFFER_SIZE - 1;

    /* Calculate initial input size */

    if (h_version >= V5) {
	addr++;
	LOW_BYTE (addr, read_size)
    } else read_size = 0;

    /* Copy initial input to local buffer */

    for (i = 0; i < read_size; i++) {
	addr++;
	LOW_BYTE (addr, c)
	buffer[i] = c;
    }
    buffer[i] = 0;

continue_input:

    /* Redraw status line (V1 to V3 only) */

    if (h_version <= V3)
	z_show_status ();

    /* Read input from current input stream */

    terminator = read_line (max_size, buffer, timeout, continued);

    /* Handle timeouts */

    if (terminator == 0)

	if (call_interrupt (routine) == 0) {

	    /* The input line may have to be redrawn */

	    if (get_line_width (cwin) == get_units_left ())
		print_string (buffer);

	    /* Continue input */

	    continued = 1;
	    goto continue_input;

	} else buffer[0] = 0;

    /* Reset the line counter */

    reset_line_count ();

    /* Handle hot keys */

    if (terminator >= HOT_KEY_MIN && terminator <= HOT_KEY_MAX)

	if (hot_key (terminator) == 0) {

	    print_string ("Input continues...\n\n");
	    print_string (">");

	    /* Continue input */

	    continued = 1;
	    goto continue_input;

	} else return;

    /* Handle return key */

    if (terminator == 13)
	display_new_line ();

    /* Handle mouse clicks */

    if (terminator == 253 || terminator == 254) {
	z_storew (h_mouse_table, 1, mouse_x);
	z_storew (h_mouse_table, 2, mouse_y);
    }

    /* Perform save_undo for V1 to V4 games */

    if (h_version <= V4)
	z_save_undo (1);

    /* Copy local buffer back to dynamic memory */

    read_size = strlen (buffer);

    for (i = 0; i < read_size; i++) {

	if (isprint (buffer[i]))
	    c = tolower (buffer[i]);
	else
	    c = euro_tolower[(unsigned char) buffer[i] - EURO_MIN];

	z_storeb (text + (h_version <= V4 ? 1 : 2), i, c);
    }

    /* Add null character (V1-V4) or write input length into 2nd byte */

    if (h_version <= V4)
	z_storeb (text, read_size + 1, 0);
    else
	z_storeb (text, 1, read_size);

    /* Tokenise line if a token buffer is present */

    if (parse != 0)
	z_tokenise (2, text, parse, 0, 0);

    /* Return terminating key to Z-machine (V5+ only) */

    if (h_version >= V5)
	store (terminator);

}/* z_read */

/*
 * z_read_char
 *
 * Read one character with optional timeout.
 *
 */

void z_read_char (int argc, zword device, zword timeout, zword routine)
{
    int key;

    /* Supply default arguments */

    if (argc < 2)
	timeout = 0;

continue_input:

    /* Read input from command file or keyboard */

    key = read_char (timeout);

    /* Handle timeouts */

    if (key == 0 && call_interrupt (routine) == 0)
	goto continue_input;

    /* Reset the line counter */

    reset_line_count ();

    /* Handle mouse clicks */

    if (key == 253 || key == 254) {
	z_storew (h_mouse_table, 1, mouse_x);
	z_storew (h_mouse_table, 2, mouse_y);
    }

    /* Return key value to Z-machine */

    store (key);

}/* z_read_char */

/*
 * z_read_mouse
 *
 * Read the current mouse status and the status of the mouse buttons
 * and write them into a table.
 *
 */

void z_read_mouse (zword table)
{

    /* Not implemented */

}/* z_read_mouse */
