
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef CPM
# include <cpm.h>
#endif
#ifdef __MSDOS__
# include <dos.h>
#endif

/* Pacific and Hi-Tech C (for DOS & CP/M) aren't quite up to the latest
 * C standards... */

#ifndef EXIT_SUCCESS
# define EXIT_FAILURE 1
# define EXIT_SUCCESS 0
#endif

#ifndef SEEK_SET
# define SEEK_SET 0
# define SEEK_CUR 1
# define SEEK_END 2
#endif

#ifdef CPM
# define AV0 "UNGAC"
#else
# ifdef __PACIFIC__
#  define AV0 "UNGAC"
# else
#  define AV0 argv[0]
# endif
#endif

#define VERSION "0.1"

#ifdef CPM
typedef unsigned char boolean;
#else
typedef int boolean;
#endif

typedef unsigned char byte;
typedef unsigned short word;

/* GAC fixed addresses */
#define PUNCTUATION	0xA1E5	/* 8 possible phrase endings */
#define NOUNS		0xA51F	/* Words that GAC recognises as nouns */
#define ADVERBS		0xA521	/* Words that GAC recognises as adverbs */
#define OBJECTS		0xA523	/* Table of objects */
#define ROOMS		0xA525	/* Table of locations */
#define HPCS		0xA527	/* High-priority conditions */
#define LCS		0xA529	/* Local conditions */
#define LPCS		0xA52B	/* Low-priority conditions */
#define MESSAGES	0xA52D	/* Message text */
#define GRAPHICS	0xA52F	/* Room graphics */
#define TOKENS		0xA531	/* Tokens used to store all forms of text */
#define STARTROOM	0xA54D	/* Initial room */
#define VERBS		0xA54F	/* Words that GAC recognises as verbs */

/* The 8 bytes that should be at PUNCTUATION. UnGAC uses this as a magic number
 * to detect a GAC database. */
char punc_magic[8] = "\0 .,-!?:";

word width = 79;	/* Screen width */

int verbose   = 0;
int skipadverbs = 0;
int skipbeg   = 0;
int skipfont  = 0;
int skipgfx   = 0;
int skiphpcs  = 0;
int skiplcs   = 0;
int skiplpcs  = 0;
int skipmsgs  = 0;
int skipnouns = 0;
int skipobjs  = 0;
int skiprooms = 0;
int skipverbs = 0; 
FILE *outfile;
char *snafile;

/* A lot of these prototypes are just to shut Turbo C up */
#define indent() fprintf(outfile, "      ")
void short_roomdesc(word match);
void short_message(word match);
void short_object(word match);
void print_word(word addr, byte match);
void init(void);
void deinit(void);
word pop(void);
void push(word value);
void get_screen_width(void);
void check_magic(void);
void prelude(void);



void get_screen_width()
{
#ifdef __MSDOS__
	/* Get screen width from the BIOS data area. This won't work on 
	 * Apricots :-( */
	byte far *ptr = MK_FP(0x40, 0x4A);
	width = ptr[0] - 1;
#endif

#ifdef CPM
	static byte scbpb[] = { 0x1A, 0 };
	/* Check for CP/M Plus */
	if ((bdos(0x0c, 0) & 0xFF) >= 0x30)
	{
		/* Read screen width from the System Control Block */
		width = bdos(0x31, &scbpb) & 0xFF;
	}
#endif
}

#ifdef TINY_MODEL
FILE *fpsna;

#define CACHE_LEN   0x2000
#define CACHE_BASE  PUNCTUATION
byte cache[CACHE_LEN];

void init(void)
{
	fpsna = fopen(snafile, "rb");
	if (!fpsna)
	{
		perror(snafile);
		exit(EXIT_FAILURE);
	}
	fseek(fpsna, (CACHE_BASE - 0x3FE5), SEEK_SET);
	if (fread(cache, 1, CACHE_LEN, fpsna) < CACHE_LEN)
	{
		perror(snafile);
		fclose(fpsna);
		exit(EXIT_FAILURE);
	}
}

void deinit()
{
	fclose(fpsna);
}


byte peek1(word addr)
{
	int c;

	if (addr < 0x4000) return 0xFF;

	if (addr >= CACHE_BASE && addr < CACHE_BASE + CACHE_LEN)
	{
		return cache[addr - CACHE_BASE];
	}

	fseek(fpsna, addr - 0x3fe5, SEEK_SET);
	c = fgetc(fpsna);
	if (c == EOF)
	{
		perror(snafile);
		fclose(fpsna);
		exit(EXIT_FAILURE);
	}
	return c;
}

word peek2(word addr)
{
	int l, h;
	if (addr < 0x4000) return 0xFFFF;
	
	if (addr >= CACHE_BASE && addr < CACHE_BASE + CACHE_LEN - 1)
	{
		l = cache[addr - CACHE_BASE];
		h = cache[addr - CACHE_BASE + 1];
	}
	else
	{
		fseek(fpsna, addr - 0x3fe5, SEEK_SET);
		l = fgetc(fpsna);
		h = fgetc(fpsna);
	}
	if (l == EOF || h == EOF)
	{
		perror(snafile);
		fclose(fpsna);
		exit(EXIT_FAILURE);
	}
	return (h << 8) | l;	
}

#else
byte zxram[49179];

void init(void)
{
	FILE *fp = fopen(snafile, "rb");
	if (!fp)
	{
		perror(snafile);
		exit(EXIT_FAILURE);
	}
	if (fread(zxram, 1, sizeof(zxram), fp) < sizeof(zxram))
	{
		fprintf(stderr, "%s: Unexpected EOF\n", snafile);
		fclose(fp);
		exit(EXIT_FAILURE);
	}
}

byte peek1(word addr)
{
	if (addr < 0x4000) return 0xFF;
	return zxram[addr - 0x3FE5];
}

void deinit()
{

}

word peek2(word addr)
{
	if (addr < 0x4000) return 0xFF;
	return zxram[addr - 0x3FE5] + 256 * zxram[addr - 0x3FE4];
}

#endif

void prelude()
{
	fprintf(outfile, "; Generated by UnGAC v%s\n", VERSION);
}

void check_magic()
{
	word n;
	for (n = 0; n < 8; n++)
	{
		if (peek1(PUNCTUATION + n) != punc_magic[n])
		{
			fprintf(stderr, "Cannot identify this snapshot as "
					"a GAC adventure.\n");
			deinit();
			exit(EXIT_FAILURE);
		}
	}
}


word find_token(int token)
{
	word addr = peek2(TOKENS);
	while (token > 0)
	{
		byte len = peek1(addr++);
		addr += len;
		token--;
	}
	return addr + 1;
}


void show_message_len(word addr, byte len, word max)
{
	byte n, a, b;
	word w, token;
	byte top;
	byte x;

	x = 6;
	for (n = 0; n < len; n += 2)	
	{
		w = peek2(addr + n);

		top = (w >> 14) & 3;	
		if (top == 3)
		{
			a = (w >> 11) & 7;
			b = peek1(PUNCTUATION+a);
			if (b == 0)
			{
				return;	/* End of string */
			}
			for (a = (w & 0xFF); a > 0; a--)
			{
				fputc(b, outfile);
				x++;
				if (x == width) 
				{
					fputc('\n', outfile);
					indent();
					x = 6;
				}
				max--;
				if (!max) 
				{
					fprintf(outfile, "...");
					return;
				}
			}
			continue;
		}
		token = find_token(w & 0x7FF);
		do
		{
			switch(top)
			{
				case 0:	a = peek1(token);
					top = 1;
					break;
				case 1: a = peek1(token);
					if (a & 0x40) a |= 0x20;
					break;
				case 2:	a = peek1(token);
					break;
			}
			fputc(a & 0x7F, outfile);
			x++;
			if (x == width) 
			{
				fputc('\n', outfile);
				indent();
				x = 6;
			}
			max--;
			if (!max) 
			{
				fprintf(outfile, "...");
				return;
			}
			++token;
		} while ((a & 0x80) == 0);
		a = (w >> 11) & 7;
		b = peek1(PUNCTUATION+a);
		if (b == 0)
		{
			return;	/* End of string */
		}
		fputc(b, outfile);
		x++;
		if (x == width) 
		{
			fputc('\n', outfile);
			indent();
			x = 6;
		}
		max--;
		if (!max) 
		{
			fprintf(outfile, "...");
			return;
		}
	}
}

void show_message(word addr, byte len)
{
	show_message_len(addr, len, (word)-1);
}


void short_message(word match)
{
	byte id, len;
	word messages = peek2(MESSAGES);

	while ((id = peek1(messages)) != 0)
	{
		len = peek1(messages + 1);
		messages += 2;
		if (id != match)
		{
			messages += len;
			continue;
		}
		show_message_len(messages, len, 50);
		return;
	}
}



void short_object(word match)
{
	byte id, len;
	word objects = peek2(OBJECTS);

	while ((id = peek1(objects)) != 0)
	{
		len = peek1(objects + 1);
		objects += 2;
		if (id != match)
		{
			objects += len;
			continue;
		}
		show_message_len(objects + 3, len - 3, 50);
		return;
	}
}



void messages(void)
{
	byte id, len;
	word messages = peek2(MESSAGES);

	while ((id = peek1(messages)) != 0)
	{
		len = peek1(messages + 1);
		fprintf(outfile, "%04x: Message %d (length %d):\n",
			messages, id, len);
		messages += 2;
		indent();
		show_message(messages, len);
		fprintf(outfile, "\n");
		messages += len;
	}
}

void v_room(word dest)
{
	if (verbose && dest != 0xFFFF)
	{
		fprintf(outfile, "\t\t; ");
		short_roomdesc(dest);
	}
}


void v_word(word table, word id)
{
	if (verbose && id != 0xFFFF)
	{
		fprintf(outfile, "\t\t; ");
		print_word(table, id);
	}

}

void v_message(word dest)
{
	if (verbose && dest != 0xFFFF && dest != 0)
	{
		fprintf(outfile, "\t\t; ");
		short_message(dest);
	}
}


void v_object(word dest)
{
	if (verbose && dest != 0xFFFF && dest != 0)
	{
		fprintf(outfile, "\t\t; ");
		short_object(dest);
	}
}



void show_object(word addr, byte len)
{
	word dest;

	indent();
	fprintf(outfile,"Weight: %d\n", peek1(addr));
	indent();
	fprintf(outfile,"Starts at: %d", dest = peek2(addr+1));
	v_room(dest);
	fputc('\n', outfile);
	indent();
	show_message(addr+3, len-3);
	fprintf(outfile, "\n");
}


void objects(void)
{
	word id, len;
	word object = peek2(OBJECTS);

	while ((id = peek1(object)) != 0)
	{
		len = peek1(object+1);
		fprintf(outfile, "%04x: Object %d (length %d):\n",
				object, id, len);
		object += 2;
		show_object(object, len);
		object += len;
	}
}


void short_roomdesc(word match)
{
	word room = peek2(ROOMS);
	word id;
	word len, base;

	if (match == 0)
	{
		fprintf(outfile, "<nowhere>");
		return;
	}
	if (match >= 0x7F00)
	{
		fprintf(outfile, "<carried>");
		return;
	}
	while ((id = peek2(room)) != 0)
	{
		len = peek2(room+2);
		room += 4;
		if (id != match)
		{
			room += len;
			continue;
		}
		base = room;
		room += 2;	/* Graphics */
		while (peek1(room)) room += 3;	/* Exits */
		room++;
		show_message_len(room, len - (room-base), 40);
		return;
	}
	fprintf(outfile, "<unknown>");
}


void show_gfx(word gfx, int len)
{
	byte cmd;
	byte param[4];

	len = peek1(gfx++);
	indent();
	fprintf(outfile, "%d instructions:\n", len);
	while (len > 0)
	{
		cmd = peek1(gfx++); --len;
		indent();
		param[0] = peek1(gfx);
		param[1] = peek1(gfx+1);
		param[2] = peek1(gfx+2);
		param[3] = peek1(gfx+3);
		switch(cmd)
		{
			default: printf("OP%02x\n", cmd);
				 break;

			case 0x01:
				 printf("BORDER\t%d\n", param[0]);
				 gfx++;
				 break;
			case 0x02:
				 printf("PLOT\t%d,%d\n", param[0], param[1]);
				 gfx += 2;
				 break;
			case 0x03:
				 printf("ELLIPSE\t%d,%d %d,%d\n",
						 param[0], param[1],
						 param[2], param[3]);
				 gfx += 4;
				 break;
			case 0x04:
				 printf("FILL\t%d,%d\n", param[0], param[1]); 
				 gfx += 2;
				 break;
			case 0x05:
				 printf("BGFILL\t%d,%d\n", param[0], param[1]); 
				 gfx += 2;
				 break;
			case 0x06:
				 printf("SHADE\t%d,%d\n", param[0], param[1]); 
				 gfx += 2;
				 break;
			case 0x07:
				 printf("CALL\t%d\n", param[1] * 256 + param[0]);
				 gfx += 2;
				 break;
			case 0x08:
				 printf("RECT\t%d,%d %d,%d\n",
						 param[0], param[1],
						 param[2], param[3]);
				 gfx += 4;
				 break;
			case 0x09:
				 printf("LINE\t%d,%d %d,%d\n",
						 param[0], param[1],
						 param[2], param[3]);
				 gfx += 4;
				 break;
			case 0x10:
				 printf("INK\t%d\n", param[0]);
				 ++gfx;
				 break;
			case 0x11:
				 printf("PAPER\t%d\n", param[0]);
				 ++gfx;
				 break;
			case 0x12:
				 printf("BRIGHT\t%d\n", param[0]);
				 ++gfx;
				 break;
			case 0x13:
				 printf("FLASH\t%d\n", param[0]);
				 ++gfx;
				 break;
		}

	}
}

void show_room(word room, word len)
{
	word base = room;
	boolean ex = 0;
	word dest;
	byte dir;

	indent();
	fprintf(outfile, "Graphics=%04x\n", peek2(room));
	room += 2;

	while (peek1(room))
	{
		if (!ex)
		{
			indent();
			fprintf(outfile, "Exits:\n");
			ex = 1;
		}
		dir = peek1(room);
		dest = peek2(room + 1);
		indent();
		indent();
		fprintf(outfile, "%d to %d", dir, dest);
		v_room(dest);
		fprintf(outfile, "\n");
		room += 3;
	}
	++room;
	indent();
	show_message(room, len - (room-base));
	fprintf(outfile, "\n");
}

void rooms(void)
{
	word id, len;
	word room = peek2(ROOMS);

	while ((id = peek2(room)) != 0)
	{
		len = peek2(room + 2);
		fprintf(outfile, "%04x: Room %d (length %d):\n",
				room, id, len);
		room += 4;
		show_room(room, len);
		room += len;
	}
}



void graphics(void)
{
	word id, len;
	word room = peek2(GRAPHICS);

	while ((id = peek2(room)) != 0)
	{
		len = peek2(room + 2);
		fprintf(outfile, "%04x: Graphics %d (length %d):\n",
				room, id, len);
		room += 4;
		len -= 4;
		show_gfx(room, len);
		room += len;
	}
}




#define STACKMAX 64
static word stack[STACKMAX];
static word sp = 0;

void push(word w)
{
	if (sp < STACKMAX) stack[sp++] = w;
}

word pop()
{
	if (sp > 0) return stack[--sp];
	return 0;
}

word showcond(word cond)
{
	word s0, s1;
	byte bt;
	boolean first = 1;

	sp = 0;
	do
	{
		bt = peek1(cond);
		if (bt == 0) 
		{
			fprintf(outfile, "\n");
			return cond + 1;
		}
		if (first) 
		{
			fprintf(outfile,"%04x: ", cond);
			first = 0;
		}
		else
		{
			indent();
		}
		if (bt & 0x80)
		{
			s0 = (bt & 0x7F) * 256 + peek1(cond+1);
			fprintf(outfile,"PUSH\t%d\n", s0);
			push(s0);
			cond += 2;
			continue;		
		}
		cond++;
		switch(bt & 0x3F)
		{
			case 0: fprintf(outfile,"OP00\n");
				return cond;
			case 1: fprintf(outfile,"AND\n");
				s0 = pop();
				s1 = pop();
				push(s1 & s0);
				break;	
			case 2: fprintf(outfile,"OR\n");
				s0 = pop();
				s1 = pop();
				push(s1 | s0);
				break;	
			case 3: fprintf(outfile,"NOT\n");
				push (!pop());
				break;	
			case 4: fprintf(outfile,"XOR\n");
				s0 = pop();
				s1 = pop();
				push(s1 ^ s0);
				break;	
			case 5: fprintf(outfile,"HOLD\n");
				pop();
				break;
			case 6: fprintf(outfile,"GET");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);	
				break;
			case 7: fprintf(outfile,"DROP");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);	
				break;
			case 8:	fprintf(outfile, "SWAP");
				s0 = pop();
				s1 = pop();
				v_object(s0);
				if (verbose)
				{
					fprintf(outfile, "\n      \t");
					v_object(s1);	
						
				}
				fputc('\n', outfile);
				break;
			case 9:	fprintf(outfile, "TO");
				s0 = pop();
				s1 = pop();
				v_room(s0);
				if (verbose)
				{
					fprintf(outfile, "\n      \t");
					v_object(s1);	
				}
				fputc('\n', outfile);
				break;
			case 0x0A:
				fprintf(outfile, "OBJ");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);
				break;

			case 0x0B:
				s0 = pop();
				fprintf(outfile,"SET\n");
				break;
			case 0x0C:
				s0 = pop();
				fprintf(outfile,"RESE\n");
				break;
			case 0x0D:
				s0 = pop();
				fprintf(outfile,"SET?\n");
				break;
			case 0x0E:
				s0 = pop();
				fprintf(outfile,"RES?\n");
				break;
			case 0x0F:
				pop();
				pop();
				fprintf(outfile, "CSET\n");
				break;
			case 0x10:
				fprintf(outfile, "CTR\n");
				pop();
				push(0xFFFF);
				break;
			case 0x11:
				fprintf(outfile, "DECR\n");
				pop();
				break;
			case 0x12:
				fprintf(outfile, "INCR\n");
				pop();
				break;
			case 0x13:
				fprintf(outfile, "EQU?\n");
				pop();
				pop();
				break;
			case 0x14:
				fprintf(outfile, "DESC");
				s0 = pop();
				v_room(s0);
				fputc('\n', outfile);	
				break;	
			case 0x15:
				fprintf(outfile,"LOOK\n");
				break;
			case 0x16:
				fprintf(outfile,"MESS");
				s0 = pop();
				v_message(s0);
				fputc('\n', outfile);	
				break;	
			case 0x17:
				fprintf(outfile,"PRIN\n");
				s0 = pop();
				break;
			case 0x18:
				fprintf(outfile,"RAND\n");
				s0 = pop();
				push(rand() % s0);
				break;
			case 0x19:
				fprintf(outfile, "<\n");
				push(0);
				break;
			case 0x1A:
				fprintf(outfile, ">\n");
				push(0);
				break;
			case 0x1B:
				fprintf(outfile, "=\n");
				push(0);
				break;
			case 0x1C:
				fprintf(outfile, "SAVE\n");
				break;
			case 0x1D:
				fprintf(outfile, "LOAD\n");
				break;
			case 0x1E:
				fprintf(outfile, "HERE");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);	
				push(0);
				break;
			case 0x1F:
				fprintf(outfile, "CARR");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);	
				push(0);
				break;
			case 0x20:
				fprintf(outfile, "CARR");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);	
				push(0);
				break;
			case 0x21:
				s0 = pop();
				s1 = pop();
				push(s0+s1);
				fprintf(outfile, "+\n");
				break;
			case 0x22:
				s0 = pop();
				s1 = pop();
				push(s0-s1);
				fprintf(outfile, "-\n");
				break;
			case 0x23:
				push(0);
				fprintf(outfile, "TURN\n");
				break;
			case 0x24: 
				fprintf(outfile,"AT");
				s0 = pop();
				v_room(s0);
				fputc('\n', outfile);	
				push(0);
				break;
			case 0x25:
				fprintf(outfile, "BRIN");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);	
				push(0);
				break;
			case 0x26:
				fprintf(outfile, "FIND");
				s0 = pop();
				v_object(s0);
				fputc('\n', outfile);	
				push(0);
				break;
			case 0x27:
				fprintf(outfile, "IN");
				s0 = pop();
				s1 = pop();
				v_object(s0);
				if (verbose)
				{
					fprintf(outfile, "\n      \t");
					v_room(s1);	
				}
				fputc('\n', outfile);	
				push(0);
				break;
			/* 0x28 and 0x29 are no-ops */
			case 0x2A:
				fprintf(outfile,"OKAY\n");
				break;
			case 0x2B:
				fprintf(outfile,"WAIT\n");
				break;
			case 0x2C:
				fprintf(outfile,"QUIT\n");
				break;
			case 0x2D:
				fprintf(outfile,"EXIT\n");
				break;
			case 0x2E:
				fprintf(outfile,"ROOM\n");
				push(0xFFFF);
				break;
			case 0x2F:
				fprintf(outfile,"NOUN");
				s0 = pop();
				v_word(peek2(NOUNS), s0);
				fprintf(outfile,"\n");
				break;
			case 0x30:
				fprintf(outfile,"VERB");
				s0 = pop();
				v_word(VERBS, s0);
				fprintf(outfile,"\n");
				break;
			case 0x31:
				fprintf(outfile,"ADVE");
				s0 = pop();
				v_word(peek2(ADVERBS), s0);
				fprintf(outfile,"\n");
				break;
			case 0x32:
				fprintf(outfile, "GOTO");
				s0 = pop();
				v_room(s0);
				fputc('\n', outfile);	
				break;
			case 0x33:
				fprintf(outfile, "NO1\n");
				push(0xFFFF);
				break;
			case 0x34:
				fprintf(outfile, "NO2\n");
				push(0xFFFF);
				break;
			case 0x35:
				fprintf(outfile, "VBNO\n");
				push(0xFFFF);
				break;
			case 0x36:
				fprintf(outfile,"LIST");
				s0 = pop();
				v_room(s0);
				fputc('\n', outfile);	
				break;
			case 0x37:
				fprintf(outfile,"PICT\n");
				break;
			case 0x38:
				fprintf(outfile,"TEXT\n");
				break;
			case 0x39:
				fprintf(outfile, "CONN");
				s0 = pop();
				v_word(VERBS, s0);
				fputc('\n', outfile);	
				break;
			case 0x3A:
				fprintf(outfile, "WEIG");
				s0 = pop();
				v_object(s0);
				push(1);
				fputc('\n', outfile);	
				break;
			case 0x3B:
				fprintf(outfile,"WITH\n");
				push(0x7FFF);
				break;
			case 0x3C:
				fprintf(outfile,"STRE\n");
				pop();
				break;
			case 0x3D: 
				fprintf(outfile, "LF\n");
				break;
			case 0x3E:
				fprintf(outfile, "IF\n");
				pop();
				break;
			case 0x3F:
				fprintf(outfile,"END\n\n");
				first = 1;
				break;

			default:
				fprintf(outfile,"OP%02x\n", (bt & 0x3F));
				break;
				
		}
	}
	while (1);
	return cond;
}


void hpcs(void)
{
	word cond;

	fprintf(outfile, "High priority conditions:\n");
	cond = peek2(HPCS);
	cond = showcond(cond);	
}


void lcs(void)
{
	word cond, room;
	
	fprintf(outfile, "Local conditions:\n");
	cond = peek2(LCS);
	while ((room = peek2(cond)))
	{
		fprintf(outfile, "%04x: Room %d", cond, room);
		v_room(room);
		fprintf(outfile, "\n");
		cond = showcond(cond + 2);
	}
}


void lpcs(void)
{
	word cond;

	fprintf(outfile,"Low priority conditions:\n");
	cond = peek2(LPCS);
	cond = showcond(cond);	
}

void begin(void)
{
	fprintf(outfile, "\n%04x: First room is %d.\n",
			STARTROOM, peek2(STARTROOM));
	if (verbose)
	{
		indent();
		v_room(peek2(STARTROOM));
		fputc('\n', outfile);
	}
	fputc('\n', outfile);
}


void print_word(word addr, byte match)
{
	byte id, a;
	word token;

	while(1)
	{
		id = peek1(addr);
		if (id == 0) return;
		if (id != match)
		{
			addr += 3;
			continue;
		}
		addr++;
		token = find_token(0x7FF & peek2(addr));
		do
		{
			a = peek1(token);
			fputc(a & 0x7F, outfile);
			token++;
		} while ((a & 0x80) == 0);
		return;
	}
}


void dump_words(word addr)
{
	byte id, a;
	word token;

	while(1)
	{
		id = peek1(addr);
		if (id == 0) return;
		fprintf(outfile, "%04x: (%d) ", addr, id);
		addr++;
		token = find_token(0x7FF & peek2(addr));
		do
		{
			a = peek1(token);
			fputc(a & 0x7F, outfile);
			token++;
		} while ((a & 0x80) == 0);
		fprintf(outfile, "\n");
		addr += 2;
	}
}

void verbs(void)
{
	word addr = VERBS;
	fprintf(outfile, "Verbs:\n");
	dump_words(addr);
	fprintf(outfile, "\n");
}


void nouns(void)
{
	word addr = peek2(NOUNS);
	fprintf(outfile, "Nouns:\n");
	dump_words(addr);
	fprintf(outfile, "\n");
}


void adverbs(void)
{
	word addr = peek2(ADVERBS);
	fprintf(outfile, "Adverbs:\n");
	dump_words(addr);
	fprintf(outfile, "\n");
}

byte mirror(byte c)
{
	word n;
	byte o;

	for (n = 0, o = 0; n < 8; n++)
	{
		o = o << 1;
		o |= (c & 1);
		c = c >> 1;
	}
	return o;
}


void font(void)
{
	byte n, m, xbpos;
	word fontbase = peek2(23606) + 256;
	if (fontbase < 0x5B00) return;	/* Using ROM font */

	fprintf(outfile, "%04x: Custom font:\n"
			"#define font_width 768\n"
			"#define font_height 8\n"
			"\n"
			"static char font_bits[] = {\n",
			fontbase);
	xbpos = 0;
	for (m = 0; m < 8; m++) for (n = 0; n < 96; n++)
	{
		fprintf(outfile,"0x%2.2x", mirror(peek1(fontbase + (8 * n) + m)));
		if (++xbpos == 12)
		{
			if ((n==(95)) && (m==7)) 
				fprintf(outfile, "};\n");
			else fprintf(outfile, ",\n ");
			xbpos=0;
		}
		else	fprintf(outfile, ", ");
	}
}


void syntax(char *progname)
{
	fprintf(stderr, "Syntax: %s { options } SNA-file\n"
			"Options are:\n"
			"   -ofilename   Output to specified file\n"
			"   -sa          Skip adverbs\n"
			"   -sb          Skip beginning room\n"
			"   -sc          Skip local conditions\n"
			"   -sf          Skip custom font\n"
			"   -sg          Skip graphics\n"
			"   -sh          Skip high priority conditions\n"
			"   -sl          Skip low priority conditions\n"
			, progname);
	fprintf(stderr,
			"   -sm          Skip messages\n"
			"   -sn          Skip nouns\n"
			"   -so          Skip objects\n"
			"   -sr          Skip rooms\n"
			"   -sv          Skip verbs\n"
			"   -v           Verbose output\n"
	      	);
	exit(EXIT_FAILURE);
}

void setskip(char *arg, char c)
{
	if (isupper(c)) c = tolower(c);
	switch(c)
	{
		case 'a': skipadverbs = 1; break;
		case 'b': skipbeg = 1; break;
		case 'c': skiplcs = 1; break;
		case 'f': skipfont = 1; break;
		case 'g': skipgfx = 1; break;
		case 'h': skiphpcs = 1; break;
		case 'l': skiplpcs = 1; break;
		case 'm': skipmsgs = 1; break;
		case 'n': skipnouns = 1; break;
		case 'o': skipobjs = 1; break;
		case 'r': skiprooms = 1; break;
		case 'v': skipverbs = 1; break;
		default: syntax(arg);
	}
}

int main(int argc, char **argv)
{
	int n;
	boolean dashdash = 0;
	outfile = stdout;

	for (n = 1; n < argc; n++)
	{
		if (!strcmp(argv[n], "--"))
		{
			dashdash = 1; 
			continue;
		}
		if (argv[n][0] == '-' && (!dashdash))
		{
			switch(argv[n][1])
			{
				case 'o':
				case 'O':
					outfile = fopen(argv[n]+2, "w");
					if (!outfile)
					{
						perror(argv[n]+2);
						return EXIT_FAILURE;
					}
					break;
				case 'V':
				case 'v':
					verbose = 1;
					break;
				case 'S':
				case 's':
					setskip(AV0, argv[n][2]);
					break;
				case 'H':
				case 'h':
				case '?':
				default:
					syntax(AV0);
					break;
			}
			continue;	
		}
		if (snafile == NULL)
		{
			snafile = argv[n];
		}
	}

	if (snafile == NULL)
	{
		syntax(AV0);
		return EXIT_FAILURE;
	}
	get_screen_width();
	init();
	check_magic();
	prelude();
	if (!skipverbs) verbs();
	if (!skipnouns) nouns();
	if (!skipadverbs) adverbs();
	if (!skiphpcs)  hpcs();
	if (!skiplcs)   lcs();
	if (!skiplpcs)  lpcs(); 
	if (!skipobjs)  objects();
	if (!skiprooms) rooms();
	if (!skipmsgs)  messages();
	if (!skipgfx)   graphics();
	if (!skipbeg)   begin();
	if (!skipfont)  font();

	if (outfile != stdout)
	{
		fclose(outfile);
	}
	deinit();
	return EXIT_SUCCESS;
}
