"""
t3OBJS.py

Provides a class representing a static object block in a t3 image file.
"""

import struct, cStringIO
from t3Block import *
from t3DataHolder import *
from t3Metaclass import *

class T3Object:
	def __init__(self, id, size, data, image):
		self.i_id = id
		self.i_size = size
		self.o_data = data
		self.image = image
		
	def report(self):
		sl = []
		sl.append(str(self.i_id) + " " + self.o_data.report_deep(self.image) + "\n")
		return "".join(sl)
		
	def __str__(self):
		return "[tads-object: " + str(self.i_id) + "]"
		
	def __repr__(self):
		return self.__str__()


class OBJS_Block(Block):
	"""
	A static object block. Each such block in a file defines one or more
	root-set object instances of a given metaclass.
	"""
	name = "Static Object Block"

	def __init__(self, datastream, image, attribs):
		Block.__init__(self, datastream, image, attribs)
		self.read_block()
		
	def read_block(self):
		"""
		Reads the data included in the block.
		"""
		nobjs, mdti, flags = struct.unpack("<3H", self.datastream.read(6))
		self.data["i_numberOfObjectsInBlock"] = nobjs
		self.data["i_metaclassDependencyTableIndex"] = mdti
		metaclassblock = self.image.get_mcld()
		metaclassentrylist = metaclassblock.data["l_entries"]
		metaclass = metaclassentrylist[mdti]["s_name"]
		self.data["s_metaclass"] = metaclass
		self.data["i_flags"] = flags
		self.data["b_largeObjects"] = (flags == 1)
		self.data["b_transientObjects"] = (flags == 2)
		if self.data["b_largeObjects"]:
			objdatalengthf = 4
		else:
			objdatalengthf = 2
		self.data["l_objects"] = self.interpret_data(nobjs, objdatalengthf)
		
	def interpret_data(self, n, objdatalengthf):
		"""
		Should be called after reading the start of the block. Returns
		a list of objects instances.
		"""
		objects = []
		for i in xrange(n):
			i_id = struct.unpack("<I", self.datastream.read(4))[0]
			if objdatalengthf == 2:
				i_objsize = struct.unpack("<H", 
								self.datastream.read(objdatalengthf))[0]
			else:
				i_objsize = struct.unpack("<I",
								self.datastream.read(objdatalengthf))[0]
			data = self.datastream.read(i_objsize)
			scls = self.data["s_metaclass"].split("/")[0]
			cls = metaclassdict.get(scls, MetaUnknownClass)
			o_data = cls(data)
			objects.append(T3Object(i_id, i_objsize, o_data, self.image))
		return objects

	def report_data(self):
		sl = []
		sl.append("number of objects in block: " + 
					str(self.data["i_numberOfObjectsInBlock"]) + "\n")
		sl.append("metaclass: " + 
					self.data["s_metaclass"] + "\n")
		sl.append("large objects: " +
				str(self.data["b_largeObjects"]) + "\n")
		sl.append("transient objects: " + \
					str(self.data["b_transientObjects"]) + "\n\n")
		for object in self.data["l_objects"]:
			sl.append(object.report())
		return "".join(sl)