#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Services Pack
 *   
 *  tsp_delegate_obj.t
 *
 *  Creates a dynamic delegation object that an object can delegate work 
 *  to. This object's state is set to that of the delegating object, so 
 *  that the delegated object knows what the caling object knows.
 *
 *  This mechanism is useful in cases where TADS 3 language 'delegated' 
 *  keyword is not appropriate. For example, suppose we have the following
 *  objects defined:
 *  
 *  xObj: object
 *  {
 *      attr1  = 'a'
 *      attr2  = 'b'
 *      attr3  = 'c'
 *
 *      method1a() { delegated yObj.method1(); }
 *      method1b() { delegateTo(yObj,&method1); }
 *
 *      method2() { "hello xObj 2 <<attr1>>"; }
 *      method3() { "hello xObj 3 <<attr3>>"; }
 *  }
 *
 *  yObj: object
 *  {
 *      attr1  = 'd'
 *      attr2  = 'e'
 *      attr4  = 'f'
 *
 *      method1() { return method2(); }
 *      method2() { "goodbye yObj 2 <<attr1>>"; }
 *      method4() { "goodbye yObj 3 <<attr3>>"; }
 *  }
 *
 *  Suppose we want xObj to delegate its method1a() call to yObj. Using
 *  the TADS 3 'delegated' keyword the call chain would look like this:
 *  
 *      xObj.method1a() calls yObj.method1()
 *      yObj.method1()  calls xObj.method2()
 *
 *  and this displays:
 *  
 *      'TADS 3 DELEGATED : hello xObj 2 a'
 *
 *  Suppose we use the delegateTo() method of tsp_delegate_obj.t. The 
 *  call chain from xObj.method1b() would look like this:
 *
 *      xObj.method1b() calls yObj.method1()
 *      yObj.method1()  calls yObj.method2()
 *
 *  and this displays:
 *
 *      'DELEGATE TO METH : goodbye yObj 2 a'
 *
 *  In this case the method calls for delegatedTo() all remain within the 
 *  yObj delegation, which has been set to the same state as the calling 
 *  object. In addition the yObj delegation synchronizes the state
 *  of the calling object, if desired. 
 */

#include "tsp_delegate_obj.h"
#include "tsp_mod_registry.h"

RegisterModule

RequiresModule('tsp_inheritance_order.t')

class DelegateObj: object
{
    /*
     *  This is a simplified version of delegationMethod(), which
     *  can take either a delegation object or a list of delegation
     *  objects, defaults to no exclusion list, and to true on caller
     *  synchronization.
     */
    delegateTo(selfObj, delObj, method, [args])
    {
        if (dataType(delObj) == TypeObject)
            delObj = [delObj];

        return delegateMethod(selfObj, delObj, [], DelegateSynch, 
            method, args...);
    }

    delegateMethod(selfObj, dList, eList, synch, method, [args])
    {
        local delObj, val;

        /* create a delegation object */
        delObj = createDelegate(selfObj, dList, eList);

        /* run the method on the delegation object */
        val = delObj.(method)(args...);

        /*
         *  Synchronize this object with the delegation object's 
         *  defined state.
         */
        if (DelegateSynch)
            synchObjToDefStateOf(selfObj, delObj, eList);

        if (synch == DelegateRet)
            /*
             *  Return a list consisting of the delegate 
             *  object and the return value from the method call.
             */
            return [delObj, val];
        else
            /* return the value of the method call */
            return val;
    }

    /* 
     * Creates a delegate object that is an instance of 
     *  the objects listed in dList, but whose state-related 
     *  properties are the same as this object, with the exclusion
     *  of properties listed in eList.
     */
    createDelegate(selfObj, dList, eList)
    {
        local delObj;

        /*
         *  Create a clone of the original objects. This object
         *  will have the identical set of properties defined in
         *  the originals.
         */
        delObj  = TadsObject.createInstanceOf(dList...);

        /*
         *  Create a TSP single-inheritance model clone of the 
         *  delegation object. This means that we have cloned all 
         *  of the superclasses of the delegation object, and 
         *  restructured the delegation object's inheritance 
         *  structure to be single-inheritance.
         */
        delObj = delObj.createTspSimClone();

        synchObjToInherStateOf(delObj, selfObj, eList);

        return delObj;
    }

    /*
     *  Sets this object's state to that of the inherited state of obj, 
     *  with the exception of those attributes listed in exclustion List.
     */
    synchObjToInherStateOf(obj1, obj2, eList)
    {
        local pList;

        pList   = obj2.getInherPropList();
        pList   -= eList;

        setStateTo(obj1, obj2, pList);
    }

    /*
     *  Sets this object's state to that of the defined state of obj, 
     *  with the exception of those attributes listed in exclustion List.
     */
    synchObjToDefStateOf(obj1, obj2, eList)
    {
        local pList;

        pList   = obj2.getPropList();
        pList   -= eList;

        setStateTo(obj1, obj2, pList);
    }

    /*
     *  Sets the properties of obj1 that are listed in pList
     *  to those of obj2.
     */
    setStateTo(obj1, obj2, pList)
    {
        pList.forEach(new function(value) 
        {
            local prop;

            prop = value;

            /*
             *  We only update object attributes that are not 
             *  the same as that of obj. 
             */
            if (obj2.propType(prop) not in (TypeDString, TypeCode,
                TypeNativeCode)
                && obj1.(prop) != obj2.(prop))
            {
                obj1.(prop) = obj2.(prop);
            }
         });
    }
}

modify TadsObject
{
    /*
     *  Builds a list of all properties inherited by this object.
     */
    getInherPropList()
    {
        local ioList, pList, pVec = new Vector(50);

        ioList = getInheritanceOrder();
        ioList.forEach(new function(val)
        {
            pList = val.getPropList();
            pVec.appendUnique(pList);
         });

        return pVec.toList();
    }

    /*
     *  Returns a linear representation of the object's 
     *  inheritance order from the object's flattened 
     *  inheritance tree.
     */
    getInheritanceOrder()
    {
        local v = new Vector(40);

        // create superclass tree for this object
        scTreeIter(v);

        // remove any empty elements
        v.removeElement(nil);

        // return the result
        return v;
    }
    scTreeIter(vector)
    {
      // clear any existing instances of self in the vector
        vector.applyAll({x: x == self ? nil : x});  
        
        // add self to the vector
        vector.append(self);

        // insert the superclasses of this element into the queue
        foreach (local o in getSuperclassList())
            o.scTreeIter(vector);
    }
}