#charset "us-ascii"

/* 
 *  Copyright (c) 2008 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Rulebooks library extension
 *
 *  association.t
 */

#include "rulebooks.h"

/*
 *  An assoication is a base class from which
 *  all other rulebooks  datatypes derive.
 *
 *  By default an Association does nothing.
 */
class Association: PreinitObject
{
    /*
     *  Indicates how this Association is to control 
     *  the evaluation of its associations.
     */
    evalCtl         = nil

    /*
     *  Indicates how this Association is to handle 
     *  the return of its exec() invocation. 
     */
    execCtl         = nil

    /* 
     *  This property is a list of the datatypes associations. Depending 
     *  on the datatypes class this list may represent relationships, 
     *  prerequisites, or members. The result of evaluation of these 
     *  associations determines whether the datatypes exec() method is 
     *  invoked.
     */
    assoc       = []

    /*
     *  An association that is evaluated by the class' 
     *  default exec() method.
     */
    execAssoc   = nil

    /* 
     *  This property indicates that this datatype is an association 
     *  of location. 
     */
    location    = nil

    /*
     *  This method is meant to be called when an author desires to 
     *  evaluate a datatype. Evaluation means that the datatypes 
     *  associations will be processed (evaluated), and if indicated, 
     *  the datatypes exec() method invoked.     
     */
    eval()
    {
        if (check())
            return execMain();

        return nil;
    }

    /*
     *  This method is executed when the datatype is evaluated, via its 
     *  eval() method. This method processes the evaluation of the datatypes 
     *  associations and returns true or nil to indicate whether the datatypes 
     *  exec() method should be invoked.     
     */
    check()
    {
        local checkRet, evalRet;

        /*
         *  Set the default return value for this method.
         */
        switch(evalCtl)
        {
            /*
             *  By default a "do all" will return this
             *  once processing of the associations is done.
             *  This is the value for an empty assoc list as
             *  well. The exec() WILL be invoked.
             */
            case EvalCtlDoAll:
                checkRet = true;
                break;

            /*
             *  By default an "all true" will return this
             *  once processing of the associations is done.
             *  This is the value for an empty assoc list as
             *  well. The exec() WILL be invoked.
             */
            case EvalCtlAllTrue:
                checkRet = true;
                break;

            /*
             *  By default an "all nil" will return this
             *  once processing of the associations is done.
             *  This is the value for an empty assoc list as
             *  well. The exec() WILL be invoked.
             */
            case EvalCtlAllNil:
                checkRet = true;
                break;

            /*
             *  By default an "exist true" will return this
             *  once processing of the associations is done.
             *  This is the value for an empty assoc list as
             *  well. The exec() will NOT be invoked.
             */
            case EvalCtlExistTrue:
                checkRet = nil;
                break;

            /*
             *  By default an "exist nil" will return this
             *  once processing of the associations is done.
             *  This is the value for an empty assoc list as
             *  well. The exec() will NOT be invoked.
             */
            case EvalCtlExistNil:
                checkRet = nil;
                break;

            /*
             *  By default any other value will return nil
             *  without any processing being done. The exec()
             *  will NOT be invoked.
             */
            default:
                return nil;
        }

        /*
         *  Evaluate associations
         */
        __checkLoop:
        foreach (local elm in assoc)
        {
            evalRet = elm.eval();

            switch(evalCtl)
            {
                /*
                 *  For a "do all", continue with
                 *  the next association.
                 */
                case EvalCtlDoAll:
                    continue;

                /*
                 *  For an "all true" we break out of the
                 *  evaluation processing as soon as an 
                 *  association returns NIL. The check() 
                 *  will return NIL. The exec() method
                 *  will NOT be invoked.
                 */
                case EvalCtlAllTrue:
                    if (evalRet == nil)
                    {
                        checkRet = nil;
                        break __checkLoop;
                    }
                    break;

                /*
                 *  For an "all NIL" we break out of the
                 *  evaluation processing as soon as an 
                 *  association returns TRUE. The check() 
                 *  will return NIL. The exec() method 
                 *  will NOT be invoked.
                 */
                case EvalCtlAllNil:
                    if (evalRet == true)
                    {
                        checkRet = nil;
                        break __checkLoop;
                    }
                    break;

                /*
                 *  For an "exist true" we break out of the
                 *  evaluation processing as soon as an 
                 *  association returns TRUE. The check() 
                 *  will return TRUE. The exec() method 
                 *  WILL be invoked.
                 */
                case EvalCtlExistTrue:
                    if (evalRet == true)
                    {
                        checkRet = true;
                        break __checkLoop;
                    }
                    break;

                /*
                 *  For an "exist NIL" we break out of the
                 *  evaluation processing as soon as an 
                 *  association returns NIL. The check() 
                 *  will return TRUE. The exec() method 
                 *  WILL be invoked.
                 */
                case EvalCtlExistNil:
                    if (evalRet == nil)
                    {
                        checkRet = true;
                        break __checkLoop;
                    }
                    break;

                /*
                 *  By default, continue with
                 *  the next association.
                 */
                default:
                    continue;
            }
        }

        return checkRet; 
    }

    /*
     *  This method invokes the exec() method and then 
     *  determines how to handle its return based on the 
     *  association's execCtl property.
     */
    execMain()
    {
        local ret; 
        
        ret = exec();
        
        switch(execCtl)
        {
            /*
             *  Return the exec() return value to the caller.
             */
            case ExecCtlRetVal:
                return ret;

            /* 
             *  Return nil to the caller.
             */
            case ExecCtlRetNil:
                return nil;

            /*
             *  Return true to the caller.
             */
            case ExecCtlRetTrue:
                return true;

            /*
             *  By default, return nil to the caller.
             */
            default:
                return nil;
        }
    }

    /*
     *  This method is invoked only when the datatypes check() 
     *  method returns true. 
     *
     *  By default this method returns nil.
     */
    exec()
    {
        if (execAssoc)
            return (execAssoc).eval();
    
        return nil;
    }

    /*
     *  Appends the value obj to the datatypes associations. If 
     *  afterObj is supplied then obj is appended to the datatypes 
     *  associations after afterObj. If afterObj is not found in the 
     *  datatypes associations then obj is appended to the end of 
     *  datatypes associations.
     */
    append(obj, [afterObj])
    {
        obj.location = self;
        if (afterObj.length())
        {
            local idx;
            idx = assoc.indexOf(afterObj.car());
            if (idx)
            {
                assoc = assoc.insertAt(idx + 1, obj);
            }
            else
                assoc = assoc.append(obj);
       }
        else
            assoc = assoc.append(obj);
    }

    /* 
     *  Returns true if obj is one of the datatypes associations. 
     *  Otherwise nil is returned.
     */
    hasAssoc(obj)
    {
        return assoc.indexOf(obj) != nil;
    }

    /* 
     *  Returns the number of associations for this datatype.
     */
    length()
    {
        return assoc.length();
    }

    /* 
     *  Prepends the value obj to the datatypes associations. If 
     *  beforeObj is supplied then obj is prepended to the datatypes 
     *  associations before beforeObj. If beforeObj is not found in 
     *  the datatypes associations then obj is prepended to the end 
     *  of datatypes associations.
     */
    prepend(obj, [beforeObj])
    {
        obj.location = self;
        if (beforeObj.length())
        {
            local idx;
            idx = assoc.indexOf(beforeObj.car());
            if (idx)
            {
                assoc = assoc.insertAt(idx, obj);
            }
            else
                assoc = assoc.prepend(obj);
        }
        else
            assoc = assoc.prepend(obj);
    }

    /* 
     *  Removes obj from this datatypes associations. If obj 
     *  is not one of the datatypes associations the method 
     *  does nothing.
     */
    removeAssoc(obj)
    {
        if (assoc.indexOf(obj) && obj.location == self)
            obj.location = nil;
        assoc -= obj;
    }

    /*
     *  If we have a location, add this object
     *  to the location's associations list. 
     */
    execute()
    {
        if (location)
            location.append(self);
    }
}