#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Structure.t
 *
 *  Provides a mechanism for analyzing the structure of an object. 
 */

#include "proteus.h"
 
modify Object
{
    /*----------------------------------------------------------------
     *  STRUCTURE LISTS
     *----------------------------------------------------------------
     */
    inherOrderIter(sVec, func)
    {
        sVec.applyAll(new function(val) {
            local ret;

            // clear any existing instances of self in the vector
            switch(dataType(val))
            {
                case TypeList:
                    val[1] == self ? ret = nil : ret = val;
                    break;

                case TypeObject:
                    val == self ? ret = nil : ret = val;
                    break;

                default:
                    ret = val;
            }

            return ret;
        });
        
        // add self to the vector
        func(self, getPropList());
        
        // insert the superclasses of this element into the queue
        foreach (local o in getSuperclassList())
            o.inherOrderIter(sVec, func);
    }

    /*
     *  Returns a structure list for the object in the format of 
     *  [ [ obj [ prop-list ], [ superclass1 [ prop-list ]], ... ]
     *
     *  Valid argument forms:
     *  
     *      getInherStrucList()
     *      getInherStrucList(func)
     *      getInherStrucList(suppress)
     *      getInherStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     */
    getInherStrucList([cond])
    {
        local sVec, func, suppress;

        func = cond.car();
        sVec = new Vector(40);
        
        if (cond.length() == 1 
            && dataType(cond[1]) is in (TypeNil, TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 
            && dataType(cond[2]) is in (TypeNil, TypeTrue))
            suppress = cond[2];
        
        inherOrderIter(sVec, new function(obj, propList) { 
            if (func == nil)
            {
                if (propList.length() != 0 || !suppress)
                    sVec = sVec.append([obj, propList]);
            }
            else 
            {
                local pVec = new Vector(50);
                
                foreach (local prop in propList)
                    if (func(obj, prop))
                        pVec = pVec.appendUnique([prop]);

                if (pVec.length() != 0 || !suppress)
                    sVec = sVec.append([obj, pVec.toList()]);
            }
        });

        /* remove any empty elements */
        sVec.removeElement(nil);
        
        return sVec.toList();
    }
        
    /*
     *  Returns a structure list.
     *
     *  These property lists only contain properties for the superclass
     *  object from which the property is inherited.
     *
     *  Valid argument forms:
     *  
     *      getInherDefStrucList()
     *      getInherDefStrucList(func)
     *      getInherDefStrucList(suppress)
     *      getInherDefStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     *
     *  If an argument is passed it should be a function of the form 
     *  func(obj, prop) that returns either true or nil. If the function
     *  returns true the object / property is added to the list,
     *  otherwise it is not.
     */
    getInherDefStrucList([cond])
    {
        local func = cond.car();
        local suppress;
        
        if (cond.length() == 1 && dataType(cond[1]) is in (TypeNil,
            TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 && dataType(cond[2]) is in (TypeNil,
            TypeTrue))
            suppress = cond[2];
        
        return getInherStrucList(new function(obj, prop) {
            if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        }, suppress);
    }
    
    /*
     *  Returns a structure list that only contains properties that are
     *  mutable by the program (i.e. not TypeDString, TypeCode,
     *  TypeNativeCode.) 
     *
     *  Furthermore, these lists only contain properties for 
     *  the superclass object from which the property is 
     *  inherited.
     *
     *  Valid argument forms:
     *  
     *      getInherStateStrucList()
     *      getInherStateStrucList(func)
     *      getInherStateStrucList(suppress)
     *      getInherStateStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     */
    getInherStateStrucList([cond])
    {
        local func = cond.car();
        local suppress;
        
        if (cond.length() == 1 && dataType(cond[1]) is in (TypeNil,
            TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 && dataType(cond[2]) is in (TypeNil,
            TypeTrue))
            suppress = cond[2];
        
        return getInherStrucList(new function(obj, prop) {
            if (obj.propType(prop) is in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (obj.propType(prop) is in (TypeNil, nil)
                && dataType(prop) == TypeProp)
                return nil;
            else if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        }, suppress);
    }
        
    /*
     *  Returns a structure list that only contains properties that are
     *  associated with actions (i.e. TypeDString, TypeCode,
     *  TypeNativeCode prefixed with dobjFor() or iobjFor() prefixes.) 
     *
     *  Valid argument forms:
     *  
     *      getInherActStrucList()
     *      getInherActStrucList(func)
     *      getInherActStrucList(suppress)
     *      getInherActStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     */
    getInherActStrucList([cond])
    {
        local func = cond.car();
        local suppress;
        
        if (cond.length() == 1 && dataType(cond[1]) is in (TypeNil,
            TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 && dataType(cond[2]) is in (TypeNil,
            TypeTrue))
            suppress = cond[2];
        
        return getInherStrucList(new function(obj, prop) {
            local str;

            str = String.toSString(prop);

            if (obj.propType(prop) not in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (!str.startsWith('&preCondDobj')
                && !str.startsWith('&preCondIobj')
                && !str.startsWith('&verifyDobj')
                && !str.startsWith('&verifyIobj')
                && !str.startsWith('&remapDobj')
                && !str.startsWith('&remapIobj')
                && !str.startsWith('&checkDobj')
                && !str.startsWith('&checkIobj')
                && !str.startsWith('&actionDobj')
                && !str.startsWith('&actionIobj'))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        }, suppress);
    }
        
    /*
     *  Returns a structure list that only contains properties that are
     *  associated with actions (i.e. TYP_DSTRING, TypeCode,
     *  TypeNativeCode prefixed with dobjFor() or iobjFor() prefixes.) 
     *
     *  Furthermore, these lists only contain properties for the superclass
     *  object from which the property is inherited.
     *
     *  Valid argument forms:
     *  
     *      getInherDefActStrucList()
     *      getInherDefActStrucList(func)
     *      getInherDefActStrucList(suppress)
     *      getInherDefActStrucList(func, suppress)
     *
     *  If suppress is true then object/propList sub-elements are 
     *  not added to the list when the propList is empty.
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the object / property sub-element is added to 
     *  the list, otherwise it is not.
     */
    getInherDefActStrucList([cond])
    {
        local func = cond.car();
        local suppress;
        
        if (cond.length() == 1 && dataType(cond[1]) is in (TypeNil,
            TypeTrue))
        {
            suppress = cond[1];
            func = nil;
        }
        else if (cond.length() > 1 && dataType(cond[2]) is in (TypeNil,
            TypeTrue))
            suppress = cond[2];
        
        return getInherStrucList(new function(obj, prop) {
            local str;

            str = String.toSString(prop);

            if (obj.propType(prop) not in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (!str.startsWith('&preCondDobj')
                && !str.startsWith('&preCondIobj')
                && !str.startsWith('&verifyDobj')
                && !str.startsWith('&verifyIobj')
                && !str.startsWith('&remapDobj')
                && !str.startsWith('&remapIobj')
                && !str.startsWith('&checkDobj')
                && !str.startsWith('&checkIobj')
                && !str.startsWith('&actionDobj')
                && !str.startsWith('&actionIobj'))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        }, suppress);
    }

    /*
     *----------------------------------------------------------------
     *  CLASS LISTS
     *----------------------------------------------------------------
     */

    /*
     *  Method returns a list of the object and the traits that make
     *  up this object's structure. 
     *
     *  If an argument is passed it should be a
     *  function of the form func(obj, propList) that returns either
     *  true or nil. If the function returns true the object is added
     *  to the classes list, otherwise it is not.
     *  
     *  Valid argument forms:
     *  
     *      getInherSuperclassList()
     *      getInherSuperclassList(func)
     *
     *  The callback function func should be of the form func(obj, 
     *  propList) that returns either true or nil. If the function 
     *  returns true the object is added to the list, otherwise it 
     *  is not.
     */
    getInherSuperclassList([cond])
    {
        local sVec, func;
        
        func = cond.car();
        sVec = new Vector(40);
                
        inherOrderIter(sVec, new function(obj, propList) {
            if (func == nil || func(obj, propList))
                sVec = sVec.appendUnique([obj]);
        });
        
        /* remove any empty elements */
        sVec.removeElement(nil);
        
        return sVec.toList();
    }

    /*
     *----------------------------------------------------------------
     *  PROPERTY LISTS
     *----------------------------------------------------------------
     */
        
    /*
     *  Method returns a list of the properties that make up this
     *  object's structure. 
     *
     *  Valid argument forms:
     *  
     *      getInherPropList()
     *      getInherPropList(func)
     *
     *  The callback function func should be of the form func(obj, 
     *  prop) that returns either true or nil. If the function 
     *  returns true the property is added to the list, otherwise it 
     *  is not.
     */
    getInherPropList([cond])
    {
        local sVec, func;

        func = cond.car();
        sVec = new Vector(1000);
        
        inherOrderIter(sVec, new function(obj, propList) {
            if (func == nil)
                sVec = sVec.appendUnique(propList); 
            else foreach (local prop in propList)
                    if (func(obj, prop))
                        sVec = sVec.appendUnique([prop]);
        });
        
        /* remove any empty elements */
        sVec.removeElement(nil);
        
        return sVec.toList();
    }
    
    /*
     *  Returns a property list.
     *
     *  The property list only contains properties for the superclass
     *  object from which the property is inherited.
     *
     *  If an argument is passed it should be a function of the form 
     *  func(obj, prop) that returns either true or nil. If the function
     *  returns true the object / property is added to the list,
     *  otherwise it is not.
     */
    getInherDefPropList([cond])
    {
        local func = cond.car();
        
        return getInherPropList(new function(obj, prop) {
            if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        });
    } 

    /*
     *  Returns a property list.
     *
     *  The property list only contains state properties for 
     *  the superclass object from which the property is inherited.
     *
     *  If an argument is passed it should be a function of the form 
     *  func(obj, prop) that returns either true or nil. If the function
     *  returns true the object / property is added to the list,
     *  otherwise it is not.
     */
    getInherStatePropList([cond])
    {
        local func = cond.car();

        return getInherPropList(new function(obj, prop) {
            if (obj.propType(prop) is in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (obj.propType(prop) is in (TypeNil, nil)
                && dataType(prop) == TypeProp)
                return nil;
            else if (obj != self.propDefined(prop, PropDefGetClass))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        });
    }  
    
    /*
     *  Returns a property list.
     *
     *  The property list only contains action properties for 
     *  the superclass object from which the property is inherited.
     *
     *  If an argument is passed it should be a function of the form 
     *  func(obj, prop) that returns either true or nil. If the function
     *  returns true the object / property is added to the list,
     *  otherwise it is not.
     */
    getInherActPropList([cond])
    {
        local func = cond.car();
        
        return getInherPropList(new function(obj, prop) {
            local str;

            str = String.toSString(prop);

            if (obj.propType(prop) not in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (!str.startsWith('&preCondDobj')
                && !str.startsWith('&preCondIobj')
                && !str.startsWith('&verifyDobj')
                && !str.startsWith('&verifyIobj')
                && !str.startsWith('&remapDobj')
                && !str.startsWith('&remapIobj')
                && !str.startsWith('&checkDobj')
                && !str.startsWith('&checkIobj')
                && !str.startsWith('&actionDobj')
                && !str.startsWith('&actionIobj'))
                return nil;
            else if (func == nil)
                return true;
            else return func(obj, prop);
        });
    }            
}