#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  ObjRef.t
 *
 *  Provides a mechanism for giving each object a unique object reference tag
 */

#include "proteus.h"

/*
 *  This preinit object assigns an object reference tag to 
 *  all objects and classes derived from TadsObject that have
 *  been defined before preinit().
 */
objRefTagger: PreinitObject
{
    /* execute this preinit after the symbol table has been built */
    execBeforeMe = [symbolTablePreinit]

    // object reference counter
    objRefCnt           = 0

    assignObjRef(obj)
    {
        /* convert the objRefCnt to hexidecimal */
        obj.objRefTag = 'tads#' + toString(objRefCnt, 16).toLower();

        /* increment the reference counter */
        objRefCnt++;
    }
    execute()
    {
        local o;

        /* reset the object reference counter */
        objRefCnt = 0;

        o = firstObj(TadsObject, ObjAll);
        while (o != nil)
        {
            assignObjRef(o);
            o = nextObj(o, TadsObject, ObjAll);
        }
    }
}

/*
 *  These modifications provide a means for objects
 *  to be 'named' in derivations of the Object class.
 */
modify Object
{    
    /* 
     *  Object class objects do not have object reference tags. Only
     *  TadsObject class does, so we return nil.
     */
    getObjRefTag() { return nil; }

   /*
     *  Returns a string representation of the object symbol
     *  or, if dynamically-created, the object's symbolic ancestor,
     *  and its object reference tag.
     */
    getObjTagAndSym([args])
    {
        local str = '', pList, val = '', len;

        if (isDynamOrAnon())
            pList = getFirstSymAncestorList();
        else
            pList = [] + self;

        len = pList.length();
        for (local i = 1; i <= len; ++i)
        {
            val += gSymbols.getSString(pList[i]);
            if (i < len)
                val += ', ';
        }

        if (isDynamOrAnon())
            str += ('<u>' + val + '</u>');
        else
            str += val;
       
       return str;
    }

    /*
     *  The meta-command reference determines whether the 
     *  object reference tag or object symbol should be used
     *  and prefixes the reference with either 'tag' or 'sym'
     *  as appropriate.
     */
    getMetaCmdReference()
    {
        if (isDynamOrAnon())
            return getObjTagAndSym();
        else
            return 'sym ' + getObjTagAndSym();
    }
}

/*
 *  These modifications provide a means for objects
 *  to be 'tagged' in derivations of the TadsObject class.
 */
modify TadsObject
{
    objRefTag = nil

    /*
     *  If the object is dynamically created, and it executes the
     *  base constructor for TadsObject, then we tag it.
     */
    construct([args])
    {
        objRefTagger.assignObjRef(self);
    }
    /*
     *  Use this method to retrieve reference tags. If the object
     *  hasn't been tagged then we tag it before returning the tag.
     */
    getObjRefTag()
    {
        local tag;

        // if the object isn't tagged, tag it
        if (objRefTag == nil)
            objRefTagger.assignObjRef(self);
        else
        {
            //if (self != objRefTag.getRefTagObj())
            tag = getRefTagObjList();

            // if the object shares the tag with others, retag it
            if (tag.length() > 1)
                objRefTagger.assignObjRef(self);
        }

        return objRefTag;
    }

    getRefTagObjList()
    {
        local lst = [];

        // include TadsObject as part of our search
        if (objRefTag.toLower() == TadsObject.objRefTag)
            lst += TadsObject;

        /* loop through all objects of class TadsObject */
        for (local o = firstObj(TadsObject, ObjAll); 
            o != nil; o = nextObj(o, TadsObject, ObjAll))
            /* if tag matches the object reference tag return the object */
            if (o.objRefTag.toLower() == objRefTag.toLower())
                lst += o;

        return lst;
    }

    /*
     *  Returns a string representation of the object symbol
     *  or, if dynamically-created, the object's static prototype's
     *  symbol, and its object reference tag.
     */
    getObjTagAndSym([args])
    {
        local str = '', pList, scList, len, val = '', emph;
        local noSuppressTag, useTag;

        noSuppressTag = args.car();

        if (isDynamOrAnon() || isUnnamedInternal() || noSuppressTag)
            useTag = true;

        if (useTag)
            str += getObjRefTag();

        scList = getSuperclassList();

        if (isDynamOrAnon())
            pList = getFirstSymAncestorList();
        else
            pList = [] + self;

        len = pList.length();
        for (local i = 1; i <= len; ++i)
        {
            emph = nil;

            if (scList.length() != 0 && scList[i] != pList[i]
            && self != pList[i])
                emph = true;

            if (emph)
                val += '<u>';

            val += gSymbols.getSString(pList[i]);

            if (emph)
                val += '</u>';

            if (i < len)
                val += ', ';
        }

        if (useTag)
            str += ' (';

        str += val;

        if (useTag && isUnnamedInternal())
        {
            local obj, val;
            str += ' *';
            obj = getObjModList()[1];
            val = gSymbols.getSString(obj);
            str += val;
            str += '*';
        }

        if (useTag)
            str += ')';

        return str;
    }

    /*
     *  The meta-command reference determines whether the 
     *  object reference tag or object symbol should be used
     *  and prefixes the reference with either 'tag' or 'sym'
     *  as appropriate.
     */
    getMetaCmdReference()
    {
            return 'tag ' + getObjRefTag();
    }
}

/*
 *  This function returns the object corresponding
 *  to the tag provided. If no object matches the 
 *  tag then it returns nil.
 */
modify String
{
    getRefTagObj()
    {
        // include TadsObject as part of our search
        if (toLower() == TadsObject.objRefTag)
            return TadsObject;

        /* loop through all objects of class TadsObject */
        for (local o = firstObj(TadsObject, ObjAll); 
            o != nil; o = nextObj(o, TadsObject, ObjAll))
            /* if tag matches the object reference tag return the object */
            if (o.objRefTag.toLower() == toLower())
                return o;

        /* no objects match tag, return nil */
        return nil;
    }
}