#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Snapshot.t
 *
 *  Provides a mechanism for saving and restoring an object's state.
 */

#include "proteus.h"
 
class SnapshotLU: ProteusLU
{
    myValuesProp = &mySnapshots
    myAddProp       = &addSnapshot
    
    instantiate()
    {
        return new SnapshotLU();
    }

    construct([args])
    {
        inherited();
        gSnapshots = self;
    }

    addSnapshot(obj, image)
    {
        storeValue(obj, image);
    }
    
    getSnapshot(obj)
    {
        local snapshotList, len;
        
        if (!hasMoreSnapshots(obj))
            throw new NoFurtherSnapshotException(self);
        
        snapshotList = getValues(obj);
    
        len = snapshotList.length();
        return snapshotList[len];
    }
    
    hasMoreSnapshots(obj)
    {
        return hasMoreValues(obj);
    }
    
    /*
     *  Removes the last snapshot from the archive.
     */
    removeSnapshot(obj, image)
    {
        local ndx, snapshotList;
        
        if (!hasMoreSnapshots(obj))
            throw new NoFurtherSnapshotException(self);
            
        snapshotList = getValues(obj);
        ndx = snapshotList.indexOf(image);
        
        if (ndx == nil)
            throw new NotFoundSnapshotException(obj);
                
        removeValue(obj, image);
    }
}

/*
 *  An Exception class for snapshot.
 */
class SnapshotException: ProteusException
{
    construct() { inherited(); }    
}

class NoFurtherSnapshotException: SnapshotException
{
    construct(obj)
    {
        local objName;
        
        objName = String.toSString(obj);
        
        exceptionMessage = '[No further snapshots
                    available for \"' + objName + '\".]';
                    
        inherited();
    }
}

class NotFoundSnapshotException: SnapshotException
{
    construct(obj)
    {
        local objName;
        
        objName = String.toSString(obj);
        
        exceptionMessage = '[Snapshot image for \"' + objName + '\" 
            not found.]';
            
        inherited();
    }
}

modify TadsObject
{
    addSnapshot(image)
    {
        gSnapshots.addSnapshot(self, image);
    }
    
    /*
     *  Returns the last snapshot taken for this object.
     */
    getSnapshot()
    {
        return gSnapshots.getSnapshot(self);
    }
    
    hasMoreSnapshots()
    {
        return gSnapshots.hasMoreSnapshots(self);
    }
    
    /*
     *  Removes the last snapshot from the archive.
     */
    removeSnapshot(image)
    {
        return gSnapshots.removeSnapshot(self, image);
    }
    
    /*
     *  Saves the state of the object as a createClone() object 
     *  stored in the snapshot list archive.
     */
    saveState()
    {
        local stateImage;
    
        stateImage = createClone();
        addSnapshot(stateImage);
    }
    
    /*
     *  Saves the state of all objects of kind TadsObject except for
     *  those listed in the bypass list (including this object). The
     *  flgs parameter indicates whether the firstObj() / nextObj() loop
     *  includes only objects, classes, or both.
     */
    saveStateAll(cls, flgs, [bypass])
    {
        local obj;
        
        /* add this object to the bypass list */
        bypass = bypass.append(self);
        
        obj = firstObj(cls, flgs);
        while (obj != nil)
        {
            if (!obj.ofKindOrUU(ProteusLU))
            {
                
                if (bypass.indexOf(obj) == nil)
                    obj.saveState();
            }
            
            obj = nextObj(obj, cls, flgs);
        }
    }
    
    /*
     *  Restores the state of this object from the last snapshot taken.
     *  After the restore it removes the snapshot from the archive.
     */
    restoreState([args])
    {
        local stateImage;
        
        stateImage = getSnapshot();
        
        setStateFromSnapshot(stateImage, args...); 
            
        removeSnapshot(stateImage);
    }
    
    /*
     *  Restores the state of all objects of kind TadsObject except for
     *  those listed in the bypass list (including this object). The
     *  flgs parameter indicates whether the firstObj() / nextObj() loop
     *  includes only objects, classes, or both.
     */
    restoreStateAll(cls, flgs, [bypass])
    {
        local obj;

        /* add this object to the bypass list */
        bypass = bypass.append(self);
        
        obj = firstObj(cls, flgs);
        while (obj != nil)
        {
            if (!obj.ofKindOrUU(ProteusLU)
            && !obj.ofKindOrUU(symbolTablePreinit)
            && obj.hasMoreSnapshots())
            {
                
                if (bypass.indexOf(obj) == nil)
                    obj.restoreState(true);
            }
            
            obj = nextObj(obj, cls, flgs);
        }
    }
    
    /*
     *  The method has two approaches. For a single object we do a 
     *  restore of its entire state property list. For 'all' we 
     *  only restore the directly defined state properties. This
     *  speeds up the restore by a factor of 40!
     */
    setStateFromSnapshot(snapshot, [args])
    {
        local prop, propList = [];
        local snapshotval, selfval;
    
        /* extract the property list for obj structure */
        if (args.car())
            propList = getPropList();
        else
            propList = getInherPropList();  

        propList = propList.subset(new function(x) {
            return (propType(x) 
                not in ( TypeDString, TypeCode, TypeNativeCode ));
        });
        
        /*
         *  Set any attribute (that can be set) to its corresponding
         *  value on the cloning object. 
         */
        foreach (prop in propList)
        {
            selfval     = self.(prop);
            snapshotval = snapshot.(prop);
                
            if (selfval != snapshotval)
                self.(prop) = snapshotval;
        }
    }

    /*
     *  Sets thisObj state from obj. Only properties that are not
     *  of TypeDString, TypeCode, or TypeNativeCode are mutable. 
     *
     *  If setToNil is 'true' then any properties defined by thisObj
     *  that are not defined by obj will be set to nil. 
     *
     *  Any property that are in the bypass list will be excluded from
     *  the setting of state from obj to thisObj.
     */
    setStateFrom(obj, setToNil, [bypass])
    {
        local prop, thisPropList = [], objPropList = [];
        local objval, selfval;
    
        /* extract the property list for obj structure */
        objPropList = obj.getInherPropList();  

        objPropList = objPropList.subset(new function(x) {
            return (obj.propType(x) 
                not in ( TypeDString, TypeCode, TypeNativeCode ));
        });
        
        /*
         *  Set any attribute (that can be set) to its corresponding
         *  value on the cloning object. 
         */
        foreach (prop in objPropList)
        {
            if (bypass.indexOf(prop))
                continue;
               
            selfval = self.(prop);
            objval  = obj.(prop);
                
            if (selfval != objval)
                self.(prop) = objval;
        }
         
        /*
         *  Set any attribute (that can be set) that is not on the
         *  cloned object to nil.
         */
        if (setToNil)
        {
            /* extract the property list for this structure */
            thisPropList = getInherPropList();

            thisPropList = thisPropList.subset(new function(x) {
                return (self.propType(x) 
                    not in ( TypeDString, TypeCode, TypeNativeCode ));
            });
            
            thisPropList -= objPropList;
    
            foreach (prop in thisPropList)
            {
                if (bypass.indexOf(prop))
                    continue;
                    
                self.(prop) = nil;
            }
        }
    }
}