#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  TravelActor.t
 *
 *  THE TRAVEL GRAPH
 *
 *      The travel graph is a multi-graph of weighted directed
 *      edges, that may contain self-loops. To simplify the automation
 *      process only travel properties that resolve to property types 
 *      of _object_ are used to build adjacent vertex and adjacent edge
 *      lists.
 *
 *      Multiple edges between a given pair of vertices are flattened
 *      out by resolving the traversal between the two vertices as 
 *      passable if any of the edges is found to be passable, and 
 *      impassable otherwise. This allows for simple paths to be 
 *      computed by dijkstra's shortest paths algorithm.
 *
 *      Weighting of edges between vertices is based upon the 
 *      actor's ability to pass all of the obstacles that may 
 *      lie along the given edge. Impassable edges are given a 
 *      weight of INFINITY, while edges for which passage is 
 *      possible are give a weight of 1. The edge weights are 
 *      maintained through all path computations for a given 
 *      travelDest (see below), and re-adjudicated only when the
 *      next travelDest is being considered. 
 *
 *      If a passable edge has been found for the two vertices 
 *      then the adjacent vertex itself is examined to determine
 *      if it is accessible to the actor. Inaccessible vertices 
 *      are temporarily removed from path computations. After a 
 *      new path has been computed the bias against the 
 *      inaccessible vertex is removed.
 *
 *  TRAVEL DESTINATIONS
 *
 *      TravelActor class allows the author to set travel destinations as 
 *      points of travelling preference along the graph. Travel 
 *      destinations are used to form the destinations of simple paths 
 *      computed by dijkstra's algorithm. 
 *
 *      When a path to a travel destination has been found to be 
 *      impassable, the runActor() will remove the impassable 
 *      vertex or edge from its new path computation and attempt to 
 *      traverse this new path. This strategy is continued until 
 *      either a path to the travel destination has been successfully 
 *      traversed; or all possible paths to the travel destination are 
 *      exhausted. If all paths have been exhausted the travel 
 *      destination is abandonned and we proceed to calculate a path 
 *      from the actor's location to the next travel destination in the list.
 *
 *  EXAMPLE
 *
 *          setTravelDest( [loc1, loc2, loc3] );
 *
 *      The above statement tells TravelActor to set its travel 
 *      destination list to [loc1, loc2, loc3]. The actor's 
 *      travelDest will be set to loc1 and a path computed 
 *      from the actor's location.
 *
 *      The actor will then follow the computed path (travel list) 
 *      until it completes it or finds an edge or vertex it cannot
 *      pass. If this is the case then a new path to its destination
 *      is computed, eliminating the obstructing edge or vertex from
 *      the calculation. 
 *
 *      After the actor has arrived at loc1 (or it has been 
 *      determined that no path to the destination exists)
 *      a new path starting from the actor's location to loc2
 *      will be computed and the process repeated.
 */

#include "adv3.h"

enum StartOfList, EndOfList, TravelCmdIssued;
enum TravelEdgeSuccess, TravelLinkSuccess, TravelNodeSuccess, TravelDestSuccess;
enum TravelEdgeBlocked, TravelLinkBlocked, TravelNodeBlocked, TravelDestBlocked;
enum TravelEdgeFailure, TravelLinkFailure, TravelNodeFailure, TravelDestFailure;

/* TravelActor states */
enum GetNextTravelDest, GetNextTravelLink;
enum ComputingPath, StopActor;

class TravelActor: object
{
    isActive                = nil
    chooseEdgeSuccess       = nil
    myDaemon                = nil
    travelState             = nil
    
    path                    = nil
    travelDestList          = []
    travelLinkList          = []
    destIndex               = 0
    travelIndex             = 0

    travelDest              = nil
    travelLink              = nil
    travelEdge              = nil  
    travelValue             = nil

    /*
     *  Method should return true if the actor is familiar with
     *  the location; otherwise it should return nil.
     */
    locationIsFamiliar(loc) 
    { 
        if (locationMemory.isKeyPresent(loc))
            return locationMemory[loc];
        else
            return true; 
    }

    /*
     *  A LookupTable keyed to [loc, dir] with values of
     *  TravelEdgeBlocked, TravelEdgeFailure, TravelEdgeSuccess.
     *  This serves as the actor's memory of the result of 
     *  having traversed the loc/dir previously.
     */
    travelEdgeMemory = nil

    /*
     *  Set the loc/dir key in the travelEdgeMemory 
     *  to the value of val. Val should be TravelEdgeBlocked,
     *  TravelEdgeFailure, or TravelEdgeSuccess.
     */
    setTravelEdgeKnown(loc, dir, val)
    {
        travelEdgeMemory[[loc,dir]] = val;
    }

    /*
     *  Returns the value kept in travelEdgeMemory
     *  for the key loc/dir. If the key isn't present
     *  the method will return nil.
     */
    travelEdgeKnown(loc, dir)
    {
        return travelEdgeMemory[[loc, dir]];
    }

    /*
     *  Returns true if travel is known to be blocked 
     *  or fails to transport the actor to the location
     *  it believes to be the destination of the loc/dir.
     *  Otherwise the method returns nil.
     */
    travelBlocked(loc, dir)
    {
        local val = travelEdgeKnown(loc, dir);

        if (val is in (TravelEdgeBlocked, TravelEdgeFailure))
            return true;
        else
            return nil;
    }

    /*
     *  Starts the actor by setting up its travel destination
     *  list, sets the actor to active status, and sets up a
     *  daemon to execute runActor().
     *
     *  This should be used once at the very beginning of a game to
     *  kick off the actor. The preTurn indicates whether runActor()
     *  should be executed once so that it runs during the current
     *  turn.
     */
    initiateTravelActor(preTurn, [travelDestList])
    {
        setTravelDestList(travelDestList...);
        startActor();
        if (preTurn)
            runActor();
        if (myDaemon == nil)
            myDaemon = new Daemon(self, &runActor, 1);
    }

    /*
     *  Starts the actor by setting it to active status. This 
     *  method should be used when we want to 'restart' an actor 
     *  that was 'paused' temporarily by stopActor().
     */
    startActor()
    {
    	isActive = true;
    }

    /*
     *  This should be used when we want to remove the daemon
     *  from the eventManager, rather than pause the actor 
     *  momentarily.
     */
    terminateTravelActor()
    {
        myDaemon.removeEvent();
        myDaemon = nil;
        stopActor();
    }

    /*
     *  This method should be used to 'pause' the actor, without
     *  disrupting the actor's daemon. Use startActor() and 
     *  stopActor() when you want to 'pause' and 'restart' an
     *  actor during doTravelScript().
     */
    stopActor()
    {
        isActive = nil;
    }

    /*
     *  Sets the travel destination list and 
     *  initializes the destination index. Sets
     *  the actor's travel state to get the next
     *  travel destination.
     */
    setTravelDestList([lst])
    {
        travelDestList  = lst;
        destIndex       = 0;
        travelState     = GetNextTravelDest;
    }

    /*
     *  The main method of the travel actor. This method is called
     *  by the event manager every turn and handles all the aspects
     *  of the actor's movement.
     */
    runActor()
    {
        local dirName, toks, travelLog, nextNodeIsDestination;

        if (!isActive) 
            return;

        travelLog = new Vector(10);

        travelStateLoop: while (travelState != nil)
        {
            switch (travelState)
            {
                case GetNextTravelDest:
                    ++destIndex;
                    if (destIndex > travelDestList.length())
                    {
                        travelValue     = EndOfList;
                        travelLog       += buildTravelActionInfo(travelValue);
                        travelState     = StopActor;
                        break;
                    }
                    travelDest = travelDestList[destIndex];
                    if (location == travelDest)
                    {
                        travelValue     = TravelDestSuccess;
                        travelLog       += buildTravelActionInfo(travelValue);
                        travelState     = GetNextTravelDest;
                        break;
                    }

                case ComputingPath:
                    if (travelLink 
                    && travelDest 
                    && travelLink.nextTravelNode == travelDest)
                        nextNodeIsDestination = true;
                    else
                        nextNodeIsDestination = nil;

                    computePath();

                    if (travelLinkList.length() == 0)
                    {
                        if (travelState == GetNextTravelDest
                        || nextNodeIsDestination)
                            travelValue = TravelDestBlocked;
                        else
                            travelValue = TravelNodeBlocked;
                        travelLog       += buildTravelActionInfo(travelValue);
                        travelState     = GetNextTravelDest;
                        break;
                    }

                case GetNextTravelLink:
                    if (location == travelDest)
                    {
                        travelValue     = TravelDestSuccess;
                        travelLog       += buildTravelActionInfo(travelValue);
                        travelState     = GetNextTravelDest;
                        break;
                    }
                    if (travelLink == nil)
                    {
                        travelValue     = StartOfList;
                        travelLog       += buildTravelActionInfo(travelValue);
                    }
                    else if (location == travelLink.nextTravelNode)
                    { 
                        if (travelValue == TravelEdgeBlocked)
                            travelValue = TravelLinkSuccess;
                        else
                            travelValue = TravelEdgeSuccess;
                        travelLog       += buildTravelActionInfo(travelValue);
                    }
                    else if (location == travelLink.currTravelNode)
                    {
                        travelValue     = TravelEdgeBlocked;
                        travelLog       += buildTravelActionInfo(travelValue);
                        setTravelEdgeKnown(location, travelEdge, 
                            TravelEdgeBlocked);
                        --travelIndex;
                    }
                    else
                    {
                        travelValue     = TravelEdgeFailure;
                        travelLog       += buildTravelActionInfo(travelValue);
                        setTravelEdgeKnown(travelLink.currTravelNode, 
                            travelEdge, TravelEdgeFailure);
                        travelState     = ComputingPath;
                        break;
                    }
                    ++travelIndex;
                    travelLink = travelLinkList[travelIndex];
                    travelEdge = getNextTravelEdge();
                    if (travelEdge == nil)
                    {
                        travelValue     = TravelLinkBlocked;
                        travelLog       += buildTravelActionInfo(travelValue);
                        travelState     = ComputingPath;
                        break;
                    }
                    /* get the name for this travel direction */
                    dirName = travelEdge.name;

                    /*
                     *  Tokenize the direction name for use as 
                     *  an actor command.
                     */
                    toks = cmdTokenizer.tokenize(dirName);

                    /*
                     *  Tell the actor to move in this direction
                     *  the next time its schedule is run.
                     */
                    addPendingCommand(true, self, toks);
                    travelValue = TravelCmdIssued;
                    travelLog   += buildTravelActionInfo(travelValue);
                    travelState = GetNextTravelLink;
                    break travelStateLoop;

                case StopActor:
                    stopActor();

                default:
                    travelState = nil;
            }
        }

        dbgTravelActionInfo(travelLog);

        doTravelScript(travelLog);
    }

    /*
     *  Compute a dijkstra path based on the actor
     *  location and the current travel destination.
     */
    computePath()
    {
        path = new DijkstraPath(self, location, 
            travelDestList[destIndex]);

        /*
         *  Set the travel list to the shortest
         *  path entry list.
         */
        travelLinkList          = path.travelLinkList;

        /* reset the track list index */
        travelIndex             = 0;

        travelLink              = nil;
        travelEdge              = nil;
    }

    /*
     *  Chooses the travel edge based on actor knowledge. If the
     *  actor is set to choose the first successful travel edge
     *  then the search terminates with that; otherwise the list
     *  of candidate edges is reduced by any that are known to
     *  have failed or have been blocked and one randomly selected.
     */
    getNextTravelEdge()
    {
        local edgeList, val;
        
        edgeList = travelLinkList[travelIndex].travelEdgeList;

        foreach (local edge in edgeList)
        {
            val = travelEdgeKnown(location, edge);

            if (chooseEdgeSuccess
            && val == TravelEdgeSuccess)
                return edge;

            if (val is in (TravelEdgeBlocked, TravelEdgeFailure))
                edgeList -= edge;
        }

        if (edgeList.length() == 0)
            return nil;
        else
            return rand(edgeList);
    }

    /*
     *  Builds a TravelActionInfo object for this stage in the
     *  travel process. TravelActionInfo objects are passed to
     *  doTravelScript() to provide an author with more 
     *  information regarding the decision process of choosing
     *  a vertex and edge.
     */
    buildTravelActionInfo(v)
    {
        local d, c, n, e;

        d = travelDest;

        if (travelLink)
        {
            c = travelLink.currTravelNode;
            n = travelLink.nextTravelNode;
        }

        e = travelEdge;

        return new TravelActionInfo(v, d, c, n, e, location);
    }

    /*
     *  TravelActors should override this method to perform
     *  special actions prior to the next actor command 
     *  execution cycle.
     */
    doTravelScript(travelLog) {}

    /*
     *  Provides debugging information for the travelLog 
     *  created by runActor().
     */
    dbgTravelActionInfo(travelLog)
    {
        local v, d, c, n, e, l;

        if (libGlobal.dbgTravelActor == nil)
            return;

        foreach (local travelActionInfo in travelLog)
        {
            v = travelActionInfo.value;
            d = travelActionInfo.travelDest;
            c = travelActionInfo.currTravelNode;
            n = travelActionInfo.nextTravelNode;
            e = travelActionInfo.travelEdge;
            l = travelActionInfo.location;

            "\nTravel Log Value :&emsp&fpmsp<B>";
            
            switch(v)
            {
                case TravelCmdIssued:
                    "TravelCmdIssued ";
                    break;

                case StartOfList:
                    "StartOfList ";
                    break;

                case EndOfList:
                    "EndOfList ";
                    break;

                case TravelDestSuccess:
                    "TravelDestSuccess ";
                    break;

                case TravelNodeSuccess:
                    "TravelNodeSuccess ";
                    break;

                case TravelLinkSuccess:
                    "TravelLinkSuccess ";
                    break;

                case TravelEdgeSuccess:
                    "TravelEdgeSuccess ";
                    break;

                case TravelDestBlocked:
                    "TravelDestBlocked ";
                    break;

                case TravelNodeBlocked:
                    "TravelNodeBlocked ";
                    break;

                case TravelLinkBlocked:
                    "TravelLinkBlocked ";
                    break;

                case TravelEdgeBlocked:
                    "TravelEdgeBlocked ";
                    break;

                case TravelEdgeFailure:
                    "TravelEdgeFailure ";
                    break;

                default:
                    "Unknown ";
            }
            "</B>";
            if (d)
                "\n\ttravelDest:&emsp&emsp&emsp&ensp<<d.name>>";
            if (c)
                "\n\tcurrTravelNode:&emsp<<c.name>>";
            if (n)
                "\n\tnextTravelNode:&emsp<<n.name>>";
            if (e)
                "\n\ttravelEdge:&emsp&emsp&emsp&fpmsp<<e.name>>";
            "\n\tlocation:&emsp&emsp&emsp&emsp&fpmsp<<l.name>>";
            "\n----------------------------------------------------";
            "\n";
        }
    }

    initializeActor()
    {
        inherited();

        travelEdgeMemory = new LookupTable();
    }
}

/*
 *  A class to encapsulate information gathered during
 *  runActor() regarding its decision-making process.
 */
class TravelActionInfo: object
{
    value               = nil
    travelDest          = nil  
    currTravelNode      = nil
    nextTravelNode      = nil
    travelEdge          = nil
    location            = nil

    construct(v, d, c, n, e, l)
    {
        value           = v;
        travelDest      = d;
        currTravelNode  = c;
        nextTravelNode  = n;
        travelEdge      = e;
        location        = l;
    }
}