#charset "us-ascii"
#include <adv3.h>

/*

Room Path Finders

the main support structures are borrowed from "Pathfinder"
copyright Mike Roberts, 2003

This file demonstrates how to set up all the different path-finders for
calculating shortest paths between rooms in standard game maps.

Feel free to use these examples however you like.

*/

/* We use the mix-in class "RoomPathFinder" (defined below) to
 * instantiate our path finder algorithms. We could as easily
 * use ApparentRoomPathFinder (also defined below).
 */

// A Room Path Finder using the DoubleBreadth algorithm:
DBRPF: RoomPathFinder, DoubleBreadth;

// A Room Path Finder using the BreadthFirst algorithm:
BFRPF: RoomPathFinder, BreadthFirst;

// A Room Path Finder using the Dijkstra's algorithm:
DRPF: RoomPathFinder, Dijkstra
    findDistance(actor, fromNode, toNode) {
        bestDist_ = nil;
        findPath(actor, fromNode, toNode);
        return bestDist_;
    }
;

/* We make a special roomPathFloyd object, because Floyd needs different
 * information than the RoomPathFinder mix-in class provides.
 *
 * We would normally make this a PreinitObject, so that the paths and
 * distances would be precalculated at compile time. But you might want
 * to include this file in a game that doesn't use Floyd, so we'll
 * define it thus:
 */
roomPathFloyd: Floyd // , PreinitObject

    populateNodeVector() {
        for(local obj = firstObj(Room) ; obj ; obj = nextObj(obj, Room))
            nodeVector += obj;
    }

    nodesConnected(node1, node2) {
        foreach (local dir in Direction.allDirections)
            if (node1.getTravelConnector(dir, myActor) == node2)
                return 1; // connections between rooms aren't weighted
        return nil;
    }

    myActor = gPlayerChar // by default

    /* We allow for the rare situation where travel connectors vary
     * their destination by which actor is traversing the connector.
     */
    recalcForActor(actor) {
        myActor = actor;
        execute();
    }
;

/* RoomPathFinder:
 *
 * This is a mix-in class designed to pass information to one of the
 * path finding algorithms about the underlying graph, which is in
 * this case the game-world map. This works for BreadthFirst,
 * DoubleBreadth, and Dijkstra.
 *   
 * This implementation traverses rooms based on the actual connections
 * in the game map. Note that this isn't appropriate for all purposes,
 * since it doesn't take into account what the actor knows about the
 * game map. This "omniscient" implementation is suitable for situations
 * where the actor's knowledge isn't relevant and we just want the
 * actual best path between the locations.  
 */

class RoomPathFinder: object
    /* find the path for a given actor from one room to another */
    findPath(actor, fromLoc, toLoc)
    {
        /* remember the actor */
        actor_ = actor;

        /* run the normal algorithm */
        return inherited(fromLoc.getOutermostRoom(), toLoc);
    }

    /* assume an unweighted graph: just count the nodes. */
    findDistance(actor, fromNode, toNode) {
        return (findPath(actor, fromNode, toNode).length() - 1);
    }

    /* 
     *   iterate over the nodes adjacent in the underlying graph to the
     *   given node 
     */
    forEachAdjacent(loc, func)
    {
        /* 
         *   run through the directions, and add the apparent destination
         *   for each one 
         */
        foreach (local dir in Direction.allDirections)
        {
            local conn;
            local dest;
            
            /* 
             *   if there's a connector, and it has an apparent
             *   destination, then the apparent destination is the
             *   adjacent node 
             */
            if ((conn = loc.getTravelConnector(dir, actor_)) != nil
                && (dest = getDestination(loc, dir, conn)) != nil
                && includeRoom(dest))
            {
                /* 
                 *   This one seems to go somewhere - process the
                 *   destination.  The standard room map has no concept of
                 *   distance, so use equal weightings of 1 for all
                 *   inter-room distances.  
                 */
                (func)(dest, 1);
            }
        }
    }

    /*
     *   Get the location adjacent to the given location, for the purposes
     *   of finding the path.  By default, we return the actual
     *   destination, but subclasses might want to use other algorithms.
     *   For example, if a subclass's goal is to make an NPC find its own
     *   way from one location to another, then it should use the APPARENT
     *   destination, from the NPC's perspective, rather than the actual
     *   destination, since we'd want to construct the path based on the
     *   NPC's knowledge of the map.  
     */
    getDestination(loc, dir, conn)
    {
        /* return the actual destination for the connector */
        return conn.getDestination(loc, actor_);
    }

    /*
     *   For easier customization, this method allows the map that we
     *   search to be filtered so that it only includes a particular
     *   subset of the map.  This returns true if a given room is to be
     *   included in the search, nil if not.  By default, we include all
     *   rooms.  Note that this is only called to begin with for rooms
     *   that have apparent connections to the starting room, so there's
     *   no need to filter out areas of the map that aren't connected at
     *   all to the starting search location.  
     */
    includeRoom(loc) { return true; }

    /* the actor who's finding the path */
    actor_ = nil
;

/* ApparentRoomPathFinder:
 *
 * This is a slight modification of the RoomPathFinder. In this case we
 * want the grid to represent the "apparent" or "known" game-map paths,
 * rather than the actual paths in the game. Otherwise, it works the
 * same as the RoomPathFinder.
 *
 * Again, this will return the "apparent" shortest path, which is not
 * necessarily the *actual* shortest path; also, if knowledge can be
 * mistaken or appearance misleading, the path returned may not be
 * viable at all.
 */
class ApparentRoomPathFinder: RoomPathFinder
    /* 
     *   Get the destination.  Unlike the base class implementation, we
     *   take into the NPC's knowledge of the map by only using connector
     *   destinations that are APPARENT to the NPC. 
     */
    getDestination(loc, dir, conn)
    {
        /* return the apparent destination of the connector */
        return conn.getApparentDestination(loc, actor_);
    }
;

