#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2003 by Kevin Forchione. All rights reserved.
 *   
 *   Portions based on work by Michael J. Roberts, used by permission.  
 *
 *  This file is part of PROTEUS, the TADS 3 Utility Classes Package
 *
 *  MultiMorphic.t
 *
 *--------------------------------------------------------
 *  !! MULTI_MOPRHIC.H must be #included in ADV.H in order
 *  !! to override the inherited() keyword.
 *--------------------------------------------------------
 *
 *  Implements the MultiMorphic class. This class simulates 
 *  dynamic object inheritance via the object's defaultHandler
 *  list.
 *
 *  For instance, the following defines a LockableOpenable using
 *  the MultiMorphic class:
 *
 *      myObj: MultiMorphic
 *      {
 *          'myobj' 'myobj'
 *          defaultHandler = [Lockable, OpenableContainer]
 *      } 
 */

/* include the proteus header */
#include "adv3.h"

/*
 *  Define the propNotDefined property and export it so that the VM can
 *  call this property when a call is made to an undefined property.
 */
property propNotDefined;
export propNotDefined;

modify Object
{
    /*
     *  Returns a linear representation of the object's 
     *  inheritance order from the object's flattened 
     *  inheritance tree.
     */
    getInheritanceOrder()
    {
        local v = new Vector(40);

        // create superclass tree for this object
        scTreeIter(v);

        // remove any empty elements
        v.removeElement(nil);

        // return the result
        return v;
    }
    scTreeIter(vector)
    {
      // clear any existing instances of self in the vector
        vector.applyAll({x: x == self ? nil : x});  
        
        // add self to the vector
        vector.append(self);

        // insert the superclasses of this element into the queue
        foreach (local o in getSuperclassList())
            o.scTreeIter(vector);
    }

    /*
     *  Determine the object that directly-defines 
     *  the property for this object.
     */
    getPropDefObj(prop, [args])
    {
        local indx, orderList;

        args.length() == 0 ? indx = 1 : indx = args.car();

        orderList = getInheritanceOrder();

        for (local i = indx; i <= orderList.length(); ++i)
            if (orderList[i].getPropList().indexOf(prop))
                return orderList[i];

        return nil;
    }

    /*
     *  Returns the appropriate PropDefXXX value for 
     *  the property. If no object in the inheritance 
     *  tree defines the property then the method 
     *  returns nil.
     */
    getPropDefVal(prop, [args])
    {
        local obj = getPropDefObj(prop, args...);

        if (obj == nil)
            return nil;
        else
            if (obj == self)
                return PropDefDirectly;
            else
                return PropDefInherits;
    }

    /*
     *  Method simulates the TADS 3 inherited() keyword, which
     *  we've overridden to call this method. 
     */
    doInheritedProp(targetProp, targetObj, definingObj, [args])
    {
        local o;
        
        o = propInherited(targetProp, targetObj, definingObj, PropDefGetClass);

        if (o == nil)
            return;
        else
            return delegated o.(targetProp)(args...);
    }
}

class MultiMorphic: PreinitObject
{
    /* a Morphic is a Thing by default */
    defaultHandler          = [Thing]
    inheritanceOrderList    = []

    propNotDefined(prop, [args])
    {
        local obj;
        
        obj = getPropDefObj(prop);

        if (obj == nil)
            return nil;
        else
            return delegated obj.(prop)(args...);
    }

    getDefaultHandler()
    {
        return defaultHandler;
    }
    
    setDefaultHandler(lst)
    {
        defaultHandler  = lst;
        setInheritanceOrderList();
    }

    insertDefaultHandler(index, [vals])
    {
        defaultHandler = defaultHandler.insertAt(index, vals...);
        setInheritanceOrderList();
    }

    removeDefaultHandler([vals])
    {
        defaultHandler -= vals;
        setInheritanceOrderList();
    }

    setInheritanceOrderList()
    {
        inheritanceOrderList = getInheritanceOrder().toList();
    }

    /*
     *  The superclass list is built by simply appending
     *  the default handler to this object's inherited 
     *  superclass list.
     */
    getSuperclassList()
    {
        return defaultHandler;
    }
    
    /*
     *  Equivalent to the ofKind() intrinsic method. 
     *  Method searches the inheritanceOrderList to 
     *  see if this object inherits from obj.
     */
    ofKind(obj)
    {
        return (inheritanceOrderList().indexOf(obj) != nil);
    }

    /*
     *  Algorithm does the same thing as 
     *  propDefined(prop, PropDefGetClass), 
     *
     *  Since propDefined() is overridden in
     *  MultiMorphics this method relies on the
     *  flat superclass tree search list, and the
     *  directly-defined property list provided 
     *  by getPropList() to determine the object
     *  that directly-defines the property for 
     *  this object.
     */
    getPropDefObj(prop, [args])
    {
        local indx;

        args.length() == 0 ? indx = 1 : indx = args.car();

        for (local i = indx; i <= inheritanceOrderList.length(); ++i)
            if (inheritanceOrderList[i].getPropList().indexOf(prop))
                return inheritanceOrderList[i];

        return nil;
    }

    /*
     *  Returns the appropriate PropDefXXX value for 
     *  the property. If no object in the inheritance 
     *  tree defines the property then the method 
     *  returns nil.
     */
    getPropDefVal(prop, [args])
    {
        local obj = getPropDefObj(prop, args...);

        if (obj == nil)
            return nil;
        else
            if (obj == self)
                return PropDefDirectly;
            else
                return PropDefInherits;
    }

    getPropParams(prop)
    {
        local obj;

        obj = getPropDefObj(prop);

        if (obj == nil)
            return nil;
        else if (obj.ofKind(definingobj))
            return inherited(prop);
        else
            return obj.getPropParams(prop);
    }

    propDefined(prop, [args])
    {
        local val;
        
        val = args.car();

        switch(val)
        {
            case PropDefGetClass:
                return getPropDefObj(prop);

            case nil:
            case PropDefAny:
                return (getPropDefVal(prop) != nil);

            case PropDefDirectly:
            case PropDefInherits:
                return (getPropDefVal(prop) == val);

            default:
                throw new MorphicRuntimeError(2306, 'invalid value for Morphic function argument');
        }
    }

    /* 
     *   Determine if a property is inherited further from the given object.
     *   definingObj is usually the value of the 'definingobj'
     *   pseudo-variable, and origTargetObj is usually the value of the
     *   'targetobj' pseudo-variable.  
     */
    propInherited(prop, origTargetObj, definingObj, flags)
    {
        local otoIndx, doIndx, pdoIndx;

        switch (flags)
        {
            case PropDefAny:
            case PropDefGetClass:
                otoIndx = inheritanceOrderList.indexOf(origTargetObj);
                doIndx  = inheritanceOrderList.indexOf(definingObj);

                if (nil is in (otoIndx, doIndx))
                    return nil;
                else
                    if (otoIndx > doIndx)
                        pdoIndx = otoIndx + 1;
                    else
                        pdoIndx = doIndx + 1;
            
                if (flags == PropDefAny)
                    return getPropDefVal(prop, pdoIndx);
                else 
                    return getPropDefObj(prop, pdoIndx);

            default:
                throw new MorphicRuntimeError(2306, 'invalid value for Morphic function argument');
        }
    }

    propType(prop)
    {
        local obj;

        obj = getPropDefObj(prop);

        if (obj == nil)
            return nil;
        else if (obj.ofKind(definingobj))
            return inherited(prop);
        else
            return obj.propType(prop);
    }

    execute()
    {
        setInheritanceOrderList();

        /* initialize the object's vocabulary and containment */
        initializeVocab();
        initializeThing();
    }
}

class MorphicRuntimeError: RuntimeError
{
    construct(errno, errmsg)
    {
        inherited(errno);

        exceptionMessage = errmsg;
    }
}