#charset "us-ascii"
/* 
 *  Copyright (c) 2006 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Data-structures Library Extension
 *
 *  Queue.t
 *
 *  Defining the Queue class.
 */

#include <tads.h>
#include <t3.h>

class Queue: DoubleLinkedList
{
    /*
     *  Returns true if the queue is empty, and nil otherwwise.
     */
    empty() { return size() == 0; }

    /*
     *  Returns the data value at the front of a non-empty queue.
     */
    front() { return nodeFirst().valThis(); }    
    
    /*
     *   Returns true if the queue has elements, and nil otherwise.
     */
    more() { return size() > 0; }
    
    /*
     *  Removes and returns the item at the front of a non-empty queue.
     */
    pop()
    {
        local value, node1, node2;

        /*
         *  Get the value of the first node in the queue.
         */
        value   = front();

        /*
         *  Get the 1st node in the queue
         */
        node1   = nodeFirst();

        /*
         *  Get the 2nd node in the queue.
         */
        node2   = node1.nextThis();

        /*
         *  If we have a 2nd node it's value
         *  and next pointer become those of
         *  the 1st node. We then remove the 
         *  2nd node from the queue manually.
         */
        if (node2)
        {
            /*
             *  Set the queue's 1st node data / type 
             *  values to those of the 2nd node. 
             */
            if (node2.typeThis())
                node1.setValThis(node2.valThis());
            else
                node1.setValThis();

            /*
             *  Set the queue's 1st node next pointer
             *  to those of the 2nd node.
             */
            node1.setNextThis(node2.nextThis());

            /*
             *  If the queue's 1st node next points to another
             *  node, set that node to point to the queue's 1st node.
             */
            if (node1.nextThis())
                node1.nextThis().setPrevThis(node1);

            /*
             *  Set the 2nd node data/type to nil, releasing 
             *  any references.
             */
            node2.setValThis();

            /*
             *  Set the 2nd node next and prev pointers to nil
             */
            node2.setPrevThis(nil);
            node2.setNextThis(nil);
        }
        else
        {
            /* 
             *  If we don't have a 2nd node in the queue
             *  then we simply set this node's data value
             *  and type to nil.
             */
            node1.setValThis();
        }

        /*
         *  Return the original value of the queue's 1st node.
         */
        return value;
    }

    purge()
    {
        /*
         *   Cut the link between the next node and this one.
         */
        if (nextThis())
        {
            nextThis().setPrevThis(nil);
            setNextThis(nil);
        }
        
        /*
         *   Cut the link between the prev node and this one.
         */
        if (prevThis())
        {
            prevThis().setNextThis(nil);
            setPrevThis(nil);
        }
        
        /*
         *   Clear the data.
         */
        setValThis();
    }
    
    /*
     *  Inserts val to the back of the queue.
     */
    push(value)
    {
        /*
         *   If the queue is empty, set the data of this node.
         */
        if (empty())
            setValThis(value);
        else
        {
            local node;
            
            /*
             *  Create a new Queue node.
             */
            node    = new Queue(value...);
            
            /*
             *  Set the new node's prev pointer to the last node.
             */
            node.setPrevThis(nodeLast);

            /*
             *  Set the last node's next pointer to the new node.
             */
            nodeLast().setNextThis(node);
        }
        
        return self;
    }

    /*
     *  Returns the total number of elements in the queue.
     */
    size() { return length(); }
}