//
//  XTHtmlCharEntityParserTests.m
//  XTads
//
//  Created by Rune Berg on 07/10/16.
//  Copyright © 2016 Rune Berg. All rights reserved.
//

#import <XCTest/XCTest.h>
#import "XTHtmlCharEntityParser.h"


@interface XTHtmlCharEntityParserTests : XCTestCase

@property XTHtmlCharEntityParser *parser;

@end


@implementation XTHtmlCharEntityParserTests

- (void)setUp {
    [super setUp];
	self.parser = [XTHtmlCharEntityParser new];
}

- (void)tearDown {
    [super tearDown];
}

//-----------

- (void)test_reset1
{
	[self.parser reset];
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_reset2
{
	[self.parser reset];
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_reset3
{
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
	[self.parser reset];
	XCTAssertEqualObjects(@"b", [self.parser parse:@"b"]);
}

- (void)test_reset4
{
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a&lt"]);
	[self.parser reset];
	XCTAssertEqualObjects(@";", [self.parser parse:@";"]);
}

- (void)test_reset5
{
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a&lt"]);
	[self.parser reset];
	XCTAssertEqualObjects(@"b", [self.parser parse:@"b"]);
}

- (void)test_reset6
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&lt"]);
	[self.parser reset];
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@";b", [self.parser parse:@";b"]);
}

- (void)test_reset6b
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&lt"]);
	// NOT: [self.parser reset];
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"<b", [self.parser parse:@";b"]);
}

- (void)test_reset7
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&lt"]);
	[self.parser reset];
	XCTAssertEqualObjects(@"b", [self.parser parse:@"b"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

//-----------

- (void)testEmptyInput1 {
	
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testEmptyInput2 {
	
	XCTAssertEqualObjects(@"", [self.parser parse:nil]);
	XCTAssertEqualObjects(@"", [self.parser parse:nil]);
}

- (void)testEmptyInput3 {
	
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

//-----------

- (void)testNamedEntity1 {

	XCTAssertEqualObjects(@"<", [self.parser parse:@"&lt;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testNamedEntity1b {
	
	XCTAssertEqualObjects(@"<", [self.parser parse:@"&lt;"]);
	XCTAssertEqualObjects(@"b", [self.parser parse:@"b"]);
}

- (void)testNamedEntity2 {
	
	XCTAssertEqualObjects(@"", [self.parser parse:@"&l"]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@"t;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testNamedEntity2b {
	
	XCTAssertEqualObjects(@"", [self.parser parse:@"&l"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@"t;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testNamedEntity3 {
	
	XCTAssertEqualObjects(@"", [self.parser parse:@"&"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"l"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"t"]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@";"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testNamedEntity3b {
	
	XCTAssertEqualObjects(@"", [self.parser parse:@"&"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"l"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"t"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@";"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testNamedEntity4 {
	
	XCTAssertEqualObjects(@"<>&", [self.parser parse:@"&lt;&gt;&amp;"]);
	XCTAssertEqualObjects(@"cc", [self.parser parse:@"cc"]);
}

//-----------

- (void)testOneChar1 {
	
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testOneChar2 {
	
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
	XCTAssertEqualObjects(@"b", [self.parser parse:@"b"]);
	XCTAssertEqualObjects(@"c", [self.parser parse:@"c"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testSeveralChars1 {
	
	XCTAssertEqualObjects(@"abc", [self.parser parse:@"abc"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testSeveralChars2 {
	
	XCTAssertEqualObjects(@"ab", [self.parser parse:@"ab"]);
	XCTAssertEqualObjects(@"cde", [self.parser parse:@"cde"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

//-----------

- (void)testTextNamedEntity1 {
	
	XCTAssertEqualObjects(@"a<", [self.parser parse:@"a&lt;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testTextNamedEntity2 {

	XCTAssertEqualObjects(@"a", [self.parser parse:@"a&"]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@"lt;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testNamedEntityText1 {
	
	XCTAssertEqualObjects(@"<a", [self.parser parse:@"&lt;a"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testNamedEntityText2 {
	
	XCTAssertEqualObjects(@"", [self.parser parse:@"&lt"]);
	XCTAssertEqualObjects(@"<a", [self.parser parse:@";a"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testTextNamedEntityText1 {

	XCTAssertEqualObjects(@"a<b", [self.parser parse:@"a&lt;b"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)testTextNamedEntityText2 {

	XCTAssertEqualObjects(@"a", [self.parser parse:@"a&l"]);
	XCTAssertEqualObjects(@"<b", [self.parser parse:@"t;b"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

//-----------

- (void)test_parse_hexCharEntity1
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#x26;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_hexCharEntity2
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#x026;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_hexCharEntity3
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#x0026;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_hexCharEntity4
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#x000026;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_hexCharEntity5
{
	XCTAssertEqualObjects(@"<", [self.parser parse:@"&#x3C;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_hexCharEntity6
{
	XCTAssertEqualObjects(@"<", [self.parser parse:@"&#x3c;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_hexCharEntity6b
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&#x3"]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@"c;"]);
	XCTAssertEqualObjects(@"xx", [self.parser parse:@"xx"]);
}

- (void)test_parse_hexCharEntity6c
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"#"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"x"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"3"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"c"]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@";"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_hexCharEntity7
{
	XCTAssertEqualObjects(@"&&&", [self.parser parse:@"&#x26;&#x26;&#x26;"]);
	XCTAssertEqualObjects(@"&.", [self.parser parse:@"&."]);
}

//-----------

- (void)test_parse_decimalCharEntity1
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#38;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_decimalCharEntity2
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#038;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_decimalCharEntity3
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#0038;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_decimalCharEntity4
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&#000038;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_decimalCharEntity5
{
	XCTAssertEqualObjects(@"<", [self.parser parse:@"&#60;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_decimalCharEntity5b
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&#6"]);
	XCTAssertEqualObjects(@"<", [self.parser parse:@"0;"]);
	XCTAssertEqualObjects(@"d", [self.parser parse:@"d"]);
}

- (void)test_parse_decimalCharEntity5c
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"#"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"6"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@"0"]);
	XCTAssertEqualObjects(@"<b", [self.parser parse:@";b"]);
	XCTAssertEqualObjects(@"c", [self.parser parse:@"c"]);
}

- (void)test_parse_decimalCharEntity6
{
	XCTAssertEqualObjects(@"&&&", [self.parser parse:@"&#38;&#38;&#38;"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

//-----------

- (void)test_parse_unknownNamedCharEntity
{
	XCTAssertEqualObjects(@"&foobar;", [self.parser parse:@"&foobar;"]);
	XCTAssertEqualObjects(@"", [self.parser parse:@""]);
}

- (void)test_parse_unknownNamedCharEntity2
{
	XCTAssertEqualObjects(@"&foobar;", [self.parser parse:@"&foobar;"]);
	XCTAssertEqualObjects(@"x", [self.parser parse:@"x"]);
}

- (void)test_parse_illegalCharInState1
{
	XCTAssertEqualObjects(@"&+", [self.parser parse:@"&+"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_illegalCharInState2
{
	XCTAssertEqualObjects(@"&#+", [self.parser parse:@"&#+"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_illegalCharInState3
{
	XCTAssertEqualObjects(@"&am+", [self.parser parse:@"&am+"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_illegalCharInState4
{
	XCTAssertEqualObjects(@"&#x+", [self.parser parse:@"&#x+"]);
	XCTAssertEqualObjects(@"ab", [self.parser parse:@"ab"]);
}

- (void)test_parse_illegalCharInState5
{
	XCTAssertEqualObjects(@"\x08+", [self.parser parse:@"&#x8+"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_illegalCharInState6
{
	XCTAssertEqualObjects(@"\x06+", [self.parser parse:@"&#6+"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_illegalCharInState6b
{
	XCTAssertEqualObjects(@"\74+", [self.parser parse:@"&#60+"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_unfinishedEntity1
{
	XCTAssertEqualObjects(@"& ", [self.parser parse:@"& "]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_unfinishedEntity2
{
	XCTAssertEqualObjects(@"&# ", [self.parser parse:@"&# "]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_unfinishedEntity3
{
	XCTAssertEqualObjects(@"&#x ", [self.parser parse:@"&#x "]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_unfinishedEntity4
{
	XCTAssertEqualObjects(@"\1 ", [self.parser parse:@"&#x1 "]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_unfinishedEntity5
{
	XCTAssertEqualObjects(@"\2 ", [self.parser parse:@"&#2 "]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_unfinishedEntity6
{
	XCTAssertEqualObjects(@"&& ", [self.parser parse:@"&& "]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_unfinishedEntity6b
{
	XCTAssertEqualObjects(@"&", [self.parser parse:@"&&"]);
	XCTAssertEqualObjects(@"& ", [self.parser parse:@" "]);
}

- (void)test_parse_unfinishedEntity7
{
	XCTAssertEqualObjects(@"&#", [self.parser parse:@"&#&"]);
	XCTAssertEqualObjects(@"& a", [self.parser parse:@" a"]);
}

- (void)test_parse_unfinishedEntity8
{
	XCTAssertEqualObjects(@"&#x", [self.parser parse:@"&#x&"]);
	XCTAssertEqualObjects(@"& a", [self.parser parse:@" a"]);
}

- (void)test_parse_unfinishedEntity9
{
	XCTAssertEqualObjects(@"&&", [self.parser parse:@"&&&"]);
	XCTAssertEqualObjects(@"&.", [self.parser parse:@"."]);
}

- (void)test_parse_unfinishedEntity10
{
	XCTAssertEqualObjects(@"<&", [self.parser parse:@"&lt&&"]);
	XCTAssertEqualObjects(@"&.", [self.parser parse:@"."]);
}

- (void)test_parse_unfinishedEntity11
{
	XCTAssertEqualObjects(@"&<&", [self.parser parse:@"&&lt&&"]);
	XCTAssertEqualObjects(@"&.", [self.parser parse:@"."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity1
{
	XCTAssertEqualObjects(@"\"xx;", [self.parser parse:@"&quotxx;"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity2
{
	XCTAssertEqualObjects(@"\"x;", [self.parser parse:@"&quotx;"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity3
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&quotx"]);
	XCTAssertEqualObjects(@"\"x<", [self.parser parse:@"<"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity4
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&quot"]);
	XCTAssertEqualObjects(@"\".", [self.parser parse:@"."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity5
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&poook"]);
	XCTAssertEqualObjects(@"&poook-", [self.parser parse:@"-"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity7
{
	XCTAssertEqualObjects(@">", [self.parser parse:@"&gt&"]);
	XCTAssertEqualObjects(@"&.", [self.parser parse:@"."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity8
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&gtc"]);
	XCTAssertEqualObjects(@">c.", [self.parser parse:@"."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity9
{
	XCTAssertEqualObjects(@"", [self.parser parse:@"&gtcast"]);
	XCTAssertEqualObjects(@">cast..", [self.parser parse:@".."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity10
{
	XCTAssertEqualObjects(@">cast fireball", [self.parser parse:@"&gtcast fireball"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity11
{
	XCTAssertEqualObjects(@"a>cast fireball", [self.parser parse:@"a&gtcast fireball"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity12
{
	XCTAssertEqualObjects(@"<> fireball", [self.parser parse:@"&lt;&gt; fireball"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity12b
{
	XCTAssertEqualObjects(@"<> fireball", [self.parser parse:@"&lt;&gt fireball"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity13
{
	XCTAssertEqualObjects(@"<>cast F", [self.parser parse:@"&lt;&gtcast F"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity14
{
	XCTAssertEqualObjects(@"\x15>cast fireball", [self.parser parse:@"&#21;&gtcast fireball"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity15
{
	XCTAssertEqualObjects(@"\x15\x15\x15>cast fireball at goblin mage", [self.parser parse:@"&#21;&#21;&#21;&gtcast fireball at goblin mage"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity16
{
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a&gtcast"]);
	XCTAssertEqualObjects(@">cast.", [self.parser parse:@"."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity16b
{
	XCTAssertEqualObjects(@"a>cast;", [self.parser parse:@"a&gtcast;"]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity17
{
	XCTAssertEqualObjects(@"<", [self.parser parse:@"&lt;&gtcast"]);
	XCTAssertEqualObjects(@">cast.", [self.parser parse:@"."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity18
{
	XCTAssertEqualObjects(@"\x15", [self.parser parse:@"&#21;&gtcast"]);
	XCTAssertEqualObjects(@">cast.", [self.parser parse:@"."]);
}

- (void)test_parse_missingSemicolonAfterNamedCharEntity19
{
	XCTAssertEqualObjects(@"\x15\x15\x15", [self.parser parse:@"&#21;&#21;&#21;&gtcast"]);
	XCTAssertEqualObjects(@">cast.", [self.parser parse:@"."]);
}

//-----------

- (void)test_parse_mixOfNamedCharEntitiesAndText1
{
	NSString *string = @"abc&gt;de&lt;f";
	XCTAssertEqualObjects(@"abc>de<f", [self.parser parse:string]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_mixOfNamedCharEntitiesAndText2
{
	NSString *string = @"abc&gt;de&lt;f";
	XCTAssertEqualObjects(@"abc>de<f", [self.parser parse:string]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_mixOfNamedCharEntitiesAndText3
{
	NSString *string = @"&gt;a&lt;F";
	XCTAssertEqualObjects(@">a<F", [self.parser parse:string]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_mixOfHexCharEntitiesAndText1
{
	XCTAssertEqualObjects(@"&a&a&", [self.parser parse:@"&#x26;a&#x26;a&#x26;"]);
	XCTAssertEqualObjects(@"a", [self.parser parse:@"a"]);
}

- (void)test_parse_mixOfHexCharEntitiesAndText2
{
	XCTAssertEqualObjects(@"a&b&c&d", [self.parser parse:@"a&#x26;b&#x26;c&#x26;d"]);
	XCTAssertEqualObjects(@"e", [self.parser parse:@"e"]);
}

- (void)test_parse_mixOfDecCharEntitiesAndText1
{
	XCTAssertEqualObjects(@"&a&b&c", [self.parser parse:@"&#38;a&#38;b&#38;c"]);
	XCTAssertEqualObjects(@"e", [self.parser parse:@"e"]);
}

- (void)test_parse_mixOfDecCharEntitiesAndText2
{
	XCTAssertEqualObjects(@"a&b&c&d", [self.parser parse:@"a&#38;b&#38;c&#38;d"]);
	XCTAssertEqualObjects(@"e", [self.parser parse:@"e"]);
}

//-----------

- (void)test_parse_singleSpecialChars1
{
	XCTAssertEqualObjects(@"\x15", [self.parser parse:@"&#21;"]);
}

- (void)test_parse_singleSpecialChars2
{
	XCTAssertEqualObjects(@"\x15", [self.parser parse:@"&#x15;"]);
}

- (void)test_parse_enspSpecialHandling
{
	XCTAssertEqualObjects(@"\u2002", [self.parser parse:@"&#8194;"]);
	
	XCTAssertEqualObjects(@"a\u2002b", [self.parser parse:@"a&#8194;b"]);
	
	XCTAssertEqualObjects(@"\u2002", [self.parser parse:@"&ensp;"]);
	
	XCTAssertEqualObjects(@"a\u2002b", [self.parser parse:@"a&ensp;b"]);
}

- (void)test_parse_emspSpecialHandling
{
	XCTAssertEqualObjects(@"\u2003", [self.parser parse:@"&#8195;"]);
	
	XCTAssertEqualObjects(@"a\u2003b", [self.parser parse:@"a&#8195;b"]);
	
	XCTAssertEqualObjects(@"\u2003", [self.parser parse:@"&emsp;"]);
	
	XCTAssertEqualObjects(@"a\u2003b", [self.parser parse:@"a&emsp;b"]);
}

@end
