////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: Composite 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _COMPOSITE_H_
#define _COMPOSITE_H_

#include <container.h>
#include <fixture.h>
#include <lockablecontainer.h>
#include <openablecontainer.h>
#include <surface.h>

#pragma C+

/*----------------------------------------------------------------------
 *  MIX-IN CLASS:   This class must always preced any non-mix-in classes
 *                  in a superclass list - this is required because a
 *                  mix-in can't override default methods in the root of
 *                  the hierarchy.
 *--------------------------------------------------------------------*/

/*
 *	Composite: object
 *
 *	The Composite class forms the root of a tree of composite objects, 
 *	objects composed of a "parent" object and component objects. The
 *	theory behind Composite class is that because components possess the
 *	vocabulary of their "parents" they will be selected with the parent
 *	object in cases such as <<open the desk>>, where the drawer is
 *	referable as the "drawer of the desk".
 *
 *	It then becomes a matter of determining which, if any, of the
 *	objects being resolved are capable of completing the specified
 *	action. In general this is accomplished by starting at the "parent"
 *	object and working down the composition tree. Each component in turn 
 *	may become a "parent" of other components. Each "parent" has
 *	precedence over its components when determining if the action can be
 *	completed. All components of the same level that are capable of 
 *	completing the action are returned, producing what is hoped to be
 *	logical disambiguation questions.
 */
class Composite: object
	componentObjList = []
	componentClassList = []
	
	makeComponents = {
		local i, j, x, vList;
		for ( i = 1; i <= length(self.componentClassList); ++i ) {
			/*
			 * create the component; add it to the componentObjList;
			 * assign its componentParent; move the component into 
			 * the parent object.
			 */
			x = new componentClassList[ i ];
			self.componentObjList += x;
			x.componentParent = self;
			x.moveInto( self );
			
			/* add nouns to the component */
			vList = getwords( self, &noun );
			for ( j = 1; j <= length( vList ); ++j ) {
				addword( x, &noun, vList[ j ] );
			}
			
			/* add adjectives to component */
			vList = getwords( self, &adjective );
			for ( j = 1; j <= length( vList ); ++j ) {
				addword( x, &adjective, vList[ j ] );
			}
			
			/* add plural to component */
			vList = getwords( self, &plural );
			for ( j = 1; j <= length( vList ); ++j ) {
				addword( x, &plural, vList[ j ] );
			}
		}
	}
	
    disambigIobj(actor, prep, dobj, verProp, actionProp, wordList, objList,
               flagList, numberWanted, isAmbiguous, silent) = 
    {	
    	local i, lst = [];
    	
		/* is the componentParent in the objList? */
		if ( self.componentParent 
		&& find( objList, self.componentParent ) != nil ) {
			lst += self.componentParent.disambigIobj( actor, prep, dobj, verProp, 
				actionProp, wordList, objList, flagList, numberWanted,
				isAmbiguous, silent );
				
			/* return component parent if no substitution */	
			if ( length( lst ) == 0 ) 
				lst += self.componentParent;
				
			return lst;
		}

		/* 
	 	 * does the verb have an actionProp? if not let this obj handle it 
	 	 */
		if ( actionProp == nil ) {
			lst += self;
			return lst;
		}

		/* can this object perform the action? */
		if ( defined( self, actionProp ) ) {
			lst += self;
			return lst;
		}
	
		/* remove this object from the objList */
		if ( find( objList, self ) ) {
	 			local tempList = [];
			tempList += self;
			objList -= tempList;
		}

		/* iterate through this object's component object list */
		for ( i = 1; i <= length( self.componentObjList ); ++i ) {			
			lst += self.componentObjList[ i ].disambigIobj( actor, 
				prep, dobj, verProp, actionProp, wordList, 
				objList, flagList, numberWanted, isAmbiguous, 
				silent );
		}

		return lst;
    }
    
    disambigDobj(actor, prep, iobj, verProp, actionProp, wordList, objList,
               flagList, numberWanted, isAmbiguous, silent) = 
    {
    	local i, lst = [];

		/* is the componentParent in the objList? */
		if ( self.componentParent 
		&& find( objList, self.componentParent ) != nil ) {
			lst += self.componentParent.disambigDobj( actor, prep, iobj, verProp, 
				actionProp, wordList, objList, flagList, numberWanted,
				isAmbiguous, silent );
				
			/* return component parent if no substitution */	
			if ( length( lst ) == 0 ) 
				lst += self.componentParent;
				
			return lst;
		}

		/* 
	 	 * does the verb have an actionProp? if not let this obj handle it 
	 	 */
		if ( actionProp == nil ) {
			lst += self;
			return lst;
		}

		/* can this object perform the action? */
		if ( defined( self, actionProp ) ) {
			lst += self;
			return lst;
		}
	
		/* remove this object from the objList */
		if ( find( objList, self ) ) {
			local tempList = [];
			tempList += self;
			objList -= tempList;
		}

		/* iterate through this object's component object list */
		for ( i = 1; i <= length( self.componentObjList ); ++i ) {			
			lst += self.componentObjList[ i ].disambigDobj( actor, 
				prep, iobj, verProp, actionProp, wordList, 
				objList, flagList, numberWanted, isAmbiguous, 
				silent );
		}

		return lst;
    }
;

class SurfaceContainer: Composite
	componentClassList = [ ComponentSurface, ComponentContainer ]
;

class SurfaceOpenableContainer: Composite
	componentClassList = [ ComponentSurface, ComponentOpenableContainer ]
;

class SurfaceLockableContainer: Composite
	componentClassList = [ ComponentSurface, ComponentLockableContainer ]
;

class Component: Composite, Fixture
;

class ComponentSurface: Component, Surface
	adjective = 'surface' 'top'
	sDesc = "top of <<self.componentParent.theDesc>>"
;

class ComponentContainer: Component, Container
	adjective = 'inside' 'insides'
	sDesc = "inside of <<self.componentParent.theDesc>>"
;

class ComponentOpenableContainer: Component, OpenableContainer
	adjective = 'inside' 'insides'
	sDesc = "inside of <<self.componentParent.theDesc>>"
;

class ComponentLockableContainer: Component, LockableContainer
	adjective = 'inside' 'insides'
	sDesc = "inside of <<self.componentParent.theDesc>>"
;

#pragma C-

#endif /* _COMPOSITE_H_ */
