/*

   == T2 Combat

   == Copyright (c) 2002, Steve Breslin

   == based in part on monster.t (c) 1991 by Dave Nault
 
   ==>>  License:
     _______________________________________________________________
    /                                                               \
    |     You can use this program however you want, but if you     |
    |     decide to distribute anything that uses, #include's or    |
    |     otherwise borrows from this program, you have to send     |
    |     a copy of any improvements or modify-cations you made     |
    |     to this program to its author, Steve Breslin.             |
    |                                                               |
    |     That way, you will help keep the program up to date for   |
    |     everybody else, and everybody else will help keep it      |
    |     up to date for you.                                       |
    |                                                               |
    |     You may redistribute this verbatim or in modified form    |
    |     only if you keep the copyright and license intact.        |
    |                                                               |
    |     Also, feel encouraged to release your source code along   |
    |     with your game, though this isn't a requirement.          |
    |                                                               |
    |     The author can be reached at <versim@hotmail.com>.        |
    \______________________________________________________________*/


/*-------------------------------------------------------------------
 * First, the new material:
 */

/* The monster(s) and player have their 'heal' method called
 * each turn.
 *
 * The actual healing is handled by the particular objects, so that
 * each can heal different amounts, at different frequencies.
 */
healDaemon: function( parm ) {

    local i;

    /* First the monsters heal.
     */
    for (i := 1 ; i <= length(global.monsterlist) ; i++) {
        global.monsterlist[i].heal;
    }

    /* Then the player heals.
     */
    parserGetMe.heal;
}

/* The 'combatant' class is a behind-the-scenes utility class which
 * the 'combatMe' and the 'monster' class inherit from. Basically, it
 * encapsulates all commonalities between 'combatMe' and 'monster'.
 * We could have instead modified the 'Actor' class, since both 'Me'
 * and 'monster' inherit from 'Actor', but this would assume that
 * all actors are combatants, which might not always be the case.
 * Also, if your game uses multiple PC objects, you might want one or
 * more of them not to be combatants; in this case, you can use
 * basicMe instead of combatMe.
 *
 * So 'combatant_' shouldn't be used as a game object. Use 'monster'
 * instead. 'Actor' should be used for non-attackable NPC's, and
 * 'basicMe' should be used if the PC doesn't engage in combat. If
 * the PC is supposed to engage in combat, use 'combatMe' instead of
 * 'basicMe'.
 */

combatant_: Actor

    AC = 10

    endurance = 10
    maxendurance = 10

    combating = nil
    wielding = nil

    /* This is what executes a the heal for each combatant. This
     * method is called each round, after the command is processed.
     * 
     * First, we check if the object's endurance is below max. If not,
     * we do nothing.
     *
     * Then we check if the object's percent chance to heal is lower
     * than a random number generated from 1 to 100. If not, we do
     * nothing.
     *
     * Then we generate a number between 1 and the object's maxheal
     * number, which is the maximum number of endurance points a
     * object can recover in a single heal. We add this number to
     * the object's endurance.
     *
     * If the object heals, we alert the PC when appropriate.
     *
     * Finally, if the object's current endurance is greater than
     * the maximum, we correct this so that it just matches, but does
     * not exceed the maximum.
     */
    heal() = {
        if (self.endurance < self.maxendurance) {
            if (rand(100) < self.percentchancetoheal ||
                /* resting doubles chances of healing */
                self.resting &&
                rand(100) < self.percentchancetoheal) {
                self.endurance += rand(self.maxheal);
                if (parserGetMe = self)
                    "\nYou feel a bit better.\n";
                else if (self.isReachable(parserGetMe))
                    "\n<<self.thedesc>> looks a bit better.\n";
                if (self.endurance > self.maxendurance) {
                    self.endurance := self.maxendurance;
                }
            self.resting := nil;
            }
        }
    }

    maxheal = 2
    percentchancetoheal = 15

    /* The player can use the 'rest' verb. This increases the PC's
     * chances to heal that turn. By default, monsters never rest.
     */
    resting = nil

    /* This method controls the object's sustaining damage.
     * If the damage sustained drops the endurance below 1, the
     * object dies. Otherwise, its endurance is updated and
     * that's it.
     *
     * Note that the actual death is handled by the death()
     * function, which is defined by the particular object or
     * by the 'monster' and 'combatMe' subclasses.
     */
    sustaindamage(amount) = {
        self.endurance -= amount;
        if (self.endurance < 1)
            self.death;
    }

    /* The attack() method determines if the present object should
     * determine the damage dealt, or if the weapon should determines
     * the damage.
     *
     * Before damage is dealt, we call the beforedamage method, which
     * doesn't do anything by default.
     */
    attack(target) = {
        if (target.combatant = nil) // if my opponent isn't already fighting
            target.combatant := self; // then I become the combatant
        self.beforedamage;
        if (wielding)
            self.wielding.damage(target);
        else
            self.damage(target);
    }

    beforedamage() = { }

    /* This does the damage when we have no weapon wielded. */
    damage(target) = {
        if (self.calchit(target)) {
            self.hitmessage(target);
            target.sustaindamage(self.calcdamage);
        }
        else
            self.missmessage(target);
    }

    /* Here we calculate the chance to hit. This is based on the attacker's
     * level and hit bonus, and is counteracted by the target's AC.
     *
     * As a sort of default, we have weighted things such that hitting
     * happens about 50% of the time.
     *
     * The author can easily adapt this by modifying either the base AC
     * and/or the hitbonus.
     *
     * If you want to modify how hits are calculated, you can modify the
     * following method. You might rather modify the caller, damage(target)
     * in some way as well. For example, some find it "more realistic" for
     * armor to soak a certain amount of damage inflicted by attacks
     * rather than making the armored PC/NPC harder to hit. Some such
     * systems portray armour as ablative, much like additional hit points
     * would be, while others treat it as a fixed reduction to the damage
     * of all attacks.
     *
     * The hazard to such systems is that a heavily-armoured PC can wade
     * into the thick of a melee and know that he/she is essentially
     * invulnerable, which detracts from combat tension.  There are
     * a couple ways to avoid that threat.  For example, as a game
     * designer you can arrange things so that the PC never has
     * access to armor so powerful that it stops all damage. Also,
     * you can introduce the concept of a "critical hit", which would be
     * a low-probability (say, 5%) event that would bypass all armor.
     * Either approach makes armor a valuable commodity (as it was in
     * most low-tech armies), but prevents it from becoming an unrealistic
     * panacea. Keep in mind, though, that realism isn't the final measure
     * for a good game, and might detract from fun rather than add to
     * believability or whatever.
     */
    calchit(target) = {
        return (rand(20) > (20 - (self.level + self.hitbonus) + (10 - target.AC)));
    }

    level = 1 // by default

    hitbonus = 10 // by deafult

    /* This determines bare hand damage dealt. */
    calcdamage() = {
        local rolls, total;
        total := 0;
        rolls := 1;
        while (rolls <= self.dice) {
            total += rand(self.sides);
            rolls++;
        }
        return total;
    }

    /* Default bare hand damage: */
    dice = 1
    sides = 4

    /* We note the endurance after the normal inspection material.
     */
    doInspect(actor) = {
        inherited.doInspect(actor);
        if (self = parserGetMe)
            "Your ";
        else
            "\n\^<<self.thedesc>>'s ";
        "endurance is
        currently <<self.endurance>> of <<self.maxendurance>>.\n";
    }

    reverseChargeAttackingACPenalty = {
        self.AC -= self.chargeAttackingACPenalty;
    }

    reverseChargeDefendingACPenalty = {
        self.AC -= self.chargeDefendingACPenalty;
    }

    chargeAttackingACPenalty = 5
    chargeDefendingACPenalty = 3
;

/* Use combatMe instead of basicMe for a PC who can fight monsters.
 */
combatMe: basicMe, combatant_

    /* Note that we give the default player an edge over default
     * monsters.
     */
    endurance = 15
    maxendurance = 15

    /* These messages are used when the player has no weapon.
     * If the player has a weapon, the weapon's hit and miss
     * messages are used instead.
     */
    hitmessage(target) = {
        "You punch <<target.thedesc>>.\n";
    }

    missmessage(target) = {
        "You swing your fist at <<target.thedesc>>, but miss.\n";
    }

    /* When the PC dies, the die() function is called.
     */
    death() = {
        die();
    }

    /* The standard roomCheck() is called on the parserGetMe before
     * the execution of a turn cycle. This in turn calls the room's
     * room check. However, other actors don't get consulted.
     *
     * So we're modifying roomCheck(), so that other
     * actors (monsters) will have a chance to attack the player
     * before his command is executed.
     */
    roomCheck(v) = {
        local i;
        for (i := 1 ; i <= length(global.monsterlist) ; i++) {
            if (global.monsterlist[i].canAttack(self))
                global.monsterlist[i].checkattack;
        }
        return inherited.roomCheck(v);
    }
    
    /* Silly message for failure message when player attacks himself */
    verDoAttackWith( actor, io ) = { "There's no need to get suicidal. "; }

    /* We provide a skeleton framework for PC levels, although we don't
     * really use this in the basic combat module. We call gainexp()
     * from monster.death, but here we pass a 0 amount and do nothing
     * more. This is just here to make an extension in this direction
     * easier for the user.
     */
    gainexp(amount) = {
        if (amount > 0) {
            "\nYou gain <<amount>> experience points.\n";
            self.exp += amount;
        }
    }
    exp = 0

    defendExpiresFuse = {
        self.AC += defendVerb.defendBonus;
    }
;

monster: combatant_

    /* When a monster is dynamically created, we want to add it to the
     * global.monsterlist when it is constructed.
     */
    construct() = {
        global.monsterlist += self;
        inherited.construct;
    }

    /* All monsters fight back when attacked, but some monsters
     * are aggressive, and attack first. However, they don't
     * attack friends. E.g., two grues shouldn't attack each other.
     */
    aggressive = true
    friendlist = []

    /* Even if I'm not naturally aggressive, I will react negatively
     * towards anyone who attacks me. I will become aggressive, unless
     * I am wimpy.
     */
    enemylist = []
    wimpy = nil

    /* This is what happens when I die.
     */
    death() = {
        "\^<<self.thedesc>> dies.\n";
        global.monsterlist -= self;
        self.createcorpse;
        self.moveInto(nil);
        if (parserGetMe.combatant = self) {
            parserGetMe.gainexp(self.expvalue);
            parserGetMe.combatant := nil;
        }
    }

    expvalue = 0

    /* The corpse will probably need to be monster-specific, so
     * we'll cheat around making a corpse in the general case.
     */
    createcorpse() = {
        "A cloud of smoke envelopes <<self.thedesc>>, and slowly
        dissipates, leaving no trace of the body.\n";
    }

    /* These messages are used when the monster has no weapon.
     * If the monster has a weapon, the weapon's hit and miss
     * messages are used instead.
     */
    hitmessage(target) = {
        "\^<<self.thedesc>> hits ";
        if (target = parserGetMe)
            "you.\n";
        else {
            target.thedesc;
            ".\n";
        }
    }

    missmessage(target) = {
        "\^<<self.thedesc>> misses ";
        if (target = parserGetMe)
            "you.\n";
        else {
            target.thedesc;
            ".\n";
        }
    }

    /* Monsters can be attacked. */
    verDoAttackWith(actor, io) = {}
    verDoAttack(actor) = {}

    /* These handle the attack to the monster. */
    doAttackWith(actor, io) = {
        if (actor.wielding <> io) {
            "\n(First wielding <<io.thedesc>>)\n";
            execCommand(parserGetMe, wieldVerb, io);
        }
        if (actor.wielding <> io)
            "Instead of doing nothing, you attack with what
            you have. ";
        self.doAttack(actor);
    }  

    doAttack(actor) = {
        actor.attack(self); // player swings at me
        if (self.combatant = nil) {
            self.combatant := actor;
            if (!self.aggressiveattacking) {
                self.counterattacking := true;
                notify(self, &counterAttackDaemon, 0);
            }
        }
    }

    /* Charge passes processing to attack, but alters the AC of the
     * attacker and target. The AC of the target is altered for
     * the attack, making the target easier to hit; the AC of the
     * attacker is similarly altered for the remainder of this turn,
     * plus the entire next turn. This will not be as deterimental if
     * the attacker has the initiative.
     */
    verDoChargeWith(actor, io) = { self.verDoAttackWith(actor, io); }
    verDoCharge(actor) = { self.verDoAttack(actor); }

    doChargeWith(actor, io) = {
        "You charge!\n";
        actor.AC += actor.chargeAttackingACPenalty;
        self.AC += self.chargeDefendingACPenalty;
        self.doAttackWith(actor, io);
        notify(actor, &reverseChargeAttackingACPenalty, 2);
        notify(self, &reverseChargeDefendingACPenalty, 1);
    }

    doCharge(actor) = {
        "You charge!\n";
        actor.AC += actor.chargeAttackingACPenalty;
        self.AC += self.chargeDefendingACPenalty;
        self.doAttack(actor);
        notify(actor, &reverseChargeAttackingACPenalty, 2);
        notify(self, &reverseChargeDefendingACPenalty, 1);
    }

    aggressiveAttack(target) = {
        if (self.canAttack(target)) {
            self.attack(target);
            self.aggressiveattacking := true;
        }
    }

    attack(target) = {
        if (!find(self.enemylist, target))
            self.enemylist += target;
        inherited.attack(target);
    }

    /* Before we attack, we wield a weapon, if we have a weapon.
     * You might like to add a message or emotive the monster sometimes
     * makes as it attacks.
     */
    beforedamage() = {
        if (self.wielding = nil) {
            local i;
            for (i := 1 ; i <= length(self.contents) ; i++) {
                if (isclass(self.contents[i], weapon)) {
                    self.wielding := self.contents[i];
                    "\n\^<<self.thedesc
                    >> wields <<self.contents[i].thedesc>>.\n";
                }
            }
        }
    }

    /* Counterattack Daemon. This is a daemon because daemons execute
     * every turn, after the action. A counter attack comes after the
     * attack, so a daemon is good for that.
     */
    counterAttackDaemon() = {
        if (self.canAttack(combatant)) {
            self.attack(combatant);
        }
        else { // target out of range: stop counterattacking
            unnotify(self, &counterAttackDaemon);
            self.combatant := nil;
            counterattacking := nil;
        }
    }

    /* This function is called by the player object's roomCheck.
     * We don't necessarily attack, but if we do, we will attack
     * before the player's command is executed.
     *
     * This method continues an aggressive attack began before the
     * current turn cycle. Also, it will begin an aggressive
     * attack in the peculiar case that the attack didn't begin
     * when the player comes in range. (This might be possible
     * if the player teleports into a room, for example, or when
     * the monster's canAttack() check will permit the monster to
     * attack players in other rooms.) Normally, the attack will
     * begin as the player enters the room.
     */
    checkattack() = {
        if (!self.counterattacking && self.combatant) {
            self.aggressiveAttack(combatant);
        }
        else if (self.aggressive && self.canAttack(parserGetMe)) {
            self.attack(parserGetMe);
        }
        else if (self.aggressive) { // I will try to attack something
            local i;
            for (i := 1 ; i <= length(global.monsterlist) ; i++) {
                if (self.canAttack(global.monsterList[i])) {
                    self.attack(global.monsterlist[i]);
                    break;
                }
            }
        }
    }

    /* By default, the canAttack() verification checks whether or not
     * the target is reachable in the ordinary way. A sniper or magician
     * might be able to make an attack from another room, attacking a
     * monster or player which is not normally reachable.
     */
    canAttack(target) = {
        return target.isReachable(self);
    }

    /* playerentered() is called when a player enters the room which the
     * monster currently occupies. This could be useful for triggering
     * greeting messages for example, but we're using it to tell
     * aggressive mobs they they now have a target.
     */
    playerentered() = {
        local actor := parserGetMe; // should expand this for all (mobile) monsters.
        if (find(self.enemylist, actor)) {
            if (self.wimpy) {
                "\^<<self.thedesc>> cowers in the corner as you enter
                the room. ";
            }
            else {
                "\^<<self.thedesc>> attacks as you enter the room.\n";
                self.aggressiveAttack(actor);
            }
            return;
        }
        if (self.aggressive &&
                 !find(self.friendlist, actor)) {
            "\^<<self.thedesc>> attacks as you enter the room. ";
            self.aggressiveAttack(actor);
        }
        return;
    }
;

weapon: item

    /* The weapon does the actual damage. */
    damage(target) = {
        if (self.calchit(target)) {
            self.hitmessage(target);
            target.sustaindamage(self.calcdamage);
        }
        else
            self.missmessage(target);
    }

    /* By default, weapons hit the target with the same success that
     * a barehand attack hits the target, as determined by the weapon's
     * wielder.
     */
    calchit(target) = {
        return (self.location.calchit(target));
    }

    /* Two small methods for generating messages for hits and misses.
     * It would be nice for weapons to make specific messages for
     * the weapon type, and vary the messages so as not to get too
     * repetitive.
     */
    missmessage(target) = {
        if (self.location = parserGetMe)
            "You ";
        else
            "\^<<self.location.thedesc>> ";
        "swing";
        if (self.location != parserGetMe)
            "s";
        " <<self.thedesc>> at ";
        if (self.location = parserGetMe)
            "<<target.thedesc>>";
        else
            "you";
        ", but miss";
        if (self.location != parserGetMe)
            "es";
        ".\n";
    }

    hitmessage(target) = {
        if (self.location = parserGetMe)
            "You ";
        else
            "\^<<self.location.thedesc>> ";
        "strike";
        if (self.location != parserGetMe)
            "s";
        if (self.location = parserGetMe)
            " <<target.thedesc>> ";
        else
            " you ";
        "with <<self.thedesc>>.\n";
    }

    /* The weapon determines the damage dealt. */
    calcdamage() = {
        local rolls, total;
        total := 0;
        rolls := 1;
        while (rolls <= self.dice) {
            total += rand(self.sides);
            rolls++;
        }
        return total;
    }

    /* The default damage dice for a weapon. */
    dice = 1
    sides = 6

    verIoAttackWith(actor) = {}
    ioAttackWith(actor, dobj) = { dobj.doAttackWith(actor, self); }
    verIoChargeWith(actor) = { self.verIoAttackWith(actor); }
    ioChargeWith(actor, dobj) = {
        "You charge!\n";
        self.ioAttackWith(actor, dobj);
    }

    verDoWield(actor) = {
        if (actor.wielding = self)
            "%You're% already wearing <<self.thedesc>>! ";
    }

    doWield(actor) = {
        if (actor.wielding) { "(Unwielding <<actor.wielding.thedesc>> first)\n"; }
        if (self.location != actor)
        {
            /* try taking it */
            "(First taking <<self.thedesc>>)\n";
            if (execCommand(actor, takeVerb, self) != 0)
                exit;

            /* 
             *   make certain it ended up where we want it - the command
             *   might have failed without actually indicating failure 
             */
            if (self.location != actor)
                exit;
        }

        /* wield it */
        "Okay, %you're% now wielding "; self.thedesc; ". ";
        actor.wielding := self;
    }

    verDoUnwield(actor) = {
        if (actor.wielding != self)
            "You're not wielding it.\n";
    }

    doUnwield(actor) = {
        actor.wielding := nil;
        "Unwielded.\n";
    }

    checkDrop = {
        if (parserGetMe.wielding = self)
        {
            "(Unwielding "; self.thedesc; " first)\n";
            parserGetMe.wielding := nil;
        }
    }

    doDrop(actor) =
    {
        self.checkDrop;
        pass doDrop;
    }

    doPutIn(actor, io) =
    {
        self.checkDrop;
        pass doPutIn;
    }

    doPutOn(actor, io) =
    {
        self.checkDrop;
        pass doPutOn;
    }

    doGiveTo(actor, io) =
    {
        self.checkDrop;
        pass doGiveTo;
    }

    /* For the moment, throw weapons aren't implemented.
     * Thus, 'throw weapon' works the same way as throw
     * anything: basically, it works the same way as drop.
     */

    doThrowAt(actor, io) =
    {
        self.checkDrop;
        pass doThrowAt;
    }

    doThrowTo(actor, io) =
    {
        self.checkDrop;
        pass doThrowTo;
    }

    doThrow(actor) =
    {
        self.checkDrop;
        pass doThrow;
    }

    moveInto(obj) =
    {
        /*
         * Catch any other movements with moveInto; this won't stop the
         * movement from happening, but it will prevent any anamolous
         * consequences caused by the object moving but still being
         * wielded.
         */
        if (self.location && self.location.wielding)
            self.location.wielding := nil;
        pass moveInto;
    }
;

/* Two verbs for wielding and unwielding weapons:
 */
wieldVerb: darkVerb
    sdesc = "wield"
    verb = 'wield' 'ready'
    doAction = 'Wield'
    doDefault( actor, prep, io ) = {
        local ret;
        ret := actor.contents;
        if (actor.wielding)
            ret -= actor.wielding;
        return(ret);
    }
;

unwieldVerb: darkVerb
    sdesc = "unwield"
    verb = 'unwield' 'unready'
    doAction = 'Unwield'
    doDefault(actor, prep, io) = {
        if (actor.wielding)
            return ([actor.wielding]);
        return [];
    }
;

armor: clothingItem

    AC = 1 // by default, armor lowers the wearer's AC by 1

    /* Wear and unwear work the same as they do for regular
     * clothing, but they also modify the wearer's AC.
     * Wearing lowers the wearer's AC by the armor's AC value,
     * and unwearing raises the wearer's AC by the same value,
     * reversing the effect.
     */
    doWear(actor) =
    {
        /* 
         *   if the actor is not directly carrying it (in other words,
         *   it's either not in the player's inventory at all, or it's
         *   within a container within the actor's inventory), take it --
         *   the actor must be directly holding the object for this to
         *   succeed 
         */
        if (self.location != actor)
        {
            /* try taking it */
            "(First taking <<self.thedesc>>)\n";
            if (execCommand(actor, takeVerb, self) != 0)
                exit;

            /* 
             *   make certain it ended up where we want it - the command
             *   might have failed without actually indicating failure 
             */
            if (self.location != actor)
                exit;
        }

        /* wear it */
        "Okay, %you're% now wearing "; self.thedesc; ". ";
        self.isworn := true;
        actor.AC -= self.AC; // update the AC
    }

    doUnwear(actor) =
    {
        /* ensure that maximum bulk is not exceeded */
        if (addbulk(actor.contents) + self.bulk > actor.maxbulk)
        {
            "%You% could drop <<self.thedesc>>, but %your% hands are
            too full to carry <<self.itobjdesc>>. ";
        }
        else
        {
            /* doff the item */
            "Okay, %you're% no longer wearing "; self.thedesc; ". ";
            self.isworn := nil;
            actor.AC += self.AC; // update the AC
        }
    }
;

/* A verb the player can use to find out his current and max endurance:
 */
healthVerb: darkVerb
    verb = 'health' 'diagnose'
    action(actor) = {
        "You current endurance is <<actor.endurance>> out
        of <<actor.maxendurance>>.\n";
    }
;

/* A verb the player uses to rest after combat. This doubles the
 * chance that the player will heal, and it's more convincing than
 * using the "wait" verb for about the same purpose.
 */
restVerb: darkVerb
    verb = 'rest'
    action(actor) = {
        "You rest your tired bones for a moment.\n";
        actor.resting := true;
    }
;

chargeVerb: deepverb
    verb = 'charge'
    doAction = 'Charge'
    prepDefault = withPrep
    ioAction(withPrep) = 'ChargeWith'
;

defendVerb: deepverb
    verb = 'defend' 'parry'
    action(actor) = {
        if (actor.combatant)
            "You defend as best you can
            against <<actor.combatant.thedesc>>.\n";
        else
            "You are not being attacked at present, but you take a
            defensive posture.\n";
        actor.AC -= self.defendBonus;
        notify (actor, &defendExpiresFuse, 1);
    }
    defendBonus = 5
;

/*-------------------------------------------------------------------
 * Now the modifications to the regular library and std. Note that
 * you will want to check to make sure none of your other modules
 * and your own code do not modify the same stuff that we're modifying
 * here. If there is a conflict, you will need to orchestrate the
 * modifications in your own code.
 */

/*-------------------------------------------------------------------
 * Modofications to adv.t
 */

/* We modify the attack verb; we want it to be smart about figuring
 * out the weapon to use when the actor is wielding a weapon. Note
 * that it is possible to attack without a weapon.
 */
modify attackVerb
    ioDefault(actor, prep) = {
        local ret := [];
        if (actor.wielding)
            ret += actor.wielding;
        return ret;
    }
    doAction = 'Attack'
;

/* Next, we modify listcontgen(). Actually we use the 'replace'
 * command, since functions cannot be modified, only replaced.
 * Basically we copy the entire function, whole hog, and make
 * our small change.
 *
 * This modification is very simple and cosmetic. We add a message
 * "(wielded)" if an object is being wielded. This is similar to
 * the message "(providing light)".
 */

replace listcontgen: function(obj, flags, indent)
{
    local i, count, tot, list, cur, disptot, prefix_count;

    /*
     *   Get the list.  If the "obj" parameter is already a list, use it
     *   directly; otherwise, list the contents of the object. 
     */
    switch(datatype(obj))
    {
    case 2:
        /* it's an object - list its contents */
        list := obj.contents;

        /* 
         *   if the CHECKVIS flag is specified, check to make sure the
         *   contents of the object are visible; if they're not, don't
         *   list anything 
         */
        if ((flags & LCG_CHECKVIS) != 0)
        {
            local contvis;

            /* determine whether the contents are visible */
            contvis := (!isclass(obj, openable)
                        || (isclass(obj, openable) && obj.isopen)
                        || obj.contentsVisible);

            /* if they're not visible, don't list the contents */
            if (!contvis)
                return;
        }
        break;
        
    case 7:
        /* it's already a list */
        list := obj;
        break;

    default:
        /* ignore other types entirely */
        return;
    }

    /* count the items in the list */
    tot := length(list);

    /* we haven't displayed anything yet */
    count := 0;

    /* 
     *   Count how many items we're going to display -- this may be fewer
     *   than the number of items in the list, because some items might
     *   not be listed at all (isListed = nil), and we'll list each group
     *   of equivalent objects with a single display item (with a count of
     *   the number of equivalent objets) 
     */
    disptot := itemcnt(list);

    /* iterate through the list */
    for (i := 1 ; i <= tot ; ++i)
    {
        /* get the current object */
        cur := list[i];

        /* if this object is to be listed, figure out how to show it */
        if (cur.isListed)
        {
            /* presume there is only one such object (i.e., no equivalents) */
            prefix_count := 1;
            
            /*
             *   if this is one of more than one equivalent items, list it
             *   only if it's the first one, and show the number of such
             *   items along with the first one 
             */
            if (cur.isEquivalent)
            {
                local before, after;
                local j;
                local sc;

                /* get this object's superclass */
                sc := firstsc(cur);

                /* scan for other objects equivalent to this one */
                for (before := after := 0, j := 1 ; j <= tot ; ++j)
                {
                    if (isIndistinguishable(cur, list[j]))
                    {
                        if (j < i)
                        {
                            /*
                             *   note that objects precede this one, and
                             *   then look no further, since we're just
                             *   going to skip this item anyway
                             */
                            ++before;
                            break;
                        }
                        else
                            ++after;
                    }
                }
                
                /*
                 *   if there are multiple such objects, and this is the
                 *   first such object, list it with the count prefixed;
                 *   if there are multiple and this isn't the first one,
                 *   skip it; otherwise, go on as normal 
                 */
                if (before = 0)
                    prefix_count := after;
                else
                    continue;
            }

            /* display the appropriate separator before this item */
            if ((flags & LCG_TALL) != 0)
            {
                local j;
                
                /* tall listing - indent to the desired level */
                "\n";
                for (j := 1; j <= indent; ++j)
                    "\t";
            }
            else
            {
                /* 
                 *   "wide" (paragraph-style) listing - add a comma, and
                 *   possibly "and", if this isn't the first item
                 */
                if (count > 0)
                {
                    if (count+1 < disptot)
                        ", ";
                    else if (count = 1)
                        " and ";
                    else
                        ", and ";
                }
            }
            
            /* list the object, along with the number of such items */
            if (prefix_count = 1)
            {
                /* there's only one object - show the singular description */
                cur.adesc;
            }
            else
            {
                /* 
                 *   there are multiple equivalents for this object -
                 *   display the number of the items and the plural
                 *   description 
                 */
                sayPrefixCount(prefix_count); " ";
                cur.pluraldesc;
            }
            
            /* show any additional information about the item */
            if (cur.isworn)
                " (being worn)";
            if (cur.islamp and cur.islit)
                " (providing light)";

// here's the new stuff:

            if (cur = parserGetMe.wielding)
                " (wielded)";

// end new stuff

            /* increment the number of displayed items */
            ++count;

            /* 
             *   if this is a "tall" listing, and there's at least one
             *   item contained inside the current item, list the item's
             *   contents recursively, indented one level deeper 
             */
            if ((flags & LCG_RECURSE) != 0 && itemcnt(cur.contents) != 0)
            {
                /* 
                 *   if this is a "wide" listing, show the contents in
                 *   parentheses 
                 */
                if ((flags & LCG_TALL) = 0)
                {
                    if (cur.issurface)
                        " (upon which %you% see%s% ";
                    else
                        " (which contains ";
                }
                
                /* show the recursive listing, indented one level deeper */
                listcontgen(cur, flags, indent + 1);

                /* close the parenthetical, if we opened one */
                if ((flags & LCG_TALL) = 0)
                    ")";
            }
        }
    }
}

/* We modify room, so that monsters can immediately attack players when
 * the player enters the room. The monster normally isn't notified when
 * a player enters a room, but the room is. So we hook into the room's
 * notification.
 */

modify room
    enterRoom(actor) = {
        local i;
        inherited.enterRoom(actor);
        for (i := 1 ; i <= length(global.monsterlist) ; i++) {
            if (global.monsterlist[i].isIn(self))
                global.monsterlist[i].playerentered;
        }
    }
;

/*-------------------------------------------------------------------
 * modifications to std.t
 */

/* We change the 'Me' object from a (non-combatant) 'basicMe' to a
 * 'combatMe'.
 */

replace Me: combatMe
;

/* We print the player's endurance before the command prompt.
 */
replace commandPrompt: function(code)
{
    /* display the normal prompt */
    "\b";
    "E:<<parserGetMe.endurance>>/<<parserGetMe.maxendurance>>";
    if (parserGetMe.combatant)
        " O:<<parserGetMe.combatant.endurance>>/<<parserGetMe.combatant.maxendurance>>";
    "&gt;";

    /* 
     *   switch the font to TADS-Input - the standard text-only
     *   interpreter will simply ignore this, so we don't have to worry
     *   about whether we're using an HTML-enabled interpreter or not 
     */
    "<font face='TADS-Input'>";
}

/* We're modifying this to add a new daemon, healDaemon,
 * which moves monsters and heals them.
 */

replace init: function
{
#ifdef USE_HTML_STATUS
    /* 
     *   We're using the adv.t HTML-style status line - make sure the
     *   run-time version is recent enough to support this code.  (The
     *   status line code uses systemInfo to detect whether the run-time
     *   is HTML-enabled or not, which doesn't work properly before
     *   version 2.2.4.)  
     */
    if (systemInfo(__SYSINFO_SYSINFO) != true
        || systemInfo(__SYSINFO_VERSION) < '2.2.4')
    {
        "\b\b\(WARNING! This game requires the TADS run-time version
        2.2.4 or higher.  You appear to be using an older version of the
        TADS run-time.  You can still attempt to run this game, but the
        game's screen display may not work properly.  If you experience
        any problems, you should try upgrading to the latest version of
        the TADS run-time.\)\b\b";
    }
#endif

    /* perform common initializations */
    commonInit();
    
    // put introductory text here
    
    version.sdesc;                // display the game's name and version number

    setdaemon(turncount, nil);                 // start the turn counter daemon
    setdaemon(sleepDaemon, nil);                      // start the sleep daemon
    setdaemon(eatDaemon, nil);                       // start the hunger daemon

// new stuff:

    setdaemon(healDaemon, nil);

// end new stuff

    parserGetMe().location := startroom;     // move player to initial location
    startroom.lookAround(true);                      // show player where he is
    startroom.isseen := true;                  // note that we've seen the room
    scoreStatus(0, 0);                          // initialize the score display
}

/* We're modifying this to add a monsterlist, which works just like the
 * lamplist.
 */
replace preinit: function
{
    local o;
    
    global.lamplist := [];

// new stuff:

    global.monsterlist := [];

// end new stuff

    o := firstobj();
    while(o <> nil)
    {
        if (o.islamp)
            global.lamplist := global.lamplist + o;

// new stuff:

        if (isclass(o, monster))
            global.monsterlist := global.monsterlist + o;

// end new stuff

        o := nextobj(o);
    }
    initSearch();

#ifdef GAMEINFO_INCLUDED
    writeGameInfo(getGameInfo());
#endif
#ifdef COMBAT_RAP
    rBuildMap();
#endif
}

/* We add a monsterlist to the global object.
 */
modify global
    monsterlist = []
;

/* We tell the player his health when he types score.
 */
replace scoreRank: function
{
    "In a total of "; say(global.turnsofar);
    " turns, you have achieved a score of ";
    say(global.score); " points out of a possible ";
    say(global.maxscore); ".\b";
    healthVerb.action(parserGetMe);
}

