/* ex:set ts=4 sw=4:
 *
 * sysverbs.t:	they're watching you, you know!
 *
 * This routine modifies the game save/restore methods, the restart and
 * the undo methods so that the programmer can detect a player who is
 * trying to solve a puzzle by repetition rather than logic.
 *
 * In order to catch lazy players, declare an object which is class 
 * 'game_scrambler' and define it in the following four methods 
 *
 *	savedGame(here)=		{...}
 *	restoredGame(here)=		{...}
 *	undoMove(here)=			{...}
 *	restartedGame(here)=	{...}
 *
 * These methods will all be called after the standard verb processing 
 * associated with "SAVE", "RESTORE", "UNDO" and "RESTART".  The 'here'
 * parameter is set to true if the game_scrambler object is in the same
 * location as the player.
 *
 * You must declare a method for each of them or a default message will be
 * displayed, announcing a bug to the player.  This is largely to ensure
 * that you think about each of the cases and warn the player at "SAVE"
 * time that "RESTORE" won't necessarily help.
 *
 * This module is Copyright (c) 1994 Jeff Laing.  Permission to use any or all
 * of this code in other TADS games is granted provided credit is given in
 * your "CREDITS" command for this code and that you leave this copyright
 * message in the source whenever distributed.  Please make all changes in
 * an backward compatible manner (where possible) and make them available
 * for free (like I did ;-)
 */
sysverbsTag: versionTag,initialization
	id="$Id: sysverbs.t_v 1.3 1994/05/09 03:29:30 jeffl Exp jeffl $\n"
	func='system verbs'
	author='Jeff Laing'

	/*
	 * called by preinit()
	 */
	preinit_phase={
		local o;
		global.scrambler_list := [];

		// search for scramblers and remember them
		for (o:=firstobj(game_scrambler);o<>nil;o:=nextobj(o,game_scrambler)) {
			global.scrambler_list += o;
		}
	}

	/*
	 * called by init()
	 */
	init_phase={
		if (global.restarting)
			notify_scramblers(#restartedGame);
	}
;

/*
 * A game_scrambler is an object that gets notified whenever the player
 * tries to take back a step in the game.  Unscrupulous players can use the
 * "TAKE OPTION 1. UNDO. TAKE OPTION 2. UNDO. TAKE OPTION 3" to find the
 * answer to puzzles that should be solved by logic.  The game_scrambler
 * is notified whenever the player has tried this and can automatically
 * "move the tiger from door 1 to door 2" for example.
 *
 * The game_scramblers are collected during preinit() and are called *after*
 * the appropriate actions are taken.  If the programmer wishes to disable
 * the function completely, override the methods in the verb definitions
 * instead.
 */
class game_scrambler: object

	// called when player saves game
	saveGame(here)=
		"\b\(Bug: programmer forgot <<self.sdesc>>.saveGame\)\b"

	// called when player restores game
	restoreGame(here)=
		"\b\(Bug: programmer forgot <<self.sdesc>>.restoreGame\)\b"

	// called when player undoes a move
	undoMove(here)=
		"\b\(Bug: programmer forgot <<self.sdesc>>.undoMove\)\b"
	
	// called when player restarts the game
	restartGame(here)=
		"\b\(Bug: programmer forgot <<self.sdesc>>.restartGame\)\b"
;

/*
 * this function is used by the various verbs to call specific methods
 * within the scramblers.
 */
notify_scramblers: function( method )
{
	local obj,list;
	list := global.scrambler_list;	// copy the list
	while (length(list)>0) {		// while its non-null
		obj:=car(list);				// send its leader
		list := cdr(list);			// and then move to the next one
		obj.(method)				// a message (passed as a parameter)
				(obj.location=Me.location);	// with a flag
	}
}

/*
 * give the programmer a chance to warn the player about saving the game.
 */
modify saveVerb

	// seperate verb processing from functional methods
    action( actor ) = {
		self.saveGame(actor,askfile( 'File to save game in' ));
		abort;
	}

	// this is called from stringObj.doSave and from saveVerb.action
    saveGame(a,savefile) = {
		if (savefile = nil or savefile = '') {
			"\b[Save cancelled]\b";
			return(nil);
		}

		if (save( savefile )) {
			"\b[Save as \"<<savefile>>\" failed]\b";
			return(nil);
		}

		"\b[Saved as \"<<savefile>>\"]\b";
		notify_scramblers(#savedGame);
		return(true);
    }
;

/*
 * give the programmer a chance to warn the player about restoring the game.
 */
modify restoreVerb

	// seperate verb processing from functional methods
    action( actor ) = {
		self.restoreGame(actor,askfile( 'File to restore game from' ));
		abort;
	}

	// this is called from stringObj.doSave and from saveVerb.action
    restoreGame(a,savefile) = {
		if (savefile = nil or savefile = '') {
			"\b[Restore cancelled]\b";
			return(nil);
		}

		if (restore( savefile )) {
			"\b[Restore \"<<savefile>>\" failed]\b";
			return(nil);
		}

		"\b[Restored \"<<savefile>>\"]\b";
		Me.location.lookAround(true);
		scoreStatus(global.score, global.turnsofar);
		notify_scramblers(#restoredGame);
		return(true);
	}
;

/*
 * modify the base string class so that SAVE "xxx" and RESTORE "xxx" also
 * use our bottlenecked code rather than their own little copy.
 */
modify class basicStrObj

	// redirect SAVE "name" to the standard verb processor
    doSave( actor ) = {
		saveVerb.saveGame(actor,self.value);
		abort;
	}

	// redirect RESTORE "name" to the standard verb processor
    doRestore( actor ) = {
		restoreVerb.restoreGame(actor,self.value);
		abort;
    }
;

/*
 * "UNDO <thing>" is not legal
 */
modify class thing
	verDoUndo(a) =
		"You can undo a single move using \"UNDO\" or a number of moves
		by \"UNDO nn\". "
;

/*
 * we allow "UNDO nnn" to go back a number of turns.
 */
modify class basicNumObj

	// redirect UNDO 10 to the standard verb processor
	verDoUndo(actor) = {}
	doUndo(actor) = {
		undoVerb.undoMove(actor,self.value+1);
		abort;
	}
;

/*
 * give the programmer a chance to warn the player about taking back moves.
 */
modify undoVerb

	// seperate verb processing from functional methods
    action(actor) = {
		self.undoMove(actor,2);
		abort;
	}

	// allow undo with a parameter
	doAction = 'Undo'

	// perform a single undo.  by making this a seperate method, we make it
	// possible to undo a programmed number of times (we can be called from
	// death methods and we need to undo() just once rather than twice.
	undoMove(a,turns) = {

		local undone;
		undone := 0;				// reset number of moves undone
		while(undone < turns) {		// loop until caller satisfied
			if (undo()) {			// undo a move
				undone++;			//		if it works, remember it
				continue;			//		and loop
			}
			if (undone>1) break;	// if we did any, announce them
									// otherwise barf
			"\b[No more undo information is available]\b";
			return(nil);
		}

		// need condition in case this was a 'restart,quit,undo' move
		if (undone>1) "\b[Undone <<undone-1>> turn<<undone=2?'':'s'>>]\b";

		// show player where they are now
		Me.location.lookAround(true);
		scoreStatus(global.score, global.turnsofar);

		// and let scramblers know
		notify_scramblers(#undoMove);

		return(true);
    }
;

/*
 * give the programmer a chance to warn the player about restarting
 */
modify restartVerb

	// seperate verb processing from functional methods
    action( actor ) = {
		scoreobj.rank;
		"\bAre you sure you want to start over? (YES or NO) > ";
		if (yorn()=1)
			self.restartGame(actor);
		else
			"\nOkay.\n";
		abort;
    }

	restartGame(actor)={
		"\n";
		scoreStatus(0,0);
		restart(initRestart, global.initRestartParam);
	}
;

/*
 * give the programmer a chance to warn the player about quitting
 */
modify quitVerb
	// add another synonym
	verb='q'

	// seperate verb processing from functional methods
    action( actor ) = {
        local yesno;

        scoreobj.rank;
		"\bIf you wish, you can save before quitting.\b";
		while (1) {
			"\nDo you really wish to quit now? (Y=yes, N=no, S=save) > ";
			switch (substr(upper(input()),1,1)) {

			default:			// N (and anything else in fact)
				"Okay. ";
				abort;

			case 'S':			// Save then terminate
				if ( not
					saveVerb.saveGame(actor,askfile('\nFile to save game in'))
				) {
					continue;
				}

			case 'Y':
				self.quitGame(actor);
				abort;
			}
		}
    }

	quitGame(actor)={
		terminate();
		quit();
	}
;

/*
 * The jigsup() function is called when the player dies.  It tells the
 * player how well he has done (with his score), and asks if he'd
 * like to start over (the alternative being quitting the game).
 *
 * We can optionally offer the chance to 'undo' although this can be
 * disabled (allowing death-puzzles which can be solved by guessing to 
 * re-randomize themselves)
 */
jigsup: function(undo_ok)
{
    scoreobj.rank;

    "\bYou may restore a saved game, start over ";
	if (undo_ok) ", undo the current command ";
	"or quit.\n";
    while ( 1 ) {
        local resp;

		"\nPlease enter RESTORE, RESTART ";
		if (undo_ok) ", UNDO";
		" or QUIT: >";
        resp := upper(input());
        if ( resp = 'RESTORE' ) {
			if (restoreVerb.restoreGame(Me,askfile( 'File to restore'))) {
				abort;
			}
		} else if ( resp = 'RESTART' ) {
			restartVerb.restartGame(Me);
		} else if ( resp = 'QUIT' ) {
			quitVerb.quitGame(Me);
        } else if (undo_ok and resp = 'UNDO') {
			if (undoVerb.undoMove(Me,1)) {
				abort;
			}
		}
    }
}
