/* ex:set ts=4 sw=4:
 *
 * bodyfunc.t: bodily functions
 *
 * This module provides a slightly more complicated set of object classes
 * for body functions - eating, sleeping, etc
 *
 * This module is Copyright (c) 1994 Jeff Laing.  Permission to use any or all
 * of this code in other TADS games is granted provided credit is given in
 * your "CREDITS" command for this code and that you leave this copyright
 * message in the source whenever distributed.  Please make all changes in
 * a backward compatible manner (where possible) and make them available
 * for free (like I did ;-)
 */

sleepDaemon: function;
eatDaemon: function;
gotoSleep: function;

bodyfuncVersion : versionTag, initialization
	id="$Id: bodyfunc.t_v 1.1 1994/05/04 06:49:35 jeffl Exp jeffl $\n"
	author='Mike Roberts and Jeff Laing'
	func='bodily functions'

	// called during preinit()
	preinit_phase={}

	// called during init()
	init_phase={
		setdaemon( sleepDaemon, nil );                 // start the sleep daemon
		setdaemon( eatDaemon, nil );                  // start the hunger daemon
	}
;

/*
 * attach our attributes to the global dumping ground
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
modify global
    awakeTime = 0               // time that has elapsed since the player slept
    sleepTime = 400     // interval between sleeping times (longest time awake)

    lastMealTime = 0              // time that has elapsed since the player ate
    eatTime = 200         // interval between meals (longest time without food)
;

/*
 * we move the sleep action onto the actor itself rather than embedding it
 * in the verb definition itself.
 */
modify sleepVerb
    action( actor ) = (actor.sleep)
;

/*
 * add a new method to the actor so that *all* actors can sleep but in a
 * different way if required
 */
modify class actor
    sleep = {
        if ( self.cantSleep )
            "%You% %are% much too anxious worrying about %your% continued
            survival to fall asleep now. ";

        else if ( global.awakeTime+1 < global.sleepTime )
            "%You're% not tired. ";

        else if ( not ( self.location.isbed or self.location.ischair ))
            "I don't know about you, but I can never sleep
            standing up. %You% should find a nice comfortable
            bed somewhere. ";

        else {
            "%You% quickly drift%s% off into dreamland...\b";
            goToSleep();
        }
    }
;

/*
 *  fooditem: item
 *
 *  An object that can be eaten.  When eaten, the object is removed from
 *  the game, and global.lastMealTime is decremented by the
 *  foodvalue property.  By default, the foodvalue property
 *  is global.eatTime, which is the time between meals.  So, the
 *  default fooditem will last for one "nourishment interval."
 */
class fooditem: item
    doEat( actor ) =
    {
        "That was delicious! ";
        global.lastMealTime := global.lastMealTime - self.foodvalue;
        self.moveInto( nil );
    }
    foodvalue = { return( global.eatTime ); }
;

/*
 *   This function is a daemon, started by init(), that monitors how long
 *   it has been since the player slept.  It provides warnings for a while
 *   before the player gets completely exhausted, and causes the player
 *   to pass out and sleep when it has been too long.  The only penalty
 *   exacted if the player passes out is that he drops all his possessions.
 *   Some games might also wish to consider the effects of several hours
 *   having passed; for example, the time-without-food count might be
 *   increased accordingly.
 */
sleepDaemon: function( parm )
{
    local a, s;

    global.awakeTime := global.awakeTime + 1;
    a := global.awakeTime;
    s := global.sleepTime;

    if ( a = s or a = s+10 or a = s+20 )
        "\bYou're feeling a bit drowsy; you should find a
        comfortable place to sleep. ";
    else if ( a = s+25 or a = s+30 )
        "\bYou really should find someplace to sleep soon, or
        you'll probably pass out from exhaustion. ";
    else if ( a >= s+35 )
    {
      global.awakeTime := 0;
      if ( Me.location.isbed or Me.location.ischair )
      {
        "\bYou find yourself unable to stay awake any longer.
        Fortunately, you are ";
        if ( Me.location.isbed ) "on "; else "in ";
        Me.location.adesc; ", so you gently slip off into
        unconsciousness.
        \b* * * * *
        \bYou awake some time later, feeling refreshed. ";
      }
      else
      {
        local itemRem, thisItem;

        "\bYou find yourself unable to stay awake any longer.
        You pass out, falling to the ground.
        \b* * * * *
        \bYou awaken, feeling somewhat the worse for wear.
        You get up and dust yourself off. ";
        itemRem := Me.contents;
        while (car( itemRem ))
        {
            thisItem := car( itemRem );
            if ( not thisItem.isworn )
	        thisItem.moveInto( Me.location );
            itemRem := cdr( itemRem );
        }
      }
    }
}

/*
 *   This function is a daemon, set running by init(), which monitors how
 *   long it has been since the player has had anything to eat.  It will
 *   provide warnings for some time prior to the player's expiring from
 *   hunger, and will kill the player if he should go too long without
 *   heeding these warnings.
 */
eatDaemon: function( parm )
{
    local e, l;

    global.lastMealTime := global.lastMealTime + 1;
    e := global.eatTime;
    l := global.lastMealTime;

    if ( l = e or l = e+5 or l = e+10 )
        "\bYou're feeling a bit peckish. Perhaps it would be a good
        time to find something to eat. ";
    else if ( l = e+15 or l = e+20 or l = e+25 )
        "\bYou're feeling really hungry. You should find some food
        soon or you'll pass out from lack of nutrition. ";
    else if ( l=e+30 or l = e+35 )
        "\bYou can't go much longer without food. ";
    else if ( l >= e+40 )
    {
        "\bYou simply can't go on any longer without food. You perish from
        lack of nutrition. ";
        jigsup(true);
    }
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function
{
    "***\bYou wake up some time later, feeling refreshed. ";
}
