////////////////////////////////////////////////////////////////////////
//  
//  Inform.t Library File: Grammar 991214
//
//  Copyright (c) 1999 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the Inform.t library extension for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#include <declare.t>

#pragma C+

/*
 *  Modified for sense passing and object reactions.
 */
modify deepverb
    dorequires = [&scope_reachable]
    iorequires = [&scope_reachable]
    
    //
    //  Common checking mechanism that loops through the verb
    //  requirements list, passing path list (plist) and requirements to
    //  the blocksPath function to determine if path is valid for the
    //  given action. 
    //
    //  If the path is not valid (blocksPath returns the failing
    //  location) then we set obj.accessfailed to a list of the 
    //  scope filter and the failed location and return nil; otherwise
    //  it returns true.
    //
    checkRequires(plist, requires) = {
        local i, len, obj, failedloc, filter;
        local path = plist[2];
                
        len = length(path);
        obj = path[len];
        
        for (i = 1; i <= length(requires); ++i)
        {
            filter = requires[i];
            failedloc = blocksPath(plist, filter);
            if (failedloc)
            {
                obj.accessfailed = [ filter, failedloc ];
                return nil;
            }
        }
        return true;
    }
	/*	
	 *	preAction Processing: 
	 *
	 *	This loops through every object in the actor's
	 *	scope (default is every 'visible' object for the 
	 *  actor). It calls the preAction method for every object 
	 *  in the list, unless an object returns an exit, exitobj or abort, 
	 *  in which case we issue an exit/exitobj/abort and do not continue 
	 *  preAction processing.
	 */
    replace verbAction( actor, dobj, prep, iobj ) =
    {   
    	local c, o, r, verb = self, str;    	
    	local loc = scopeCeiling(actor, nil, &scope_visible);

#ifdef USE_COMMAND_MESSAGING
    	//
    	//  Turn outcapture off. Display was captured from multisdesc,
    	//  to eliminate obj.sdesc; ":"; style messages being printed
    	//  by the parser during multiple direct object handling.
    	//
    	//  Since global.stat may be nil (if no multiple-object handling
    	//  has occurred) we must explicitly check for a '0' value. Also 
    	//  we must set global.stat back to nil if we've captured
    	//  output.
    	//
    	if (global.stat == 0)
    	{
            str = outcapture(global.stat);
            global.stat = nil;
    	}

#endif  /* USE_COMMAND_MESSAGING */

	    command.actorPtr = actor;
	    command.verbptr = self;
	    command.dobjPtr = dobj;
	    command.prepPtr = prep;
	    command.iobjPtr = iobj;

    	//
    	//  ACTOR PREACTION
    	//
    	
        r = actor.actorPreAction(actor, verb, dobj, prep, iobj);
        if (r == EC_EXITOBJ) exitobj;
        if (r == EC_EXIT) exit;
        if (r == EC_ABORT) abort;
        
        //
        //  ACTOR SCOPECEILING LOCATION PREACTION
        //
        
        if (loc != actor )
        {
            r = loc.roomPreAction(actor, verb, dobj, prep, iobj);
            if (r == EC_EXITOBJ) exitobj;
            if (r == EC_EXIT) exit;
            if (r == EC_ABORT) abort;
        }
        
        //
        //  ACTOR SCOPE OBJECTS PREACTION
        //
        
  		c = scope( actor.location, nil, &scope_visible);
  		
  		if (c == nil)
  			c = scope( actor.location, nil, &scope_objtree);
  		
     	o = car( c );
     	while( o )
     	{
     		if ( o != actor 
     		&& o != loc 
     		&& o != dobj 
     		&& o != iobj )
       		{
       			r = o.scopePreAction( actor, verb, dobj, prep, iobj );
        		if (r == EC_EXITOBJ) exitobj;
        		if (r == EC_EXIT) exit;
        		if (r == EC_ABORT) abort;
       		}
         	c = cdr( c );
        	o = car( c );
    	}
    	
    	//
    	//  IOBJ PREACTION
    	//
    	
    	if (iobj)
        {
            r = iobj.ioPreAction(actor, verb, dobj, prep, iobj);
            if (r == EC_EXITOBJ) exitobj;
            if (r == EC_EXIT) exit;
            if (r == EC_ABORT) abort;
        }
        
    	//
    	//  DOBJ PREACTION
    	//
    	
        if (dobj)
        {
            r = dobj.doPreAction(actor, verb, dobj, prep, iobj);
            if (r == EC_EXITOBJ) exitobj;
            if (r == EC_EXIT) exit;
            if (r == EC_ABORT) abort;
        }
   	}
    //
    //  Use all parsed objects
    //
    replace validDoList(actor, prep, iobj) = {
        command.actorPtr    = actor;
        command.verbPtr     = self;
        command.iobjPtr     = iobj;
        command.prepPtr     = prep;
	    command.doWords     = objwords(1);
        return (nil);
    }
    
    //
    //  Build scope path, then check that path is valid
    //
    replace validDo(actor, obj, seqno) = {
        //
        //  If the target object is Floating, maintain the scope list.
        //
        if (isclass(obj, floatingItem)) maintainScopelist(obj);
        
        command.plist       = path(actor, obj);
        
        return (self.checkRequires(command.plist, self.dorequires));
    }
    
    //
    //  Use all parsed objects
    //
    replace validIoList(actor, prep, dobj) = {
        command.actorPtr    = actor;
        command.verbPtr     = self;
        command.dobjPtr     = dobj;
        command.prepPtr     = prep;
	    command.ioWords     = objwords(2);
        return (nil);
    }
    
    //
    //  Build scope path, then check that path is valid
    //
    replace validIo(actor, obj, seqno) = {
        //
        //  If the target object is Floating, maintain the scope list.
        //
        if (isclass(obj, floatingItem)) maintainScopelist(obj);

        command.plist       = path(actor, obj);
        
        return (self.checkRequires(command.plist, self.iorequires));
    }
    
	//
	//  This function is called when nothing matched by the
	//  do/io vocabulary is accessible.
	//
	cantReach(actor, dolist, iolist, prep) = {
		
	    local	i, list, obj, filter, failloc;
	    
		if (dolist != nil) {
			list = dolist;
			command.cantReachWords = objwords(1);
		}
		else {
			list = iolist;
			command.cantReachWords = objwords(2);
		}

        if (!actor.isactor)
        {
            "\^<<actor.thedesc>> <<actor.doesdesc>> not respond.";
            return;
        }
        if (!isclass(actor, movableActor))
        {
            "There's no point in talking to <<actor.sdesc>>.";
            return;
        }
        
        for (i = 1; i <= length(list); ++i)
        {
            obj     = list[i];
            filter  = obj.accessfailed[1];
            failloc = obj.accessfailed[2];
                                
            switch(filter)
            {
                case &scope_visible:
                    failloc.cantSee(actor);
                    break;
                case &scope_audible:
                    failloc.cantHear(actor);
                    break;
                case &scope_olfactory:
                    failloc.cantSmell(actor);
                    break;
                case &scope_reachable:
                    failloc.cantReach(actor);
                    break;
            }
        }
	}
;

/*----------------------------------------------------------------------
 *  VERBS FOR SENSE OF SIGHT
 *--------------------------------------------------------------------*/

class visibleVerb: deepverb
    dorequires = [ &scope_visible ]
;

replace lookVerb: visibleVerb
    verb = 'look' 'l' 'look around' 'l around'
    sdesc = "look"
    action(actor) = {
        actor.location.lookAround(actor, true, true);
    }
;
replace inspectVerb: visibleVerb
    verb = 'inspect' 'examine' 'look at' 'l at' 'x'
    sdesc = "inspect"
    doAction = 'Inspect'
;
replace lookInVerb: visibleVerb
    verb = 'look in' 'look on' 'l in' 'l on'
    sdesc = "look in"
    doAction = 'Lookin'
;
replace lookThruVerb: visibleVerb
    verb = 'look through' 'look thru' 'l through' 'l thru'
    sdesc = "look through"
    doAction = 'Lookthru'
;
replace lookUnderVerb: visibleVerb
    verb = 'look under' 'look beneath' 'l under' 'l beneath'
    sdesc = "look under"
    doAction = 'Lookunder'
;
replace lookBehindVerb: visibleVerb
    verb = 'look behind' 'l behind'
    sdesc = "look behind"
    doAction = 'Lookbehind'
;
replace searchVerb: visibleVerb
    verb = 'search'
    sdesc = "search"
    doAction = 'Search'
;

/*----------------------------------------------------------------------
 *  VERBS FOR SENSE OF HEARING
 *--------------------------------------------------------------------*/

class audibleVerb: darkVerb
    dorequires = [ &scope_audible ]
;
listenVerb: audibleVerb
    verb = 'listen'
    action(actor) = {
        actor.location.listenAround(actor, true, true);
    }
    sdesc = "listen"
;
listentoVerb: audibleVerb
    verb = 'listen to'
    doAction = 'Listento'
    sdesc = "listen to"
;

/*----------------------------------------------------------------------
 *  VERBS FOR THE SENSE OF SMELL
 *--------------------------------------------------------------------*/

class olfactoryVerb: darkVerb
    dorequires = [ &scope_olfactory ]
;
smellVerb: olfactoryVerb
    verb = 'smell'
    doAction = 'Smell'
    action(actor) = {
        actor.location.smellAround(actor, true, true);
    }
    sdesc = "smell"
;

/*----------------------------------------------------------------------
 *  VERBS FOR THE SENSE OF TOUCH & TASTE
 *--------------------------------------------------------------------*/

class tactileVerb: deepverb
    dorequires = [ &scope_reachable ]
;
replace touchVerb: tactileVerb
    verb = 'touch' 'feel'
    sdesc = "touch"
    doAction = 'Touch'
;
replace pokeVerb: tactileVerb
    verb = 'poke' 'jab'
    sdesc = "poke"
    doAction = 'Poke'
;
tasteVerb: tactileVerb
    verb = 'taste'
    doAction = 'Taste'
    sdesc = "taste"
;

/*
 *	class waitingVerb: darkVerb
 *
 *	This class initiates the Waiting Process. Only the fuses and 
 *	daemons are executed, allowing events in the game to unfold while
 * 	waiting. It will iterate through the process until (a) it has advanced
 *	timesys through timesys.waittime minutes, (b) the game
 *	is terminated through the actions of a daemon or fuse, (c)
 *	stopwaiting() is issued through either a daemon or fuse.
 */
class waitingVerb: darkVerb
	action( actor ) = 
    {
		local t, i, t1, t2, e1, e2;
		
		
		t = timesys.waittime;
		if ( actor )
		{
			if (timesys.waittime)
				"Time passes...";
			else
				"Time doesn't pass...";
		}
		i = timesys.timerate;
		timesys.timerate = 1;
		timesys.waiting = true;
		while ( t > 0 )
		{
			--t;
//			timesys.advance( nil );
			if ( timesys.informWait 
			or (t%i == 0 && t != 0))
			{
			    rundaemons();
			    runfuses();
			}

			endCommand_turns();		
		    endCommand_reactions( parserGetMe() );

			if ( ! timesys.waitqblock )
			{
				if ( t > 0 && timesys.waitquery )
				{
					timesys.waitquery = nil;
					"<P>Do you want to continue waiting? <Y/N> ";
					if ( yorn() )
					{ 
						timesys.waiting = true;
					}
					else
					{
						break;
					}
				}
				if ( ! timesys.waiting )
				{
					break;
				}
			}
 		}
		timesys.waiting = nil;
		timesys.waitquery = nil;
		timesys.waitqblock = nil;
		timesys.timerate = i;
	}
;

/* 
 *	waitVerb: waitingVerb
 *
 *	Allows us to wait until a certain time. For instance, we can
 *	"wait until 4:00 pm" or simply "wait until 4:00". 
 *
 *	Allows us to wait for a certain time. For instance, we can
 *	"wait for 10 minutes" or "wait 10 hours". The 'for' is optional. If
 *	neither minutes or hours is specified the default is minutes. 
 *
 *	Note that "wait" without any amount of time indicated uses the
 *	timesys.timerate value.
 */
replace waitVerb: waitingVerb
	sdesc = "wait"
	verb = 'wait' 'waitfor' 'waituntil' 'z'
	action( actor ) =
	{
		if ( timesys.waittime == 0 )
		{
		    local ret = parsetime(timesys.timestring);
		    
		    timesys.waittime = ret[2];
		}
		inherited.action( actor );
		timesys.waittime = 0;
		abort;
	}
; 

/*======================================================================
 *	BOARDING AND UNBOARDING VERBS
/*====================================================================*/

disembarkVerb: deepverb
    verb = 'disembark'
    sdesc = "disembark"
    doAction = 'Unboard'
    action(actor) = { askdo; }
    doDefault(actor, prep, io) =
    {
        if (actor.location and actor.location.location)
            return ([] + actor.location);
        else
            return [];
    }
;

replace getOffVerb: deepverb
    verb = 'getoff' 'getoffof'
    sdesc = "get off of"
    doAction = 'Getoff'
    action(actor) = { askdo; }
    doDefault(actor, prep, io) =
    {
        if (actor.location and actor.location.location)
            return ([] + actor.location);
        else
            return [];
    }
;
replace getOutVerb: deepverb
    verb = 'getout' 'getoutof'
    sdesc = "get out of"
    doAction = 'Getout'
    action(actor) = { askdo; }
    doDefault(actor, prep, io) =
    {
        if (actor.location and actor.location.location)
            return ([] + actor.location);
        else
            return [];
    }
;
replace boardVerb: deepverb
    verb = 'board'
    sdesc = "board"
    doAction = 'Board'
;
getinVerb: deepverb
    verb = 'getin' 'getinto'
    sdesc = "get in"
    doAction = 'Getin'
;
getonVerb: deepverb
    verb = 'geton' 'getonto'
    sdesc = "get on"
    doAction = 'Geton'
;

/*======================================================================
 *	TRAVEL VERB CLASS AND VERBS
 *
 *  The modifications allow for the separation of display-text and 
 *  state-change code from the direction's location pointer.
 *====================================================================*/    

/*
 *  travelVerb: darkVerb
 *  These verbs allow the player to move about.
 *  All travel verbs have the property isTravelVerb set true.
 */
replace class travelVerb: darkVerb
    isTravelVerb = true
    travelDir(actor) = {
        local aPtr = self.dirAction;
        local dPtr = self.dir;
        local ret = actor.location.(aPtr)(actor);
        "\n";
        if (ret == true) return nil;
        return actor.location.(dPtr); 
    }
;

replace dVerb: travelVerb
    verb = 'd' 'down' 'go down'
    dir = &down
    dirAction = &downAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "down"
;
replace eVerb: travelVerb
    verb = 'e' 'east' 'go east'
    dir = &east
    dirAction = &eastAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "east"
;
replace inVerb: travelVerb
    verb = 'in' 'go in' 'enter' 'go to' 'go into'
    doAction = 'Enter'
    ioAction(inPrep) = 'EnterIn'
    ioAction(onPrep) = 'EnterOn'
    ioAction(withPrep) = 'EnterWith'
    dir = &in
    dirAction = &inAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "enter"
;
replace nVerb: travelVerb
    verb = 'n' 'north' 'go north'
    dir = &north
    dirAction = &northAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "north"
;
replace neVerb: travelVerb
    verb = 'ne' 'northeast' 'go ne' 'go northeast'
    dir = &ne
    dirAction = &neAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "northeast"
;
replace nwVerb: travelVerb
    verb = 'nw' 'northwest' 'go nw' 'go northwest'
    dir = &nw
    dirAction = &nwAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "northwest"
;
replace outVerb: travelVerb
    verb = 'out' 'go out' 'exit' 'leave'
    doAction = 'Exit'
    dir = &out
    dirAction = &outAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "exit"
;
;
replace sVerb: travelVerb
    verb = 's' 'south' 'go south'
    dir = &south
    dirAction = &southAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "south"
;
replace seVerb: travelVerb
    verb = 'se' 'southeast' 'go se' 'go southeast'
    dir = &se
    dirAction = &seAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "southeast"
;
replace swVerb: travelVerb
    verb = 'sw' 'southwest' 'go sw' 'go southwest'
    dir = &sw
    dirAction = &swAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "southwest"
;
replace uVerb: travelVerb
    verb = 'u' 'up' 'go up'
    dir = &up
    dirAction = &upAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "up"
;
replace wVerb: travelVerb
    verb = 'w' 'west' 'go west'
    dir = &west
    dirAction = &westAction
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    sdesc = "west"
;

/*======================================================================
 *	SYSTEM VERBS
 *====================================================================*/    

/*
 * 	bannerVerb: sysverb
 *
 *	This is simply used as a toggle for the sample
 *	TimeSys displays, and allows us to move back
 *	and forth between TimeSys and classic TADS
 *	banners.
 */
bannerVerb: sysverb
	sdesc = "banner"
	verb = 'banner'
	action( actor ) =
	{
		if ( global.timeStatus == 2)
		{
			global.timeStatus = nil;
			"<P><B>";
			"[ Basic TimeSys banner display ]";
			"</B><P>";
		}
		else if ( global.timeStatus == nil )
		{
			global.timeStatus = 1;
			"<P><B>";
			"[ Advanced TimeSys banner display ]";
			"</B><P>";
		}
		else if ( global.timeStatus == 1 )
		{
			global.timeStatus = 2;
			"<P><B>";
			"[ Classic TADS banner display ]";
			"</B><P>";
		}
		abort;
	}
;

/*	
 *	timeVerb: sysverb
 *
 *	The time verb produces a display for those systems running a
 *	non-HTML run-time. This allows the game to display the time
 *	whether a banner is present or not.
 */
timeVerb: sysverb
	verb = 'time'
	action( actor ) =
	{
        "\b"; say(gettimesys( TS_DAY, timesys.day, nil ));
        if (global.timeStatus == 1)
        {
        	", "; say(gettimesys( TS_DATE, timesys.date, nil ));
        }
        " "; say(gettimesys( TS_TIME, timesys.time, nil ));
        " ";
        abort;
	}
;

/*
 *  MODIFED TO USE parserGetMe().location.senseAround()
 */
modify undoVerb
    replace undoMove(actor) = {
        /* do TWO undo's - one for this 'undo', one for previous command */
        if (undo() && undo())
        {
            "(Undoing one command)\b";
            parserGetMe().location.senseAround(actor, true);
            scoreStatus(global.score, global.turnsofar);
        }
        else
            "No more undo information is available. ";
    }
;

/*
 *  MODIFIED TO USE parserGetMe().location.senseAround()
 */
modify verboseVerb
    replace verboseMode(actor) =
    {
        "Okay, now in VERBOSE mode.\n";
        global.verbose = true;
        parserGetMe().location.senseAround(actor, true);
    }
;

#pragma C-
