////////////////////////////////////////////////////////////////////////
//  
//  Inform.t Library File: Function 991214
//
//  Copyright (c) 1999 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the Inform.t library extension for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#include <declare.t>

#pragma C+

////////////////////////////////////////////////////////////////////////
//
//  PARSER HOOKS
//
////////////////////////////////////////////////////////////////////////

/*
 *  MODIFICATION: add all preparse objects to global.preparseObjList.  
 *
 *   preinit() is called after compiling the game, before it is written
 *   to the binary game file.  It performs all the initialization that can
 *   be done statically before storing the game in the file, which speeds
 *   loading the game, since all this work has been done ahead of time.
 *
 *   This routine puts all lamp objects (those objects with islamp == true) into
 *   the list global.lamplist.  This list is consulted when determining whether
 *   a dark room contains any light sources.
 */
replace preinit: function
{
    local o;
    
    //
    //  Set up floatingList
    //
    initSearch();
    
    o = firstobj(preinitObj);
    while( o )
    {
    	o.piomethod;
        o = nextobj(o, preinitObj);
    }
}
;

/*
 *  preparse: function( cmd )
 *
 *  This replacement implements the processing of preparse objects. It
 *  allows for the definition of multiple preparse routines and is an
 *  adjunct to ACTOR GRAMMAR.
 */
replace preparse: function( cmd ) 
{
	local o, p, ret;
	p = global.preparseObjList;
	o = car( p );
	while( o )
	{
		ret = o.ppomethod( cmd );
		if ( ret != true ) return( ret );
		p = cdr( p );
		o = car( p );
	};
	return( true );
}
;

/*
 *  preCommand( actor, verb, dolist, prep, iobj )
 *
 *  This function is used to store command elements that may have come
 *  about through execCommand().
 */
preCommand: function( actor, verb, dolist, prep, iobj ) 
{
    command.actorPtr    = actor;
    command.verbPtr     = verb;
    command.dolistPtr   = dolist;
    command.prepPtr     = prep;
    command.iobjPtr     = iobj;
}

/*	
 *	postAction: function( actor, verb, dobj, prep, iobj, status )
 *
 *	This function loops through all the objects in the actor's
 *  scope. If any of the objects return true then we stop 
 *	processing with that object; otherwise we continue to the 
 *	next object.
 */
postAction: function( actor, verb, dobj, prep, iobj, status )
{
	local c, o, r, loc = scopeCeiling( actor, nil, &scope_visible );
	
	invokeMessagepump( 1 );

	if (status == EC_ABORT) return;
	
    //
	//  ACTOR SCOPE POSTACTION
	//
	
  	c = scope( actor.location, nil, &scope_visible);

  	if (c == nil)
  		c = scope( actor.location, nil, &scope_objtree);

	o = car( c );
    while( o )
	{
		if ( o != actor 
		&& o != loc 
		&& o != dobj 
		&& o != iobj )
     	{
     		r = o.scopePostAction( actor, verb, dobj, prep, iobj, 
     			    status );
         	if (r == EC_EXITOBJ) exitobj;
         	if (r == EC_EXIT) exit;
         	if (r == EC_ABORT) abort;
     	}
        c = cdr( c );
        o = car( c );
    }
    
    //
    //  ACTOR SCOPECEILING LOCATION POSTACTION
    //
    
    if (loc != actor)
    {
        
	    r = loc.roomPostAction( actor, verb, dobj, prep, iobj, status );
	    if (r == EC_EXITOBJ) exitobj;
	    if (r == EC_EXIT) exit;
	    if (r == EC_ABORT) abort;
    }
	
	//
	//  IOBJ POSTACTION
	//
	
    if (iobj)
    {
	    r = iobj.ioPostAction( actor, verb, dobj, prep, iobj, status );
	    if (r == EC_EXITOBJ) exitobj;
	    if (r == EC_EXIT) exit;
	    if (r == EC_ABORT) abort;
    }
	
	//
	//  DOBJ POSTACTION
	//
	
    if (dobj)
    {
	    r = dobj.doPostAction( actor, verb, dobj, prep, iobj, status );
	    if (r == EC_EXITOBJ) exitobj;
	    if (r == EC_EXIT) exit;
	    if (r == EC_ABORT) abort;
    }
}

/*
 *	endCommand: function( actor, verb, dobjList, prep, iobj, status )
 *	
 *	This function handles end_of_turn logic. 
 *	if status is not abort:
 *		a. calls incturn()
 *		b. updates global turnsofar
 *		c. advances the time
 *		d. performs actor.location.roomEndCommand() logic
 *      e. performs obj.scopeEndCommand() logic
 *      f. performs time-related events
 *	It then calls scoreStatus()
 *	
 */
endCommand: function( actor, verb, dobjList, prep, iobj, status )
{
	invokeMessagepump( 2 );
    if (status != EC_ABORT)
    {
        endCommand_turns();               
        endCommand_reactions( actor );
    }
	invokeMessagepump( 3 );
	scoreStatus( global.score, global.turnsofar );	       
    command.actorPtr    = parserGetMe();
    command.verbPtr     = nil;
    command.dolistPtr   = [];
    command.dobjPtr     = nil;
    command.prepPtr     = nil;
    command.iobjPtr     = nil;
    command.plist       = [];
}

endCommand_turns: function
{
	local c, o, loc; 

    // Fuses advanced
	incturn();
    // Turns Counter Incremented
	global.turnsofar = global.turnsofar + 1;
		
	// 24-hour Clock Advanced
	timesys.advance( nil );
}

endCommand_reactions: function( actor )
{
    local c, o, loc;
    
    //
	//  ACTOR LOCATION SCOPECEILING ENDCOMMAND
	//
		
    loc = scopeCeiling(actor, nil, &scope_visible);
    if (loc != actor)
        loc.roomEndCommand;
        
	//
	//  ACTOR SCOPE ENDCOMMAND 
	//
		
	c = scope(actor, nil, &scope_visible);
	o = car( c );
	while( o )
	{
	    if (o != actor 
		&& o != loc)
		    o.scopeEndCommand;
		        
	    c = cdr( c );
	    o = car( c );
	}
	
	//
	//  TIMESYS EVENTS
	//
	timesys.events;
}

////////////////////////////////////////////////////////////////////////
//
//  INITIALIZATION & FINALIZATION FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

/*
 *	Modification:
 *      commented out setdaemon for turncount 
 *                    setdaemon for sleepDaemon
 *                    setdaemon for eatDaemon
 *
 *   The init() function is run at the very beginning of the game.
 *   It should display the introductory text for the game, start
 *   any needed daemons and fuses, and move the player's actor ("Me")
 *   to the initial room, which defaults here to "startroom".
 */
replace init: function
{
#ifdef USE_HTML_STATUS
    /* 
     *   We're using the adv.t HTML-style status line - make sure the
     *   run-time version is recent enough to support this code.  (The
     *   status line code uses systemInfo to detect whether the run-time
     *   is HTML-enabled or not, which doesn't work properly before
     *   version 2.2.4.)  
     */
    if (systemInfo(__SYSINFO_SYSINFO) != true
        || systemInfo(__SYSINFO_VERSION) < '2.2.4')
    {
        "\b\b\(WARNING! This game requires the TADS run-time version
        2.2.4 or higher.  You appear to be using an older version of the
        TADS run-time.  You can still attempt to run this game, but the
        game's screen display may not work properly.  If you experience
        any problems, you should try upgrading to the latest version of
        the TADS run-time.\)\b\b";
    }
#endif
    
    story.main;
}
;

/*
 *   initRestore() - the system calls this function automatically at game
 *   startup if the player specifies a saved game to restore on the
 *   run-time command line (or through another appropriate
 *   system-dependent mechanism).  We'll simply restore the game in the
 *   normal way.
 */
replace initRestore: function(fname)
{
    /* perform common initializations */
    story.initialise_common;

    /* tell the player we're restoring a game */
    "\b[Restoring saved position...]\b";
    
    story.restore_introduction;

    /* go restore the game in the normal way */
    mainRestore(fname);
}
;

/*
 *  mainRestore: function
 *  
 *  MODIFIED TO CALL parserGetMe().location.senseAround(), INSTEAD
 *  OF parserGetMe().location.lookAround(actor, true, true).
 *
 *  Restore a saved game.  Returns true on success, nil on failure.  If
 *  the game is restored successfully, we'll update the status line and
 *  show the full description of the current location.  The restore command
 *  uses this function; this can also be used by initRestore (which must be
 *  separately defined by the game, since it's not defined in this file).
 */
replace mainRestore: function(fname)
{
    /* try restoring the game */
    switch(restore(fname))
    {
    case RESTORE_SUCCESS:
        /* update the status line */
        scoreStatus(global.score, global.turnsofar);

        /* tell the user we succeeded, and show the location */
        "Restored.\b";
        parserGetMe().location.senseAround(parserGetMe(), true);
        
        /* success */
        return true;

    case RESTORE_FILE_NOT_FOUND:
        "The saved position file could not be opened. ";
        return nil;

    case RESTORE_NOT_SAVE_FILE:
        "This file does not contain a saved game position. ";
        return nil;

    case RESTORE_BAD_FMT_VSN:
        "This file was saved by another version of TADS that is
        not compatible with the current version. ";
        return nil;

    case RESTORE_BAD_GAME_VSN:
        "This file was saved by a different game, or a different
        version of this game, and cannot be restored with this game. ";
        return nil;

    case RESTORE_READ_ERROR:
        "An error occurred reading the saved position file; the file
        might be corrupted.  The game might have been partially restored,
        which could make it impossible to continue playing; if the game
        does not seem to be working properly, you should quit the game
        and start over. ";
        return nil;

    case RESTORE_NO_PARAM_FILE:
        /* 
         *   ignore the error, since the caller was only asking, and
         *   return nil 
         */
        return nil;

    default:
        "Restore failed. ";
        return nil;
    }
}
;

/*
 *   The win() function is called when the player wins.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */
win: function
{
    "\b*** You have died ***\b";
    scoreRank();
    askContinue();
}

/*
 *  MODIFIED The continuation question has been broken out into
 *  askContinue.
 *
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */
replace die: function
{
    "\b*** You have died ***\b";
    scoreRank();
    askContinue();
}

askContinue: function
{
    "\bYou may restore a saved game, start over, quit, or undo
    the current command.\n";
    while (true)
    {
        local resp;

        "\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
        resp = upper(input());
        if (resp == 'RESTORE')
        {
            resp = askfile('File to restore',
                            ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE);
            if (resp == nil)
                "Restore failed. ";
            else if (restore(resp))
                "Restore failed. ";
            else
            {
                parserGetMe().location.senseAround(parserGetMe(), true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
        }
        else if (resp == 'RESTART')
        {
            scoreStatus(0, 0);
            restart();
        }
        else if (resp == 'QUIT')
        {
            terminate();
            quit();
            abort;
        }
        else if (resp == 'UNDO')
        {
            if (undo())
            {
                "(Undoing one command)\b";
                parserGetMe().location.senseAround(parserGetMe(), true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
            else
                "Sorry, no undo information is available. ";
        }
    }
}
;

/*
 *	scoreRank: function
 *
 *	The TimeSys scoreRank is replaced here to
 *	demonstrate the toggling of feature of bannerVerb
 *	and to allow games to be SCORED in either turns
 *	or time.
 */
replace scoreRank: function
{
	if ( global.timeStatus == nil or global.timeStatus == 1)
	{
		"In a total of "; say( gettimesys( TS_ELAPSE, timesys.elapsetime, nil ) );
		", you have achieved a score of ";
		say( global.score ); " points out of a possible ";
		say( global.maxscore ); ".\n";
	}
	else
	{
   		 "In a total of "; say(global.turnsofar);
   		 " turns, you have achieved a score of ";
    		say(global.score); " points out of a possible ";
    		say(global.maxscore); ".\n";
	}
}
;

////////////////////////////////////////////////////////////////////////
//  
//  SCOPE FILTER FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

scope_objtree: function(o) { return (true); }

scope_visible: function(o) { return (o.contentsVisible); }

scope_audible: function(o) { return (o.contentsAudible); }

scope_olfactory: function(o) { return (o.contentsOlfactory); }

scope_reachable: function(o) { return (o.contentsReachable); }

////////////////////////////////////////////////////////////////////////
//  
//  SCOPE ROOM LISTING FILTER FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

scope_qVisible: function(o) {
    return (scope_visible(o)
        && !o.isqVisible);
}

scope_qAudible: function(o) { 
    return (scope_audible(o) 
        && !o.isqAudible); 
}

scope_qOlfactory: function(o) { 
    return (scope_olfactory(o) 
        && !o.isqOlfactory); 
}

////////////////////////////////////////////////////////////////////////
//
//  SCOPE LIST FUNCTIONS: 
//
//  These functions produce a "bubble-up/filter-down" style list of 
//  all objects valid for the given scope filter.
//
////////////////////////////////////////////////////////////////////////

/*
 *  inScope: function( vantage, target, limit, filter )
 *
 *  Function returns true if the target is within the scope list
 *  produced for the given scope level for the vantage; otherwise
 *  it returns nil.
 */
inScope: function( vantage, target, limit, filter )
{	
	local value = scope( vantage, limit, filter );
	if ( find( value, target ) ) 
	    return true;
	return nil;
}
    
/*
 *	scope: function( vantage, limit, filter )
 *
 *	Function returns a list of all objects that are valid for a given vantage
 *	and scope filter. 
 *
 *	First it determines the scopeCeiling for a given vantage and scope filter. 
 *	This becomes the first entry in the list. 
 *
 *  limit can be useful in cases where the author wishes to limit the
 *  scope to within a select location, even though the scope filter might
 *  extend beyond the location.
 *
 *	Next it calls scopeList with the location, vantage, scope ceiling, and 
 *	scope filter and adds the return to the list. 
 *
 *	Next each object in global.floatingList is checked using scopeList and 
 *	the results are added to the list.
 */
scope: function( vantage, limit, filter )
{
    local c, o, loc, ceiling, lst = [];
 
    ceiling = scopeCeiling( vantage, limit, filter );
        
    loc = ceiling;

    lst += ceiling;
    lst += scopeList( loc, vantage, ceiling, filter );

    /*
     *  Each floatingItem object is evaluated. If it has the same
     *  scope ceiling as the vantage or its scope ceiling is the
     *  vantage then it is added to scope along with any appropriate
     *  contents.
     */
	c = global.floatingList;
	o = car( c );
	while( o )
	{
		loc = o;
		ceiling = scopeCeiling( o, limit, filter );
		if ( ceiling == lst[1]
		|| ceiling == vantage)
		{
			lst += o;
			if (isScopedef(loc, filter))
			    lst += scopeList( loc, o, ceiling, filter );
		}
		
		c = cdr( c );
		o = car( c );
	}
    
    return lst;
}

/*
 *  scopeList: function( loc, vantage, ceiling, filter )
 *
 *  Function returns a list of all valid objects for a given vantage, scope
 *	ceiling and scope filter. It recurses down each object in the location's
 *	contents.
 */
scopeList: function( loc, vantage, ceiling, filter )
{
    local ret = [];
    local i, lst, len;

    /* don't look in "nil" objects */
    if (loc == nil)
        return ret;

	/*
	 *	The location is added to the list if,
	 *		a. it is the vantage or the scope ceiling
	 *		b. it meets the requirements of a scope definition for the given
	 *		   scope filter.
	 */
	if (loc == vantage 
	||  loc == ceiling
	||  isScopedef(loc, filter))
    {
        lst = loc.contents;
        len = length(lst);
        ret += lst;
        for (i = 1 ; i <= len ; ++i)
        {
            /* recurse into this object */
      		ret += scopeList( lst[i], vantage, ceiling, filter );
        }
    }

    return ret;
}

/* 
 *	scopeCeiling: function( vantage, limit, filter )
 *
 *	Function returns the scopeCeiling for a given vantage and scope filter.
 *	The scopeCeiling is the highest level location that the vantage has access
 *	to for the given scope filter.
 *
 *  limit acts to constrain the function a given scope ceiling "height",
 *  but the function will always return a valid scope ceiling for a
 *  valid location.
 *
 */
scopeCeiling: function( vantage, limit, filter )
{
    local loc;
    
    if (vantage == limit )
    	return vantage;
    	
    if (vantage.location)
    	loc = vantage.location;
    else
    	loc = vantage;
    	
    while( loc && loc.location ) 
   	{
   	    if ( loc == limit )
   	        break;
   	        
   	    if (!isScopedef(loc, filter) ) 
            break;
                        
        loc = loc.location;
    }
    return loc;
}

/*
 *  isScopedef: function( loc, filter )
 *
 *  Function returns true if the location fits the specified scope
 *  filter definition; otherwise it returns nil. This allows scope-
 *  related definitions to be maintained outside of the scoping
 *  functions.
 */
isScopedef: function( loc, filter )
{
    switch( datatype(filter) )
    {
        case DTY_TRUE: 
            return true;
        case DTY_FUNCPTR:
            return (filter(loc));
        case DTY_PROP:
            return (loc.(filter));
        default:
            return nil;
    }
}

////////////////////////////////////////////////////////////////////////
//  
//  SCOPE PATH FUNCTIONS: 
//
//  These functions produce a PATH list of locations connecting the
//  vantage to the object. They also validate this PATH for each
//  location along the PATH.
//
////////////////////////////////////////////////////////////////////////

/*
 *  path( vantage, target )
 *
 *  Function builds the plist for the vantage and the target.
 *  The plist is the logical path along which action is to flow,
 *  not necessarily the valid path for the action, which is then
 *  determined by the blocksPath() function.
 *
 *  The plist consists of two element lists:
 *      a. lowest common containment (or nil, if none)
 *      b. the logical path 
 */
path: function( vantage, target )
{
    local i, f, loc, lcc;
    local vlist = [], olist = [], nlist = [], path = [];
     
    //
    //  Get the containment lists for both vantage and target.
    //
    vlist = vantage.clist;
    olist = target.clist;
    
    //
    //  intersect the vlist and olist. The first element
    //  should be the lowest common containment (lcc).
    //
    nlist = intersect(vlist, olist);
    lcc = car(nlist);
     
    //
    //  First we build the path by walking up the vantage
    //  containment list (vlist) until we either reach the lowest common
    //  containment (lcc) or the end of the list.
    //
    if (lcc) 
        f = find(vlist, lcc);
    else
        f = length(vlist);
        
    for (i = 1; i <= f; ++i)
        path += vlist[i];
     
    //
    //  Continue to build the path by walking down the
    //  target list (olist). We start at the object just below the
    //  lowest common containment (lcc), if it exists; otherwise we
    //  start from the end of the olist.
    //
    if (lcc)
        f = find(olist, lcc) - 1;
    else
        f = length(olist);
            
    for (i = f; i > 0; --i)
        path += olist[i];
 
    return [lcc path ];
}

/*
 *  blocksPath( plist, filter )
 *
 *  Checks every object along the scope path to determine if access is
 *  valid for the given action/scope filter. It returns the object that
 *  fails the checks; otherwise it returns nil. 
 *
 *  This function determines how objects are to be checked:
 *
 *      - vantage: passToLocation()
 *      - other vlist objects: passToLocation, isScopedef()
 *      - lowest common containment (lcc): passAcrossContents()
 *      - other olist objects: passToContents(), isScopedef()
 *      - target: passToContents()
 *
 */
blocksPath: function( plist, filter )
{
    local i, len, vantage, target;
    local lcc, path;
    
    lcc     = plist[1];
    path    = plist[2];
    
    len     = length(path);
    vantage = path[1];
    target  = path[len];
            
    for (i = 1; i <= len; ++i)
    {
        if (path[i] == vantage
        && path[i] == target)
        {
            if (!path[i].passAcrossContents(plist, filter))
                return path[i];
        }
        else if (path[i] == vantage)
        {
            if (!path[i].passToLocation(plist, filter))
                return path[i];
        }
        else if (path[i] == lcc)
        {
            if (!path[i].passAcrossContents(plist, filter))
                return path[i];
        }
        else if (path[i] == target)
        {
            if (!path[i].passToContents(plist, filter))
                return path[i];
        }
        else if (find(vantage.clist, path[i]))
        {
            if (!(path[i].passToLocation(plist, filter)
            && isScopedef(path[i], filter)))
                return path[i];
        }
        else if (find(target.clist, path[i]))
        {
            if (!(path[i].passToContents(plist, filter)
            && isScopedef(path[i], filter)))
                return path[i];
        }
    }
        
    return nil;
}

/*
 *  maintainScopelist(o)
 *  
 *  This function is called from preinit(), moveInto(), and travelTo()
 *  in order to keep the containment list and top-level list current.
 */
maintainScopelist: function(o)
{
    local i, list, f;
    
    //
    //  Get the scopelist for this object, so that we build the clist
    //  for it and all of its contents.
    //
    list = scope(o, o, &scope_objtree);
    
    for (i = 1; i <= length(list); ++i)
    {
        //
        //  Build the containment list for the object. This list includes
        //  the object itself and the top-level location.
        //
        list[i].clist = buildClist(list[i]);
    }
    
    //
    //  Add the object to the global.tlist if it is a top-level location
    //  and it isn't in the list already. It removes the object from the 
    //  global.tlist if it is on the list and it is no longer a 
    //  top-level location. 
    //
    if (o.location == nil
    && isclass(o, room)
    && !isclass(o, floatingItem))
    {
        f = find(global.tlist, o);
    
        if (o.location == nil)
        {
            if (f == nil)
            {
                global.tlist += o;
            }
        }
        else
        {
            if (f)
            {
                global.tlist -= o;
            }
        }
    }
}

/*
 *  buildClist(o)
 *
 *  returns a containment list for the object.
 */
buildClist: function(o)
{
    local loc, tmpList = [];
    
    loc = o;
    while(loc)
    {
        tmpList += loc;
        loc = loc.location;
    }
    return tmpList;    
}

/*
 *  getPlist(list, flag)
 *
 *  Function returns the object or list corresponding to the PTH
 *  constant passed. It returns an object for the lcc, vantage, or
 *  obj. It returns a list for the tloc or path.
 *
 *  list is optional. If no list is provided then the function defaults
 *  to command.plist.
 *
 *  flag should be a PTH constant.
 */
getPlist: function(...)
{
    local flag, len, lcc, vantage, obj, t1, t2;
    local list = command.plist;
    
    switch(argcount)
    {
        case 1:
            flag = getarg(1);
            break;
        case 2:
            list = getarg(1);
            flag = getarg(2);
            break;
        default:
            return (nil);
    }
    
    switch(flag)
    {
        case PTH_LCC:
            return list[1];
        case PTH_PATH:
            return list[2];
        case PTH_VANTAGE:
            return list[2][1];
        case PTH_TARGET:
            len = length(list[2]);
            return list[2][len];
        case PTH_TLOC:            
            lcc = getPlist(list, PTH_LCC);
            
            if (lcc)
                return [lcc];
                
            vantage = getPlist(list, PTH_VANTAGE);
            len     = length(vantage.clist);
            t1      = vantage.clist[len];
            
            obj     = getPlist(list, PTH_TARGET);
            len     = length(obj.clist);
            t2      = obj.clist[len];
            
            return [t1, t2];
        default:
            return (nil);   
    }
}

/*
 *  inPlist(list, o, flag)
 *
 *  Function determines if o is found in the specified part of the
 *  plist. If o is found then the function returns true; otherwise it
 *  returns nil.
 *
 *  The both the list and flag arguments are optional. If no list is
 *  provided the function defaults to command.plist.
 *
 *  The flags parameter should be a PTH constant. If no flag is provided
 *  the function defaults to PTH_PATH.
 */
inPlist: function(...)
{
    local arg1, arg2, arg3;
    local list, o, flag, p;
    
    switch(argcount)
    {
        case 3:
            arg3 = getarg(3);
        case 2:
            arg2 = getarg(2);
        case 1:
            arg1 = getarg(1);
            break;
        default:
            return (nil);
    }
    switch(argcount)
    {
        case 3:
            list    = arg1;
            o       = arg2;
            flag    = arg3;
            break;
        case 2:
            switch(datatype(arg1))
            {
                case DTY_LIST:
                    list    = arg1;
                    o       = arg2;
                    flag    = PTH_PATH;
                    break;
                case DTY_OBJECT:
                    list    = command.plist;
                    o       = (arg1);
                    flag    = PTH_PATH;
                    break;
                default:
                    return (nil);
            }
            break;
        case 1:
            list    = command.plist;
            o       = arg1;
            flag    = PTH_PATH;
            break;
        default:
            return (nil);
    }
    
    if (datatype(list) != DTY_LIST) return (nil);
    if (datatype(o) != DTY_OBJECT) return (nil);
    if (datatype(flag) != DTY_NUMBER) return (nil);
           
    p = getPlist(list, flag);
    
    if (p == nil) return (nil);
    
    switch(flag)
    {
        case PTH_LCC:
        case PTH_VANTAGE:
        case PTH_TARGET:
            if (o == p) 
                return (true);
            else
                return (nil);
        case PTH_PATH:
        case PTH_TLOC:
            if (find(p, o))
                return (true);
            else
                return (nil);
        default:
            return (nil);
    }
}

/*
 *  proximity(vantage, target)
 *
 *  Function returns a list of two elements. The first is a proximity
 *  flag PRX_XXXX and the second is a list of all the scope filters
 *  that are valid for the path from vantage to target.
 */
proximity: function(vantage, target)
{
    local i, plist, filter, lcc, prx_flag, vsList = [];
    local filterList = [ &scope_visible, &scope_audible, &scope_olfactory,
        &scope_reachable ];
    
    plist = path(vantage, target);
    
    lcc = getPlist(plist, PTH_LCC);
    
    if (vantage == target)
        prx_flag = PRX_REFLEXIVE;
    else if (vantage == lcc)
        prx_flag = PRX_POSSESSIVE;
    else if (target == lcc)
        prx_flag = PRX_CONTAINED;
    else if (lcc && find(global.tlist, lcc))
        prx_flag = PRX_LOCAL;
    else if (lcc)
        prx_flag = PRX_IMMEDIATE;
    else 
        prx_flag = PRX_REMOTE;
    
    for (i = 1; i <= length(filterList); ++i)
    {
        filter = filterList[i];
        
        if (blocksPath(plist, filter) == nil)
            vsList += filter;
    }
    return [ prx_flag vsList ];
}

////////////////////////////////////////////////////////////////////////
//  
//  DISPLAY-FORMATTING & LISTING-RELATED FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

/*
 *  printWysiwyg: function(list)
 *
 *  This function prints a list of strings, preventing the spacing
 *  compression normal with TADS. This allows for the production of
 *	accurate maps and signs.
 */
printWysiwyg: function(list)
{
    local i;
    for (i = 1; i <= length(list); ++i)
    {
#ifdef USE_HTML_STATUS
        if (systemInfo(__SYSINFO_SYSINFO) == true
        && systemInfo(__SYSINFO_HTML) == 1)
            "<BR>"; 
        else
#endif /* USE_HTML_STATUS */
            "\n";
             
        say(nocompress(list[i]));
    }  
}

/*
 *	nocompress: function( str )
 *
 *	A function for eliminating the compression of spaces, especially useful for
 *	printing room maps. It takes a (single-quote) string as its parameter and 
 *	returns a string which has the spaces treated to eliminate TADS natural 
 *	compression.
 */
nocompress: function( str )
{
	local i, char, newstr = '';
	for (i = 1; i <= length(str); ++i)
	{
		char = substr(str,i,1);
		if (char == ' ')
		{
#ifdef USE_HTML_STATUS
            if (systemInfo(__SYSINFO_SYSINFO) == true
            && systemInfo(__SYSINFO_HTML) == 1)
                newstr += '&nbsp';
            else
#endif /* USE_HTML_STATUS */
			    newstr += '\ ';		// equivalent to '&nbsp'
		}
		else
			newstr += char;
	}
	return newstr;
}

listTogether: function(list)
{
    local i, j, k, m = 1, x, value;
    local matchfound, lt_save, disptot = 0;
    local newList = [], tmpList = [], delList = [];
create_obj: ;
    x = new listobj;
    x.list_together = lt_save;
    newList += x;
    for (i = m; i <= length(list); ++i)
    {
        matchfound = nil;
        value = list[i].list_together;
        for (j = 1; j <= length(newList); ++j)
        {
            if (newList[j].list_together == value)
            {
                matchfound = true;
                newList[j].contents += list[i];;
            }
        }
        if (!matchfound)
        {
            m = i;
            lt_save = value;
            goto create_obj;
        }
    }
    for (k = length(newList); k > 0; --k)
    {
        if (length(newList[k].contents) == 0)
            delList += newList[k];
        list = newList[k].contents;
        disptot += itemcnt(list);
        tmpList += newList[k];
    }
    newList = [disptot] + tmpList - delList;
    
    return newList;
}

/*
 *  MODIFIED TO USE LISTDESC
 *
 *  listcontgen: function(obj, flags, indent)
 *
 *  This is a general-purpose object lister routine; the other object lister
 *  routines call this routine to do their work.  This function can take an
 *  object, in which case it lists the contents of the object, or it can take
 *  a list, in which case it simply lists the items in the list.  The flags
 *  parameter specifies what to do.  LCG_TALL makes the function display a "tall"
 *  listing, with one object per line; if LCG_TALL isn't specified, the function
 *  displays a "wide" listing, showing the objects as a comma-separated list.
 *  When LCG_TALL is specified, the indent parameter indicates how many
 *  tab levels to indent the current level of listing, allowing recursive calls
 *  to display the contents of listed objects.  LCG_CHECKVIS makes the function
 *  check the visibility of the top-level object before listing its contents.
 *  LCG_RECURSE indicates that we should recursively list the contents of the
 *  objects we list; we'll use the same flags on recursive calls, and use one
 *  higher indent level.  To specify multiple flags, combine them with the
 *  bitwise-or (|) operator.
 */
replace listcontgen: function(obj, flags, indent)
{
    local i, count, tot, list, cur, disptot, prefix_count;
    local k, o, num, newList;

    /*
     *   Get the list.  If the "obj" parameter is already a list, use it
     *   directly; otherwise, list the contents of the object. 
     */
    switch(datatype(obj))
    {
    case 2:
        /* it's an object - list its contents */
        list = obj.contents;

        /* 
         *   if the CHECKVIS flag is specified, check to make sure the
         *   contents of the object are visible; if they're not, don't
         *   list anything 
         */
        if ((flags & LCG_CHECKVIS) != 0)
        {
            local contvis;

            /* determine whether the contents are visible */
            contvis = (!isclass(obj, openable)
                        || (isclass(obj, openable) && obj.isopen)
                        || obj.contentsVisible);

            /* if they're not visible, don't list the contents */
            if (!contvis)
                return;
        }
        break;
        
    case 7:
        /* it's already a list */
        list = obj;
        break;

    default:
        /* ignore other types entirely */
        return;
    }

    newList = listTogether(list);
        

    /* 
     *   Count how many items we're going to display -- this may be fewer
     *   than the number of items in the list, because some items might
     *   not be listed at all (isListed = nil), and we'll list each group
     *   of equivalent objects with a single display item (with a count of
     *   the number of equivalent objets) 
     */
    disptot = car(newList);
    newList = cdr(newList);

    /* we haven't displayed anything yet */
    count = 0;

    for (k = 1; k <= length(newList); ++k)
    {
        list = newList[k].contents;
        
        /* count the items in the list */
        tot = length(list);   
        
        if (proptype(newList[k], &list_together) == DTY_SSTRING)
        {
            num = length(newList[k].contents);
            if (num > 1)
            {
                if ((flags & LCG_TALL) != 0)
                {
                    local j;
            
                    /* wide listing - indent to the desired level */
                    "\n";
                    for (j = 1; j <= indent; ++j)
                        "\t";
                }
                sayPrefixCount(num);
                " ";
                say(newList[k].list_together);
                if ((flags & LCG_TALL) == 0)
                    " (";
            }
        }

        /* iterate through the list */
        for (i = 1 ; i <= tot ; ++i)
        {
            /* get the current object */
            cur = list[i];

            /* if this object is to be listed, figure out how to show it */
            if (cur.isListed)
            {
                /* presume there is only one such object (i.e., no equivalents) */
                prefix_count = 1;
            
                /*
                 *   if this is one of more than one equivalent items, list it
                 *   only if it's the first one, and show the number of such
                 *   items along with the first one 
                 */
                if (cur.isEquivalent)
                {
                    local before, after;
                    local j;
                    local sc;

                    /* get this object's superclass */
                    sc = firstsc(cur);

                    /* scan for other objects equivalent to this one */
                    for (before = after = 0, j = 1 ; j <= tot ; ++j)
                    {
                        if (isIndistinguishable(cur, list[j]))
                        {
                            if (j < i)
                            {
                                /*
                                 *   note that objects precede this one, and
                                 *   then look no further, since we're just
                                 *   going to skip this item anyway
                                 */
                                ++before;
                                break;
                            }
                            else
                                ++after;
                        }
                    }
                
                    /*
                     *   if there are multiple such objects, and this is the
                     *   first such object, list it with the count prefixed;
                     *   if there are multiple and this isn't the first one,
                     *   skip it; otherwise, go on as normal 
                     */
                    if (before == 0)
                        prefix_count = after;
                    else
                        continue;
                }

                /* display the appropriate separator before this item */
                if ((flags & LCG_TALL) != 0)
                {
                    local j;
                
                    /* wide listing - indent to the desired level */
                    "\n";
                    for (j = 1; j <= indent; ++j)
                        "\t";
                }
                else
                {
                    /* 
                     *   "wide" (paragraph-style) listing - add a comma, and
                     *   possibly "and", if this isn't the first item
                     */
                    if (count > 0)
                    {
                        if (count+1 < disptot)
                            ", ";
                        else if (count == 1)
                            " and ";
                        else
                            ", and ";
                    }
                }

#ifdef USE_COMMON_SENSE
            
                /* list the object, along with the number of such items */
                if (prefix_count == 1)
                {
                    /* there's only one object - show the singular description */
                    cur.listdesc(&adesc);
                }
                else
                {
                    /* 
                     *   there are multiple equivalents for this object -
                     *   display the number of the items and the plural
                     *   description 
                     */
                    sayPrefixCount(prefix_count); " ";
                    cur.listdesc(&pluraldesc);
                }
                            
#else /* USE_COMMON_SENSE */
            
                /* list the object, along with the number of such items */
                if (prefix_count == 1)
                {
                    /* there's only one object - show the singular description */
                    cur.adesc;
                }
                else
                {
                    /* 
                     *   there are multiple equivalents for this object -
                     *   display the number of the items and the plural
                     *   description 
                     */
                    sayPrefixCount(prefix_count); " ";
                    cur.pluraldesc;
                }
            
                /* show any additional information about the item */
                if (cur.isworn)
                    " (being worn)";
                if (cur.islamp and cur.islit)
                    " (providing light)";
                
#endif /* USE_COMMON_SENSE */

                /* increment the number of displayed items */
                ++count;

                /* 
                 *   if this is a "tall" listing, and there's at least one
                 *   item contained inside the current item, list the item's
                 *   contents recursively, indented one level deeper 
                 */
                if ((flags & LCG_RECURSE) != 0 && itemcnt(cur.contents) != 0)
                {
                    /* 
                     *   if this is a "wide" listing, show the contents in
                     *   parentheses 
                     */
                    if ((flags & LCG_TALL) == 0)
                    {
                        if (cur.issurface)
                            " (upon which %you% see%s% ";
                        else
                            " (which contains ";
                    }
                
                    /* show the recursive listing, indented one level deeper */
                    listcontgen(cur, flags, indent + 1);

                    /* close the parenthetical, if we opened one */
                    if ((flags & LCG_TALL) == 0)
                        ")";
                }
            }
        }
        if (proptype(newList[k], &list_together) == DTY_SSTRING
        && (flags & LCG_TALL) == 0
        && num > 1)
            ")";

    }
    o = car(newList);
    while(o)
    {
        delete o;
        newList = cdr(newList);
        o = car(newList);
    }
}

/*
 *  MODIFIED TO ACCEPT A PASSED LIST
 *
 *  listcontcont: function(obj)
 *
 *  This function lists the contents of the contents of an object.
 *  It displays full sentences, so no introductory or closing text
 *  is required.  Any item in the contents list of the object
 *  obj whose contentsVisible property is true has
 *  its contents listed.  An Object whose isqcontainer or
 *  isqsurface property is true will not have its
 *  contents listed.
 */
replace listcontcont: function(obj)
{
    local list, i, tot;
    
    /*
     *   Get the list.  If the "obj" parameter is already a list, use it
     *   directly; otherwise, list the contents of the object. 
     */
    switch(datatype(obj))
    {
    case 2:
        /* it's an object - list its contents */
        list = obj.contents;
        break;
        
    case 7:
        /* it's already a list */
        list = obj;
        break;

    default:
        /* ignore other types entirely */
        return;
    }

    tot = length(list);
    i = 1;
    while (i <= tot)
    {
        showcontcont(list[i]);
        i = i + 1;
    }
}
;

/*
 *  MODIFIED TO DISPLAY ACTORS
 *
 *  showcontcont: function(obj)
 *
 *  showcontcont:  list the contents of the object, plus the contents
 *  of an fixeditem's contained by the object.  A complete sentence 
 *  is shown. This is an internal routine used by listcontcont and 
 *  listfixedcontcont.
 */
replace showcontcont: function(obj)
{
    local i, o, cnt = 0, actor;

    actor = parserGetObj(PO_ACTOR);
    if (actor == nil) actor = parserGetMe();

    for (i = 1; i <= length(obj.contents); ++i)
    {
        o = obj.contents[i];
        if (o.isactor && o != actor)
        {
            o.location.dispParagraph;
            o.actorDesc;
            ++cnt;
        }
    }
    if (cnt)
        o.location.dispParagraph;

    if (itemcnt(obj.contents))
    {
        if (obj.issurface)
        {
            if (not obj.isqsurface)
            {
                caps();
                if (cnt)    // We add an also, to smooth out the transition
                    "also ";
                "sitting on "; obj.thedesc;" is "; listcont(obj);
                ". ";
            }
        }
        else if (obj.contentsVisible and not obj.isqcontainer)
        {
            caps();
            obj.thedesc;
            if (cnt)        // again, also smooths out the transition
                " also";
            " seem";
            if (!obj.isThem) "s";
            " to contain ";
            listcont(obj);
            ". ";
        }
    }
    if (obj.contentsVisible and not obj.isqcontainer)
        listfixedcontcont(obj);
}
;

////////////////////////////////////////////////////////////////////////
//  
//  TIMESYS-RELATED FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

/* 
 *	settimesys: function( t, r, d )
 *
 *	A function to allow us to set the time for the global object. This should be called
 *	in the Init routine to set up the initial game time. The values to be passed are:
 *
 *	t == time - in minutes from 0 (midnight) or as a parm, such as '9:30 a.m.'
 *	r == rate -  rate that time passes per turn in minutes
 *	d == day - numeric day value (nil=no day displayed,
 *			1=Sunday, 2=Monday, 3=Tuesday, etc.)
 *		date - parm value in yyyyddd format, e.g. '2000001' for 
 *			January 1, 2000.
 */
settimesys: function( t, r, d )
{
	local yyyy, ddd;

	if ( t )
	{
		if (datatype( t ) == 3)
		{
			local ret = parsetime( t );
			
			if (ret)
				timesys.time = ret[3];
			else
				timesys.time = 0;
		}
		else
			timesys.time = t;
	}
	else
	{
		timesys.time = 0;
	}
	if ( r )
	{
		timesys.timerate = r;
	}
	else
	{
		timesys.timerate = 1;
	}
	if ( d )
	{
		if ( length( d ) == 1 )
		{
			d = '199900' + d;
		}
		else if ( length( d ) == 7 )
		{
			timesys.day = nil;
		}
		else
		{
			d = '1999001';
		}
	}
	else
	{
		d = '2000365';
		timesys.day = nil;
	}

	yyyy = cvtnum( substr( d, 1, 4 ) );
	ddd = cvtnum( substr( d, 5, 3 ) );
	timesys.date = yyyy*365 + ( yyyy - 1 ) / 4 + ddd;
	timesys.day = ( timesys.date % 7 ) + 1;

	timesys.advance( 0 );				// sets the display values for the statusLine without 								// advancing the time.
}

/*
 *	gettimesys: function( disp, val, parm )
 *	
 *	Used to simplify the call to the various display methods
 *	available to the author.
 *
 */
gettimesys: function( disp, val, parm )
{
	local i = lower( disp );
	switch( i )
	{
		case TS_TIME:
			return timesys.timeDisplay( val, parm );
			break;
		case TS_DAY:
			return timesys.dayDisplay( val, parm );
			break;
		case TS_DATE:
			return timesys.dateDisplay( val, parm );
			break;
		case TS_ELAPSE:
			return timesys.elapsetimeDisplay( val, parm );
			break;
		default:
			return nil;
	}
}

/*
 *	advtimesys: function( incr, style )
 *	
 *	A function to advance the time through your game code
 *	and providing the option to execute the rundaemons() and
 *	runfuses() or bypass them. This function should be the last 
 *	line of code executed in a block because it will issue an 
 *	abort upon completion regardless of the style chosen.
 *
 *	If style is true then the waitingVerb.action() method is used 
 *	to advance time. This means that daemons and fuses are run, and
 *	the advance is subject to any issued stopwaiting(). 
 *
 *	if style is nil then the timesys.advance() method is called 
 *	directly. No daemons or fuses will be run and no opportunity
 *	for stopwaiting will be available. This method, however, is 
 *	much quicker than waitingVerb.action().
 */
advtimesys: function( incr, style )
{
	local t;
	if ( timesys.waiting )
	{
		return nil;
	}
	if (datatype(incr) == 3)
	{
		local ret = parsetime( incr );
	
		if (ret)
			t = ret[2];
		else
			t = 0;
	}
	else
		t = incr;
		
	timesys.waitqblock = true;	// block waitquery

	if ( style )
	{
		timesys.waittime = t;
		waitingVerb.action( nil );
	}
	else
	{
		timesys.advance( t );
		timesys.waitqblock = nil;
		abort;
	}
}

/* 
 *	stopwaiting: function( parm )
 *
 *	This function allows you to halt the waiting process, and
 *	informing the player of some timely event. The parameter
 *	should be either true or nil. True will request that the waitquery
 *	question 'Do you wish to continue waiting? [Y/N]' be issued, 
 *	giving the player a choice. Passing nil will tell the waiting process
 *	to terminate without prompting the player for a response.
 */
stopwaiting: function( parm )
{
	timesys.waiting = nil;
	timesys.waitquery = parm;
}

/*
 *	parsetime: function( cmd )
 *
 *	This function parses cmd to determine a timesys.waittime. If
 *  successful, it returns a list consisting of 3 elements:
 *      ret[1]: a sublist containing the original reSearch() return.
 *      ret[2]: the calculated waittime
 *      ret[3]: for o'clock-style formats this is the calculated time in 
 *			minutes past midnight; for hours/minutes format it contains
 *          true if 24-hour restriction was enforced, nil otherwise.
 *  If unsuccessful the function returns nil.
 */
parsetime: function( cmd )
{
	local ret, grp, specRet, restr24;
	local tot = 0, hh = 0, mm = 0, wtime = 0;
    
	cmd = lower(cmd);
	
	/*	
	 *  FORMAT: MIDNIGHT/NOON
	 *
	 *  Convert midnight to 12:00 a.m. and noon to 12:00 p.m. 
	 */
	ret = reSearch('(midnight)|(noon)', cmd );
    if (ret)
    {
    	specRet = ret;
    	grp = reGetGroup(1);
        if (reGetGroup(1)[3] == 'midnight')
        	cmd = '12:00 a.m';
        else if (reGetGroup(2)[3] == 'noon')
            cmd = '12:00 p.m';
    }    
    
    /*
     *  FORMAT: HH:MM <AM/PM> 
     *
     *  Our ultimate goal is to determine how many minutes we need
     *  to advance from the current time (timesys.time) to reach the 
     *  desired o'clock passed in the parameter. 
     */
	ret = reSearch('([0-9]?[0-9]):([0-9][0-9]) *([ap]%.?m%.?)?', 
	    cmd );
    if (ret)
    {
        local ampmInd;
        
        grp = reGetGroup(1);
        hh = cvtnum(grp[3]);
        grp = reGetGroup(2);
        mm = cvtnum(grp[3]);        
        grp = reGetGroup(3);        
        
        /*
         *  Validate the range for hours and minutes. This code 
         *  returns nil if the hours and minutes are not within
         *  the appropriate ranges either for 24-hour or 12-hour
         *  clock-time.
         */
        if (grp && grp[2])
        {
            if (! (hh > 0 && hh < 13))
                return nil;
            else
                ampmInd = grp[3];
        }
        else
        {
            if (! (hh >= 0 && hh < 24))
                return nil;
           else if (hh > 12)
                {
                    ampmInd = 'p.m.';
                    hh -= 12;
                }
        }
        if (! (mm >= 0 && mm < 60))
        	return nil;
        
        tot = 60*hh + mm;        	
         
        /*
         *  AM/PM ADJUSTMENT.
         *
         *  At this point total represents an unadjusted total number
         *  of minutes past midnight. Adjustments need to be made for
         *  the presence or lack of an am/pm indicator.
         */
        switch( ampmInd )
        {
            case 'am':
            case 'a.m':
            case 'a.m.':
                /* adjust 12:00 a.m. - 12:59 a.m. */
                if ( tot >= 720 )
                    tot -= 720;
                break;
                
            case 'pm':
            case 'p.m':
            case 'p.m.':
                /* Adjust time for p.m. */
                if ( tot < 720 )
                    tot += 720;
                break;

   			/*
	         * 	This handles the cases where "wait until 4:00" is requested
		 	 *	and it's 2:00 pm. In this case we probably really want to 
		 	 * 	"wait until 4:00 pm", not "wait until 4:00 am" which would 
		 	 * 	happen without this adjustment.
		 	 */
            default:
			    /* 
			     *  Adjusts for tot between 12:00 pm - 12:59 pm. This
			     *  converts HH:MM to HH:MM A.M.
			     */
			    if ( timesys.time == 720 && tot >= 720 )		
				    tot -=720;
				
				/* 
				 *  Adjusts for when it's afternoon and we've asked
				 *  for a time later in the afternoon. This converts
				 *  HH:MM to HH.MM P.M.
				 */
			    if ( timesys.time > 720 
			    && (tot > timesys.time - 720 ))
				    tot += 720;
				    
				/* 
				 *  Adjusts for when its morning and we've asked
				 *  for a time in the afternoon. This converts 
				 *  HH:MM to HH:MM P.M.
				 */
			    if ( timesys.time < 720 
			    && tot <= timesys.time )
				    tot += 720;
		}
		
		/*
		 *  COMPUTE WAITTIME
		 *
		 *  At this stage it is assumed that we have derived a point
		 *  along a 24-hour clock. It remains only to determine how
		 *  many minutes we need to cycle forward from our current
		 *  time (timesys.time) in order to reach the desired time.
		 */
		wtime = tot;
		
		if ( wtime > timesys.time )
		{
			wtime -= timesys.time;
		}
		else if ( wtime < timesys.time )
		{   
			wtime += ( 1440 - timesys.time );
		}      
		else
		{
			wtime = 1440;
		}
        		
		/* handles parsing of 'midnight/noon' */
        if (specRet)
        	return [specRet wtime tot];
        else
        	return [ret wtime tot];
    }
    
    /*
     *	EDIT: If we have gotten this far, anything with a ':' is not a valid
     *	time-format.
     */
    ret = reSearch(':', cmd);
    if (ret) return nil;

    /*
     *  FORMAT: HH AM/PM
     *
     *  Restructure the format to HH:MM am/pm. This code recurses into the 
     *	parsetime() function, after formatting parm in HH:MM AM/PM. 
     */
    ret = reSearch('([0-9]?[0-9]) *([ap]%.?m%.?)', cmd);
    if (ret)
    {
        specRet = ret;
        cmd = reGetGroup(1)[3] + ':00 ' + reGetGroup(2)[3];
        ret = parsetime( cmd );
        if (ret)
        {
        	wtime = ret[2];
        	tot = ret[3];
        	return [specRet wtime tot];
        }
    }
    
    /*
     *  FORMAT HH HOUR(S) (AND) MM MINUTE(S)
     *
     *  Calculate the number of minutes to advance for the given
     *  format. 
     */
	ret = reSearch('([0-9]*[0-9]) *hours? *(and)? *([0-9]*[0-9]) *minutes?', 
	    cmd );
    if (ret)
    {
        grp = reGetGroup(1);
        hh = cvtnum(grp[3]);
        grp = reGetGroup(3);
        mm = cvtnum(grp[3]);
        
        if (hh > PARSETIME_LIM_HH || mm > PARSETIME_LIM_MM)
        	return nil;
        	
        wtime = 60*hh + mm;
        
        return [ret wtime];
    }
    
    /*
     *  FORMAT HH HOUR(S)
     *
     *  Calculate the number of minutes to advance for the given
     *  format.
     */
	ret = reSearch('([0-9]*[0-9]) *hours?', cmd );
    if (ret)
    {
        grp = reGetGroup(1);
        hh = cvtnum(grp[3]);
        
        if (hh > PARSETIME_LIM_HH)
        	return nil;
        
        wtime = 60*hh + mm;
        return [ret wtime];
    }
    
    /*
     *  FORMAT MM MINUTE(S)
     *
     *  Calculate the number of minutes to advance for the given 
     *  format.
     */
	ret = reSearch('([0-9]*[0-9]) *minutes?', cmd );
    if (ret)
    {
        grp = reGetGroup(1);
        mm = cvtnum(grp[3]);
        
        if (mm > PARSETIME_LIM_MM)
        	return nil;
        
        wtime = 60*hh + mm;
        return [ret wtime];
    }

    /*
     *  FORMAT: MM
     *
     *  Restructure the format to MM minutes. This code recurses into the 
     *	parsetime() function, after formatting parm in MM minutes. 
     */
	ret = reSearch('([0-9]*[0-9])', cmd );
    if (ret)
    {
        specRet = ret;
        cmd = reGetGroup(1)[3] + ' minutes';
        ret = parsetime( cmd );
        if (ret)
        {
        	wtime = ret[2];
        	return [specRet wtime];
        }
    }
   	return nil;
}

////////////////////////////////////////////////////////////////////////
//
//  MESSAGEPUMP FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

/*
 *  invokeMessagepump: function( queueStage )
 *
 *  Function is called from the first step of postAction(), the first
 *  step of endCommand(), and the last step of endCommand(). It requests
 *  the messagepump to process its schedule.
 */
invokeMessagepump: function( queueStage )
{
	messagepump.processQueue( queueStage );
}

/*
 *  commandEvent: function( o, p, v, s, r, f, c, actor, verb, dobj, 
 *      prep, iobj ) 
 *
 *  The function takes a variable number of arguments, although o and p
 *  are required.
 *
 *  This function defaults to the following:
 *  
 *      + processing queue: beginning of endCommand()
 *      + retain:           not retained after processing
 *      + frequency:        0, validates for processing every turn
 *      + command arg:      include command in validate / process
 *                          parameters
 *      + validatePtr:      none
 */
commandEvent: function( o, p, ... )
{
	local s = 2, r = nil, f = 0, c = true, v = nil;
	local actor, verb, dobj, prep, iobj;

	actor = parserGetObj(PO_ACTOR);
	verb = parserGetObj(PO_VERB);
	dobj = parserGetObj(PO_DOBJ);
	prep = parserGetObj(PO_PREP);
	iobj = parserGetObj(PO_IOBJ);
	
	if (argcount > 2)   v = getarg(3);
	if (argcount > 3)   s = getarg(4);
	if (argcount > 4)   r = getarg(5);
	if (argcount > 5)   f = getarg(6);
	if (argcount > 6)   c = getarg(7);
	if (argcount > 7)   actor = getarg(8);
	if (argcount > 8)   verb = getarg(9);
	if (argcount > 9)   dobj = getarg(10);
	if (argcount > 10)  prep = getarg(11);
	if (argcount > 11)  iobj = getarg(12);

	messagepump.generateEvent(o, p, v, s, r, f, c, actor, verb, dobj, 
	    prep, iobj);
}

/*
 *  alertEvent: function( o, p, v, s, r, f, c, actor, verb, dobj, 
 *      prep, iobj ) 
 *
 *  The function takes a variable number of arguments, although o and p
 *  are required.
 *  This function defaults to the following:
 *  
 *      + processing queue: immediate, or first available
 *      + retain:           not retained after processing
 *      + frequency:        0, validates for processing every turn
 *      + command arg:      exclude command in validate / process
 *                          parameters
 *      + validatePtr:      none
 */
alertEvent: function( o, p, ... )
{
    local s = nil, r = nil, f = 0, c = nil, v = nil;
    local actor, verb, dobj, prep, iobj;
    
    if (argcount > 2)   v = getarg(3);
	if (argcount > 3)   s = getarg(4);
	if (argcount > 4)   r = getarg(5);
	if (argcount > 5)   f = getarg(6);
	if (argcount > 6)   c = getarg(7);
	if (argcount > 7)   actor = getarg(8);
	if (argcount > 8)   verb = getarg(9);
	if (argcount > 9)   dobj = getarg(10);
	if (argcount > 10)  prep = getarg(11);
	if (argcount > 11)  iobj = getarg(12);

    messagepump.generateEvent(o, p, v, s, r, f, c, actor, verb, dobj, 
        prep, iobj);
}

////////////////////////////////////////////////////////////////////////
//
//  ACTOR-RELATED FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

/*
 * 	parseUnknownVerb: function( actor, wordlist, typelist, errnum )
 *
 *	ACTOR GRAMMAR allows customized handling of unknown verb 
 *	commands, where the command is either essentially a noun
 *	phrase or a more complex sentence structure where the verb
 *	is not easily identified by the parser. 
 *
 *	The advantages of ACTOR GRAMMAR are two-fold. First it allows  
 *	unknown verb processing to be handled by a specific actor and 
 *	customize an action appropriate for that actor. Secondly, object
 *	class inheritance allows for the processing of inherited grammars.
 *	An unknown verb could be processed for an actor for each of its
 *	inherited classes, finally passing to the default movableActor
 *	grammar if unsuccessful.
 *
 * 	This function should return one of three values:
 *
 *	Returning true indicates that the function has successfully 
 *	handled the entire command itself; the parser does not display 
 *	any error messages, it executes fuses and daemons as normal, and 
 *	it proceeds to continue parsing any remaining text on the command 
 *	line (after a period or "then"). 
 *
 *	Returning a number (greater than zero) indicates success, just as 
 *	true does, but also indicates that the function parsed only the    
 *	given number of words, and that the remaining words are to be      
 *	considered an additional command. The parser will run fuses and    
 *	daemons as normal, and then will resume its normal parsing,    
 *	starting with the next word after skipping the number of words    
 *	specified in the return value. 
 *
 *	You can use this if you find "and" following a noun phrase that you    
 *	parse, or if for any other reason you find that another sentence     
 *	follows and should be parsed separately.
 *
 *	Returning nil indicates that the function has failed to handle the   
 *	command, and wants the parser to display the default error 
 *  message. The parser will display the message in the normal 
 *  fashion, using parseErrorParam or parseError as appropriate, and 
 *  will abort the command. No fuses or daemons will be executed, and 
 *  any remaining text on the command line will be discarded. 
 *
 *	If this function uses "abort" to end the command, the parser will    
 *	not execute any fuses or daemons, and it will ignore any remaining     
 *	text on the command line. The difference between returning nil and       
 *	executing an "abort" statement is that the parser will display the      
 *	default message when this function returns nil, but will not      
 *	display anything if the function uses "abort".
 */
parseUnknownVerb: function( actor, tokenlist, typelist, errnum  )
{
	return actor.grammar( tokenlist, typelist, errnum );
}

/*
@switchPlayer: function
Switch the player character to a new actor.  This function uses the
\tt parserSetMe()\ built-in function to change the parser's internal
record of the player character object, and in addition performs some
necessary modifications to the outgoing and incoming player character
objects.  First, because the player character object is by convention
never part of its room's contents list, we add the outgoing actor to
its location's contents list, and remove the incoming actor from its
location's contents list.  Second, we remove the vocabulary words
"me" and "myself" from the outgoing object, and add them to the
incoming object.
*/
replace switchPlayer: function(newPlayer, flag)
{
    /* if this is the same as the current player character, do nothing */
    if (parserGetMe() == newPlayer)
        return;

    /* 
     *   add the outgoing player character to its location's contents --
     *   it's going to be an ordinary actor from now on, so it should be
     *   in its location's contents list 
     */
    if (parserGetMe() != nil && parserGetMe().location != nil)
        parserGetMe().location.contents += parserGetMe();

    /* 
     *   remove the incoming player character from its location's contents
     *   -- by convention, the active player character is never in its
     *   location's contents list 
     */
    if (newPlayer != nil && newPlayer.location != nil)
        newPlayer.location.contents -= newPlayer;

    /* 
     *   remove "me" and "myself" from the old object's vocabulary -- it's
     *   no longer "me" from the player's perspective 
     */
    delword(parserGetMe(), &noun, 'me');
    delword(parserGetMe(), &noun, 'myself');

    /* establish the new player character object in the parser */
    parserSetMe(newPlayer);

    /* add "me" and "myself" to the new player character's vocabulary */
    addword(newPlayer, &noun, 'me');
    addword(newPlayer, &noun, 'myself');
    
    global.srDisplayPlayer = flag;
}
;

////////////////////////////////////////////////////////////////////////
//
//  OBJECT TREE FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

/*
object tree The hierarchy of containment between objects of metaclass Object, 
i.e., of concretely existing objects. Each has a 'parent', though this may be 'nothing' 
(to indicate that it is uncontained, as for instance rooms are) and possibly some 
'children' (the objects directly contained within it). The 'child' of an object is the 
'eldest' of these children, the one most recently moved within it or, if none have been 
moved into it since the start of play, the first one defined as within it. The 'sibling' 
of this child is then the next eldest, or may be 'nothing' if there is no next eldest. 
Note that if A is inside B which is itself inside C, then C 'directly contains' B but 
only 'indirectly contains' A: and we do not call A one of the children of C. 
*/

/*
 *  parent: function(o)
 *
 *  Function merely returns the object location. If the object has no
 *  location then it returns nil.
 */
parent: function(o) 
{ 
    if (o == nil) return nil;
    return o.location; 
}

/*
 *  child: function(o)
 *
 *  Function merely returns the last object of a contents list.
 *  If the object has no children then the function returns nil.
 */
child: function(o)
{
    local len;
    if (o == nil) return nil;
    len = length(o.contents);
    if (len == 0) return nil;
    return o.contents[len];
}

/*
 *  children: function(o)
 *
 *  Function returns the number of objects directly contained within obj.
 */
children: function(o)
{
    if (o == nil) return 0;
    return length(o.contents); 
}

/*
 *  sibling: function(o)
 *
 */
sibling: function(o)
{
    local f, list, p;
    
    if (o == nil) return nil;
    p = parent(o);
    if (p == nil) return nil;
    list = p.contents;
    f = find(list, o);
    if (f == 1) return nil;
    return list[f-1];
}

/*
 *  commonAncestor: function(o1, o2)
 *
 *  Find the nearest object indirectly containing o1 and o2,
 *  or return nil if there is no common ancestor.
 */
commonAncestor: function(o1, o2)
{
    local i, j;
    i = o1;
    while (i != nil)
    {
        j = o2;
        while (j != nil)
        {   
            if (j == i) 
                return i;
            j = parent(j);
        }
        i = parent(i);
    }
    return nil;
}

/*
 *  indirectlyContains(o1, o2)
 *  
 *  Returns true if the common ancestor of o1 and o2 is o1; otherwise
 *  returns nil.
 */
indirectlyContains: function(o1, o2)
{
    while (o2 != nil)
    {   
        if (o1 == o2)
            return true;
        o2 = parent(o2);
    }
    return nil;
}

////////////////////////////////////////////////////////////////////////
//
//  UTILITY FUNCTIONS
//
////////////////////////////////////////////////////////////////////////

/*
 *  replaceWith(value, target, replacement, flags)
 *
 *  The function searches the value string, replacing the target string
 *  with the replacement string. 
 *
 *  Bit-flags can be passed to control the search. 
 *
 *      RW_REPLACE_ONCE replace only one occurrence of the target in the 
 *           value string.
 *
 *          The default for replaceWith() is to replace all occurrences 
 *          of the target in the value string.
 *
 *      RW_MATCH_WORD target must match whole words. For example:
 *
 *          target 'get in' will search for '%<get>% *%<in%>'
 *          which will match 'get    in the chair', but not 
 *          'get into the chair'
 *
 *      RW_MATCH_CASE target must match the case in value. 
 *
 *          The default for replaceWith() is case-insensitive. A target
 *          string of 'get into' will match on 'GET INTO THE CAR' as
 *          well as 'Get into the car' unless RW_MATCH_CASE is used. 
 *
 *      RW_RET_NIL function returns nil if no match for the target found
 *
 *          The default for replaceWith() returns the value unchanged.
 *          Using RW_RET_NIL will return the value only if replacement
 *          occurred; otherwise the function will return nil.
 */
replaceWith: function(value, target, replacement, ...)
{
    local ret, new_value = '';
    local valuesave, targetsave, replacementsave;
    local flags = 0;
    
    if (argcount > 3) flags = getarg(4);
    if ((flags & RW_MATCH_WORD)!= 0) 
    {
        local tmptarget = '%<';
        tmptarget += replaceWith(target, ' ', '%> *%<');
        tmptarget += '%>';
        target = tmptarget;
    }
    
    do
    {
        if ((flags & RW_MATCH_CASE) == 0)
        {
            valuesave = value;
            targetsave = target;
            replacementsave = replacement;
            value = lower(value);
            target = lower(target);
            replacement = lower(replacement);
        }
        ret = reSearch(target, value);
        if ((flags & RW_MATCH_CASE) == 0)
        {
            value = valuesave;
            target = targetsave;
            replacement = replacementsave;
        }
        if (ret)
        {
            local len, tmp = '';
            len = length(value) + 1;
            if (ret[1] - 1)
                tmp += substr(value, 1, ret[1]-1);
            tmp += replacement;
            new_value += tmp;
            if (len - (ret[1]+ret[2]))
                value = substr(value, ret[1]+ret[2], len - (ret[1]+ret[2]));
            else
                value = '';
            if ((flags & RW_REPLACE_ONCE) != 0) break;
        }
        else if ((flags & RW_RET_NIL)!= 0
            && new_value == '') 
                return nil;
    }
    while( ret != nil);
    new_value += value;
    return new_value;
}

#pragma C-
