/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      dmb     In real life:   David M. Baggett
 *
 *      djp     In real life:   David J. Picton
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 *
 * Modification History
 *
 * CCR
 * ===
 *
 *  1-Jan-93    dmb     rec.arts.int-fiction BETA release (source only)
 *                      For beta testing only -- not for general
 *                      distribution.
 *
 * 20-Apr-93    dmb     Added the stream decoration to Inside_Building
 *                      you can get water there.
 *
 * 29-Apr-96    djp     Supplied some suggested enhancements/bugfixes
 *                      to dmb.  These were incorporated in Version 2.0.
 *
 * 24-Mar-99    djp     Preview version (1.0)
 *
 * AD551
 * =====
 *
 * 14-Apr-99    djp     Initial release of Adventure 551 (1.01)
 *                      Changes since version 1.0:
 *                      * Corrected code to drop emerald when 'plover' is
 *                      used at Y2.
 *                      * Added nouns for MarksInTheDust
 *                      * Added a heredesc for the shadowy figure
 *                      * Corrected the ne direction from the Hall of the
 *                        Mountain King when in oldgame mode.
 *
 * 23-Apr-99    djp     New release (1.10) of Adventure 551.
 *                      Changes since version 1.01:
 *                      * Implemented 'put x in y', where x is a container
 *                        for liquids and y is a stream or similar object.
 *
 * 30-Apr-99    djp     New release (1.11) of Adventure 551.
 *                      Changes since version 1.10:
 *                      * Implemented response to attempting to swim a
 *                        streamItem object.
 *                      * Clover now brings good luck in eliminating
 *                        'crawled around' messages (except for Witt's
 *                        End, where the chance of escaping is improved)
 *                      * Fixed an obscure bug affecting the troll (if you
 *                        paid at the sw side, then went back and entered
 *                        the ne side via the Lost River, the troll would
 *                        let you cross the bridge.)
 *                      * Fixed the passing of the 'actor' variable to
 *                        RicketyBridge doCross and cross methods.
 *                        Moved the check for the bear as actor to the
 *                        correct place (doCross).
 *
 * 17-May-99    djp     New release (1.20)
 *                      Changes since version 1.11
 *                      * If you try to sweep the rocks for a second time,
 *                        you are told to read the inscription.
 *                      * 'Read inscription' (but not 'read rocks') gave the
 *                        wrong answer.
 *                      * Once you have found the Tight Crack, this is
 *                        chosen for you on subsequent occasions (provided
 *                        that nothing has been dropped in the Dead End Crack).
 *                      * Similarly, once you have found the way from the
 *                        Swiss Cheese to the Oriental room, nw from Swiss
 *                        Cheese will always find the Oriental room.
 *                      * Suppress the code for 'spotting' the treasure
 *                        chest if the room is dark.
 *                      * Placed more objects in the floatingdecoration and
 *                        makecontents classes (which are now handled more
 *                        efficiently)
 *                      * Implemented a moveLoclist method to allow
 *                        contents lists to be updated automatically when
 *                        a floatingdecoration loclist is changed.
 *                      * Adjusted some location methods to cope with the
 *                        mod to toplocation (which now returns nil unless
 *                        the top location-level is a real room i.e. class
 *                        'room' but not class 'nestedroom'.)
 *
 * 20-May-99    djp     New release (1.21)
 *                      Changes since version 1.20:
 *                      * Some adjectives removed due to a change in the
 *                        TADS 2.4.0 parser, which fixes a bug preventing
 *                        the use of an adjective as a noun when another
 *                        object uses the same word as a noun.  This is
 *                        now permitted, making it necessary to exercise more
 *                        caution in the use of adjectives.  For example, the
 *                        crystal bridge defined 'rod' as an adjective; this
 *                        meant that 'wave rod' would refer to the crystal
 *                        bridge if the rod wasn't present.
 *
 * 15-Jul-99    djp     New release (2.00)
 *                      Changes in this version:
 *                      * Implemented the 'middle' travel verb.
 *                      * Reworked the cracks in the east end of the long
 *                        hall.  If they are examined, the player is told
 *                        how to select which one to enter.  Otherwise the
 *                        tight crack or dead-end crack will be chosen (or
 *                        only the tight crack, if the player has found it
 *                        and no objects are in the dead-end crack).
 *                        The middle crack can only be entered by typing
 *                        MIDDLE.
 *                      * Changed the default response to some travel verbs to
 *                        'I don't know how to apply that word here'
 *                      * For consistency, made 'climb up' the same as
 *                        'climb' in the west pit.  (UP climbs out of the
 *                        pit, whereas CLIMB climbs the plant if it's
 *                        grown.)
 *                      * Moved the NPCexit1 method for the west pit to
 *                        its correct position.
 *                      * Systematically made 'board' a synonym for 'enter'.
 *                      * Made the building object a 'distantItem' when seen
 *                        from the hill.
 *                      * Implemented 'enter room' and 'exit room'.
 *                      * Changed the safe to use dobjCheck and iobjCheck,
 *                        simplifying the implementation.
 *                      * Added a few travel methods to reflect the fact that
 *                        'opening' (as a verb) is now a synonym for 'passage'.
 *                      * Changed the coding when going south from the
 *                        east end of the long hall.  If the dead-end crack
 *                        has not been seen, the left or right crack is
 *                        chosen randomly.  (However, the unpromising-looking
 *                        middle crack is never chosen.)
 *                      * Implemented 'wave hands'.
 *                      * Added sensible responses for debris/mud.
 *
 * 3-Aug-99     djp     Bugfix release - Version 2.01
 *
 *              djp     Bugfix Version 2.02 (not released)
 *                        Changes in this version
 *                      * Changed the coding which warns about leaving on the
 *                        lamp.  It now looks for a special nolampwarn
 *                        property set by lit rooms outside the cave for
 *                        which the warning is to be suppressed.
 *                      * Fixed the handling of telephones in the default
 *                        room listendesc.
 *
 * 17-Feb-00    djp     New release - Version 2.20
 *                        Changes in this version
 *                      * Further generalization of version handling
 *
 * POLYADV
 * =======
 * 24-Aug-99    bjs     Made modifications for 550-point version.
 *          djp+bjs     Incorporated ad551 mods up to 2.20
 *
 * 3-Mar-00     djp     Initial beta release - Version 1.00
 *                      * Defined doCount methods for some scenery objects.
 *                      * Changed newgame properties to game551 (for this
 *                        file, this was overlooked in v2.20)
 *                      * Changed floatingdecoration location method to
 *                        return the first location in the list, rather than
 *                        nil, when the player isn't in any of the rooms.
 *                      * Removed redundant location code from
 *                        floatingdecoration objects.
 *                      * Added code to lock the grate when the blob
 *                        is summoned.
 *                      * Removed redundant 'spring' object from building and
 *                        added 'spring' to the Stream vocabulary.
 *                      * Added a new asknum function for dials, telephones
 *                        etc.
 *
 * 4-Apr-00     djp     Version 1.01: bugfix release
 *                      Changes in this version:
 *                      * Bugfix to Y2_Rock (obsolete reachablecontainers
 *                        property updated to canReachContents)
 *                      * Canyon mirror made a distantItem object
 *                      * Fixed the locations of some volcano items to be
 *                        visible from the valley.
 *                      * Added iswavetarget property for cases when the
 *                        rod is waved at a bridge or chasm etc.
 *                      * Re-coded the rusty door as a doorway item.
 *                      * Slight changes to Blueberries class (which, in
 *                        accordance with normal Polyadv practice, no longer
 *                        defines game-version properties) and to Blue1 (which
 *                        is now called a bush when the berries have gone)
 *
 * 4-May-00     djp     Version 1.02: bugfix release
 *                      Changes in this version:
 *                      * Fixed the description of In_Hall_Of_Mists to mention
 *                        the crack in the 550-point game.   Added a crack
 *                        object and direction for this location.
 *
 * 18-Sep-00    djp     Version 2.00: New version with 701-point game
 *                      Changes in this version:
 *                      * Corrected description of At_Recent_Cave_In (space
 *                        required after first '.').
 *              
 * 25-Sep-00    djp     Version 2.01: bugfix release
 *                      Changes in this version:
 *                      * Minor correction to text for sitting on the
 *                        throne without wearing crown
 *                      * Added At_East_End_Of_Twopit_Room to Bothhole object
 *                        loclist
 *                      * Changed Bothhole to allow climbing up and down of
 *                        holes only if the uphole or downhole properties
 *                        are set to true.
 *                      * Added methods for attempting to climb the hole
 *                        from the west pit.
 *                      * Allowed the north hole at the east end of the long
 *                        hall to be climbed down.
 *
 * 20-Dec-00     djp    Version 2.02: bugfix release
 *                      Changes in this version:
 *                      * Corrected spelling of 'whirlpool'.
 *                      * Removed 'high' as adjective for Passage, to
 *                        avoid disambiguation problems when entering a hole
 *                        at the east end of the Long Hall.
 *                      * Added a dust object in the Dusty Rocks room.
 *
 * 8-Feb-02     djp     Version 2.03: bugfix release
 *                      Changes in this version
 *                      * More sensible results when the dust is swept
 *                        in various rooms.
 *
 * 17-Aug-01    djp     Version 2.06: bugfix release with e-mail address update
 *                      Changes in this version
 *                      * In 701-point game, don't set global.bonustime (which
 *                        sends the player back to the Cylindrical Room) when 
 *                        the player tries to unlock the grate from the 
 *                        outside after cave closure.
 *                      * Incorporated classes NotFarIn and NoNPC in class 
 *                        Outside.  Changed room classes to reflect this.
 *
 * 22-Nov-01    djp     Version 2.08: bugfix release
 *                      Changes in this version
 *                      * Defined methods for 'leaving' or 'exiting' certain
 *                        objects, e.g. the 'room' object.  (Equivalent to 
 *                        typing 'out').
 *                      * Added a version-specific loclist for the 701-point
 *                        game.
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      Changes in this version
 *                      * Fixed a bug affecting 'enter waterfall' in the
 *                        551-point game.  The message was issued, but the
 *                        player didn't go anywhere!
 *                      * The high hole (at the east end of the Long Hall in
 *                        the 551 or 701-point version) is now a 
 *                        distantItem.
 *                      * Correction to verification code for 'read walls';
 *                        we should see nothing in Dead_End_7 unless the
 *                        message is there.
 *                      * Added alternative adjectives, e.g. 'w' as an
 *                        alternative to 'west'.
 *                      * Moved asknum function to ccr-fun.t
 *                      * Added methods for the Secret Canyon to restrict
 *                        the reachability of objects on the 'wrong' side of 
 *                        the dragon.
 *                      * Added a new Internal_Room class for a room within
 *                        a room.
 *                      * Made the handling of SEARCH and LOOK IN more
 *                        consistent with current adv.t practice - see comments
 *                        in history.t.  Eliminated the use of unnecessary
 *                        doLookin/doSearch methods for containers.
 *                      * Added a new object representing the north wall of
 *                        In_Arched_Hall and the south wall of EW_Corridor_E. 
 *                      * Removed the Debris item from In_Arched_Hall in the
 *                        550-point and 701-point game.
 *                      * Moved the updated CCR_lockableDoorway class 
 *                        (previously Multidoor) into this file.  Simplified
 *                        the Grate coding to use this file.
 *                      * Made provision for location methods to depend on 
 *                        an actor other than the player.
 *                      * Regrow mushrooms if lost in the waterfall.
 *                      * Fix to stop the sealed flask from being used when
 *                        liquids are taken.
 *                      * Removed redundant loclist701 properties.
 *
 * 31-May-03    djp     Version 2.10: minor bugfixes and enhancements
 *                      * Implemented OPEN SAFE WITH keys/dial/other
 *
 * 12-Aug-03    bjs     Version 2.11: added 580-point mode.
 *
 * 23-Jan-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *                      * Slight change to In_Hall_Of_Mt_King exits.  The east
 *                        passage from the Hall of the Mountain King is now
 *                        separate from the stairs (and enters the Hall of
 *                        Mists via the north exit).  Unlike the stairs, it
 *                        can be traversed when the gold is carried.
 *                      * Moved the code for OIL DOOR into the general
 *                        CCR_doorway class.
 *                      * Added a new Noexit object (to facilitate 
 *                        dynamically-changed exit properties)
 *                      * Added the ability to put objects on the crystal
 *                        bridge.  If the bridge is destroyed, the objects
 *                        fall down to the Cavern with Waterfall (based on
 *                        the 550-point game) or the Lost River Canyon
 *                        (551-point game and derivatives).
 *                      * Various minor changes in connection with the new
 *                        701+ point mode. These include:
 *                      * The player's possessions are now dropped in the
 *                        Hall of Mists if he tries to go down the pit with
 *                        the gold nugget.  (Important at Green level - we
 *                        don't want the gray rod to be dropped at the top
 *                        of the pit, because this locks the player out of
 *                        the Bird-Chamber area.)
 *         
 * 15-Jan-05    djp     Version 3.01: bugfix release.
 *                      * Removed adjectives from 'ceiling'.
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 *                      * Changes to At_Slit_In_Streambed.down to cater for
 *                        variant gameplay after a 'throne mishap' (restart
 *                        caused by sitting on the throne without a crown).
 *                        If the player managed to finish the game without
 *                        transporting the Wumpi, hurrah() should be called
 *                        regardless of whether the upper Transindection 
 *                        rooms are still active.
 *
 * 26-May-05   djp      Version 3.10 Rev. C
 *                      * Blueberries could not be eaten after the mushroom.
 *                        The 'loss of appetite' check is now based on
 *                        total portions eaten, not actor.maxweight.
 *                      * Blue1.count was incremented after loss of
 *                        appetite.
 *                      * Save Blue1.location after sitting on the throne.
 *                      * Added a heredesc for Blue1 so that the bush 
 *                        appears in the room description, once it's been
 *                        discovered.
 *                      * 'vanish' method for the bridge takes an extra
 *                        argument for an object to be excluded (i.e. the
 *                        rod being waved; if this is on the bridge, the
 *                        implication is that the player has temporarily
 *                        picked it up.  It is therefore dropped at the
 *                        player's location).
 *                       
 * 16-Jul-06    djp     Version 3.20: Bugfixes and extensions
 *                      * Addition of room properties (wino_trollstop) in
 *                        connection with drinking wine from the cask - if
 *                        the troll blocks the route to the normal locations,
 *                        the player ends up at the fork in the path. 
 *
 * 3-Mar-07     djp     Version 3.20 Rev. B
 *                      * Added Green_Debris_Room to the Debris loclist.
 *
 * 20-Oct-07    djp     Version 3.20 Rev. C
 *                      * Made the Dome into a floatingdecoration (seen at
 *                        three levels)
 *                      * Changed the test for Large_Gold_Nugget.isIn(actor) to
 *                        isInside(actor) in all cases.
 *
 * 30-Mar-09    djp     Version 3.20 Rev. D
 *                      * Improved the code for BridgeFissure.ioSpinOver to
 *                        respond correctly to SPIN MULTICOLORED ROD OVER 
 *                        FISSURE (when the star has not been attached).
 *
 * 21-Nov-10    djp     Version 3.21
 *                      * Correction to hallcracks description (at Blue
 *                        level the middle and left cracks are blocked)
 *
 *
 * 22-Oct-13    djp     Version3.21A
 *                      * Correction to Green_NE_Corner travel destinations
 *                        (which mostly went to Red level)
 *
 */

/* Note on compound words.  Compound word constructions were used in the
past to circumvent problems when verbs were used as adjectives; for example,
'actor, west' wouldn't parse correctly if an item defined 'west' as an
adjective.   This fix is no longer needed, and has been backed out from
version 2.00 onwards.

For the benefit of people using TADS 2.4.0 and later, we now also avoid
the use of adjectives which are used as a noun by other objects.
A bugfix in the interpreter allows the use of adjectives in these
circumstances, and this caused some confusion; for example, 'bird' could
refer to the cage in the absence of the bird, and 'rod' could refer to the
crystal bridge in the absence of the rod.
 */

/*
 * This file defines all the generic classes for rooms and fixed items, and
 * multi-location items which are present in all game versions.  Rooms which
 * are present in the basic 350-point game are defined here, together with
 * fixed items which are seen in those rooms.
 */

/*
 * The first few definitions override the ccr-adv.t file which is in the
 * process of being standardized.
 *
 * Room classes have been rationalized so that 'room' is now the generic
 * default for lit rooms, and the subclass 'darkroom' is used for dark rooms.  
 * Care must be taken when converting the old room classes e.g. CCR_room,
 * which made dark rooms the default.
 *
 * For Polyadv, the following convention is used to ensure that rooms are
 * easily identifiable:
 *
 * The class list for rooms should start with either 'room' or 'darkroom'. 
 * (N.B. this does not apply to nestedroom objects).
 *
 * All generic room classes are a subclass of the room object (not darkroom).
 *
 * In the original Fortran versions of the game, you could move around
 * by naming adjacent places.  Since we want to retain this capability,
 * we need to augment the existing set of room movement methods with
 * extra travel verbs (in ccr-verb.t).
 *
 * We also need a few things to help our NPC's move about the cave
 * without getting stuck or wandering where they're not supposed to.
 *
 * Notes on exit properties:
 * 1.  If a travel property is defined as a method, it should always return
 * a value explicitly using a return statement.  Unpredictable effects can
 * occur if you forget to return a value when TADS expects the method to do
 * so.
 *
 * 2.  If travel in a given direction is impossible, the code should issue
 * appropriate text then return nil.  (If travel may be possible in some
 * cases but not others, the method can use a pass statement to issue the
 * default method.)
 *
 * 3.  It is permissible for travel property code to issue travelTo itself.
 * The method should then return nil to indicate that no further travel is
 * possible.
 *
 * 4.  Note that the die() function now issues the abort statement, so any
 * code after die() will not be reached.
 *
 * 5.  Travel verbs now set global.travelActor and global.currentActor
 * to the actor doing the travelling, and this may be used by travel 
 * properties which are methods.
 *     
 *     If travel methods are to be called by other code, they should now
 * use the enhanced argument list e.g.
 *
 *     Bear.travelTo(Bear.location,&up)
 *
 *     This enables the travelTo routine to set up the global variables
 * before the travel destination is evaluated, obviating the need to 
 * set environment variables in the code beforehand.  The travelTo methods
 * now provide two further optional arguments: a "refloc" argument defining
 * the viewpoint from which location methods are to be evaluated, if different
 * from the actor's location, and a "nocheck" argument which suppresses 
 * checks for valid destinations at closing time.  If the refloc argument is
 * used, global.currentActor will be set to dummyactor and dummyactor.location
 * will be set to refloc.
 *
 * 6.  If you add new exit properties, be sure to update the exitlist in
 * ccr-npc.t (or NPC's won't know to look at them) and also the exitlist in
 * CCR_room.back (or the "back" command may not always work).
 */

/* Note on game versions

   Potentially we could have any number of game versions but at present
   there are six.

   To delete a room from a particular game version, set:

   gamevvv to true for the relevant versions - for example, setting just
   game551 to true will result in the room being deleted from other
   versions.

   Objects in class 'floatingdecoration', and objects with a simple location,
   will be moved into nil automatically when they are deleted.  In other
   cases the location method may need to be hand-coded to ensure that the
   object can only be seen in the appropriate game versions.

   To change the location of an object (with a simple location) in a
   given version, use a locationvvv variable e.g.

   location551 = Pantry

   If it is necessary to change the loclist of a floatingdecoration object for
   a given version, set the appropriate loclistvvv to a non-nil
   value, e.g.

   loclist551 = [Inside_Building Pantry]

   Note that it's perfectly OK for a loclist to contain rooms which are
   present in some versions but not others.  For example, you can code just
  
   loclist    = [Inside_Building Pantry Sword_Point]
  
   without worrying about the fact that some of the locations aren't available
   in all game versions.  However, you would need to use the mechanism if
   there is a difference which will be visible to the player, e.g.

   loclist    = [Inside_Building Pantry Y2]  // default room list
   loclist701 = [Pantry Fake_Y2]             // list for 701-point game
   
   Exactly one of these global properties will evaluate as true:
   global.oldgame - if the game lacks the 551-point extensions.  This
                    property is set to true for the 350-point and 550-point 
                    games.

   global.newgame - if the game is the 551-point version, or a later version
                    derived from it.  This property is set to true for the
                    551-point and 701-point games.

   The following properties are also available.

   global.game550 - true if the game is the 550-point version (or in future, a
                    version derived from it e.g. the 660-point version).

   global.game701 - true if the game is the 701-point version.

   There is also a variable, global.vnumber, which holds the version
   identification.

   The 'place', 'scatter' and 'move' methods for the dwarves/pirate
   avoid placing the NPC's in a room unless its NPCvalid method is set to
   true.  By default, this will be the case if none of 'noNPCs', 'deleted'
   or 'version_NoNPCs' is set to true:

   The 'noNPCs' property should be set to 'true' for rooms which are always 
   out of bounds to NPCs, or nil for rooms which may be available to them.
   Note that this property is used at the 'preinit' stage to build a table
   of rooms and exits which may be available to NPCs.  For this reason it
   wouldn't make logical sense to change the value of noNPCs during a game,
   or to code noNPCs as a method whose return value may change.

   The 'deleted' property of a room should also not be changed by user code.
   This property is set automatically at the 'init' stage to indicate
   that a room is unavailable (to the player as well as NPCs) in the selected
   game version.

   If NPCs need to be excluded from a room on a conditional basis, e.g.
   game version or type of NPC, a version_NoNPCs property should be provided.
   Again, it should return true if NPCs are to be excluded from the room.
   Take care when doing this - you'll get warning messages if you cause
   NPC's to get stuck in a room with no valid exits.

   Note that the NPC movement code now considers doors to be valid 
   destinations, unless they have the noNPC property set to true.  The
   actual destination is determined by the NPCdest method, which will normally
   return the doordest method if two conditions are met:

   (i)  the doordest property is set to a room object, or is a method and
        the doordestOK property is set to true.  In the latter case the
        method should return a room object (or nil), but must not print text.
  
   (ii) the door can be traversed without first issuing an explicit OPEN or 
        UNLOCK command.
*/

/*
 * A class for the rooms in the "alike" maze.
 */
class CCR_alike_maze_room: room
    sdesc = "Maze of Twisty Little Passages, All Alike"
    ldesc = {
        I(); "%You% %are% in a maze of twisty little passages,
        all alike.";
    }
    myhints = [Mazehint]
;

/*
 * A class for rooms that are forbidden to non-player characters.
 * (This only means dwarves and pirate.)
 *
 * All pits inherit this class because the original source claims
 * that dwarves won't follow the player into a pit, even though
 * they do.  It makes more sense for them to give up the chase
 * when the player scrambles down into a pit, and for that matter
 * it may sound a bit funny for a combat to occur in a little pit,
 * so I've add NoNPC to all the pit rooms.
 *
 * BJS: Sounds reasonable to me. However, I think what they meant
 * is that the dwarves won't follow players into "You are at the bottom
 * of the pit with a broken neck."
 * Note that the 550-point version does set the pits as no-dwarf locations.
 */
class NoNPC: room
    noNPCs = true
;

/*
 * A class for the dead ends.
 */
class CCR_dead_end_room: room
    sdesc = "At a Dead End"
    ldesc = {
        I(); "%You% %have% reached a dead end.";
    }
;

/*
 * A class for rooms that are outside the cave.
 * These rooms are off limits when the cave is closing (except when the player
 * has escaped from the Cylindrical Room; the inside rooms are then
 * off-limits.)
 */
class Outside: room, NotFarIn, NoNPC
    isoutside = true
;

/*
 * A class for rooms that are in a building or cavern but outside the
 * cave proper.
 */
class Indoor: room, Outside
    isindoor = true
;

/*
 * A class for rooms that aren't far enough in to earn the player
 * the bonus for getting "well in."
 *
 * See the definition of basicMe in advmods.t for info on how this is used.
 */
class NotFarIn: room, NoNPC
    notfarin = true
;

class Internal_Room: fixeditem, room, openable, qcontainer
/* Class for a room-within-a-room - DJP. */

    reachable = ( [] + self )
    contentsVisible = { 
        local actor := getActor(&verbActor);
        return (self.isopen or actor.isIn(self)); 
    }

    istoproom = true  // The game should regard it as the top room.
    notakeallfrom = true // stop 'take all' (from outside) from referring
                         // to the contents.
    //multisdesc and sdesc should be set up as in the following example:
    // multisdesc = "pantry"
    // sdesc = "In the Pantry"

    adesc = {"a "; self.multisdesc;}
    thedesc = {"the "; self.multisdesc;}
    //inldesc should be the room description when seen from inside;
    //outldesc the long description when seen from outside.
    ldesc = {
        local actor;
        if(global.verbActor) 
            actor := global.verbActor;
        else
            actor := parserGetMe();
        if (actor.isIn(self)) self.inldesc; else self.outldesc;
    }    

    isopen = true   // open by default

    hasfloor = true // has floor rather than ground

    // default methods for entering and exiting.  The containing room
    // should normally also define 'in' as a travel method.
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    verDoEnter(actor) = {
        if(actor.isIn(self)) {
            "%You% are already in ";self.thedesc;". ";
        }
    }
    doEnter(actor) = {
        actor.travelTo(self);
    }
    doUnboard(actor) = {
        actor.travelTo(self,&out);
    }
    out=(self.location)
    // Exit info. for 'back' command:
    exithints = {return [] + self.location + &out;}
    // objects can normally be placed inside the room, but the vase
    // breaks unless the pillow has been placed there.
    ioPutIn( actor, dobj ) = {
        if (self.checksmash(actor,dobj,inPrep,self))
            pass ioPutIn;
    }
    verDoLookthru(actor) = {self.verDoLookin(actor);}
    doLookthru(actor) = {self.doLookin(actor);}
;
/* class makecontents: fixeditem */
/* This is a class for scenery items like walls and ceilings which are 
 * present in all rooms, or in a large number of rooms.

 * The presence of an object in a room is indicated by an exact match between 
 * the actor's location and the value returned by the object's location method.
 * The preinit routine adds the object to the contents lists of all rooms
 * which satisfy this condition.
 *
 * Note that this class is mainly intended for objects which are present in
 * a fixed set of locations, regardless of the game version.  Of course, it
 * is possible to make variations - for example, the ceiling of a room might
 * disappear after an explosion - but it would then be necessary to hand-code
 * the adjustments to the contents lists.  
 *
 * Note that the actor will not always be Me, and the location method should
 * in general check the location of the actor defined by global.currentActor,
 * defaulting to Me if this property is nil.
 */

class makecontents: fixeditem
;

/* class floatingdecoration: fixeditem */
/*
 * This class lets us easily define an object that's in multiple
 * places at once.  You just list the locations it appears in in
 * a list called loclist, which must normally contain top-level rooms (not
 * nested rooms like the boat and the Y2 rock).  If the current actor is
 * in one of the listed rooms, the location method returns that room.  If
 * the actor is elsewhere, the location defaults to the first element in 
 * the loclist.
 *
 * This is particularly nice for things like water, which can be
 * manipulated -- we only have one object name to consider (Stream)
 * for the streams in the game from which water can be taken.  (DJP - there
 * are now other water-bearing objects, e.g. the lake and the
 * reservoir.)
 *
 * Items of type floatingdecoration are no longer handled as true floating
 * items.  Instead, the items are included in the contents lists of all
 * the rooms listed in the loclist.
 *
 * In common with other multiple-location classes, the floatingdecoration 
 * method now references the position of a 'current actor' (defined by 
 * global.currentActor, but defaulting to Me if this property is 
 * nil.)  If the actor is in one of the rooms defined by the loclist, the 
 * relevant element of the loclist is returned.  If the player isn't in 
 * any of these rooms, the first element of the loclist is returned as 
 * a default.
 *
 * An loclistvvv method may now be defined, to cause the initial loclist
 * to be different in game version 'vvv'.  Note that it's perfectly OK if
 * some of the rooms in a loclist don't exist in the game version being
 * played.  For this reason, it's usually unnecessary to use loclistvvv
 * methods.
 *
 */
class floatingdecoration: fixeditem
    locationOK = true       // OK for location to be method
    // The default location method.  Note that this assumes that
    // the loclist contains only 'top-level' rooms.

    // Modified to allow the actor to be changed, e.g. during NPC movement.

    location = {
        local actor,toploc;
        if(global.currentActor) 
            actor := global.currentActor;
        else
            actor := parserGetMe();
        toploc := toplocation(actor);
        if ( find(self.loclist,toploc) )
            return toploc;
        else if (length(self.loclist) = 0)
            return nil;
        else return self.loclist[1];
    }
    // change the loclist and adjust contents lists
    moveLoclist(newlist) = {
        local i,l,oldloc,newloc;
        if (global.floatcontents) {
            // Add to contents lists of new locations where
            // necessary
            l := length(newlist);
            for (i := 1; i <= l; i++) {
                newloc := newlist[i];
                if (find(self.loclist,newloc) = nil) {
                    newloc.contents += self;
                }
            }
            // Remove from contents lists of old locations
            // not in the new list
            l := length(self.loclist);
            for (i := 1; i <= l; i++) {
                oldloc := self.loclist[i];
                if (find(newlist,oldloc) = nil) {
                    oldloc.contents -= self;
                }
            }
        }
        // Update the loclist.
        self.loclist := newlist;
    }
;

class roomliquid: fixeditem

    // Name of liquid (one word, e.g. water)
    // liquid = "water"

    // List of possible containers in order of preference.  Don't
    // include the vase by default.

    contlist = [bottle flask cask]
    getcont = {
        local actor := getActor(&verbActor);
        local i, o, l, cur;
        i := 1; o := nil; l := length(self.contlist);
        while (i <= l and not (o <> nil)) {
            cur := self.contlist[i];
            // find the first empty, unsealed container in the list
            if(cur.isIn(actor) and cur.isReachable(actor)
            and not (cur.haswater or cur.hasoil or cur.haswine or
            cur.issealed))
                o := cur;
            i++;
        }
        return o;
    }

    verDoTake(actor) = {
        local cont := self.getcont;
        if (cont = nil) {
            "%You% %have% nothing in which to carry the ";self.liquid;". ";
        }
        else if (cont.issealed) {
            "%You%'d have to open <<cont.thedesc>> before you could put
            anything into it. ";
        }
    }
/* flag to stop 'put in' from trying to do an implied 'take' - this could
   cause a loop */
    noImpliedTake = true
    doTake(actor) = {
        local cont;
        cont := self.getcont;
        cont.ioPutIn(actor, self);
    }
    verDoPutIn(actor, io) = {}
/* This method is only called by ordinary containers - not by containers
   for liquids which only use ioPutIn. */
    doPutIn(actor, io) = {
        caps(); io.thedesc; " would leak all over the
        place if %you% tried to carry liquids in it. ";
    }
    verIoFillWith(actor) = {}
    ioFillWith(actor,dobj) = {dobj.ioPutIn(actor, self);}
    verIoPutIn(actor) = {}
    ioPutIn(actor,dobj) = {
        if(isclass(dobj,liquidcont)) dobj.ioPutIn(actor,self);
        else
            "I don't see the point of putting <<dobj.thedesc>> into
            <<self.thedesc>>. ";
    }
;

class upclimbable: item // DJP - for things which you can climb UP but not DOWN
                        // It is assumed that verDoClimb and doClimb methods
                        // are defined for the object.

    verDoClimbup(actor) = {
        self.verDoClimb(actor);
    }
    verDoClimbdown(actor) = {
        "You could try to climb UP <<self.thedesc>>, but not DOWN. ";
    }
    doClimbup(actor) = {
        self.doClimb(actor);
    }
;

class downclimbable: item // DJP - for things which you can climb DOWN.
                          // It is assumed that verDoClimb and doClimb methods
                          // are defined for the object.

    verDoClimbup(actor) = {
        "You could try to climb DOWN <<self.thedesc>>, but not UP. ";
    }
    verDoClimbdown(actor) = {
        self.verDoClimb(actor);
    }
    doClimbdown(actor) = {
        self.doClimb(actor);
    }
;

class picklock: item //for lockable doors etc.
    verDoPick(actor) = { "%You% %have% no tools to pick the lock with."; }
;

/* These are similar to doorway/lockableDoorway, but changed as follows:

   * An optional doorlist property can be defined, so that more than two
     doors can be opened/closed/locked/unlocked at the same time (for the
     elfin door in the 551-point version, represented by four objects)

   * A misfit(io) method is defined to print an appropriate message when
     the key won't work.

   * If the property externalDoor is set to true (the default for the
     CCR_lockableDoorway class), the door cannot be unlocked or opened at 
     closing time or during a Security Alert.

     Note that the mykey property is no longer set to nil at closing time.
     Attempts to unlock an external door at closing time are intercepted by
     the relevant methods in CCR_doorway.

 */
class CCR_doorway: doorway, picklock
    // New destination method says 'unlock' instead of 'open' if a locked
    // door would open automatically.
    destination =
    {
        if (self.islocked and not self.noAutoOpen) {
            "%You%'ll have to unlock << self.thedesc >> first. ";
            setit(self);
            return nil;
        }    
        else pass destination;
    }
    // message to be issued when the key doesn't fit (defined in advmods.t)

    misfit(io) = { inherited keyedLockable.misfit(io);} // DJP

    // message to be issued when trying to unlock or open the door at
    // closing time.  (Detected when misfit(io) is called despite io being
    // the same as self.mykey)

    closingmessage = {
        if (global.closed) {
            P(); global.closingmess;
            if (not self.panicked and not self.location.isoutside and not
            global.fully_closed) {
                self.panicked := true;
                global.bonustime := global.panictime;
            }
        }
    }
    setIsopen(setting) =
    {
        /* update my status and if there are other doors, update them as well*/
        local cur, i, tot;
        tot := length (self.doorlist);
        for (i := 1; i <= tot; i++) {
            cur := self.doorlist[i];
            cur.isopen := setting;
        }
    }
    setIslocked(setting) =
    {
        /* update my status and if there are other doors, update them as well*/
        local cur, i, tot;
        self.islocked := setting;
        tot := length (self.doorlist);
        for (i := 1; i <= tot; i++) {
            cur := self.doorlist[i];
            cur.islocked := setting;
        }
    }
    // methods for locking and unlocking differ from keyedLockable only if
    // mykey = nil (indicating that the door can be unlocked or locked without
    // a key).  Set mykey = office_keys if you want the door to be 
    // permanently locked (see ccr-endg.t).
    doLock(actor) =
    {
        local i,l,outhideStatus;
        if (self.mykey = nil) 
            pass doLock;
        else inherited keyedLockable.doLock(actor);
    }
    doUnlock(actor) =
    {
        local i,l,outhideStatus;
        if (self.mykey = nil) 
            pass doUnlock;
        else inherited keyedLockable.doUnlock(actor);
    }
    // changed to use misfit property
    doLockWith(actor, io) =
    {
        if (io = self.mykey)
        {
            "Locked. ";
            self.setIslocked(true);
        }
        else
            self.misfit(io);
    }
    // changed to use the misfit property and check for closing and Security
    // Alert conditions.
    doUnlockWith(actor, io) =
    {
        if (io = self.mykey and not
        (self.externalDoor and (global.triggered_alert or global.closed)))
        {
            "Unlocked. ";
            self.setIslocked(nil);
        }
        else self.misfit(io);
    }
    verDoLookthru( actor ) = {
        if (not self.isopen) {"%You% can't see anything through ";
        self.thedesc;" because it's closed. ";}
    }

    // changed to check for closure and Security Alert conditions.
    doOpen( actor ) = {
        if (not
        (self.externalDoor and (global.triggered_alert or global.closed)))
            pass doOpen;
        else {
            caps; self.thedesc; " seems to be stuck - it refuses to open! ";
            self.closingmessage;
        }
    }
    doLookthru (actor) = {"%You% see nothing unexpected.";}
    // The list of doors to act upon.  By default, this is set to the 
    // door itself, plus the otherside door if defined.

    doorlist = {
        local dlist := [] + self;
        if (self.otherside) dlist += self.otherside;
        return dlist;
    }

    externalDoor = nil // by default, set to nil for nonlockable doors.
    // N.B. The NPCdest property is defined in advmods.t for all doorway
    // objects.

    verDoOil(actor) = {}
    doOil(actor) = {
        if (bottle.isIn(actor) and bottle.hasoil)
            bottle.doPourOn(actor, self);
        else if (cask.isIn(actor) and cask.hasoil)
            cask.doPourOn(actor, self);
        else if (flask.isIn(actor) and flask.hasoil)
            flask.doPourOn(actor, self);
        else
            "%You% %have% nothing to oil it with.";
    }
    setIsoiled(setting) =
    {
        /* update my status and if there are other doors, update them as well*/
        local cur, i, tot;
        tot := length (self.doorlist);
        for (i := 1; i <= tot; i++) {
            cur := self.doorlist[i];
            cur.isoiled := setting;
        }
    }

;

class CCR_lockableDoorway: CCR_doorway
    externalDoor = true // by default, set to true for lockable doors.
    islockable = true
    islocked = true
;

Noexit: CCR_doorway // Dummy object for dynamic changes to travel tables
    destination = {
        local actor := getActor(&travelActor);
        return actor.location.noexit;
    }
    doordest = nil   // Leads nowhere
    doordestOK = nil // Don't use for NPC travel
;

/*
 * The locations
 */
/* 1 */
At_End_Of_Road: room, Outside
    sdesc = "At End Of Road"
    ldesc = {
        I(); "%You% %are% standing at the end of a road before a
        small brick building. Around %you% is a forest.  A
        small stream flows out of the building and down a
        gully.";
    }
    road = At_Hill_In_Road
    west = At_Hill_In_Road
    up = At_Hill_In_Road
    building = Inside_Building
    in = Inside_Building
    east = Inside_Building
    downstream = In_A_Valley
    gully = In_A_Valley
    stream = In_A_Valley
    south = In_A_Valley
    down = In_A_Valley
    forest = In_Forest_1
    north = In_Forest_1
    depression = Outside_Grate
    // added for 551-point version
    knoll = Knoll
    hole = Thunder_Hole
    thunder = Thunder_Hole

    // This was in the original fortran code, but conflicts
    // with the fact that the building is to the east:
    // east = In_Forest

    // This property suppresses warnings about leaving the lamp on.
    nolampwarn = true

;
Building: floatingdecoration
    sdesc = "building"
    ldesc = {
        local actor;
        if(global.verbActor) 
            actor := global.verbActor;
        else
            actor := parserGetMe();

        if (actor.isIn(Inside_Building))
            "%You're% in it.";
        else
            "It's a small brick building.  It seems to be
            a well house.";
    }
    noun = 'building' 'house' 'wellhouse'
    adjective = 'well' 'small' 'brick'
    loclist = [ At_End_Of_Road  At_Hill_In_Road  Inside_Building
        Pantry]

    verDoEnter(actor) = {
        if (actor.isIn(At_Hill_In_Road))
            "It's too far away. ";
        else if (actor.isIn(Inside_Building))
            "You're already in the building! ";
    }
    doEnter(actor) = {
        actor.travelTo(Inside_Building);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    verDoUnboard(actor) = {
        if (not actor.isIn(Inside_Building))
            "You aren't in the building. ";
    }
    doUnboard(actor) = {
        actor.travelTo(At_End_Of_Road);
    }

    // The building is a distant item if seen from the hill in the
    // road.
    dobjGen(a, v, i, p) =
    {
        if (a.isIn(At_Hill_In_Road) and
        (v <> inspectVerb) and (v <> gonearVerb) and (v <> countVerb)) {
            "It's too far away. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb) return;
        self.dobjGen(a, v, d, p);
    }
;
Road: floatingdecoration
    sdesc = "road"
    noun = 'road' 'street' 'path'
    adjective = 'dirt'
    loclist = [ At_End_Of_Road  At_Hill_In_Road  In_Forest_2 ]
;
Forest: floatingdecoration, upclimbable
    sdesc = "forest"
    adesc = "forest"
    ldesc = {
        "The trees of the forest are large hardwood oak and
        maple, with an occasional grove of pine or spruce.
        There is quite a bit of undergrowth, largely birch
        and ash saplings plus nondescript bushes of various
        sorts.  This time of year visibility is quite
        restricted by all the leaves, but travel is quite
        easy if you detour around the spruce and berry
        bushes.";
    }

    noun = 'forest' 'tree' 'trees' 'oak' 'maple' 'grove' 'pine'
        'spruce' 'birch' 'ash' 'saplings' 'bushes' 'leaves'
        'berry' 'berries'
    adjective = 'surrounding' 'open' 'hardwood' 'oak' 'maple' 'pine'
        'spruce' 'birch' 'ash' 'berry'
    loclist = [
        At_End_Of_Road  At_Hill_In_Road  In_A_Valley
        In_Forest_1  In_Forest_2  In_Forest_3
    ]
    verDoClimb(actor) = {
        "None of the trees appear to be easily climbable.";
    }
;
class Streamitem: roomliquid // DJP - new generic class for bodies of water
    liquid = "water"
    verDoDrink(actor) = {}
    doDrink(actor) = {
        "%You% %have% taken a drink from <<self.thedesc>>.  The water
        tastes strongly of minerals, but is not unpleasant.
        It is extremely cold.";
    }
    verDoSwim(actor) = {"You don't know how.  ";}
;
Stream: Streamitem, floatingdecoration // DJP - this is now only one of the
                                   // water-bearing objects in the cave.

    sdesc = "stream"
    ldesc = {
        local actor;
        if(global.verbActor) 
            actor := global.verbActor;
        else
            actor := parserGetMe();

        if (actor.isIn(Inside_Building))
            "The stream flows out through a pair of 1
            foot diameter sewer pipes.";
        else
            pass ldesc;
    }
    noun = 'stream' 'water' 'brook' 'river' 'spring'
/* DJP: allow the expression 'stream water' */
    adjective = 'small' 'tumbling' 'splashing' 'babbling' 'rushing'
        'reservoir' 'stream'

    loclist = [
        At_End_Of_Road  In_A_Valley  At_Slit_In_Streambed
        In_Pit  In_Cavern_With_Waterfall
        Inside_Building Pantry Green_Lake_Room Muddy_Defile
        Fairy_Grotto Bubble_Chamber Red_Rock_Crawl
    ]

    listendesc = "You hear the sound of running water, splashing along the
                 bed of the stream. "
;
Gully: floatingdecoration
    sdesc = "gully"
    noun = 'gully'
    loclist = [ At_End_Of_Road  At_Slit_In_Streambed  Outside_Grate ]
;

/* 2 */
At_Hill_In_Road: room, Outside
    sdesc = "At Hill In Road"
    ldesc = {
        I(); "%You% %have% walked up a hill, still in the forest.
        The road slopes back down the other side of the
        hill.  There is a building in the distance.";
    }
    road = At_End_Of_Road
    building = At_End_Of_Road
    forwards = At_End_Of_Road
    east = At_End_Of_Road
    north = At_End_Of_Road
    down = At_End_Of_Road
    forest = {
        if (global.newgame and rand(100) <= 30) return In_Forest_3;
        else return In_Forest_1;
    }
    south = In_Forest_1
    // it is left to the adventurer to discover that you can go west
    // in the 551-point game!
    west = In_Forest_3
;

Hill: fixeditem, distantItem
    sdesc = "hill"
    ldesc = "It's just a typical hill."
    noun = 'hill' 'bump' 'incline'
    location = At_Hill_In_Road
;
OtherSideOfHill: fixeditem, distantItem
    sdesc = "other side of hill"
    thedesc = "the other side of the hill"
    adesc = { self.thedesc; }
    ldesc = "Why not explore it yourself?"
    noun = 'side'
    adjective = 'other'
    location = At_Hill_In_Road
;

/* 3 */
Inside_Building: room, Outside, Indoor
    sdesc = "Inside Building"
    ldesc = {
        I(); "%You% %are% inside a building, a well house for a
        large spring.  ";
        if(global.newgame)
            "On the north side, through an open doorway, is a small
             pantry.";
    }

    hasfloor = true   // described as 'floor' not 'ground'
    hasfloordesc = true // floor has a special description
    floordesc = "It's just an ordinary concrete floor."
    out = At_End_Of_Road
    outdoors = At_End_Of_Road
    west = At_End_Of_Road
    xyzzy = In_Debris_Room
    plugh = At_Y2
    downstream = { return self.stream; }
    stream = {
        "The stream flows out through a pair of 1 foot
        diameter sewer pipes. It would be advisable to use
        the exit.";

        return nil;
    }
    pantry = {
        local actor := getActor(&travelActor);
        if(global.newgame)return Pantry.enter(actor);
        else pass pantry;
    }
    in = {
        local actor := getActor(&travelActor);
        if(global.newgame)return Pantry.enter(actor);
        else pass in;
    }
    north = {
        local actor := getActor(&travelActor);
        if(global.newgame)return Pantry.enter(actor);
        else pass north;
    }
    passage = {
        local actor := getActor(&travelActor);
        if(global.newgame)return Pantry.enter(actor);
        else pass passage;
    }
    click = Rainbow_Room // if slippers are worn (551-point game)

    // Exit info. for 'back' command:
    exithints = [Pantry, &pantry]

    // This property suppresses warnings about leaving the lamp on.
    nolampwarn = true

;

Safe: fixeditem, openable // DJP
    game551 = true // 551-point game only
    sdesc = "steel safe"

    ldesc = {
        "It's a very solid-looking combination safe, embedded in
        the wall. ";
        "At present it is ";
        if (self.isopen) "open"; else "closed";".\n";
        if (not self.isopen) {
            P(); "To the right of the door there is a dial numbered from
            0 to <<Dial.maxsetting>>.  You note that the 0 setting is
            marked \"Reset\".  The dial is currently set
            to <<Dial.setting>>. ";
        }
        if (self.isopen) {
            if(itemcnt(self.contents) > 0) {
                P(); "It contains ";listcont(self);". ";
            }
        }
    }

    noun = 'safe' 'door'
    adjective = 'steel' 'wall' 'combination' 'safe'

    heredesc = {P(); I(); "A steel safe is embedded in the wall";
        if (self.hidden)", behind the poster. ";
        else ". ";
    }

    location = nil
    maxbulk = 1000
    nobird = true
    isopen = nil
    verDoOpen(actor) = {
        if(not self.isopen) "You'll have to tell me how to do that.";
        else pass verDoOpen;
    }
    verDoUnlock(actor) = {
        if (not self.isopen) "You'll have to tell me how to do that.";
        else "It's already unlocked!";
    }
    verDoLock(actor) = {self.verDoClose(actor);}
    doLock(actor) = {self.doClose(actor);}

    verDoLockWith(actor,io) = {
        if (self.isopen) {
            "I think you could lock this safe simply by closing it. ";
        }
        else "It's already locked.";
    }

    verDoUnlockWith(actor,io) = {
        if (isclass(io,keyItem)) {
            "This is a combination safe. "; caps(); io.thedesc; " won't
            help.";
        }
        else if (io = Dial)
            "It's obvious that the dial has something to do with opening
            the safe, but you'll have to be more specific than that! ";
        else if (objwords(2) = ['combination']) {
            if (not SafeCombination.isseen)
                "You don't know the combination. ";
            else
                "You'll have to give me more precise instructions than
                that, I'm afraid. ";
        }
        else {
            "You can't unlock a combination safe with ";io.adesc;"! ";
        }
    }
    verDoOpenWith(actor,io) = {self.verDoUnlockWith(actor,io);}
    doClose(actor) = {
        "The safe's door clicks shut.\n";
        pass doClose;
    }
    ioPutIn ( actor, dobj ) = {
        if(dobj.islong) {
            caps(); dobj.thedesc; " is too long to go into ";
            self.thedesc; ".  ";
        }
        else if(dobj.islarge) {
            caps(); dobj.thedesc; " is too large to go into ";
            self.thedesc; ".  ";
        }
        else if(dobj.ishuge) {
            caps(); dobj.thedesc; " is far too large to go into ";
            self.thedesc; ".  ";
        }
        else if(dobj = ming_vase) {
            caps(); dobj.thedesc; " won't quite fit into ";
            self.thedesc; ".  ";
        }
        else if((dobj = wicker_cage) and
        little_bird.isIn(wicker_cage)){
            "Are you kidding?  Do you want to suffocate the poor bird?";
        }
        else {
            if (isclass(dobj,pendantItem) and not (dobj = broken_pendant)) {
                "A strange feeling of insecurity comes over you as you place
                <<dobj.thedesc>> in the safe.  Somehow you \(know\) that
                it belongs with you, not in there! "; 
            }
            pass ioPutIn;
        }
    }
    // We now use xxxxCheck methods to simplify the implementation,
    // avoiding the need for two objects:
    dobjCheck(a, v, i, p) = {
        if (self.hidden and (v <> gonearVerb) and (v <> countVerb)) {
            "%You% had better remove the poster first.";
            exit;
        }
    }
    iobjCheck(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb) return;
        self.dobjCheck(a, v, d, p);
    }
    verIoUnlockWith(actor) = {
        if (objwords(2) = ['combination'])
            "It doesn't have a combination lock. ";
        else     
            "I'm not sure how you could unlock that with <<self.thedesc>>. ";
    }
    verIoOpenWith(actor) = {
        if (objwords(2) = ['combination'])
            "It doesn't have a combination lock. ";
        else     
            "I'm not sure how you could open that with <<self.thedesc>>. ";
    }
;

Dial: fixeditem, dialItem // DJP
    // In this implementation, the combination is a random set of
    // three different numbers from 1 to 50.  Turning the dial to
    // the zero position resets the dial.  A reset is needed whenever
    // the dial has been turned to an incorrect number.

    sdesc = "dial"
    ldesc = {
        "It's just to the right of the safe's door. ";
        inherited.ldesc;
        " You notice that the 0 setting is marked \"Reset\". ";
    }
    noun = 'dial'
    setting = 0
    maxsetting = 50
    minsetting = 0
    counter = 0 // counts the number of correct settings so far
    abscounter = 0 // counts the absolute number of settings
    comblen = 3 // number of digits in safe combination
    combination = [0 0 0] // the combination - set up either when we first
                      // turn the dial, or when the dusty rocks are
                      // swept.
    /* Set the safe combination (different each time!) */
    setcomb = {
        local i,j,randcom,done,ok;
        if(global.newgame) {
            Dial.combination := [];
            for (i := 1; i <= Dial.comblen; i++) {
                done := nil;
                while (not done) {
                    ok := true;
                    randcom := rand(self.maxsetting);
                    for (j := 1; j < i; j++) {
                        if (randcom = Dial.combination[j]) ok := nil;
                    }
                    if (ok) done := true;
                }
                Dial.combination += randcom;
            }
        }
    self.combination_set := true;
    }


    verDoTurnTo( actor, io ) = {
        if (Safe.isopen) "The dial won't turn while the safe is open.";
        else pass verDoTurnTo;
    }
    verDoTurn( actor) = {
        if (Safe.isopen) "The dial won't turn while the safe is open.";
        else pass verDoTurn;
    }
    doTurnTo( actor, io ) = {
        local old_setting := self.setting;
        /* check that indirect object is a number */
        if (io = numObj) {
            /* Limit the scope for trial and error.  Before the player
            learns the combination, the dial can be set to a nonzero
            value up to 15 times, allowing (at most) five combinations to
            be tried. */
            if(io.value <> 0 and (not SafeCombination.isseen) and
            (self.abscounter > (self.comblen*5)) and
            (io.value <> self.setting) and (not Safe.hasopened)) {
                "This is getting ridiculous!  It's highly unlikely that
                you could open the safe by trial and error, so I suggest
                that you wait until you've found the combination. ";
                return;
            }
            /* use inherited method to turn dial */
            inherited.doTurnTo(actor,io);
            /* do nothing if dial hasn't moved */
            if (old_setting = self.setting) return;
            /* Set the combination if necessary */
            if (not self.combination_set) self.setcomb;
            abscounter += 1;
            /* check for correct setting */
            if (not self.failed and
            io.value = self.combination[counter + 1]) {
                counter := counter + 1; P();
                    if(SafeCombination.isseen or counter = self.comblen
                    or Safe.hasopened)
                        "You hear a satisfying \"Click\"";
                    else
                        "You hear a \"click\", but you have no way
                        of knowing whether the setting is correct";
                if (counter = self.comblen) {
                    Safe.isopen := true;
                    Safe.hasopened := true;
                    if(rare_book.undiscovered) {
                        global.checklist += rare_book;
                        rare_book.undiscovered := nil;
                    }
                    ", and the safe's door smoothly swings open";
                    if(length(Safe.contents) > 0) {
                         " revealing ";listlist(Safe.contents);
                    }
                    "! ";
                    counter := 0;
                }
                else ". ";
            }
            /* for incorrect setting, reset counter to 0. */
            else {
                P(); counter := 0;
                if(io.value = 0) {
                    "You hear a \"clunk\" as the mechanism resets
                    itself. ";
                    self.failed := nil;
                }
                else if(SafeCombination.isseen or Safe.hasopened) {
                    self.failed := true;
                    "You hear an unsatisfying \"click\".
                    Maybe if %you% rubbed %your% fingers with
                    sandpaper ...";
                }
                else {
                    self.failed := true;
                    "You hear a \"click\", but you have no way
                    of knowing whether the setting is correct. ";
                }
            }
        }
        /* if indirect object is non-numerical, pass to inherited method
           for error message. */
        else pass doTurnTo;
    }
    verIoUnlockWith(actor) = {
        "I'm not sure how you could unlock that with <<self.thedesc>>. "; 
    }
    verIoOpenWith(actor) = {
        "I'm not sure how you could open that with <<self.thedesc>>. ";
    }
;

SewerPipes: fixeditem
    sdesc = "pair of 1 foot diameter sewer pipes"
    location = Inside_Building
    noun = 'pipes' 'pipe'
    adjective = 'one' 'foot' '1-foot' 'diameter' 'sewer'
;

/* 4 */
In_A_Valley: room, Outside
    sdesc = "In A Valley"
    ldesc = {
        I(); "%You% %are% in a valley in the forest beside a
        stream tumbling along a rocky bed.";
    }
    upstream = At_End_Of_Road
    building = At_End_Of_Road
    north = At_End_Of_Road
    forest = In_Forest_1
    east = In_Forest_1
    west = In_Forest_1
    up = In_Forest_1
    downstream = At_Slit_In_Streambed
    south = At_Slit_In_Streambed
    down = At_Slit_In_Streambed
    depression = Outside_Grate
;
Streambed: floatingdecoration
    sdesc = "streambed"
    noun = 'bed' 'streambed' 'rock'
    adjective = 'stream' 'water' 'river' 'small' 'tumbling' 'splashing'
                    'babbling' 'rushing' 'rocky' 'bare' 'dry'
    loclist = [ In_A_Valley  At_Slit_In_Streambed  Outside_Grate ]
;
Valley: floatingdecoration
    sdesc = "valley"
    ldesc = {
        local actor;
        if(global.verbActor) 
            actor := global.verbActor;
        else
            actor := parserGetMe();

        if (actor.isIn(In_A_Valley))
            "%You're% in it.";
        else
            pass ldesc;
    }
    noun = 'valley'
    adjective = 'deep'
    loclist = [ In_A_Valley  In_Forest_1  In_Forest_2 ]
;

/* 5 */
In_Forest_1: room, Outside
    sdesc = "In Forest"
    ldesc = {
        I(); "%You% %are% in open forest, with a deep valley to
        one side. ";
        if (global.newgame) {
            "Not far is a large billboard.";
        }
    }
    valley = In_A_Valley
    east = In_A_Valley
    down = In_A_Valley

    // An approximation of the original code:
    forest = {
        if (rand(100) <= 50)
            return In_Forest_1;
        else
            return In_Forest_2;
    }

    forwards = In_Forest_1
    north = In_Forest_1
    west = In_Forest_1
    south = In_Forest_1
    // Exit info. for 'back' command:
    exithints = [In_Forest_2, &forest]
;

Billboard: fixeditem, readable
    game551 = true // present in 551-point game only
    sdesc = "billboard"
    ldesc = {
        "The billboard reads:
        \"Visit Beautiful Colossal Cave.  Open Year Around.  Fun for
        the entire family, or at least for those who survive.\"
        Below the headline is an impossibly complicated map showing how
        to find Colossal Cave.  Not that it matters, because all the
        directions are written in Elvish.  However, one illustration
        catches your attention.  It seems to show a group of elves
        involved in a strong-man competition.  For some reason, they
        appear to be gorging themselves on blueberries! ";P();

        // blueberries added by DJP
        "You notice a blueberry bush nearby. ";

        Blue1.moveInto(self.location);

    }
    noun = 'billboard'
    adjective = 'large'
    location = In_Forest_1
;

class Blueberries: fixeditem
    isThem = true
    sdesc = "blueberries"
    adesc = "blueberries"
    ldesc = "The berries look delicious!  You're tempted to sample them."
    noun = 'berries' 'blueberries' 'bush' 'bushes'
    adjective = 'blue'
    verDoPick(actor) = {self.versample;}
    verDoTake(actor) = {self.versample;}
    verDoEat(actor) = {self.versample;}
    verDoPutIn(actor,io) = {self.versample;}
    versample = {}
    count = 0

    sample = {
        local actor;
        if(global.verbActor) 
            actor := global.verbActor;
        else
            actor := parserGetMe();

        if(actor.blueberries_eaten >= 5) {
            "You reach for the blueberries, but then have second
            thoughts.  For some reason you seem to have lost your
            appetite! ";
        }
        else {
            "You take a few of the blueberries, and can't resist eating
            them ... \n
            MMMM! They're delicious, and now you somehow feel a little
            stronger than you did before.";
            // Note that the player is allowed to carry more weight, but
            // not more bulk.  The sack must be carried.
            actor.maxweight += 2;
            // Total portions eaten
            actor.blueberries_eaten++;
            // portions taken from this object
            self.count++;
        }
    }
    doPick(actor) = {self.sample;}
    doTake(actor) = {self.sample;}
    doEat(actor) = {self.sample;}
    doPutIn(actor,io) = {self.sample;}
;

Blue1: Blueberries
    game551 = true
    isThem = nil
    sdesc = {
        if (self.count >= 2)
            "blueberry bush";
        else
            "blueberries";
    }
    ldesc = {
        if (self.count >= 2)
            "The bush has been picked clean now - there aren't any berries
            left. ";
        else
            "Someone else has almost picked the bush clean, but there are
            still a few left.  You're tempted to sample them. ";
    }
    heredesc = {
        P(); I(); "There is a blueberry bush here. "; 
    }

    noun = 'berries' 'blueberries' 'bush'

    sample = {
        if (self.count >= 2) {
            "You have now picked the bush clean.  There are no
            more blueberries left. ";
            return;
        }
        pass sample;
    }

;
/* 6 */
In_Forest_2: room, Outside
    sdesc = "In Forest"
    ldesc = {
        I(); "%You% %are% in open forest near both a valley and a
        road.";
    }
    road = At_End_Of_Road
    north = At_End_Of_Road
    valley = In_A_Valley
    east = In_A_Valley
    west = In_A_Valley
    down = In_A_Valley
    forest = In_Forest_1
    south = In_Forest_1
;

/* 7 */
At_Slit_In_Streambed: room, Outside
    sdesc = "At Slit In Streambed"
    ldesc = {
        I(); "At %your% feet all the water of the stream
        splashes into a ";
        if (global.closed) "2-foot"; // The only way to get here
                                     // with the cave closed
        else "2-inch";               // is in 550-point mode
        " slit in the rock.  Downstream
        the streambed is bare rock.";
    }
    building = At_End_Of_Road
    upstream = In_A_Valley
    north = In_A_Valley
    forest = In_Forest_1
    east = In_Forest_1
    west = In_Forest_1
    downstream = Outside_Grate
    rock = Outside_Grate
    bed = Outside_Grate
    south = Outside_Grate

    slit = { return self.down; }
    stream = { return self.down; }
    down = {
        // horror() should be called if the player has transported the Wumpi
        // but not de-activated the Upper Transindection Chambers.

        if (global.closed) {
            local i;
            // Look for any more treasures which should go to the Troll.
            for(i:=firstobj(item); i<>nil; i:=nextobj(i, item)) {
                if ((isclass(i,CCR_treasure_item) and not 
                i.awardedpointsfordepositing) or (i=tarnished_pendant)
                or (i = dull_pendant)) {
                    // exclude treasures already in the possession of the
                    // Troll
                    if (i.isInside(Troll_Treasure)) 
                        continue;
                    // exclude the charged pendants if the discharged pendants
                    // exist
                    if ((i = pendant) and (tarnished_pendant.location != nil))
                        continue;
                    if ((i = pendant2) and (dull_pendant.location != nil))
                        continue; 
                    // exclude the discharged pendants if they don't exist,
                    // or if they're in the player's possession
                    if ((i = tarnished_pendant) or (i = dull_pendant)) {
                        if ((i.location = nil) or i.isInside(parserGetMe()))
                            continue;
                    }
                    // exclude deleted treasures, or not-yet-found treasures
                    // in the 701+ point game.
                    if (i.deleted or (i.bonustreasure and not i.bonusfound))
                        continue;
                    // special handling for discharged pendants
                    if (i = pendant) {
                        if (tarnished_pendant.isInside(parserGetMe()))
                            continue;
                    }
                    else if (i = pendant2) {
                        if (dull_pendant.isInside(parserGetMe()))
                            continue;
                    }
                    else { 
                        global.wrongtreasloc++; 
                        i.moveInto(Troll_Treasure);
                    }
                }
            }
            if(Wumpi.phase > 0 and Green_Upper_Trans_Room.isdotroom)
                horror();
            else
                hurrah();
        }
        else
            "%You% %do%n't fit through a two-inch slit!";
        return nil;
    }
;
Slit: fixeditem
    sdesc = { if(global.closed) "2-foot slit"; else "2-inch slit"; }
    ldesc = {
        if(not global.closed)
            "It's just a 2-inch slit in the
            rock, through which the stream is flowing.";
        else
            "It's a 2-foot slit in the rock, through which the
            stream is flowing. You could probably fit through
            it, if necessary.";
    }
    location = At_Slit_In_Streambed
    noun = 'slit'
    adjective = 'two' 'inch' 'foot' '2-inch' 'two-inch' '2-foot' 'two-foot'
    verDoEnter(actor) = {}
    doEnter(actor) = { return At_Slit_In_Streambed.down; }
;

/* 8 */
Outside_Grate: room, Outside
    sdesc = "Outside Grate"
    ldesc = {
        I(); "%You% %are% in a 20-foot depression floored with
        bare dirt.  Set into the dirt is a strong steel grate
        mounted in concrete.  A dry streambed leads into the
        depression.";
    }
    forest = In_Forest_1
    east = In_Forest_1
    west = In_Forest_1
    south = In_Forest_1
    building = At_End_Of_Road
    upstream = At_Slit_In_Streambed
    gully = At_Slit_In_Streambed
    north = At_Slit_In_Streambed

    in = { return self.down; }
    down = Grate

    myhints = [Cavehint1]

    // This property suppresses warnings about leaving the lamp on.
    nolampwarn = true
;

Depression: fixeditem
    sdesc = "20-foot depression"
    ldesc = "%You're% standing in it."
    location = Outside_Grate
    noun = 'depression' 'dirt'
    adjective = '20-foot' 'twenty' 'foot' 'twenty-foot' 'bare'
;

// DJP - reworked using the new class.  Note that we'd normally define
// two separate objects, but we've kept it as one object to avoid changing
// code elsewhere.

Grate: floatingdecoration, CCR_lockableDoorway

    isopen = nil
    islocked = true
    sdesc = "steel grate"
    ldesc = {
        if (self.isIn(Outside_Grate)) {
            "It just looks like an ordinary grate
            mounted in concrete.";
        }
        else {
            "It's just a 3x3 steel grate mounted
            in the ceiling.";
        }

        " It is ";
        if (self.isopen)
            "open.";
        else if (self.islocked)
            "closed and locked.";
        else
            "closed.";
    }
    noun = 'grate' 'lock' 'gate' 'grille'
    adjective = 'metal' 'strong' 'steel' 'open' 'closed' 'locked'
        'unlocked'

    doordest = {
            if(self.isIn(Outside_Grate)) 
                 return Below_The_Grate;
            else
                 return Outside_Grate;
    }
    doordestOK = true // Tell backVerb code that the doordest is OK to use.

    locationOK = true // Tell compiler OK for location to be method
    loclist = [Outside_Grate Below_The_Grate]

    mykey = set_of_keys
    verDoLookthru(actor) = {}
    doLookthru (actor) = {
            if(self.isIn(Outside_Grate)) 
                 "You see a cobbled floor below the grate. ";
            else
                 pass doLookthru;
    }
;

/* 9 */
Below_The_Grate: NotFarIn
    sdesc = "Below the Grate"
    ldesc = {
        I(); "%You% %are% in a small chamber beneath a 3x3 steel
        grate to the surface. A low crawl over cobbles leads
        inward to the west.";
    }
    crawl = In_Cobble_Crawl
    cobble = In_Cobble_Crawl
    in = In_Cobble_Crawl
    west = In_Cobble_Crawl
    pit = At_Top_Of_Small_Pit
    debris = In_Debris_Room

    outdoors = { return self.up; }  // DMB: added
    out = { return self.up; }
    up = Grate

;
Cobbles: floatingdecoration
    sdesc = "cobbles"
    adesc = "cobbles"
    ldesc = "They're just ordinary cobbles."
    noun = 'cobble' 'cobbles' 'cobblestones' 'cobblestone' 'stones'
        'stone'
    adjective = 'cobble'
    loclist = [ Below_The_Grate  In_Cobble_Crawl  In_Debris_Room ]
;

/* 10 */
In_Cobble_Crawl: NotFarIn
    sdesc = "In Cobble Crawl"
    ldesc = {
        I(); "%You% %are% crawling over cobbles in a low passage.
        There is a dim light at the east end of the
        passage.";
    }
    out = Below_The_Grate
    tosurface = Below_The_Grate
    east = Below_The_Grate
    in = In_Debris_Room
    dark = In_Debris_Room
    west = In_Debris_Room
    debris = In_Debris_Room
    pit = At_Top_Of_Small_Pit

    // DMB: added the following, in accordance with its presence
    // in In_Debris_Room and rooms beyond.
    depression = {
        local actor := getActor(&travelActor), oldloc;
        oldloc := actor.location;

        Grate.doEnter(actor);

        // If the player didn't move, the grate must be
        // locked.  Move the player underneath the grate.
        if (actor.location = oldloc) {
            "\b";
            actor.travelTo(Below_The_Grate);
        }

        // We've already moved the player, but we have to
        // return a value, so just return nil (which results
        // in no more movement).
        return nil;
    }
    // Exit info. for 'back' command:
    exithints = [
        Outside_Grate, &depression,
        Below_The_Grate, &depression
    ]
;

/* 11 */
In_Debris_Room: darkroom, NotFarIn
    sdesc = "In Debris Room"
    ldesc = {
        if(self.west=Debris_West) {
            I(); "%You% %are% in a debris room filled with stuff
            washed in from the surface. A low wide passage with cobbles is
            now only partially blocked with debris, and continues to the west.
            An awkward canyon leads upward. "; P();
        }
        else {
            I(); "%You% %are% in a debris room filled with stuff
            washed in from the surface. A low wide passage with
            cobbles becomes plugged with mud and debris here, but
            an awkward canyon leads upward and west."; P();
        }

        I(); "A note on the wall says, \"Magic word XYZZY.\"";
    }
    entrance = Below_The_Grate
    crawl = In_Cobble_Crawl
    cobble = In_Cobble_Crawl
    passage = In_Cobble_Crawl
    low = In_Cobble_Crawl
    east = In_Cobble_Crawl
    canyon = In_Awkward_Sloping_E_W_Canyon
    in = In_Awkward_Sloping_E_W_Canyon
    up = In_Awkward_Sloping_E_W_Canyon
    west = In_Awkward_Sloping_E_W_Canyon
    xyzzy = Inside_Building
    pit = At_Top_Of_Small_Pit

    // The original occasionally allowed the player to walk
    // large distances in one turn.  This is just one example.
    depression = {return In_Cobble_Crawl.depression;}

    // Exit info. for 'back' command:
    exithints = [
        Outside_Grate, &depression,
        Below_The_Grate, &depression
    ]
;
Debris: floatingdecoration
    sdesc = "debris"
    ldesc = "You see nothing of interest."
    noun = 'debris' 'stuff' 'mud'
    loclist = [ In_Debris_Room  In_Arched_Hall ]
    // DJP - no mention of debris in 550-point version of Arched Hall
    // Added some extra locations for the 701+ point game.
    loclist550 = [ In_Debris_Room Blue_Debris_Room Debris_West 
    Blue_Debris_West Green_Debris_Room Blue_Plaque_Room Green_Plaque_Room] 
    verDoSearch(actor) = {}
    doSearch(actor) = {"You sift through the debris, but find nothing
    of interest. ";}
    verDoMove(actor) = {"You move some of the looser pieces of debris, but
    this reveals nothing of interest. ";}
    verDoTake(actor) = {"You have come here to find treasures, not to
    cart around useless rubble, debris or mud.  I suggest that you
    leave it where it is. ";}
    verDoPutIn(actor,io) = {self.verDoTake(actor);}
;
XyzzyNote: fixeditem, readable
    sdesc = "note"
    ldesc = "The note says \"Magic word XYZZY\"."
    noun = 'note'
    location = In_Debris_Room
;

/* 12 */
In_Awkward_Sloping_E_W_Canyon: darkroom, NotFarIn
    sdesc = "In Awkward Sloping E/W Canyon"
    ldesc = {
        I(); "%You% %are% in an awkward sloping east/west
        canyon.";
    }

    entrance = Below_The_Grate
    down = In_Debris_Room
    east = In_Debris_Room
    debris = In_Debris_Room
    in = In_Bird_Chamber
    up = In_Bird_Chamber
    west = In_Bird_Chamber
    pit = At_Top_Of_Small_Pit
    depression = {return In_Cobble_Crawl.depression;}

    // Exit info. for 'back' command:
    exithints = [
        Outside_Grate, &depression,
        Below_The_Grate, &depression
    ]

;

/* 13 */
In_Bird_Chamber: darkroom, NotFarIn
    sdesc = "In Bird Chamber"
    ldesc = {
            I(); "%You% %are% in a splendid chamber thirty feet high.
            The walls are frozen rivers of orange stone.  An
            awkward canyon and a good passage exit from east and
            west sides of the chamber.";
    }

    entrance = Below_The_Grate
    debris = In_Debris_Room
    canyon = In_Awkward_Sloping_E_W_Canyon
    east = In_Awkward_Sloping_E_W_Canyon
    passage = At_Top_Of_Small_Pit
    pit = At_Top_Of_Small_Pit
    west = At_Top_Of_Small_Pit
    depression = {return In_Cobble_Crawl.depression;}

    // Exit info. for 'back' command:
    exithints = [
        Outside_Grate, &depression,
        Below_The_Grate, &depression
    ]
;

Orange_Stone: floatingdecoration, downclimbable
    sdesc = "orange stone"
    ldesc = "It's a column of travertine - a beautiful mineral found in wet
        limestone.  You could climb down it but you wouldn't be able to 
        climb up. "
    noun = 'stone' 'travertine' 'column'
    adjective = 'orange'
    loclist = [In_Bird_Chamber, Blue_Bird_Chamber, At_Brink_Of_Pit]
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&down);
    }
;

/* 14 */
At_Top_Of_Small_Pit: darkroom, NotFarIn
    sdesc = "At Top of Small Pit"
    ldesc = {
        // Note: this used to say "An east passage ends here..."
        // but that's obviously a mistake.

        I(); "At %your% feet is a small pit breathing traces of
        white mist.  A passage from the east ends here except for a
        small crack leading on.";

    }
    entrance = Below_The_Grate
    debris = In_Debris_Room
    passage = In_Bird_Chamber
    east = In_Bird_Chamber
    crack = { return self.west; }
    west = {
        "The crack is far too small for %you% to follow.";
        return nil;
    }

    down = {
        local actor := getActor(&travelActor);
        if (large_gold_nugget.isInside(actor)) {
            // for consistency with Green level, we now drop the actor's
            // possessions at the bottom of the pit instead of the top.
            actor.moveInto(In_Hall_Of_Mists);
            return broken_neck.death;
        }
        else
            return In_Hall_Of_Mists;
    }

    depression = {return In_Cobble_Crawl.depression;}

    // Exit info. for 'back' command:
    exithints = [
        Outside_Grate, &depression,
        Below_The_Grate, &depression,
        In_Hall_Of_Mists, &down
    ]
;
PitSteps: floatingdecoration, downclimbable
    sdesc = "rough stone steps"
    ldesc = {
        local actor := getActor(&verbActor);
        if(large_gold_nugget.isInside(actor))
            "It wouldn't be safe to descend the steps with what you're
            carrying. ";
        else
            "The rough stone steps lead down the pit.";
    }
    loclist = [At_Top_Of_Small_Pit, Green_Top_Of_Small_Pit]
    noun = 'stair' 'stairs' 'staircase' 'steps'
    adjective = 'rough' 'stone'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&down);
    }
    heredesc = { 
        local actor := getActor(&verbActor);
        if(not large_gold_nugget.isInside(actor)) {
            P();I(); "Rough stone steps lead down the pit.";
        }
    }
    verDoCount(actor) = {"You have better things to do than count steps
        and stairs. ";}

;
SmallPit:floatingdecoration
    sdesc = "small pit"
    ldesc = {
        local actor := getActor(&verbActor);
        "The pit is breathing traces of white mist.\n";
        if (self.location.analevel = 1)
            "The steps have been removed.  The elevator is the only way down
            at this Level. ";
        else if (large_gold_nugget.isInside(actor))
            "I'm not sure %you%'ll be able to climb down it
            safely with what %you're% carrying.";
        else
            "It looks like %you% might be able to climb down it.";
    }
    loclist = [At_Top_Of_Small_Pit, Blue_Top_Of_Small_Pit, 
              Green_Top_Of_Small_Pit]
    noun = 'pit'
    verDoClimb(actor) = {PitSteps.verDoClimb(actor);}
    verDoClimbup(actor) = {PitSteps.verDoClimbup(actor);}
    verDoClimbdown(actor) = {PitSteps.verDoClimbdown(actor);}
    doClimb(actor) = {PitSteps.doClimb(actor);}
    doClimbup(actor) = {PitSteps.doClimbup(actor);}
    doClimbdown(actor) = {PitSteps.doClimbdown(actor);}
;
PitCrack: floatingdecoration
    sdesc = "crack"
    ldesc = "The crack is very small -- far too small for %you% to follow."
    loclist = [At_Top_Of_Small_Pit, Blue_Top_Of_Small_Pit,
              Green_Top_Of_Small_Pit]
    noun = 'crack'
    adjective = 'small'
    verDoEnter(actor) = {self.ldesc;}
    verDoBoard(actor) = {self.verDoEnter(actor);}
    verDoLookin(actor) = {
        "There's nothing in the crack (in this version of Adventure,
        anyway). ";
    }
    verDoSearch(actor) = {self.verDoLookin;}
    analevel = {
        if(self.location)
             return self.location.analevel;
        else
             return 0;
    }
;
Mist: floatingdecoration, distantItem
    sdesc = "mist"
    ldesc = {
        "Mist is a white vapor, usually water, seen from time
        to time in caverns.  It can be found anywhere but is
        frequently a sign of a deep pit leading down to
        water.";
    }
    noun = 'mist' 'vapor' 'wisps'
    adjective = 'white' // 'water' removed for TADS 2.4.0
    loclist = [
        At_Top_Of_Small_Pit In_Hall_Of_Mists
        On_East_Bank_Of_Fissure  At_Window_On_Pit_1
        At_West_End_Of_Hall_Of_Mists In_Misty_Cavern
        In_Mirror_Canyon  At_Reservoir At_Window_On_Pit_2
        On_Sw_Side_Of_Chasm
    ]
    dobjGen(a, v, i, p) =
    {
        if ((v <> inspectVerb) and (v <> gonearVerb))
        {
            "It's too diffuse to do anything to.";
            exit;
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb) return;
        self.dobjGen(a, v, d, p);
    }

;

/* 15 */
In_Hall_Of_Mists: darkroom
    sdesc = "In Hall of Mists"
    ldesc = {
        if (global.game550) {
            I(); "%You% are at one end of a vast hall stretching
            forward out of sight to the west, filled with wisps of
            white mist that sway to and fro almost as if alive.  There
            is a passage at the top of a dome above %youm%.  A wide
            staircase runs downward into the darkness;  a chill wind
            blows up from below.  There are small passages to the north
            and south, and a small crack leads east.";
        }
        else {
            I(); "%You% %are% at one end of a vast hall stretching
            forward out of sight to the west.  There are openings
            to either side.  Nearby, a wide stone staircase leads
            downward.  The hall is filled with wisps of white
            mist swaying to and fro almost as if alive.  A cold
            wind blows up the staircase.  ";"There is a passage at
            the top of a dome behind %youm%.";
        }
    }

    left = In_Nugget_Of_Gold_Room
    south = In_Nugget_Of_Gold_Room
    forwards = On_East_Bank_Of_Fissure
    hall = On_East_Bank_Of_Fissure
    west = On_East_Bank_Of_Fissure
    east = {
        if (global.game550) return Sandstone_Chamber;
        else return self.up;
    }
    crack = {
        if (global.game550) return self.east;
        else pass crack;
    }
    stairs = In_Hall_Of_Mt_King
    // N.B. going north is regarded as a different route to going
    // down the stairs.  In the 660/770 point games this is not the case:
    // the staircase is the north exit.
    down = In_Hall_Of_Mt_King
    north = {
        local actor := getActor(&travelActor);
        actor.nextRoute := 1; 
        return In_Hall_Of_Mt_King;
    }
    y2 = { // DJP - changed for 551-point version
        if(global.newgame) {
            "%You% locate a hidden passage on the north side of the
            hall, and climb%s% down a wall of broken rock ...";P();
            // We've combined two moves into one, but we still
            // penalize backtracking when the Wumpus is chasing ...
            if(Wumpus.ischasing) Wumpus.moveInto(Jumble_Of_Rock);
            return At_Y2;
        }
        else return Jumble_Of_Rock;
    }
    up = {
        local actor := getActor(&travelActor);
        if (large_gold_nugget.isInside(actor)) {
            "The dome is unclimbable.";
            return nil;
        }
        else {
            return At_Top_Of_Small_Pit;
        }
    }
    passage = {
        "There's more than one opening - please tell me which
        direction you want to go. ";
        return nil;
    }

    NPCexit1 = {
        if(global.newgame) return At_Y2;
        else return Jumble_Of_Rock;
    }
    NPCexit2 = {
        if (global.game550) return Sandstone_Chamber;
        else return nil;
    }

    // Exit info. for 'back' command:
    exithints = [
        At_Y2, &y2,
        Jumble_Of_Rock, &y2,
        At_Top_Of_Small_Pit, &up,
        Sandstone_Chamber, &east
    ]
    climb = {
        "You'll have to tell me whether you want to climb UP or DOWN.";
        return nil;
    }
    ana2 = Blue_Hall_Of_Mists
    ne = {
        if (not Blue_Hall_Of_Mists.isseen) 
            pass ne;
        else
            "You stand in the northeast corner of the room, in the position
            of the elevator shaft at Blue level. ";
            P();
            return Red_NE_Corner;
    }
;

class NE_Corner: beditem, owntrans
    sdesc = "northeast corner"
    ldesc = "It's the \"footprint\" of the Blue level elevator, located in
        the northeast corner of the room. "
// The objects are initially not accessible, but are moved to 
// the red-level and green-level Hall of Mists when Blue_Hall_Of_Mists is 
// entered    
    location = nil 
    noun = 'corner'
    adjective = 'northeastern' 'northeast' 'ne'

// This is really just part of the room, so all objects are reachable

    reachable = {return self.location.allcontents;}
    doEnter(actor) = {
        actor.travelTo(self.location, &ne);
    }
    doBoard(actor) = {self.doEnter(actor);}
    verDoStandon(actor) = {
        if(actor.posture != 'standing') return;
        if (actor.location = self)
        {
            "%You're% already on "; self.thedesc; "! ";
        }
    }
    doStandon(actor) = {self.doEnter(actor);}
    isseen = true
    isdroploc = true // objects can be dropped here
    default_posture = 'standing'
;
Red_NE_Corner: NE_Corner
    ana2 = In_Elevator
    out = In_Hall_Of_Mists
    sw   = In_Hall_Of_Mists
    west = In_Hall_Of_Mists
    south = In_Hall_Of_Mists
    north = {return In_Hall_Of_Mists.north;}
    y2 = {return In_Hall_Of_Mists.y2;}
    up = {return In_Hall_Of_Mists.up;}
    east = {return In_Hall_Of_Mists.east;}
;
Green_NE_Corner: NE_Corner
    kata = In_Elevator
    out = Green_Hall_Of_Mists
    sw   = Green_Hall_Of_Mists
    west = Green_Hall_Of_Mists
    south = Green_Hall_Of_Mists
    north = {return Green_Hall_Of_Mists.north;}
    y2 = {return Green_Hall_Of_Mists.y2;}
    up = {return Green_Hall_Of_Mists.up;}
    east = {return Green_Hall_Of_Mists.east;}
;

Game550crack: floatingdecoration
    game550 = true
    sdesc = "small crack"
    ldesc = "It's just large enough for you to enter. "
    noun = 'crack'
    adjective = 'small'
    loclist = [In_Hall_Of_Mists, Green_Hall_Of_Mists]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&east);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
;

Staircase: floatingdecoration, downclimbable
    sdesc = "wide stone staircase"
    ldesc = "The staircase leads down."
    loclist = [In_Hall_Of_Mists, Blue_Hall_Of_Mists, Green_Hall_Of_Mists]
    noun = 'stair' 'stairs' 'staircase'
    adjective = 'wide' 'stone'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&down);
    }
    verDoCount(actor) = {"You have better things to do than count steps
        and stairs. ";}

;
DomeSteps: floatingdecoration, upclimbable
    sdesc = "rough stone steps"
    ldesc = {
        local actor := getActor(&verbActor);
        if(large_gold_nugget.isInside(actor))
            "You won't be able to climb the steps with what you're
            carrying.";
        else
            "The rough stone steps lead up the dome.";
    }
    loclist = [In_Hall_Of_Mists, Green_Hall_Of_Mists]
    noun = 'steps'
    adjective = 'rough' 'stone'
    heredesc = {
        local actor := getActor(&verbActor);
        if(not large_gold_nugget.isInside(actor)) {
            P();I(); "Rough stone steps lead up the dome. ";
        }
    }
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&up);
    }
    verDoCount(actor) = {"You have better things to do than count steps
        and stairs. ";}
;
Dome: floatingdecoration
    sdesc = "dome"
    ldesc = {
        local actor := getActor(&verbActor);
        if (self.location.analevel=1)
            "The rough stone steps have been removed. The elevator is the only
            way up at this Level. ";
        else if (large_gold_nugget.isInside(actor))
            "I'm not sure %you%'ll be able to get up it
            with what %you're% carrying.";
        else
            "It looks like %you% might be able to climb up it.";
    }
    loclist = [In_Hall_Of_Mists, Blue_Hall_Of_Mists, Green_Hall_Of_Mists]
    noun = 'dome'

    verDoClimb(actor) = {DomeSteps.verDoClimb(actor);}
    verDoClimbup(actor) = {DomeSteps.verDoClimbup(actor);}
    verDoClimbdown(actor) = {DomeSteps.verDoClimbdown(actor);}
    doClimb(actor) = {DomeSteps.doClimb(actor);}
    doClimbup(actor) = {DomeSteps.doClimbup(actor);}
    doClimbdown(actor) = {DomeSteps.doClimbdown(actor);}

;

/* 17 */
On_East_Bank_Of_Fissure: darkroom
    sdesc = "On East Bank of Fissure"
    ldesc = {
        I(); "%You% %are% on the east bank of a fissure slicing
        clear across the hall. The mist is quite thick here,
        and the fissure is too wide to jump.";

    }
    hall = In_Hall_Of_Mists
    east = In_Hall_Of_Mists

    forwards = { return self.jump; }
    jump = {
        if (CrystalBridge.exists) {
            "I respectfully suggest %you% go%es% across the
            bridge instead of jumping.";

            return nil;
        }
        else
            return didnt_make_it.death;
    }

    over = { return self.across; }
    west = { return self.across; }
    cross = { return self.across; }
    across = {
        local actor := getActor(&travelActor);
        CrystalBridge.doCross(actor);
        return nil;
    }

    //
    // NPC's can go across too, but only if the bridge exists.
    //
    NPCexit1 = {
        if (CrystalBridge.exists)
            return West_Side_Of_Fissure;
        else
            return nil;
    }
    isfissureroom = true // DJP
    // Exit info. for 'back' command:
    exithints = [West_Side_Of_Fissure, &west]
    listendesc = {
        if (global.oldgame) pass listendesc;
        global.listenadd := true;
        inherited.listendesc;
        P(); "You hear a distant roar, like the sound of a fast-flowing
             river, from the depths of the fissure. ";
        global.listenadd := nil;
    }
    ana2 = Blue_East_Bank_Of_Fissure
;
BridgeFissure: floatingdecoration
    iswavetarget = true // magic can be worked by waving the rod at it ...
    sdesc = "fissure"
    ldesc = {
        if (self.location.analevel = 2)
            "An iron bridge spans the fissure. ";
        else if (CrystalBridge.exists)
            "A crystal bridge now spans the fissure. ";
        else
            "The fissure looks far too wide to jump. ";
    }
    noun = 'fissure'
    adjective = 'wide'
    loclist = [ West_Side_Of_Fissure,  On_East_Bank_Of_Fissure,
    Green_West_Side_Of_Fissure, Green_East_Bank_Of_Fissure ]
    verDoJump(actor) = {}
    doJump(actor) = {
        actor.travelTo(actor.location,&jump);
    }
    listendesc = {
        if (global.oldgame)
            pass listendesc;
        else
            "You hear a distant roar, like the sound of a fast-flowing
            river, from the depths of the fissure. ";
    }
    // Span x over fissure (implemented in connection with the scrawled note
    // in the Control Room manual)
    verIoSpanOver(actor) = {}
    ioSpanOver(actor,dobj) = {
        if(dobj.islong)
            "Good try, but even <<dobj.thedesc>> <<dobj.isdesc>> too 
            short to span over the fissure. ";
        else
            "Good try, but <<dobj.thedesc>> is far too short to span 
            over <<self.thedesc>>.";
    }
    // Spin x over fissure (correct interpretation of scrawled note)
    verIoSpinOver(actor) = {}
    ioSpinOver(actor,dobj) = {
        if (find(global.spinrodlist,dobj)) {
            dobj.isspun := true;
            if(not (self.location.analevel = 2)) {
                "You hold the rod over the fissure while spinning it in your
                hand.  It turns very freely, but nothing happens. ";
                if(manual.isread) "Maybe you'd better take another look at the
                manual ... ";
            }
            else {
                dobj.doTurn(actor);
                if (dobj.needs_activation and not dobj.activated) {
                     "You continue across the bridge, but nothing exciting
                     happens ... ";
                     P();
                }
                else
                     "Nothing happens at first, so you continue across the 
                     bridge. ";
                IronBridge.doCross(actor);
            }
        }
        else
            "Peculiar!  Anyway, nothing exciting happens. ";
    }
;
// Modified so that objects can be put onto the bridge.  However, they
// will fall into the fissure if the bridge disappears again.
CrystalBridge: floatingdecoration, surface
    iswavetarget = true // magic can be worked by waving the rod at it ...
    exists = nil
    sdesc = "Crystal bridge"
    ldesc = "It spans the fissure, thereby providing %youm% a way across."
    heredesc = {
        if (self.exists) {
            P(); I();
            "A crystal bridge now spans the fissure.";
        }
    }
    locationOK = true       // tell compiler OK for location to be method
    loclist = []
    noun = 'bridge'
    adjective = 'crystal' 'magic' // 'rod' deleted for TADS 2.4.0

    verDoCross(actor) = {}
    doCross(actor) = {
        if (self.exists) {
            if (actor.isIn(On_East_Bank_Of_Fissure))
                actor.travelTo(West_Side_Of_Fissure);
            else
                actor.travelTo(On_East_Bank_Of_Fissure);
        }
        else
            "There is no way across the fissure.";
    }
    verDoEnter(actor) = {}
    verDoBoard(actor) = {}
    verDoStandon(actor) = {}
    doEnter(actor) = {
        "You stand on the bridge, but you can't help looking down into the
        depths of the fissure.  This makes you feel very insecure, so you
        continue across ... ";
        P();
        self.doCross(actor);
    }
    doBoard(actor) = {self.doEnter(actor);}
    doStandon(actor) = {self.doEnter(actor);}
    vanish(actor,...) = {
        local i,o,l, objlist := self.contents, wheretogo, righthere := nil;
        local exclude := nil;
        if(argcount > 1) exclude := getarg(2); 
        if(self.isIn(toplocation(actor))) {
            "The crystal bridge has vanished!";
            righthere := true;
        }
        self.moveLoclist([]);
        self.exists := nil;
        if(itemcnt(self.contents) > 0) {
            if(global.newgame)
                wheretogo := Lost_Canyon_End;
                // based on Wumpus destination
            else
                wheretogo := In_Cavern_With_Waterfall;
                // based on an unimplemented feature of the 550-point game;
                // objects thrown into the fissure should go here
            l := length(objlist);
            for(i := 1; i <= l; i++) {
                o := objlist[i];
                if (not o.isfixed) {
                    if(righthere and not (o = exclude)) {
                        "\n"; caps(); o.thedesc;" falls into the depths of 
                        the fissure!\n ";
                    }
                    // deal with fragile items
                    if (o = ming_vase) {
                        o.shatter;
                        o := shards;
                    }
                    else if (o = glass_vial)
                        wheretogo := nil;
                    else if (o = exclude)
                        wheretogo := toplocation(actor);
                    o.moveInto(wheretogo);
                }
            }
        }
        if(Wumpus.ischasing) {
            if (Wumpus.prevloc.isfissureroom and
            (parserGetMe().location <> Wumpus.prevloc) and
            (Wumpus.locstay = 1))
                Wumpus.demise;
        }
    }
    appear(actor) = {
        if(toplocation(actor).isfissureroom)
            "A crystal bridge now spans the fissure!";
        self.exists := true;
        self.moveLoclist([West_Side_Of_Fissure On_East_Bank_Of_Fissure]);
    }
;

/* 18 */
In_Nugget_Of_Gold_Room: darkroom
    sdesc = "In Nugget of Gold Room"
    ldesc = {
        I(); "This is a low room with a crude note on the
        wall. "; NuggetNote.readdesc;
    }
    hall = In_Hall_Of_Mists
    out = In_Hall_Of_Mists
    north = In_Hall_Of_Mists
    ana2 = Blue_Nugget_Of_Gold_Room
;
NuggetNote: fixeditem, readable
    sdesc = "note"
    ldesc = {
        "The note says, \"You won't get it up the steps\".";
    }
    location = In_Nugget_Of_Gold_Room
    noun = 'note'
    adjective = 'crude'
;

/* 19 */
In_Hall_Of_Mt_King: darkroom
    sdesc = "In Hall of Mt King"
    ldesc = {
        I(); "%You% %are% in the hall of the mountain king, with
        passages off in all directions.";

    }
    stairs = {
        local actor := getActor(&travelActor);
        if (large_gold_nugget.isInside(actor) and global.newgame) {
            "The staircase is now unclimable.";
            return nil;
        }
        else {
            return In_Hall_Of_Mists;
        }
    }
    up = {return self.stairs;}
    // N.B. Going east is now regarded as a different route to the Hall of
    // Mists, via the north exit.  (The two routes are one and the same in the
    // 660/770 point game).
    east = {
        local actor := getActor(&travelActor);
        actor.nextRoute := 1; 
        return In_Hall_Of_Mists;
    }

    left = { return self.north; }
    north = {
        if (self.snakecheck)
            return Low_N_S_Passage;
        else
            return nil;
    }

    right = { return self.south; }
    south = {
        if (self.snakecheck)
            return In_South_Side_Chamber;
        else
            return nil;
    }

    forwards = { return self.west; }
    west = {
        if (self.snakecheck)
            return In_West_Side_Chamber;
        else
            return nil;
    }

    /*
     * An interesting little bit of trivia here:
     * 35% of the time you can slip past the snake and into
     * the secret canyon.  (This is in the original Fortran
     * code.)  But if you say "secret" you will *always* sneak
     * by it.
     */
    sw = {
        if (rand(100) <= 35) {
            return In_Secret_E_W_Canyon;
        }
        else {
            if (self.snakecheck)
                return In_Secret_E_W_Canyon;
            else
                return nil;
        }
    }
    secret = In_Secret_E_W_Canyon

    snakecheck = {
        local actor := getActor(&travelActor);
        if (Snake.isIn(actor.location)) {
            "%You% can't get by the snake.";
            return nil;
        }
        else
            return true;
    }

    ne = {
        local actor := getActor(&travelActor);
        if (global.oldgame and not global.game550) pass ne;
        if (not self.snakecheck) return nil;
        if (global.game550) return Morion;
        if (Throne_Room.isseen) return Throne_Room;
        if (rand(100) < 75 and not clover.isInside(actor))
            return crawled_around.message;
        else return Throne_Room;
    }

    nw = {
        if (not global.game550) pass nw;
        if (not self.snakecheck) return nil;
        return Corr_Divis;
    }

    se = {
        if (not global.game550) pass se;
        if (not self.snakecheck) return nil;
        return Corridor_1;
    }

    down = {
        if (not global.game550) pass down;
        if (not self.snakecheck) return nil;
        return Vault;
    }

    //
    // Let the NPC's go everywhere out of here
    //
    NPCexit1 = Low_N_S_Passage
    NPCexit2 = In_South_Side_Chamber
    NPCexit3 = In_West_Side_Chamber
    NPCexit4 = Throne_Room
    NPCexit5 = In_Hall_Of_Mists
    NPCexit6 = Corridor_1
    NPCexit7 = Corr_Divis
    NPCexit8 = Morion
    NPCexit9 = Vault // DJP

    // Exit info. for 'back' command:
    exithints = [
        Low_N_S_Passage, &north,
        In_South_Side_Chamber, &south,
        In_West_Side_Chamber, &west,
        Throne_Room, &ne,
        Morion, &ne,
        Corr_Divis, &nw,
        Corridor_1, &se,
        Vault, &down,
        In_Hall_Of_Mists, &up
    ]
    myhints = [Snakehint Snakehint2]
    ana2 = Blue_Hall_Of_Mt_King
;

Staircase2: floatingdecoration, upclimbable
    sdesc = "wide stone staircase"
    ldesc = "The staircase leads up."
    loclist = [In_Hall_Of_Mt_King, Blue_Hall_Of_Mt_King, Green_Hall_Of_Mt_King]
    noun = 'stair' 'stairs' 'staircase'
    adjective = 'wide' 'stone'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(self.location,&up);
    }

;
/* 20-22 are messages */
/* 23 */
At_West_End_Of_Twopit_Room: darkroom
    sdesc = "At West End of Twopit Room"
    ldesc = {
        I(); "%You% %are% at the west end of the twopit room.
        There is a large hole in the wall above the pit at
        this end of the room.";

        if (PlantStickingUp.isIn(self)) {
            P(); I(); PlantStickingUp.ldesc;
        }
    }
    east = At_East_End_Of_Twopit_Room
    across = At_East_End_Of_Twopit_Room
    west = In_Slab_Room
    slab = In_Slab_Room
    down = In_West_Pit
    pit = In_West_Pit

    up = { return self.hole; }      // DMB: added
    hole = {
        "It is too far up for you to reach.";
        return nil;
    }

;
HoleAbovePit_1: fixeditem, distantItem
    sdesc = "hole above pit"
    ldesc = {
        "The hole is in the wall above the pit at
        this end of the room.";
    }
    noun = 'hole'
    adjective = 'large'
    location = At_West_End_Of_Twopit_Room

    verDoEnter(actor) = { "It is too far up for %you% to reach."; }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    verDoClimb(actor) = { "It is too far up for %you% to reach."; }
    verDoClimbup(actor) = { "It is too far up for %you% to reach."; }

    dobjGen(a, v, i, p) =
    {
        if ((v <> inspectVerb) and (v <> gonearVerb) and (v <> inVerb)
        and (v <> countVerb))
        {
            "It's too far away.";
            exit;
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb) return;
        self.dobjGen(a, v, d, p);
    }
    doSynonym('Enter') = 'Gothrough'

;
PlantStickingUp: fixeditem,distantItem
    sdesc = {
        if (Plant.size = 1)
            "top of 12-foot-tall beanstalk";
        else
            "huge beanstalk";
    }
    ldesc = {
        if (Plant.size = 1)
            "The top of a 12-foot-tall beanstalk is
            poking out of the west pit.";
        else
            "There is a huge beanstalk growing out of the
            west pit up to the hole.";
    }
    noun = 'plant' 'beanstalk' 'stalk'
    adjective = 'bean' 'giant' 'tiny' 'little' 'murmuring'
        '12-foot-tall' 'twelve' 'foot' 'tall' 'bellowing'

    location = {
        local actor := getActor(&currentActor);
        if (Plant.size = 0)
            return nil;
        else if (actor.isIn(At_West_End_Of_Twopit_Room))
            return At_West_End_Of_Twopit_Room;
        else
            return At_East_End_Of_Twopit_Room;
    }
;

/* 24 */
In_East_Pit: darkroom, NoNPC
    sdesc = "In East Pit"
    ldesc = {
        I(); "%You% %are% at the bottom of the eastern pit in the
        twopit room.  There is a small pool of oil in one
        corner of the pit.";
    }
    up = At_East_End_Of_Twopit_Room
    out = At_East_End_Of_Twopit_Room
    climb = {return self.up;}
;
EastPit: floatingdecoration, downclimbable // DJP - reworked.
    sdesc = "eastern pit"
    ldesc = "The best way to examine the pit would be to go down into
        it. "
    noun = 'pit' 'corner'
    adjective = 'east' 'eastern' 'e'
    loclist = [At_East_End_Of_Twopit_Room
        At_West_End_Of_Twopit_Room]

    verDoClimb(actor) = {
        if(actor.isIn(At_West_End_Of_Twopit_Room)) {
            "You'll have to go over to the east end of the room first.";
        }
    }
    verDoEnter(actor) = {
        if(actor.isIn(At_West_End_Of_Twopit_Room)) {
            "You'll have to go over to the east end of the room first.";
        }
    }
    doEnter(actor) = {
        actor.travelTo(actor.location,&down);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doClimb(actor) = {
        self.doEnter(actor);
    }

    // The pit is a distant item in some locations.
    dobjGen(a, v, i, p) = {
        if (self.isIn(At_West_End_Of_Twopit_Room)) {
            if ((v <> inspectVerb) and (v <> gonearVerb) and
            ( v <> countVerb )) {
                "It's too far away. ";
                exit;
            }
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb) return;
        self.dobjGen(a, v, d, p);
    }

;
EastPit2: fixeditem, upclimbable
    sdesc = "eastern pit"
    ldesc = {In_East_Pit.ldesc;}
    noun = 'east-pit' 'pit' 'corner'
    adjective = 'eastern'
    location = In_East_Pit

    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&up);
    }
    verDoEnter(actor) = {
        "You're already in the pit!";
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&up);
    }
;

Oil: fixeditem, roomliquid
    liquid = "oil"
    sdesc = "pool of oil"
    noun = 'pool' 'oil'
/* BUGFIX - allow 'pool of oil' in disambiguation */
    adjective = 'small' 'pool' 'of' 'oil'
/* END OF BUGFIX */
    location = In_East_Pit
;

/* 25 */
In_West_Pit: darkroom, NoNPC
    sdesc = "In West Pit"
    ldesc = {
        I(); "%You% %are% at the bottom of the western pit in the
        twopit room.  There is a large hole in the wall about
        25 feet above you.";
    }
    up = At_West_End_Of_Twopit_Room
    out = At_West_End_Of_Twopit_Room
    climb = { // DJP - reworked
        local actor := getActor(&travelActor);
        if (Plant.size < 1 or Plant.size > 2) {
            "\n(the pit)\n";
            return self.up;
        }
        else {
            "\n(the plant)\n";
            Plant.doClimb(actor);
            return nil;
        }
    }
    climbup = {return self.climb;}
    // Allow NPCs to follow the player up the beanstalk
    NPCexit1 = {
        if (Plant.size <> 2) return nil;
        else return In_Narrow_Corridor;
    }
    // Exit info. for 'back' command:
    exithints = [In_Narrow_Corridor, &climb]
;
Plant: fixeditem, upclimbable
    size = 0

    sdesc = {
        if (self.size = 0)
            "plant";
        else if (self.size = 1)
            "beanstalk";
        else if (self.size = 2)
            "giant beanstalk";
    }
    ldesc = {
        if (self.size = 0)
            "There is a tiny little plant in the pit,
            murmuring \"Water, water, ...\"";
        else if (self.size = 1)
            "There is a 12-foot-tall beanstalk stretching
            up out of the pit, bellowing \"Water!!
            Water!!\"";
        else if (self.size = 2)
            "There is a gigantic beanstalk stretching all
            the way up to the hole.";
    }
    heredesc = {P(); I(); self.ldesc;}
    location = In_West_Pit
    noun = 'plant' 'beanstalk' 'stalk'
    adjective = 'bean' 'giant' 'tiny' 'little' 'murmuring'
        '12-foot-tall' 'twelve' 'foot' 'tall' 'bellowing'

    verDoClimb(actor) = {
        if (self.size = 0)
            "It's just a little plant!";
    }
    doClimb(actor) = {
        if (self.size = 1) {
            "%You% %have% climbed up the plant and out of the
            pit.\b";

            actor.travelTo(At_West_End_Of_Twopit_Room);
        }
        else {
            "%You% clamber up the plant and scurry through
            the hole at the top.\b";

            actor.travelTo(In_Narrow_Corridor);
        }
    }

    verDoWater(actor) = {}
    doWater(actor) = {
    /* added cask and flask */
        if (bottle.isIn(actor) and bottle.haswater)
            bottle.doPourOn(actor, self);
        else if (cask.isIn(actor) and cask.haswater)
            cask.doPourOn(actor, self);
        else if (flask.isIn(actor) and flask.haswater)
            flask.doPourOn(actor, self);
        else
            "%You% %have% nothing to water the plant with.";
    }
    verDoOil(actor) = {}
    doOil(actor) = {
    /* added cask and flask */
        if (bottle.isIn(actor) and bottle.hasoil)
            bottle.doPourOn(actor, self);
        else if (cask.isIn(actor) and cask.hasoil)
            cask.doPourOn(actor, self);
        else if (flask.isIn(actor) and flask.hasoil)
            flask.doPourOn(actor, self);
        else
            "%You% %have% nothing to oil the plant with.";
    }

    // The plant's not going anywhere.
    verifyRemove(actor) = {
        "The plant has exceptionally deep roots and cannot be
        pulled free.";
    }
    verDoPull(actor) = {self.verifyRemove(actor);}
    verDoTake(actor) = {self.verifyRemove(actor);}
    water = {
        self.size := self.size + 1;

        if (self.size = 1)
            {P(); I();"The plant spurts into furious growth for a
            few seconds.";}
        else if (self.size = 2)
            {P(); I();"The plant grows explosively, almost filling
            the bottom of the pit.";}
        else {
            {P(); I(); "You've over-watered the plant!  It's
            shriveling up!  It's, it's...";}

            self.size := 0;
        }

        P(); I(); Plant.ldesc;
    }
;

WestPit: floatingdecoration, downclimbable // DJP - reworked.
    sdesc = "western pit"
    ldesc = "The best way to examine the pit would be to go down into
        it. "
    noun = 'pit' 'corner'
    adjective = 'west' 'western' 'w'
    loclist = [At_West_End_Of_Twopit_Room
           At_East_End_Of_Twopit_Room]

    verDoClimb(actor) = {
        if(actor.isIn(At_East_End_Of_Twopit_Room)) {
            "You'll have to go over to the west end of the room first.";
        }
    }
    verDoEnter(actor) = {
        if(actor.isIn(At_East_End_Of_Twopit_Room)) {
            "You'll have to go over to the west end of the room first.";
        }
    }
    doClimb(actor) = {
        self.doEnter(actor);
    }
    doEnter(actor) = {
        actor.travelTo(actor.location,&down);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}

    // The pit is a distant item in some locations.
    dobjGen(a, v, i, p) = {
        if (self.isIn(At_East_End_Of_Twopit_Room)) {
            if ((v <> inspectVerb) and (v <> gonearVerb) and
            (v <> countVerb)) {
                "It's too far away. ";
                exit;
            }
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb) return;
        self.dobjGen(a, v, d, p);
    }

;
WestPit2: fixeditem, upclimbable
    sdesc = "western pit"
    ldesc = {In_West_Pit.ldesc;}
    noun = 'pit' 'corner'
    adjective = 'west' 'western'
    location = In_West_Pit

    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&up);
    }
    verDoEnter(actor) = {
        "You're already in the pit!";
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&up);
    }
;
HoleAbovePit_2: fixeditem,distantItem
    sdesc = "hole above pit"
    ldesc = "The hole is in the wall above you."
    noun = 'hole'
    adjective = 'large'
/* DJP: this hole is seen from the west pit. */
    location = In_West_Pit

    verDoClimbup(actor) = {
        self.verDoEnter(actor);
    }

    verDoClimb(actor) = {
        self.verDoEnter(actor);
    }

    verDoEnter(actor) = {
        if(Plant.size <> 2) {
            "%You're% not anywhere near the hole -- it's far
            overhead.";
        }
    }

    doClimbup(actor) = {
        self.doEnter(actor);
    }

    doClimb(actor) = {
        self.doEnter(actor);
    }

    doEnter(actor) = {
        actor.travelTo(actor.location,&climb);
    }

    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'

;

/* 27 */
West_Side_Of_Fissure: darkroom
    sdesc = "West Side of Fissure"
    ldesc = {
        I(); "You are on the west side of the fissure in the
        hall of mists.";
    }

    west = At_West_End_Of_Hall_Of_Mists

    forwards = { return self.jump; }
    jump = {
        if (CrystalBridge.exists) {
            "I respectfully suggest you go across the
            bridge instead of jumping.";

            return nil;
        }
        else
            return didnt_make_it.death;
    }

    over = { return self.across; }
    east = { return self.across; }
    cross = { return self.across; }
    across = {
        local actor := getActor(&travelActor);
        CrystalBridge.doCross(actor);
        return nil;
    }

    north = {
        local actor := getActor(&travelActor);
        actor.nextRoute := 1; // DJP
        "%You% %have% crawled through a very low wide passage
        parallel to and north of the hall of mists.\b";

        return At_West_End_Of_Hall_Of_Mists;
    }

    //
    // NPC's can go across too, but only if the bridge exists.
    //
    NPCexit1 = {
        if (CrystalBridge.exists)
            return On_East_Bank_Of_Fissure;
        else
            return nil;
    }
    isfissureroom = true // DJP
    // Exit info. for 'back' command:
    exithints = [On_East_Bank_Of_Fissure, &east]
    listendesc = {
        if (global.oldgame) pass listendesc;
        global.listenadd := true;
        inherited.listendesc;
        P(); "You hear a distant roar, like the sound of a fast-flowing
             river, from the depths of the fissure. ";
        global.listenadd := nil;
    }
    ana2 = Blue_East_Bank_Of_Fissure
;

/* 28 */
Low_N_S_Passage: darkroom
    sdesc = "Low N/S Passage"
    ldesc = {
        I(); "You are in a low N/S passage at a hole in the
        floor.  The hole goes down to an E/W passage.";
    }
    hall = In_Hall_Of_Mt_King
    out = In_Hall_Of_Mt_King
    south = In_Hall_Of_Mt_King
    north = At_Y2
    y2 = At_Y2
    down = In_Dirty_Passage
    hole = In_Dirty_Passage
;

/* 29 */
In_South_Side_Chamber: darkroom
    sdesc = "In South Side Chamber"
    ldesc = {
        I(); "You are in the south side chamber.";
    }
    hall = In_Hall_Of_Mt_King
    out = In_Hall_Of_Mt_King
    north = In_Hall_Of_Mt_King
;

/* 30 */
In_West_Side_Chamber: darkroom
    sdesc = "In West Side Chamber"
    ldesc = {
        I(); "You are in the west side chamber of the hall of
        the mountain king. A passage continues west and up
        here.";
    }
    hall = In_Hall_Of_Mt_King
    out = In_Hall_Of_Mt_King
    east = In_Hall_Of_Mt_King
    west = Crossover
    up = Crossover
;

/* 31 - 32 are messages */
/* 33 */
At_Y2: darkroom
    sdesc = "At \"Y2\""
    ldesc = {
        local actor := getActor(&verbActor);
        I(); "You are in a large room, with a passage to the
        south, a passage to the west, and a wall of broken
        rock to the east. There is a large \"Y2\" on ";

        if (actor.isIn(Y2Rock))
             "the rock you are sitting on.";
        else
             "a rock in the room's center.";
    }
    plugh = Inside_Building
    south = Low_N_S_Passage
    east = Jumble_Of_Rock
    wall = Jumble_Of_Rock
    broken = Jumble_Of_Rock
    west = At_Window_On_Pit_1
    plover = {
        local actor := getActor(&travelActor);
        // DJP - corrected to use new isInside function so that
        // the emerald is dropped even if it's in a closed
        // container!
        if (egg_sized_emerald.isInside(actor))
            egg_sized_emerald.moveInto(At_Y2);
        return In_Plover_Room;
    }

    hollowvoice = {
        if (not parserGetMe().isIn(self))return;
        if (rand(100) <= 25) {
            P(); I(); "A hollow voice says, \"Plugh.\"\n";
        }
    }

;

Y2Rock: fixeditem, chairitem, readable
    sdesc = "\"Y2\" rock"
    ldesc = "There is a large \"Y2\" painted on the rock."
    noun = 'rock'
    adjective = 'y2'

    location = At_Y2

    plugh = { return self.location.plugh; }
    plover = { return self.location.plover; }

    onroom = true   // We sit ON the rock, not IN it.

    //
    // We want the player to be able to pick things in the
    // room up while sitting on the rock.
    reachable = {return self.location.allcontents;}
    // This makes the contents of containers accessible as well:
    canReachContents = true

;

Jumble_Of_Rock: darkroom
    sdesc = "Jumble of Rock"
    ldesc = {
        I(); "You are in a jumble of rock, with cracks
        everywhere.";
    }
    down = At_Y2
    y2 = At_Y2
    up = In_Hall_Of_Mists
;

/* 34 */
At_Window_On_Pit_1: darkroom
    sdesc = "At Window on Pit"
    ldesc = {
        I(); "%You're% at a low window overlooking a huge pit,
        which extends up out of sight.  A floor is
        indistinctly visible over 50 feet below.  Traces of
        white mist cover the floor of the pit, becoming
        thicker to the right. Marks in the dust around the
        window would seem to indicate that someone has been
        here recently.  Directly across the pit from you and
        25 feet away there is a similar window looking into a
        lighted room.  A shadowy figure can be seen there ";
        "peering back at %you%.";
    }
    east = At_Y2
    y2 = At_Y2
    jump = {
        if (not Window.isopen) {
            "\n(opening the window)\n";
        }
        return broken_neck.death;
    }
    reflect = {
        // stolen from the 550-point game
        if (global.newgame) return At_Window_On_Pit_2;
        else pass reflect;
    }
;
Window: floatingdecoration
    iswavetarget = true
    sdesc = "window"
    ldesc = "It looks like a regular window."
    noun = 'window'
    adjective = 'low'
    loclist = [ At_Window_On_Pit_1  At_Window_On_Pit_2 ]
    isopen = nil

    verDoOpen(actor) = {if (self.isopen) "It's already open. ";}
    doOpen(actor) = {
        "OK, the window is now open.  You notice that
        the shadowy figure opened his window at the same time. ";
        self.isopen := true;
    }
    verDoClose(actor) = {if (not self.isopen) "It's already closed. ";}
    doClose(actor) = {
        "OK, the window is now closed.  You notice that
        the shadowy figure closed his window at the same time. ";
        self.isopen := nil;
}
;
WindowPit: floatingdecoration,distantItem
    sdesc = "huge pit"
    ldesc = {
        "It's so deep you can barely make out the floor below,
        and the top isn't visible at all.";
    }
    noun = 'pit'
    adjective = 'deep' 'large'
    loclist = [ At_Window_On_Pit_1  At_Window_On_Pit_2 ]
;
MarksInTheDust: floatingdecoration
    sdesc = "marks in the dust"
    adesc = { self.sdesc; }
    ldesc = "Evidently %you're% not alone here."
    noun = 'marks' 'dust' // DJP
    loclist = [ At_Window_On_Pit_1  At_Window_On_Pit_2 ]
    doCleanWith( actor, io ) = {theFloor.doCleanWith( actor, io );}
    doSweepWith( actor, io ) = {theFloor.doSweepWith( actor, io );}
;
ShadowyFigure: floatingdecoration,distantItem
    iswavetarget = true
    sdesc = "shadowy figure"
    ldesc = {
        "The shadowy figure looks very similar to you.
        Perhaps he's another adventurer.  He seems to be trying to attract
        your attention. ";
    }
    heredesc = { P(); I();
        "The shadowy figure seems to be trying to attract your
        attention. ";
    }
    noun = 'figure' 'shadow' 'person' 'individual'
    adjective = 'shadowy' 'mysterious'
    loclist = [ At_Window_On_Pit_1  At_Window_On_Pit_2 ]
;
/* 36 */
In_Dirty_Passage: darkroom
    sdesc = "In Dirty Passage"
    ldesc = {
        I(); "You are in a dirty broken passage.  To the east
        is a crawl.  To the west is a large passage.  Above
        you is a hole to another passage.";
    }
    east = On_Brink_Of_Pit
    crawl = On_Brink_Of_Pit
    up = Low_N_S_Passage
    hole = Low_N_S_Passage
    west = In_Dusty_Rock_Room
    bedquilt = In_Bedquilt
    slab = In_Slab_Room     // DMB: this is only in some versions
;

/* 37 */
On_Brink_Of_Pit: darkroom
    sdesc = "On Brink of Pit"
    ldesc = {
        I(); "You are on the brink of a small clean climbable
        pit.  A crawl leads west.";
    }
    west = In_Dirty_Passage
    crawl = In_Dirty_Passage
    down = In_Pit
    pit = In_Pit
    climb = In_Pit
    in = In_Pit     // DMB: added
;
CleanPit: fixeditem, downclimbable
    sdesc = "small pit"
    ldesc = "It looks like you might be able to climb down into it."
    noun = 'pit'
    adjective = 'small' 'clean' 'climable'
    location = On_Brink_Of_Pit

    verDoClimb(actor) = {}

    doClimb(actor) = {        
        actor.travelTo(self.location,&climb); 
    }
    verDoEnter(actor) = {}
    doEnter(actor) = { self.doClimb(actor); }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
;
CleanPit2: fixeditem, upclimbable
    sdesc = "small pit"
    ldesc = {In_Pit.ldesc;}
    noun = 'pit'
    adjective = 'small' 'clean' 'climable'
    location = In_Pit

    verDoClimb(actor) = {}

    doClimb(actor) = {
        actor.travelTo(self.location,&up); 
    }
    verDoEnter(actor) = {"You're already in the pit!"; }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&up);
    }
;
/* 38 */
In_Pit: darkroom, NoNPC
    sdesc = "In Pit"
    ldesc = {
        I(); "You are in the bottom of a small pit with a
        little stream, which enters and exits through tiny
        slits.";
    }
    climb = On_Brink_Of_Pit
    up = On_Brink_Of_Pit
    out = On_Brink_Of_Pit

    slit = { return self.down; }
    stream = { return self.down; }
    upstream = { return self.down; }
    downstream = { return self.down; }
    down = {
        // In the original, the same message given
        // in At_Slit_In_Streambed was used here.
        // Since it's not quite right (and was probably only
        // reused to save space), I've changed it slightly.

        "You don't fit through the tiny slits!";
        return nil;
    }
;
PitSlits: decoration
    sdesc = "tiny slits"
    adesc = { self.sdesc; }
    ldesc = {
        "The slits form a complex pattern in the rock.";
    }
    location = In_Pit
    noun = 'slit' 'slits'
    adjective = 'tiny'
;

/* 39 */
In_Dusty_Rock_Room: darkroom
    sdesc = "In Dusty Rock Room"
    ldesc = {
        I(); "You are in a large room full of dusty rocks.
        There is a big hole in the floor.  There are cracks
        everywhere, and a passage leading east.";
    }
    east = In_Dirty_Passage
    passage = In_Dirty_Passage
    down = At_Complex_Junction
    hole = At_Complex_Junction
    floor = At_Complex_Junction
    bedquilt = In_Bedquilt
;
DustyRocks: fixeditem, readable
    // DJP - for the 551-point game, we change the description to hint
    // that they should be swept.  The magazines also hint at the same
    // puzzle.  They can't be swept in a 350-point game, due to the lack
    // of a broom.
    sdesc = "dusty rocks"
    ldesc = {
        if (global.oldgame)
            "They're just rocks.  (Dusty ones, that is.)";
        else if (self.areswept)
            "They are covered with a thick coating of dust.  However, the
            dust has been swept away from one of the larger rocks,
            revealing a carved inscription. ";
        // this happens if we sit on the throne without a crown
        else if (SafeCombination.isseen)
            "They are covered with a thick coating of dust.  I know that
            there is an inscription on one of the rocks, but I can no
            longer see it - you'll have to sweep the rock again. ";
        else
            "They are covered with a thick coating of dust.  You'll have
            to find some way to remove it before you can examine them
            properly. ";

    }
    readdesc = {
        local i;
        if(not self.areswept) {
            if (SafeCombination.isseen)
                "Sorry, but I can't quite remember what the inscription
                reads.  You'll have to sweep the rock again. ";
            else
                "Even if there was anything readable, I wouldn't be able
                to see it for all the dust!";
        }
        else {
            "In the rock is carved the message: \"";
            for (i := 1; i <= Dial.comblen; i++) {
                say(Dial.combination[i]); if (i < Dial.comblen) "-";
            }
            "\". ";
        }
    }
    location = In_Dusty_Rock_Room
    noun = 'rocks' 'boulders' 'stones' 'rock' 'boulder' 'stone'
    adjective = 'dusty' 'dirty'
    doCleanWith( actor, io ) =
    {
        if(io = whiskbroom) {
            if(not self.areswept) self.sweep;
            else "Enough dusting, already!  %You're% making me sneeze.
                If %you% want to read the inscription, please say so. ";
        }
        else pass doCleanWith;
    }
    doSweepWith( actor, io ) =
    {
        if(io = whiskbroom) {
            if(not self.areswept) self.sweep;
            else "Enough dusting, already!  %You're% making me sneeze.
                If %you% want to read the inscription, please say so. ";
        }
        else pass doSweepWith;
    }
    sweep = {
        local i;
        if (not Dial.combination_set) Dial.setcomb;
        "Brushing the dust from one of the larger rocks reveals some carved
        characters.  They appear to read: \"";
        for (i := 1; i <= Dial.comblen; i++) {
            say(Dial.combination[i]); if (i < Dial.comblen) "-";
        }
        "\". ";
        self.areswept := true;
        SafeCombination.moveInto(self.location);
        SafeCombination.isseen := true;
    }
;

RockDust: fixeditem
    sdesc = "dust"
    ldesc = "It looks like ordinary dust. "
    noun = 'dust'
    location = In_Dusty_Rock_Room
    verDoTake(actor) = {
        if(DustyRocks.areswept) {
            "You've already removed as much dust as you need to. ";
        }
        else if (global.newgame) {
            "You'll have to tell me how to do that. ";
        }
        else pass verDoTake;
    }   
    verDoTakeOut(actor, io) = {
        if (io = DustyRocks or io = theFloor) self.verDoTake(actor);
        else {
            caps(); "<<self.thedesc>> is on the rocks, not <<io.thedesc>>! ";
        }
    }
    doCleanWith( actor, io ) = {DustyRocks.doCleanWith( actor, io );}
    doSweepWith( actor, io ) = {DustyRocks.doSweepWith( actor, io );}
;

SafeCombination: fixeditem, readable
    sdesc = "carved inscription"
    ldesc = "They are just 2-inch high characters, carved into the rock."
    readdesc = {
        local i;
        "They read: ";
        for (i := 1; i <= Dial.comblen; i++) {
            say(Dial.combination[i]); if (i < Dial.comblen) "-";
        }
        ". ";
    }
    noun = 'characters' 'inscription' 'writing' 'combination'
    adjective = 'carved'
    location = nil
;
/* 40 is a message */
/* 41 */
At_West_End_Of_Hall_Of_Mists: darkroom
    sdesc = "At West End of Hall of Mists"
    ldesc = {
        I(); "You are at the west end of the hall of mists.
        A low wide crawl continues west and another goes
        north.  To the south is a little passage 6 feet off
        the floor.";
    }
    south = Alike_Maze_1
    up = Alike_Maze_1
    passage = Alike_Maze_1
    climb = Alike_Maze_1
    east = West_Side_Of_Fissure
    west = At_East_End_Of_Long_Hall
    crawl = At_East_End_Of_Long_Hall

    north = {
        local actor := getActor(&travelActor);
        actor.nextRoute := 1; // DJP
        "%You% %have% crawled through a very low wide passage
        parallel to and north of the hall of mists.\b";

        return West_Side_Of_Fissure;
    }
    ana2 = Blue_West_End_Of_Hall_Of_Mists
;

/* 42 */
Alike_Maze_1: CCR_alike_maze_room
    up = At_West_End_Of_Hall_Of_Mists
    north = Alike_Maze_1
    east = Alike_Maze_2
    south = Alike_Maze_4
    west = Alike_Maze_11
    // Help NPC's escape from the maze
    NPCexit1 = At_West_End_Of_Hall_Of_Mists
    NPCexit2 = At_West_End_Of_Hall_Of_Mists
;

/* 43 */
Alike_Maze_2: darkroom, CCR_alike_maze_room
    west = Alike_Maze_1
    south = Alike_Maze_3
    east = Alike_Maze_4
    // Help NPC's escape from the maze
    NPCexit1 = At_West_End_Of_Hall_Of_Mists
    NPCexit2 = At_West_End_Of_Hall_Of_Mists
;

/* 44 */
Alike_Maze_3: darkroom, CCR_alike_maze_room
    east = Alike_Maze_2
    down = Dead_End_3
    south = Alike_Maze_6
    north = Dead_End_10 // DJP - corrected.  This wrongly led to the
                        // pirate's dead end!
;

/* 45 */
Alike_Maze_4: darkroom, CCR_alike_maze_room
    west = Alike_Maze_1
    north = Alike_Maze_2
    east = Dead_End_1
    south = Dead_End_2
    up = Alike_Maze_14
    down = Alike_Maze_14
    // Help NPC's escape from the maze
    NPCexit1 = At_West_End_Of_Hall_Of_Mists
    NPCexit2 = At_West_End_Of_Hall_Of_Mists
;

/* 46 */
Dead_End_1: darkroom, CCR_dead_end_room
    west = Alike_Maze_4
    out = Alike_Maze_4
;

/* 47 */
Dead_End_2: darkroom, CCR_dead_end_room
    east = Alike_Maze_4
    out = Alike_Maze_4
;

/* 48 */
Dead_End_3: darkroom, CCR_dead_end_room
    up = Alike_Maze_3
    out = Alike_Maze_3
;

/* 49 */
Alike_Maze_5: darkroom, CCR_alike_maze_room
    east = Alike_Maze_6
    west = Alike_Maze_7
;

/* 50 */
Alike_Maze_6: darkroom, CCR_alike_maze_room
    east = Alike_Maze_3
    west = Alike_Maze_5
    down = Alike_Maze_7
    south = Alike_Maze_8
;

/* 51 */
Alike_Maze_7: darkroom, CCR_alike_maze_room
    west = Alike_Maze_5
    up = Alike_Maze_6
    east = Alike_Maze_8
    south = Alike_Maze_9
;

/* 52 */
Alike_Maze_8: darkroom, CCR_alike_maze_room
    west = Alike_Maze_6
    east = Alike_Maze_7
    south = Alike_Maze_8
    up = Alike_Maze_9
    north = Alike_Maze_10
    down = Dead_End_12
;

/* 53 */
Alike_Maze_9: darkroom, CCR_alike_maze_room
    west = Alike_Maze_7
    north = Alike_Maze_8
    south = Dead_End_4
;

/* 54 */
Dead_End_4: darkroom, CCR_dead_end_room
    west = Alike_Maze_9
    out = Alike_Maze_9
;

/* 55 */
Alike_Maze_10: darkroom, CCR_alike_maze_room
    west = Alike_Maze_8
    north = Alike_Maze_10
    down = Dead_End_5
    east = At_Brink_Of_Pit
;

/* 56 */
Dead_End_5: darkroom, CCR_dead_end_room
    up = Alike_Maze_10
    out = Alike_Maze_10
;

/* 57 */
At_Brink_Of_Pit: darkroom
    sdesc = "At Brink of Pit"
    ldesc = {
        I(); "You are on the brink of a thirty foot pit with
        a massive orange column down one wall.  You could
        climb down here but you could not get back up.  The
        maze continues at this level.";
    }
    down = In_Bird_Chamber
    climb = In_Bird_Chamber
    west = Alike_Maze_10
    south = Dead_End_6
    north = Alike_Maze_12
    east = Alike_Maze_13
    // Allow NPC's out of here (to the Hall of Mists)
    NPCexit1 = In_Hall_Of_Mists
    NPCexit2 = In_Hall_Of_Mists
;
OrangeColumn: fixeditem, downclimbable
    sdesc = "massive orange column"
    ldesc = "It looks like you could climb down it."
    noun = 'column'
    adjective = 'massive' 'orange' 'big' 'huge'
    location = At_Brink_Of_Pit

    verDoClimb(actor) = {}
    doClimb(actor) = {        actor.travelTo(self.location,&down); }
;
ThirtyFootPit: fixeditem, downclimbable
    sdesc = "pit"
    ldesc = "You'll have to climb down to find out anything more..."
    noun = 'pit'
    adjective = 'thirty' 'foot' 'thirty-foot' '30-foot'
    location = At_Brink_Of_Pit

    verDoClimb(actor) = {}
    doClimb(actor) = {        actor.travelTo(self.location,&down); }
    verDoEnter(actor) = {}
    doEnter(actor) = { self.doClimb(actor); }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
;

/* 58 */
Dead_End_6: darkroom, CCR_dead_end_room
    east = At_Brink_Of_Pit
    out = At_Brink_Of_Pit
;

/* 59 is a message */
/* 60 */
At_East_End_Of_Long_Hall: darkroom
    sdesc = "At East End of Long Hall"
    ldesc = {I();
    "You are at the east end of a very long hall apparently without side
    chambers.  ";
    if(global.newgame) {
        "In the south wall are several wide cracks and a high
        hole, but the hole is far above ";"%your% head. ";
    }
    "To the east a wide crawl slants up.  To the north a round two foot
    hole slants down.";}
    east = At_West_End_Of_Hall_Of_Mists
    up = At_West_End_Of_Hall_Of_Mists
    crawl = At_West_End_Of_Hall_Of_Mists
    west = At_West_End_Of_Long_Hall
    north = Crossover
    down = Crossover
    hole = Crossover
    south = {
        local i,l,o,movecontents;
        if (global.oldgame) pass south;
        // Determine the moveable contents of the dead-end crack
        movecontents := Dead_End_Crack.contents;
        l := length(Dead_End_Crack.contents);
        for (i := 1; i <= l; i++) {
            o := Dead_End_Crack.contents[i];
            if (o.isfixed) movecontents -= o;
        }
        // Note that we play a slight trick on the player.   We don't
        // choose the middle crack (which looks unpromising).  If
        // the right crack has been seen (and is therefore known to
        // be a dead end), we choose the left one.
        if ((not Dead_End_Crack.isseen) or length(movecontents) <> 0) {
            "\n(choosing one of the wider cracks at random)\n";
            if (rand(100) <= 50) return Tight_Crack;
            else return Dead_End_Crack;
        }
        else {
            "\n(choosing the left crack)\n";
            return Tight_Crack;
        }
    }
    crack = {return self.south;}
    climb = {if(global.oldgame) pass climb;
            "It is too far up for you to reach.";return nil;}

    left = {if(global.newgame) return Tight_Crack; else pass left;}
    middle = {if(global.newgame) return Tight_Crack_2; else pass middle;}
    right = {if(global.newgame) return Dead_End_Crack; else pass right;}

    NPCexit1 = {
        if(global.newgame) return Tight_Crack;
        else return nil;
    }
    NPCexit2 = {
        if(global.newgame) return Tight_Crack_2;
        else return nil;
    }
    NPCexit3 = {
        if(global.newgame) return Dead_End_Crack;
        else return nil;
    }
    exithints = [
        Tight_Crack, &left, Tight_Crack_2, &middle,
        Dead_End_Crack, &right
    ]
    ana2 = Blue_East_End_Of_Long_Hall
;


/* Define cracks, high hole, round hole as objects and allow them to
be entered */
Hallcracks: floatingdecoration
    game551 = true // in 551-point game only
    sdesc = "cracks"
    ldesc = {
        if(self.location.analevel = 1) {
            "There are three cracks in the south wall, but the middle and
            left cracks have been blocked with concrete.  ";
        }
        else {
            "There are three cracks in the south wall.  To choose which
            one to enter, type LEFT, MIDDLE or RIGHT.  ";
            if(self.location.analevel <> 2) {
                "The middle crack is narrower than the other two";
                if (Tight_Crack_2.isseen)
                    ". ";
                else 
                    " and it doesn't appear that you can proceed very far down 
                    it. ";
            }
        }
    }
    noun = 'cracks'
    loclist = [At_East_End_Of_Long_Hall, Blue_East_End_Of_Long_Hall,
        Green_East_End_Of_Long_Hall]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&crack);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
;

LeftCrack: floatingdecoration
    game551 = true // in 551-point game only
    sdesc = "left crack"
    ldesc = "It's one of three cracks on the south wall.  It looks large
            enough to enter. "
    loclist = [At_East_End_Of_Long_Hall, Blue_East_End_Of_Long_Hall, 
        Green_East_End_Of_Long_Hall]

    noun = 'crack'
    adjective = 'left'
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&left);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
;

RightCrack: floatingdecoration
    game551 = true // in 551-point game only
    sdesc = "right crack"
    ldesc = {
        "It's one of three cracks on the south wall.  It looks large
        enough to enter";
        if(self.location.analevel=1)
           ", but unfortunately it has been blocked up with concrete. ";
        else
           ". ";
    }
    noun = 'crack'
    adjective = 'right'
    loclist = [At_East_End_Of_Long_Hall, Blue_East_End_Of_Long_Hall, 
        Green_East_End_Of_Long_Hall]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&right);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
;

MiddleCrack: floatingdecoration
    game551 = true // in 551-point game only
    sdesc = "middle crack"
    ldesc = {
        if(self.location.analevel = 2)
            "It's one of three cracks on the south wall.  It may once have been
            narrow like its counterpart at Red level, but it appears to have
            been widened at some time in the past. ";
        else {
            "It's one of three cracks on the south wall.  It's very
            narrow, although it looks just large enough to enter";
            if(self.location.analevel = 1)
                ".  Unfortunately you can't, because it has been blocked up
                with concrete. ";
            else if (Tight_Crack_2.isseen)
                ". ";
            else 
                ", but I doubt whether you could proceed very far down it. ";
        }
    }
    noun = 'crack'
    adjective = 'middle' 'center' 'centre'

    loclist = [At_East_End_Of_Long_Hall, Blue_East_End_Of_Long_Hall, 
        Green_East_End_Of_Long_Hall]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&middle);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
;

Northhole: floatingdecoration, downclimbable
    sdesc = "round hole"
    ldesc = {
        if(self.location.analevel = 1)
            "It's a round eight-foot hole, going north.  Unfortunately it
            is blocked after a short distance by a solid steel door with no
            keyhole or handle. ";
        else if(self.location.analevel = 2)
            "It's a round two-foot hole, going north.  Unfortunately it
            is blocked after a short distance by a locked iron gate. ";
        else
            "It's a round two-foot hole, going north. ";
    }
    noun = 'hole'
    adjective = 'n' 'north' 'northern' 'two-foot' 'eight-foot' 'round'
    loclist = [At_East_End_Of_Long_Hall, Blue_East_End_Of_Long_Hall,
        Green_East_End_Of_Long_Hall]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&north);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&north);
    }
;

Highhole: floatingdecoration, upclimbable,distantItem
    game551 = true // only present in 551-point game
    sdesc = "high hole"
    ldesc = "It's high above your head."
    noun = 'hole'
    adjective = 'high'
    loclist = [At_East_End_Of_Long_Hall, Blue_East_End_Of_Long_Hall,
               Green_East_End_Of_Long_Hall]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&climb);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&climb);
    }
;

/* 61 */
At_West_End_Of_Long_Hall: darkroom
    sdesc = "At West End of Long Hall"
    ldesc = {
         I(); "You are at the west end of a very long
         featureless hall.  The hall joins up with a narrow
         north/south passage.";
    }
    east = At_East_End_Of_Long_Hall
    north = Crossover
    south = Different_Maze_1
    ana2 = Blue_West_End_Of_Long_Hall
;

/* 62 */
Crossover: darkroom
    sdesc = "N/S and E/W Crossover"
    ldesc = {
         I(); "You are at a crossover of a high N/S passage
         and a low E/W one.";
    }
    west = At_East_End_Of_Long_Hall
    north = Dead_End_7
    east = In_West_Side_Chamber
    south = At_West_End_Of_Long_Hall
;

TheCrossover: fixeditem
    sdesc = "crossover"
    ldesc = "You know as much as I do at this point."
    noun = 'crossover' 'over'
    adjective = 'cross'
    location = Crossover
;

/* 63 */
Dead_End_7: darkroom, CCR_dead_end_room
    ldesc = {
          I(); "You have reached a dead end.";
         }
    south = Crossover
    out = Crossover
;

StatueMessage: fixeditem, readable // DJP: changed to use a heredesc
    game550 = true
    sdesc = "message on the wall"
    ldesc = {
         "The message reads, \"Stand where the statue gazes, and
         make use of the proper tool.\"";
    }
    heredesc = {
         P(); I();
         "Scratched on the wall is the
          message, \"Stand where the statue gazes, and make use of
         the proper tool.\"";
    }
    noun = 'message' 'scrawl' 'writing' 'script'
    adjective = 'scratched'
    location = Dead_End_7
;

/* 64 */
At_Complex_Junction: darkroom
    sdesc = "At Complex Junction"
    ldesc = {
         I(); "You are at a complex junction.  A low hands and
         knees passage from the north joins a higher crawl
         from the east to make a walking passage going west.
         There is also a large room above.  The air is damp
         here.";
    }
    up = In_Dusty_Rock_Room
    climb = In_Dusty_Rock_Room
    toroom = In_Dusty_Rock_Room
    hole = In_Dusty_Rock_Room
    west = In_Bedquilt
    bedquilt = In_Bedquilt
    north = In_Shell_Room
    shell = In_Shell_Room
    east = In_Anteroom
;

/* 65 */
In_Bedquilt: darkroom
    sdesc = "In Bedquilt"
    ldesc = {
         I(); "You are in bedquilt, a long east/west passage
         with holes everywhere. To explore at random select
         north, south, up, or down.";
    }
    east = At_Complex_Junction
    west = In_Swiss_Cheese_Room
    south = {
        local actor := getActor(&travelActor);
        if (rand(100) <= 80 and not
        clover.isInside(actor))
            return crawled_around.message;
        else
            return self.slab;
    }
    slab = In_Slab_Room

    up = {
        local actor := getActor(&travelActor);
        if (rand(100) <= 80 and not
        clover.isInside(actor))
            return crawled_around.message;
        else if (rand(100) <= 50)
            return In_Secret_N_S_Canyon_1;
        else
            return In_Dusty_Rock_Room;
    }
    /* allow holes to be climbed up or down */
    uphole = true
    downhole = true

/* additional direction  - DJP */

    secret = In_Secret_N_S_Canyon_1

    north = {
        local actor := getActor(&travelActor);
        if (rand(100) <= 60 and not
        clover.isInside(actor))
            return crawled_around.message;
        else if (rand(100) <= 75)
            return In_Large_Low_Room;
        else
            return At_Junction_Of_Three_Secret_Canyons;
    }

    down = {
        local actor := getActor(&travelActor);
        if (rand(100) <= 80 and not
        clover.isInside(actor))
            return crawled_around.message;
        else
            return In_Anteroom;
    }
    hole = {
        "There are holes everywhere, so you'll have to tell me which
        direction you want to go in.";
    }

    //
    // Let the NPC's go everywhere out of here too.
    //
    NPCexit1 = In_Dusty_Rock_Room
    NPCexit2 = In_Large_Low_Room
    NPCexit3 = At_Junction_Of_Three_Secret_Canyons
    NPCexit4 = In_Anteroom
    // Exit info. for 'back' command:
    exithints = [
        In_Secret_N_S_Canyon_1, &secret,
        In_Large_Low_Room, &north,
        At_Junction_Of_Three_Secret_Canyons, &north,
        In_Anteroom, &down
    ]
;

/* 66 */
In_Swiss_Cheese_Room: darkroom
    sdesc = "In Swiss Cheese Room"
    ldesc = {
        I(); "You are in a room whose walls resemble swiss
        cheese.  Obvious passages go west, east, ne, and nw.
        Part of the room is occupied by a large bedrock
        block.";
    }
    ne = In_Bedquilt
    west = At_East_End_Of_Twopit_Room
    south = {
        local actor := getActor(&travelActor);
        if (rand(100) <= 80 and not
        clover.isInside(actor))
            return crawled_around.message;
        else
            return self.canyon;
    }
    canyon = In_Tall_E_W_Canyon

    east = In_Soft_Room
    nw = {
        local actor := getActor(&travelActor);
        if (self.foundorient)
            return self.oriental;
        if (rand(100) <= 50 and not
        clover.isInside(actor))
            return crawled_around.message;
        else
            self.foundorient := true;
            return self.oriental;
    }
    oriental = {
        self.foundorient := true;
        return In_Oriental_Room;
    }

    // let NPC's out of here
    NPCexit1 = In_Oriental_Room

    exithints = [ In_Oriental_Room, &oriental ]
;
BedrockBlock: fixeditem
    sdesc = "bedrock block"
    ldesc = "It's just a huge block."
    noun = 'block'
    adjective = 'bedrock' 'large'
    location = In_Swiss_Cheese_Room

    verDoLookunder(actor) = { "Surely you're joking."; }
    verDoMove(actor) = { self.verDoLookunder(actor); }
    verifyRemove(actor) = { self.verDoLookunder(actor); }
;

/* 67 */
At_East_End_Of_Twopit_Room: darkroom
    sdesc = "At East End of Twopit Room"
    ldesc = {
        I(); "You are at the east end of the twopit room.
        The floor here is littered with thin rock slabs,
        which make it easy to descend the pits. There is a
        path here bypassing the pits to connect passages from
        east and west.  There are holes all over, but the
        only big one is on the wall directly over the west
        pit where you can't get to it.";

        if (PlantStickingUp.isIn(self)) {
            P(); I(); PlantStickingUp.ldesc;
        }
    }
    east = In_Swiss_Cheese_Room
    west = At_West_End_Of_Twopit_Room
    across = At_West_End_Of_Twopit_Room
    down = In_East_Pit
    pit = In_East_Pit
    hole = {
        "The only hole you could enter is on the wall directly over the
        west pit, and you can't get to it from here. ";
        return nil;
    }
;
Slabs: fixeditem
    sdesc = "thin rock slabs"
    adesc = { self.sdesc; }
    ldesc = "They almost form natural stairs down into the pit."
    noun = 'slabs' 'slab' 'rocks' 'stairs'
    adjective = 'thin' 'rock'
    location = At_East_End_Of_Twopit_Room

    verDoLookunder(actor) = { "Surely you're joking."; }
    verDoMove(actor) = { self.verDoLookunder(actor); }
    verifyRemove(actor) = { self.verDoLookunder(actor); }
;

/* 68 */
In_Slab_Room: darkroom
    sdesc = "In Slab Room"
    ldesc = {
        I(); "You are in a large low circular chamber whose
        floor is an immense slab fallen from the ceiling
        (slab room).  East and west there once were large
        passages, but they are now filled with boulders.  Low
        small passages go north and south, and the south one
        quickly bends west around the boulders.";
    }
    south = At_West_End_Of_Twopit_Room
    up = In_Secret_N_S_Canyon_0
    climb = In_Secret_N_S_Canyon_0
    north = In_Bedquilt
;
Slab: fixeditem
    sdesc = "slab"
    ldesc = "It is now the floor here."
    noun = 'slab'
    adjective = 'immense'
    location = In_Slab_Room
;
SlabBoulders: fixeditem
    sdesc = "boulders"
    ldesc = "They're just ordinary boulders."
    noun = 'boulder' 'boulders' 'rocks' 'stones'
    location = In_Slab_Room
;

/* 69 */
In_Secret_N_S_Canyon_0: darkroom
    sdesc = "In Secret N/S Canyon"
    ldesc = {
        I(); "You are in a secret N/S canyon above a large
        room.";
    }
    down = In_Slab_Room
    slab = In_Slab_Room

    south = In_Secret_Canyon

    north = In_Mirror_Canyon
    reservoir = At_Reservoir

;

/* 70 */
In_Secret_N_S_Canyon_1: darkroom
    sdesc = "In Secret N/S Canyon"
    ldesc = {
        I(); "You are in a secret N/S canyon above a sizable
        passage.";
    }
    north = At_Junction_Of_Three_Secret_Canyons
    down = In_Bedquilt
    passage = In_Bedquilt
    south = Atop_Stalactite
;

/* 71 */
At_Junction_Of_Three_Secret_Canyons: darkroom
    sdesc = "At Junction of Three Secret Canyons"
    ldesc = {
        I(); "You are in a secret canyon at a junction of
        three canyons, bearing north, south, and se.  The
        north one is as tall as the other two combined.";
    }
    se = In_Bedquilt
    south = In_Secret_N_S_Canyon_1
    north = At_Window_On_Pit_2
;

/* 72 */
In_Large_Low_Room: darkroom
    // for some reason the directions in the 551-point game have
    // been changed, so the room description, travel and exithints methods
    // all have to check for the global.newgame variable.  The 551-point
    // room description has been changed so that the order of the
    // directions corresponds: Dead_End_Crawl, In_Oriental_Room,
    // In_Sloping_Corridor.
    sdesc = "In Large Low Room"
    ldesc = {
        I(); "You are in a large low room.  Crawls lead
        north, "; if (global.newgame) "sw, and ne.";
        else "se, and sw.";
    }
    bedquilt = In_Bedquilt
    ne = {
        if (global.newgame) return In_Sloping_Corridor;
        else pass ne;
    }
    sw = {
        if (global.newgame) return In_Oriental_Room;
        else return In_Sloping_Corridor;
    }
    north = Dead_End_Crawl
    se = {
        if(global.oldgame) return In_Oriental_Room;
        else pass se;
    }
    oriental = In_Oriental_Room

    // let NPC's out of here
    NPCexit1 = In_Sloping_Corridor
    // Exit info. for 'back' command:
    exithints = {
        if (global.newgame) return [In_Sloping_Corridor, &ne];
        else return [In_Sloping_Corridor, &sw];
    }
;

/* 73 */
Dead_End_Crawl: darkroom, CCR_dead_end_room
    sdesc = "Dead End Crawl"
    ldesc = {
        I(); "This is a dead end crawl.";
    }
    south = In_Large_Low_Room
    crawl = In_Large_Low_Room
    out = In_Large_Low_Room
;

/* 74 */
In_Secret_E_W_Canyon: darkroom
    sdesc = "In Secret E/W Canyon Above Tight Canyon"
    ldesc = {
        I(); "You are in a secret canyon which here runs E/W.
        It crosses over a very tight canyon 15 feet below.
        If you go down you may not be able to get back up.";
    }
    east = In_Hall_Of_Mt_King
    west = In_Secret_Canyon
    down = In_N_S_Canyon

;

/* 75 */
In_N_S_Canyon: darkroom
    sdesc = "In N/S Canyon"
    ldesc = {
        I(); "You are at a wide place in a very tight N/S
        canyon.";
    }
    south = Canyon_Dead_End
    north = In_Tall_E_W_Canyon
;

/* 76 */
Canyon_Dead_End: darkroom, CCR_dead_end_room
    sdesc = "Canyon Dead End"
    ldesc = {
        I(); "The canyon here becomes too tight to go further
        south.";
    }
    north = In_N_S_Canyon
;

/* 77 */
In_Tall_E_W_Canyon: darkroom
    sdesc = "In Tall E/W Canyon"
    ldesc = {
        I(); "You are in a tall E/W canyon.  A low tight
        crawl goes 3 feet north and seems to open up.";
    }
    east = In_N_S_Canyon
    west = Dead_End_8
    north = In_Swiss_Cheese_Room
    crawl = In_Swiss_Cheese_Room
;

/* 78 */
Dead_End_8: darkroom, CCR_dead_end_room
    ldesc = {
        I(); "The canyon runs into a mass of boulders -- dead
        end.";
        if (global.game550)
            "  Scratched on one of the boulders are
            the words, \"Jerry Cornelius was here.\"";
    }
    south = In_Tall_E_W_Canyon
    out = In_Tall_E_W_Canyon        // DMB: added
;
MassOfBoulders: fixeditem
    sdesc = "mass of boulders"
    ldesc = {
        "They just like ordinary boulders.";
        if (global.game550)
             "  Scratched on one of the boulders are the words, \"Jerry
             Cornelius was here.\"";
    }
    noun = 'boulders' 'mass'
    location = Dead_End_8
;
MessageOnBoulder: fixeditem
    game550 = true
    sdesc = "message on boulder"
    ldesc = "It reads, \"Jerry Cornelius was here.\""
    noun = 'words' 'message' 'boulder'
    adjective = 'boulder' 'scratched'
    location = Dead_End_8
;

/* 80 */
Alike_Maze_11: darkroom, CCR_alike_maze_room
    north = Alike_Maze_1
    west = Alike_Maze_11
    south = Alike_Maze_11
    east = Dead_End_9
;

/* 81 */
Dead_End_9: darkroom, CCR_dead_end_room
    west = Alike_Maze_11
    out = Alike_Maze_11
;

/* 82 */
Dead_End_10: darkroom, CCR_dead_end_room
    south = Alike_Maze_3
    out = Alike_Maze_3
;

/* 83 */
Alike_Maze_12: darkroom, CCR_alike_maze_room
    south = At_Brink_Of_Pit
    east = Alike_Maze_13
    west = Dead_End_11
;

/* 84 */
Alike_Maze_13: darkroom, CCR_alike_maze_room
    north = At_Brink_Of_Pit
    west = Alike_Maze_12
    nw = Dead_End_13
;

/* 85 */
Dead_End_11: darkroom, CCR_dead_end_room
    east = Alike_Maze_12
    out = Alike_Maze_12
;

/* 86 */
Dead_End_12: darkroom, CCR_dead_end_room
    up = Alike_Maze_8
    out = Alike_Maze_8
;

/* 87 */
Alike_Maze_14: darkroom, CCR_alike_maze_room
    up = Alike_Maze_4
    down = Alike_Maze_4
;

/* 88 */
In_Narrow_Corridor: darkroom
    sdesc = "In Narrow Corridor"
    ldesc = {
        I(); "You are in a long, narrow corridor stretching
        out of sight to the west.  At the eastern end is a
        hole through which you can see a profusion of
        leaves.";
    }
    down = In_West_Pit
    climb = In_West_Pit
    east = In_West_Pit
    jump = { return broken_neck.death; }
    west = In_Giant_Room
    giant = In_Giant_Room
    // Let NPC's out of here even if we haven't grown the beanstalk
    NPCexit1 = At_West_End_Of_Twopit_Room
;
Leaves: fixeditem
    sdesc = "leaves"
    ldesc = {
        "The leaves appear to be attached to the beanstalk
        you climbed to get here.";
    }
    location = In_Narrow_Corridor
    noun = 'leaf' 'leaves' 'plant' 'tree' 'stalk' 'beanstalk' 'profusion'
;

/* 91 */
At_Steep_Incline_Above_Large_Room: darkroom
    sdesc = "At Steep Incline Above Large Room"
    ldesc = {
        I(); "You are at the top of a steep incline above a
        large room.  You could climb down here, but you would
        not be able to climb up.  There is a passage leading
        back to the north.";
    }
    north = In_Cavern_With_Waterfall
    cavern = In_Cavern_With_Waterfall
    passage = In_Cavern_With_Waterfall
    down = In_Large_Low_Room
    climb = In_Large_Low_Room
;

/* 92 */
In_Giant_Room: darkroom
    sdesc = "In Giant Room"
    ldesc = {
        I(); "You are in the giant room.  The ceiling here is
        too high up for your lamp to show it.  Cavernous
        passages lead east, north, and south.  On the west
        wall is scrawled the inscription, \"Fee fie foe foo\"
        [sic].";
    }
    south = In_Narrow_Corridor
    east = At_Recent_Cave_In
    north = In_Immense_N_S_Passage
;
Inscription: fixeditem, readable
    sdesc = "scrawled inscription"
    ldesc = "It says, \"Fee fie foe foo [sic].\""
    noun = 'inscription' 'writing' 'scrawl'
    adjective = 'scrawled'
    location = In_Giant_Room
;

/* 93 */
At_Recent_Cave_In: darkroom
    sdesc = "At Recent Cave-in"
    ldesc = {
        I(); if (global.game550) {
            "You are in a low tunnel with an irregular ceiling. ";
            CaveIn.ldesc;
        }
        else "The passage here is blocked by a recent cave-in. ";
    }
    south = In_Giant_Room
    giant = In_Giant_Room
    out = In_Giant_Room
    north = {
        if (global.game550) return Glassy_Room;
        else "The tunnel is blocked (in this version). "; return nil;
    }
    NPCexit1 = Glassy_Room
    exithints = [Glassy_Room, &north]
;
CaveIn: fixeditem
    sdesc = "cave-in"
    ldesc = {
        if (global.game550)
            "To the north, the tunnel is partially
            blocked by a recent cave-in, but you can probably
            get past the blockage without much trouble. ";
        else
            "To the north, the tunnel is blocked by a recent cave-in. ";
    }
    noun = 'in' 'cave-in'
    adjective = 'cave'
    location = At_Recent_Cave_In
;

/* 94 */
In_Immense_N_S_Passage: darkroom
    sdesc = "In Immense N/S Passage"
    ldesc = {
        I(); "You are at one end of an immense north/south
        passage. ";
    }
    south = In_Giant_Room
    giant = In_Giant_Room
    passage = In_Giant_Room

    enter = { return self.north; }
    cavern = { return self.north; }
    north = RustyDoor

    // Enhancements to the NPC movement code now allow them to use the door
    // so no NPCexit properties are needed.

    // Exit info. for 'back' command:
    exithints = [In_Cavern_With_Waterfall, &north]
;

// The odd implementation of the rusty door is for code testing purposes - we
// want to check that the locations of floatingdecoration objects will be
// correctly evaluated in NPC movement code.  So the 'real' door is a 
// single object with two locations, like the grate.

// The second 'door' in the group handles travel to the giant room
// from the waterfall cavern.  

class RustyDoor_Class: floatingdecoration, CCR_doorway
    isoiled = nil
    noAutoOpen = (not self.isoiled) // don't open automatically unless oiled

    doorlist = [RustyDoor, RustyDoor_2]
    destination = {
        if (not self.isoiled and not self.isopen) {
            "(Opening << self.thedesc >>)\n
            The door is extremely rusty and refuses to open. ";
            return nil;
        }
        else
            pass destination;
    }
    sdesc = "rusty door"
    ldesc = "It's just a big iron door."
    isopen = nil

    noun = 'door' 'hinge' 'hinges'
    adjective = 'massive' 'rusty' 'iron'

    verDoOpen(actor) = {
        if (self.isopen)
            "It's already open.";
        else if (not self.isoiled)
            "The hinges are quite thoroughly rusted now
            and won't budge.";
    }
    verDoClose(actor) = {
        if (not self.isopen)
            "No problem there -- it's already closed.";
        else if (not self.isoiled)
            "The hinges are quite thoroughly rusted now
            and won't budge.";
    }
;

RustyDoor: floatingdecoration, RustyDoor_Class
    loclist = [In_Immense_N_S_Passage In_Cavern_With_Waterfall]

    // Note that we don't mention the door in the waterfall room, so players
    // probably won't notice it in there!
    heredesc = {
        if (not (self.location = In_Immense_N_S_Passage)) return;
        P(); I();
        if (not self.isoiled and not self.isopen)
            "The way north is barred by a massive, rusty,
            iron door.";
        else
            "The way north leads through a massive,
            rusty, iron door.";
    }
    doordest = {
        if(self.location = In_Immense_N_S_Passage)
             return In_Cavern_With_Waterfall;
        else
             return In_Immense_N_S_Passage;
    }
    doordestOK = true // OK to use doordest in NPC travel
;

RustyDoor_2: RustyDoor_Class // Special 'door' to handle travel from the
                             // waterfall cavern to the Giant room.

    location = nil  // avoid unwanted disambiguation messages
    doordest = In_Giant_Room
;


/* 95 */
In_Cavern_With_Waterfall: darkroom
    sdesc = "In Cavern With Waterfall"
    ldesc = {
        I(); "You are in a magnificent cavern with a rushing
        stream, which cascades over a sparkling waterfall
        into a roaring whirlpool which disappears through a
        hole in the floor.  Passages exit to the south and
        west.";
    }
    south = RustyDoor
    out = RustyDoor
    giant = RustyDoor_2
    west = At_Steep_Incline_Above_Large_Room
    // The response to down depends on the game version - in the
    // 551-point version we go to Sword_Point, in the 550 and 701-point
    // versions we go to N_Of_Reservoir minus our inventory (except the lamp
    // and any items which are worn).
    down = {
        if (global.oldgame and not global.game550) pass down;
        else if (global.newgame and not global.game701) {
            dragged_down.message;
            return Sword_Point;
        }
        else if (global.game550) {
            "What, into the whirlpool?";
            Waterfall.rhetoricalturn := global.turnsofar;
            return nil;
        }
    }
    hole = {return self.down;}
/* Note: dwarves are excluded from the sword point area, and the north of the reservoir.*/
;
Waterfall: fixeditem
    rhetoricalturn = -999  // see yesVerb in CCR-VERB.T
    sdesc = "waterfall"
    ldesc = {
        if(global.oldgame and not global.game550)
            "Wouldn't want to go down it in a barrel!";
        else
            /* Changed to hint that the player might try going down. */
            "It's a roaring whirlpool.  Only the most foolhardy
            adventurers would try going down it.";
    }
    noun = 'waterfall' 'whirlpool'
    adjective = 'sparkling' 'whirling'
    location = In_Cavern_With_Waterfall
    verDoEnter(actor) = {
        if(global.oldgame and not global.game550) "You've got to be kidding!";
    }
    doEnter(actor) = {
        actor.travelTo(In_Cavern_With_Waterfall,&down);
    }
    verDoBoard(actor) = { return self.verDoEnter(actor); }
    doBoard(actor) = { return self.doEnter(actor); }

    // BJS: added these verbs:
    verDoRide(actor) = { return self.verDoEnter(actor); }
    doRide(actor) = { return self.doEnter(actor); }

    // BJS: routine for 550-point and 701-point games.
    // DJP: changed to keep items which are worn - and to remark about the
    // crown when it's still there.
    enter() = { 
        local obj, ripped := nil, wornkept := 0, crownkept := nil;
        local actor := getActor(&travelActor);

        if (brass_lantern.isIn(actor)) "You plunge into the water and
        are sucked down by the whirlpool.  ";
        else "You plunge into the water and are sucked down by
                the whirlpool into pitch darkness.  ";
        for(obj:=firstobj(item); obj<>nil; obj:=nextobj(obj,
        item)) {
            if (obj.isfixed) continue; // DJP - exclude fixed items
            if(obj.location = actor and obj<>brass_lantern and not 
            obj.isworn) {
                obj.moveInto(nil); // Remove any carried objects
                if (obj = mushrooms or obj = mushroom) // Regrow mushrooms
                    notify(obj,&regrow,obj.growtime);
                ripped := true;    // except for the lamp.
            }
            else if (obj.location = actor and obj.isworn) {
                wornkept++;
                if (obj=crown) crownkept := true;
            }
        }
        if (ripped and itemcnt(actor.contents) > 0) {
            if (brass_lantern.isIn(actor)) {
                "The current is incredibly
                strong, and you barely manage to hold
                on to your lamp;  everything else ";
                if (wornkept) "(except what you are wearing) ";
                "is pulled from your grasp and is lost
                in the swirling waters. ";
            }
            else {
                "The current is incredibly strong, and
                everything that you are carrying ";
                if (wornkept) "(except what you are wearing) ";
                "is ripped from your grasp and is lost in
                the swirling waters. ";
            }
        }
        P();
        "The swirling waters deposit you, not ungently, on solid
         ground. ";
        if (crownkept) "Instinctively, you reach up to
            check that the crown is still there.  Miraculously, it is! ";
        P();
        if (global.game701) actor.travelTo(Sword_Point);
        else  actor.travelTo(N_Of_Reservoir);
    }
;

/* 96 */
In_Soft_Room: darkroom
    sdesc = "In Soft Room"
    ldesc = {
        I(); "You are in the soft room.  The walls are
        covered with heavy curtains, the floor with a thick
        pile carpet.  Moss covers the ceiling.";
    }
    west = In_Swiss_Cheese_Room
    out = In_Swiss_Cheese_Room
    softfloor = true // we can safely drop the vase here
    hasfloor = true // described as 'floor' not 'ground'
    hasfloordesc = true // custom floor description
    floordesc = "The carpet is quite plush."
    // extra nouns and adjectives for the floor in this room
    floorvocab = ['carpet', 'pile']
    floorvocaba = ['thick', 'plush', 'pile']
;

Curtains: fixeditem
    sdesc = "curtains"
    ldesc = "They seem to absorb sound very well."
    noun = 'curtain' 'curtains'
    adjective = 'heavy' 'thick'
    location = In_Soft_Room

    verifyRemove(actor) = { "Now don't go ripping up the place!"; }
    verDoLookbehind(actor) = {}
    // This will have to be different for the 440-point and 660-point
    // versions!
    doLookbehind(actor) = {
        "You don't find anything exciting behind the curtains (in this version
        of Adventure, anyway). ";
    }
;
Moss: fixeditem
    sdesc = "moss"
    ldesc = "It just looks like your typical, everyday moss."
    noun = 'moss'
    adjective = 'typical' 'everyday'
    location = In_Soft_Room

    verifyRemove(actor) = { "It's too high up for you to reach."; }
    verDoEat(actor) = { "Eeeewwwww."; }
;

/* 97 */
In_Oriental_Room: darkroom
    sdesc = "In Oriental Room"
    ldesc = {
        I(); "This is the oriental room.  Ancient oriental
        cave drawings cover the walls.  A gently sloping
        passage leads upward to the north, another passage
        leads se, and a hands and knees crawl leads ";
        if(global.newgame) "east."; else "west.";
    }
    se = In_Swiss_Cheese_Room
    crawl = In_Large_Low_Room
/* changed direction for 551-point version */
    east = {
        if (global.newgame) return In_Large_Low_Room;
        else return self.noexit;
    }
    west = {
        if (global.oldgame) return In_Large_Low_Room;
        else return self.noexit;
    }
    up = In_Misty_Cavern
    north = In_Misty_Cavern
    cavern = In_Misty_Cavern
;
CaveDrawings: fixeditem
    sdesc = "ancient oriental drawings"
    ldesc = "They seem to depict people and animals."
    noun = 'paintings' 'drawings' 'art'
    adjective = 'cave' 'ancient' 'oriental'
    location = In_Oriental_Room
;

/* 98 */
In_Misty_Cavern: darkroom
    sdesc = "In Misty Cavern"
    ldesc = {
        I(); "You are following a wide path around the outer
        edge of a large cavern. Far below, through a heavy
        white mist, strange splashing noises can be heard.
        The mist rises up through a fissure in the ceiling. ";
        if(global.oldgame) {
            "The path exits to the south and west. ";
        }
        else {
            "The path hugs the cavern's rim to the NE and south, while
            another branch forks west.  A round chute with extremely
            smooth walls angles sharply up to the southwest. ";
        }
    }
    south = In_Oriental_Room
    oriental = In_Oriental_Room
    west = In_Alcove
    /* 'Up' works only 3% of the time in the 551-point game.  In the 701-point
    version we disable it altogether, because the adventurer should not
    be able to bring any objects into the area (except items which are worn).
    */
    up = {
       if (global.oldgame) return self.noexit;
       if (not global.game701 and rand(100) <= 3) return Top_Of_Slide;
       else if (rand(100) <= 75)
           "You managed to climb about halfway up before losing
           %your% hold and sliding back.";
           else "You were only a few yards from the top when you slipped
           and tumbled all the way back down.";
           return nil;
    }
    sw = { return self.up; }
    ne = Dantes_Rest

    // This verb has been taken out, because it looks like a game-testing
    // short cut.

    // chimney = {
    //    if(global.newgame) return Sword_Point;
    //    else pass chimney;
    // }

    // Exit info. for 'back' command:
    exithints = [Top_Of_Slide, &up]
    myhints = [Slidehint]
    listendesc = {
        if (global.oldgame) pass listendesc;
        global.listenadd := true;
        inherited.listendesc;
        P(); "You hear strange splashing noises from the cavern far
             below you. ";
        global.listenadd := nil;
    }
;
CeilingFissure: fixeditem
    sdesc = "fissure"
    ldesc = "You can't really get close enough to examine it."
    noun = 'fissure'
    location = In_Misty_Cavern
;

Chute: fixeditem, upclimbable
    newgame = true
    sdesc = "chute"
    ldesc = "Its walls are very smooth and steep.  If you tried to climb
        it, you'd likely slide back down again! "
    noun = 'chute'
    adjective = 'round' 'steep'
    location = In_Misty_Cavern
    verDoClimb(actor) = {}
    doClimb(actor) = {return actor.location.up;}
;

/* 99 */
In_Alcove: darkroom
    sdesc = "In Alcove"
    ldesc = {
        I(); "You are in an alcove.  A small northwest path seems
        to widen after a short distance.  An extremely tight
        tunnel leads east.  It looks like a very tight
        squeeze.  An eerie light can be seen at the other
        end.";
    }
    nw = In_Misty_Cavern
    cavern = In_Misty_Cavern
    passage = { return self.east; }
    east = {
        local actor := getActor(&travelActor);

        //
        // The player must be carrying only the emerald or
        // nothing at all to fit through the tight tunnel.
        //
        if (length(actor.contents) > 1)
            return wontfit.message;
        else if (length(actor.contents) = 1) {
            if (egg_sized_emerald.location = actor)
                return In_Plover_Room;
            else
                return wontfit.message;
        }
        else
            return In_Plover_Room;
    }

    //
    // Let NPC's go in the plover room regardless of what
    // they're carrying.  (Life's not fair in the Colossal Cave.)
    //
    NPCexit1 = In_Plover_Room
    // Exit info. for 'back' command:
    exithints = [In_Plover_Room, &east]
;

/* 100 */
In_Plover_Room: room
    sdesc = "In Plover Room"
    ldesc = {
        I(); "%You're% in a small chamber lit by an eerie green
        light.  An extremely narrow tunnel exits to the west.
        A dark corridor leads northeast.";
    }

    passage = { return self.west; }
    out =  { return self.west; }
    west = {
        local actor := getActor(&travelActor);

        //
        // The player must be carrying only the emerald or
        // nothing at all to fit through the tight tunnel.
        //
        if (length(actor.contents) > 1)
            return wontfit.message;
        else if (length(actor.contents) = 1) {
            if (egg_sized_emerald.location = actor)
                return In_Alcove;
            else
                return wontfit.message;
        }
        else
            return In_Alcove;
    }

    ne = In_Dark_Room
    dark = In_Dark_Room

    // DJP - corrected to drop emerald, and use the isInside function
    // so it will be dropped even if it's in a closed container.
    plover = {
        local actor := getActor(&travelActor);
        if (egg_sized_emerald.isInside(actor))
            egg_sized_emerald.moveInto(In_Plover_Room);
        return At_Y2;
    }

    //
    // Let NPC's leave the plover room regardless of what
    // they're carrying.  (Life's not fair in the Colossal Cave.)
    //
    NPCexit1 = In_Alcove
    // Exit info. for 'back' command:
    exithints = [In_Alcove, &west]
    myhints = [Ploverhint]
;

/* 101 */
In_Dark_Room: darkroom
    sdesc = "In Dark Room"
    ldesc = {
        I(); "%You're% in the dark-room.  A corridor leading
        south is the only exit."; P();

        I(); StoneTablet.ldesc;
    }
    south = In_Plover_Room
    plover = In_Plover_Room
    out = In_Plover_Room
;
StoneTablet: fixeditem
    sdesc = "stone tablet"
    ldesc = {
        "A massive stone tablet imbedded in the wall reads:
        \"Congratulations on bringing light into the
        dark-room!\"";
    }
    location = In_Dark_Room
    noun = 'tablet'
    adjective = 'massive' 'stone'
;

/* 102 */
In_Arched_Hall: darkroom
    sdesc = "In Arched Hall"
    ldesc = {I();
    if(global.game550) {
        "You are in an arched hall.  A coral passage continues up and east.
        The air smells of sea water. ";
        if (self.jericho)
            "The north wall has partially crumbled, exposing a
            connecting hole to another room. ";
    }
    else if(self.jericho)
        "You are in an arched hall.  The remnants of a now-plugged coral
        passage lie to the east.  The north wall has partially crumbled,
        exposing a connecting hole to another room. ";
    else
        "You are in an arched hall.  A coral passage
        once continued up and east from here, but is now
        blocked by debris.  The air smells of sea water. ";
    }

/* This flag will be set to true when the horn is blown */
    jericho = nil

    east = {if (global.game550) return Coral_Passage;
        "The coral passage is blocked - in this version, anyway. ";
        return nil;
    }
    down = In_Shell_Room
    shell = In_Shell_Room
    out = In_Shell_Room
    up = { return self.east; }
    north = {
        local actor := getActor(&travelActor);

        if (not self.jericho) pass north;
        if (giant_bivalve.isIn(actor)) {
            if (giant_bivalve.opened)
                "You can't fit this five-foot oyster
                through the hole. ";
            else
                "You can't fit this five-foot clam
                through the hole. ";

            return nil;
        }
        else
            return EW_Corridor_E;
    }
    hole = {
        if (not self.jericho) pass hole;
        else return self.north;
    }
// Let npc's through, after the horn has been blown.  Note that
// NPC's are allowed to go from EW_Corridor_E to here even before
// the horn has been blown; this is to prevent them from getting
// trapped.
    NPCexit1 = {if(self.jericho)return EW_Corridor_E;
               else return nil;}
    NPCexit2 = Coral_Passage
    // Exit info. for 'back' command:
    exithints = [EW_Corridor_E, &north,
                Coral_Passage, &east]
    listendesc = {
        if (self.jericho or global.oldgame)
            pass listendesc;
        else
            "You pace around the room, listening carefully to your
            footsteps.  You have a strong feeling that the wall to your
            north is hollow. ";
    }
;

In_Arched_Hall_Walls: floatingdecoration
    sdesc = "walls"
    adesc = "walls"
    ldesc = {
        if(not global.newgame) 
            "I've already told all I know about the walls. ";
        else if(In_Arched_Hall.jericho)
            "A hole has now appeared in the <<self.holedir>> wall. ";
        else
            "You can't see anything unusual about the walls. ";
    }
    noun = 'walls' 'wall'
    adjective = 'north' 'south' 'n' 's' 'northern' 'southern' 'hollow'
    loclist = [In_Arched_Hall EW_Corridor_E]
    holedir = {
        local actor := getActor(&travelActor);
        if (actor.isIn(In_Arched_Hall)) return 'north'; 
        else return 'south';
    }
    verDoKnock(actor) = {}
    doKnock(actor) = {"The <<self.holedir>> wall sounds hollow. ";}
    verDoAttack(actor) = {}
    doAttack(actor) = {
        if(In_Arched_Hall.jericho)
            "You've already done enough damage! ";
        else
            "The <<self.holedir>> wall sounds hollow, but holds firm against 
            your onslaught. ";
    }       
    verDoAttackWith(actor,io) = {}
    doAttackWith(actor,io) = {self.doAttack(actor);}
    verDoBreak(actor) = {}
    doBreak(actor) = {self.doAttack(actor);}
;

/* 103 */
In_Shell_Room: darkroom
    sdesc = "In Shell Room"
    ldesc = {
        I(); "%You're% in a large room carved out of
        sedimentary rock.  The floor and walls are littered
        with bits of shells imbedded in the stone.  A shallow
        passage proceeds downward, and a somewhat steeper one
        leads up.  A low hands and knees passage enters from
        the south.";
    }
    up = In_Arched_Hall
    hall = In_Arched_Hall
    down = In_Ragged_Corridor

    south = {
        local actor := getActor(&travelActor);

        if (giant_bivalve.isIn(actor)) {
            if (giant_bivalve.opened)
                "You can't fit this five-foot oyster
                through that little passage!";
            else
                "You can't fit this five-foot clam
                through that little passage!";

            return nil;
        }
        else
            return At_Complex_Junction;
    }

    //
    // Let NPC's through.
    //
    NPCexit1 = At_Complex_Junction
    // Exit info. for 'back' command:
    exithints = [At_Complex_Junction, &south]
;

/* 104 */
In_Ragged_Corridor: darkroom
    sdesc = "In Ragged Corridor"
    ldesc = {
        I(); "You are in a long sloping corridor with ragged
        sharp walls.";
    }
    up = In_Shell_Room
    shell = In_Shell_Room
    down = In_A_Cul_De_Sac
;

/* 105 */
In_A_Cul_De_Sac: darkroom
    sdesc = "In a Cul-de-Sac"
    ldesc = {
        I(); "You are in a cul-de-sac about eight feet
        across.";
    }
    up = In_Ragged_Corridor
    out = In_Ragged_Corridor
    shell = In_Shell_Room
;

/* 106 */
In_Anteroom: darkroom
    sdesc = "In Anteroom"
    ldesc = {
        I(); "You are in an anteroom leading to a large
        passage to the east.  Small passages go west and up.
        The remnants of recent digging are evident."; P();

        I(); "A sign in midair here says \"Cave under
        construction beyond this point. Proceed at own risk.
        [Witt Construction Company]\"";
    }
    up = At_Complex_Junction
    west = In_Bedquilt
    east = At_Witts_End
;
WittSign: fixeditem, readable
    sdesc = "hanging sign"
    ldesc = "It's hanging way above %your% head."
    readdesc = {
        "It says \"Cave under construction beyond this point.
        Proceed at own risk.  [Witt Construction Company]\"";
    }
    noun = 'sign'
    adjective = 'hanging'
    location = In_Anteroom

    verifyRemove(actor) = { "No chance.  It's too far up."; }
;

/*
 * This maze is off limits to NPC's again, in keeping with the 551-point
 * version.  (And because it's hard for NPC's to get out of this maze once
 * they've entered it.)
 */
/* 107 */
Different_Maze_1: darkroom, NoNPC
    sdesc = "Maze of Twisty Little Passages, All Different"
    ldesc = {
        I(); "You are in a maze of twisty little passages,
        all different.";
    }
    south = Different_Maze_3
    sw = Different_Maze_4
    ne = Different_Maze_5
    se = Different_Maze_6
    up = Different_Maze_7
    nw = Different_Maze_8
    east = Different_Maze_9
    west = Different_Maze_10
    north = Different_Maze_11
    down = At_West_End_Of_Long_Hall
;

/* 108 */
At_Witts_End: darkroom
    sdesc = "At Witt's End"
    ldesc = {
        I(); "You are at Witt's End.  Passages lead off in
        *all* directions.";
    }
    east = {
        local wittpct := 95;
        local actor := getActor(&travelActor);
        if (clover.isInside(actor)) wittpct := 80;
        if (rand(100) <= wittpct)
            return crawled_around.message;
        else {
            if (global.game580) return S_Of_Center;
            else return In_Anteroom;
        }
    }
    west = {
        "%You% %have% crawled around in some little holes and
        found %your% way blocked by a recent cave-in.  You are
        now back in the main passage.";

        return nil;
    }

    north = { return self.east; }
    south = { return self.east; }
    ne = { return self.east; }
    se = { return self.east; }
    sw = { return self.east; }
    nw = { return self.east; }
    up = { return self.east; }
    down = { return self.east; }

    //
    // Let NPC's out of here with no trouble; they always go back 
    // to the anteroom, since they have no way out of the computer 
    // center.
    //
    NPCexit1 = In_Anteroom
    // but if we try to go back, we're blocked by the cave-in.
    // Exit info. for 'back' command:
    exithints = [In_Anteroom, &west]
    myhints = [Witthint]
;

/* 109 */
In_Mirror_Canyon: darkroom
    sdesc = "In Mirror Canyon"
    ldesc = {
        I(); "You are in a north/south canyon about 25 feet
        across.  The floor is covered by white mist seeping
        in from the north.  The walls extend upward for well
        over 100 feet.  Suspended from some unseen point far
        above you, an enormous two-sided mirror is hanging
        parallel to and midway between the canyon walls. ";

        P(); I();
        "The mirror is obviously provided for the use of the
        dwarves, who as you know, are extremely vain.";

        P(); I(); "A small window can be seen in either wall,
        some fifty feet up.";
    }
    south = In_Secret_N_S_Canyon_0
    north = At_Reservoir
    reservoir = At_Reservoir
;
Mirror_1: fixeditem, distantItem
    sdesc = "enormous mirror"
    ldesc = "It looks like an ordinary, albeit enormous, mirror."
    location = In_Mirror_Canyon
    noun = 'mirror'
    adjective = 'enormous' 'huge' 'big' 'large' 'suspended'
        'hanging' 'vanity' 'dwarvish'

    verDoBreak(actor) = { self.verifyRemove(actor); }
    verifyRemove(actor) = { "You can't reach it from here."; }

    // N.B. SEARCH and LOOK IN are not equivalent for mirrors but are for
    // many other objects e.g. containers, crystal ball.
    verDoLookin(actor) = {}
    doLookin(actor) = {
        "All you can see is a reflection of the canyon walls.  The
        mirror is too high up to show your reflection. ";
    }
;
// DJP - eliminated duplicate object CanyonMirror and added a window object

CanyonWindows: distantItem
    sdesc = "windows"
    ldesc = "A window can be seen in both the east and the west walls
        of the canyon, about 50 feet up. "
    noun = 'window' 'windows'
    location = In_Mirror_Canyon
;

/* 110 */
At_Window_On_Pit_2: darkroom
    sdesc = "At Window on Pit"
    ldesc = {
        I(); "%You're% at a low window overlooking a huge pit,
        which extends up out of sight.  A floor is
        indistinctly visible over 50 feet below.  Traces of
        white mist cover the floor of the pit, becoming
        thicker to the left. Marks in the dust around the
        window would seem to indicate that someone has been
        here recently.  Directly across the pit from ";"%you% and
        25 feet away there is a similar window looking into a
        lighted room.  A shadowy figure can be seen there
        peering back at ";"%you%.";
    }
    west = At_Junction_Of_Three_Secret_Canyons
    jump = { return broken_neck.death; }
    reflect = At_Window_On_Pit_1
;

/* 111 */
Atop_Stalactite: darkroom
    sdesc = "Atop Stalactite"
    ldesc = {
        I(); "A large stalactite extends from the roof and
        almost reaches the floor below.  You could climb down
        it, and jump from it to the floor, but having done so
        you would be unable to reach it to climb back up.";
    }
    north = In_Secret_N_S_Canyon_1

    jump = { return self.down; }
    climb = { return self.down; }
    down = {
        if (rand(100) <= 40)
            return Alike_Maze_6;
        else if (rand(100) <= 50)
            return Alike_Maze_9;
        else
            return Alike_Maze_4;
    }

    //
    // Let NPC's through to the maze
    //
    NPCexit1 = Alike_Maze_6
    NPCexit2 = Alike_Maze_9
    NPCexit3 = Alike_Maze_4
;
Stalactite: fixeditem
    sdesc = "stalactite"
    ldesc = {
        "You could probably climb down it, but you can forget
        coming back up.";
    }
    noun = 'stalactite' 'stalagmite' 'stalagtite'
    adjective = 'large'
    location = Atop_Stalactite

    verDoLookunder(actor) = { "Do get a grip on yourself."; }
    verDoMove(actor) = { self.verDoLookunder(actor); }
    verifyRemove(actor) = { self.verDoLookunder(actor); }
;

/* 122 */
Different_Maze_2: darkroom, NoNPC
    sdesc = "Little Maze of Twisting Passages, All Different"
    ldesc = {
        I(); "You are in a little maze of twisting passages,
        all different.";
    }
    sw = Different_Maze_3
    north = Different_Maze_4
    east = Different_Maze_5
    nw = Different_Maze_6
    se = Different_Maze_7
    ne = Different_Maze_8
    west = Different_Maze_9
    down = Different_Maze_10
    up = Different_Maze_11
    south = Dead_End_14
;

/* 113 */
At_Reservoir: darkroom
    sdesc = { if (global.game550) "South Edge of Reservoir";
                else "At Reservoir"; }
    ldesc = {
        I(); // This one changes a lot between the three versions.
        if (global.game550)
            "You are at the southern edge of a
            large underground reservoir.  A thick cloud of white mist
            fills the room and rises rapidly upward.  The lake is fed
            by a stream, which tumbles out of a hole in the wall about
            10 feet overhead and splashes noisily into the water near
            the reservoir's northern wall.  A dimly-seen passage exits
            through the northern wall, but you can't get across the
            water to get to it.  Another passage leads south from here.";
        else {
            "You are at the edge of a large underground
            reservoir.  An ";
            if (global.newgame)
                "almost "; // DJP - added for 551-point version
            "opaque cloud of white mist fills the
            room and rises rapidly upward.  The lake is fed by a
            stream, which tumbles out of a hole in the wall about
            10 feet overhead and splashes noisily into the water
            somewhere within the mist.  ";
            if (global.newgame)
                "The indistinct shape of the
                opposite shore can be dimly seen through the mist. ";
            "The only passage goes back toward the south. ";
        }
    }
    south = In_Mirror_Canyon
    out = In_Mirror_Canyon
    north = {
        local actor := getActor(&travelActor);
        Reservoir.doCross(actor);
        return nil;
    }
    cross = (self.north) // BJS: Added
    across = (self.north)

    //
    // The original had another exit, but the verb it was attached
    // to didn't exist.  I have no idea what was intended here...
    //
    //  BJS: Most likely it was supposed to be the (non-existent) verb
    // for the mirror canyon, since the verb number is the same as the
    // canyon's room number. -BJS
    //
    // ??? = In_Mirror_Canyon
    listendesc = "You hear the sound of the stream, noisily splashing
    into the water. "
;

Reservoir: Streamitem, floatingdecoration
    sdesc = "reservoir"
    ldesc = {
        "It's a large reservoir, from which a cloud of mist rises. ";
        if(global.newgame or global.game550)
            "The opposite shore can be seen indistinctly through
            the mist.  However, you can't swim and there is no
            way across the reservoir from here. ";
        else
            "There is no way across the reservoir. ";
    }
    verDoCross(actor) = {}
    doCross(actor) = { if(global.game550) {
        if(Turtle.location = self.location) {
            "You step gently on Darwin the Tortoise's back,
            and he carries you smoothly over to the southern
            side of the reservoir.  He then blows a couple of
            bubbles at you and sinks back out of sight.";
            Turtle.moveInto(nil); P();
            actor.travelTo(At_Reservoir);
        }
        else
            "I can't swim, or walk on water.  You'll have to find
            some other way across, or get someone to assist you.";
        }
        else if(global.newgame)
            "I can't swim, or walk on water.  You'll
            have to find some other way across.";
        else
            "I can't swim, or walk on water. There is no way across.";
    }
    noun = 'reservoir' 'lake' 'water' 'stream'
    adjective = 'deep'
    loclist = [At_Reservoir Sword_Point N_Of_Reservoir]
;
//
// Here's where the pirate(s) keeps his treasure (as well as any loot
// he's swiped from the player).  Once the chest has been been found
// here, turn off the pirate(s) completely.  (This is how the original
// handled it, and it's thankfully merciful so I've kept it the same.)
//
// DJP - corrected a bug in enterRoom so that the pirate really is turned
// off as intended.
//
/* 114 */
Dead_End_13: darkroom, CCR_dead_end_room, NoNPC
    se = Alike_Maze_13
    out = Alike_Maze_13     // DMB: added

    ldesc = "This is the pirate's dead end."

    enterRoom(actor) = { // DJP - isIn(Me) changed to isIn(self)
        if (treasure_chest.isIn(self) and not treasure_chest.spotted
        and self.islit) {
            P(); I(); "You've found the pirate's treasure chest!";
            P();
            Pirates.loclist := [];
            unnotify(Pirates, &move);
            treasure_chest.spotted := true;
            PirateMessage.moveInto(nil);
        }

        pass enterRoom;
    }
;

/* 115,116 are at the end of this file */
/* 117 */
On_Sw_Side_Of_Chasm: darkroom
    sdesc = "On SW Side of Chasm"
    ldesc = {
        I(); "You are on one side of a large, deep chasm.  A
        heavy white mist rising up from below obscures all
        view of the far side.  A southwest path leads away
        from the chasm into a winding corridor. ";

        RicketyBridge.xdesc;
    }
    sw = In_Sloping_Corridor

    across = { return self.over; }
    cross = { return self.over; }
    ne = { return self.over; }
    over = {
        local actor := getActor(&travelActor);
        RicketyBridge.doCross(actor);
        return nil;
    }

    jump = {
        if (RicketyBridge.exists) {
            "I respectfully suggest you go across the
            bridge instead of jumping.";

            return nil;
        }
        else
            return didnt_make_it.death;
    }

    //
    // No NPC exits because we don't want the pirate to be able
    // to go across the bridge and steal the player's return toll.
    // (All rooms on the other side of the bridge are off limits
    // to NPC's.)
    //
    // It would be OK for dwarves to show up over there, except
    // that they might run into the bear, a situation for which we don't
    // have any code.  (This is how the original was as well.)
    //
    // Exit info. for 'back' command:
    exithints = [On_Ne_Side_Of_Chasm, &over]
;

TrollChasm: rfd
    iswavetarget = true // magic can be worked by waving the rod at it ...
    loclist = [On_Sw_Side_Of_Chasm, On_Ne_Side_Of_Chasm]
    sdesc = "chasm"
    noun = 'chasm'
    adjective = 'deep' 'large'
    verDoJump(actor) = {}
    doJump(actor) = {
        actor.travelTo(actor.location,&jump);
    }

;
RicketyBridge: floatingdecoration
    iswavetarget = true // magic can be worked by waving the rod at it ...
    exists = true

    sdesc = "rickety bridge"
    ldesc = "It just looks like an ordinary, but unstable, bridge."

    xdesc = {
        if (self.exists) {
            "A rickety wooden bridge extends across the
            chasm, vanishing into the mist."; P();

            I(); "A sign posted on the bridge reads,
            \"Stop! Pay troll!\"";
        }
        else {  if(self.isburnt) "The charred remains of a wooden
            bridge can be seen at the bottom of the chasm.";
            else "The wreckage of a bridge (and a dead bear)
            can be seen at the bottom of the chasm.";
        }
    }

    noun = 'bridge'
    adjective = 'rickety' 'unstable' 'wobbly' 'rope'

    locationOK = true       // tell compiler OK for location to be method
    loclist = [On_Sw_Side_Of_Chasm, On_Ne_Side_Of_Chasm ]

    verDoCross(actor) = {}
    doCross(actor) = {
        if (actor = Bear) {
            // DJP - added code for bear as actor
            "That bridge looks very rickety.  The bear refuses
            to cross it ahead of you.";
            return nil;
        }
        if (self.exists) {
            if (Troll.ispaid = toplocation(actor)
                or Troll.location = nil) {
                self.cross(actor);
            }
            else {
                if (Troll.isIn(self.location)) {
                    "The troll refuses to let you
                    cross.";
                }
                else {
                    "The troll steps out from
                    beneath the bridge and blocks
                    %your% way.";

                    Troll.moveInto(self.location);
                }
            }
        }
        else
            "There is no longer any way across the chasm.";
    }

    cross(actor) = {
        if (Bear.isfollowing) {
            // DJP - check whether the upgraded rod has
            // been used here.
            if(not self.strengthened) {
                "Just as you reach the other side, the bridge
                buckles beneath the weight of the bear, which
                was still following you around. You scrabble
                desperately for support, but as the bridge
                collapses you stumble back and fall into the
                chasm.";
            }
            else {
                "The bridge is no longer rickety!  In fact it
                feels very secure as you cross, and easily
                supports the weight of the bear, which was still
                following you around.   Unfortunately the spell
                seems to wear off at the instant you reach the
                other side.   With a loud \"Crack!\" the bridge
                collapses under the weight of the bear, which
                scrambles frantically
                to reach your side but fails to make it.";
            }

            // Get rid of the bridge in case the
            // player gets reincarnated and
            // continues the game.
            self.exists := nil;
            self.moveLoclist([]);
            On_Sw_Side_Of_Chasm.contents -= self;
            On_Ne_Side_Of_Chasm.contents -= self;
            // No more bear!
            Bear.exists := nil;
            DeadBear.moveLoclist([On_Sw_Side_Of_Chasm,
            On_Ne_Side_Of_Chasm]);
            /* DJP: The player survives if the bridge has been
            strengthened. */
            if(self.strengthened) {
                self.strengthened := nil;
                P();
                actor.travelTo(On_Sw_Side_Of_Chasm);
            }
            else {
                /* DJP: in accordance with the original game, move
                the adventurer to the other side to make his
                possessions accessible after reincarnation. */
                actor.moveInto(On_Sw_Side_Of_Chasm);
                die();
            }
        }
        else if (Troll.isduped and global.game550
        and Troll.ispaid = nil) {
            "As you reach the middle of the bridge, the troll
        appears from out of the tunnel behind you, wearing
            a large backpack.  \"So, Mister Magician,\" he
            shouts, \"you like to use magic to steal back my
            hard-earned toll?  Let's see how you like a little
            of MY magic!!\"  With that, he aims a tube running
            from the backpack directly at the bridge and pulls
            a trigger.  A spout of magical fire roars out and
            incinerates the bridge supports, causing the bridge
            to sway giddily and collapse into the chasm.  You
            plunge down to your death.";
            self.exists := nil;
            self.isburnt := true;
            self.moveLoclist([]);
            BurntBridge.moveLoclist([On_Sw_Side_Of_Chasm,
            On_Ne_Side_Of_Chasm]);
            /* DJP: in accordance with the original game, move
               the adventurer to the other side to make his
               possessions accessible after reincarnation. */
            actor.moveInto(On_Sw_Side_Of_Chasm);
            die();
        }
        else if (actor.isIn(On_Sw_Side_Of_Chasm)) {
            Troll.ispaid := nil;
            /* DJP - issue message if upgraded rod has been used */
            if(self.strengthened) {
                "As you cross, you get a vague sense that something
                isn't quite right.  You realize what it is -
                although the bridge still \(looks\) very rickety,
                it doesn't actually \(feel\) rickety! ";
                P();
                self.strengthened := nil;
                self.prevstrength := true;
            }
            else if(self.prevstrength) {
                "You notice that the bridge is now as rickety
                as it looks.  You deduce that the magic wears off
                after one crossing. "; P();
                self.prevstrength := nil;
            }
            actor.travelTo(On_Ne_Side_Of_Chasm);
        }
        else {
            Troll.ispaid := nil;
            /* DJP - issue message if upgraded rod has been used */
            if(self.strengthened) {
                "As you cross, you get a vague sense that something
                isn't quite right.  You realize what it is -
                although the bridge still \(looks\) very rickety,
                it doesn't actually \(feel\) rickety! ";
                P();
                self.strengthened := nil;
                self.prevstrength := true;
            }
            else if(self.prevstrength) {
                "You notice that the bridge is now as rickety
                as it looks.  You deduce that the magic wears off
                after one crossing. "; P();
                self.prevstrength := nil;
            }
            actor.travelTo(On_Sw_Side_Of_Chasm);
        }
    }
;

BurntBridge: floatingdecoration,distantItem
    sdesc = "bridge wreckage"
    ldesc = "The charred remains of the bridge lie at the bottom of the
            chasm.  They are too far away to examine closely."
    noun = 'remains' 'bridge' 'wreckage'
    adjective = 'burnt' 'wrecked' 'destroyed' 'charred' 'burned' 'wooden'
    loclist = []
;

/* Special treasure room for the troll. An ldesc is defined because we might
   be able to view it after the sapphire has been used as a toll.  It's
   otherwise off limits to players.

   The original a-code implementation of the 550-point version has similar
   rooms.  LIMBO is a repository for objects which are unavailable to the 
   player, including tolls.  YLEM is a 'circular file' for objects which have 
   been irretrievably lost or destroyed.  Note that the TADS implementation 
   does not have an YLEM and allows certain objects to reappear after they 
   have been lost.  For example, the axe may reappear, the lamp will reappear 
   after reincarnation (if not destroyed), and the eggs can always be sent to 
   the Giant room. 

 */

Troll_Treasure: room, NoNPC
    sdesc = "Troll's treasure room"
    ldesc = {
        I(); "You are in a large chamber, lit by a strange orange
        glow.  It appears to be the room where the troll stores his
        treasures!  Fixed to the walls
        are numerous shelves, stacked with gold, silver and platinum
        articles, musical instruments, vases, clocks and ornaments
        of all descriptions.
        Part of the room is stacked to the ceiling with treasure
        chests, all full to overflowing with precious stones.
        Another area is full of antique furniture which the troll has
        collected over the centuries - some of the items must be
        worth an absolute fortune!
        In the center of the room a table is piled high with
        valuable coins, and an open ledger records the tolls paid
        by adventurers over the years.  ";
        if((Troll.location = nil) and not (global.closed)) { 
        //troll chased away by the bear
        P(); I();
            "You see the troll here, counting money and cataloging
            his treasures.  You hear him muttering something about
            \"that pesky bear\".";
        }
    }
;

/* 118 */
In_Sloping_Corridor: darkroom
    sdesc = "In Sloping Corridor"
    ldesc = {
        I(); "You are in a long winding corridor sloping out
        of sight in both directions.";
    }
    down = In_Large_Low_Room
    up = On_Sw_Side_Of_Chasm
;

/* 119 to 121 */

/* (This was implemented as 3 locations in the original version, but now
 * there is just one location.  While the dragon is present, the reachability
 * of objects is checked in a similar way to the glass bridge in The Mulldoon
 * Legacy.  Items are reachable only if they were dropped on the same side
 * as the dragon.)
 */
#define EAST_EXIT 0
#define NORTH_EXIT 1

In_Secret_Canyon: darkroom

    // allow taking of objects only when taken at the correct end of the
    // room.
    special_cantreach(obj,actor,chain) = {
        local topcont, actor_side, obj_side;
        // check that the dragon still exists, and that the object being
        // tested isn't the room itself. 
        if (Dragon.location = self and obj <> self) {
            topcont := chain[2]; /* top-level container of obj */
            if(topcont.bothsides)return nil;
            actor_side := EAST_EXIT; 
            if(actor.prevloc = In_Secret_N_S_Canyon_0)
                actor_side := NORTH_EXIT;
            obj_side := EAST_EXIT; 
            if(topcont.meprevloc = In_Secret_N_S_Canyon_0)
                obj_side := NORTH_EXIT;
            if(actor_side <> obj_side)return true;
        }
        return nil;
    }
    cantReach(actor) = // DJP - modified to call cantReachRoom for objects
                       // which are in the room.
    {
        if (actor.location = nil)
        {
            /*
             *   The actor isn't in a room.
             */
            "I don't see that here.";
        }
        else
        {
            /*
             *   The actor is inside the room, or something within the
             *   room, but from which the room's contents are not
             *   reachable.  Tell the actor's location to generate an 
             *   appropriate message.
             */
            actor.location.cantReachRoom(self);
        }
    }
    cantReachRoom(otherRoom) = {
        if(toplocation(otherRoom) = self) 
        {
            "That's on the far side of the dragon.  You can't reach it from 
            here. ";
        }
        else pass cantReachRoom;
    }
    sdesc = "In Secret Canyon"
    ldesc = {
        I(); "You are in a secret canyon which exits to the
        north and east.";
    }
    north = {
        local actor := getActor(&travelActor);
        if (actor.prevloc = In_Secret_N_S_Canyon_0) {
            return actor.prevloc;
        }
        else {
            if (self.dragoncheck)
                return In_Secret_N_S_Canyon_0;
            else
                return nil;
        }
    }
    east = {
        local actor := getActor(&travelActor);
        if (actor.prevloc <> In_Secret_N_S_Canyon_0 ) {
            return In_Secret_E_W_Canyon;
        }
        else {
            if (self.dragoncheck)
                return In_Secret_E_W_Canyon;
            else
                return nil;
        }
    }

    forwards = {
        local actor := getActor(&travelActor);
        if (actor.prevloc = In_Secret_N_S_Canyon_0)
            return self.east;
        else
            return self.north;
    }
    out = {
        local actor := getActor(&travelActor);
        if (actor.prevloc = In_Secret_N_S_Canyon_0)
            return self.north;
        else
            return self.east;
    }

    dragoncheck = {
        if (Dragon.isIn(self)) {
            "The dragon looks rather nasty.  You'd best
            not try to get by.";

            return nil;
        }
        else
            return true;
    }

    //
    // Let NPC's by the dragon without incident (they're all on
    // the same team, after all).
    //
    NPCexit1 = In_Secret_N_S_Canyon_0
    NPCexit2 = In_Secret_E_W_Canyon
    // Exit info. for 'back' command:
    exithints = [
        In_Secret_N_S_Canyon_0, &north,
        In_Secret_E_W_Canyon, &east
    ]
;
/* 122 */
On_Ne_Side_Of_Chasm: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "On NE Side of Chasm"
    ldesc = {
        I(); "You are on the far side of the chasm.  A
        northeast path leads away from the chasm on this
        side. ";

        RicketyBridge.xdesc;
    }
    ne = In_Corridor

    across = { return self.over; }
    cross = { return self.over; }
    sw = { return self.over; }
    over = {
        local actor := getActor(&travelActor);
        RicketyBridge.doCross(actor);
        return nil;
    }

    jump = {
        if (RicketyBridge.exists) {
            "I respectfully suggest you go across the
            bridge instead of jumping.";

            return nil;
        }
        else
            return didnt_make_it.death;
    }

    fork = At_Fork_In_Path
    view = At_Breath_Taking_View
    barren = In_Front_Of_Barren_Room
    // Exit info. for 'back' command:
    exithints = [On_Sw_Side_Of_Chasm, &over]
;

/* 123 */
In_Corridor: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Corridor"
    ldesc = {
        I(); "%You're% in a long east/west corridor.  A faint
        rumbling noise can be heard in the distance.";
    }
    west = On_Ne_Side_Of_Chasm
    east = At_Fork_In_Path
    fork = At_Fork_In_Path
    view = At_Breath_Taking_View
    barren = In_Front_Of_Barren_Room
    listendesc = "You hear a faint rumbling noise in the distance. "
;

/* 124 */
At_Fork_In_Path: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At Fork in Path"
    ldesc = {
        I(); "The path forks here.  The left fork leads
        northeast.  A dull rumbling seems to get louder in
        that direction.  The right fork leads southeast down
        a gentle slope.  The main corridor enters from the
        west.";
    }
    west = In_Corridor
    ne = At_Junction_With_Warm_Walls
    left = At_Junction_With_Warm_Walls
    se = In_Limestone_Passage
    right = In_Limestone_Passage
    down = In_Limestone_Passage
    view = At_Breath_Taking_View
    barren = In_Front_Of_Barren_Room
    listendesc = "You hear a faint rumbling noise in the distance. "
;

/* 125 */
At_Junction_With_Warm_Walls: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At Junction With Warm Walls"
    ldesc = {
        I(); "The walls are quite warm here.  From the north
        can be heard a steady roar, so loud that the entire
        cave seems to be trembling.  Another passage leads
        south, and a low crawl goes east.";
    }
    south = At_Fork_In_Path
    fork = At_Fork_In_Path
    north = At_Breath_Taking_View
    view = At_Breath_Taking_View
    east = In_Chamber_Of_Boulders
    crawl = In_Chamber_Of_Boulders
    listendesc = "You hear a steady roar from the north. "
;

/* 126 */
At_Breath_Taking_View: room, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "At Breath-Taking View"
    ldesc = {
        I(); "%You% %are% on the edge of a breath-taking view. Far
        below %you% is an active volcano, from which great
        gouts of molten lava come surging out, cascading back
        down into the depths.  The glowing rock fills the
        farthest reaches of the cavern with a blood-red
        glare, giving everything an eerie, macabre
        appearance. The air is filled with flickering sparks
        of ash and a heavy smell of brimstone.  The walls are
        hot to the touch, and the thundering of the volcano
        drowns out all other sounds.  Embedded in the jagged
        roof far overhead are myriad twisted formations
        composed of pure white alabaster, which scatter the
        murky light into sinister apparitions upon the walls.
        To one side is a deep gorge, filled with a bizarre
        chaos of tortured rock which seems to have been
        crafted by the devil himself.  An immense river of
        fire crashes out from the depths of the volcano,
        burns its way through the gorge, and plummets into a
        bottomless pit far off to ";"%your% left.  ";
        P();
        if (global.game550) {
            "Across the gorge, the entrance to
            a valley is dimly visible.  ";
        }
        "To the right, an immense geyser of blistering steam erupts
        continuously from a barren island in the center of a
        sulfurous lake, which bubbles ominously.  The far
        right wall is aflame with an incandescence of its
        own, which lends an additional infernal splendor to
        the already hellish scene.  A dark, forboding passage
        exits to the south.";
    }
    south = At_Junction_With_Warm_Walls
    passage = At_Junction_With_Warm_Walls
    out = At_Junction_With_Warm_Walls
    fork = At_Fork_In_Path
    // DJP - changed this to pass on the actor.  We must make sure that
    // the bear doesn't go where he shouldn't!
    valley = {
        local actor := getActor(&travelActor);
        if (global.game550)
            WheatStoneBridge.doCross(actor);
        else "There is no valley here. Perhaps in another version...";
            return nil;
    }
    cross = { 
        local actor := getActor(&travelActor);
        WheatStoneBridge.doCross(actor); 
        return nil; 
    }
    gorge = { return self.cross; }
    north = { return self.cross; }
    jump = { return self.down; }
    down = {
        "Don't be ridiculous!";
        return nil;
    }
    listendesc = "The sound of the volcano is almost deafening. "
    exithints = [ Valley_Faces, &north ]
;
Volcano: floatingdecoration, distantItem
    iswavetarget = true // magic can be worked by waving the rod at it ...
                        // (in the 550-point game)
    sdesc = "active volcano"
    ldesc = {
        "Great gouts of molten lava come surging out of the
        volvano and go cascading back down into the depths.
        The glowing rock fills the farthest reaches of the
        cavern with a blood-red glare, giving everything an
        eerie, macabre appearance.";
    }
    loclist = [At_Breath_Taking_View, Valley_Faces]
    noun = 'volcano' 'rock'
    adjective = 'active' 'glowing' 'blood' 'blood-red' 'red' 'eerie'
        'macabre'
;
Sparks: decoration
    sdesc = "sparks of ash"
    adesc = { self.sdesc; }
    ldesc = {
        "The sparks are too far away for you to get a good look at
        them.";
    }
    location = At_Breath_Taking_View
    noun = 'spark' 'sparks' 'ash' 'air'
    adjective = 'flickering'
;
JaggedRoof: decoration
    sdesc = "jagged roof"
    ldesc = {
        "Embedded in the jagged roof far overhead are myriad
        twisted formations composed of pure white alabaster,
        which scatter the murky light into sinister
        apparitions upon the walls.";
    }
    location = At_Breath_Taking_View
    noun = 'roof' 'formations' 'light' 'apparitions'
    adjective = 'jagged' 'twisted' 'murky' 'sinister'
;
DeepGorge: floatingdecoration, distantItem
    iswavetarget = true // magic can be worked by waving the rod at it ...
    sdesc = "deep gorge"
    ldesc = {
        local actor := getActor(&verbActor);

        "The gorge is filled with a bizarre chaos of tortured
        rock which seems to have been crafted by the devil
        himself. ";
        if (global.game550 and actor.isIn(At_Breath_Taking_View))
        "Across the gorge, the entrance to a valley is dimly
        visible.";
    }
    loclist = [At_Breath_Taking_View, Valley_Faces]
    noun = 'gorge' 'chaos' 'rock'
    adjective = 'deep' 'bizarre' 'tortured'
    verDoCross(actor) = {}
    doCross(actor) = {WheatStoneBridge.doCross(actor);}

;
Gorgevalley: decoration, distantItem
    iswavetarget = true // magic can be worked by waving the rod at it ...
    sdesc = "valley"
    game550 = true
    ldesc = "It's dimly visible across the gorge. "
    location = At_Breath_Taking_View
    noun = 'valley'
;

RiverOfFire: floatingdecoration, distantItem
    iswavetarget = true // magic can be worked by waving the rod at it ...
    sdesc = "river of fire"
    ldesc = {
        local actor := getActor(&verbActor);

        "The river of fire crashes out from the depths of the
        volcano, burns its way through the gorge, and
        plummets into a bottomless pit far off to ";
        "%your% "; if (actor.isIn(At_Breath_Taking_View))"left.";
        else "right.";
    }
    loclist = [At_Breath_Taking_View, Valley_Faces]
    noun = 'river' 'fire' 'depth' 'pit'
    adjective = 'fire' 'firey' 'bottomless'
;
Geyser: floatingdecoration, distantItem
    iswavetarget = true // magic can be worked by waving the rod at it ...
    sdesc = "immense geyser"
    ldesc = {
        "The geyser of blistering steam erupts continuously
        from a barren island in the center of a sulfurous
        lake, which bubbles ominously.";
    }
    loclist = [At_Breath_Taking_View, Valley_Faces]
    noun = 'geyser' 'steam' 'island' 'lake'
    adjective = 'immense' 'blistering' 'barren' 'sulfrous'
        'sulfurous' 'sulphurous' 'sulphrous' 'bubbling'
;

/* 127 */
In_Chamber_Of_Boulders: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Chamber of Boulders"
    ldesc = {
        I(); "You are in a small chamber filled with large
        boulders.  The walls are very warm, causing the air
        in the room to be almost stifling from the heat.  The
        only exit is a crawl heading west, through which is
        coming a low rumbling.";
    }
    west = At_Junction_With_Warm_Walls
    out = At_Junction_With_Warm_Walls
    crawl = At_Junction_With_Warm_Walls
    fork = At_Fork_In_Path
    view = At_Breath_Taking_View
    listendesc = "You hear a low rumbling sound. ";
ChamberBoulders: fixeditem
    sdesc = "boulders"
    ldesc = "They're just ordinary boulders.  They're warm."
    noun = 'boulder' 'boulders' 'rocks' 'stones'
    location = In_Chamber_Of_Boulders
;

/* 128 */
In_Limestone_Passage: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Limestone Passage"
    ldesc = {
        I(); "You are walking along a gently sloping
        north/south passage lined with oddly shaped limestone
        formations.";
    }
    north = At_Fork_In_Path
    up = At_Fork_In_Path
    fork = At_Fork_In_Path
    south = In_Front_Of_Barren_Room
    down = In_Front_Of_Barren_Room
    barren = In_Front_Of_Barren_Room
    view = At_Breath_Taking_View
;
LimestoneFormations: decoration
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "limestone formations"
    ldesc = {
        "Every now and then a particularly strange shape
        catches %your% eye.";
    }
    location = In_Limestone_Passage
    noun = 'formations' 'shape' 'shapes'
    adjective = 'lime' 'limestone' 'stone' 'oddly' 'shaped'
        'oddly-shaped'
;


/* 129 */
In_Front_Of_Barren_Room: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Front of Barren Room"
    ldesc = {
        I(); "You are standing at the entrance to a large,
        barren room.  A sign posted above the entrance reads:
        \"Caution!  Bear in room!\"";
    }
    west = In_Limestone_Passage
    up = In_Limestone_Passage
    fork = At_Fork_In_Path
    east = In_Barren_Room
    in = In_Barren_Room
    barren = In_Barren_Room
    enter = In_Barren_Room
    view = At_Breath_Taking_View
;
BarrenSign: fixeditem, readable
    sdesc = "barren room sign"
    ldesc = {
        "The sign reads, \"Caution!  Bear in room!\"";
    }

    noun = 'sign'
    adjective = 'barren' 'room'

    location = In_Front_Of_Barren_Room
;

/* 130 */
In_Barren_Room: darkroom, NoNPC
    wino_trollstop = true // troll stops a wino from getting to cloak_pits
    sdesc = "In Barren Room"
    ldesc = {
        I(); "You are inside a barren room.  The center of
        the room is completely empty except for some dust.
        Marks in the dust lead away toward the far end of the
        room.  The only exit is the way you came in.";
    }
    west = In_Front_Of_Barren_Room
    out = In_Front_Of_Barren_Room
    fork = At_Fork_In_Path
    view = At_Breath_Taking_View
;
Dust: fixeditem
    sdesc = "dust"
    ldesc = { "It just looks like ordinary dust."; }
    location = In_Barren_Room
    noun = 'dust' 'marks'
;

/* 131 */
Different_Maze_3: darkroom, NoNPC
    sdesc = "Maze of Twisting Little Passage, All Different"
    ldesc = {
        I(); "You are in a maze of twisting little passages,
        all different.";
    }
    west = Different_Maze_1
    se = Different_Maze_4
    nw = Different_Maze_5
    sw = Different_Maze_6
    ne = Different_Maze_7
    up = Different_Maze_8
    down = Different_Maze_9
    north = Different_Maze_10
    south = Different_Maze_11
    east = Different_Maze_2
;

/* 132 */
Different_Maze_4: darkroom, NoNPC
    sdesc = "Little Maze of Twisty Passages, All Different"
    ldesc = {
        I(); "You are in a little maze of twisty passages,
        all different.";
    }
    nw = Different_Maze_1
    up = Different_Maze_3
    north = Different_Maze_5
    south = Different_Maze_6
    west = Different_Maze_7
    sw = Different_Maze_8
    ne = Different_Maze_9
    east = Different_Maze_10
    down = Different_Maze_11
    se = Different_Maze_2
;

/* 133 */
Different_Maze_5: darkroom, NoNPC
    sdesc = "Twisting Maze of Little Passages, All Different"
    ldesc = {
        I(); "You are in a twisting maze of little passages,
        all different.";
    }
    up = Different_Maze_1
    down = Different_Maze_3
    west = Different_Maze_4
    ne = Different_Maze_6
    sw = Different_Maze_7
    east = Different_Maze_8
    north = Different_Maze_9
    nw = Different_Maze_10
    se = Different_Maze_11
    south = Different_Maze_2
;

/* 134 */
Different_Maze_6: darkroom, NoNPC
    sdesc = "Twisting Little Maze of Passages, All Different"
    ldesc = {
        I(); "You are in a twisting little maze of passages,
        all different.";
    }
    ne = Different_Maze_1
    north = Different_Maze_3
    nw = Different_Maze_4
    se = Different_Maze_5
    east = Different_Maze_7
    down = Different_Maze_8
    south = Different_Maze_9
    up = Different_Maze_10
    west = Different_Maze_11
    sw = Different_Maze_2
;

/* 135 */
Different_Maze_7: darkroom, NoNPC
    sdesc = "Twisty Little Maze of Passages, All Different"
    ldesc = {
        I(); "You are in a twisty little maze of passages,
        all different.";
    }
    north = Different_Maze_1
    se = Different_Maze_3
    down = Different_Maze_4
    south = Different_Maze_5
    east = Different_Maze_6
    west = Different_Maze_8
    sw = Different_Maze_9
    ne = Different_Maze_10
    nw = Different_Maze_11
    up = Different_Maze_2
;

/* 136 */
Different_Maze_8: darkroom, NoNPC
    sdesc = "Twisty Maze of Little Passages, All Different"
    ldesc = {
        I(); "You are in a twisty maze of little passages,
        all different.";
    }
    east = Different_Maze_1
    west = Different_Maze_3
    up = Different_Maze_4
    sw = Different_Maze_5
    down = Different_Maze_6
    south = Different_Maze_7
    nw = Different_Maze_9
    se = Different_Maze_10
    ne = Different_Maze_11
    north = Different_Maze_2
;

/* 137 */
Different_Maze_9: darkroom, NoNPC
    sdesc = "Little Twisty Maze of Passages, All Different"
    ldesc = {
        I(); "You are in a little twisty maze of passages,
        all different.";
    }
    se = Different_Maze_1
    ne = Different_Maze_3
    south = Different_Maze_4
    down = Different_Maze_5
    up = Different_Maze_6
    nw = Different_Maze_7
    north = Different_Maze_8
    sw = Different_Maze_10
    east = Different_Maze_11
    west = Different_Maze_2
;

/* 138 */
Different_Maze_10: darkroom, NoNPC
    sdesc = "Maze of Little Twisting Passages, All Different"
    ldesc = {
        I(); "You are in a maze of little twisting passages,
        all different.";
    }
    down = Different_Maze_1
    east = Different_Maze_3
    ne = Different_Maze_4
    up = Different_Maze_5
    west = Different_Maze_6
    north = Different_Maze_7
    south = Different_Maze_8
    se = Different_Maze_9
    sw = Different_Maze_11
    nw = Different_Maze_2
;

/* 139 */
Different_Maze_11: darkroom, NoNPC
    sdesc = "Maze of Little Twisty Passages, All Different"
    ldesc = {
        I(); "You are in a maze of little twisty passages,
        all different.";
    }
    sw = Different_Maze_1
    nw = Different_Maze_3
    east = Different_Maze_4
    west = Different_Maze_5
    north = Different_Maze_6
    down = Different_Maze_7
    se = Different_Maze_8
    up = Different_Maze_9
    south = Different_Maze_10
    ne = Different_Maze_2
;

/*
 * We don't allow NPC's here because it would be *really* bogus
 * if the pirate stole the batteries right after the player bought
 * them.
 */
/* 140 */
Dead_End_14: darkroom, CCR_dead_end_room, NoNPC
    sdesc = "At a Dead End, in Front of a Massive Vending Machine"
    ldesc = {
        I(); "%You% %have% reached a dead end. There is a massive
        vending machine here.";

        if (PirateMessage.isIn(self)) {
            P();
            I(); "Hmmm...  There is a message here
            scrawled in the dust in a flowery script.";
        }
    }

    north = Different_Maze_2
    out = Different_Maze_2
;
VendingMachine: fixeditem, readable
    sdesc = "vending machine"
    ldesc = {
        "The instructions on the vending machine read,
        \"Insert coins to receive fresh batteries.\"";
    }
    location = Dead_End_14
    noun = 'machine' 'slot'
    adjective = 'vending' 'massive' // Some adjectives removed
                                    // for TADS 2.4.0

    verIoPutIn(actor) = {}
    ioPutIn(actor, dobj) = {
        local newbatt;
        if (isclass(dobj,coinitem)) {
            "Soon after %you% insert%s% ";
            if (dobj = bag) "the pieces of eight";
            else dobj.thedesc;
            " in the coin slot, the vending machine makes a
            grinding sound, and a set of fresh batteries falls at
            %your% feet. ";
            if (dobj = bag) {
                "You discard the empty bag, which disappears
                out of sight through a crevice in the floor. ";
            }
            dobj.moveInto(nil);
            newbatt := new fresh_batteries;
            if(isclass(dobj,CCR_treasure_item))
                global.vendingtreasures++;
        }
        else {
            "The machine seems to be designed to take
            coins.";
        }
    }

    verDoKick(actor) = {}
    doKick(actor) = {
        "<WHUMP!> You boot the machine, but to no avail.";
    }
    verDoBreak(actor) = {}
    doBreak(actor) = {
        "The machine is quite sturdy and survives %your% attack
        without getting so much as a scratch.";
    }
    verDoAttack(actor) = { self.verDoBreak(actor); }
    doAttack(actor) = { self.doBreak(actor); }
    verDoAttackWith(actor, io) = { self.verDoAttack(actor); }
    doAttackWith(actor) = { self.doAttack(actor); }

    verDoLookbehind(actor) = {
        "You don't find anything behind the vending machine.";
    }
    verDoLookunder(actor) = {
        "You don't find anything under the vending machine.";
    }
    verDoMove(actor) = { "The vending machine is far too heavy to move."; }
;
PirateMessage: fixeditem, readable
    sdesc = "message in the dust"
    ldesc = {
        "The message reads, \"This is not the maze where the
        pirate leaves his treasure chest.\"";
    }
    noun = 'message' 'scrawl' 'writing' 'script'
    adjective = 'scrawled' 'flowery'
    location = nil  // moved to Dead_End_14 when pirate spotted
    doCleanWith( actor, io ) =
    {
        if(io = whiskbroom) {
            "You sweep away the message. ";
            self.moveInto(nil);
        }
        else pass doCleanWith;
    }
    doSweepWith( actor, io ) =
    {
        if(io = whiskbroom) self.doCleanWith(actor,io);
        else pass doSweepWith;
    }
;

/*
 * Miscellaneous messages
 */
broken_neck: object
    death = {
        "You are at the bottom of the pit with a broken neck.";
        die();
        return nil;
    }
;
didnt_make_it: object
    death = {
        "You didn't make it.";
        die();
        return nil;
    }
;
crawled_around: object
    message = {
        "%You% %have% crawled around in some little holes and
        wound up back in the main passage.";

        return nil;
    }
;
wontfit: object
    message = {
        "Something %you're% carrying won't fit through the
        tunnel with you. You'd best take inventory and drop
        something.";

        return nil;
    }
;

/*
 * Room feature decorations.
 * These don't give any new information, but they make the program
 * seem less brain-damaged.
 */
TheRoom: fixeditem, makecontents
    sdesc = "room"
    ldesc = {
        local actor := getActor(&verbActor);
        // Upon "examine room" we just give the standard
        // description for the current location.
        actor.location.lookAround(true);
    }
    noun = 'room' 'anteroom' 'dark-room' 'darkroom'
    adjective = 'low' 'twopit' 'two-pit' 'large' 'lighted'
        'giant' 'soft' 'oriental' 'dark' 'immense' 'barren'
    locationOK = true        // tell compiler OK for location to be method
    location = {
        local actor := getActor(&currentActor);
        return actor.location;     // always where player is
    }
    doCleanWith( actor, io ) =
    {
        if(io = whiskbroom) {
            if(toplocation(actor) = In_Dusty_Rock_Room and not
            DustyRocks.areswept) DustyRocks.sweep;
            else "Enough dusting, already!  %You're% making me sneeze.";
        }
        else pass doCleanWith;
    }
    doSweepWith( actor, io ) =
    {
        if(io = whiskbroom) {
            if(toplocation(actor) = In_Dusty_Rock_Room and not
            DustyRocks.areswept) DustyRocks.sweep;
            else "Enough dusting, already!  %You're% making me sneeze.";
        }
        else pass doSweepWith;
    }
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&in);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
;
Hands: fixeditem   // the player's hands
    sdesc = "your hands"
    adesc = { self.sdesc; }
    thedesc = { self.sdesc; }
    // This ldesc is stolen from Delusions - DJP
    ldesc = "Two of 'em.  Five fingers each. They look pretty normal to
        me! "
    noun = 'hands'
    adjective = 'my' 'bare' 'hands'

    locationOK = true        // tell compiler OK for location to be method
    location = {
        local actor := getActor(&currentActor);
        return actor.location;     // always where player is
    }
/* DJP - 'take hands' and 'drop hands' get more sensible replies */
    verDoTake ( actor ) = {
        "%You% already have %your% hands!";
    }
    verDoDrop ( actor ) = {
        "They can't be dropped (in the normal sense, anyway). ";
    }
/* DJP - also 'wave hands' */
    verDoWave ( actor ) = {}
    doWave(actor) = {
        /* clear global.noAskWave (see thing.doWave) */
        global.noAskWave := nil;
        if (self.isIn(Window.location)) {
            "The shadowy figure waves back at you. ";
        }
        else pass doWave;
    }
    doCount(actor) = {
        "You have two hands.  How many did you expect? ";
    }
;

Heels: fixeditem, makecontents   // DJP - the player's heels
    sdesc = "your heels"
    adesc = { self.sdesc; }
    thedesc = { self.sdesc; }

    ldesc = "They look pretty normal to me."
    noun = 'heels'
    adjective = 'my'

    locationOK = true        // tell compiler OK for location to be method
    location = {
        local actor := getActor(&currentActor);
        return actor.location;     // always where player is
    }
    verDoTake ( actor ) = {
        "%You% already have %your% heels! ";
    }
    verDoDrop ( actor ) = {
        "I'm not sure what you mean. ";
    }
    verDoClick (actor ) = {}
    doClick ( actor ) = {clickVerb.action(actor);}
    doCount(actor) = {
        "%You% %have% two heels.  How many did you expect? ";
    }
;



Footsteps: fixeditem, makecontents // DJP - exists in all game versions
                      // but normally noticed only in the Arched hall in
                      // the 551-point version.
    sdesc = "your footsteps"
    adesc = { self.sdesc; }
    thedesc = { self.sdesc; }

    ldesc = {
        local actor := getActor(&verbActor);

        if (global.oldgame or not (actor.isIn(In_Arched_Hall) or
        (actor.isIn(EW_Corridor_E))) or In_Arched_Hall.jericho) {
            "They sound pretty normal to me.";
        }
        else {
            "They reverberate hollowly around the chamber.  You have the
            feeling that the <<In_Arched_Hall_Walls.holedir>> wall is 
            maybe not as solid as it looks. ";
        }
    }
    noun = 'footsteps'
    adjective = 'my'

    locationOK = true        // tell compiler OK for location to be method
    location = {
        local actor := getActor(&currentActor);
        return actor.location;     // always where player is
    }
    heredesc = {
        local actor := getActor(&verbActor);
        if (global.oldgame) return;
        if (not (actor.isIn(In_Arched_Hall) or
        (actor.isIn(EW_Corridor_E))) or In_Arched_Hall.jericho) return;
        P(); I(); "Your footsteps echo hollowly throughout the chamber.  ";
    }
    verDoTouch(actor) = {"Don't be ridiculous!";}
    verDoRub(actor) = {self.verDoTouch(actor);}
    verDoPush(actor) = {self.verDoTouch(actor);}
    // reference to 'takeable' sounds in SoFar.z8
    verDoTake(actor) = {"I believe you have another game in mind. ";}
    verifyRemove(actor) = {self.verDoTouch(actor);}
    listendesc = {self.ldesc;}
    verDoCount(actor) = {"That's not something you can count. ";}
;

class rfd: floatingdecoration
    ldesc = "You know as much as I do at this point."
;
Crawl: rfd
    sdesc = "crawl"
    noun = 'crawl' 'crawls'
    adjective = 'cobble' 'low' 'wide' 'higher' 'dead' 'end' 'tight' 'narrow'
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
    loclist = [
        Below_The_Grate  In_Cobble_Crawl  In_Debris_Room
        In_Dirty_Passage  On_Brink_Of_Pit
        At_West_End_Of_Hall_Of_Mists  At_East_End_Of_Long_Hall
        At_Complex_Junction  In_Large_Low_Room  Dead_End_Crawl
        In_Tall_E_W_Canyon  In_Oriental_Room
        At_Junction_With_Warm_Walls  In_Chamber_Of_Boulders
        Nondescript   By_Pentagram   Crack_3  Crack_4
    ]
;

Chamber: rfd
    sdesc = "chamber"
    noun = 'chamber'
    adjective = 'small' 'splendid' 'side' 'large'
                'low' 'circular' 'icy' 'sandstone'
                'golden' 'nondescript' 'non-descript' 'treasure'
                'bare' 'warm' 'south' 'west' 'southern' 'western' 's' 'w'
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
    loclist = [
        Below_The_Grate  In_Bird_Chamber  In_South_Side_Chamber
        In_West_Side_Chamber  In_Slab_Room  In_Plover_Room
        Warm_Room  Balcony  In_Chamber_Of_Boulders
        Nondescript  Golden  Sandstone_Chamber
        Crack_4  Ice_Cave_Exit
    ]
;
Passage: rfd
    sdesc = "passage"
    noun = 'passage' 'opening' 'openings' 'corridor' 'corridors'
            'path' 'paths' 'tunnel'
    adjective = 'low' 'wide' 'plugged' 'good' 'east' 'eastern' 'e' 
            'small' 'twisty' 'little' 'n/s' 'e/w' 'dirty' 'broken' 'long'
            'large' 'walking' 'sizeable' 'sizable' 'cavernous'
            'blocked' 'immense' 'gently' 'sloping' 'coral' 'dimly-seen'
            'dimly' 'seen' 'rumbling' 'rough' 'winding' 'narrow'
            'shallow' 'somewhat' 'steeper' 'dark' 'forboding' 'tight'
            'fourier' // 'cairn' removed for TADS 2.4.0
                      // 'high' removed to avoid disambiguation problems
                      // when trying to enter holes at the east end of the
                      // Long Hall
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&passage);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
    loclist = [
        In_Cobble_Crawl In_Debris_Room
        In_Awkward_Sloping_E_W_Canyon In_Bird_Chamber
        At_Top_Of_Small_Pit In_Hall_Of_Mists
        On_East_Bank_Of_Fissure In_Nugget_Of_Gold_Room
        In_Hall_Of_Mt_King At_West_End_Of_Twopit_Room
        In_East_Pit In_West_Pit West_Side_Of_Fissure
        Low_N_S_Passage In_South_Side_Chamber
        In_West_Side_Chamber At_Y2 Jumble_Of_Rock
        At_Window_On_Pit_1 In_Dirty_Passage On_Brink_Of_Pit
        In_Pit In_Dusty_Rock_Room
        At_West_End_Of_Hall_Of_Mists Alike_Maze_1
        Alike_Maze_2 Alike_Maze_3 Alike_Maze_4 Dead_End_1
        Dead_End_2 Dead_End_3 Alike_Maze_5 Alike_Maze_6
        Alike_Maze_7 Alike_Maze_8 Alike_Maze_9 Dead_End_4
        Alike_Maze_10 Dead_End_5 At_Brink_Of_Pit Dead_End_6
        At_East_End_Of_Long_Hall At_West_End_Of_Long_Hall
        Crossover Dead_End_7 At_Complex_Junction In_Bedquilt
        In_Swiss_Cheese_Room At_East_End_Of_Twopit_Room
        In_Slab_Room In_Secret_N_S_Canyon_0
        In_Secret_N_S_Canyon_1
        At_Junction_Of_Three_Secret_Canyons In_Large_Low_Room
        Dead_End_Crawl
        In_Secret_E_W_Canyon In_N_S_Canyon
        Canyon_Dead_End In_Tall_E_W_Canyon Dead_End_8
        Alike_Maze_11 Dead_End_9 Dead_End_10 Alike_Maze_12
        Alike_Maze_13 Dead_End_11 Dead_End_12 Alike_Maze_14
        In_Narrow_Corridor At_Steep_Incline_Above_Large_Room
        In_Giant_Room At_Recent_Cave_In
        In_Immense_N_S_Passage In_Cavern_With_Waterfall
        In_Soft_Room In_Oriental_Room In_Misty_Cavern
        In_Alcove In_Plover_Room In_Dark_Room In_Arched_Hall
        In_Shell_Room In_Ragged_Corridor In_A_Cul_De_Sac
        In_Anteroom Different_Maze_1 At_Witts_End
        In_Mirror_Canyon At_Window_On_Pit_2 Atop_Stalactite
        Different_Maze_2 At_Reservoir N_Of_Reservoir Dead_End_13
        At_Ne_End At_Sw_End On_Sw_Side_Of_Chasm In_Sloping_Corridor
        In_Secret_Canyon  On_Ne_Side_Of_Chasm In_Corridor
        At_Fork_In_Path At_Junction_With_Warm_Walls South_Fog
        At_Breath_Taking_View In_Chamber_Of_Boulders Plain_Center
        In_Limestone_Passage In_Front_Of_Barren_Room Nondescript
        South_Basilisk North_Basilisk Basilisk_Fork Fake_Y2
        Winding_Pass Golden Translucent In_Barren_Room
        Different_Maze_3 Different_Maze_4 Different_Maze_5
        Different_Maze_6 Different_Maze_7 Different_Maze_8
        Different_Maze_9 Different_Maze_10 Different_Maze_11
        Dead_End_14 Morion Corr_Divis In_Cubicle Spherical_Room
        Glassy_Room Sorc_Lair Brink_North Brink_East Crack_2
        Crack_3 Ice_Room Coral_Passage Coral_Pass_2 Arch_Fork
        Jonah Fourier Throne_Room_East Top_Of_Steps
        Blue_Grotto_East Bubble_Chamber Tight_Crack Tight_Crack_2
        Spiral_Staircase Spiral_Staircase_Blocked Cloakroom
        Cloak_Pits Tongue_of_Rock Star_Chamber Elbow_in_Passage
        Tunnel_Intersection Narrow_E_W_Passage Rotunda Vestibule
        Fairy_Grotto Yellow_Path Cold_Passage Green_Lake_Room
        Passage_End_at_Hole Thunder_Hole River_Styx_Approach
        River_Styx River_Styx_E Dantes_Rest
    ]
;
Canyon: rfd
    sdesc = "canyon"
    noun = 'canyon' 'canyons'
    adjective = 'awkward' 'sloping' 'secret' 'e/w' 'n/s' 'tight'
                'tall' 'three'
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
    loclist = [
        In_Debris_Room  In_Awkward_Sloping_E_W_Canyon
        In_Bird_Chamber  In_Secret_N_S_Canyon_0
        In_Secret_N_S_Canyon_1  At_Junction_Of_Three_Secret_Canyons
        In_Secret_E_W_Canyon  In_N_S_Canyon
        Canyon_Dead_End  In_Tall_E_W_Canyon  Dead_End_8
        In_Mirror_Canyon  In_Secret_Canyon
    ]
;
Walls: fixeditem, makecontents
    sdesc = "walls"
    ldesc = "I've already told all I know about the walls."
    noun = 'walls'
    // 'Orange' taken out for 701+ point game to avoid a clash with the
    // orange button - also because we have an object for the orange 
    // travertine in the Bird Chamber 
    adjective = 'swiss' 'cheese' 'ragged' 'sharp' 'warm' 'hot' 'wall'
    // giving 'wall' as an adjective is a kludge so that if we define
    // a special wall in a room, 'X WALL' will refer to it without
    // disambiguation.
    locationOK = true
    location = {
        local actor := getActor(&currentActor);
        local toploc := toplocation(actor);
        if(toploc = nil) return nil;
        if(toploc = At_End_Of_Road) return toploc;
        if(((not toploc.isoutside) or toploc.isindoor)
              and toploc <> Foggy_Plain and toploc <> Plain_Center
              and toploc <> In_Safe and toploc <> In_Arched_Hall 
              and toploc <> EW_Corridor_E)
            return toploc;
        else
            return nil;
    }
    verDoClean(actor) = {  // DJP
        if (toplocation(actor) = Bat_Cave) {
            "I suggest that you leave the bats alone.";
        }
        else pass verDoClean;
    }
    verDoSweep(actor) = {self.verDoClean(actor);}
    verDoCleanWith(actor,io) = {self.verDoClean(actor);}
    verDoSweepWith(actor,io) = {self.verDoSweep(actor);}
    verDoWake(actor) = {"Leave the bats alone, please!";}
    verDoRead(actor) = {
        local toploc := toplocation(actor);
        if (toploc <> River_Styx_Approach and
            toploc <> In_Debris_Room and
            toploc <> In_Nugget_Of_Gold_Room and
            toploc <> Sorc_Lair and
            toploc <> Tool_Room and
            (not (toploc = Dead_End_7) and (StatueMessage.isIn(toploc)))
           ) pass verDoRead;
    }
    doRead(actor) = {
        if(toplocation(actor) = River_Styx_Approach) {
            "Translated, it means:";
            P();I();
            "\"Abandon Hope all Ye who Enter Here.\"";
        }
        else if(toplocation(actor) = In_Debris_Room)
            XyzzyNote.readdesc;
        else if(toplocation(actor) = In_Nugget_Of_Gold_Room)
            NuggetNote.readdesc;
        else if(toplocation(actor) = Dead_End_7)
            StatueMessage.ldesc;
        else if(toplocation(actor) = Tool_Room)
            ToolMessage.ldesc;
        else if(toplocation(actor) = Sorc_Lair)
            "The only readable words are, \"noside samoht.\"";

    }
    doCount(actor) = {
        if(actor.isIn(Octagonal_Room))
            "It's an octagonal room.  That means it has 8 walls! ";
        else if(actor.isIn(Inside_Building) or actor.isIn(Pantry))
            "This room has - wait for it - four walls! ";
        else if(actor.isIn(At_End_Of_Road))
            "The outside of the building has - wait for it - four walls! ";
        else if(actor.isIn(Cylindrical))
            "The wall of this room is one continuous surface. ";
        else if(actor.isIn(Spherical_Room))
            "It's a spherical room.  That means that the wall, floor
            and ceiling are all one continuous surface, broken only
            by the exit passage. ";
        else
            "Counting walls isn't easy in a cave, due to the irregular
            shapes of most rooms. ";
    }
;
Ceiling: fixeditem, distantItem, makecontents
    sdesc = "ceiling"
    noun = 'ceiling'
    locationOK = true
    location = {return Walls.location;}
    ldesc = {
        local actor := getActor(&verbActor);
        if (actor.location = In_Slab_Room) {
            "You realize that this room doesn't have a ceiling any more!
            It looks as if you could go up to another room.";
        }
        else if (actor.location = Beach or actor.location = Beach_Shelf)
            "Actually, this room has no ceiling.";
        else rfd.ldesc;
    }
;
DeadEnd: rfd
    sdesc = "dead end"
    noun = 'end'
    adjective = 'dead'
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
    loclist = [
        Dead_End_1 Dead_End_2 Dead_End_3 Dead_End_4
        Dead_End_5 Dead_End_6 Dead_End_7 Dead_End_Crawl
        Canyon_Dead_End Dead_End_8 Dead_End_9 Dead_End_10
        Dead_End_11 Dead_End_12 Dead_End_13 Dead_End_14
        Salt_Dead_End Dead_End_15 Dead_End_Crack Lost_Canyon_E
    ]
;
Hall: rfd
    sdesc = "hall"
    noun = 'hall'
    adjective = 'vast' 'long' 'featureless' 'arched'
                'audience' 'cavernous' 'jonah'
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
    loclist = [
        In_Hall_Of_Mists  On_East_Bank_Of_Fissure
        In_Hall_Of_Mt_King  West_Side_Of_Fissure
        In_West_Side_Chamber  At_West_End_Of_Hall_Of_Mists
        At_East_End_Of_Long_Hall At_West_End_Of_Long_Hall
        In_Arched_Hall  West_Audience  East_Audience
        Jonah  In_Jonah
    ]
;
// DJP: the hole object is now split into four.  Please note that only one
// of downhole, uphole, bothhole and sidehole should be present at a given
// location.

Downhole: rfd, downclimbable // DJP - hole below player
    sdesc = "hole"
    noun = 'hole' 'holes'
    adjective = 'large' 'big' 'round' 'two' 'foot' 'two-foot'
    loclist = [
        Low_N_S_Passage
        In_Dusty_Rock_Room
        In_Narrow_Corridor  In_Cavern_With_Waterfall
        At_High_Hole
        Passage_End_at_Hole Crypt
    ]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&hole);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&hole);
    }
;
Uphole: rfd, upclimbable // DJP - hole above player
    sdesc = "hole"
    noun = 'hole' 'holes'
    adjective = 'large' 'big' 'round' 'two' 'foot' 'two-foot'
    loclist = [
        In_Dirty_Passage At_Complex_Junction Green_Lake_Room
        Top_of_Stalagmite At_Reservoir N_Of_Reservoir
        Sword_Point Grotto_West
    ]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&hole);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'
    verDoClimb(actor) = {}
    doClimb(actor) = {
        actor.travelTo(actor.location,&hole);
    }
;
Bothhole: rfd, upclimbable // DJP - holes all around (e.g. Bedquilt)
    sdesc = "hole"
    noun = 'hole' 'holes'
    adjective = 'large' 'big' 'round' 'two' 'foot' 'two-foot'
    loclist = [
        In_Bedquilt At_East_End_Of_Twopit_Room
    ]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&hole);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'
    /* Allow holes to be climbed up or down only if the location has
    the uphole or downhole method defined as true */
    verDoClimbup(actor) = {
        local toploc := toplocation(actor);
        if(toploc.uphole) return;
        "I can't see a hole you could climb up. ";
    }
    doClimbup(actor) = {
        actor.travelTo(actor.location,&up);
    }
    verDoClimbdown(actor) = {
        local toploc := toplocation(actor);
        if(toploc.downhole) return;
        "I can't see a hole you could climb down. ";
    }
    doClimbdown(actor) = {
        actor.travelTo(actor.location,&down);
    }
;
Sidehole: rfd // DJP - hole at same level as player
    sdesc = "hole"
    noun = 'hole' 'holes'
    adjective = 'large' 'big' 'round' 'two' 'foot' 'two-foot'
    loclist = []
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&hole);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'
;
Cracks: rfd
    sdesc = "cracks"
    noun = 'crack' 'cracks'
    loclist = [
        Jumble_Of_Rock In_Dusty_Rock_Room Brink_East
        Crack_1 Crack_2 Crack_3 Crack_4
    ]
    verDoEnter(actor) = {}
    doEnter(actor) = {
        actor.travelTo(actor.location,&crack);
    }
    verDoBoard(actor) = {self.verDoEnter(actor);}
    doBoard(actor) = {self.doEnter(actor);}
    doSynonym('Enter') = 'Gothrough'
;
Junction: rfd
    sdesc = "junction"
    noun = 'junction'
    adjective = 'complex'
    /* Note the use of a version-dependent loclist, so that the
       misty cavern is seen as a junction only in the 551-point and
       701-point games.
     */

    loclist = [
        Low_N_S_Passage At_Complex_Junction
        At_Junction_Of_Three_Secret_Canyons
        At_Junction_With_Warm_Walls
    ]

    loclist551 = [
        Low_N_S_Passage At_Complex_Junction
        At_Junction_Of_Three_Secret_Canyons
        At_Junction_With_Warm_Walls In_Misty_Cavern
    ]

    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(actor.location,&out);
    }
;
Air: rfd, makecontents // If they MUST examine EVERYTHING
    sdesc = "air"
    adesc = "air"
    noun = 'air' 'environment' 'atmosphere' 'wind'
    adjective = 'damp' 'hot' 'stifling'
    locationOK = true        // tell compiler OK for location to be method
    location = {
        local actor := getActor(&currentActor);
        return actor.location;     // always where player is
    }
    verDoSmell(actor) = {}
    doSmell(actor) = {
        "The air smells pretty much like you would expect.";
    }
;
