package iageserver;

/**
 * Interpreter class - used to run scripted, user-written code.
 */

public class interpreter {
	
	/*** Contains the returnvalue property set by the executing script code */
	public String returnvalue = "";
	/*** Set to true if CancelEvent is called in the code */
	public boolean cancelled = false;
	
	private player thisplayer = null;
	private parsestring thisparser = null;
	private int instruction = 1;
	private int programcounter = 1;
	private String source = "";
	private iagecode thecode = null;
	private iagecollection codeline = null;
	private boolean isget = false;
	private iagecollection localvariables = new iagecollection();
	private iagecollection localvariablevalues = new iagecollection();
	
	
	/*** Creates a new interpreter for the specified player, using the specifed parser */
	public interpreter (player p, parsestring pa) {
		
		thisplayer = p;
		thisparser = pa;
		
	}
	
	/** Executes the iagecode collection passed in. The source is a string to identify the
	 *  code being executed if an error occurs.
	 */
	public void runcode(iagecode tcode, String codesource) {
		
		source = codesource;
		cancelled = false;
		programcounter = 1;
		thecode = tcode;
		parsestring p = null;
		String curcodeline = "";
		
		// Drop out now if there are no lines of code
		if (thecode.getCount() == 0) return;
		
		while (programcounter <= thecode.getCount()) {
			
			// start at the first instruction
			instruction = 1;
			
			// Remove all leading spaces from the code line
			curcodeline = thecode.get(programcounter);
			curcodeline = data.removeleadingspaces(curcodeline);
			
			// Trim the codeline to make sure there are no tabs
			curcodeline = curcodeline.trim();

			// Break up the code line
			p = new parsestring(curcodeline, thisplayer, " ");
			
			// Get our line of code
			codeline = p.vwords;
			
			// Make sure we have some code
			if (codeline.getCount() != 0) {
				
				// Cycle through each instruction, interpreting it
				while (instruction <= codeline.getCount()) {
					try {
						interpret();
					}
					catch(Exception e) {
						e.printStackTrace();
						data.oerror.RaiseError(source, "Interpreter error - " + e.toString(), programcounter, instruction);
						e.printStackTrace(thisplayer.socket_outputstream);
					}
				}
			}
			
			programcounter++;	
		}
	}
	
	/** Executes the iagecode collection passed in. The source is a string to identify the
	 *  code being executed if an error occurs.
	 *  This overloaded version allows you to start from a procedure
	 */
	public void runcode(iagecode tcode, String codesource, 
						String procname, String[] args, int noargs,
						String originalsource, int oldinstruction, int oldprogcount) {
		
		source = codesource;
		cancelled = false;
		programcounter = 1;
		thecode = tcode;
		parsestring p = null;
		String curcodeline = "";
		String firstword = "";
		String secondword = "";
		int i = 1;
		
		// Drop out now if there are no lines of code
		if (thecode.getCount() == 0) return;
	
		// Find the proc 
		while (programcounter <= thecode.getCount()) {
			curcodeline = thecode.get(programcounter);
			p = new parsestring(curcodeline.trim(), thisplayer, " ");
			if (p.vwords.getCount() > 1) {
				firstword = (String) p.vwords.get(1);
				secondword = (String) p.vwords.get(2);
				if (firstword.equalsIgnoreCase("proc") && 
					secondword.equalsIgnoreCase(procname)) {
					
					// We have our proc, we now need to create variables
					// containing the argument values.
					
					// Break up the proc declaration
					p = new parsestring(curcodeline.trim(), thisplayer, " ");
					i = 3;
					while (i <= p.vwords.getCount()) {
						localvariables.add((String)p.vwords.get(i));
						localvariablevalues.add(args[i-2]);
						i++;					
					}		
					
					// Drop out into the code flow		
					break;
				}	
			}
			programcounter++;
		}
		
		// If we didn't find a proc, quit with an error,
		// UNLESS oldprogcount is 0, which means it was spawned from the engine
		// instead of user code so quit with no error.
		if (programcounter > thecode.getCount()) {
			if  (oldprogcount > 0) {
			data.oerror.RaiseError(originalsource, "Procedure " + procname + " does not exist in " + codesource + ".", oldprogcount, oldinstruction);
			}
			return;
		}
		
		programcounter++;

		while (programcounter <= thecode.getCount()) {
			
			// start at the first instruction
			instruction = 1;
			
			// Remove all leading spaces from the code line
			curcodeline = thecode.get(programcounter);
			curcodeline = data.removeleadingspaces(curcodeline);
			
			// Trim the codeline to make sure there are no tabs
			curcodeline = curcodeline.trim();

			// Break up the code line
			p = new parsestring(curcodeline, thisplayer, " ");
			
			// Get our line of code
			codeline = p.vwords;
			
			// Make sure we have some code
			if (codeline.getCount() != 0) {
				
				// Cycle through each instruction, interpreting it
				while (instruction <= codeline.getCount()) {
					try {
						interpret();
					}
					catch(Exception e) {
						e.printStackTrace();
						data.oerror.RaiseError(source, "Interpreter error - " + e.toString(), programcounter, instruction);
						e.printStackTrace(thisplayer.socket_outputstream);
					}	
				}
			}
			
			programcounter++;	
		}
	}
	
	/** Executes the iagecode collection passed in. The source is a string to identify the
	 *  code being executed if an error occurs.
	 *  This overloaded version allows you to specify a program counter to start at
	 */
	public void runcode(iagecode tcode, String codesource, int progcountstartat) {
		
		source = codesource;
		cancelled = false;
		programcounter = progcountstartat;
		thecode = tcode;
		parsestring p = null;
		String curcodeline = "";
		
		// Drop out now if there are no lines of code
		if (thecode.getCount() == 0) return;
		
		while (programcounter <= thecode.getCount()) {
			
			// start at the first instruction
			instruction = 1;
			
			// Remove all leading spaces from the code line
			curcodeline = thecode.get(programcounter);
			curcodeline = data.removeleadingspaces(curcodeline);
			
			// Break up the code line
			p = new parsestring(curcodeline, thisplayer, " ");
			
			// Get our line of code
			codeline = p.vwords;
			
			// Make sure we have some code
			if (codeline.getCount() != 0) {
				
				// Cycle through each instruction, interpreting it
				while (instruction <= codeline.getCount()) {
					try {
						interpret();
					}
					catch(Exception e) {
						data.oerror.RaiseError(source, "Interpreter error - " + e.toString(), programcounter, instruction);
						e.printStackTrace(thisplayer.socket_outputstream);
					}
				}
			}
			
			programcounter++;	
		}
	}

	/*** Interprets and executes the current instruction */
	private void interpret() {
		
		// Here we scan for commands/objects and call their appropriate routines.
	    
	    // if this is a line label, ignore it - we can look it up later
	    if (checkcommand(":")) {instruction++; return;}
	    
	    // Ignore the entire line if they are comments
	    if (checkcommand("'")) {instruction = codeline.getCount() + 1; return;}
	    
	    // Ignore the entire line if it is a procedure declaration
	    if (checkcommand("proc ")) {instruction = codeline.getCount() + 1; return;}
	    
	    // Any properties encountered at this level will be to set.
	    isget = false;
	    
	    // Shorthand IF statements
	    if (checkcommand(";")) {Cmd_IfVerb(); return;}
	    if (checkcommand("#")) {Cmd_IfAdverb(); return;}
	    
	    // Check for references to objects
	    if (checkcommand("message(")) {Obj_Message((String) codeline.get(instruction)); return;}
	    if (checkcommand("flag(")) {Obj_Flag((String) codeline.get(instruction)); return;}
	    if (checkcommand("game.")) {Obj_Game((String) codeline.get(instruction)); return;}
	    if (checkcommand("currentplayer.")) {Obj_CurrentPlayer((String) codeline.get(instruction)); return;}
	    if (checkcommand("playerarray(")) {Obj_PlayerArray((String) codeline.get(instruction)); return;}
	    if (checkcommand("player(")) {Obj_Player((String) codeline.get(instruction)); return;}
	    if (checkcommand("location(")) {Obj_Location((String) codeline.get(instruction)); return;}
	    if (checkcommand("item(")) {Obj_Item((String) codeline.get(instruction)); return;}
	    if (checkcommand("internal.")) {Obj_Internal((String) codeline.get(instruction)); return;}
	    if (checkcommand("character(")) {Obj_Character((String) codeline.get(instruction)); return;}
	    if (checkcommand("input.")) {Obj_Input((String) codeline.get(instruction)); return;}
	    if (checkcommand("returnvalue")) {Prop_ReturnValue(); return;}
	    if (checkcommand("array")) {Obj_Array((String) codeline.get(instruction)); return;}
	    
	    // Inbuilt statements and commands
	    // Inbuilt statements and commands
	    if (checkcommand("if")) {Cmd_If(); return;}
	    if (checkcommand("goto")) {Cmd_Goto(); return;}
	    if (checkcommand("call")) {Cmd_Call(); return;}
	    if (checkcommand("while")) {Cmd_While(); return;}
	    if (checkcommand("for")) {Cmd_For(); return;}
	    if (checkcommand("var")) {Cmd_Var(); return;}
	    if (checkcommand("ask")) {Cmd_Ask(); return;} 
	    if (checkcommand("getitemfromnoun")) { Cmd_GetItemFromNoun(); return;}
	    if (checkcommand("outputcontentsof")) { Cmd_OutputContentsOf(); return;}
	    if (checkcommand("printallexcept")) {	instruction++; 
	    										String pindex = performevaluate((String) codeline.get(instruction));
	    										int pind = Integer.parseInt(pindex);
	    										instruction++;
	    										String expr = (String) codeline.get(instruction);
	    										instruction++;
	    										int i = 1;
	    										player theplayer = null;
	    										while (i <= data.oplayers.getCount()) {
	    											theplayer = (player) data.oplayers.get(i);
	    											if (theplayer.Index == pind) {
	    												vdu.TransmitAllEx(performevaluate(expr), theplayer);
	    												return;
	    											}
	    											i++;	
	    										}
	    									}
	    if (checkcommand("printallinexcept")) {	instruction++; 
	    										String loc = performevaluate((String) codeline.get(instruction));
	    										long locid = Long.parseLong(loc);
	    										instruction++;
	    										String pindex = performevaluate((String) codeline.get(instruction));
	    										int pind = Integer.parseInt(pindex);
	    										instruction++;
	    										String expr = (String) codeline.get(instruction);
	    										instruction++;
	    										int i = 1;
	    										player theplayer = null;
	    										while (i <= data.oplayers.getCount()) {
	    											theplayer = (player) data.oplayers.get(i);
	    											if (theplayer.Index == pind) {
														break;
	    											}
	    											i++;	
	    										}
	    										if (theplayer != null) {
	    									    	vdu.TransmitAllInLocation(performevaluate(expr), theplayer, locid);
	    									    }
	    										return;
	    									}
	    if (checkcommand("printallin")) 	{	instruction++; 
	    										String loc = performevaluate((String) codeline.get(instruction));
	    										long locid = Long.parseLong(loc);
	    										instruction++;
	    										String expr = (String) codeline.get(instruction);
	    										instruction++;
	    									    vdu.TransmitAllInLocation(performevaluate(expr), thisplayer, locid);
	    									    if (thisplayer !=null) {
	    									    	if (thisplayer.CurrentLocation == locid) {vdu.Transmit(performevaluate(expr), thisplayer);}
	    									    }
	    										return;
	    									}
	    
	    // Dictionary
	    if (checkcommand("addverb")) {instruction++;
	    							  String tid = (String) codeline.get(instruction);
	    							  instruction++;
	    							  String ttext = (String) codeline.get(instruction);
	    							  verb ve = new verb();
	    							  ve.ID = Long.parseLong(tid);
	    							  ve.Text = ttext;
	    							  data.overbs.add(ve);
	    							  instruction++;
	    							  return;
	    							 }
		if (checkcommand("addadverb")) {instruction++;
	    							  String tid = (String) codeline.get(instruction);
	    							  instruction++;
	    							  String ttext = (String) codeline.get(instruction);
	    							  adverb adve = new adverb();
	    							  adve.ID = Long.parseLong(tid);
	    							  adve.Text = ttext;
	    							  data.oadverbs.add(adve);
	    							  instruction++;
	    							  return;
	    							 }
	    							 
		if (checkcommand("addnoun")) {instruction++;
	    							  String tid = (String) codeline.get(instruction);
	    							  instruction++;
	    							  String ttext = (String) codeline.get(instruction);
	    							  noun no = new noun();
	    							  no.ID = Long.parseLong(tid);
	    							  no.Text = ttext;
	    							  data.onouns.add(no);
	    							  instruction++;
	    							  return;
	    							 }
	    							 
	    if (checkcommand("additem")) {instruction++;
	    							  String tid = (String) codeline.get(instruction);
	    							  item im = new item();
	    							  im.ID = Long.parseLong(tid);
	    							  data.oitems.add(im);
	    							  instruction++;
	    							  return;
	    							  }
	    							  
		if (checkcommand("addlocation")) {instruction++;
	    							  String tid = (String) codeline.get(instruction);
	    							  location lo = new location();
	    							  lo.ID = Long.parseLong(tid);
	    							  data.olocations.add(lo);
	    							  instruction++;
	    							  return;
	    							  }
	    							  
		if (checkcommand("addnpc")) {instruction++;
	    							  String tid = (String) codeline.get(instruction);
	    							  character cc = new character();
	    							  cc.ID = Long.parseLong(tid);
	    							  data.ocharacters.add(cc);
	    							  instruction++;
	    							  return;
	    							  }
	    							  
	    if (checkcommand("removeitem")) {instruction++;
	    								 String tid = performevaluate((String) codeline.get(instruction));
	    								 long lid = Long.parseLong(tid);
	    								 // Find item
	    								 int i = 1;
	    								 item im = null;
	    								 while (i <= data.oitems.getCount()) {
	    								 	im = (item) data.oitems.get(i);
	    								 	if (im.ID == lid) {
	    								 		data.oitems.remove(i);
	    								 		break;
	    								 	}
	    								 	i++;
	    								 }
	    								 instruction++;
	    							}
	    							
	    if (checkcommand("removelocation")) {instruction++;
	    								 String tid = performevaluate((String) codeline.get(instruction));
	    								 long lid = Long.parseLong(tid);
	    								 // Find item
	    								 int i = 1;
	    								 location im = null;
	    								 while (i <= data.olocations.getCount()) {
	    								 	im = (location) data.olocations.get(i);
	    								 	if (im.ID == lid) {
	    								 		data.olocations.remove(i);
	    								 		break;
	    								 	}
	    								 	i++;
	    								 }
	    								 instruction++;
	    							}
	    							
	    if (checkcommand("removenpc")) {instruction++;
	    								 String tid = performevaluate((String) codeline.get(instruction));
	    								 long lid = Long.parseLong(tid);
	    								 // Find item
	    								 int i = 1;
	    								 character im = null;
	    								 while (i <= data.ocharacters.getCount()) {
	    								 	im = (character) data.ocharacters.get(i);
	    								 	if (im.ID == lid) {
	    								 		data.ocharacters.remove(i);
	    								 		break;
	    								 	}
	    								 	i++;
	    								 }
	    								 instruction++;
	    							}
	    							 
	    // Media
	    if (checkcommand("playsound")) {Cmd_PlaySound(); return;}
	    if (checkcommand("playmovie")) {Cmd_PlayMovie(); return;}
	    if (checkcommand("playmidi")) {Cmd_PlayMIDI(); return;}
	    if (checkcommand("stopmidi")) {Cmd_StopMIDI(); return;}
	    if (checkcommand("stopmovie")) {Cmd_StopMovie(); return;}
	    if (checkcommand("openurl")) {Cmd_OpenURL(); return;}
	    if (checkcommand("showpicture")) {Cmd_ShowPicture(); return;}
	    
	    
	    // Local Variables
	    int i = 1;
	    String varname = "";
	    while (i <= localvariables.getCount()) {
	    	varname = (String) localvariables.get(i);
	    	if (checkcommand(varname)) {
	    		if (data.debugging) System.out.println("Altering local variable " + varname);
	    		if (data.debugging) System.out.println("was " + (String)localvariablevalues.get(i));
	    		// They are trying to alter the value of a local variable
	    		// let's find it - next instruction should be = sign.
	    		instruction++;
	    		// Next instruction is expression - assign it
	    		instruction++;
	    		// Set the local variable value over the top
	    		localvariablevalues.set(i, performevaluate((String) codeline.get(instruction)));
	    		if (data.debugging) System.out.println("should be " + performevaluate((String) codeline.get(instruction)));
	    		if (data.debugging) System.out.println("now " + (String)localvariablevalues.get(i));
	    		// Found it so break now
	    		break;
	    	}
	    	i++;	
	    }
	    
	    
	    // Skippers - ie. commands we don//t actually run, they//re just markers
	    if (checkcommand("else")) {
	        // Find the end if and skip to after there.
	        findendif();
			return;
	 	}
	 	
	    if (checkcommand("endif")) {
	        // if we find an endif, skip to the next line
	        instruction = codeline.getCount() + 1;
	        return;
	 	}
	 	
	 	if (checkcommand("endwhile")) {
	        // if we find an endwhile, find it's while and go back
	        findwhile();
	        return;
	 	}
	    
	    // Code terminators. Note that end HAS to appear after
	    // endif otherwise we//d break in the wrong places.
	    if (checkcommand("end")) {Cmd_End(); return;}
	    if (checkcommand("cancelevent")) {Cmd_CancelEvent(); return;}
	    
	    // if we came this far and nothing has happened then increment the instruction to carry on
	    instruction++;
	}
	
	/*** Returns the object ID referenced in an instruction by looking
	   * for the first opening and closing brackets and returning an 
	   * evaluation of what's between them.
	   */
	public long getobjectitem(String s) {
		int spos = 0;
		int epos = 0;
		long retvalue = 0;
		
		// Remember property let/get status
		boolean letget = isget;
		
		spos = s.indexOf("(");
		epos = s.indexOf(")");
		
		String retstr = performevaluate(s.substring(spos + 1, epos));
		retvalue = Long.parseLong(retstr);
		
		// Restore property let/get status
		isget = letget;
		
		return retvalue;
	}
	
	/**
     *  Returns the property/method used with an object
     *  this is done by return everything after the dot
     *  in the instruction.
     */
	public String getobjectcall(String s) {
		return s.substring(s.indexOf(".") + 1, s.length());
	}
	
	/*** Checks to see if the command matches */
	private boolean checkcommand(String ccheck) {		
		// Check it's in there
		String ts = (String) codeline.get(instruction);
		ts = data.trimstring(ts);
		if (ts.startsWith(ccheck)) {
			return true;
		}
		else
		{
			return false;
		}
	}
	
	private boolean checkwhole(String ccheck) {
		String ts = (String) codeline.get(instruction);
		ts = ts.trim();
		return ts.equalsIgnoreCase(ccheck.toLowerCase());
	}
	
	/*** Checks to see if a string starts with another */
	private boolean checkstart(String checkstr, String checkagainst) {
		
		checkstr = checkstr.toLowerCase();
		checkagainst = checkagainst.toLowerCase();
		checkagainst = data.trimstring(checkagainst);
		if (checkagainst.startsWith(checkstr)) {
			return true;
		}
		else
		{
			return false;
		}
				
	}
	
	/*** Checks to see if one string exists in another string */
	public boolean checkexistence(String ccheck, String tofind) {
		// Force to lower case
		ccheck = ccheck.toLowerCase();
		tofind = tofind.toLowerCase();
		// Check it's there
		if (tofind.indexOf(ccheck) != -1 ) {
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/*** Does property assignment checks and performs evaluation. The result of the
	   * evaluation is return for assignment, formatted as a string.
	   */
	private String propertyAssign() {
		instruction++;
		// check for property assignment
		String ts = (String)codeline.get(instruction);
		if (ts.equals("=")) {
			// found one, so return the evaluation result
			instruction++;
			return performevaluate((String)codeline.get(instruction));
		}
		else data.oerror.RaiseError(source, "Invalid property assignment", programcounter, instruction);
		return "";
	}
	
	private String Obj_Message(String s) {
		
		int i = 1;
		message m = null;
		boolean foundobj = false;
		boolean executed = false;
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// find the object - not if it's a count request though - 
		// we don't need to find it for that
		if (!getobjectcall(s).equals("count")) {
			while (i <= data.omessages.getCount()) {
				m = (message) data.omessages.get(i);
				if (m.ID == getobjectitem(s)) {
					foundobj = true;
					break;
				}
				i++;	
			}
			
			// if we didn't find it, raise an error
			if (!foundobj) {
				data.oerror.RaiseError(source, "Message " + Long.toString(getobjectitem(s)) + " does not exist.", programcounter, instruction);
				instruction++;
				return "";
			}
		}
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
		
		// ------------- METHODS -------------
		
		if (tcall.equals("show")) {vdu.Transmit(m.Text, thisplayer); executed = true;}
		if (tcall.equals("showothers")) {vdu.TransmitAllInLocation(m.Text, thisplayer, thisplayer.CurrentLocation);}
		if (tcall.equals("showallin")) {
											// Get location ID from next instruction
											instruction++;
											if (instruction > codeline.getCount()) {
												data.oerror.RaiseError(source, "No location specified for ShowAllIn command.", programcounter, instruction);
											}
											else
											{
												vdu.TransmitAllInLocation(m.Text, thisplayer, Long.parseLong((String)codeline.get(instruction)));
											}
										}
										
		// ------------- PROPERTIES ------------
		if (tcall.equals("id")) {if (isget) {rval = Long.toString(m.ID); } else {m.ID = Long.parseLong(propertyAssign());}}
		if (tcall.equals("text")) {if (isget) {rval = m.Text; } else {m.Text = propertyAssign();}}
		
		if (tcall.equals("count"))	{ rval = Integer.toString(data.omessages.getCount()); }
		
		instruction++;
		return rval;
	}
	
	private String Obj_Character(String s) {
		
		int i = 1;
		character c = null;
		boolean foundobj = false;
		boolean executed = false;
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// find the object - not if it's a count request though - 
		// we don't need to find it for that
		if (!getobjectcall(s).equals("count")) {
			while (i <= data.ocharacters.getCount()) {
				c = (character) data.ocharacters.get(i);
				if (c.ID == getobjectitem(s)) {
					foundobj = true;
					break;
				}
				i++;	
			}
			
			// if we didn't find it, raise an error
			if (!foundobj) {
				data.oerror.RaiseError(source, "Character " + Long.toString(getobjectitem(s)) + " does not exist.", programcounter, instruction);
				instruction++;
				return "";
			}
		}
		
	    // Check for custom properties. Interestingly, custom properties
	    // can override inbuilt properties.
	    int x = 0;
	    int y = 1;
	    String flg = "";
	    int flgno = 1;
	    
	    if (c != null) {
		    x = c.CustomProperties.indexOf(getobjectcall(s).toLowerCase());
		    if (x != -1) {
		        // Find the comma
		        x = c.CustomProperties.indexOf(",", x);
		        if (x != -1) {
		            // Find the pipe
		            y = c.CustomProperties.indexOf("|", x);
		            // If we don't have one, use the end of the string
		            if (y == -1) y = c.CustomProperties.length();
		            // Get the flag number
		            flg = c.CustomProperties.substring(x + 1, y);
		            // Do standard property code with flag instead
		            flgno = Integer.parseInt(flg);
		            if (isget) {rval = data.oflags[flgno].Value; } else {data.oflags[flgno].Value = propertyAssign();}
		            // Move to next instruction and return
		            instruction++;
					return rval;
		     	}
		 	}
		}
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
		
		// ------------- METHODS -------------
		if (tcall.indexOf("setvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 boolean foundupdate = false;
												 String valname = tcall.substring(startname, endname);
												 
												 // Get the replacement value
												 instruction++;
												 instruction++;
												 String newval = performevaluate((String) codeline.get(instruction));
												 
												 c.setNameValue(valname, newval);
											}
											
		if (tcall.indexOf("getvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 String valname = tcall.substring(startname, endname);
												 
												 rval = c.getNameValue(valname);
											}
											
		if (tcall.equals("codeclear")) { // Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Test it
										 if (eventname.equalsIgnoreCase("onaction"))
										 	c.OnAction = new iagecode();
										 	
										 if (eventname.equalsIgnoreCase("ontalk"))
										 	c.OnTalk = new iagecode();
										 	
										 if (eventname.equalsIgnoreCase("ontimer"))
										 	c.OnTimer = new iagecode(); 	
										}
										
 		if (tcall.equals("codeaddline")) {// Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Get the new code line
										 instruction++;
										 String newcode = (String) codeline.get(instruction);
										 
										 // Test it
										 if (eventname.equalsIgnoreCase("onaction"))
										 	c.OnAction.add(newcode);
										 	
										 if (eventname.equalsIgnoreCase("ontalk"))
										 	c.OnTalk.add(newcode);
										 	
										 if (eventname.equalsIgnoreCase("ontalk"))
										 	c.OnTimer.add(newcode); 	
										}
										
		if (tcall.equals("fireinlineevent")) { // Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Get the inline event to fire
										 instruction++;
										 String inevent = (String) codeline.get(instruction);
										 
										 // Test it and run code
										 if (eventname.equalsIgnoreCase("OnAction")) {
										 	interpreter inlevent = new interpreter(thisplayer, thisparser);
	                						inlevent.runcode(c.OnAction, "Character(" + c.Name + ").OnAction", inevent, new String[4], 0, "InlineEvent." + inevent, 0, 0);
										 	}
										 if (eventname.equalsIgnoreCase("OnTalk")) {
										 	interpreter inlevent = new interpreter(thisplayer, thisparser);
	                						inlevent.runcode(c.OnTalk, "Character(" + c.Name + ").OnTalk", inevent, new String[4], 0, "InlineEvent." + inevent, 0, 0);
										 	}
										 if (eventname.equalsIgnoreCase("OnTimer")) {
										 	interpreter inlevent = new interpreter(thisplayer, thisparser);
	                						inlevent.runcode(c.OnTimer, "Character(" + c.Name + ").OnTimer", inevent, new String[4], 0, "InlineEvent." + inevent, 0, 0);
										 	}
										}

										
		// ------------- PROPERTIES ------------
		if (tcall.equals("id")) {if (isget) {rval = Long.toString(c.ID); } else {c.ID = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("name")) {if (isget) {rval = c.Name; } else {c.Name = propertyAssign();}}
		if (tcall.equals("currentlocation")) {if (isget) {rval = Long.toString(c.CurrentLocation); } else {c.CurrentLocation = Long.parseLong(propertyAssign());}}
		if (tcall.equals("containerlocation")) {if (isget) {rval = Long.toString(location.NPCBASE + c.ID); }}
		if (tcall.equals("hitpoints")) {if (isget) {rval = Long.toString(c.HitPoints); } else {c.CurrentLocation = Long.parseLong(propertyAssign());}}
		if (tcall.equals("damageindicator")) {if (isget) {rval = Long.toString(c.DamageIndicator); } else {c.CurrentLocation = Long.parseLong(propertyAssign());}}
		if (tcall.equals("money")) {if (isget) {rval = Long.toString(c.Money); } else {c.Money = Long.parseLong(propertyAssign());}}
		if (tcall.equals("autoattack")) {if (isget) {rval = data.booleanToString(c.AutoAttack); } else {c.AutoAttack = Boolean.getBoolean(propertyAssign());}}
		if (tcall.equals("attackwhenattacked")) {if (isget) {rval = data.booleanToString(c.AttackWhenAttacked); } else {c.AttackWhenAttacked = Boolean.getBoolean(propertyAssign());}}
		if (tcall.equals("attackingwho")) {if (isget) {rval = Integer.toString(c.AttackingWho); } else {c.AttackingWho = Integer.parseInt(propertyAssign());}}									
		if (tcall.equals("description")) {if (isget) {rval = c.Description; } else {c.Description = propertyAssign();}}
		if (tcall.equals("defaultexamine")) {if (isget) {rval = c.DefaultExamine; } else {c.DefaultExamine = propertyAssign();}}
		if (tcall.equals("nounid")) {if (isget) {rval = Long.toString(c.NounID); } else {c.NounID = Long.parseLong(propertyAssign());}}
		if (tcall.equals("timerinterval")) {if (isget) {rval = Long.toString(c.TimerInterval); } else {c.TimerInterval = Long.parseLong(propertyAssign());}}
		if (tcall.equals("timerindex")) {if (isget) {rval = Long.toString(c.TimerIndex); } else {c.TimerIndex = Long.parseLong(propertyAssign());}}
		if (tcall.equals("timetonextrun")) {if (isget) {rval = Long.toString(c.TimeToNextRun); } else {c.TimeToNextRun = Long.parseLong(propertyAssign());}}
		if (tcall.equals("movedfromoriginallocation")) {if (isget) {rval = data.booleanToString(c.MovedFromOriginalLocation); } else {c.MovedFromOriginalLocation = Boolean.getBoolean(propertyAssign());}}
		if (tcall.equals("aimode")) {if (isget) {rval = Byte.toString(c.AIMode); } else {c.AIMode = Byte.parseByte(propertyAssign());}}
		if (tcall.equals("followplayerindex")) {if (isget) {rval = Long.toString(c.FollowPlayerIndex); } else {c.FollowPlayerIndex = Long.parseLong(propertyAssign());}}
		
		if (tcall.equals("count"))	{ rval = Integer.toString(data.ocharacters.getCount()); }
		
		instruction++;
		return rval;
	}
	
	private String Obj_Array(String s) {
		
		String rval = "";
		
		s = s.toLowerCase();
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
	
		if (tcall.equals("new")) {
			
						// Get the name
						instruction++;			
						String arrname = (String) codeline.get(instruction);
						
						// Get initial values
						String[] args = new String[codeline.getCount()];
						
						int i = 1;
						instruction++;
						while (instruction <= codeline.getCount()) {
							args[i] = performevaluate((String)codeline.get(instruction));
							instruction++;
							i++;
						}
						
						String arrvalue = "";
						if (i > 1) {
							// Create initialised values	
							int j = 1;
							while (j <= i) {
								arrvalue = arrvalue + args[j];
								if (j < i) arrvalue = arrvalue + "|";
								j++;					
							}
						}
						
						// Now create the array as a local variable
						localvariables.add(arrname);
						localvariablevalues.add(arrvalue);
					}
					
		if (tcall.equals("add")) {		
		
						// Get the name
						instruction++;
						String arrname = (String) codeline.get(instruction);
						
						// Get the value
						instruction++;
						String arrvalue = performevaluate((String)codeline.get(instruction));
						
						// Find the array var
						int i = 1;
						String nametest = "";
						while (i <= localvariables.getCount()) {
							nametest = (String) localvariables.get(i);
							if (nametest.equalsIgnoreCase(arrname)) {
								
								// We got it - add our new value
								String arrayval = (String) localvariablevalues.get(i);
								if (arrayval.length() > 0) arrayval = arrayval + "|";
								arrayval = arrayval + arrvalue;
								localvariablevalues.set(i, arrayval);
								
								break;
							}	
							i++;	
						}
					}
					
		if (tcall.equals("get")) {
			
						// Get the name
						instruction++;
						String arrname = (String) codeline.get(instruction);
						
						// Get the item number
						instruction++;
						int arrnum = Integer.parseInt(performevaluate((String)codeline.get(instruction)));
						
						// Find the array var
						int i = 1;
						String nametest = "";
						while (i <= localvariables.getCount()) {
							nametest = (String) localvariables.get(i);
							if (nametest.equalsIgnoreCase(arrname)) {
								
								// Create a parser and break the array
								String arrayval = (String) localvariablevalues.get(i);
								parsestring pa = new parsestring(arrayval, thisplayer, "|");
								
								// Get our value for return
								rval = (String) pa.vwords.get(arrnum);
								break;
							}	
							i++;	
						}
					}
					
		if (tcall.equals("set")) {
			
						// Get the name
						instruction++;
						String arrname = (String) codeline.get(instruction);
						
						// Get the item number
						instruction++;
						int arrnum = Integer.parseInt(performevaluate((String)codeline.get(instruction)));
						
						// Get the new value
						instruction++;
						String newval = performevaluate((String)codeline.get(instruction));
						
						// Find the array var
						int i = 1;
						String nametest = "";
						while (i <= localvariables.getCount()) {
							nametest = (String) localvariables.get(i);
							if (nametest.equalsIgnoreCase(arrname)) {
								
								// Create a parser and break the array
								String arrayval = (String) localvariablevalues.get(i);
								parsestring pa = new parsestring(arrayval, thisplayer, "|");
								
								// Set the new value
								pa.vwords.set(arrnum, newval);
								
								// Rebuild the array
								int j = 1;
								arrayval = "";
								while (j <= pa.vwords.getCount()) {
									arrayval = arrayval + pa.vwords.get(j);
									if (j < pa.vwords.getCount()) arrayval = arrayval + "|";
									j++;
								}

								break;
							}	
							i++;	
						}
					}
						
		if (tcall.equals("remove")) {
			
						// Get the name
						instruction++;
						String arrname = (String) codeline.get(instruction);
						
						// Get the item number
						instruction++;
						int arrnum = Integer.parseInt(performevaluate((String)codeline.get(instruction)));
						
						// Find the array var
						int i = 1;
						String nametest = "";
						while (i <= localvariables.getCount()) {
							nametest = (String) localvariables.get(i);
							if (nametest.equalsIgnoreCase(arrname)) {
								
								// Create a parser and break the array
								String arrayval = (String) localvariablevalues.get(i);
								parsestring pa = new parsestring(arrayval, thisplayer, "|");
								
								// Remove the value
								pa.vwords.remove(arrnum);
								
								// Rebuild the array
								int j = 1;
								arrayval = "";
								while (j <= pa.vwords.getCount()) {
									arrayval = arrayval + pa.vwords.get(j);
									if (j < pa.vwords.getCount()) arrayval = arrayval + "|";
									j++;
								}
								
								// Store it
								localvariablevalues.set(i, arrayval);
								
								break;
							}	
							i++;	
						}
					}		
					
		if (tcall.indexOf("getcount") != -1) {
			
						// Get the name
						// Find bool check
						int startname = tcall.indexOf("(") + 1;
						int endname = tcall.indexOf(")");
						String arrname = tcall.substring(startname, endname);
						
						// Find the array var
						int i = 1;
						String nametest = "";
						while (i <= localvariables.getCount()) {
							nametest = (String) localvariables.get(i);
							if (nametest.equalsIgnoreCase(arrname)) {
								
								// Create a parser and break the array
								String arrayval = (String) localvariablevalues.get(i);
								parsestring pa = new parsestring(arrayval, thisplayer, "|");
								
								// Return the count
								rval = Integer.toString(pa.vwords.getCount() - 1);
								
								break;
							}	
							i++;	
						}
					}						
						
		
		instruction++;
		return rval;
	}
	
	/** Reference to the current player */
	private String Obj_CurrentPlayer(String s) {
		
		player p = thisplayer;
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
		rval = Obj_PlayerAction(tcall, p);
		
		instruction++;
		return rval;
	}
	
	private String Obj_Flag(String s) {
		
		flag f = null;
		String rval = "";
		long flagno = 0;
		int flg = 0;
		
		// Find our flag
		flagno = getobjectitem(s);
		flg = Integer.parseInt(Long.toString(flagno));
		f = data.oflags[flg];
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
										
		// ------------- PROPERTIES ------------
		if (tcall.equals("value")) {if (isget) {rval = f.Value; } else {f.Value = propertyAssign();}}
		
		if (tcall.equals("count")) {rval = Integer.toString(data.oflags.length);}
		
		
		instruction++;
		return rval;
	}
	
	private String Obj_Input(String s) {
		
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
										
		// ------------- PROPERTIES ------------
		if (tcall.equals("verb")) {if (isget) {rval = Long.toString(thisparser.Verb); } else {thisparser.Verb = Long.parseLong(propertyAssign());} }
		if (tcall.equals("verb2")) {if (isget) {rval = Long.toString(thisparser.Verb2); } else {thisparser.Verb2 = Long.parseLong(propertyAssign());}}
		if (tcall.equals("verb3")) {if (isget) {rval = Long.toString(thisparser.Verb3); } else {thisparser.Verb3 = Long.parseLong(propertyAssign());}}
		if (tcall.equals("noun")) {if (isget) {rval = Long.toString(thisparser.Noun); } else {thisparser.Noun = Long.parseLong(propertyAssign());}}
		if (tcall.equals("noun2")) {if (isget) {rval = Long.toString(thisparser.Noun2); } else {thisparser.Noun2 = Long.parseLong(propertyAssign());}}
		if (tcall.equals("noun3")) {if (isget) {rval = Long.toString(thisparser.Noun3); } else {thisparser.Noun3 = Long.parseLong(propertyAssign());}}
		if (tcall.equals("adverb")) {if (isget) {rval = Long.toString(thisparser.Adverb); } else {thisparser.Adverb = Long.parseLong(propertyAssign());}}
		if (tcall.equals("adverb2")) {if (isget) {rval = Long.toString(thisparser.Adverb2); } else {thisparser.Adverb2 = Long.parseLong(propertyAssign());}}
		if (tcall.equals("adverb3")) {if (isget) {rval = Long.toString(thisparser.Adverb3); } else {thisparser.Adverb3 = Long.parseLong(propertyAssign());}}		
		if (tcall.equals("sverb")) {if (isget) {rval = thisparser.SVerb; } else {thisparser.SVerb = propertyAssign();}}
		if (tcall.equals("sverb2")) {if (isget) {rval = thisparser.SVerb2; } else {thisparser.SVerb2 = propertyAssign();}}
		if (tcall.equals("sverb3")) {if (isget) {rval = thisparser.SVerb3; } else {thisparser.SVerb3 = propertyAssign();}}
		if (tcall.equals("sadverb")) {if (isget) {rval = thisparser.SAdverb; } else {thisparser.SAdverb = propertyAssign();}}
		if (tcall.equals("sadverb2")) {if (isget) {rval = thisparser.SAdverb2; } else {thisparser.SAdverb2 = propertyAssign();}}
		if (tcall.equals("sadverb3")) {if (isget) {rval = thisparser.SAdverb3; } else {thisparser.SAdverb3 = propertyAssign();}}
		if (tcall.equals("snoun")) {if (isget) {rval = thisparser.SNoun; } else {thisparser.SNoun = propertyAssign();}}
		if (tcall.equals("snoun2")) {if (isget) {rval = thisparser.SNoun2; } else {thisparser.SNoun2 = propertyAssign();}}
		if (tcall.equals("snoun3")) {if (isget) {rval = thisparser.SNoun3; } else {thisparser.SNoun3 = propertyAssign();}}
		if (tcall.equals("isnpcaddress")) {if (isget) {rval = data.booleanToString(thisparser.isnpcaddress);}}
		
		//-------------- METHODS ----------------
		if (tcall.indexOf("word") != -1) {
											// return the word at the index specified
											int startname = tcall.indexOf("(") + 1;
											int endname = tcall.indexOf(")");
											String wordval = tcall.substring(startname, endname);
											int wordindex = (int) Long.parseLong(performevaluate(wordval));
											rval = (String) thisparser.vwords.get(wordindex);
										}
		if (tcall.equals("totalwords")) {if (isget) {rval = Integer.toString(thisparser.vwords.getCount());}}
		if (tcall.indexOf("contains") != -1) {
											// return true or false if the word fragment
											// appears anywhere in the parsed string
											int startname = tcall.indexOf("(") + 1;
											int endname = tcall.indexOf(")");
											String wordval = tcall.substring(startname, endname);
											String wordtofind = performevaluate(wordval);
											if (thisparser.originalstring.toLowerCase().indexOf(wordtofind.toLowerCase()) != -1)
												rval = "true";
											else
												rval = "false";
										}
										
		
		
		instruction++;
		return rval;
	}
	
	private String Obj_Item(String s) {
		int i = 1;
		item im = null;
		boolean foundobj = false;
		boolean executed = false;
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// find the object - not if it's a count request though - 
		// we don't need to find it for that
		if (!getobjectcall(s).equals("count")) {
			while (i <= data.oitems.getCount()) {
				im = (item) data.oitems.get(i);
				if (im.ID == getobjectitem(s)) {
					foundobj = true;
					break;
				}
				i++;	
			}
			
			// if we didn't find it, raise an error
			if (!foundobj) {
				data.oerror.RaiseError(source, "Item " + Long.toString(getobjectitem(s)) + " does not exist.", programcounter, instruction);
				instruction++;
				return "";
			}
		}
		
	    // Check for custom properties. Interestingly, custom properties
	    // can override inbuilt properties.
	    int x = 0;
	    int y = 1;
	    String flg = "";
	    int flgno = 1;
	    
	    if (im != null) {
		    x = im.CustomProperties.indexOf(getobjectcall(s).toLowerCase());
		    if (x != -1) {
		        // Find the comma
		        x = im.CustomProperties.indexOf(",", x);
		        if (x != -1) {
		            // Find the pipe
		            y = im.CustomProperties.indexOf("|", x);
		            // If we don't have one, use the end of the string
		            if (y == -1) y = im.CustomProperties.length();
		            // Get the flag number
		            flg = im.CustomProperties.substring(x + 1, y);
		            // Do standard property code with flag instead
		            flgno = Integer.parseInt(flg);
		            if (isget) {rval = data.oflags[flgno].Value; } else {data.oflags[flgno].Value = propertyAssign();}
		            // Move to next instruction and return
		            instruction++;
					return rval;
		     	}
		 	}
		}
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
		
		// ------------- METHODS -------------
		if (tcall.indexOf("getuserboolean") != -1) { // Find bool check
													 int startname = tcall.indexOf("(") + 1;
													 int endname = tcall.indexOf(")");
													 String boolname = tcall.substring(startname, endname);
													 // Search for our name in user booleans
													 if (im.UserBooleans.indexOf(boolname) != -1) {
													 	rval = "true";
													 }
													 else
													 {
													 	rval = "false";
													 }
												   }
												   
		if (tcall.indexOf("adduserboolean") != -1) { // Find bool check
													 int startname = tcall.indexOf("(") + 1;
													 int endname = tcall.indexOf(")");
													 String boolname = tcall.substring(startname, endname);
													 // Add to list
													 im.UserBooleans = im.UserBooleans + " " + boolname;
												   }
												   
	if (tcall.indexOf("removeuserboolean") != -1) { // Find bool check
													 int startname = tcall.indexOf("(") + 1;
													 int endname = tcall.indexOf(")");
													 String boolname = tcall.substring(startname, endname);
													 // Remove from list
													 StringBuffer sb = new StringBuffer(im.UserBooleans);
													 int spos = im.UserBooleans.indexOf(boolname);
													 if (spos != -1) {
													 	if (im.UserBooleans.equalsIgnoreCase(boolname)) {
													 		im.UserBooleans = "";
													 	}
													 	else
													 	{
														 	sb = vdu.stringbuffreplace(sb, spos, spos + boolname.length(), "");
														 	im.UserBooleans = sb.toString();
													 	}
													 }
												   }
												   
		if (tcall.indexOf("getvaluebyexp") != -1) { 
		
												 // Just like getvalue except the named argument
												 // can be an expression.
												 
												 // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 String valname = performevaluate(tcall.substring(startname, endname));
												 
												 rval = im.getNameValue(valname);
											}
											
		if (tcall.indexOf("setvaluebyexp") != -1) { 
		
		
												 // Just like setvalue except the named argument
												 // can be an expression.
		
												 // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 boolean foundupdate = false;
												 String valname = performevaluate(tcall.substring(startname, endname));
												 
												 // Get the replacement value
												 instruction++;
												 instruction++;
												 String newval = performevaluate((String) codeline.get(instruction));
												 
												 im.setNameValue(valname, newval);
											}
												   
		if (tcall.indexOf("setvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 boolean foundupdate = false;
												 String valname = tcall.substring(startname, endname);
												 
												 // Get the replacement value
												 instruction++;
												 instruction++;
												 String newval = performevaluate((String) codeline.get(instruction));
												 
												 im.setNameValue(valname, newval);
											}
											
		if (tcall.indexOf("getvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 String valname = tcall.substring(startname, endname);
												 
												 rval = im.getNameValue(valname);
											}
											
											
		if (tcall.equals("codeclear")) { // Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Test it
										 if (eventname.equalsIgnoreCase("onaction"))
										 	im.OnAction = new iagecode();	
										}
										
 		if (tcall.equals("codeaddline")) {// Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Get the new code line
										 instruction++;
										 String newcode = (String) codeline.get(instruction);
										 
										 // Test it
										 if (eventname.equalsIgnoreCase("onaction"))
										 	im.OnAction.add(newcode);	
										}
										
		if (tcall.equals("fireinlineevent")) { // Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Get the inline event to fire
										 instruction++;
										 String inevent = (String) codeline.get(instruction);
										 
										 // Test it and run code
										 if (eventname.equalsIgnoreCase("OnAction")) {
										 	interpreter inlevent = new interpreter(thisplayer, thisparser);
	                						inlevent.runcode(im.OnAction, "Item(" + im.Name + ").OnAction", inevent, new String[4], 0, "InlineEvent." + inevent, 0, 0);
										 	}
										}
										
		// ------------- PROPERTIES ------------
		if (tcall.equals("id")) {if (isget) {rval = Long.toString(im.ID); } else {im.ID = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("name")) {if (isget) {rval = im.Name; } else {im.Name = propertyAssign();}}
		if (tcall.equals("thename")) {if (isget) {rval = im.Name.substring(im.Name.indexOf(" ") + 1, im.Name.length()); } else {im.Name = propertyAssign();}}
		if (tcall.equals("currentlocation")) {if (isget) {rval = Long.toString(im.CurrentLocation); } else {im.CurrentLocation = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("containerlocation")) {if (isget) {rval = Long.toString(location.CONTAINERBASE + im.ID); }}
		if (tcall.equals("surfacelocation")) {if (isget) {rval = Long.toString(location.SURFACEBASE + im.ID); }}
		if (tcall.equals("description")) {if (isget) {rval = im.Description; } else {im.Description = propertyAssign();}}
		if (tcall.equals("defaultexamine")) {if (isget) {rval = im.DefaultExamine; } else {im.DefaultExamine = propertyAssign();}}
		if (tcall.equals("weight")) {if (isget) {rval = Long.toString(im.Weight); } else {im.Weight = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("containedweight")) {if (isget) {rval = Long.toString(im.ContainedWeight); } else {im.ContainedWeight = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("size")) {if (isget) {rval = Long.toString(im.Size); } else {im.Size = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("nounid")) {if (isget) {rval = Long.toString(im.NounID); } else {im.NounID = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("containedsize")) {if (isget) {rval = Long.toString(im.ContainedSize); } else {im.ContainedSize = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("containedweight")) {if (isget) {rval = Long.toString(im.ContainedWeight); } else {im.ContainedWeight = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("containedsize")) {if (isget) {rval = Long.toString(im.ContainedSize); } else {im.ContainedSize = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("islightsource")) {if (isget) {rval = data.booleanToString(im.IsLightSource); } else {im.IsLightSource = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("islit")) {if (isget) {rval = data.booleanToString(im.IsLit); } else {im.IsLit = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("istransparent")) {if (isget) {rval = data.booleanToString(im.Transparent); } else {im.Transparent = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("iswearable")) {if (isget) {rval = data.booleanToString(im.IsWearable); } else {im.IsWearable = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("isworn")) {if (isget) {rval = data.booleanToString(im.IsWorn); } else {im.IsWorn = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("invisible")) {if (isget) {rval = data.booleanToString(im.Invisible); } else {im.Invisible = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("iscontainer")) {if (isget) {rval = data.booleanToString(im.IsContainer); } else {im.IsContainer = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("isweapon")) {if (isget) {rval = data.booleanToString(im.IsWeapon); } else {im.IsWeapon = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("isfixed")) {if (isget) {rval = data.booleanToString(im.IsFixed); } else {im.IsWeapon = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("fixedmessage")) {if (isget) {rval = im.FixedMessage; } else {im.FixedMessage = propertyAssign();}}
		if (tcall.equals("userbooleans")) {if (isget) {rval = im.UserBooleans; } else {im.UserBooleans = propertyAssign();}}
		if (tcall.equals("hassurface")) {if (isget) {rval = data.booleanToString(im.HasSurface); } else {im.HasSurface = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("canbestoodon")) {if (isget) {rval = data.booleanToString(im.CanBeStoodOn); } else {im.CanBeStoodOn = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("canbesaton")) {if (isget) {rval = data.booleanToString(im.CanBeSatOn); } else {im.CanBeSatOn = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("canbelaidon")) {if (isget) {rval = data.booleanToString(im.CanBeLaidOn); } else {im.CanBeLaidOn = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("canbegotin")) {if (isget) {rval = data.booleanToString(im.CanBeGotIn); } else {im.CanBeGotIn = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("canopenclose")) {if (isget) {rval = data.booleanToString(im.CanOpenClose); } else {im.CanOpenClose = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("openclosestate")) {if (isget) {rval = data.booleanToString(im.OpenCloseState); } else {im.OpenCloseState = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("isedible")) {if (isget) {rval = data.booleanToString(im.IsEdible); } else {im.IsEdible = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("ediblehitpoints")) {if (isget) {rval = Long.toString(im.EdibleHitPoints); } else {im.EdibleHitPoints = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("isreadable")) {if (isget) {rval = data.booleanToString(im.IsReadable); } else {im.IsReadable = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("readabletext")) {if (isget) {rval = im.ReadableText; } else {im.ReadableText = propertyAssign();}}
		if (tcall.equals("issubitem")) {if (isget) {rval = data.booleanToString(im.IsSubItem); } else {im.IsEdible = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("subitemof")) {if (isget) {rval = Long.toString(im.SubItemOf); } else {im.SubItemOf = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("damageindicator")) {if (isget) {rval = Long.toString(im.DamageIndicator); } else {im.DamageIndicator = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("movedfromoriginallocation")) {if (isget) {rval = data.booleanToString(im.MovedFromOriginalLocation); } else {im.MovedFromOriginalLocation = data.stringToBoolean(propertyAssign());}}       		
		
		if (tcall.equals("count"))	{ rval = Integer.toString(data.oitems.getCount()); }
		
		instruction++;
		return rval;
	}
	
	private String Obj_Location(String s) {
		
		int i = 1;
		location l = null;
		boolean foundobj = false;
		boolean executed = false;
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// find the object - not if it's a count request though - 
		// we don't need to find it for that
		if (!getobjectcall(s).equals("count")) {
			while (i <= data.olocations.getCount()) {
				l = (location) data.olocations.get(i);
				if (l.ID == getobjectitem(s)) {
					foundobj = true;
					break;
				}
				i++;	
			}
			
			// if we didn't find it, raise an error
			if (!foundobj) {
				data.oerror.RaiseError(source, "Location " + Long.toString(getobjectitem(s)) + " does not exist.", programcounter, instruction);
				instruction++;
				return "";
			}
		}
		
	    // Check for custom properties. Interestingly, custom properties
	    // can override inbuilt properties.
	    int x = 0;
	    int y = 1;
	    String flg = "";
	    int flgno = 1;
	    
	    if (l != null) {
		    x = l.CustomProperties.indexOf(getobjectcall(s).toLowerCase());
		    if (x != -1) {
		        // Find the comma
		        x = l.CustomProperties.indexOf(",", x);
		        if (x != -1) {
		            // Find the pipe
		            y = l.CustomProperties.indexOf("|", x);
		            // If we don't have one, use the end of the string
		            if (y == -1) y = l.CustomProperties.length();
		            // Get the flag number
		            flg = l.CustomProperties.substring(x + 1, y);
		            // Do standard property code with flag instead
		            flgno = Integer.parseInt(flg);
		            if (isget) {rval = data.oflags[flgno].Value; } else {data.oflags[flgno].Value = propertyAssign();}
		            // Move to next instruction and return
		            instruction++;
					return rval;
		     	}
		 	}
		}
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
		
		// ------------- METHODS ---------------
		if (tcall.indexOf("setvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 boolean foundupdate = false;
												 String valname = tcall.substring(startname, endname);
												 
												 // Get the replacement value
												 instruction++;
												 instruction++;
												 String newval = performevaluate((String) codeline.get(instruction));
												 
												 l.setNameValue(valname, newval);
											}
											
		if (tcall.indexOf("getvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 String valname = tcall.substring(startname, endname);
												 
												 rval = l.getNameValue(valname);
											}
											
		if (tcall.equals("codeclear")) { // Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Test it
										 if (eventname.equalsIgnoreCase("oninput"))
										 	l.OnInput = new iagecode();	
										 if (eventname.equalsIgnoreCase("ondisplay"))
										 	l.OnDisplay = new iagecode();	
										}
										
 		if (tcall.equals("codeaddline")) {// Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Get the new code line
										 instruction++;
										 String newcode = (String) codeline.get(instruction);
										 
										 // Test it
										 if (eventname.equalsIgnoreCase("oninput"))
										 	l.OnInput.add(newcode);	
										 if (eventname.equalsIgnoreCase("ondisplay"))
										 	l.OnDisplay.add(newcode);	
										}
										
		if (tcall.equals("fireinlineevent")) { // Get the event name
										 instruction++;
										 String eventname = (String) codeline.get(instruction);
										 eventname = performevaluate(eventname);
										 
										 // Get the inline event to fire
										 instruction++;
										 String inevent = (String) codeline.get(instruction);
										 
										 // Test it and run code
										 if (eventname.equalsIgnoreCase("oninput")) {
										 	interpreter inlevent = new interpreter(thisplayer, thisparser);
	                						inlevent.runcode(l.OnInput, "Location(" + l.Name + ").OnInput", inevent, new String[4], 0, "InlineEvent." + inevent, 0, 0);
										 	}
										 if (eventname.equalsIgnoreCase("ondisplay")) {
										 	interpreter inlevent = new interpreter(thisplayer, thisparser);
	                						inlevent.runcode(l.OnDisplay, "Location(" + l.Name + ").OnDisplay", inevent, new String[4], 0, "InlineEvent." + inevent, 0, 0);
										 }
										}
		
										
		// ------------- PROPERTIES ------------
		if (tcall.equals("id")) {if (isget) {rval = Long.toString(l.ID); } else {l.ID = Long.parseLong(propertyAssign());}}									
		if (tcall.equals("name")) {if (isget) {rval = l.Name; } else {l.Name = propertyAssign();}}
		if (tcall.equals("imagepath")) {if (isget) {rval = l.ImagePath; } else {l.ImagePath = propertyAssign();}}
		if (tcall.equals("description")) {if (isget) {rval = l.Description; } else {l.Description = propertyAssign();}}
		if (tcall.equals("isdark")) {if (isget) {rval = data.booleanToString(l.IsDark); } else {l.IsDark = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("n")) {if (isget) {rval = Long.toString(l.N); } else {l.N = Long.parseLong(propertyAssign());}}
		if (tcall.equals("s")) {if (isget) {rval = Long.toString(l.S); } else {l.S = Long.parseLong(propertyAssign());}}
		if (tcall.equals("e")) {if (isget) {rval = Long.toString(l.E); } else {l.E = Long.parseLong(propertyAssign());}}
		if (tcall.equals("w")) {if (isget) {rval = Long.toString(l.W); } else {l.W = Long.parseLong(propertyAssign());}}
		if (tcall.equals("u")) {if (isget) {rval = Long.toString(l.U); } else {l.U = Long.parseLong(propertyAssign());}}
		if (tcall.equals("d")) {if (isget) {rval = Long.toString(l.D); } else {l.D = Long.parseLong(propertyAssign());}}
		if (tcall.equals("ne")) {if (isget) {rval = Long.toString(l.NE); } else {l.NE = Long.parseLong(propertyAssign());}}
		if (tcall.equals("nw")) {if (isget) {rval = Long.toString(l.NW); } else {l.NW = Long.parseLong(propertyAssign());}}
		if (tcall.equals("se")) {if (isget) {rval = Long.toString(l.SE); } else {l.SE = Long.parseLong(propertyAssign());}}
		if (tcall.equals("sw")) {if (isget) {rval = Long.toString(l.SW); } else {l.SW = Long.parseLong(propertyAssign());}}
		
		
		if (tcall.equals("count"))	{ rval = Integer.toString(data.olocations.getCount()); }
		
		instruction++;
		return rval;
	}
	
	/*** Players referenced by index */
	private String Obj_Player(String s) {

		int i = 1;
		player p = null;
		boolean foundobj = false;
		boolean executed = false;
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// find the object - not if it's a count request though - 
		// we don't need to find it for that
		if (!getobjectcall(s).equals("count")) {
			while (i <= data.oplayers.getCount()) {
				p = (player) data.oplayers.get(i);
				if (p.Index == getobjectitem(s)) {
					foundobj = true;
					break;
				}
				i++;	
			}
			
			// if we didn't find it, raise an error
			if (!foundobj) {
				data.oerror.RaiseError(source, "Player " + Long.toString(getobjectitem(s)) + " does not exist.", programcounter, instruction);
				instruction++;
				return "";
			}
		}
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
		rval = Obj_PlayerAction(tcall, p);
		
		instruction++;
		return rval;
	}

	private String Obj_PlayerArray(String s) {

		int i = 1;
		player p = null;
		boolean foundobj = false;
		boolean executed = false;
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// find the object - not if it's a count request though - 
		// we don't need to find it for that
		if (!getobjectcall(s).equals("count")) {
			p = (player) data.oplayers.get(Integer.parseInt(Long.toString(getobjectitem(s))));
			
			// if we didn't find it, raise an error
			if (p == null) {
				data.oerror.RaiseError(source, "Player " + Long.toString(getobjectitem(s)) + " does not exist.", programcounter, instruction);
				instruction++;
				return "";
			}
		}
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
		
		rval = Obj_PlayerAction(tcall, p);
		
		instruction++;
		return rval;
	}
	
	private String Obj_PlayerAction(String tcall, player p) {
		
		String rval = "";
		
		// ------------- METHODS -------------
		
		if (tcall.equals("quit")) {p.quit(false);}
		if (tcall.equals("showscore")) {p.showscore();}
		if (tcall.equals("showstatus")) {p.showstatus();}
		if (tcall.equals("print")) {vdu.Transmit(performevaluate((String)codeline.get(instruction+1)), p); instruction++;}
		if (tcall.equals("dropallitems")) {item.MoveAll(p.Index + location.PLAYERBASE, p.CurrentLocation);}
		if (tcall.equals("displaycurrentlocation")) {game.displaylocation(p);}
		if (tcall.indexOf("getfirstboolitem") != -1) {
							// Special method - has bracketed arg containing boolean type
							// Returns zero if no item found.
							 int startname = tcall.indexOf("(") + 1;
							 int endname = tcall.indexOf(")");
							 String boolname = tcall.substring(startname, endname);
							 // Enumerate objects player has
							 item im = null;
							 int i = 1;
							 while (i <= data.oitems.getCount()) {
							 	im = (item) data.oitems.get(i);
							 	if (im.CurrentLocation == (p.Index + location.PLAYERBASE)) {
							 		// Check if it has the bool we are looking for
							 		if (im.UserBooleans.indexOf(boolname) != -1) {
							 			// It does! - return it!
							 			return Long.toString(im.ID);
							 		}	
							 	}
							 	i++;
							 }
							 return "0";	
						}
		if (tcall.indexOf("iscarrying") != -1) {	
		 					 // Returns true or false if player is carrying
		 					 // item in brackets		
							 int startname = tcall.indexOf("(") + 1;
							 int endname = tcall.indexOf(")");
							 String itemref = tcall.substring(startname, endname);
							 // Enumerate objects player has
							 item im = null;
							 long itemid = Long.parseLong(performevaluate(itemref));
							 int i = 1;
							 while (i <= data.oitems.getCount()) {
							 	im = (item) data.oitems.get(i);
							 	if (im.ID == itemid && im.CurrentLocation == (thisplayer.Index + location.PLAYERBASE)) {
									// They are carrying it
									return "true";
							 	}
							 	i++;
							 }
							 return "false";
						}
						
		if (tcall.indexOf("savestate") != -1) {
							 // Saves the player's state:
							 
							 // identifier
							 instruction ++;
							 String identifier = performevaluate((String) codeline.get(instruction));
							 
							 // password
							 instruction ++;
							 String password = performevaluate((String) codeline.get(instruction));
							 
							 // Do the save
							 p.SaveState(password, identifier);
						}
		
		if (tcall.indexOf("restorestate") != -1) {
							  // Restores a saved state
							 
							 // identifier
							 instruction ++;
							 String identifier = performevaluate((String) codeline.get(instruction));
							  
							 // password
							 instruction ++;
							 String password = performevaluate((String) codeline.get(instruction));
							 
							 // Do the restore and set the returnvalue
							 this.returnvalue = data.booleanToString(p.RestoreState(password, identifier));
						}	
						
		if (tcall.indexOf("setvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 String valname = tcall.substring(startname, endname);
												 
												 // Get the replacement value
												 instruction++;
												 instruction++;
												 String newval = performevaluate((String) codeline.get(instruction));
												 
												 p.setNameValue(valname, newval);
												 
											}
											
		if (tcall.indexOf("getvalue") != -1) { // Find value name										   
												 int startname = tcall.indexOf("(") + 1;
												 int endname = tcall.indexOf(")");
												 String valname = tcall.substring(startname, endname);
												 rval = p.getNameValue(valname);
											}		
		if (tcall.indexOf("askparserquestion") != -1) { // Get replace noun
														instruction++;
														String newval = performevaluate((String) codeline.get(instruction));
														p.askParserQuestion(thisparser, Integer.parseInt(newval));
													}				
		
		
		// ------------- PROPERTIES ------------
		if (tcall.equals("index")) {if (isget) {rval = Integer.toString(p.Index); } else {p.Index = Integer.parseInt(propertyAssign());}}
		if (tcall.equals("name")) {if (isget) {rval = p.Name; } else {p.Name = propertyAssign();}}
		if (tcall.equals("displayname")) {if (isget) {rval = p.DisplayName; } else {p.DisplayName = propertyAssign();}}
		if (tcall.equals("score")) {if (isget) {rval = Long.toString(p.Score); } else {p.Score = Long.parseLong(propertyAssign());}}
		if (tcall.equals("turns")) {if (isget) {rval = Long.toString(p.Turns); } else {p.Turns = Long.parseLong(propertyAssign());}}
		if (tcall.equals("currentlocation")) {if (isget) {rval = Long.toString(p.CurrentLocation); } else {p.CurrentLocation = Long.parseLong(propertyAssign());}}
		if (tcall.equals("containerlocation")) {if (isget) {rval = Long.toString(location.PLAYERBASE + p.Index); }}
		if (tcall.equals("ipaddress")) {if (isget) {rval = p.IPAddress; } else {p.IPAddress = propertyAssign();}}
		if (tcall.equals("nounid")) {if (isget) {rval = Long.toString(p.Index + noun.PLAYERNOUNBASE);}}
		if (tcall.equals("cansee")) {if (isget) {rval = data.booleanToString(p.CanSee); } else {p.CanSee = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("textonly")) {if (isget) {rval = data.booleanToString(p.TextOnly); } else {p.TextOnly = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("outputtoplayer")) {if (isget) {rval = data.booleanToString(p.OutputToPlayer); } else {p.OutputToPlayer = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("state")) {if (isget) {rval = Byte.toString(p.State); } else {p.State = Byte.parseByte(propertyAssign());}}
		if (tcall.equals("stateitem")) {if (isget) {if (p.StateItem == null) {rval = "0";} else {rval = Long.toString(p.StateItem.ID);} } else {instruction++; instruction++; p.StateItem = item.getitemfromid(Long.parseLong(performevaluate((String)codeline.get(instruction))));}}
		if (tcall.equals("lastnoun")) {if (isget) {rval = Long.toString(p.LastNoun); } else {p.LastNoun = Long.parseLong(propertyAssign());}}
		if (tcall.equals("lastcommand")) {if (isget) {rval = p.LastCommand; } else {p.LastCommand = propertyAssign();}}
		if (tcall.equals("weightcarried")) {if (isget) { 
														// Find how much weight player is carrying
														int i = 1;
														long itemweight = 0;
														item im = null;
														while (i <= data.oitems.getCount()) {
															im = (item) data.oitems.get(i);
															if (im.CurrentLocation == (p.Index + location.PLAYERBASE)) {	
																itemweight = itemweight + im.Weight;
															}
															i++;
														}
														rval = Long.toString(itemweight); }};
		if (tcall.equals("itemscarried")) {if (isget) { 
														// Find how many items player is carrying
														int i = 1;
														int itemcount = 0;
														item im = null;
														while (i <= data.oitems.getCount()) {
															im = (item) data.oitems.get(i);
															if (im.CurrentLocation == (p.Index + location.PLAYERBASE)) {	
																itemcount++;
															}
															i++;
														}
														rval = Integer.toString(itemcount); }};
														
														
		if (tcall.equals("sizecarried")) {if (isget) { 
														// Find how much size player is carrying
														int i = 1;
														long itemsize = 0;
														item im = null;
														while (i <= data.oitems.getCount()) {
															im = (item) data.oitems.get(i);
															if (im.CurrentLocation == (p.Index + location.PLAYERBASE)) {	
																itemsize = itemsize + im.Size;
															}
															i++;
														}
														rval = Long.toString(itemsize); }};
														
		if (tcall.equals("verbosemode")) {if (isget) {rval = data.booleanToString(p.VerboseMode); } else {p.VerboseMode = data.stringToBoolean(propertyAssign());}}
		if (tcall.equals("hitpoints")) {if (isget) {rval = Long.toString(p.HitPoints); } else {p.HitPoints = Long.parseLong(propertyAssign());}}
		if (tcall.equals("damageindicator")) {if (isget) {rval = Long.toString(p.DamageIndicator); } else {p.DamageIndicator = Long.parseLong(propertyAssign());}}
		if (tcall.equals("chanceofhitting")) {if (isget) {rval = Long.toString(p.ChanceOfHitting); } else {p.ChanceOfHitting = Long.parseLong(propertyAssign());}}
		if (tcall.equals("money")) {if (isget) {rval = Long.toString(p.Money); } else {p.Money = Long.parseLong(propertyAssign());}}
		
		if (tcall.equals("cp1")) {if (isget) {rval = p.CP1; } else {p.CP1 = propertyAssign();}}
		if (tcall.equals("cp2")) {if (isget) {rval = p.CP2; } else {p.CP2 = propertyAssign();}}
		if (tcall.equals("cp3")) {if (isget) {rval = p.CP3; } else {p.CP3 = propertyAssign();}}
		if (tcall.equals("cp4")) {if (isget) {rval = p.CP4; } else {p.CP4 = propertyAssign();}}
		if (tcall.equals("cp5")) {if (isget) {rval = p.CP5; } else {p.CP5 = propertyAssign();}}
		if (tcall.equals("cp6")) {if (isget) {rval = p.CP6; } else {p.CP6 = propertyAssign();}}
		if (tcall.equals("cp7")) {if (isget) {rval = p.CP7; } else {p.CP7 = propertyAssign();}}
		if (tcall.equals("cp8")) {if (isget) {rval = p.CP8; } else {p.CP8 = propertyAssign();}}
		if (tcall.equals("cp9")) {if (isget) {rval = p.CP9; } else {p.CP9 = propertyAssign();}}
		if (tcall.equals("cp10")) {if (isget) {rval = p.CP10; } else {p.CP10 = propertyAssign();}}
		if (tcall.equals("cp11")) {if (isget) {rval = p.CP11; } else {p.CP11 = propertyAssign();}}
		if (tcall.equals("cp12")) {if (isget) {rval = p.CP12; } else {p.CP12 = propertyAssign();}}
		if (tcall.equals("cp13")) {if (isget) {rval = p.CP13; } else {p.CP13 = propertyAssign();}}
		if (tcall.equals("cp14")) {if (isget) {rval = p.CP14; } else {p.CP14 = propertyAssign();}}
		if (tcall.equals("cp15")) {if (isget) {rval = p.CP15; } else {p.CP15 = propertyAssign();}}
		if (tcall.equals("cp16")) {if (isget) {rval = p.CP16; } else {p.CP16 = propertyAssign();}}
		if (tcall.equals("cp17")) {if (isget) {rval = p.CP17; } else {p.CP17 = propertyAssign();}}
		if (tcall.equals("cp18")) {if (isget) {rval = p.CP18; } else {p.CP18 = propertyAssign();}}
		if (tcall.equals("cp19")) {if (isget) {rval = p.CP19; } else {p.CP19 = propertyAssign();}}
		if (tcall.equals("cp20")) {if (isget) {rval = p.CP20; } else {p.CP20 = propertyAssign();}}
		if (tcall.equals("cp21")) {if (isget) {rval = p.CP21; } else {p.CP21 = propertyAssign();}}
		if (tcall.equals("cp22")) {if (isget) {rval = p.CP22; } else {p.CP22 = propertyAssign();}}
		if (tcall.equals("cp23")) {if (isget) {rval = p.CP23; } else {p.CP23 = propertyAssign();}}
		if (tcall.equals("cp24")) {if (isget) {rval = p.CP24; } else {p.CP24 = propertyAssign();}}
		if (tcall.equals("cp25")) {if (isget) {rval = p.CP25; } else {p.CP25 = propertyAssign();}}
		if (tcall.equals("cp26")) {if (isget) {rval = p.CP26; } else {p.CP26 = propertyAssign();}}
		if (tcall.equals("cp27")) {if (isget) {rval = p.CP27; } else {p.CP27 = propertyAssign();}}
		if (tcall.equals("cp28")) {if (isget) {rval = p.CP28; } else {p.CP28 = propertyAssign();}}
		if (tcall.equals("cp29")) {if (isget) {rval = p.CP29; } else {p.CP29 = propertyAssign();}}
		if (tcall.equals("cp30")) {if (isget) {rval = p.CP30; } else {p.CP30 = propertyAssign();}}		
		
		if (tcall.equals("count"))	{ rval = Integer.toString(data.oplayers.getCount()); }
		
		return rval;

	}
	
	private String Obj_Internal(String s) {
		
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
	
		// Get our arguments for further on
		int noargs = codeline.getCount() - instruction;
		String[] args = new String[6];
		int i = 1;
		instruction++;
		while (i <= noargs) {
			args[i] = (String) codeline.get(instruction);
			instruction++;
			i++;	
		}
		// Some evaulate args3 as noun2 - it needs to be able to be parsed
		// as a long so if we don't have one, set it to zero.
		if (args[3] == null) {
			args[3] = "0";}
		else
		{
			if (args[3].equals("")) {
				args[3] = "0";
			}	
		}
		
		// Create a new parser to run these things on
		parsestring np = new parsestring("D,D", thisplayer, ",");
	
		if (tcall.equals("attackplayer")) {np.Noun = noun.PLAYERNOUNBASE + thisplayer.Index;
			np.Noun2 = np.getnounfromitemid(Long.parseLong(performevaluate(args[3]))); np.vAttack();}
		if (tcall.equals("attacknpc")) {np.Noun = np.getnounfromcharacterid(Long.parseLong(performevaluate(args[1])));
		}	np.Noun2 = np.getnounfromitemid(Long.parseLong(performevaluate(args[3]))); np.vAttack();
		if (tcall.equals("close")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vClose();}
		if (tcall.equals("drop")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vDrop();}
		if (tcall.equals("eat")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vEat();}
		if (tcall.equals("examine")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vExamine();}
		if (tcall.equals("extinguish")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vExtinguish();}
		if (tcall.equals("get")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vGet(true);}
		if (tcall.equals("getin")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vGetInside();}
		if (tcall.equals("getoutof")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vGetOut();}
		if (tcall.equals("go")) {
			String dr = args[1].trim();
			if (dr.equalsIgnoreCase("n")) {np.Adverb = constant.ADVB_NORTH; np.vGo(true);}
			if (dr.equalsIgnoreCase("s")) {np.Adverb = constant.ADVB_SOUTH; np.vGo(true);}
			if (dr.equalsIgnoreCase("e")) {np.Adverb = constant.ADVB_EAST; np.vGo(true);}
			if (dr.equalsIgnoreCase("w")) {np.Adverb = constant.ADVB_WEST; np.vGo(true);}
			if (dr.equalsIgnoreCase("u")) {np.Adverb = constant.ADVB_UP; np.vGo(true);}
			if (dr.equalsIgnoreCase("d")) {np.Adverb = constant.ADVB_DOWN; np.vGo(true);}
			if (dr.equalsIgnoreCase("ne")) {np.Adverb = constant.ADVB_NORTHEAST; np.vGo(true);}
			if (dr.equalsIgnoreCase("se")) {np.Adverb = constant.ADVB_SOUTHEAST; np.vGo(true);}
			if (dr.equalsIgnoreCase("sw")) {np.Adverb = constant.ADVB_SOUTHWEST; np.vGo(true);}
			if (dr.equalsIgnoreCase("nw")) {np.Adverb = constant.ADVB_NORTHWEST; np.vGo(true);}
			}
		if (tcall.equals("inventory")) {np.vInventory();}
		if (tcall.equals("lieon")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vLie();}
		if (tcall.equals("light")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vLight();}
		if (tcall.equals("open")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vOpen();}
		if (tcall.equals("put")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.Noun2 = np.getnounfromitemid(Long.parseLong(performevaluate(args[3])));np.Adverb = 13; np.vPut();}
		if (tcall.equals("read")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vRead();}
		if (tcall.equals("remove")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.Noun2 = np.getnounfromitemid(Long.parseLong(performevaluate(args[3])));np.vRemove();}
		if (tcall.equals("siton")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vSit();}
		if (tcall.equals("stand")) {np.vStand();}
		if (tcall.equals("unwear")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vUnwear();}
		if (tcall.equals("wear")) {np.Noun = np.getnounfromitemid(Long.parseLong(performevaluate(args[1]))); np.vWear();}
		
		return rval;
	}
	
	private String Obj_Game(String s) {
		
		String rval = "";
		
		// drop to lower case for matching
		s = s.toLowerCase();
		
		// Find appropriate property or method and call it:
		String tcall = getobjectcall(s);
		tcall = tcall.toLowerCase();
										
		// ------------- METHODS ------------
		if (tcall.equals("displaycurrentlocation")) {data.ogame.displaylocation(thisplayer);}
		if (tcall.equals("displayversion")) {data.ogame.displayversion(thisplayer);}
		if (tcall.equals("getinventory")) { int startname = tcall.indexOf("(") + 1;
											int endname = tcall.indexOf(")");
											String locidexpr = tcall.substring(startname, endname);
											instruction++;
											return game.getinventory(Long.parseLong(performevaluate(locidexpr)));
										  }
		
		instruction++;
		return rval;
		
	}
	
	/*** Sets the cancelled flag for other routines to pick up. Eg. if the routine
	   * is OnGet for something, calling cancelevent will stop the item being
	   * picked up.
	   */
	private void Cmd_CancelEvent() {
		cancelled = true;
		instruction = codeline.getCount() + 1;
	}
	
	/*** Terminates code execution */
	private void Cmd_End() {
		instruction = codeline.getCount() + 1;
		programcounter = thecode.getCount() + 1;		
	}
	
	private void Cmd_For() {
		
	}

	/*** Forces program execution to jump to line number or label */	
	private void Cmd_Goto() {
		
		String labeltogoto;
	    int i = 1;
	    
	    instruction++;
	    labeltogoto = (String) codeline.get(instruction);

		while (i <= thecode.getCount()) {

			if (checkexistence(":" + labeltogoto, (String) thecode.get(i))) {
	            programcounter = i - 1;
	            instruction = codeline.getCount() + 1;
	            return;
	     	}
	     	i++;
	 	}
	    
	    data.oerror.RaiseError(source, "Line label " + labeltogoto + " does not exist.", programcounter, instruction);
	}

	/**		 
		* If structures should follow certain rules:
		* 1. each evaluation should appear in brackets with a space either side.
		* 2. THEN is used as the terminator.
		* 3. Logical operators should appear outside the brackets.
		* 4. The if structure takes one line - it should break to the next
		*    immediately after the Then.
		*
		* eg. If ( Flags(24).Value = 1 ) Then
		*         Message(1).Show
		*     Else
		*         Message(2).Show
		*     EndIf
		* eg. If ( Flags(24).Value = 1 ) AND ( Flags(26).Value = 1 ) Then
     */	
	private void Cmd_If() {
	    
	    iagecollection conditions = new iagecollection();
	    String lastoperator = "";
	    boolean runningstatus = false;
	    boolean currentstatus = false;
	    int i;
	    
	    // Build our conditions list
	    while (!checkexistence("then", (String) codeline.get(instruction))) {
	    
	    	if (checkexistence("(", (String) codeline.get(instruction))) {
	            
	            instruction++;
	            iagecondition con = new iagecondition();
	            
	            con.Item1Value = performevaluate((String) codeline.get(instruction));
	            instruction++;
	            con.Operator = (String) codeline.get(instruction);
	            instruction++;
	            con.Item2Value = performevaluate((String) codeline.get(instruction));
	            instruction++;
	            con.LogicalOperatorFromPrevious = lastoperator;
	            
	            // Add the condition to the list
	            conditions.add(con);
	            
	            // Should be ")"
	            if (!checkexistence(")", (String) codeline.get(instruction))) {
	            	data.oerror.RaiseError(source, "Missing ')' in If.", programcounter, instruction);
	            	return;
	            }
	            instruction++;
	            
	            // This will either be our operator for the next item, or
	            // a then. If it is a then, break out of the loop because
	            // we are ready to test!.
	            if (checkexistence("then", (String) codeline.get(instruction))) {
	            	break;
	            }
	            else
	            {
	                lastoperator = (String) codeline.get(instruction);
	         	}
	     	}
	        instruction++;
	        if (instruction > codeline.getCount()) break;
	 	}
	    
	    // We should now have an evaluatable condition:
	    i = 1;
	    while (i <= conditions.getCount()) {
	    
	        currentstatus = false;
	        
	        // Get a true/false outcome for the expression
	        iagecondition curcondition = (iagecondition) conditions.get(i);
	        
	        if (curcondition.Operator.equals("=")) {currentstatus = (curcondition.Item1Value.equals(curcondition.Item2Value));}
	        if (curcondition.Operator.equals("<>")) {currentstatus = (!curcondition.Item1Value.equals(curcondition.Item2Value));}
	        if (curcondition.Operator.equals(">")) {currentstatus = (Long.parseLong(curcondition.Item1Value) > Long.parseLong(curcondition.Item2Value));}
	        if (curcondition.Operator.equals("<")) {currentstatus = (Long.parseLong(curcondition.Item1Value) < Long.parseLong(curcondition.Item2Value));}
	        if (curcondition.Operator.equals("like")) {currentstatus = (curcondition.Item1Value.equalsIgnoreCase(curcondition.Item2Value));}
	        
	        // Operate with this expression against running status
	        if (curcondition.LogicalOperatorFromPrevious.equals("and")) {runningstatus = runningstatus && currentstatus;}
	        if (curcondition.LogicalOperatorFromPrevious.equals("or")) {runningstatus = runningstatus || currentstatus;}
	        if (curcondition.LogicalOperatorFromPrevious.equals("")) {runningstatus = currentstatus;}
	        
	        i++;
	 	}
	    
	    // RunningStatus now holds whether our expression evaluated to true.
	    // If it did, we drop back from this routine, letting the instructions
	    // exhaust so the program counter gets incremented and the code
	    // following the if is executed.
	    if (runningstatus) {
	        instruction = codeline.getCount() + 1;
	        return;
	 	}
	 	else
	 	{
	 		// resume from the endif
			findendif();
	 	}
		
	}
	/*** Performs "if (input.verb = <calculate ID from text supplied>) then " */
	private void Cmd_IfVerb() {
		
		boolean foundmatch = false;
		
		String verbtest = (String) codeline.get(instruction);
		verbtest = verbtest.substring(1, verbtest.length());
		
		parsestring ps = new parsestring(verbtest, thisplayer, ",");
		
		int z = 1;
		while (z <= ps.vwords.getCount()) {
		
			// Find that verb to get the ID
			verb v = null;
			long verbid = 0;
			int i = 1;
			while (i <= data.overbs.getCount()) {
				v = (verb) data.overbs.get(i);	
				if (v.Text.equalsIgnoreCase((String) ps.vwords.get(z))) {
					verbid = v.ID;
					if (verbid == thisparser.Verb) {
						// We have a positive match!
						foundmatch = true;
						break;
					}
				}
				i++;
			}
			if (foundmatch) break; // Quit here if we already found a match
			z++;	
		}
		
		// Exhaust instructions because we've run them now.
		instruction = codeline.getCount() + 1;
		
		if (foundmatch) {
		
			// We have a match! Resume from next line
			return;
		}
		else
		{
			// Find the endif for this if and resume from there	
			findendif();
		}
		
	}
	
	/*** Performs "if (input.adverb = <calculate ID from text supplied>) then " */
	private void Cmd_IfAdverb() {
		
		boolean foundmatch = false;
		
		String adverbtest = (String) codeline.get(instruction);
		adverbtest = adverbtest.substring(1, adverbtest.length());
		
		parsestring ps = new parsestring(adverbtest, thisplayer, ",");
		
		int z = 1;
		while (z <= ps.vwords.getCount()) {
		
			// Find that adverb to get the ID
			adverb v = null;
			long adverbid = 0;
			int i = 1;
			while (i <= data.oadverbs.getCount()) {
				v = (adverb) data.oadverbs.get(i);	
				if (v.Text.equalsIgnoreCase((String) ps.vwords.get(z))) {
					adverbid = v.ID;
					if (adverbid == thisparser.Adverb) {
						// We have a positive match!
						foundmatch = true;
						break;
					}
				}
				i++;
			}
			if (foundmatch) break; // Quit here if we already found a match
			z++;	
		}
		
		// Exhaust instructions because we've run them now.
		instruction = codeline.getCount() + 1;
		
		if (foundmatch) {
		
			// We have a match! Resume from next line
			return;
		}
		else
		{
			// Find the endif for this if and resume from there	
			findendif();
		}
		
	}
	
	/*** Asks user a question */
	private String Cmd_Ask() {
		
		// Determine whether this is the second time this line has been
		// run (ie. Did we ask the player and this is the response?)
		if (thisplayer.WaitingForAskResponse) {
			// Stop waiting
			thisplayer.WaitingForAskResponse = false;
			thisplayer.AskProgramCounter = 1;
			thisplayer.AskCode = null;
			thisplayer.AskCodeSource = "";
			thisplayer.AskInterpreter = null;
			// Return the value out of this routine to continue code execution
			return thisplayer.UserResponseToAsk;
		}
		
		// Remember the position of this command
		thisplayer.AskProgramCounter = programcounter;
		
		// Remember the code for later
		thisplayer.AskCode = thecode;
		thisplayer.AskCodeSource = source;
		thisplayer.AskInterpreter = this;
		
		// Start waiting for a response
		thisplayer.WaitingForAskResponse = true;
		
		// Do player.print with the parameter following the ask
		instruction++;
		vdu.Transmit(performevaluate((String)codeline.get(instruction)), thisplayer);
		
		// Stop the code now
		Cmd_End();
		
		// Return nowt
		return "*";
		
	}
	
	private String Cmd_GetItemFromNoun() {
		
		// Assumes the next instruction contains an expression
		// evaluating to a noun. This finds the item with that
		// noun and returns it.
		
		// Get our noun id expression
		instruction++;
		String nid = performevaluate((String) codeline.get(instruction));
		long thenounid = Long.parseLong(nid);
		int i = 1;
		item im = null;
		while (i <= data.oitems.getCount()) {
			im = (item) data.oitems.get(i);
			if (im.NounID == thenounid) {
				// Found our matching item - return it.
				
				return Long.toString(im.ID);
			}
			i++;	
		}
		return "0";
	}
	
	private void Cmd_OutputContentsOf() {
		
		// Assumes the next two instructions contain an expression
		// evaluating to a playerindex and a locationid
		// This routine then shows that player the contents of that
		// location
		
		// Get our playerindex expression
		instruction++;
		String pdx = performevaluate((String) codeline.get(instruction));
		int playerindex = Integer.parseInt(pdx);
		
		// Get our locationid expression
		instruction++;
		String lid = performevaluate((String) codeline.get(instruction));
		long locationid = Long.parseLong(pdx);
		
		// Get an inventory of the location
		String outstring = game.getinventory(locationid);
		
		// Send it
		int i = 1;
		player theplayer = null;
		while (i <= data.oplayers.getCount()) {
			theplayer = (player) data.oplayers.get(i);
			if (theplayer.Index == playerindex) {
				vdu.Transmit(outstring, theplayer);
				return;
			}
			i++;	
		}
		return;
	}
	
	private void Cmd_OpenURL() {
		
	}
	
	private void Cmd_PlayMIDI() {
		
	}
	
	private void Cmd_PlaySound() {
		
	}
	
	private void Cmd_PlayMovie() {
		
	}

	/*** Tells client to display image */	
	private void Cmd_ShowPicture() {
		instruction++;
		String s = performevaluate((String) codeline.get(instruction));
		vdu.Transmit("<img src=" + data.ogame.MediaBase + s + ">", thisplayer);
	}
	
	private void Cmd_StopMIDI() {
		
	}
	
	private void Cmd_StopMovie() {
		
	}
	
	/*** Used to create variables and assign initial values */
	private void Cmd_Var() {
	
	    instruction++;
	    
	    if (instruction > codeline.getCount()) {
	    	data.oerror.RaiseError(source, "Var without variable name.", programcounter, instruction);
	     	return;
	     }
	    
	    String varname = (String) codeline.get(instruction);
	    
	    // Check to see if a variable with this name already exists - if
	    // it does, destroy it so it can be reinitialised safely:
	    int i = 1;
	    String testval = "";
	    while (i <= localvariables.getCount()) {
	    	testval = (String) localvariables.get(i);
	    	if (testval.equalsIgnoreCase(varname)) {
	    		localvariables.remove(i);
	    		localvariablevalues.remove(i);
	    		break;
	    	}
	    	i++;	
	    }
	    
	    // Create variable without a value.
	    localvariables.add(varname);
	    
	    // If the next item isn't an equals or we run out of words then terminate code now
	    // and leave the variable blank
	    instruction++;
	    if (instruction > codeline.getCount()) {localvariablevalues.add(new String("")); return;}
	    	    
	    // Get value of var and store it
	    instruction++;	    
	    localvariablevalues.add(performevaluate((String) codeline.get(instruction)));
	       
	    // Exhaust instructions
	    instruction = codeline.getCount() + 1;
	}
	
	/*** Deals with while commands. Syntax is "While <if statement> then",
	   * using "endwhile" as terminator. Whiles can be nested as with if
	   * statements.*/
	private void Cmd_While() {
		
	    iagecollection conditions = new iagecollection();
	    String lastoperator = "";
	    boolean runningstatus = false;
	    boolean currentstatus = false;
	    int i;
	    
	    // Build our conditions list
	    while (!checkexistence("then", (String) codeline.get(instruction))) {
	    
	    	if (checkexistence("(", (String) codeline.get(instruction))) {
	            
	            instruction++;
	            iagecondition con = new iagecondition();
	            
	            con.Item1Value = performevaluate((String) codeline.get(instruction));
	            instruction++;
	            con.Operator = (String) codeline.get(instruction);
	            instruction++;
	            con.Item2Value = performevaluate((String) codeline.get(instruction));
	            instruction++;
	            con.LogicalOperatorFromPrevious = lastoperator;
	            
	            // Add the condition to the list
	            conditions.add(con);
	            
	            // Should be ")"
	            if (!checkexistence(")", (String) codeline.get(instruction))) {
	            	data.oerror.RaiseError(source, "Missing ')' in While.", programcounter, instruction);
	            	return;
	            }
	            instruction++;
	            
	            // This will either be our operator for the next item, or
	            // a then. If it is a then, break out of the loop because
	            // we are ready to test!.
	            if (checkexistence("then", (String) codeline.get(instruction))) {
	            	break;
	            }
	            else
	            {
	                lastoperator = (String) codeline.get(instruction);
	         	}
	     	}
	        instruction++;
	        if (instruction > codeline.getCount()) break;
	 	}
	    
	    // We should now have an evaluatable condition:
	    i = 1;
	    while (i <= conditions.getCount()) {
	    
	        currentstatus = false;
	        
	        // Get a true/false outcome for the expression
	        iagecondition curcondition = (iagecondition) conditions.get(i);
	        
	        if (curcondition.Operator.equals("=")) {currentstatus = (curcondition.Item1Value.equals(curcondition.Item2Value));}
	        if (curcondition.Operator.equals("<>")) {currentstatus = (!curcondition.Item1Value.equals(curcondition.Item2Value));}
	        if (curcondition.Operator.equals(">")) {currentstatus = (Long.parseLong(curcondition.Item1Value) > Long.parseLong(curcondition.Item2Value));}
	        if (curcondition.Operator.equals("<")) {currentstatus = (Long.parseLong(curcondition.Item1Value) < Long.parseLong(curcondition.Item2Value));}
	        if (curcondition.Operator.equals("like")) {currentstatus = (curcondition.Item1Value.equalsIgnoreCase(curcondition.Item2Value));}
	        
	        // Operate with this expression against running status
	        if (curcondition.LogicalOperatorFromPrevious.equals("and")) {runningstatus = runningstatus && currentstatus;}
	        if (curcondition.LogicalOperatorFromPrevious.equals("or")) {runningstatus = runningstatus || currentstatus;}
	        if (curcondition.LogicalOperatorFromPrevious.equals("")) {runningstatus = currentstatus;}
	        
	        i++;
	 	}
	    
	    // RunningStatus now holds whether our expression evaluated to true.
	    // If it did, we drop back from this routine, letting the instructions
	    // exhaust so the program counter gets incremented and the code
	    // following the while is executed.
	    if (runningstatus) {
	        instruction = codeline.getCount() + 1;
	        return;
	 	}
	 	else
	 	{
	 		// resume from the endwhile
			findendwhile();
	 	}
		
	}
	
	/** Deals with getting/setting the returnvalue property which all routines have */
	private String Prop_ReturnValue() {
		String rval = "";
		if (isget) {
				rval = returnvalue;
		}
		else
	    {
	        returnvalue = propertyAssign();
	    }
	    instruction++;
	    return rval;
	}
	
	/**
     *  Evaluates IAGEscript expressions and returns the result
     *  as a string. 
     *
     *  This routine has to do some major trickery
     *  when performing maths due to the strong type-casting
     *  required by Java.
     *
     *  If an expression item evaluates to a number, it is taken
     *  to be a float during evaluation.
     *  Boolean true/false values are left as strings and not 
     *  examined in any unique form, so you should never perform
     *  expressions containing anything where you rely on true and false
     *  being anything but strings.
     */
	private String performevaluate(String s) {

	    iagecollection ei = null;
	    int i = 1;
	    int z = 1;
	    String CV = "";
	    String LV = "";
	    String Op = "";
	    	
	    // Any object properties encountered will be to get in an expression
	    isget = true;
	
	    // Given an expression, it will return an intrinsic value if there is one
	    
	    // Seperate items by expression seperators (_)
	    ei = new parsestring(s, thisplayer, "_").vwords;
	    
	    while (i <= ei.getCount()) {
	    
	        CV = (String) ei.get(i);
	    
	        // Check for object references - note here that if we find one
	        // we subtract an instruction. This is because these routines
	        // increment the instruction counter and in an evaluate we
	        // do not want this to happen because we don't know what called
	        // this.
	        if (checkstart("message(", CV)) {CV = Obj_Message(CV); instruction--;}
	        if (checkstart("flag(", CV)) {CV = Obj_Flag(CV); instruction--;}
	        if (checkstart("game.", CV)) {CV = Obj_Game(CV); instruction--;}
	        if (checkstart("currentplayer.", CV)) {CV = Obj_CurrentPlayer(CV); instruction--;}
	        if (checkstart("playerarray(", CV)) {CV = Obj_PlayerArray(CV); instruction--;}
	        if (checkstart("player(", CV)) {CV = Obj_Player(CV); instruction--;}
	        if (checkstart("location(", CV)) {CV = Obj_Location(CV); instruction--;}
	        if (checkstart("getitemfromnoun", CV)) {CV = Cmd_GetItemFromNoun();}
	        if (checkstart("item(", CV)) {CV = Obj_Item(CV); instruction--;}
	        if (checkstart("character(", CV)) {CV = Obj_Character(CV); instruction--;}
	        if (checkstart("input.", CV)) {CV = Obj_Input(CV); instruction--;}
	        if (checkstart("returnvalue", CV)) {CV = Prop_ReturnValue(); instruction--;}
	        if (checkstart("array.", CV)) {CV = Obj_Array(CV); instruction--;}
	        
	        // Ask is a bit special - if it returns a * we should abandon all code
	        // and not attempt to do anything, because we want the user's input before
	        // continuing.
	        if (checkstart("ask", CV)) {CV = Cmd_Ask(); if (CV.equals("*")) return ""; else instruction--;}
	        
	        // Internal functions
	        if (checkstart("random", CV)) {CV = Double.toString(Math.random());}
	        if (checkstart("makeinteger", CV)) {CV = "0"; LV = LV.substring(0, LV.indexOf("."));}
	        if (checkstart("space", CV)) {CV = " ";}
	        
	        // String functions
	        
	        // Instr(Str>>Str) - returns a 1 if string 2 is present
	        // in string 1. 0 for no match.
	        if (checkstart("instr(", CV)) {
	        	// Get both strings
	        	int startname = CV.indexOf("(") + 1;
				int endname = CV.indexOf(")");
				String totalexp = CV.substring(startname, endname);
				// Now split them on the >> chars which separates them
				int bpoint = totalexp.indexOf(">>");
				String exp1 = totalexp.substring(0, bpoint);
				String exp2 = totalexp.substring(bpoint + 2, totalexp.length());
				// evaluate them
				exp1 = performevaluate(exp1);
				exp2 = performevaluate(exp2);
				// Work out what to return
				if (exp1.indexOf(exp2) != -1) 
					CV = "1";
				else
					CV = "0";			
	     	}
	     
	        // Local Variables
	        z = 1;
	        while (z <= localvariables.getCount()) {
	        	if (checkstart((String) localvariables.get(z), (String) ei.get(i))) {
	        		CV = (String) localvariablevalues.get(z);
	                break;
	         	}
	         	z++;
	     	}
	    
	        // If we have an operator from the last instruction, then apply the
	        // the items together. Because we move up to double precision, we check
	        // afterwards if the result is a whole number (ie. Ends with .0) - if it
	        // is, we throw away the decimal point and what's after it.
	        if (!Op.equals("")) {
	        	if (Op.equals("+")) {LV = Double.toString(Double.valueOf(LV).doubleValue() + Double.valueOf(CV).doubleValue()); if (LV.endsWith(".0")) LV = LV.substring(0, LV.length()-2);}
	        	if (Op.equals("-")) {LV = Double.toString(Double.valueOf(LV).doubleValue() - Double.valueOf(CV).doubleValue()); if (LV.endsWith(".0")) LV = LV.substring(0, LV.length()-2);}
	        	if (Op.equals("*")) {LV = Double.toString(Double.valueOf(LV).doubleValue() * Double.valueOf(CV).doubleValue()); if (LV.endsWith(".0")) LV = LV.substring(0, LV.length()-2);}
	        	if (Op.equals("/")) {LV = Double.toString(Double.valueOf(LV).doubleValue() / Double.valueOf(CV).doubleValue()); if (LV.endsWith(".0")) LV = LV.substring(0, LV.length()-2);}
	        	if (Op.equals("&")) {LV += CV;}
	     	}
	    
	        // Check the operator (if there is one)
	        if (i < ei.getCount()) {
	            i++;
	            Op = (String) ei.get(i);
	            if (LV.equals("")) {LV = CV;}
	     	}    
	        i++;
	 	}
	    
	    // If we have no cumulative value, set the last one as result
	    if (LV.equals("")) {LV = CV;}
	    
	    // If there are any ^ characters present, then the result is a string
	    // and we need to convert those ^ characters to spaces.
	    LV = LV.replace('^', ' ');
	    
	    return LV;

	}
	
	/*** Locates the logical endif for a given if statement */
	private void findendif() {
		
	    // If it did not evaluate to true, we look for an else or an endif (whichever comes first)
	    // and resume the code from the line after that point.
	    // We have to do this a bit intelligently, so if we find another IF
	    // statement after this one, we add up a tally and use this to determine
	    // if we have found the right else or endif for this particular IF statement.
	    int tally = 0;
	    programcounter++; // Start at next line
	    while (programcounter <= thecode.getCount()) {
	    	if (checkstart("else", thecode.get(programcounter))) { if (tally == 0) break; }
	    	if (checkstart("endif", thecode.get(programcounter))) { if (tally == 0) break; else tally--; }
	    	if (checkstart("if ", thecode.get(programcounter))) {tally++; }
	    	if (checkstart(";", thecode.get(programcounter))) {tally++; }
	    	if (checkstart("#", thecode.get(programcounter))) {tally++; }
	    	programcounter++;
	    }
	    	
	    if (programcounter > thecode.getCount()) {
	    	data.oerror.RaiseError(source, "if without endif", programcounter, instruction);
	    	return;
	 	}
	    // Exhaust the instructions for this line.
	    instruction = codeline.getCount() + 1;
	    // Drop out so we carry on from the next bit of program.
		
	}
	
	/*** Locates the logical endwhile for a given while statement */
	private void findendwhile() {
		
	    // If it did not evaluate to true, we look for an endwhile 
	    // and resume the code from the line after that point.
	    // We have to do this a bit intelligently, so if we find another while
	    // statement after this one, we add up a tally and use this to determine
	    // if we have found the right endwhile for this particular while statement.
	    int tally = 0;
	    programcounter++; // Start at next line
	    while (programcounter <= thecode.getCount()) {
	    	if (checkstart("endwhile", thecode.get(programcounter))) { if (tally == 0) break; else tally--; }
	    	if (checkstart("while ", thecode.get(programcounter))) {tally++; }
	    	programcounter++;
	    }
	    	
	    if (programcounter > thecode.getCount()) {
	    	data.oerror.RaiseError(source, "while without endwhile", programcounter, instruction);
	    	return;
	 	}
	    // Exhaust the instructions for this line.
	    instruction = codeline.getCount() + 1;
	    // Drop out so we carry on from the next bit of program.
		
	}
	
	/*** Locates the logical while for a given endwhile statement */
	private void findwhile() {
		
	    int tally = 0;
	    programcounter--; // Start at previous line
	    while (programcounter > 0) {
	    	if (checkstart("endwhile", thecode.get(programcounter))) { tally++; }
	    	if (checkstart("while ", thecode.get(programcounter))) {if (tally == 0) break; else tally--; }
	    	programcounter--;
	    }
	    
	    if (programcounter == 0) {
	    	data.oerror.RaiseError(source, "Endwhile without While", programcounter, instruction);
	    }

	    // First instruction
	    instruction = 1;
	    // Grab the codeline
	    String curcodeline = (String) thecode.get(programcounter);
	    curcodeline = data.removeleadingspaces(curcodeline);
		parsestring p = new parsestring(curcodeline, thisplayer, " ");
		codeline = p.vwords;
	    // Drop out so we execute the while
		
	}
	
	private void Cmd_Call() {
		
		//Found our procedure name
		instruction++;
		String procname = (String) codeline.get(instruction);
		
		//Find out how many arguments there were
		int argno = (codeline.getCount() - instruction);
		
		// Make storage for them
		String[] args = new String[argno + 1];
			
		// Grab them and evaluate them
		int i = 1;
		instruction++;
		while (instruction <= codeline.getCount()) {
			args[i] = performevaluate((String)codeline.get(instruction));
			i++;
			instruction++;	
		}
		
		// Retrieve the separate module and procedure name
		int sepchar = procname.indexOf(".");
		if (sepchar == -1) {
			// Invalid proc call - no qualifier
			data.oerror.RaiseError(source, "Invalid procedure call - " + procname, programcounter, instruction);
			instruction = codeline.getCount() + 1;
			return;
		}
		
		String modname = procname.substring(0, sepchar);
		String proc = procname.substring(sepchar + 1, procname.length());
		
		// Create a new interpreter and pass it this parser and
		// the correct code module
		interpreter runproc = new interpreter(thisplayer, thisparser);
		
		// Start it up with the procedure name and args
		i = 1;
		codemodule mod = null;
		while (i <= data.omodules.getCount()) {
			mod = (codemodule) data.omodules.get(i);
			if (mod.Name.trim().equalsIgnoreCase(modname.trim())) {
				runproc.runcode(mod.Code, modname, proc, args, argno, source, instruction, programcounter);
				// Bubble back returnvalues and cancelled
				this.cancelled = runproc.cancelled;
				this.returnvalue = runproc.returnvalue;
				return;
			}
			i++;
		}
		
	}
	
}