//#define USE_DEBUG_CONSOLE
/*
	HEWIN.C

	Win32 front-end for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	This file contains Windows API interfacing; Hugo Engine interfacing
	is performed by hemsvc.c.
*/

#define HE_PROGRAM_NAME		_T("hewin.exe")

#define TYPED_STORY_COMMANDS

#ifdef UNDER_CE
#define GetProp WinGetProp
#endif
#include <windows.h>
#undef GetProp

#ifdef UNDER_CE
#include <commdlg.h>
#endif

#ifdef USE_DEBUG_CONSOLE
#include <fcntl.h>
#include <io.h>
#endif

#include <shlobj.h>	// for folder-browse dialog

#include "heheader.h"
#define APSTUDIO_INVOKED
#ifdef UNDER_CE
#include "ceres.h"
#else
#include "hewin.h"
#endif
#include "hewin32.h"
#include "hemenu.h"
#ifdef SPEECH_ENABLED
#include "hetalk.h"
#endif
#ifdef UNDER_CE
#include "hewince.h"
#endif
#ifndef NO_SOUND
#include "mikmod.h"
#endif

#if defined (DEBUGGER)
#undef APSTUDIO_INVOKED
#include "hdwin.h"
#endif


/* Specific to hewin.c/hemsvc.c: */
void TypeCommand(char *cmd, char clear, char linefeed);
void CenterWindow(HWND, HWND);
void UpdateClient(int visible_update);
void ClipVisibleClient(void);
void CheckResizeShift(void);
void MakeFullScreen(int full);
void SetClientDimensions(void);
void MatchFont(int flags);
void ResetCharDimensions(int f);
void SelectNewFont(int which_font);
void SelectDefaultColor(int which_color);
void CheckMenuItems(void);
void NewEngineInstance(void);
void ShowCompassRose(void);
void LoadRegistryDefaults(void);
void SaveRegistryDefaults(void);
void SetCurrentDirectories(void);

/* For Windows: */
LRESULT CALLBACK WndProc (HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);
void CleanupAndQuit(void);
BOOL CALLBACK FrontEndDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK AboutDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
UINT APIENTRY CenterHookProc(HWND hwndDlg, UINT uiMsg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK SelectColorDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK wndCompassProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK ScrollbackDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK SetupDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);

#if defined (DEBUGGER)
void *AllocMemory(size_t size);
#endif

#if defined (FRONT_END)
int he_main(int argc, char **argv);     /* from he.c */
#endif


int prop_charwidth, prop_lineheight;
int fixed_charset, prop_charset;

/* From herun.c: */
extern int physical_lowest_windowbottom, lowest_windowbottom;

/* From mikmod.h */
extern int forbid;

/* Windows API management info: */
HINSTANCE AppInstance;
HWND wndMain;
#ifndef NO_COMPASS
HWND wndCompassBmp;
#endif
// Keeping dcMain global causes video playback problems
HDC dcMem;
HBITMAP bmpMem;
HFONT fontDefault = NULL, fontCurrent = NULL;
LOGFONT lfCurrent;
HMENU menuMain;
HACCEL accelMain;

/* For handling updating: */
char client_needs_updating = false;
char post_picture_update = false;
char client_painted = false;
int scroll_offset = 0, scroll_offset_fill = 0;
int caret_height = 0;

/* For handling resizing */
BOOL client_is_minimized = false, override_resize_handling = false;
BOOL client_is_resizing = false;

/* For the compass rose: */
#ifndef NO_COMPASS
HWND wndCompass = NULL;
HBITMAP bmpEmbossed, bmpSolid;
HDC dcEmbossed, dcSolid;
#endif

int client_width, client_height;
char disable_graphics = false, disable_sounds = false;
char waiting_for_key = false;
char processed_accelerator_key = false;
char show_compass_rose = false;

/* Registry/initialization information:
   (Unless loaded otherwise from the registry, these are the defaults)
*/
#define DEFAULT_XPOS 50
#define DEFAULT_YPOS 20

#ifndef UNDER_CE
int reg_xPos = DEFAULT_XPOS, reg_yPos = DEFAULT_YPOS;
int reg_Width = 648;	/* ~80 character display w/default fixed font */
int reg_Height = 480;
int reg_AutoTranslateLatin1 = 1;
int reg_DisplayGraphics = 1, reg_PlaySounds = 1;
#define DEF_FIXED_FONT_SIZE 10
#define DEF_PROP_FONT_SIZE 11

#else	// UNDER_CE
int reg_xPos = CW_USEDEFAULT, reg_yPos = CW_USEDEFAULT;
int reg_Width = CW_USEDEFAULT, reg_Height = CW_USEDEFAULT;
int reg_AutoTranslateLatin1 = 0;
int reg_DisplayGraphics = 0, reg_PlaySounds = 0;
#define DEF_FIXED_FONT_SIZE 8
#define DEF_PROP_FONT_SIZE 10
#endif	// UNDER_CE

#define DEF_FIXED_FONT_FACE _T("Terminal")
#define DEF_PROP_FONT_FACE _T("Arial")
TCHAR reg_FixedFont[LF_FACESIZE] = DEF_FIXED_FONT_FACE,
	reg_PropFont[LF_FACESIZE] = DEF_PROP_FONT_FACE;
int reg_FixedPoint = DEF_FIXED_FONT_SIZE,
	reg_PropPoint = DEF_PROP_FONT_SIZE;
int reg_FixedFlags = 0,
	reg_PropFlags = PROP_FONT;
int reg_FixedCharset = OEM_CHARSET,
	reg_PropCharset = ANSI_CHARSET;
int reg_DefForeground, reg_DefBackground,
	reg_DefSLForeground, reg_DefSLBackground;
int reg_Maximized = 1;
int reg_FullScreen = 0, reg_FastScrolling = 0;
int reg_ShowCompass = 0;
int reg_CompassX = DEFAULT_XPOS+20, reg_CompassY = DEFAULT_XPOS+20;
int reg_GraphicsSmoothing = 0;
COLORREF reg_CustomColors[16];

/* Directories: */
TCHAR reg_GameDir[MAXPATH] = _T("");
TCHAR reg_SaveDir[MAXPATH] = _T("");
TCHAR current_games_dir[MAXPATH];
TCHAR current_save_dir[MAXPATH];

/* Other variables: */
TCHAR executable_module[MAXPATH];	/* i.e., path of hewin.exe */
TCHAR filename[MAXFILENAME];
TCHAR *invocationpath;

/* Most Recently Used file list: */
#define MRU_FILE_COUNT 4
TCHAR MRU_filename[MRU_FILE_COUNT][MAXPATH];

#define SCROLLBACK_SIZE 32768
TCHAR scrollback_buffer[SCROLLBACK_SIZE];
int scrollback_pos = 0;
char scrollback_active = false;

#ifdef UNICODE
WCHAR unicode_buffer[MAXBUFFER];
#endif

#if defined (DEBUGGER)
extern HWND wndDebug;		// debugger window handle
#endif


/*
 * Hugo Engine initialization:
 *
 */

/* WinMain

	The Windows entry point, looking after initializing the
	application, setting up windows, device contexts, and the bitmap
	used for buffering output.

	All output is first done to the bitmap (bmpMem) in the memory
	device context (dcMem); when an update is to be done, everything
	is copied to the main window's device context (dcMain).
*/

int WINAPI 
#if defined (DEBUGGER)
he_WinMain
#else
WinMain
#endif
(HINSTANCE hInstance, HINSTANCE hPrevInstance,
#ifdef UNDER_CE
		LPWSTR lpCmdLine,
#else
		char *szCmdLine,
#endif
		int iCmdShow)
{
	static TCHAR szAppName[MAXPATH + 32];
	WNDCLASS wndclass;
	LPCTSTR szClassName = _T("HugoEngineClass");
	int argc = 2;
	char **argv, *arg2, inquote = 0;
	HFONT fontOld;
	HDC dcMain;
	int i;
#ifdef SINGLE_LAUNCH
	HANDLE hPrevInstMutex;
#endif
#if defined (USE_DEBUG_CONSOLE) && defined (_DEBUG)
	int hCrt;
	FILE *hf;
#endif

#ifdef SINGLE_LAUNCH
	/* Check to see if another instance is already running;
	   can't use hPrevInstance because it's always null for 
	   Win32 apps
	*/
	hPrevInstMutex = CreateMutex(NULL, FALSE, _T("HugoEngineUniqueMutex"));
	if (GetLastError()!=ERROR_SUCCESS)
	{
		HWND oldWnd = FindWindow(szClassName, NULL);
		if (oldWnd==NULL)
		{
#ifdef DEBUGGER
			oldWnd = FindWindow(NULL, _T("Hugo Debugger"));
#else
			oldWnd = FindWindow(NULL, _T("Hugo Engine"));
#endif
		}
		if (oldWnd)
		{
			SetForegroundWindow(oldWnd);
			SetFocus(oldWnd);
		}
		exit(0);
	}
#endif

#if defined (USE_DEBUG_CONSOLE) && defined (_DEBUG)
	AllocConsole();
	hCrt = _open_osfhandle(
		(long) GetStdHandle(STD_OUTPUT_HANDLE),
		_O_TEXT );
	hf = _fdopen( hCrt, "w" );
	*stdout = *hf;
	setvbuf( stdout, NULL, _IONBF, 0 ); 
#endif
   
   	/* Get the path and filename of this, the executable
	   module (hewin.exe)
	*/
	GetModuleFileName(GetModuleHandle(HE_PROGRAM_NAME),
       		executable_module, MAXPATH);

	/* Set up the default colors--may be overwritten
	   by LoadRegistryDefaults immediately following
	*/
	reg_DefForeground = hugo_color(DEF_FCOLOR);
	reg_DefBackground = hugo_color(DEF_BGCOLOR);
	reg_DefSLForeground = hugo_color(DEF_SLFCOLOR);
	reg_DefSLBackground = hugo_color(DEF_SLBGCOLOR);

	/* Get global application info */
	LoadRegistryDefaults();		/* i.e., reg_... variables */
	AppInstance = hInstance;

	/* Init COM */
#if !defined (NO_SOUND) && !defined (USE_NT_SOUND_ONLY)
	CoInitialize(NULL);
#endif

	/* Get invocation path and make up ANSI C's argc and argv in
	   rather caveman-esque fashion
	*/
 	invocationpath = GetCommandLine();
	/* Trim trailing spaces */
	while (invocationpath[_tcslen(invocationpath)-1]==' ')
		invocationpath[_tcslen(invocationpath)-1] = '\0';
	argv = malloc(2*sizeof(char **));

#ifdef UNICODE
	argv[0] = (char *)malloc(_tcslen(executable_module)*sizeof(char));
	strcpy_UtoA(argv[0], executable_module);
#else
	argv[0] = executable_module;
#endif

	/* If there is no second argument, i.e., no supplied file to run,
	   open a dialogbox to prompt for one
	*/
#ifdef UNICODE
	arg2 = (char *)malloc(_tcslen(invocationpath)*sizeof(char));
	strcpy_UtoA(arg2, invocationpath);
#else
	arg2 = invocationpath;
#endif

#ifdef UNDER_CE
	/* Have to do this nonsense because of the different way Pocket PC
	   2000 and 2002 seem to set up the arguments
	*/
	if (_tcsstr(invocationpath, executable_module))
	{
		strcpy_UtoA(arg2, _tcsstr(invocationpath, executable_module)+_tcslen(executable_module));
		if (*arg2=='\"') arg2++;
	}
#endif

NextArg:

#ifndef UNDER_CE
	// Pocket PC 2002 (at least) doesn't quote arguments with spaces
	while (*arg2!='\0' && (*arg2!=' ' || inquote))
	{
		if (*arg2=='\"') inquote = !inquote;
		arg2++;
	}
#endif

	/* If there's no second argument */
	if (*arg2=='\0')
	{
		/* Open "Filename to run" dialog */
#if defined (DEBUGGER)
		if (DialogBox(hInstance, MAKEINTRESOURCE(HE_FRONTEND_DIALOG),
			wndDebug, FrontEndDialog)!=TRUE)
#else
		if (DialogBox(hInstance, MAKEINTRESOURCE(HE_FRONTEND_DIALOG),
			NULL, FrontEndDialog)!=TRUE)
#endif
				exit(0);
#ifdef UNICODE
		argv[1] = (char *)malloc(_tcslen(filename)*sizeof(char));
		strcpy_UtoA(argv[1], filename);
#else
		argv[1] = filename;
#endif
	}
	/* If there's an argument giving switches */
	else if (*(++arg2)=='-')
	{
NextSwitch:
		if (*(arg2+1)=='g') disable_graphics = true;
		if (*(arg2+1)=='s') disable_sounds = true;

		if (*(arg2+1)==' ') goto NextArg;
		arg2++;
		goto NextSwitch;
	}
	/* If not, it must be the filename */
	else
	{
		argv[1] = arg2;
	}

	while (*argv[1]==' ') *argv[1]++;
	if (*argv[1]=='\"') *argv[1]++, argv[1][strlen(argv[1])-1]='\0';
	argv[1][0] = toupper(argv[1][0]);

#if defined (DEBUGGER)
	sprintf(szAppName, _T("Hugo Debugger - Output"));
#else
#ifdef UNICODE
#ifndef UNDER_CE
	strcpy_AtoU(unicode_buffer, argv[1]);
	_stprintf(szAppName, _T("Hugo Engine - \"%s\""), unicode_buffer);
#else
	_tcscpy(szAppName, _T("Hugo Engine"));
#endif	// UNDER_CE
#else
	sprintf(szAppName, "Hugo Engine - \"%s\"", argv[1]);
#endif	// !UNICODE
#endif	// !DEBUGGER

	/* To prevent division-by-zero violations starting up: */
	FIXEDCHARWIDTH = FIXEDLINEHEIGHT = charwidth = lineheight = 1;

	/* The application window class */
#ifdef USE_TEXTBUFFER
	wndclass.style          = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
#else
	wndclass.style          = CS_HREDRAW | CS_VREDRAW;
#endif
	wndclass.lpfnWndProc    = WndProc;
	wndclass.cbClsExtra     = 0;
	wndclass.cbWndExtra     = 0;
	wndclass.hInstance      = hInstance;
	wndclass.hIcon          = LoadIcon(hInstance, MAKEINTRESOURCE(PROGRAM_ICON));
	wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground  = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wndclass.lpszMenuName   = NULL;
	wndclass.lpszClassName  = szClassName;
	RegisterClass(&wndclass);

	wndMain = CreateWindowEx(reg_Maximized?0:WS_EX_CLIENTEDGE,
				szClassName,            // window class name
				szAppName,              // window caption
#ifdef UNDER_CE
				WS_VISIBLE |		// CE window style
					0,// WS_SYSMENU,
#else
				WS_OVERLAPPEDWINDOW |	// regular window style
					WS_CAPTION,
#endif
				reg_xPos,		// initial x position
				reg_yPos,		// initial y position
				reg_Width,
				reg_Height,
				NULL,                   // parent window handle
				NULL,                   // window menu handle
				hInstance,              // program instance handle
				NULL);                  // creation parameters

	/* Set up main menubar and accelerator keys */
#ifdef UNDER_CE
	SetupCEWindow(hInstance, wndMain);
#else
	menuMain = LoadMenu(hInstance, MAKEINTRESOURCE(HEWIN_MENUBAR));
	SetMenu(wndMain, menuMain);
#endif
	accelMain = LoadAccelerators(hInstance, MAKEINTRESOURCE(HEWIN_MENUBAR_ACCELERATORS));
	
	/* Show the compass rose, if required */
	if (reg_ShowCompass) show_compass_rose = true;

	/* Now actually bring the window up */
#ifndef UNDER_CE
	if (reg_FullScreen)
	{
		MakeFullScreen(true);
		ShowWindow(wndMain, iCmdShow);
	}
	else
#endif
	{
		ShowWindow(wndMain, reg_Maximized?SW_MAXIMIZE:iCmdShow);
	}

	if (reg_Maximized)
		SendMessage(wndMain, WM_MOVE, 0, 0);

	SetClientDimensions();
	UpdateWindow(wndMain);
	SetForegroundWindow(wndMain);

	/* Try to initialize the audio player--do it here so we can set the
	   disable_sounds value.  (Only call InitPlayer() if we haven't
	   explicitly disabled it.)
	*/
#if !defined (NO_SOUND)
	if ((!disable_sounds) && !InitPlayer()) disable_sounds = true;
#ifdef SPEECH_ENABLED
	if (!disable_sounds) InitSpeech();
#endif
#endif

	/* Make sure menu items are properly checked as per registry defaults */
	CheckMenuItems();


#ifndef NO_SOUND
	/* Make an adjustment if CheckMenuItems() adjusted reg_PlaySounds */
	if (!reg_PlaySounds && !disable_sounds)
	{
		SuspendAudio();
	}
#endif
	/* Get the device context of the main window and create a compatible
	   DC in memory (for dynamic output, to be copied to the screen/client
	   itself)
	*/
	dcMain = GetDC(wndMain);
	dcMem = CreateCompatibleDC(dcMain);

	/* Because 16-color display is painfully slow */
#ifndef UNDER_CE
	if (GetDeviceCaps(dcMain, BITSPIXEL)<8)
	{
		MessageBox(wndMain, _T("Your display is currently set for 16 colors.  Display \
updating may be very slow at this color depth.  You may want to select a greater number \
of colors (using the Control Panel).  At least 16-bit color (65,536 colors) is recommended."),
			_T("Hugo Engine - Display"), MB_ICONEXCLAMATION);
	}
#endif

	/* Create a memory bitmap twice the maximum client height (i.e.,
	   the screen height) to accommodate fast scrolling.  Basically,
	   if fast scrolling is in effect, the display is never physically
	   scrolled when not in a window--i.e., when in the bottom, full-
	   screen text area.  Only the portion of text that will eventually
	   be updated is shifted down.  Then, when it's time for an update,
	   the appropriate "window" of the larger bitmap is blitted to the
	   visible client.  See hugo_scrollwindow() and UpdateClient().
	*/

	/* Compatible bitmap--since a DIB doesn't seem to work for
	   font-smoothing (with some video card drivers): */
#ifndef UNDER_CE
	bmpMem = CreateCompatibleBitmap(dcMain,
		GetSystemMetrics(SM_CXSCREEN),
		GetSystemMetrics(SM_CYSCREEN)*2);
#else
	// Under Windows CE the bitmap has to be big enough to change orientation
	bmpMem = CreateCompatibleBitmap(dcMain,
		(GetSystemMetrics(SM_CXSCREEN)>GetSystemMetrics(SM_CYSCREEN))?GetSystemMetrics(SM_CXSCREEN):GetSystemMetrics(SM_CYSCREEN),
		(GetSystemMetrics(SM_CXSCREEN)>GetSystemMetrics(SM_CYSCREEN))?GetSystemMetrics(SM_CXSCREEN)*2:GetSystemMetrics(SM_CYSCREEN)*2);
#endif

	ReleaseDC(wndMain, dcMain);

	if (!bmpMem) PrintFatalError("Unable to create output window");

	SelectObject(dcMem, bmpMem);

	/* Set up the proportional font dimensions */
	MatchFont(PROP_FONT);
	fontDefault = SelectObject(dcMem, fontCurrent);
	ResetCharDimensions(PROP_FONT);

	/* Set up the fixed font dimensions */
	MatchFont(NORMAL_FONT);
	fontOld = SelectObject(dcMem, fontCurrent);
	DeleteObject(fontOld);
	ResetCharDimensions(NORMAL_FONT);

	/* Start with a fixed font as the default */
	hugo_font(NORMAL_FONT);
	hugo_settextmode();

	/* Set up functions to be called on normal termination */
	atexit((void *)CleanupAndQuit);

	/* Just as we're about to start this game, add it to the
	   MRU file list
	*/
#ifdef UNICODE
	strcpy_AtoU(unicode_buffer, argv[1]);

	for (i=0; i<MRU_FILE_COUNT; i++)
	{
		if (!_tcsicmp(unicode_buffer, MRU_filename[i]))
			break;
	}
	while (i>=MRU_FILE_COUNT-1) i--;

	for (; i>=0; i--)
	{
		if (_tcsicmp(unicode_buffer, MRU_filename[i]))
			_tcscpy(MRU_filename[i+1], MRU_filename[i]);
	}
	_tcscpy(MRU_filename[0], unicode_buffer);
#else
	for (i=0; i<MRU_FILE_COUNT; i++)
	{
		if (!_tcsicmp(argv[1], MRU_filename[i]))
			break;
	}
	while (i>=MRU_FILE_COUNT-1) i--;

	for (; i>=0; i--)
	{
		if (_tcsicmp(argv[1], MRU_filename[i]))
			_tcscpy(MRU_filename[i+1], MRU_filename[i]);
	}
	_tcscpy(MRU_filename[0], argv[1]);
#endif

#ifndef UNDER_CE
	SHAddToRecentDocs(SHARD_PATH, argv[1]);
#endif

	return he_main(argc, argv);
}

void CleanupAndQuit(void)
{
#ifdef SPEECH_ENABLED
	ShutdownSpeech();
#endif
#ifndef NO_SOUND
	hugo_stopvideo();
	hugo_stopsample();
	hugo_stopmusic();
#ifndef USE_NT_SOUND_ONLY
	// MikMod initializes COM last, so it must uninit it first
	// (or not, because we're on the same thread)
	if (!disable_sounds) MD_Exit();
	CoUninitialize();
#endif
#endif
	SaveRegistryDefaults();

	PostQuitMessage(0);
}


/*
 * Main window callback:
 *
 */

/* WndProc

	WndProc is the callback for the main window, responsible for
	processing all messages related to window updating, resizing,
        movement, etc.  As well, we have to use key messages and
	PushKeypress() to record keypresses so that hugo_getkey() can
	restore them to simulate getch().
*/

LRESULT CALLBACK WndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	static int minimized_from;
	static int pre_resize_height;
	static char skip_next_mouse_click = false;
	static char pending_client_resize_shift = false;
	int r, tf;
	POINT rightclick;

#ifdef UNDER_CE
	/* Give the WinCE callback first crack at the message */
	if (WinCEWndProc(hwnd, iMsg, wParam, lParam)==1)
		return 0;
#endif

	switch (iMsg)
	{
		case WM_CREATE:
			minimized_from = 0;
			return 0;

		case WM_ACTIVATE:
			if (LOWORD(wParam)==WA_CLICKACTIVE)
				skip_next_mouse_click = true;
			if (HIWORD(wParam))
				override_resize_handling = true;
			return 0;
		
		case WM_SETFOCUS:
		{
			if (reg_FullScreen)
			{
				MakeFullScreen(true);
			}

#if defined (DEBUGGER)
			if (active_screen==DEBUGGER && during_input)
			{
				PostMessage(wndDebug, WM_COMMAND, HD_RUN_GO, 0);
			}
#endif
			/* Old WinNT uses a much poorer audio factory
			   than other Win32 builds (i.e., no DirectSound)
			*/
#ifndef NO_SOUND
			if (NT_sound_system) ResumeAudio();
#endif
	
#ifdef SPEECH_ENABLED
			ResumeSpeaking();
#endif
			if (!client_painted)
				SetClientDimensions();
			post_picture_update = false;
			if (client_painted)
				UpdateClient(true);

			/* Create a '|' caret */
			CreateCaret(wndMain, NULL, 2, caret_height);
			ConstrainCursor();

			// Needed for wndVideo/XP:
			if (video_running)
			{
				RECT rect;
				GetClientRect(wndMain, &rect);
				InvalidateRect(wndMain, &rect, false);
			}

			return 0;
		}

		case WM_KILLFOCUS:
		{
			if (reg_FullScreen && !just_stopped_video)
			{
				SetWindowPos(wndMain, HWND_BOTTOM, 0, 0, 0, 0,
					SWP_NOACTIVATE | SWP_NOSIZE | SWP_NOMOVE);
			}
			just_stopped_video = false;
#ifndef NO_SOUND
			if (NT_sound_system) SuspendAudio();
#endif
#ifdef SPEECH_ENABLED
			PauseSpeaking();
#endif
			DestroyCaret();

			// Needed for wndVideo/XP:
			if (video_running)
			{
				RECT rect;
				GetClientRect(wndMain, &rect);
				InvalidateRect(wndMain, &rect, false);
			}
			return 0;
		}

		case WM_CLOSE:
#ifndef NO_SOUND
			if (NT_sound_system)
			{
				hugo_stopsample();
				hugo_stopmusic();
				if (player_started)
				{
					MD_PlayStop();
					MD_Exit();
					player_started = false;
				}
			}
#endif
			return 0;

		case WM_INITMENU:
			CheckMenuItems();
			return 0;

		case WM_EXITMENULOOP:
			// Needed for wndVideo/XP:
			if (video_running)
			{
				RECT rect;
				GetClientRect(wndMain, &rect);
				InvalidateRect(wndMain, &rect, false);
			}
			return 0;

#ifndef UNDER_CE
		case WM_ENTERSIZEMOVE:
			client_is_resizing = true;
			break;

		case WM_EXITSIZEMOVE:
			client_is_resizing = false;
			if (pending_client_resize_shift)
			{
				// This is necessary to distinguish WM_*SIZEMOVE
				// calls for sizing vs. moving
				SetClientDimensions();
				CheckResizeShift();
				pending_client_resize_shift = false;
			}
			UpdateClient(true);
			break;
#endif

		case WM_MOVE:
			/* if not fullscreen or maximized */
			if (!reg_FullScreen && !IsZoomed(wndMain))
			{
				RECT rect;
				GetWindowRect(wndMain, &rect);
				reg_xPos = rect.left;
				reg_yPos = rect.top;
			}

			/* fall through to WM_PAINT */

		case WM_PAINT:
			if (client_painted)
			{
				PAINTSTRUCT ps;
				BeginPaint(hwnd, &ps);
				UpdateClient(true);
				EndPaint(hwnd, &ps);
			}
			return 0;

		case WM_SIZE:
		{
			RECT rect;

			if (wndMain==NULL) wndMain = hwnd;	// sanity check

			// Don't need to do client-internal resizing if we're
			// not the current application (i.e., if this WM_SIZE
			// is due to something system-side, like Show Desktop).
			// On the other hand, we definitely want to go through
			// the first time for WinCE, before SCREENWIDTH gets set.
#ifndef UNDER_CE
			if (GetForegroundWindow()!=wndMain)
			{
				return 0;
			}
#endif
			/* If we're overriding custom processing or minimizing,
			   let the default Windows process deal with it
			*/
			if (override_resize_handling || wParam==SIZE_MINIMIZED)
			{
				override_resize_handling = false;
				break;
			}
#ifndef UNDER_CE
			else if (wParam==SIZE_RESTORED)
			{
				reg_Maximized = false;
				SetWindowLong(wndMain, GWL_EXSTYLE, WS_EX_CLIENTEDGE);
			}
			else if (wParam==SIZE_MAXIMIZED)
			{
				reg_Maximized = true;
				SetWindowLong(wndMain, GWL_EXSTYLE, 0);
			}
#endif

			SetClientDimensions();

			if (!client_is_resizing)
			{
				CheckResizeShift();
			}
			else
			{
				pending_client_resize_shift = true;
			}

			hugo_settextmode();	// reset screen variables
			display_needs_repaint = true;

			/* If not fullscreen or maximized or minimized */
			if (!reg_FullScreen && !IsZoomed(wndMain) && !IsIconic(wndMain))
			{
				GetWindowRect(wndMain, &rect);
				reg_xPos = rect.left;
				reg_yPos = rect.top;
				reg_Width = rect.right-rect.left;
				reg_Height = rect.bottom-rect.top;
			}

			// Needed for wndVideo/XP:
			if (video_running)
			{
				GetClientRect(wndMain, &rect);
				InvalidateRect(wndMain, &rect, false);
			}

			/* Recopy the screen buffer contents to the main DC */
			UpdateClient(true);

			return 0;
		}

#ifndef UNDER_CE
		case WM_GETMINMAXINFO:
			if (reg_FullScreen)
			{
				LPMINMAXINFO lpmmi = (LPMINMAXINFO)lParam;

				lpmmi->ptMaxTrackSize.x = 
					GetSystemMetrics(SM_CXSIZEFRAME)*2 +
					GetSystemMetrics(SM_CXSCREEN);
					
				lpmmi->ptMaxTrackSize.y =
					GetSystemMetrics(SM_CYSIZEFRAME)*2 +
					GetSystemMetrics(SM_CYSCREEN) +
					GetSystemMetrics(SM_CYCAPTION);
			}
			break;
#endif	// #ifndef UNDER_CE

		case WM_SYSCOMMAND:
			switch (wParam)
			{
#ifndef UNDER_CE
				// Imperfect, but we need to do these SetWindowLong()
				// calls before we get to the WM_SIZE message
				case SC_RESTORE:
					if ((reg_Maximized && !IsIconic(wndMain)) ||
						(IsIconic(wndMain) && !reg_Maximized && !reg_FullScreen))
					{
						SetWindowLong(wndMain, GWL_EXSTYLE, WS_EX_CLIENTEDGE);
						break;
					}
					// else fall through
				case SC_MAXIMIZE:
					SetWindowLong(wndMain, GWL_EXSTYLE, 0);
					break;
#endif	// #ifndef UNDER_CE
				case SC_CLOSE:
#if defined (DEBUGGER)
					MessageBox(hwnd,
"You must quit the debugger in order to close the output window.",
						"Hugo Debugger",
						MB_ICONEXCLAMATION);
					return 0;
#else
					goto VerifyExit;
#endif
			}
			break;

	/* Process menu selections: */

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case HE_FILE_OPEN:
					// Save defaults so the new
					// instance can read them
					SaveRegistryDefaults();
					NewEngineInstance();
					break;

				case HE_FILE_EXIT:
#ifndef DEBUGGER
VerifyExit:
#endif
					if (MessageBox(hwnd,
						_T("Are you sure you want to abandon the current story?"),
						_T("Quit Current Story"),
						MB_ICONEXCLAMATION | MB_YESNO)==IDYES)
					{
							PostQuitMessage(0);
					}
					break;

				case HE_STORY_RESTART:
				case HE_STORY_RESTORE:
				case HE_STORY_SAVE:
				case HE_STORY_UNDO:
					if (!during_player_input)
					{
						MessageBox(wndMain,
							_T("Only valid at command input"),
							_T("File Operation"),
							MB_ICONEXCLAMATION);
						return 0;
					}

					/* The printing and new-prompt simulation
					   are because the normal flow of a Hugo program
					   expects "save", "restore", etc. to be entered
					   as typed commands.
					*/
#ifdef TYPED_STORY_COMMANDS
					// In case we're "typing" the others"
					if (LOWORD(wParam)==HE_STORY_RESTART)
#endif
						Printout("");

					switch (LOWORD(wParam))
					{
						case HE_STORY_RESTART:
							if (MessageBox(wndMain,
								_T("Are you sure you wish to abandon the current story?"),
								_T("Restart Current Story"),
								MB_ICONEXCLAMATION |
								MB_YESNO)==IDYES)
							{
								/* Font may change: */
								tf = currentfont &= ~ITALIC_FONT;
								/* So restarts aren't nested: */
								during_player_input = false;
								getline_active = false;
								r = RunRestart();
								during_player_input = true;
								getline_active = true;
								hugo_font(tf);
							}
							else
							{
								Printout("[Restart cancelled]");
								goto NewPrompt;
							}
							break;

#ifndef TYPED_STORY_COMMANDS
						case HE_STORY_RESTORE:
							Printout("[RESTORE]");
							r = RunRestore();
							break;
						case HE_STORY_SAVE:
							Printout("[SAVE]");
							r = RunSave();
							break;
						case HE_STORY_UNDO:
							Printout("[UNDO]");
							r = Undo();
							break;
#else
						case HE_STORY_RESTORE:
							TypeCommand("Restore", true, true);
							return 0;
						case HE_STORY_SAVE:
							TypeCommand("Save", true, true);
							return 0;
						case HE_STORY_UNDO:
							TypeCommand("Undo", true, true);
							return 0;
#endif
					}


					if (!r && LOWORD(wParam)!=HE_STORY_UNDO)
					{
						MessageBox(wndMain,
							_T("Unable to perform file operation"),
							_T("File Error"),
							MB_ICONEXCLAMATION);
					}
#ifndef TYPED_STORY_COMMANDS
					else if (!r)
					{
						MessageBox(wndMain,
							_T("Unable to undo"),
							_T("Undo Error"),
							MB_ICONEXCLAMATION);
					}
#endif

NewPrompt:
					/* Simulate a new prompt a la GetCommand() */
					full = 0;
					hugo_settextpos(1, physical_windowheight/lineheight+1);
					hugo_print("\n");
					hugo_print(GetWord(var[prompt]));
					FlushBuffer();
					ConstrainCursor();	/* reposition caret */
					post_picture_update = false;
					UpdateClient(true);
					processed_accelerator_key = true;

					/* for RunGame() */
					full_buffer = false;

					break;

#ifndef UNDER_CE
				case HE_FONTS_FIXED_WIDTH:
					SelectNewFont(NORMAL_FONT);
					display_needs_repaint = true;
					break;

				case HE_FONTS_PROPORTIONAL:
					SelectNewFont(PROP_FONT);
					display_needs_repaint = true;
					break;
#endif

#ifdef USE_SMARTFORMATING
				case HE_FONTS_SMARTFORMATTING:
					smartformatting = !smartformatting;
					break;
#endif

#ifndef UNDER_CE
				case HE_COLORS_FOREGROUND:
				case HE_COLORS_BACKGROUND:
				case HE_COLORS_SLFOREGROUND:
				case HE_COLORS_SLBACKGROUND:
					SelectDefaultColor(LOWORD(wParam));
					display_needs_repaint = true;
					break;

				case HE_COLORS_RESTORE:
					/* Restore defaults from heheader.h */
					reg_DefForeground = hugo_color(DEF_FCOLOR);
					reg_DefBackground = hugo_color(DEF_BGCOLOR);
					reg_DefSLForeground = hugo_color(DEF_SLFCOLOR);
					reg_DefSLBackground = hugo_color(DEF_SLBGCOLOR);
					display_needs_repaint = true;
					break;

				case HE_OPTIONS_FULLSCREEN:
					reg_FullScreen = !reg_FullScreen;
					MakeFullScreen(reg_FullScreen);
					return 0;
#endif

				case HE_OPTIONS_SCROLLING:
					reg_FastScrolling = !reg_FastScrolling;
					break;

#ifdef USE_TEXTBUFFER
				case HE_OPTIONS_TEXTSELECT:
					allow_text_selection = !allow_text_selection;
					break;
#endif

				case HE_OPTIONS_SHOWGRAPHICS:
					reg_DisplayGraphics = !reg_DisplayGraphics;
					display_needs_repaint = true;
					goto UnfreezeScreen;

				case HE_OPTIONS_GRAPHICSSMOOTHING:
					reg_GraphicsSmoothing = !reg_GraphicsSmoothing;
					break;

#ifndef NO_SOUND
				case HE_OPTIONS_PLAYSOUNDS:
					if (reg_PlaySounds)
					{
						SuspendAudio();
						reg_PlaySounds = false;
					}
					else
					{
						reg_PlaySounds = true;
						ResumeAudio();
					}
					break;
#endif

				case HE_OPTIONS_RESET:
					if (MessageBox(wndMain,
						_T("Erase existing display?"),
						_T("Reset Display"),
						MB_ICONEXCLAMATION | MB_YESNO)
						!=IDYES)
					{
						break;
					}

					if (!inwindow)
						hugo_clearfullscreen();
					else
						ClipVisibleClient();

					/* fall through */

				case HE_OPTIONS_UNFREEZE:
UnfreezeScreen:
					hugo_settextwindow(1, 1,
						SCREENWIDTH/FIXEDCHARWIDTH,
						SCREENHEIGHT/FIXEDLINEHEIGHT);
					physical_lowest_windowbottom = lowest_windowbottom = 0;
					ConstrainCursor();
					UpdateClient(true);
					break;

#ifndef NO_COMPASS
				case HE_OPTIONS_COMPASS:

				/* If compass rose is queued to be shown,
					   don't do it now
					*/
					if (show_compass_rose) break;

					if (!reg_ShowCompass)
						ShowCompassRose();
					else
						DestroyWindow(wndCompass);
					reg_ShowCompass = !reg_ShowCompass;

					break;
#endif

#ifdef SCROLLBACK_DEFINED
				case HE_OPTIONS_SCROLLBACK:
					scrollback_active = true;
					DialogBox(AppInstance,
						MAKEINTRESOURCE(HE_SCROLLBACK_DIALOG),
						wndMain, ScrollbackDialog);
					scrollback_active = false;
					break;
#endif

#ifdef SPEECH_ENABLED
				case HE_OPTIONS_SPEECH:
					if (!disable_sounds)
					{
						DialogBox(AppInstance,
							MAKEINTRESOURCE(HE_SPEECH_DIALOG),
							wndMain, SpeechDialog);
					}
					break;
#endif

#ifndef UNDER_CE
				case HE_OPTIONS_SETUP:
					DialogBox(AppInstance,
						MAKEINTRESOURCE(HE_SETUP_DIALOG),
						wndMain, SetupDialog);
					break;
#endif

				case HE_HELP_ABOUT:
					DialogBox(AppInstance,
						MAKEINTRESOURCE(HE_ABOUT_DIALOG),
					        wndMain, AboutDialog);
					break;

#if !defined (COMPILE_V25)
				default:
				{
					char *cc;
					int i, n, noreturn = 0;

					/* See if this is a context command from
					   HandleContextCommand()
					*/
					if ((n = LOWORD(wParam))<_APS_NEXT_COMMAND_VALUE)
						break;

					/* If it is, then pass it to the input line */
					n -= _APS_NEXT_COMMAND_VALUE;

					cc = context_command[n];
					
					// If command ends with "...", don't push 
					// Enter at the end
					if ((strlen(cc)>=4) && !strcmp(cc+strlen(cc)-3, "..."))
					{
						noreturn = 1;
					}

					// Suspend updating
					PushKeypress(OVERRIDE_UPDATE_CLIENT);
					// Esc to clear input
					PushKeypress(27);
					// Each letter of the command
					for (i=0; i<(int)strlen(cc)-(noreturn?4:0); i++)
						PushKeypress(cc[i]);
					PushKeypress(UPDATE_CLIENT);

					if (!noreturn)
						// Enter
						PushKeypress(13);
					else
					{
						PushKeypress(cc[i]);
						PushKeypress(' ');
					}
				}
#endif	// !defined (COMPILE_V25)
			}
			return 0;

	/* Process non-character keys: */

		case WM_KEYDOWN:
			switch (wParam)
			{
				/* Pushed keypresses are read by hugo_getkey()
				*/
				case VK_LEFT:
					if (GetKeyState(VK_CONTROL)&0x8000000)
						PushKeypress(CTRL_LEFT_KEY);
					else
						PushKeypress(8);
					break;
				case VK_RIGHT:
					if (GetKeyState(VK_CONTROL)&0x8000000)
						PushKeypress(CTRL_RIGHT_KEY);
					else
						PushKeypress(21);
					break;
				case VK_UP:
					PushKeypress(11);
					break;
				case VK_DOWN:
					PushKeypress(10);
					break;
				case VK_HOME:
					PushKeypress(HOME_KEY);
					break;
				case VK_END:
					PushKeypress(END_KEY);
					break;
				case VK_INSERT:
					PushKeypress(INSERT_KEY);
					break;
				case VK_DELETE:
					PushKeypress(DELETE_KEY);
					break;
				case VK_PRIOR:	// page up
					PostMessage(wndMain, WM_COMMAND, HE_OPTIONS_SCROLLBACK, 0);
					break;
				case VK_APPS:	// Application menu key
				{
					lParam = 0;
					goto RightButtonDown;
				}
			}
			return 0;

		case WM_MOUSEMOVE:
			/* This will erase any highlight in the compass rose
			   window if the mouse pointer moves into the main
			   window
			*/
#ifndef NO_COMPASS
			if (wndCompass)
			{
				SendMessage(wndCompass, WM_MOUSEMOVE, 0, 0);
			}
#endif
			return 0;

#ifdef USE_TEXTBUFFER
		case WM_LBUTTONDBLCLK:
		{
			/* A doubleclick enters a word from the textbuffer
			   into the input line */
			char *w = TB_FindWord(LOWORD(lParam), HIWORD(lParam));
			if (w && during_player_input)
			{
				if (current_text_x + hugo_textwidth(w) < physical_windowwidth)
				{
					TypeCommand(w, false, false);
				}
			}
			return 0;
		}
#endif

		case WM_LBUTTONDOWN:
			if (skip_next_mouse_click)
			{
				skip_next_mouse_click = false;
				return 0;
			}

			if (getline_active && context_commands)
			{
#ifdef USE_TEXTBUFFER
				DWORD time = GetTickCount();
				while (GetTickCount()-time < GetDoubleClickTime() && allow_text_selection)
				{
					MSG msg;
					if (PeekMessage(&msg, wndMain,
						WM_LBUTTONUP,
						WM_LBUTTONUP,
						PM_NOREMOVE))
					{
						return 0;
					}
				}
#endif
				goto RightButtonDown;
			}

			// Block to allow local scope mouse_* variables
			{
				int mouse_x = LOWORD(lParam) - physical_windowleft;
				int mouse_y = HIWORD(lParam) - physical_windowtop;
				PushKeypress(1);
				PushKeypress(mouse_x/FIXEDCHARWIDTH + 1);
				PushKeypress(mouse_y/FIXEDLINEHEIGHT + 1);
			}
			return 0;

		case WM_RBUTTONDOWN:
RightButtonDown:
			if (skip_next_mouse_click)
			{
				skip_next_mouse_click = false;
				return 0;
			}
			/* A right-button press will bring up the "Story" menu
			   as a popup menu
			*/
			rightclick.x = LOWORD(lParam);
			rightclick.y = HIWORD(lParam);
			ClientToScreen(wndMain, &rightclick);

#if !defined (COMPILE_V25)
			/* Context-command capable games will display
			   the context-command menu if it's not
			   empty
			*/
			if (game_version > 25 && context_commands)
			{
				HandleContextCommands(rightclick.x, rightclick.y);

				// Needed for wndVideo/XP:
				if (video_running)
				{
					RECT rect;
					GetClientRect(wndMain, &rect);
					InvalidateRect(wndMain, &rect, false);
				}

				return 0;
			}
#endif

#ifndef UNDER_CE
#if defined (DEBUGGER)
			/* Because the debugger has no File menu */
			TrackPopupMenu(GetSubMenu(menuMain, 1),
#else
			TrackPopupMenu(GetSubMenu(menuMain, 2),
#endif
				TPM_RIGHTBUTTON,
				rightclick.x, rightclick.y,
				0, wndMain, NULL);
#else	// UNDER_CE
			// The WinCE version:
//			TrackPopupMenu(GetSubMenu(menuMain, 2),
//				TPM_CENTERALIGN | TPM_VCENTERALIGN,
//				rightclick.x, rightclick.y,
//				0, wndMain, NULL);
			return 0;
#endif

			// Needed for wndVideo/XP:
			if (video_running)
			{
				RECT rect;
				GetClientRect(wndMain, &rect);
				InvalidateRect(wndMain, &rect, false);
			}

			return 0;

	/* Process character keys: */

		case WM_CHAR:
#if defined (DEBUGGER)
			/* Tab to debugger */
			if ((int)wParam==VK_TAB)
			{
				SetForegroundWindow(wndDebug);
				return 0;
			}
#endif
			if ((int)wParam==VK_BACK)	/* backspace */
				PushKeypress(BACKSPACE_KEY);
			else
				PushKeypress((int)wParam);
			return 0;

		case WM_DESTROY:
		{
			HDC dcMain;
			dcMain = GetDC(wndMain);
			SelectObject(dcMain, fontDefault);	/* restore original font */
			DeleteObject(fontCurrent);
			ReleaseDC(wndMain, dcMain);
#if !defined (DEBUGGER)
			CleanupAndQuit();
#endif

#ifndef NO_SOUND
			if (NT_sound_system)
			{
				hugo_stopsample();
				hugo_stopmusic();
				if (player_started)
				{
					MD_PlayStop();
					MD_Exit();
					player_started = false;
				}
			}
#endif	// #ifndef NO_SOUND

			return 0;
		}

	}

	return DefWindowProc(hwnd, iMsg, wParam, lParam);
}


/* FrontEndDialog

	Callback for the "Filename to run" dialog.

	(There are a couple of "#ifdef DEBUGGER" sections, since
	the same dialog resource is used for both builds.)
*/

BOOL CALLBACK FrontEndDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	TCHAR fn[MAXFILENAME] = _T("");
#if defined (DEBUGGER)
	TCHAR *filetypes = 
_T("All Hugo executables (*.hdx, *.hex)\0*.hex;*.hdx\0\
Hugo debuggable files (*.hdx)\0*.hdx\0\
All files (*.*)\0*.*\0");
#else
	TCHAR *filetypes =
_T("Hugo executable files (*.hex)\0*.hex\0\
Hugo debuggable files (*.hdx)\0*.hdx\0\
All .HEX/.HDX files\0*.hex;*.hdx\0\
All files (*.*)\0*.*\0");
#endif
	int wNotifyCode, wID;
	HWND control;
	OPENFILENAME ofn;

	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			int i;
			for (i=0; i<MRU_FILE_COUNT; i++)
			{
				SendMessage(GetDlgItem(hwndDlg, HE_FILENAME_COMBO), CB_ADDSTRING, 0, (LPARAM)&MRU_filename[i]);
			}
			SetFocus(GetDlgItem(hwndDlg, HE_FILENAME_COMBO));
#ifdef SINGLE_LAUNCH
			SetWindowPos(hwndDlg, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
#endif
			return 0;       /* since focus has been set */
		}

#ifndef UNDER_CE
		case WM_DROPFILES:
		{
			HDROP hdrop = (HDROP)wParam;
			if (DragQueryFile(hdrop, 0, fn, MAXPATH))
			{
				SetDlgItemText(hwndDlg, HE_FILENAME_COMBO, fn);
				GetDlgItemText(hwndDlg, HE_FILENAME_COMBO, filename, MAXFILENAME);
				EndDialog(hwndDlg, 1);
			}
			break;
		}
#endif

		case WM_COMMAND:
			wNotifyCode = HIWORD(wParam);
			wID = LOWORD(wParam);
			control = (HWND)lParam;

			switch (wID)
			{
				case HE_BROWSE_BUTTON:
				{
					memset(&ofn, 0, sizeof(OPENFILENAME));
					ofn.lStructSize = sizeof(OPENFILENAME);
					ofn.hwndOwner = hwndDlg;
					ofn.hInstance = NULL;
					ofn.lpstrFilter = filetypes;
					ofn.lpstrCustomFilter = NULL;
					ofn.nFilterIndex = 0;
					ofn.lpstrFile = fn;
					ofn.nMaxFile = MAXPATH;
					ofn.lpstrFileTitle = NULL;
					ofn.lpstrInitialDir = &current_games_dir[0];
					ofn.lpstrTitle = _T("Select File to Run");
					ofn.Flags = OFN_FILEMUSTEXIST | OFN_EXPLORER |
						OFN_HIDEREADONLY;
#if defined (DEBUGGER)
					ofn.lpstrDefExt = _T("hdx");
#else
					ofn.lpstrDefExt = _T("hex");
#endif
					if (GetOpenFileName(&ofn))
						SetDlgItemText(hwndDlg, HE_FILENAME_COMBO, fn);
					else
						break;
				}

				case IDOK:
					GetDlgItemText(hwndDlg, HE_FILENAME_COMBO,
						filename, MAXFILENAME);

					if (!_tcscmp(filename, _T("")))
					{
						MessageBox(hwndDlg,
							_T("Select a filename to run or click Cancel"),
							_T("No file selected"),
							MB_ICONEXCLAMATION);
					}
					else
					{
#ifndef UNDER_CE
						if (IsDlgButtonChecked(hwndDlg, HE_NOGRAPHICS_CHECK))
							disable_graphics = true;
						if (IsDlgButtonChecked(hwndDlg, HE_NOSOUND_CHECK))
							disable_sounds = true;
#endif

						GetCurrentDirectory(MAXPATH, current_games_dir);

						EndDialog(hwndDlg, 1);
					}
					break;

				case IDCANCEL:
					PostQuitMessage(0);
					EndDialog(hwndDlg, 0);
					break;
			}
			return 0;
	}

	return 0;
}


/* NewEngineInstance

	Starts a new instance of hewin.exe.  Since no arguments are
	supplied, the front-end "Select file to open" dialog box comes up.
*/

#ifndef UNDER_CE

void NewEngineInstance(void)
{
	int r;
	STARTUPINFO startupinfo;
	PROCESS_INFORMATION processinfo;
	LPVOID lpEnvironment;

	GetStartupInfo(&startupinfo);
	startupinfo.wShowWindow = SW_SHOWNORMAL;
	lpEnvironment = GetEnvironmentStrings();

	r = CreateProcess(executable_module,
		NULL,           /* lpCommandLine */
		NULL, NULL,     /* security attributes */
		TRUE,           /* handle inheritance */
		HIGH_PRIORITY_CLASS,
		lpEnvironment,
		NULL,           /* current directory */
		&startupinfo,
		&processinfo );

	/* if CreateProcess() failed */
	if (!r)
	{
		sprintf(line, "Unable to start \"%s\"", executable_module);
		MessageBox(wndMain, line, "Open Error", MB_ICONERROR);
	}
}

#else	// UNDER_CE
void NewEngineInstance(void)
{}
#endif


/* AboutDialog

	Callback for the "About" window.
*/

BOOL CALLBACK AboutDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			TCHAR about_text[512];

			_stprintf(about_text,
#if defined (NO_GRAPHICS) && defined (NO_SOUND)
_T("Hugo %s v%d.%d%s - %s build (%s)\n\
Copyright  1995-2006 by Kent Tessman\n\
The General Coffee Company Film Productions\n\n\
All rights reserved.\n\nPlease see the Hugo License for details."),
#else
_T("Hugo %s v%d.%d%s - %s build (%s)\n\
Copyright  1995-2006 by Kent Tessman\n\
The General Coffee Company Film Productions\n\n\
JPEG graphics display based in part on the work of the Independent \
JPEG Group.  MikMod sound system used under license.\n\n\
All rights reserved.  Please see the Hugo License for details."),
#endif
#if defined (DEBUGGER)
			_T("Debugger"),
#else
			_T("Engine"),
#endif
			HEVERSION, HEREVISION, _T(HEINTERIM),
#if defined (WINNT)
			_T("WinNT"),
#elif defined (UNDER_CE)
			_T("WinCE"),
#else
			_T("Win32"),
#endif
			_T(__DATE__));
			SetDlgItemText(hwndDlg, HE_ABOUT_TEXT, about_text);
			return TRUE;
		}

		case WM_COMMAND:
			if (LOWORD(wParam)==IDCANCEL || LOWORD(wParam)==IDOK)
				EndDialog(hwndDlg, 0);

			return 0;

		case WM_DESTROY:
			return 0;
	}

	return 0;
}


/* PrintFatalError

	Necessary because Windows can't just dump an error message to
	stdout--there is no standard error output
*/

void PrintFatalError(char *a)
{
	if (a[strlen(a)-1]=='\n') a[strlen(a)-1] = '\0';
	if (a[0]=='\n') a = &a[1];
#ifdef UNICODE
	strcpy_AtoU(unicode_buffer, a);
	MessageBox(wndMain, unicode_buffer, _T("Hugo Engine"), MB_ICONERROR);
#else 
	MessageBox(wndMain, a, "Hugo Engine", MB_ICONERROR);
#endif
	exit(255);
}


/* CenterHookProc

	Because the common dialog boxes supplied by Windows all
	position themselves at (0, 0), which looks dumb.
*/

UINT APIENTRY CenterHookProc(HWND hwndDlg, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	if (uiMsg==WM_INITDIALOG)
	{
		if (!wParam) hwndDlg = GetParent(hwndDlg);

		CenterWindow(hwndDlg, GetDesktopWindow());
		return TRUE;
	}
	return FALSE;
}


/* CenterWindow

	Centers one window over another.
*/

void CenterWindow(HWND wndChild, HWND wndParent)
{
	RECT rect;
	POINT p;
	int width, height;

	/* Find center of parent window */
	GetWindowRect(wndParent, &rect);
	p.x = (rect.right + rect.left)/2;
	p.y = (rect.bottom + rect.top)/2;

	GetWindowRect(wndChild, &rect);
	width = rect.right - rect.left;
	height = rect.bottom - rect.top;

	SetWindowPos(wndChild, NULL,
		p.x - width/2, p.y - height/2,
		0, 0,
		SWP_NOSIZE | SWP_NOZORDER);
}


/*
 * Visible client updating/formatting:
 *
 */

/* UpdateClient

	Because for Windows we're using a buffered system to display, first
	writing everything to a memory bitmap, then blitting it to the screen,
	UpdateClient() is called whenever the visible client needs to be
	updated.

	Also, since we're using a scroll offset for fast scrolling (i.e.,
	instead of blitting the entire bitmap for every scrolled line,
	we're downward-shifting the portion of the bitmap that will
	ultimately be copied), take that into account.  Fast scrolling only
	matters if we're not in a window--physical_lowest_windowbottom gives
	us the number of lines (pixels, not text) to protect at the top of
	the screen (set in RunWindows() from herun.c).
*/

static BOOL window_icon_set = false;

void UpdateClient(int visible_update)
{
	HGDIOBJ penNew, brushNew, penDefault, brushDefault;
	HDC dcMain;

	if (override_update_client) return;

	/* A convenient place to check (once) if the window icon has been set--must
	   be done after the game is loaded
	*/
	if (mem && !window_icon_set)
	{
		defseg = gameseg;
		if (Peek(H_DEBUGGABLE))
			SendMessage(wndMain, WM_SETICON, ICON_SMALL, (LPARAM)LoadIcon(AppInstance, MAKEINTRESOURCE(HDX_ICON)));
		else
			SendMessage(wndMain, WM_SETICON, ICON_SMALL, (LPARAM)LoadIcon(AppInstance, MAKEINTRESOURCE(HEX_ICON)));
		window_icon_set = true;
	}

	dcMain = GetDC(wndMain);

	if (scroll_offset)
	{
		/* If we've just been shifting scroll_offset instead of
		   physically scrolling the display, update the memory
		   bitmap to remove the scroll offset
		*/
		BitBlt(	dcMem, 0, physical_lowest_windowbottom,
			client_width, client_height-physical_lowest_windowbottom,
			dcMem, 0, physical_lowest_windowbottom+scroll_offset,
			SRCCOPY );

		/* Erase the unseen portion of the scroll-offset area */
		penNew = CreatePen(PS_SOLID, 1, scroll_offset_fill);
		brushNew = CreateSolidBrush(scroll_offset_fill);
		penDefault = SelectObject(dcMem, penNew);
		brushDefault = SelectObject(dcMem, brushNew);

		Rectangle(dcMem, 0, client_height, client_width,
			GetSystemMetrics(SM_CYSCREEN)+lineheight);

		/* Re-select the default GDI objects */
		SelectObject(dcMem, penDefault);
		SelectObject(dcMem, brushDefault);
		DeleteObject(penNew);
		DeleteObject(brushNew);
	}

	// Prevent unnecessary updating when drawing multiple images
	if (post_picture_update) visible_update = false;

	/* To update, simply blit the memory bitmap to the visible client */
	if (visible_update && dcMain && dcMem)
	{
#ifndef NO_VIDEO
		// If video is playing, blit around it, otherwise
		// just blit the entire DC
		if (!video_running)
		{
			BitBlt(	dcMain, 0, 0,
				client_width, client_height,
				dcMem, 0, 0,
				SRCCOPY );
		}
		else
		{
			if (video_window.bottom < client_height)
			{
				BitBlt( dcMain, 0, video_window.bottom,
					client_width, client_height-video_window.bottom,
					dcMem, 0, video_window.bottom,
					SRCCOPY );
			}
			if (video_window.top > 0)
			{
				BitBlt( dcMain, 0, 0,
					client_width, video_window.top,
					dcMem, 0, 0,
					SRCCOPY );
			}
			if (video_window.left > 0)
			{
				BitBlt( dcMain, 0, video_window.top,
					video_window.left, video_window.bottom-video_window.top,
					dcMem, 0, video_window.top,
					SRCCOPY );
			}
			if (video_window.right < client_width)
			{
				BitBlt( dcMain, video_window.right, video_window.top,
					client_width-video_window.right, video_window.bottom-video_window.top,
					dcMem, video_window.right, video_window.top,
					SRCCOPY );
			}
		}

#else	// NO_VIDEO
		BitBlt(	dcMain, 0, 0, client_width, client_height, dcMem, 0, 0, SRCCOPY );
#endif
	}

	ReleaseDC(wndMain, dcMain);

	scroll_offset = 0;
	if (visible_update)
		client_needs_updating = false;

	// Set the window class background brush to NULL after the first client
	// update in order to prevent black flashing on resizing; WM_PAINT will
	// handle calling this function to paint the empty client area.
#ifndef UNDER_CE
	if (!client_painted)
	{
		SetClassLong(wndMain, GCL_HBRBACKGROUND, 0);
	}
#endif

	client_painted = true;
}


/* ConstrainCursor */

void ConstrainCursor(void)
{
	if (client_is_resizing)
		return;

	if (current_text_x < physical_windowleft)
		current_text_x = physical_windowleft;
	if (current_text_x > physical_windowright)
		current_text_x = physical_windowright;
	if (current_text_y > physical_windowbottom-lineheight)
		current_text_y = physical_windowbottom-lineheight;
	if (current_text_y < physical_windowtop)
		current_text_y = physical_windowtop;

	// Checking dcMem is probably not strictly necessary,
	// but let's keep BoundsChecker happy:
	if (dcMem)
		SetCaretPos(current_text_x, current_text_y);
}


/* ClipVisibleClient

	Because when resizing the window, we have to trim off whatever might
	have been visible on a larger window since it won't get scrolled for
	a smaller window, and revealing a larger client space will show all
	the non-updated garbage.
*/

void ClipVisibleClient(void)
{
	HGDIOBJ penNew, brushNew, penDefault, brushDefault;

	int right, bottom;

	penNew = CreatePen(PS_SOLID, 1, current_back_color);
	brushNew = CreateSolidBrush(current_back_color);
	penDefault = SelectObject(dcMem, penNew);
	brushDefault = SelectObject(dcMem, brushNew);

	right = GetSystemMetrics(SM_CXSCREEN);
	bottom = GetSystemMetrics(SM_CYSCREEN)*2;

	/* right side */
	Rectangle(dcMem, SCREENWIDTH, 0, right, bottom);

	/* bottom area */
	Rectangle(dcMem, 0, SCREENHEIGHT, right, bottom);

	SelectObject(dcMem, penDefault);
	SelectObject(dcMem, brushDefault);
	DeleteObject(penNew);
	DeleteObject(brushNew);

	client_painted = true;
}


/* CheckResizeShift

	After resizing, checks to see if we should shift the (previously
	larger) bitmap up so that output doesn't start overwriting older
	output.
*/

void CheckResizeShift(void)
{
	if (client_height < current_text_y+lineheight)
	{
		BitBlt( dcMem, 0, 0, client_width, client_height,
			dcMem, 0, current_text_y+lineheight-client_height,
			SRCCOPY );
	}

	hugo_settextmode();	// reset screen dimensions for ClipVisibleClient()
	ClipVisibleClient();

	client_needs_updating = true;
}


/* MakeFullScreen(full)

	Enters or exits fullscreen mode, depending on whether full is true.
*/

#ifndef UNDER_CE

void MakeFullScreen(int full)
{
	static WINDOWPLACEMENT max_wp;
	static RECT max_rect;
	long style;

	if (full)
	{
		int x_off, y_off, width, height;

		reg_FullScreen = true;

		if (reg_Maximized)
		{
			max_wp.length = sizeof(WINDOWPLACEMENT);
			GetWindowPlacement(wndMain, &max_wp);
			GetWindowRect(wndMain, &max_rect);
		}

		SetWindowLong(wndMain, GWL_EXSTYLE, 0);
		 
		x_off = GetSystemMetrics(SM_CXSIZEFRAME);
		y_off = GetSystemMetrics(SM_CYCAPTION) +
			GetSystemMetrics(SM_CYSIZEFRAME);

		width = x_off*2 + GetSystemMetrics(SM_CXSCREEN);
		height = y_off + GetSystemMetrics(SM_CYSCREEN) +
			GetSystemMetrics(SM_CYSIZEFRAME);

		SetWindowPos(wndMain, HWND_TOPMOST,
			-x_off, -y_off, width, height,
			SWP_NOACTIVATE);

		style = GetWindowLong(wndMain, GWL_STYLE);
		style = style & ~WS_SYSMENU;
		SetWindowLong(wndMain, GWL_STYLE, style);
	}
	else
	{
		style = GetWindowLong(wndMain, GWL_STYLE);
		style = style | WS_SYSMENU;
		SetWindowLong(wndMain, GWL_STYLE, style);

		if (reg_Maximized)
		{
			max_wp.showCmd = SW_SHOWMAXIMIZED;
			SetWindowPlacement(wndMain, &max_wp);
			SetWindowPos(wndMain, HWND_NOTOPMOST,
				max_wp.ptMaxPosition.x, max_wp.ptMaxPosition.y,
				max_rect.right-max_rect.left+1,
				max_rect.bottom-max_rect.top+1,
				SWP_NOACTIVATE);
		}
		else
		{
			SetWindowLong(wndMain, GWL_EXSTYLE, WS_EX_CLIENTEDGE);

			SetWindowPos(wndMain, HWND_NOTOPMOST,
				reg_xPos, reg_yPos, reg_Width, reg_Height,
				SWP_NOACTIVATE);
		}

		reg_FullScreen = false;
	}
}

#else	// UNDER_CE
void MakeFullScreen(int full)
{}
#endif


/* SetClientDimensions

	Easier than continually calling GetClientRect() and dissecting the
	result.
*/

void SetClientDimensions(void)
{
	RECT rect;

	GetClientRect(wndMain, &rect);
	client_width = rect.right - rect.left + 1;
	client_height = rect.bottom - rect.top + 1;

#ifdef UNDER_CE
	client_height -= MENU_HEIGHT;
#endif

	// Resizing corrections might as well get done here, too
	if (!inwindow)
	{
		physical_windowright = client_width;
		physical_windowwidth = client_width - physical_windowleft + 1;
		physical_windowbottom = client_height;
		physical_windowheight = client_height - physical_windowtop + 1;

		if (currentline > physical_windowheight/lineheight)
			currentline = physical_windowheight / lineheight;
	}

	return;
}


/*
 * Font selection/measurement:
 *
 */
 
/* MatchFont

	The 'flags' is basically a collection of font flags (from heheader.h)
	comprising one or more of PROP_FONT, ITALIC_FONT, BOLD_FONT, and
	UNDERLINE_FONT.
*/

void MatchFont(flags)
{
	int s, default_flags;
	LOGFONT lf;

	/* The default flags for the proportional or fixed font may
	   already contain bold, italic, or underline
	*/
	default_flags = (flags&PROP_FONT)?reg_PropFlags:reg_FixedFlags;

	lf.lfWidth = lf.lfEscapement = lf.lfOrientation = 0;

	if ((flags & BOLD_FONT) || (default_flags & BOLD_FONT))
		lf.lfWeight = FW_BOLD;
	else lf.lfWeight = FW_NORMAL;

	lf.lfItalic = (flags & ITALIC_FONT) || (default_flags & ITALIC_FONT);
	lf.lfUnderline = (flags & UNDERLINE_FONT) || (default_flags & UNDERLINE_FONT);
	lf.lfCharSet = (flags & PROP_FONT) ? reg_PropCharset : reg_FixedCharset;
	lf.lfStrikeOut = 0;
	lf.lfOutPrecision = OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
	lf.lfQuality = DEFAULT_QUALITY;

	if (flags & PROP_FONT)
	{
		s = reg_PropPoint;
		lf.lfPitchAndFamily = VARIABLE_PITCH;
		_tcscpy(lf.lfFaceName, reg_PropFont);
	}
	else
	{
		s = reg_FixedPoint;
		lf.lfPitchAndFamily = FIXED_PITCH;
		_tcscpy(lf.lfFaceName, reg_FixedFont);
	}

	/* Convert point to logical height */
	lf.lfHeight = -(long)(MulDiv(s, GetDeviceCaps(dcMem, LOGPIXELSY), 72));

	fontCurrent = CreateFontIndirect(&lf);
	if (fontCurrent==NULL)
		MessageBox(wndMain, _T("Unable to load font(s)"), _T("Hugo Engine"), MB_ICONWARNING);
	else
		/* Whatever we requested, figure out what we got */
		GetObject(fontCurrent, sizeof(LOGFONT), &lfCurrent);
}


/* ResetCharDimensions

	Reset default charwidth and lineheight for the selected font, fixed-
	width or proportional, as appropriate.  For proportional printing,
	also set up a lookup table for character widths, based on the normal
	proportional font--i.e., normal, bold, bold-italic, etc. all use the
	same lookup table.
*/

void ResetCharDimensions(int f)
{
	TEXTMETRIC tm;

	GetTextMetrics(dcMem, &tm);

	/* proportional */
	if (f & PROP_FONT)
	{
		prop_charwidth = tm.tmMaxCharWidth;
		prop_lineheight = tm.tmHeight;
		prop_charset = tm.tmCharSet;
	}

	/* fixed-width */
	else
	{
		FIXEDCHARWIDTH = tm.tmAveCharWidth;
		FIXEDLINEHEIGHT = tm.tmHeight;
		fixed_charset = tm.tmCharSet;
	}
}


/* SelectNewFont

	Uses a ChooseFont built-in dialog to select either the fixed or
	proportional font.  The usage of which_font assumes that if the
	fixed-width font is being set, which_font = 0 (NORMAL_FONT).
*/

#ifndef UNDER_CE

void SelectNewFont(int which_font)
{
	TCHAR style[32];
	CHOOSEFONT cf;
	LOGFONT lf;
	int flags, t, tempfont;

	/* First set up a LOGFONT structure with details on the current
	   font (for defaults in the ChooseFont dialog)
	*/
	lf.lfCharSet = (which_font==PROP_FONT)?reg_PropCharset:reg_FixedCharset;
	_tcscpy(lf.lfFaceName, which_font?reg_PropFont:reg_FixedFont);

	/* Convert point to logical height */
	lf.lfHeight = -(long)(MulDiv(which_font?reg_PropPoint:reg_FixedPoint,
		GetDeviceCaps(dcMem, LOGPIXELSY), 72));

	flags = (which_font==PROP_FONT)?reg_PropFlags:reg_FixedFlags;
	_tcscpy(style, _T("Regular"));
	if (flags & BOLD_FONT && flags & ITALIC_FONT)
       		_tcscpy(style, _T("Bold Italic"));
	else if (flags & BOLD_FONT) _tcscpy(style, _T("Bold"));
	else if (flags & ITALIC_FONT) _tcscpy(style, _T("Italic"));

	/* Now initialize and launch the ChooseFont dialog: */

	cf.lStructSize = sizeof(CHOOSEFONT);
	cf.hwndOwner = wndMain;
	cf.lpLogFont = &lf;
	cf.lpszStyle = style;
	cf.Flags = CF_SCREENFONTS | CF_FORCEFONTEXIST |
		(which_font?0:CF_FIXEDPITCHONLY) | 
		CF_USESTYLE | CF_INITTOLOGFONTSTRUCT | CF_ENABLEHOOK;
	// hook function to center the dialog:
	cf.lpfnHook = (LPCFHOOKPROC)CenterHookProc;

	if (!ChooseFont(&cf)) return;	/* return if failed */

	/* Set up the new font */
	_tcscpy((which_font==PROP_FONT)?reg_PropFont:reg_FixedFont, lf.lfFaceName);

	/* Figure out the selected point size */
	t = -(long)(MulDiv(lf.lfHeight, 72, GetDeviceCaps(dcMem, LOGPIXELSY)));
	if (which_font==PROP_FONT)
		reg_PropPoint = t;
	else
		reg_FixedPoint = t;

	/* Figure out the selected charset */
	if (which_font==PROP_FONT)
		reg_PropCharset = lf.lfCharSet;
	else
		reg_FixedCharset = lf.lfCharSet;
		
	/* Figure out the selected style flags */
	t = NORMAL_FONT;
	if (!_tcscmp(cf.lpszStyle, _T("Bold Italic"))) t = BOLD_FONT | ITALIC_FONT;
	else if (!_tcscmp(cf.lpszStyle, _T("Bold"))) t = BOLD_FONT;
	else if (!_tcscmp(cf.lpszStyle, _T("Italic"))) t = ITALIC_FONT;

	if (which_font==PROP_FONT)
		reg_PropFlags = t;
	else
		reg_FixedFlags = t;

	tempfont = currentfont;
	// Need to do this here to make sure hugo_font() sets the font
	last_hugo_font = -1;
	hugo_font(currentfont = (which_font==PROP_FONT)?PROP_FONT:NORMAL_FONT);
	ResetCharDimensions((which_font==PROP_FONT)?PROP_FONT:NORMAL_FONT);
	hugo_font(currentfont = tempfont);
}


/*
 * Color-selection dialog:
 *
 */

/* SelectDefaultColor

	Uses a color-selection dialog to select one of the default
	colors.
*/

#ifndef OLD_COLOR_SELECTION_METHOD
void SelectDefaultColor(int which_color)
{
	CHOOSECOLOR cc;
	int old_background = reg_DefBackground;

	switch (which_color)
	{
		case HE_COLORS_FOREGROUND:
			cc.rgbResult = reg_DefForeground; break;
		case HE_COLORS_BACKGROUND:
			cc.rgbResult = reg_DefBackground; break;
		case HE_COLORS_SLFOREGROUND:
			cc.rgbResult = reg_DefSLForeground; break;
		case HE_COLORS_SLBACKGROUND:
			cc.rgbResult = reg_DefSLBackground; break;
	}

	cc.lStructSize = sizeof(CHOOSECOLOR);
	cc.hwndOwner = wndMain;
	cc.lpCustColors = (LPDWORD)reg_CustomColors;
	cc.Flags = CC_RGBINIT | CC_ENABLEHOOK;
	// hook function to center the dialog:
	cc.lpfnHook = (LPCCHOOKPROC)CenterHookProc;
 	
	if (!ChooseColor(&cc)) return;	// failed

	switch (which_color)
	{
		case HE_COLORS_FOREGROUND:
			reg_DefForeground = cc.rgbResult; break;
		case HE_COLORS_BACKGROUND:
			reg_DefBackground = cc.rgbResult; break;
		case HE_COLORS_SLFOREGROUND:
			reg_DefSLForeground = cc.rgbResult; break;
		case HE_COLORS_SLBACKGROUND:
			reg_DefSLBackground = cc.rgbResult; break;
	}

	if (old_background != reg_DefBackground && during_player_input)
		current_back_color = reg_DefBackground;
}

#else	// #ifdef OLD_COLOR_SELECTION_METHOD

int selected_color;

void SelectDefaultColor(int which_color)
{
	int old_background = reg_DefBackground;

	selected_color = which_color;

	if (DialogBox(AppInstance, MAKEINTRESOURCE(HE_COLOR_DIALOG),
		wndMain, SelectColorDialog))
	{
		switch (which_color)
		{
			/* selected_color will have been changed to the
			   selected palette index
			*/
			case HE_COLORS_FOREGROUND:
				reg_DefForeground = selected_color; break;
			case HE_COLORS_BACKGROUND:
				reg_DefBackground = selected_color; break;
			case HE_COLORS_SLFOREGROUND:
				reg_DefSLForeground = selected_color; break;
			case HE_COLORS_SLBACKGROUND:
				reg_DefSLBackground = selected_color; break;
		}

		if (old_background != reg_DefBackground && during_player_input)
			current_back_color = reg_DefBackground;
	}
}


/* SelectColorDialog

	Callback for the "Select Color" dialog.  Note that the HE_COLOR_LIST
	list box does not have the LBS_SORT style, so the list is NOT sorted
	alphabetically--this is helpful, because neither is the color list.
*/

BOOL CALLBACK SelectColorDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	char *colorname[16] = {"Black", "Blue", "Green", "Cyan", "Red",
		"Magenta", "Brown", "White",
		"Dark Gray", "Light Blue", "Light Green", "Light Cyan",
		"Light Red", "Light Magenta", "Yellow", "Bright White"};
	int i, current;
	HWND hwndList;
	HDC dcDlg;
	HGDIOBJ penNew, brushNew, penDefault, brushDefault;

	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			hwndList = GetDlgItem(hwndDlg, HE_COLOR_LIST);
			for (i=0; i<16; i++)
			{
				SendMessage(hwndList,
					LB_ADDSTRING, 0,
					(LPARAM)colorname[i]);
				SendMessage(hwndList, LB_SETITEMDATA, i, (LPARAM)i);
			}

			switch (selected_color)
			{
				case HE_COLORS_FOREGROUND:
					current = reg_DefForeground; break;
				case HE_COLORS_BACKGROUND:
					current = reg_DefBackground; break;
				case HE_COLORS_SLFOREGROUND:
					current = reg_DefSLForeground; break;
				case HE_COLORS_SLBACKGROUND:
					current = reg_DefSLBackground; break;
			}
			SendMessage(hwndList, LB_SETCURSEL, current, 0);

			SetFocus(hwndList);
			return FALSE;
		}

		case WM_PAINT:
			goto PaintSampleBox;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					return 0;

				case HE_COLOR_LIST:
				{
					/* If the selection in the color list
					   has changed, redraw the sample
					*/
					if (HIWORD(wParam)==LBN_SELCHANGE)
					{
						RECT rect;
PaintSampleBox:

						hwndList = GetDlgItem(hwndDlg, HE_COLOR_LIST);
						dcDlg = GetDC(hwndDlg);
						current = SendMessage(hwndList,
							LB_GETCURSEL, 0, 0);

						penNew = CreatePen(PS_SOLID, 1, 0);
						brushNew = CreateSolidBrush(hugo_color(current));
						penDefault = SelectObject(dcDlg, penNew);
						brushDefault = SelectObject(dcDlg, brushNew);

						rect.left = 103;
						rect.top = 18;
						rect.right = 160;
						rect.bottom = 33;

						MapDialogRect(hwndDlg, &rect);
						
						Rectangle(dcDlg, rect.left, rect.top, rect.right, rect.bottom);

						SelectObject(dcDlg, penDefault);
						SelectObject(dcDlg, brushDefault);
						DeleteObject(penNew);
						DeleteObject(brushNew);
						ReleaseDC(hwndDlg, dcDlg);
					}

					if (HIWORD(wParam)!=LBN_DBLCLK)
						break;
					/* fall through on doubleclick */
				}

				case IDOK:
					hwndList = GetDlgItem(hwndDlg, HE_COLOR_LIST);
					selected_color = SendMessage(hwndList,
							LB_GETCURSEL, 0, 0);
					EndDialog(hwndDlg, 1);
					return 0;
			}
			return 0;
	}

	return 0;
}

#endif	// OLD_COLOR_SELECTION_METHOD

#endif	// !UNDER_CE


/*
 * Menu management:
 *
 */

/* CheckMenuItems

	Called to enable/disable or check/uncheck menu items as appropriate.
*/

void CheckMenuItems(void)
{
#if defined (DEBUGGER)
	// Disable the compass rose under the debugger
	EnableMenuItem(menuMain, HE_OPTIONS_COMPASS, MF_GRAYED);
#endif

	if (disable_graphics)
	{
// Comment MF_GRAYED out so that disable_graphics isn't unchangeable
//		EnableMenuItem(menuMain, HE_OPTIONS_SHOWGRAPHICS, MF_GRAYED);
		reg_DisplayGraphics = false;
		disable_graphics = false;	/* so it doesn't stick */
	}

	if (disable_sounds)
		EnableMenuItem(menuMain, HE_OPTIONS_PLAYSOUNDS, MF_GRAYED);

#ifdef USE_SMARTFORMATTING
	CheckMenuItem(menuMain, HE_FONTS_SMARTFORMATTING,
		smartformatting?MF_CHECKED:MF_UNCHECKED);
#endif
	CheckMenuItem(menuMain, HE_OPTIONS_FULLSCREEN,
		reg_FullScreen?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem(menuMain, HE_OPTIONS_SCROLLING,
		reg_FastScrolling?MF_CHECKED:MF_UNCHECKED);
#ifdef USE_TEXTBUFFER
	CheckMenuItem(menuMain, HE_OPTIONS_TEXTSELECT,
		allow_text_selection?MF_CHECKED:MF_UNCHECKED);
#else
	DeleteMenu(menuMain, HE_OPTIONS_TEXTSELECT, MF_BYCOMMAND);
#endif
	CheckMenuItem(menuMain, HE_OPTIONS_SHOWGRAPHICS,
		reg_DisplayGraphics?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem(menuMain, HE_OPTIONS_GRAPHICSSMOOTHING,
		reg_GraphicsSmoothing?MF_CHECKED:MF_UNCHECKED);
	EnableMenuItem(menuMain, HE_OPTIONS_GRAPHICSSMOOTHING,
		reg_DisplayGraphics?MF_ENABLED:MF_GRAYED);

#ifndef UNDER_CE
#ifdef SPEECH_ENABLED
	if (disable_sounds)
		EnableMenuItem(menuMain, HE_OPTIONS_SPEECH, MF_GRAYED);
#else
	EnableMenuItem(menuMain, HE_OPTIONS_SPEECH, MF_GRAYED);
#endif
#endif	// ifndef UNDER_CE

	CheckMenuItem(menuMain, HE_OPTIONS_PLAYSOUNDS,
		reg_PlaySounds?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem(menuMain, HE_OPTIONS_COMPASS,
		reg_ShowCompass?MF_CHECKED:MF_UNCHECKED);
}


/* TypeCommand

	Called to "fake" a typed command.
*/

void TypeCommand(char *cmd, char clear, char linefeed)
{
	int i;

	PushKeypress(OVERRIDE_UPDATE_CLIENT);
	if (clear) PushKeypress(27);
	// Each letter of the command
	for (i=0; i<(int)strlen(cmd); i++)
		PushKeypress(cmd[i]);
	if (linefeed)
	{
		PushKeypress(UPDATE_CLIENT);
		PushKeypress(13);
	}
	else
	{
		PushKeypress(' ');
		PushKeypress(UPDATE_CLIENT);
	}
}


/*
 * Compass rose data and functions:
 *
 */

#ifndef NO_COMPASS

int GetGridSelection(int x, int y);

/* Pixel measurements from the actual bitmap(s): */
#define COMPASS_WIDTH	252
#define COMPASS_HEIGHT	135

/* A grid of the boundaries for every direction hotspot on the compass
   rose:  12 directions (8 compass, up/down, int/out) * 4 coordinates in
   left, top, right, bottom order
*/
int grid[12][4] =
{
	/* x1, y1, x2, y2 */
	{108, 5, 147, 45},	// N
	{148, 18, 182, 49},	// NE
	{150, 50, 210, 84},	// E
	{148, 85, 182, 115},	// SE
	{108, 85, 147, 126},	// S
	{65, 85, 107, 115},	// SW
	{40, 50, 105, 84},	// W
	{65, 18, 107, 49},	// NW
	{205, 15, 230, 55},	// in
	{205, 80, 232, 118},	// out
	{18, 15, 55, 40},	// up
	{18, 90, 55, 119},	// down
};

char *compass_point[12] =
{
	"North", "Northeast", "East", "Southeast",
	"South", "Southwest", "West", "Northwest",
	"In", "Out", "Up", "Down"
};


/* ShowCompassRose

	Careful with calling this:  it doesn't check to see if multiple
	compasses are being created.
*/

void ShowCompassRose(void)
{
	WNDCLASS wndclass;
	HDC dcMain, dcCompass;

	dcMain = GetDC(wndMain);

	/* 24-bit bitmaps, 3 bytes/pixel: */
	bmpEmbossed = LoadBitmap(AppInstance, _T("COMPASS_EMBOSSED"));
	bmpSolid = LoadBitmap(AppInstance, _T("COMPASS_SOLID"));

	/* Create device contexts to enable blitting */
	dcEmbossed = CreateCompatibleDC(dcMain);
	dcSolid = CreateCompatibleDC(dcMain);

	/* Select the bitmaps into the memory device contexts */
	SelectObject(dcEmbossed, bmpEmbossed);
	SelectObject(dcSolid, bmpSolid);

	/* The compass rose window class */
	wndclass.style          = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc    = wndCompassProc;
	wndclass.cbClsExtra     = 0;
	wndclass.cbWndExtra     = 0;
	wndclass.hInstance      = AppInstance;
	wndclass.hIcon          = LoadIcon(AppInstance, MAKEINTRESOURCE(PROGRAM_ICON));
	wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground  = (HBRUSH)GetStockObject(BLACK_BRUSH);
	wndclass.lpszMenuName   = NULL;
	wndclass.lpszClassName  = _T("Compass Rose");
	RegisterClass(&wndclass);

	/* Make sure the compass rose is going to actually be drawn on the
	   screen.  Seems like we're doing redundant checking, but it is
	   possible that reg_CompassX/Y + COMPASS_WIDTH/HEIGHT will wrap
	   if there has been a positioning error
	*/
	if (reg_CompassX > GetSystemMetrics(SM_CXSCREEN) ||
		reg_CompassX + COMPASS_WIDTH > GetSystemMetrics(SM_CXSCREEN))
	{
		reg_CompassX = GetSystemMetrics(SM_CXSCREEN)/2;
	}
	if (reg_CompassY > GetSystemMetrics(SM_CYSCREEN) ||
		reg_CompassY + COMPASS_HEIGHT > GetSystemMetrics(SM_CYSCREEN))
	{
		reg_CompassY = GetSystemMetrics(SM_CYSCREEN)/2;
	}

	/* Create the compass rose window */
#ifdef UNDER_CE
	wndCompass = CreateWindowEx(WS_EX_NOACTIVATE,
#else
	wndCompass = CreateWindowEx(0,
#endif
			_T("Compass Rose"),	// class name
			_T(""),			// caption
			WS_POPUP | WS_BORDER,	// flags
			reg_CompassX, reg_CompassY,
			COMPASS_WIDTH + GetSystemMetrics(SM_CXBORDER)*2,
			COMPASS_HEIGHT + GetSystemMetrics(SM_CYBORDER)*2,
			wndMain, NULL,
			AppInstance, NULL);

	/* Create the bitmap (static control) for the actual image */
	wndCompassBmp = CreateWindow(
			_T("STATIC"),
			_T("COMPASS_EMBOSSED"),
			WS_CHILD | SS_BITMAP,
			0, 0, 0, 0,
			wndCompass, NULL,
			AppInstance, NULL);

	dcCompass = GetDC(wndCompassBmp);
	if (wndCompass==NULL || wndCompassBmp==NULL || dcCompass==NULL)
	{
		MessageBox(wndMain,
			_T("Unable to create window(s).  (Exiting this application is recommended.)"),
			_T("Hugo Engine"),
			MB_ICONEXCLAMATION);
		return;
	}

	ReleaseDC(wndMain, dcMain);
	ReleaseDC(wndCompass, dcCompass);

	ShowWindow(wndCompassBmp, SW_SHOWNORMAL);
	ShowWindow(wndCompass, SW_SHOWNORMAL);
}


/* wndCompassProc

	The callback for the compass rose window.
*/

LRESULT CALLBACK wndCompassProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	int x, y, fwkeys;
	static char dragging;
	static int selected, last_selected;
	static RECT rect;
	static POINT dragpoint;
	HDC dcCompass;

	switch (iMsg)
	{
		case WM_CREATE:
			selected = -1;
			dragging = 0;
			return 0;

		case WM_SETFOCUS:
			/* The compass window never gets keyboard focus */
			SetFocus(wndMain);
			return 0;

		case WM_KILLFOCUS:
			ReleaseCapture();
			selected = -1;
			dragging = 0;
			return 0;

		case WM_MOUSEMOVE:
			fwkeys = wParam;
			x = (signed short)LOWORD(lParam);
			y = (signed short)HIWORD(lParam);

			dcCompass = GetDC(wndCompass);

			/* See if the mouse pointer is within a selectable
			   region of the compass bitmap
			*/
			selected = GetGridSelection(x, y);
			if (dragging || FindWord("north")==UNKNOWN_WORD) selected = -1;

			/* Restore the embossed background for
			   the old area
			*/
			if (selected!=last_selected)
			{
				BitBlt(dcCompass,
					rect.left, rect.top,
					rect.right-rect.left,
					rect.bottom-rect.top,
					dcEmbossed,
					rect.left, rect.top,
					SRCCOPY);
			}

			/* Unless it's the same selected area, in
			   which case don't bother
			*/ 
			else if (selected!=-1)
			{
				ReleaseDC(wndCompass, dcCompass);
				return 0;
			}

			/* If in a selectable region, not dragging, the application is
			   active, and we're able to accept a command (i.e., at player
			   input)...
			*/
			if (selected!=-1 && GetActiveWindow() 
				&& !dragging && during_player_input)
			{
				rect.left = grid[selected][0];
				rect.top = grid[selected][1];
				rect.right = grid[selected][2];
				rect.bottom = grid[selected][3];

				/* Draw the solid foreground for the
				   new area
				*/
				BitBlt(dcCompass,
					rect.left, rect.top,
					rect.right-rect.left,
					rect.bottom-rect.top,
					dcSolid,
					rect.left, rect.top,
					SRCCOPY);
			}

			/* ...or somewhere in the rest of the window, in
			   which case we can drag the compass window itself
			   if the left button is pressed
			*/
			else if (dragging && fwkeys & MK_LBUTTON)
			{
				reg_CompassX += (x - dragpoint.x);
				reg_CompassY += (y - dragpoint.y);

				MoveWindow(wndCompass,
					reg_CompassX, reg_CompassY,
					COMPASS_WIDTH + GetSystemMetrics(SM_CXBORDER)*2,
					COMPASS_HEIGHT + GetSystemMetrics(SM_CYBORDER)*2,
					TRUE);

				GetWindowRect(wndCompass, &rect);
				reg_CompassX = rect.left;
				reg_CompassY = rect.top;

				// Redraw window behind moving compass window
				RedrawWindow(wndMain, NULL, NULL, RDW_UPDATENOW);
			}

			last_selected = selected;

			ReleaseDC(wndCompass, dcCompass);

			return 0;

		case WM_MOUSEACTIVATE:
                        /* Pass the mouse message, but don't activate the
                           compass window
                        */
			return MA_NOACTIVATE;

		case WM_LBUTTONDOWN:
			dragpoint.x = LOWORD(lParam);
			dragpoint.y = HIWORD(lParam);

			// CE doesn't track an unpressed pointer
			if (!dragging)
			{
				selected = GetGridSelection(dragpoint.x, dragpoint.y);

				if (selected != -1)
				{
					rect.left = grid[selected][0];
					rect.top = grid[selected][1];
					rect.right = grid[selected][2];
					rect.bottom = grid[selected][3];

					/* Draw the solid foreground for the
					   new area
					*/
					dcCompass = GetDC(wndCompass);
					BitBlt(dcCompass,
						rect.left, rect.top,
						rect.right-rect.left,
						rect.bottom-rect.top,
						dcSolid,
						rect.left, rect.top,
						SRCCOPY);
				}
			}

			/* If no direction is highlighted */
			if (selected==-1)
			{
				SetCapture(wndCompass); /* until further notice */
				dragging = true;
			}

			/* Otherwise, send the direction to the input line
			   (only if we're at an input, of course)
			*/
			else if (during_player_input)
			{
				TypeCommand(compass_point[selected], true, true);
				dragging = false;
			}
			return 0;

		case WM_LBUTTONUP:
		{
			// CE doesn't track an unpressed pointer
			if (!dragging)
			{
				dcCompass = GetDC(wndCompass);
				BitBlt(dcCompass,
					rect.left, rect.top,
					rect.right-rect.left,
					rect.bottom-rect.top,
					dcEmbossed,
					rect.left, rect.top,
					SRCCOPY);
			}

			// Needed for wndVideo/XP:
			if (video_running)
			{
				RECT rect;
				GetClientRect(wndMain, &rect);
				InvalidateRect(wndMain, &rect, false);
			}

			ReleaseCapture();
			dragging = false;
			return 0;
		}

		case WM_DESTROY:
			// Needed for wndVideo/XP:
			if (video_running)
			{
				RECT rect;
				GetClientRect(wndMain, &rect);
				InvalidateRect(wndMain, &rect, false);
			}
			DeleteDC(dcEmbossed);
			DeleteDC(dcSolid);
			DeleteObject(bmpEmbossed);
			DeleteObject(bmpSolid);
			return 0;
	}
	return DefWindowProc(hwnd, iMsg, wParam, lParam);
}


/* GetGridSelection

	Returns the number (0-12) of the selected area on the compass map
	(as per the grid[] array).  Returns -1 if the point (x, y) is
	not in a specific area.
*/

int GetGridSelection(int x, int y)
{
	int i;

	for (i=0; i<12; i++)
	{
		if (x>=grid[i][0] && y>=grid[i][1] &&
			x<=grid[i][2] && y<=grid[i][3])
		{
			return i;
		}
	}

	return -1;
}

#endif	// !NO_COMPASS


/*
 * Scrollback window:
 *
 */

#ifdef SCROLLBACK_DEFINED

/* hugo_sendtoscrollback

	The only routine in hewin.c directly called by the engine in order to
	copy printed text to the scrollback window buffer.
*/

void hugo_sendtoscrollback(char *a)
{
	int i;

	/* If the scrollback buffer is getting close to running out of room,
	   shift it backward, truncating at the head of the buffer
	*/
	if (scrollback_pos >= SCROLLBACK_SIZE-(int)strlen(a))
	{
		_tcscpy(scrollback_buffer, scrollback_buffer+2048);
		scrollback_pos -= 2048;
	}

	for (i=0; i<(int)strlen(a); i++)
	{
		if (a[i]=='\n')
		{
			scrollback_buffer[scrollback_pos++] = L'\r';
			scrollback_buffer[scrollback_pos++] = L'\n';
		}

		else if ((unsigned char)a[i]>=' ')
			scrollback_buffer[scrollback_pos++] = a[i];
	}
	scrollback_buffer[scrollback_pos] = '\0';
}


/* ScrollbackDialog

	Callback for the "Scrollback Window" dialog.
*/

BOOL CALLBACK ScrollbackDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	static HWND hwndEdit;
	RECT rect;
	POINT point;
	static DWORD start_pos, end_pos, visible_line;  /* added by DGT Nov, 1999 - used in focus messages on hwndEdit */

	switch (uMsg)
	{
		case WM_INITDIALOG:
			/* Position the dialog itself so that it covers the
			   client area of the main window
			*/
			GetWindowRect(wndMain, &rect);
#ifndef UNDER_CE
			point.x = 0, point.y = 0;
			ClientToScreen(wndMain, &point);
			point.x -= GetSystemMetrics(SM_CXSIZEFRAME);
#else
			/* Have to be careful that dialog position doesn't get shifted
			   up by SIP/keyboard, or else we'll never be able to close it.
			*/
			point.x = 0;
			point.y = MENU_HEIGHT;
			rect.bottom = DEVICE_SCREEN_HEIGHT;
#endif	// UNDER_CE

			SetWindowPos(hwndDlg, reg_FullScreen?HWND_TOPMOST:HWND_TOP,
				point.x, point.y,
				rect.right - point.x,
				rect.bottom - point.y,
				0);			// flags
			/* Position the edit box so that it stretches to
		      	   the bottom-right of the dialog client area
			   - moved to WM_SIZE by DGT, Nov 1999
			*/
			hwndEdit = GetDlgItem(hwndDlg, HE_SCROLLBACK_EDIT);

#ifndef UNDER_CE
			SendMessage(hwndDlg, WM_SETICON, ICON_SMALL, GetClassLong(wndMain, GCL_HICONSM));

			/* Set the scrollback font */
			SendMessage(hwndEdit, WM_SETFONT,
				(WPARAM)GetStockObject(ANSI_VAR_FONT), 0);
#endif

			/* Load the scrollback buffer text */
			scrollback_buffer[scrollback_pos] = '\0';
			SendMessage(hwndEdit, WM_SETTEXT,
				0, (LPARAM)scrollback_buffer);

			/* scroll the edit box all the way to the end */
			SendMessage(hwndEdit, EM_LINESCROLL, 0, SendMessage(hwndEdit, EM_GETLINECOUNT, 0, 0));
			/* Position the caret at the end of the buffer */
			SendMessage(hwndEdit, EM_SETSEL, SCROLLBACK_SIZE, SCROLLBACK_SIZE);

			/* added by DGT, Nov 1999 - initialize variables */
			start_pos = -1;
			end_pos = 0;
			visible_line = SendMessage(hwndEdit, EM_GETFIRSTVISIBLELINE, 0, 0);

			SetFocus(hwndEdit);

// Need to fall through under CE because the WM_SIZE message never gets sent?
#ifndef UNDER_CE
			return 0;	/* since focus was set */
#endif

		case WM_SIZE:
			/* Position the edit box so that it stretches to
		      	   the bottom-right of the dialog client area
			   - moved from WM_INITDIALOG by DGT, Nov 1999
			*/
			GetClientRect(hwndDlg, &rect);
#ifndef UNDER_CE
			rect.top = 36;		// arbitrary
#else
			rect.bottom -= 26;	// MENU_HEIGHT
#endif
			SetWindowPos(hwndEdit, HWND_TOP,
				0, rect.top,
				rect.right - rect.left,
				rect.bottom - rect.top,
				0);
			return 0;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case HE_SCROLLBACK_COPY:
					SendMessage(GetDlgItem(hwndDlg, HE_SCROLLBACK_EDIT), WM_COPY, 0, 0);
					return 0;

				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					return 0;

				/*
				** added by DGT, Nov 1999
				** don't select everything by default when the
				** edit control gets focus
				*/
#ifndef UNDER_CE
				case HE_SCROLLBACK_EDIT:
					switch (HIWORD(wParam))
					{
						case EN_KILLFOCUS:
							SendMessage(hwndEdit, EM_GETSEL, (WPARAM) &start_pos, (LPARAM) &end_pos);
							visible_line = SendMessage(hwndEdit, EM_GETFIRSTVISIBLELINE, 0, 0);
							return 0;

						case EN_SETFOCUS:
							LockWindowUpdate(hwndEdit);
							SendMessage(hwndEdit, EM_SETSEL, start_pos, end_pos);
							SendMessage(hwndEdit, EM_LINESCROLL, 0, (visible_line - SendMessage(hwndEdit, EM_GETFIRSTVISIBLELINE, 0, 0)));
							LockWindowUpdate(NULL);
							return 0;
					}
#endif
			}
			break;

		case WM_CLOSE:
			EndDialog(hwndDlg, 0);
			return 0;
	}

	return 0;
}

#endif	// SCROLLBACK_DEFINED


/*
 * Setup dialog:
 *
 */

#ifndef UNDER_CE

int CALLBACK BrowseCallbackProc(HWND hwnd, UINT uMsg, LPARAM lParam, LPARAM lpData)
{
	switch (uMsg)
	{
		/* Select the default folder on init */
		case BFFM_INITIALIZED:
			SendMessage(hwnd, BFFM_SETSELECTION, TRUE, lpData);
			break;
	}
	return 0;
}


/* SetupDialog

	Callback for the "Setup" dialog.
*/

BOOL CALLBACK SetupDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
			SetDlgItemText(hwndDlg, HE_SETUP_GAMES, reg_GameDir);
			SetDlgItemText(hwndDlg, HE_SETUP_SAVE, reg_SaveDir);
			CheckDlgButton(hwndDlg, HE_SETUP_LATIN1,
				reg_AutoTranslateLatin1?BST_CHECKED:BST_UNCHECKED);
			return 1;	/* since focus wasn't set */

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDOK:
					GetDlgItemText(hwndDlg, HE_SETUP_GAMES, reg_GameDir, MAXPATH-1);
					GetDlgItemText(hwndDlg, HE_SETUP_SAVE, reg_SaveDir, MAXPATH-1);
					SetCurrentDirectories();

					reg_AutoTranslateLatin1 = IsDlgButtonChecked(hwndDlg, HE_SETUP_LATIN1);

					EndDialog(hwndDlg, 1);
					return 0;

				case HE_BROWSE_BUTTON:
				case HE_BROWSE_BUTTON2:
				{
					BROWSEINFO bi;
					LPITEMIDLIST pidlBrowse, pidlStart = NULL;
					LPMALLOC g_pMalloc;

					/* Get shell's global malloc interface */
					SHGetMalloc(&g_pMalloc);

					bi.hwndOwner = hwndDlg;
					bi.ulFlags = 0;
					bi.lpfn = BrowseCallbackProc;

					strcpy(line, "Select default folder for ");
					
					if (wParam==HE_BROWSE_BUTTON)
					{
						bi.pszDisplayName = reg_GameDir;
						strcat(line, "Game Files");
					}
					else
					{
						bi.pszDisplayName = reg_SaveDir;
						strcat(line, "Saved Games");
					}

					bi.lParam = (LPARAM)bi.pszDisplayName;
#ifdef UNICODE
					strcpy_AtoU(unicode_buffer, line);
					bi.lpszTitle = unicode_buffer;
#else
					bi.lpszTitle = line;
#endif
					bi.pidlRoot = pidlStart;
					// Must have called CoInitialize() to
					// use BIF_USENEWUI
					//bi.ulFlags = BIF_EDITBOX | BIF_USENEWUI;
					bi.ulFlags = BIF_EDITBOX | 0x40;
					pidlBrowse = SHBrowseForFolder(&bi);

					if (pidlBrowse)
					{
						SHGetPathFromIDList(pidlBrowse, bi.pszDisplayName);

						/* Free the ID pointer */
						g_pMalloc->lpVtbl->Free(g_pMalloc, pidlBrowse); 

						/* Reset editbox text */
						SendMessage(hwndDlg, WM_INITDIALOG, 0, 0);
					}

					return 0;
				}

				case HE_CLEARFILES_BUTTON:
				{
					int i;
					for (i=0; i<MRU_FILE_COUNT; i++)
						strcpy(MRU_filename[i], "");
					return 0;
				}

				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					return 0;
			}
			return 0;

		case WM_CLOSE:
			EndDialog(hwndDlg, 0);
			return 0;
	}

	return 0;
}

#endif


/*
 * Context-command menu handling:
 *
 */

#if !defined (COMPILE_V25)

HMENU menuContext;

int BuildContextMenu(HMENU hmenu)
{
	int i, count = 0;

	for (i=0; i<context_commands; i++)
	{
		// separator
		if (context_command[i][0]=='-')
		{
			InsertMenu(hmenu, i,
				MF_BYPOSITION | MF_SEPARATOR,
				0, NULL);
			count++;
		}

		// normal menu item
		else
		{
			context_command[i][0] = toupper(context_command[i][0]);
#ifdef UNICODE
			strcpy_AtoU(unicode_buffer, context_command[i]);
			InsertMenu(hmenu, i,
				MF_BYPOSITION | MF_STRING,
				_APS_NEXT_COMMAND_VALUE+i,
				unicode_buffer);
#else
			InsertMenu(hmenu, i,
				MF_BYPOSITION | MF_STRING,
				_APS_NEXT_COMMAND_VALUE+i,
				context_command[i]);
#endif
			count++;
		}
	}

	return count;
}

int HandleContextCommands(int x, int y)
{
	int result;
	MSG msg;

	if (!getline_active) return 0;

	if (context_commands==0) return 0;
	if (!(menuContext = CreatePopupMenu())) return 0;

	BuildContextMenu(menuContext);

	result = TrackPopupMenu(menuContext,
// No difference?
//		TPM_LEFTBUTTON | TPM_RIGHTBUTTON,
		TPM_RIGHTBUTTON,
		x, y,
		0, wndMain, NULL);

	// Without this, we won't be able to dismiss the
	// popup menu by clicking outside it--it'll just
	// call up another popup menu
	PeekMessage(&msg, wndMain,
		WM_LBUTTONDOWN, WM_LBUTTONDOWN, PM_REMOVE);

	DestroyMenu(menuContext);

	return result;
}

#endif	// !defined(COMPILE_V25)


/*
 * Hugo Engine registry management:
 *
 */

/* LoadRegistryDefaults */

void LoadRegistryDefaults(void)
{
	HKEY hkey;
	DWORD type;
	unsigned int bottom = 0, right = 0;
	size_t size;
	int i;
	TCHAR str[16];

#if defined (DEBUGGER)
	if (RegOpenKeyEx(HKEY_CURRENT_USER,
		_T("Software\\General Coffee Co.\\Hugo\\Debugger\\Engine"), 0, KEY_READ,
		&hkey)==ERROR_SUCCESS)
#else
	if (RegOpenKeyEx(HKEY_CURRENT_USER,
		_T("Software\\General Coffee Co.\\Hugo\\Engine"), 0, KEY_READ,
		&hkey)==ERROR_SUCCESS)
#endif
	{
		/* Values are grouped by type and type size */

		/* REG_DWORDs stored as ints, first: */
		type = REG_DWORD;
		size = sizeof(int);
#ifndef UNDER_CE
		RegQueryValueEx(hkey, _T("xPos"), 0, &type, (LPBYTE)&reg_xPos, &size);
		RegQueryValueEx(hkey, _T("yPos"), 0, &type, (LPBYTE)&reg_yPos, &size);
		RegQueryValueEx(hkey, _T("Width"), 0, &type, (LPBYTE)&reg_Width, &size);
		RegQueryValueEx(hkey, _T("Height"), 0, &type, (LPBYTE)&reg_Height, &size);

		RegQueryValueEx(hkey, _T("FixedCharset"), 0, &type,
			(LPBYTE)&reg_FixedCharset, &size);
		RegQueryValueEx(hkey, _T("FixedFlags"), 0, &type,
			(LPBYTE)&reg_FixedFlags, &size);
		RegQueryValueEx(hkey, _T("PropCharset"), 0, &type,
			(LPBYTE)&reg_PropCharset, &size);
		RegQueryValueEx(hkey, _T("PropFlags"), 0, &type,
			(LPBYTE)&reg_PropFlags, &size);

		RegQueryValueEx(hkey, _T("DefForeground"), 0, &type,
			(LPBYTE)&reg_DefForeground, &size);
		RegQueryValueEx(hkey, _T("DefBackground"), 0, &type,
			(LPBYTE)&reg_DefBackground, &size);
		RegQueryValueEx(hkey, _T("DefSLForeground"), 0, &type,
			(LPBYTE)&reg_DefSLForeground, &size);
		RegQueryValueEx(hkey, _T("DefSLBackground"), 0, &type,
			(LPBYTE)&reg_DefSLBackground, &size);
		RegQueryValueEx(hkey, _T("FullScreen"), 0, &type,
			(LPBYTE)&reg_FullScreen, &size);
		RegQueryValueEx(hkey, _T("Maximized"), 0, &type,
			(LPBYTE)&reg_Maximized, &size);

#ifdef USE_TEXTBUFFER
		RegQueryValueEx(hkey, _T("TextSelection"), 0, &type,
			(LPBYTE)&i, &size);
		allow_text_selection = i;
#endif

		RegQueryValueEx(hkey, _T("ShowCompass"), 0, &type,
			(LPBYTE)&reg_ShowCompass, &size);
		RegQueryValueEx(hkey, _T("CompassX"), 0, &type,
			(LPBYTE)&reg_CompassX, &size);
		RegQueryValueEx(hkey, _T("CompassY"), 0, &type,
			(LPBYTE)&reg_CompassY, &size);
		RegQueryValueEx(hkey, _T("AutoTranslateLatin1"), 0, &type,
			(LPBYTE)&reg_AutoTranslateLatin1, &size);

		/* REG_BINARY data next: */
		type = REG_BINARY;
		size = sizeof(reg_CustomColors);
		RegQueryValueEx(hkey, _T("CustomColors"), 0, &type,
			(LPBYTE)&reg_CustomColors, &size);

		/* REG_SZ strings next: */
		type = REG_SZ;
		size = LF_FACESIZE;
		RegQueryValueEx(hkey, _T("FixedFont"), 0, &type,
			(LPBYTE)&reg_FixedFont, &size);
		size = LF_FACESIZE;	/* have to reset this */
		RegQueryValueEx(hkey, _T("PropFont"), 0, &type,
			(LPBYTE)&reg_PropFont, &size);

		size = sizeof(reg_GameDir);
		RegQueryValueEx(hkey, _T("GameDir"), 0, &type,
			(LPBYTE)&reg_GameDir, &size);
		size = sizeof(reg_SaveDir);
		RegQueryValueEx(hkey, _T("SaveDir"), 0, &type,
			(LPBYTE)&reg_SaveDir, &size);

		/* Now tweak the window position to make sure it's on
		   the screen, etc.
		*/
		bottom = GetSystemMetrics(SM_CYSCREEN);
		right = GetSystemMetrics(SM_CXSCREEN);
		if (reg_Width!=CW_USEDEFAULT && (unsigned)reg_Width > right)
			reg_Width = right;
		if (reg_Height!=CW_USEDEFAULT && (unsigned)reg_Height > bottom)
			reg_Height = bottom;
		if (reg_xPos!=CW_USEDEFAULT && (unsigned)reg_xPos+(unsigned)reg_Width > right)
			reg_xPos = right-(unsigned)reg_Width;
		if (reg_yPos!=CW_USEDEFAULT && (unsigned)reg_yPos+(unsigned)reg_Height > bottom)
			reg_yPos = bottom-(unsigned)reg_Height;

#else	// UNDER_CE

		type = REG_DWORD;
		size = sizeof(int);
		RegQueryValueEx(hkey, _T("MinimalWindows"), 0, &type,
			(LPBYTE)&reg_MinimalWindows, &size);
		RegQueryValueEx(hkey, _T("NoBlankLines"), 0, &type,
			(LPBYTE)&reg_NoBlankLines, &size);
		RegQueryValueEx(hkey, _T("UseColors"), 0, &type,
			(LPBYTE)&reg_UseColors, &size);

#endif

		// Both Win32 and WinCE:

		type = REG_DWORD;
		size = sizeof(int);
		RegQueryValueEx(hkey, _T("FixedPoint"), 0, &type,
			(LPBYTE)&reg_FixedPoint, &size);
		RegQueryValueEx(hkey, _T("PropPoint"), 0, &type,
			(LPBYTE)&reg_PropPoint, &size);

#ifdef USE_SMARTFORMATTING
		RegQueryValueEx(hkey, _T("SmartFormatting"), 0, &type,
			(LPBYTE)&smartformatting, &size);
#endif
		RegQueryValueEx(hkey, _T("FastScrolling"), 0, &type,
			(LPBYTE)&reg_FastScrolling, &size);
		RegQueryValueEx(hkey, _T("DisplayGraphics"), 0, &type,
			(LPBYTE)&reg_DisplayGraphics, &size);
		RegQueryValueEx(hkey, _T("GraphicsSmoothing"), 0, &type,
			(LPBYTE)&reg_GraphicsSmoothing, &size);
		RegQueryValueEx(hkey, _T("PlaySounds"), 0, &type,
			(LPBYTE)&reg_PlaySounds, &size);

#ifdef SPEECH_ENABLED
		RegQueryValueEx(hkey, _T("SpeakText"), 0, &type,
			(LPBYTE)&speech_speaking, &size);
		RegQueryValueEx(hkey, _T("SuppressSounds"), 0, &type,
			(LPBYTE)&speech_suppress_sounds, &size);
		RegQueryValueEx(hkey, _T("SpeakStatus"), 0, &type,
			(LPBYTE)&speech_speak_status, &size);
		type = REG_BINARY;
		size = sizeof(speech_current_voice);
		RegQueryValueEx(hkey, _T("CurrentVoice"), 0, &type,
			(LPBYTE)&speech_current_voice, &size);
#endif
		for (i=0; i<MRU_FILE_COUNT; i++)
		{
			size = sizeof(MRU_filename[i]);
			_stprintf(str, _T("RecentFile%d"), i+1);
			RegQueryValueEx(hkey, str, 0, &type,
				(LPBYTE)&MRU_filename[i], &size);
		}

		RegCloseKey(hkey);

		SetCurrentDirectories();
	}
}


/* SaveRegistryDefaults */

void SaveRegistryDefaults(void)
{
	HKEY hkey;
	int version;
	int i;
	TCHAR str[16];

#if defined (DEBUGGER)
	if (RegCreateKeyEx(HKEY_CURRENT_USER,
		_T("Software\\General Coffee Co.\\Hugo\\Debugger\\Engine"),
		0, NULL, 0, KEY_WRITE, NULL, &hkey, NULL)==ERROR_SUCCESS)
#else
	if (RegCreateKeyEx(HKEY_CURRENT_USER,
		_T("Software\\General Coffee Co.\\Hugo\\Engine"),
		0, NULL, 0, KEY_WRITE, NULL, &hkey, NULL)==ERROR_SUCCESS)
#endif
	{
		/* Save the version in case we ever need it */
		version = HEVERSION*10 + HEREVISION;
		RegSetValueEx(hkey, _T("Version"), 0, REG_DWORD,
			(LPBYTE)&version, sizeof(version));
#ifndef UNDER_CE
		RegSetValueEx(hkey, _T("xPos"), 0, REG_DWORD,
			(LPBYTE)&reg_xPos, sizeof(reg_xPos));
		RegSetValueEx(hkey, _T("yPos"), 0, REG_DWORD,
			(LPBYTE)&reg_yPos, sizeof(reg_yPos));
		RegSetValueEx(hkey, _T("Width"), 0, REG_DWORD,
			(LPBYTE)&reg_Width, sizeof(reg_Width));
		RegSetValueEx(hkey, _T("Height"), 0, REG_DWORD,
			(LPBYTE)&reg_Height, sizeof(reg_Height));

		RegSetValueEx(hkey, _T("FixedFont"), 0, REG_SZ,
			(LPBYTE)reg_FixedFont, (_tcslen(reg_FixedFont)+1)*sizeof(TCHAR));
		RegSetValueEx(hkey, _T("FixedCharset"), 0, REG_DWORD,
			(LPBYTE)&reg_FixedCharset, sizeof(reg_FixedCharset));
		RegSetValueEx(hkey, _T("FixedFlags"), 0, REG_DWORD,
			(LPBYTE)&reg_FixedFlags, sizeof(reg_FixedFlags));

		RegSetValueEx(hkey, _T("PropFont"), 0, REG_SZ,
			(LPBYTE)reg_PropFont, (_tcslen(reg_PropFont)+1)*sizeof(TCHAR));
		RegSetValueEx(hkey, _T("PropCharset"), 0, REG_DWORD,
			(LPBYTE)&reg_PropCharset, sizeof(reg_PropCharset));
		RegSetValueEx(hkey, _T("PropFlags"), 0, REG_DWORD,
			(LPBYTE)&reg_PropFlags, sizeof(reg_PropFlags));

		RegSetValueEx(hkey, _T("DefForeground"), 0, REG_DWORD,
			(LPBYTE)&reg_DefForeground, sizeof(reg_DefForeground));
		RegSetValueEx(hkey, _T("DefBackground"), 0, REG_DWORD,
			(LPBYTE)&reg_DefBackground, sizeof(reg_DefBackground));
		RegSetValueEx(hkey, _T("DefSLForeground"), 0, REG_DWORD,
			(LPBYTE)&reg_DefSLForeground, sizeof(reg_DefSLForeground));
		RegSetValueEx(hkey, _T("DefSLBackground"), 0, REG_DWORD,
			(LPBYTE)&reg_DefSLBackground, sizeof(reg_DefSLBackground));

		RegSetValueEx(hkey, _T("FullScreen"), 0, REG_DWORD,
			(LPBYTE)&reg_FullScreen, sizeof(reg_FullScreen));
		RegSetValueEx(hkey, _T("Maximized"), 0, REG_DWORD,
			(LPBYTE)&reg_Maximized, sizeof(reg_Maximized));
#ifdef USE_TEXTBUFFER
		i = allow_text_selection;
#else
		i = 0;
#endif
		RegSetValueEx(hkey, _T("TextSelection"), 0, REG_DWORD,
			(LPBYTE)&i, sizeof(i));

		RegSetValueEx(hkey, _T("ShowCompass"), 0, REG_DWORD,
			(LPBYTE)&reg_ShowCompass, sizeof(reg_ShowCompass));
		RegSetValueEx(hkey, _T("CompassX"), 0, REG_DWORD,
			(LPBYTE)&reg_CompassX, sizeof(reg_CompassX));
		RegSetValueEx(hkey, _T("CompassY"), 0, REG_DWORD,
			(LPBYTE)&reg_CompassY, sizeof(reg_CompassY));

		RegSetValueEx(hkey, _T("GameDir"), 0, REG_SZ,
			(LPBYTE)reg_GameDir, _tcslen(reg_GameDir)+1);
		RegSetValueEx(hkey, _T("SaveDir"), 0, REG_SZ,
			(LPBYTE)reg_SaveDir, _tcslen(reg_SaveDir)+1);

		RegSetValueEx(hkey, _T("AutoTranslateLatin1"), 0, REG_DWORD,
			(LPBYTE)&reg_AutoTranslateLatin1, sizeof(reg_AutoTranslateLatin1));

		RegSetValueEx(hkey, _T("CustomColors"), 0, REG_BINARY,
			(LPBYTE)&reg_CustomColors, sizeof(reg_CustomColors));

#else	// UNDER_CE

		RegSetValueEx(hkey, _T("MinimalWindows"), 0, REG_DWORD,
			(LPBYTE)&reg_MinimalWindows, sizeof(reg_MinimalWindows));
		RegSetValueEx(hkey, _T("NoBlankLines"), 0, REG_DWORD,
			(LPBYTE)&reg_NoBlankLines, sizeof(reg_NoBlankLines));
		RegSetValueEx(hkey, _T("UseColors"), 0, REG_DWORD,
			(LPBYTE)&reg_UseColors, sizeof(reg_UseColors));
#endif

		// Both Win32 and WinCE:

		RegSetValueEx(hkey, _T("FixedPoint"), 0, REG_DWORD,
			(LPBYTE)&reg_FixedPoint, sizeof(reg_FixedPoint));
		RegSetValueEx(hkey, _T("PropPoint"), 0, REG_DWORD,
			(LPBYTE)&reg_PropPoint, sizeof(reg_PropPoint));

#ifdef USE_SMARTFORMATTING
		RegSetValueEx(hkey, _T("SmartFormatting"), 0, REG_DWORD,
			(LPBYTE)&smartformatting, sizeof(smartformatting));
#endif
		RegSetValueEx(hkey, _T("FastScrolling"), 0, REG_DWORD,
			(LPBYTE)&reg_FastScrolling, sizeof(reg_FastScrolling));
		RegSetValueEx(hkey, _T("DisplayGraphics"), 0, REG_DWORD,
			(LPBYTE)&reg_DisplayGraphics, sizeof(reg_DisplayGraphics));
		RegSetValueEx(hkey, _T("GraphicsSmoothing"), 0, REG_DWORD,
			(LPBYTE)&reg_GraphicsSmoothing, sizeof(reg_GraphicsSmoothing));
		RegSetValueEx(hkey, _T("PlaySounds"), 0, REG_DWORD,
			(LPBYTE)&reg_PlaySounds, sizeof(reg_PlaySounds));

#ifdef SPEECH_ENABLED
		RegSetValueEx(hkey, _T("SpeakText"), 0, REG_DWORD,
			(LPBYTE)&speech_speaking, sizeof(speech_speaking));
		RegSetValueEx(hkey, _T("SuppressSounds"), 0, REG_DWORD,
			(LPBYTE)&speech_suppress_sounds, sizeof(speech_suppress_sounds));
		RegSetValueEx(hkey, _T("SpeakStatus"), 0, REG_DWORD,
			(LPBYTE)&speech_speak_status, sizeof(speech_speak_status));
		RegSetValueEx(hkey, _T("CurrentVoice"), 0, REG_BINARY,
			(LPBYTE)&speech_current_voice, sizeof(speech_current_voice));
#endif
		for (i=0; i<MRU_FILE_COUNT; i++)
		{
			_stprintf(str, _T("RecentFile%d"), i+1);
			RegSetValueEx(hkey, str, 0, REG_SZ,
				(LPBYTE)MRU_filename[i], (_tcslen(MRU_filename[i])+1)*sizeof(TCHAR));
		}

		RegCloseKey(hkey);

	}

#ifdef UNDER_CE
	SaveDefaultCESettingsFile();
#endif

	/* Do this here, since we're shutting down anyway.  NT 3.51
	   (at least) seems to strand the compass window--why, I
	   don't know
	*/
#ifndef NO_COMPASS
		if (wndCompass) DestroyWindow(wndCompass);
#endif
}


/* SetCurrentDirectories */

void SetCurrentDirectories(void)
{
	_tcscpy(current_games_dir, reg_GameDir);
	SetEnvironmentVariable(_T("HUGO_GAMES"), reg_GameDir);
	SetEnvironmentVariable(_T("HUGO_OBJECT"), reg_GameDir);

	_tcscpy(current_save_dir, reg_SaveDir);
	SetEnvironmentVariable(_T("HUGO_SAVE"), reg_GameDir);
}


/* TryToOpenHandle

	Like TrytoOpen() in hemisc.c, except for a HANDLE instead
	of a FILE.
	
	Tries to open a particular filename (based on a given environment
	variable), trying the current directory first.
*/

HANDLE TryToOpenHandle(char *f, char *d)
{
	char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];
	char envvar[32];
	HANDLE tempfile; char temppath[MAXPATH];

	/* Try to open the given, vanilla filename */
#ifdef UNICODE
	strcpy_AtoU(unicode_buffer, f);
	if ((tempfile = CreateFile(unicode_buffer, GENERIC_READ, FILE_SHARE_READ, NULL,
		OPEN_EXISTING, 0, NULL)) != INVALID_HANDLE_VALUE)
#else
	if ((tempfile = CreateFile(f, GENERIC_READ, FILE_SHARE_READ, NULL,
		OPEN_EXISTING, 0, NULL)) != INVALID_HANDLE_VALUE)
#endif
	{
		return tempfile;
	}
	
	
	hugo_splitpath(f, drive, dir, fname, ext);      /* file to open */

	/* If the given filename doesn't already specify where to find it */
	if (!strcmp(drive, "") && !strcmp(dir, ""))
	{
		/* Check gamefile directory */
		hugo_makepath(temppath, "", gamepath, fname, ext);
#ifdef UNICODE
		strcpy_AtoU(unicode_buffer, temppath);
		if ((tempfile = CreateFile(unicode_buffer, GENERIC_READ, FILE_SHARE_READ, NULL,
			OPEN_EXISTING, 0, NULL)) != INVALID_HANDLE_VALUE)
#else
		if ((tempfile = CreateFile(temppath, GENERIC_READ, FILE_SHARE_READ, NULL,
			OPEN_EXISTING, 0, NULL)) != INVALID_HANDLE_VALUE)
#endif
		{
			strcpy(f, temppath);    /* the new pathname */
			return tempfile;
		}

		/* Check environment variables */
		strcpy(envvar, "hugo_");        /* the actual var. name */
		strcat(envvar, d);

		if (getenv(strupr(envvar)))
		{
			hugo_makepath(temppath, "", getenv(strupr(envvar)), fname, ext);

#ifdef UNICODE
			strcpy_AtoU(unicode_buffer, temppath);
			if ((tempfile = CreateFile(unicode_buffer, GENERIC_READ, FILE_SHARE_READ, NULL,
				OPEN_EXISTING, 0, NULL)) != INVALID_HANDLE_VALUE)
#else
			if ((tempfile = CreateFile(temppath, GENERIC_READ, FILE_SHARE_READ, NULL,
				OPEN_EXISTING, 0, NULL)) != INVALID_HANDLE_VALUE)
#endif
			{
				strcpy(f, temppath);  /* the new pathname */
				return tempfile;
			}
		}
	}

	return INVALID_HANDLE_VALUE;	/* not openable */
}


#ifdef UNICODE

/* Unicode utilities */

char *strcpy_UtoA(char *a, WCHAR *u)
{
	int i = 0;
	while (u[i])
	{
		a[i] = (char)u[i];
		i++;
	}
	a[i] = 0;
	return a;
}

WCHAR *strcpy_AtoU(WCHAR *u, char *a)
{
	int i = 0;
	while (a[i])
	{
		u[i] = (WCHAR)a[i];
		i++;
	}
	u[i] = 0;
	return u;
}

#endif
