/*
	HESOUND_NT.C
	(Old-style sound factory not using DirectSound, since older NT's
	ability to get along with DirectX is poor at best.  Everything
	in here is called from hesound.c, i.e., NT_InitPlayer() is
	called by InitPlayer in hesound.c.

	Music (MOD, S3M, XM format modules) and sound (WAV sample)
	playing routines for LCC-Win32:

		hugo_playmusic
		hugo_stopmusic
		hugo_playsample
		hugo_stopsample

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	Both hugo_playmusic() and hugo_playsample() are called with the
	file hot--i.e., opened and positioned to the start of the resource.

	NOTE:  The version of the MikMod library used here is 2.10.
*/

#include <windows.h>
#undef GetProp		// from win.h
#include <mmsystem.h>

#include "heheader.h"
#include "mikmod.h"


int NT_hugo_playmusic(HUGO_FILE infile, long reslength, char loop_flag);
void NT_hugo_musicvolume(int vol);
void NT_hugo_stopmusic(void);
int NT_hugo_playsample(HUGO_FILE infile, long reslength, char loop_flag);
void NT_hugo_samplevolume(int vol);
void NT_hugo_stopsample(void);

/* Now some originals from hesound.c (which will be directed back to
   functions here, but saves us having to hack up the code even more:
*/
int hugo_playsample(FILE *infile, long reslength, char loop_flag);
void hugo_stopsample(void);

int NT_InitPlayer(void);
static void tickhandler(void);
static int SoundDriverError(int retry);
void WaitForMusic(void);
void ResumeSample(void);

/* From heres.c: */
#undef FindResource
long FindResource(char *filename, char *resname);

static UNIMOD *modfile;
static char music_is_playing = false;
static int music_volume = 100;
static MSAMPLE *sample;
static char sample_is_playing = false;
static int sample_volume = 100;
static time_t music_last_started = 0;	/* music restart timing */
static char sound_driver_failed = 0;
char player_started = 0;

static struct
{
	char filename[MAXPATH];
	char resname[MAXPATH];
	char loop_flag;
	char was_playing;
} resume_sample;

/* From heres.c: */
extern FILE *resource_file;
extern char loaded_filename[];
extern char loaded_resname[];

/* From hewin.c: */
extern int reg_PlaySounds;
extern HWND wndMain;
extern char disable_sounds;

/* From MikMod mplayer.c: */
extern int forbid;

#define MAXCHANNELS 31			/* +1 for a sample */


/* InitPlayer

*/

int NT_InitPlayer(void)
{
	ML_RegisterLoader(&load_mod);	/* module loaders */
	ML_RegisterLoader(&load_s3m);
	ML_RegisterLoader(&load_xm);

//	MD_RegisterDriver(&drv_nos);	/* sound drivers */
	MD_RegisterDriver(&drv_w32);

	MD_RegisterPlayer(tickhandler);

	md_mixfreq = 44100;			/* standard mixing frequency */
	md_mode = DMODE_16BITS|DMODE_STEREO;	/* standard mixing mode */
	md_dmabufsize = 2000;			/* standard DMA buffer size */
	md_device = 0;				/* standard device: autodetect */

	if (!MD_Init())			/* initialize driver */
	{
		/* We'll later check sound_driver_failed when we actually
		   try to play a sample or song--don't bother printing any
		   error message until it we've tried to play a sound.
		*/
		sound_driver_failed = true;
		return false;
	}

	/* Start the player here and keep it running globally */
	md_numchn = MAXCHANNELS + 1;	/* +1 for a sample */
// Can do this later so that we don't start sound unnecessarily
//	MD_PlayStart();

	/* Kill the driver for now, having started it--we'll start it again
	   before calling MD_PlayStart()
	*/
	MD_Exit();

	return true;
}


/* SuspendAudio and ResumeAudio stand in for real suspend and
   resume functionality
*/

static int audio_suspended = 0, suspended_volume = 0;

void NT_SuspendAudio(void)
{
	suspended_volume = mp_volume;
	mp_volume = 0;
	hugo_stopsample();
	audio_suspended = true;
}

void NT_ResumeAudio(void)
{
	if (audio_suspended)
	{
		mp_volume = suspended_volume;
		audio_suspended = false;
		ResumeSample();
	}
}


/* tickhandler() is used to manage MikMod's MOD-playing loop */

static void tickhandler(void)
{
	if (disable_sounds) return;

	/* Need to do this here since we can't install an interrupt
       	   in Windows a la the DOS implementation
	*/
// On the other hand, _why_ do we need to call this?
//	if (!forbid and sample_is_playing) MD_Update();

	/* If we don't do this check, MikMod will crash us if there's a
	   sample but no module playing
	*/
	if (!music_is_playing) return;

	MP_HandleTick();    	/* play 1 tick of the module */
	MD_SetBPM(mp_bpm);
}


/* SoundDriverError

	Returns true to keep request to waiting for an audio device.
*/

static int SoundDriverError(int retry)
{
	char *title = "Hugo Engine - Sound Trouble";

	switch (retry)
	{
		case 0:
			MessageBox(wndMain,
				"Unable to initialize the audio device.  \
Another application may be using the device, or it may not be configured \
properly.",
				title,
				MB_ICONEXCLAMATION);
			return 0;

		case 1:
			if (MessageBox(wndMain,
				"Unable to restart the audio device.  \
Another application may be using the device.  Keep waiting?",
				title,
				MB_APPLMODAL |
				MB_ICONEXCLAMATION |
				MB_YESNO)==IDYES)
			{
				return true;
			}
			else
				return false;
	}
	return false;
}


/* hugo_playmusic */

int NT_hugo_playmusic(HUGO_FILE f, long reslength, char loop_flag)
{
	if (disable_sounds)
	{
		/* Wait until we've actually tried to play a sound before
		   saying we couldn't initialize the device
		*/
		if (sound_driver_failed) SoundDriverError(0);
		sound_driver_failed = 0;
		return false;
	}

	if (music_is_playing)
	{
		/* Formerly, this chunk of code was hugo_stopmusic() */
		music_is_playing = false;
		modfile->numpos = 0;

		Sleep(20);		/* wait for the player to stop */
		ML_Free(modfile);

		/* Cheat the last-start-time to be long enough ago to
		   not cause a delay in hugo_playmusic()
		*/
		music_last_started -= 10;
	}

	/* Set the rewind/SEEK_SET reference used by MikMod to the
	   start of the resource
	*/
	_mm_setiobase(ftell(f));

	modfile = ML_LoadFP(f);
	fclose(f);

	if (modfile)
	{
		if (!player_started)
		{
			MD_Init();
			MD_PlayStart();
			player_started = true;
		}

		/* Be sure to wait since the last mod was started to prevent
		   overlapping crashes
		*/
		WaitForMusic();

		MP_Init(modfile);

		/* Force the module to fit into MAXCHANNELS (which shouldn't
		   really be much of a constraint), since we started the
		   driver globally and want to keep it running--i.e., we
		   won't be calling MD_PlayStart() with a new md_numchn
		*/
// Or should modfile->numchn be constrained instead?
		md_numchn = modfile->numchn + 1;	/* +1 for a sample */
		if (md_numchn > MAXCHANNELS+1) md_numchn = MAXCHANNELS+1;
		modfile->numchn = md_numchn - 1;
		
		if (loop_flag)
		{
			forbid = 1;		/* suspend interrupts */
			modfile->reppos = 1;
			mp_loop = 1;
			forbid = 0;		/* restore interrupts */
		}
		else mp_loop = 0;

		mp_volume = music_volume;	/* 0 - 100 */
		if (!reg_PlaySounds) mp_volume = 0;

		music_is_playing = true;
		return true;
	}
#if defined (DEBUGGER)
	else
		DebugMessageBox("Sound Library Error", myerr);
#endif
	return false;
}


/* WaitForMusic

	When switching to/from or stopping/starting music, it usually pays to
	wait a couple of seconds for the buffer to clear out.
*/

void WaitForMusic(void)
{
	time_t t;

	if (music_is_playing)
	{
		do
		{
			time(&t);
			t = t - music_last_started;
		}
		while ((music_last_started) && t < 3);
		time(&music_last_started);
	}
}


/* hugo_musicvolume */

void NT_hugo_musicvolume(int vol)
{
	music_volume = vol;
}


/* hugo_stopmusic */

void NT_hugo_stopmusic(void)
{
	/* Formerly, the actual player-stopping code from hugo_playmusic()
	   went here, but all that's really needed unless you're starting
	   a new song is to pull the song level to 0
	*/
	mp_volume = 0;
}


/* hugo_playsample */

int NT_hugo_playsample(HUGO_FILE f, long reslength, char loop_flag)
{
	int channel;

	if (disable_sounds)
	{
		if (sound_driver_failed) SoundDriverError(0);
		sound_driver_failed = 0;
		return false;
	}

	if (sample_is_playing) hugo_stopsample();

	sample = MW_LoadWavFP(f);
	fclose(f);

	if (sample)
	{
		/* for ResumeSample() */
		strcpy(resume_sample.filename, loaded_filename);
		strcpy(resume_sample.resname, loaded_resname);
		resume_sample.loop_flag = loop_flag;
		
		if (!player_started)
		{
			MD_Init();
			MD_PlayStart();
			player_started = true;
		}

		/* Always play a sample on the last/highest channel */
		channel = md_numchn - 1;

		if (!reg_PlaySounds)
			MD_VoiceSetVolume((UBYTE)channel, 0);
		else
			MD_VoiceSetVolume((UBYTE)channel, 
				(UBYTE)((reg_PlaySounds)?(sample_volume/100)*64:0));
		MD_VoiceSetPanning((UBYTE)channel, 128);
		MD_VoiceSetFrequency((UBYTE)channel, sample->c2spd);

		if (audio_suspended) return true;

		if (loop_flag)
		{
			sample->flags|=SF_LOOP;
			MD_VoicePlay((UBYTE)channel, sample->handle, 0,
				sample->length, 0, sample->length, sample->flags);
		}
		else
		{
			sample->flags&=~SF_LOOP;
			MD_VoicePlay((UBYTE)channel, sample->handle, 0,
				sample->length, 0, 0, sample->flags);
		}

		sample_is_playing = true;
		return true;
	}
#if defined (DEBUGGER)
	else
		DebugMessageBox("Sound Library Error", myerr);
#endif
	return false;
}


/* hugo_samplevolume */

void NT_hugo_samplevolume(int vol)
{
	sample_volume = vol;
}


/* hugo_stopsample */

void NT_hugo_stopsample(void)
{
	if (sample_is_playing)
	{
		resume_sample.was_playing = true;

		sample_is_playing = false;
		sample->length = 0;
		sample->flags&=~SF_LOOP;
		Sleep(20);		/* wait for the sample to stop */
		MW_FreeWav(sample);
	}
	else
		resume_sample.was_playing = false;
}


/* ResumeSample

	Needed because the player won't automatically resume playing a sample
	after switching back to the application or turning sound off/on.
	(This will only be called after hugo_stopsample(), so 
	resume_sample.was_playing will be accurate.)
*/

void ResumeSample(void)
{
	if (resume_sample.was_playing && resume_sample.loop_flag)
	{
		if (!FindResource(resume_sample.filename, resume_sample.resname))
			return;
		hugo_playsample(resource_file, 0, resume_sample.loop_flag);
	}
}
