/*
	HEBLANK.C

	Template for non-portable system-specific functions:

		hugo_blockalloc         hugo_clearfullscreen
		hugo_blockfree          hugo_clearwindow
					hugo_settextmode
		hugo_splitpath          hugo_settextwindow
		hugo_makepath           hugo_settextpos
		hugo_getfilename        hugo_scrollwindowup
		hugo_overwrite          hugo_font
		hugo_closefiles         hugo_settextcolor
					hugo_setbackcolor
		hugo_getkey             hugo_color
		hugo_getline
		hugo_waitforkey         hugo_print
		hugo_iskeywaiting	hugo_charwidth
		hugo_timewait		hugo_textwidth
					hugo_strlen
		hugo_addcommand         hugo_specialchar
		hugo_restorecommand
		hugo_stripaccents

		hugo_init_screen
		hugo_hasgraphics
		hugo_setgametitle
        hugo_setctype
		hugo_cleanup_screen

	for the Hugo Engine

	Copyright (c) 1995-1998 by Kent Tessman

*/

#include "heheader.h"
#include <conio.h>
#include <sys\video.h>      /* EMX library header for v_*() functions */

/* Function prototypes: */

int hugo_color(int c);
int hugo_hasgraphics(void);
void hugo_setctype(int);            /* for setting the cursor */

void hugo_addcommand(void);
void hugo_restorecommand(int n);
void hugo_stripaccents(char *a);
void ConstrainCursor(void);

#if defined (DEBUGGER)
void *AllocMemory(size_t size);         /* from HDMISC.C */
#endif


/* Definitions and variables: */

/* Defined Hugo colors: */
#define HUGO_BLACK         0
#define HUGO_BLUE          1
#define HUGO_GREEN         2
#define HUGO_CYAN          3
#define HUGO_RED           4
#define HUGO_MAGENTA       5
#define HUGO_BROWN         6
#define HUGO_WHITE         7
#define HUGO_DARK_GRAY     8
#define HUGO_LIGHT_BLUE    9
#define HUGO_LIGHT_GREEN   10
#define HUGO_LIGHT_CYAN    11
#define HUGO_LIGHT_RED     12
#define HUGO_LIGHT_MAGENTA 13
#define HUGO_YELLOW        14
#define HUGO_BRIGHT_WHITE  15

#define HOME_KEY	128
#define END_KEY		129
#define DELETE_KEY	130
#define INSERT_KEY	131
#define BACKSPACE_KEY	132
#define CTRL_LEFT_KEY	133
#define CTRL_RIGHT_KEY	134

int insert_mode = true;

int text_windowleft, text_windowtop, text_windowright, text_windowbottom,
    text_windowwidth;
int current_text_row = 1, current_text_col = 1;

char *gline;        /* for v_putline() and v_getline() */


#define ITALIC_COLOR       HUGO_YELLOW
#define UNDERLINE_COLOR    HUGO_LIGHT_BLUE

#define HISTORY_SIZE    16              /* for command-line editing */
int hcount = 0;
char HUGO_PTR *history[HISTORY_SIZE];


/*
    MEMORY ALLOCATION:

    hugo_blockalloc(), and hugo_blockfree() are necessary because not
    all systems (MS-DOS, for one) can handle memory allocation of more
    than 64K.  For most systems, these will simply be normal ANSI
    function calls.
*/

void HUGO_PTR *hugo_blockalloc(long num)
{
	return malloc(num * sizeof(char));
}

void hugo_blockfree(void HUGO_PTR *block)
{
	free(block);
}


/*
    FILENAME MANAGEMENT:

*/

void hugo_splitpath(char *path, char *drive, char *dir, char *fname, char *ext)
{
    char *c;

    strcpy(drive, path);
    c = strchr(drive, ':');
    if (c == NULL) {
        if (strchr(drive, '\\') == drive)
            strcpy(dir, drive + 1);
        else
            strcpy(dir, drive);
        strcpy(drive, "");
    }
    else {
        c = strchr(drive, '\\');
        strcpy(dir, ++c);
        strcpy(--c, "");
    }
    c = strrchr(dir, '\\');
    if (c == NULL) {
        strcpy(fname, dir);
        strcpy(dir, "");
    }
    else {
        strcpy(fname, ++c);
        strcpy(--c, "");
    }
    c = strrchr(fname , '.');
    if (c == NULL)
        strcpy(ext, "");
    else {
        strcpy(ext, ++c);
        strcpy(--c, "");
    }
}

void hugo_makepath(char *path, char *drive, char *dir, char *fname, char *ext)
{
    char *c;
    strcpy(path, "");
    if (strcmp(drive, "")) {
        strcat(path, drive);
        c = strchr(path, 0);
        *c++ = '\\';
        *c = '\0';
    }
    if (strcmp(dir, "")) {
        strcat(path, dir);
        c = strchr(path, 0);
        *c++ = '\\';
        *c = '\0';
    }
    strcat(path, fname);    /* a filename must be provided */
    if (strcmp(ext, "")) {
        strcat(path, ".");
        strcat(path, ext);
    }
}


/* hugo_getfilename */

void hugo_getfilename(char *a, char *b)
{
	unsigned int i, p;

	sprintf(line, "Enter path and filename %s.", a);

	AP(line);

	sprintf(line,"%c(Default is %s): \\;", NO_CONTROLCHAR, b);
	AP(line);

	p = var[prompt];
	var[prompt] = 0;        /* null string */

	RunInput();

	var[prompt] = p;

	remaining = 0;

	strcpy(line, "");
	if (words==0)
		strcpy(line, b);
	else
	{
		for (i=1; i<=(unsigned int)words; i++)
			strcat(line, word[i]);
	}
}


/* hugo_overwrite */

int hugo_overwrite(char *f)
{
	FILE *tempfile;

	tempfile = fopen(f, "rb");
	if (tempfile==NULL)                     /* if file doesn't exist */
		return true;

	fclose(tempfile);

	sprintf(pbuffer, "Overwrite existing \"%s\" (Y or N)?", f);
	RunInput();

	if (words==1 and (!strcmp(strupr(word[1]), "Y") or !strcmp(strupr(word[1]), "YES")))
		return true;

	return false;
}


/* hugo_closefiles */

void hugo_closefiles()
{
	fcloseall();                    /* non-ANSI function */
}


/* hugo_getkey */

/* the integer argument is necessary because _read_kbd(), while
   similar to kbhit(), does not simply 'peek' into the keyboard
   buffer, but always returns the value of a keypress if one is
   waiting (as in hugo_waitforkey or hugo_iskeywaiting, which 
   was eliminated */

int hugo_getkey(int pre)
{
    int b;

    if (pre == -1)
        b = getch();
    else
        b = pre;

	if (b==8) b = BACKSPACE_KEY;

	if (b==0)
	{
		switch (b=getch())
		{
			case 71:		/* Home */
				{b = HOME_KEY; break;}
			case 72:		/* up-arrow */
				{b = 11; break;}
			case 75:		/* left-arrow */
				{b = 8; break;}
			case 77:		/* right-arrow */
				{b = 21; break;}
			case 79:		/* End */
				{b = END_KEY; break;}
			case 80:        	/* down-arrow */
				{b = 10; break;}
			case 82:		/* Insert */
				{b = INSERT_KEY; break;}
			case 83:		/* DEL */
				{b = DELETE_KEY; break;}
			case 115:
				{b = CTRL_LEFT_KEY; break;}
			case 116:
				{b = CTRL_RIGHT_KEY; break;}

			default:                /* anything else */
				b += 256;
		}
	}

	return b;
}


/* hugo_getline */

#define CURRENT_CHAR(c) ((c<(signed)strlen(buffer))?buffer[c]:' ')

void hugo_getline(char *p)
{
	char ch[2];
	int a, b, thiscommand;
	int c;                          /* c is the character being added */
	int x, y, oldx, oldy;

NewPrompt:
	hugo_settextcolor(fcolor);
	hugo_setbackcolor(bgcolor);

	strcpy(buffer, "");
	c = 0;
	thiscommand = hcount;

	oldx = (currentpos+FIXEDCHARWIDTH)/FIXEDCHARWIDTH;
	oldy = currentline;
	y = oldy;
	x = oldx + hugo_strlen(p);

	hugo_print(p);

GetKey:

	hugo_settextpos(x, y);
	if (insert_mode)
		hugo_setctype(1);
	else
		hugo_setctype(6);

	b = hugo_getkey(-1);

	hugo_settextcolor(icolor);
	hugo_setbackcolor(bgcolor);

	/* Now, start key checking */
	switch (b)
	{
#if defined (DEBUGGER)
		case (9):                       /* Tab */
		{
			during_input = true;
			Debugger();
			during_input = false;

			/* If the debugger is stopping execution, quitting,
			   etc., a blank string must be returned in the
			   input buffer.
			*/
			if (debugger_collapsing)
			{
				strcpy(buffer, "");
				if (active_screen==GAME)
					hugo_scrollwindowup();
				return;
			}

			goto GetKey;
		}
#endif

		case (13):                      /* Enter */
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			full = 0;

			if (script) fprintf(script, "%s%s", p, buffer);

			while (_read_kbd(0, 0, 0) != -1);
			hugo_settextpos(1, y + 2);
			hugo_scrollwindowup();
			if (script) fprintf(script, "\n");

			strcpy(buffer, Rtrim(buffer));
			hugo_addcommand();
			hugo_stripaccents(buffer);

			hugo_setctype(1);

			return;
		}
		case INSERT_KEY:
		{
			insert_mode = not insert_mode;
			goto GetKey;
		}
		case BACKSPACE_KEY:
		case DELETE_KEY:
		{
			if (strlen(buffer)>0)
			{
				if (b==BACKSPACE_KEY)
				{
					if (c==0) goto GetKey;
					c--;
				}

				for (a=c; a<=(signed)strlen(buffer); a++)
					buffer[a] = buffer[a+1];

				if (b==BACKSPACE_KEY) x--;

				hugo_settextpos(oldx+hugo_strlen(p), y);
				hugo_print(buffer);
				hugo_settextpos(oldx+hugo_strlen(p)+strlen(buffer), y);
				hugo_print("  ");
			}
			goto GetKey;
		}
		case (8):                       /* left-arrow */
		{
			if (c > 0)
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x--, c--;
			}
			goto GetKey;
		}
		case (21):                      /* right-arrow */
		{
			if (c<(signed)strlen(buffer))
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				x++, c++;
			}
			goto GetKey;
		}
		case CTRL_LEFT_KEY:
		{
			if (c)
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				do
				{
					do
						c--, x--;
					while (c and buffer[c-1]!=' ');
				}
				while (c and buffer[c]==' ');
			}
			goto GetKey;
		}
		case CTRL_RIGHT_KEY:
		{
			if (c<(signed)strlen(buffer))
			{
				hugo_settextpos(x, y);
				sprintf(ch, "%c", CURRENT_CHAR(c));
				hugo_print(ch);
				do
				{
					do
						c++, x++;
					while (c<(signed)strlen(buffer) and
						buffer[c-1]!=' ');
				}
				while (c<(signed)strlen(buffer) and buffer[c]==' ');
			}
			goto GetKey;
		}
		case HOME_KEY:
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = 0;
			x = oldx + hugo_strlen(p) - 1;
			goto GetKey;
		}
		case END_KEY:
		{
			hugo_settextpos(x, y);
			sprintf(ch, "%c", CURRENT_CHAR(c));
			hugo_print(ch);
			c = strlen(buffer);
			x = oldx + hugo_strlen(p) + strlen(buffer);
			goto GetKey;
		}
		case (27):                      /* Escape */
		case (24):                      /* CTRL-X */
		case (1<<8):                    /* Esc scan code */
		{
BlankLine:
			hugo_settextpos(oldx+hugo_strlen(p), y);

			memset(buffer, ' ', text_windowwidth);

			buffer[text_windowwidth-oldx-hugo_strlen(p)] = '\0';
			hugo_print(buffer);
			hugo_settextpos(oldx, y);

			goto NewPrompt;
		}
		case (61<<8):                   /* F3 scan code */
		case (11):                      /* up-arrow */
		{
			if (--thiscommand<0)
			{
				thiscommand = 0;
				goto GetKey;
			}
			a = strlen(buffer);
RestoreCommand:
			hugo_restorecommand(thiscommand);
			x = oldx + strlen(buffer) + hugo_strlen(p);
			hugo_settextpos(oldx+hugo_strlen(p), y);
			hugo_print(buffer);
			while (a >= (int)strlen(buffer))
				{hugo_print(" ");
				a--;}
			hugo_settextpos(x-1, y);
			c = strlen(buffer);
			goto GetKey;
		}
		case (10):                      /* down-arrow */
		{
			a = strlen(buffer);
			if (++thiscommand>=hcount) goto BlankLine;
			goto RestoreCommand;
		}
	}

	/* Disallow invalid keystrokes--characters greater than ASCII
	   127 are okay because they are PC accented characters
	*/
	if (b < 32 or b>=256) goto GetKey;

	/* Hugo circa v2.2 allowed '^' and '~' for '\n' and '\"',
	   respectively
	*/
	if (game_version<=22 and (b=='^' or b=='~')) goto GetKey;

	if (oldx+strlen(buffer) >= text_windowwidth-2)
		goto GetKey;

	hugo_settextpos(x++, y);
	sprintf(ch, "%c", b);                   /* add the new character */
	hugo_print(ch);
	buffer[strlen(buffer)+1] = '\0';
	if (c<(signed)strlen(buffer) and insert_mode)
	{
		hugo_settextpos(x, y);
		hugo_print(buffer+c);
		for (a=strlen(buffer); a>c; a--)
			buffer[a] = buffer[a-1];
	}
	buffer[c] = (char)b;

	c++;

	goto GetKey;
}


/* hugo_waitforkey */

int hugo_waitforkey(void)
{
    int b;

	while ((b = _read_kbd(0, 0, 0)) != -1);                /* clear key buffer */
    while ((b = _read_kbd(0, 0, 0)) == -1);

	return hugo_getkey(b);
}


/* hugo_timewait */

int hugo_timewait(int n)
{
    clock_t goal;
	
	goal = CLOCKS_PER_SEC/n + clock();
	while (goal > clock());

	return true;
}


/*
    COMMAND HISTORY

*/

void hugo_addcommand(void)
{
	int i;

	if (!strcmp(buffer, "")) return;

	if (hcount>=HISTORY_SIZE)
	{
		hugo_blockfree(history[0]);
		for (i=0; i<HISTORY_SIZE-1; i++)
			history[i] = history[i+1];
		hcount = HISTORY_SIZE-1;
	}

	/* Because the debugger might use (up to) all available memory for
	   code line storage, a different means of memory allocation is
	   needed (at least in MS-DOS due to limited available memory to
	   begin with).
	*/
#if !defined (DEBUGGER)
	if ((history[hcount] = hugo_blockalloc((long)((strlen(buffer)+1)*sizeof(char))))==NULL)
#else
	if ((history[hcount] = AllocMemory((size_t)((strlen(buffer)+1)*sizeof(char))))==NULL)
#endif
	{
		hugo_blockfree(history[0]);
		if (hcount)
		{
			for (i=0; i<hcount; i++)
				history[i] = history[i+1];
			hcount--;
		}
		return;
	}

	for (i=0; i<=(int)strlen(buffer); i++)
		history[hcount][i] = buffer[i];
	hcount++;
}

void hugo_restorecommand(int n)
{
	int i;

	if (n < 0 or (n>=hcount and hcount!=HISTORY_SIZE-1)) return;

	i = 0;
	do
		buffer[i] = history[n][i];
	while (history[n][i++]!='\0');
}


/* hugo_stripaccents */

void hugo_stripaccents(char *a)
{

    int i;

	for (i=0; i<(int)strlen(a); i++)
	{
		if ((unsigned char)a[i]>=0x80)
		{
			switch ((unsigned char)a[i])
			{
				case 131:               /* 'A' accents */
				case 132:
				case 133:
				case 134:
				case 160:
					{a[i] = 'a';
					break;}
				case 142:
				case 143:
					{a[i] = 'A';
					break;}

				case 135:               /* 'C' cedilla */
					{a[i] = 'c';
					break;}
				case 128:
					{a[i] = 'C';
					break;}

				case 130:               /* 'E' accents */
				case 136:
				case 137:
				case 138:
					{a[i] = 'e';
					break;}
				case 144:
					{a[i] = 'E';
					break;}

				case 139:               /* 'I' accents */
				case 140:
				case 141:
				case 161:
					{a[i] = 'i';
					break;}

				case 164:               /* 'N' tilde */
					{a[i] = 'n';
					break;}
				case 165:
					{a[i] = 'N';
					break;}

				case 147:               /* 'O' accents */
				case 148:
				case 149:
				case 162:
					{a[i] = 'o';
					break;}
				case 153:
					{a[i] = 'O';
					break;}

				case 129:               /* 'U' accents */
				case 150:
				case 151:
				case 163:
					{a[i] = 'u';
					break;}
				case 154:
					{a[i] = 'U';
					break;}

				case 152:               /* 'Y' accents */
					{a[i] = 'y';
					break;}

				case 174:               /* '<<' */
				case 175:               /* '>>' */
					{a[i] = '\"';
					break;}

				default:
				{
					strcpy(a+i, a+i+1);
				}
			}
		}
	}
}


/* 

DISPLAY CONTROL

*/

void hugo_init_screen(void)
{

    v_init();
}

int hugo_hasgraphics(void)
{

    return false;
}

void hugo_setgametitle(char *t)
{
}

void hugo_cleanup_screen(void)
{
    hugo_clearfullscreen();
    free(gline);
}

void hugo_clearfullscreen(void)
{

    hugo_settextwindow(1, 1, SCREENWIDTH, SCREENHEIGHT);
    hugo_clearwindow();
}

void hugo_clearwindow(void)
{

    int i;

    for (i = physical_windowtop; i <= physical_windowbottom; i++) {
        v_gotoxy(physical_windowleft, i);
        v_putn(' ', physical_windowwidth);
    }
    hugo_settextpos(1, 1);
}

void hugo_settextmode(void)
{

    charwidth = FIXEDCHARWIDTH = 1;
	lineheight = FIXEDLINEHEIGHT = 1;
    v_dimen(&SCREENWIDTH, &SCREENHEIGHT);
    gline = malloc(SCREENWIDTH * 2);

	hugo_settextwindow(1, 1,
		SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);
}

void hugo_settextwindow(int left, int top, int right, int bottom)
{

    text_windowtop = top;        text_windowbottom = bottom;
	text_windowleft = left;      text_windowright = right;
	text_windowwidth = text_windowright-text_windowleft+1;

	physical_windowleft = (left-1)*FIXEDCHARWIDTH;
	physical_windowtop = (top-1)*FIXEDLINEHEIGHT;
	physical_windowright = right*FIXEDCHARWIDTH-1;
	physical_windowbottom = bottom*FIXEDLINEHEIGHT-1;
	physical_windowwidth = (right-left+1)*FIXEDCHARWIDTH;
	physical_windowheight = (bottom-top+1)*FIXEDLINEHEIGHT;

	ConstrainCursor();
}


void hugo_settextpos(int x, int y)
{

	current_text_col = text_windowleft-1+x;
	current_text_row = text_windowtop-1+y;
	ConstrainCursor();
    v_gotoxy(current_text_col - 1, current_text_row - 1);

	currentline = y;
	currentpos = (x-1)*FIXEDCHARWIDTH;   /* Note:  zero-based */
}

void ConstrainCursor(void)
{
	if (current_text_col > text_windowright)
			current_text_col = text_windowright;
	if (current_text_col < text_windowleft)
			current_text_col = text_windowleft;
	if (current_text_row > text_windowbottom)
			current_text_row = text_windowbottom;
	if (current_text_row < text_windowtop)
			current_text_row = text_windowtop;
}

void hugo_print(char *a)
{

	int i, len;

	len = strlen(a);

	for (i=0; i<len; i++)
	{
		switch (a[i])
		{
			case '\n':
				++current_text_row;
				break;
			case '\r':
				current_text_col = text_windowleft;
				break;
			default:
			{
                v_gotoxy(current_text_col - 1, current_text_row - 1);
				v_putn((unsigned char)a[i], 1);

				if (++current_text_col > text_windowright)
				{
					current_text_col = text_windowleft;
					++current_text_row;
				}
			}
    	}

		if (current_text_row > text_windowbottom)
		{
			current_text_row = text_windowbottom;
			hugo_scrollwindowup();
		}
	}
}

void hugo_scrollwindowup()
{
    int i;
    for (i = physical_windowtop; i < physical_windowbottom; i++) {
        v_getline(gline, physical_windowleft, i + 1, physical_windowwidth);
        v_putline(gline, physical_windowleft, i, physical_windowwidth);
    }
    v_gotoxy(physical_windowleft, physical_windowbottom);
    v_putn(' ', physical_windowwidth);
    v_gotoxy(current_text_col - 1, current_text_row - 1);
}

void hugo_font(int f)
{

    int bold_color;

    hugo_settextcolor(fcolor);
	if (f & UNDERLINE_FONT)
		hugo_settextcolor((bgcolor!=UNDERLINE_COLOR) ? UNDERLINE_COLOR : HUGO_BLACK);

	if (f & ITALIC_FONT)
		hugo_settextcolor((bgcolor!=ITALIC_COLOR) ? ITALIC_COLOR : HUGO_BLACK);

	if (f & BOLD_FONT)
	{
		bold_color = hugo_color((int)fcolor)+8;
		if (bold_color>=16) bold_color -=8;
		hugo_settextcolor(((int)bgcolor!=bold_color) ? bold_color : HUGO_BLACK);
	}
}

void hugo_settextcolor(int c)           /* foreground (print) color */
{
    int attr;

    attr = v_getattr();
    attr &= 0xF0;
    c = hugo_color(c);
    if (c > 7) {
        c -= 8;
        attr |= INTENSITY;
    }
    attr |= c;
    v_attrib(attr);
}

void hugo_setbackcolor(int c)           /* background color */
{
    int attr;

    attr = v_getattr();
    attr &= 0x0F;
    c = hugo_color(c);
    if (c > 7) {
        c -= 8;
        if (c == attr)
            c = (c==HUGO_BLACK)?HUGO_WHITE:HUGO_BLACK;
    }
    attr |= (c * 0x10);
    v_attrib(attr);
}

int hugo_color(int c)
{

	if (c==16)      c = DEF_FCOLOR;
	else if (c==17) c = DEF_BGCOLOR;
    else if (c==18) c = DEF_SLFCOLOR;
	else if (c==19) c = DEF_SLBGCOLOR;
	else if (c==20) c = hugo_color(fcolor);

	return c;
}

void hugo_setctype(int lines) {

    int cend;

    switch (v_hardware()) {
    case V_COLOR_8:
        cend = 7;
        break;
    case V_COLOR_12:
        cend = 11;
        break;
    default:
        return;
    }

    v_ctype(cend - lines, cend);
}

/*

     CHARACTER TRANSLATION

*/

char hugo_specialchar(char *a, int *i)
{

	char r, s, skipbracket = 0;

	r = a[*i];
	s = r;

	if (r=='\"') return r;

	if (game_version <= 22)
		if (r=='~' or r=='^') return r;

	if (r=='(')
		{r = a[++*i];
		skipbracket = true;}

	switch (r)
	{
		case '`':               /* accent grave */
		{
			s = a[++*i];
			switch (s)
			{
				case 'a':  {s = 133; break;}
				case 'e':  {s = 138; break;}
				case 'i':  {s = 141; break;}
				case 'o':  {s = 149; break;}
				case 'u':  {s = 151; break;}
			}
			break;
		}
		case '\'':              /* accent acute */
		{
			s = a[++*i];
			switch (s)
			{
				case 'a':  {s = 160; break;}
				case 'e':  {s = 130; break;}
				case 'i':  {s = 161; break;}
				case 'o':  {s = 162; break;}
				case 'u':  {s = 163; break;}
				case 'E':  {s = 144; break;}
			}
			break;
		}
		case '~':               /* tilde */
		{
			s = a[++*i];
			switch (s)
			{
				case 'n':  {s = 164; break;}
				case 'N':  {s = 165; break;}
			}
			break;
		}
		case '^':               /* circumflex */
		{
			s = a[++*i];
			switch (s)
			{
				case 'a':  {s = 131; break;}
				case 'e':  {s = 136; break;}
				case 'i':  {s = 140; break;}
				case 'o':  {s = 147; break;}
				case 'u':  {s = 150; break;}
			}
			break;
		}
		case ':':               /* umlaut */
		{
			s = a[++*i];
			switch(s)
			{
				case 'a':  {s = 132; break;}
				case 'e':  {s = 137; break;}
				case 'i':  {s = 139; break;}
				case 'o':  {s = 148; break;}
				case 'u':  {s = 129; break;}
				case 'y':  {s = 152; break;}
				case 'O':  {s = 153; break;}
				case 'U':  {s = 154; break;}
			}
			break;
		}
		case ',':               /* cedilla */
		{
			s = a[++*i];
			switch (s)
			{
				case 'C':  {s = 128; break;}
				case 'c':  {s = 135; break;}
			}
			break;
		}
		case '<':               /* Spanish left quotation marks */
			{s = 174; break;}
		case '>':               /* Spanish right quotation marks */
			{s = 175; break;}
		case '!':               /* upside-down exclamation mark */
			{s = 173; break;}
		case '?':               /* upside-down question mark */
			{s = 168; break;}
		case 'a':               /* ae ligature */
			{s = 145; ++*i; break;}
		case 'A':               /* AE ligature */
			{s = 146; ++*i; break;}
		case 'c':               /* cents symbol */
			{s = 155; break;}
		case 'L':               /* British pound */
			{s = 156; break;}
		case 'Y':               /* Japanese Yen */
			{s = 157; break;}
		case '-':               /* em dash */
			{s = 196; break;}
		case '#':               /* 3-digit decimal ASCII code */
		{
			s = (char)((a[++*i]-'0')*100);
			s += (a[++*i]-'0')*10;
			s += (a[++*i]-'0');
		}
	}

	if (skipbracket)
	{
		++*i;
		if (a[*i+1]==')') ++*i;
		if (s==')') s = r;
	}

	return s;
}


/* 

    CHARACTER AND TEXT MEASUREMENT

*/

int hugo_charwidth(char a)
{
	/* As given here, this function works only for non-proportional
	   printing.  For proportional printing, hugo_charwidth() should
	   return the width of the supplied character in the current
	   font and style.
	*/

	if (a==FORCED_SPACE)
		return FIXEDCHARWIDTH;         /* same as ' ' */

	else if ((unsigned char)a >= ' ') /* alphanumeric characters */

		return FIXEDCHARWIDTH;         /* for non-proportional */

	return 0;
}

int hugo_textwidth(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else
			len += hugo_charwidth(a[i]);
	}

	return len;
}

int hugo_strlen(char *a)
{
	int i, slen, len = 0;

	slen = (int)strlen(a);

	for (i=0; i<slen; i++)
	{
		if (a[i]==COLOR_CHANGE) i+=2;
		else if (a[i]==FONT_CHANGE) i++;
		else len++;
	}

	return len;
}


/*

    GRAPHICS AND SOUND

*/

int hugo_displaypicture(FILE *infile)   /* from hejpeg.c */
{
	return true;
}

int hugo_playmusic(FILE *infile, char loop_flag)
{
	return true;	/* not an error */
}

void hugo_musicvolume(int vol)
{}

void hugo_stopmusic(void)
{}

int hugo_playsample(FILE *infile, char loop_flag)
{
	return true;	/* not an error */
}

void hugo_samplevolume(int vol)
{}

void hugo_stopsample(void)
{}
