!\-----------------------------------------------------------------------
	Sorting arrays of objects in Hugo
	Copyright 2001 by Christopher Tate <ctate@acm.org>

	A "comparison routine", as used here, is a routine that returns
	non-zero when the order of its two arguments should be
	swapped.  The arguments are always passed in the order they
	currently appear in the array.

	For example, if you want to sort the array by the objects'
	"height" property in such a way that the shortest object -- the
	one with the smallest "height" -- is first, you could use this
	comparison routine:

		routine AscendingHeightComparison(obj1, obj2)
		{
			return (obj1.height > obj2.height)
		}

	If obj1's height is greater than obj2's, it means that the taller
	object currently appears earlier in the array, so the two elements
	should be swapped.

	Assuming that your array of objects is called "lineup" and the
	number of objects in the array is in the variable "num_suspects,"
	you would sort the array like this:

		SortArray(lineup, num_suspects, &AscendingHeightComparison)

	Note that you need the '&' character to indicate that the argument
	is the address of the comparison routine, not the result of running it.
-----------------------------------------------------------------------\!

#ifclear _SORT_H
#set _SORT_H

!-----------------------------------------------------------------------
! SortArray() arguments:
!
! data = array to sort
! num = number of elements in the array
! comp_routine = the address of the comparison routine to use

routine SortArray(data, num, comp_routine)
{
	local i
	local did_swap

	for (i = 0; i < num; i = i + 1)
	{
		local j
		did_swap = false
		for (j = num - 1; j > i; j = j - 1)
		{
			local swap
			swap = call comp_routine(array data[j-1], array data[j])
			if swap
			{
				local tmp
				tmp = array data[j]
				array data[j] = array data[j-1]
				array data[j-1] = tmp
				did_swap = true
			}
		}
		! if we didn't swap any elements, then the array is sorted
		if not did_swap : return
	}
}

!-----------------------------------------------------------------------
! SORT_ASCENDING() and SORT_DESCENDING()
!
! These simple comparison routines are provided to handle the
! common cases of sorting a numeric array in ascending or
! descending order.  For example, to sort an entire numeric array
! called "data" in ascending order, just do this:
!
! SortArray(data, data[], &SORT_ASCENDING)

routine SORT_ASCENDING(obj1, obj2)
{
	return (obj1 > obj2)
}

routine SORT_DESCENDING(obj1, obj2)
{
	return (obj1 < obj2)
}

#endif
