/* main.c: Top-level source file
        for GlkDOS, curses.h/allegro implementation of the Glk API
    Designed by L. Ross Raszewski <lraszewski@justice.loyola.edu>
    
    based upon GlkTerm by Andrew Plotkin <erkyrath@netcom.com>
        
    GlkDOS Library: version 0.10.1.
    Glk API which this implements: version 0.60.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/index.html
*/

#include "gtoption.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cscr.h"
#include "glk.h"
#include "glkdos.h"
#include "glkstart.h"
#include "gtpref.h"
#include <unistd.h>

/* external declarations */
void readconfig();
int print_glk_credits();
extern chtype style_mask[];
extern chtype def_style_mask[];
void glkdos_set_basename(char *c);


/* Some constants for my wacky little command-line option parser. */
#define ex_Void (0)
#define ex_Int (1)
#define ex_Bool (2)

static int errflag = FALSE;
static int inittime = FALSE;

static int extract_value(int argc, char *argv[], char *optname, int type,
    int *argnum, int *result, int defval);
static int string_to_bool(char *str);
#ifdef UNIX
static char *basename(char *s)
{
 char *c=strrchr(s,'/');
 if (c) return c;
 else return s;
}
#endif
static char *canonicalize(char *s)
{
 static char str[256];
 char *c;
 strcpy(str,basename(s));
 if ((c = strrchr(str,'.')))
  *c=0;
 return str;
}
int main(int argc, char *argv[])
{
    int ix, jx, val;
    glkunix_startup_t startdata;

    
    /* Test for compile-time errors. If one of these spouts off, you
        must edit glk.h and recompile. */
    if (sizeof(glui32) != 4) {
        printf("Compile-time error: glui32 is not a 32-bit value. Please fix glk.h.\n");
        return 1;
    }
    if ((glui32)(-1) < 0) {
        printf("Compile-time error: glui32 is not unsigned. Please fix glk.h.\n");
        return 1;
    }
    if (sizeof(window_t *) > 4) {
        printf("Compile-time error: Pointers cannot fit in a glui32. Start writing hashtable code.\n");
        return 1;
    }
   readconfig();
    glkdos_set_basename(canonicalize(argv[0]));
    /* Now some argument-parsing. This is probably going to hurt. */
    startdata.argc = 0;
    startdata.argv = (char **)malloc(argc * sizeof(char *));
    
    /* Copy in the program name. */
    startdata.argv[startdata.argc] = argv[0];
    startdata.argc++;
    
    for (ix=1; ix<argc && !errflag; ix++) {
        glkunix_argumentlist_t *argform;
        int inarglist = FALSE;
        char *cx;
        
        for (argform = glkunix_arguments; 
            argform->argtype != glkunix_arg_End && !errflag; 
            argform++) {
            
            if (argform->name[0] == '\0') {
                if (argv[ix][0] != '-') {
                    startdata.argv[startdata.argc] = argv[ix];
                    startdata.argc++;
                    inarglist = TRUE;
                }
            }
            else if ((argform->argtype == glkunix_arg_NumberValue)
                && !strncmp(argv[ix], argform->name, strlen(argform->name))
                && (cx = argv[ix] + strlen(argform->name))
                && (atoi(cx) != 0 || cx[0] == '0')) {
                startdata.argv[startdata.argc] = argv[ix];
                startdata.argc++;
                inarglist = TRUE;
            }
            else if (!strcmp(argv[ix], argform->name)) {
                int numeat = 0;
                
                if (argform->argtype == glkunix_arg_ValueFollows) {
                    if (ix+1 >= argc) {
                        printf("%s: %s must be followed by a value\n", 
                            argv[0], argform->name);
                        errflag = TRUE;
                        break;
                    }
                    numeat = 2;
                }
                else if (argform->argtype == glkunix_arg_NoValue) {
                    numeat = 1;
                }
                else if (argform->argtype == glkunix_arg_ValueCanFollow) {
                    if (ix+1 < argc && argv[ix+1][0] != '-') {
                        numeat = 2;
                    }
                    else {
                        numeat = 1;
                    }
                }
                else if (argform->argtype == glkunix_arg_NumberValue) {
                    if (ix+1 >= argc
                        || (atoi(argv[ix+1]) == 0 && argv[ix+1][0] != '0')) {
                        printf("%s: %s must be followed by a number\n", 
                            argv[0], argform->name);
                        errflag = TRUE;
                        break;
                    }
                    numeat = 2;
                }
                else {
                    errflag = TRUE;
                    break;
                }
                
                for (jx=0; jx<numeat; jx++) {
                    startdata.argv[startdata.argc] = argv[ix];
                    startdata.argc++;
                    if (jx+1 < numeat)
                        ix++;
                }
                inarglist = TRUE;
                break;
            }
        }
        if (inarglist || errflag)
            continue;
            
        if (argv[ix][0] != '-') {
            printf("%s: unwanted argument: %s\n", argv[0], argv[ix]);
            errflag = TRUE;
            break;
        }
        
/* help/version options */
        if (extract_value(argc, argv, "?", ex_Void, &ix, &val, FALSE))
            errflag = TRUE;
        else if (extract_value(argc, argv, "help", ex_Void, &ix, &val, FALSE))
            errflag = TRUE;
        else if (extract_value(argc, argv, "version", ex_Void, &ix, &val, FALSE))
            glk_preferences[pref_printversion] = val;
        else if (extract_value(argc, argv, "v", ex_Void, &ix, &val, FALSE))
            glk_preferences[pref_printversion] = val;

/* Allegro specific options */
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc,argv,"dvorak", ex_Bool, &ix, &val, glk_preferences[pref_dvorak]))
           glk_preferences[pref_dvorak]=val;
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc,argv,"res", ex_Int, &ix, &val, glk_preferences[pref_reso]))
           glk_preferences[pref_reso]=val;
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc,argv,"cpl", ex_Int, &ix, &val, glk_preferences[pref_cpl]))
           glk_preferences[pref_cpl]=val;
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc,argv,"vresx", ex_Int, &ix, &val, VHORIZ_REZ))
           VHORIZ_REZ=val;
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc,argv,"vresy", ex_Int, &ix, &val, VVERT_REZ))
           VVERT_REZ=val;
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc,argv,"font", ex_Int, &ix, &val, glk_preferences[pref_font]))
           glk_preferences[pref_font]=val;
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc, argv, "db", ex_Bool, &ix, &val, glk_preferences[pref_dbl_buf]))
            glk_preferences[pref_dbl_buf] = val;
        else if (cscr_version()==CSCR_ALLEGRO &&
                 extract_value(argc, argv, "fl", ex_Bool, &ix, &val, glk_preferences[pref_fontlist]))
            glk_preferences[pref_fontlist] = val;

/* General options */
        else if (extract_value(argc, argv, "historylen", ex_Int, &ix, &val, 20))
            glk_preferences[pref_historylen] = val;
        else if (extract_value(argc, argv, "hl", ex_Int, &ix, &val, 20))
            glk_preferences[pref_historylen] = val;
        else if (extract_value(argc, argv, "ml", ex_Bool, &ix, &val, glk_preferences[pref_messageline]))
            glk_preferences[pref_messageline] = val;
        else if (extract_value(argc, argv, "sound", ex_Bool, &ix, &val, glk_preferences[pref_sound]))
            glk_preferences[pref_sound] = val;
        else if (extract_value(argc, argv, "latin", ex_Bool, &ix, &val, glk_preferences[pref_latin1]))
            glk_preferences[pref_latin1] = val;
        else if (extract_value(argc, argv, "credits", ex_Bool, &ix, &val, glk_preferences[pref_credits]))
            glk_preferences[pref_credits] = val;
/* pager options */
        else if (extract_value(argc, argv, "more", ex_Bool, &ix, &val, glk_preferences[pref_more_pager]))
            glk_preferences[pref_more_pager] = val;
        else if (extract_value(argc, argv, "percent", ex_Bool, &ix, &val, glk_preferences[pref_per_pager]))
            glk_preferences[pref_per_pager] = val;
        else if (extract_value(argc, argv, "fraction", ex_Bool, &ix, &val, glk_preferences[pref_rat_pager]))
            glk_preferences[pref_rat_pager] = val;

/* appearance options */
 /* Group 1: geometry */
        else if (extract_value(argc, argv, "width", ex_Int, &ix, &val, 80))
            glk_preferences[pref_screenwidth] = val;
        else if (extract_value(argc, argv, "w", ex_Int, &ix, &val, 80))
            glk_preferences[pref_screenwidth] = val;
        else if (extract_value(argc, argv, "height", ex_Int, &ix, &val, 24))
            glk_preferences[pref_screenheight] = val;
        else if (extract_value(argc, argv, "h", ex_Int, &ix, &val, 24))
            glk_preferences[pref_screenheight] = val;
        else if (extract_value(argc, argv, "border", ex_Bool, &ix, &val, glk_preferences[pref_window_borders]))
            glk_preferences[pref_window_borders] = val;
 /* Group 2: Color */
        else if (extract_value(argc, argv, "fg", ex_Int, &ix, &val, glk_preferences[pref_fg_color]))
            glk_preferences[pref_fg_color] = val;
        else if (extract_value(argc, argv, "bg", ex_Int, &ix, &val, glk_preferences[pref_bg_color]))
            glk_preferences[pref_bg_color] = val;
        else if (extract_value(argc, argv, "revgrid", ex_Bool, &ix, &val, glk_preferences[pref_reverse_textgrids]))
            glk_preferences[pref_reverse_textgrids] = val;
        else if (extract_value(argc, argv, "normbg", ex_Bool, &ix, &val, glk_preferences[pref_normalize_wipe]))
            glk_preferences[pref_normalize_wipe] = val;
 /* Group 3: Styles */
        else if (extract_value(argc, argv, "normal", ex_Int, &ix, &val, style_mask[style_Normal]))
            def_style_mask[style_Normal] = val;
        else if (extract_value(argc, argv, "emphasis", ex_Int, &ix, &val, style_mask[style_Emphasized]))
            def_style_mask[style_Emphasized] = val;
        else if (extract_value(argc, argv, "preform", ex_Int, &ix, &val, style_mask[style_Preformatted]))
            def_style_mask[style_Preformatted] = val;
        else if (extract_value(argc, argv, "header", ex_Int, &ix, &val, style_mask[style_Header]))
            (def_style_mask[style_Header]) = val;
        else if (extract_value(argc, argv, "subheader", ex_Int, &ix, &val, style_mask[style_Subheader]))
            (def_style_mask[style_Subheader]) = val;
        else if (extract_value(argc, argv, "alert", ex_Int, &ix, &val, style_mask[style_Alert]))
            (def_style_mask[style_Alert]) = val;
        else if (extract_value(argc, argv, "note", ex_Int, &ix, &val, style_mask[style_Note]))
            (def_style_mask[style_Note]) = val;
        else if (extract_value(argc, argv, "blockquote", ex_Int, &ix, &val, style_mask[style_BlockQuote]))
            (def_style_mask[style_BlockQuote]) = val;
        else if (extract_value(argc, argv, "link", ex_Int, &ix, &val, style_mask[style_Link]))
            (def_style_mask[style_Link]) = val;
        else if (extract_value(argc, argv, "user1", ex_Int, &ix, &val, style_mask[style_User1]))
            (def_style_mask[style_User1]) = val;
        else if (extract_value(argc, argv, "user2", ex_Int, &ix, &val, style_mask[style_User2]))
            (def_style_mask[style_User2]) = val;
  /* Group 4: Links */
        else if (extract_value(argc, argv, "gl", ex_Bool, &ix, &val, glk_preferences[pref_grid_link]))
            glk_preferences[pref_grid_link] = val;
        else if (extract_value(argc, argv, "bl", ex_Bool, &ix, &val, glk_preferences[pref_buf_link]))
            glk_preferences[pref_buf_link] = val;

        else {
            printf("%s: unknown option: %s\n", argv[0], argv[ix]);
            errflag = TRUE;
        }
    }
    
    if (errflag) {
        printf("usage: %s [ options ... ]\n", argv[0]);
        if (glkunix_arguments[0].argtype != glkunix_arg_End) {
            glkunix_argumentlist_t *argform;
            printf("game options:\n");
            for (argform = glkunix_arguments; 
                argform->argtype != glkunix_arg_End; 
                argform++) {
                printf("  %s\n", argform->desc);
            }
        }
        printf("library options:\n");
        printf(" General options:\n");
        printf("  -ml BOOL: use message line (default: 'yes')\n");
        printf("  -historylen NUM: length of command history (default: 20)\n");
        printf("  -latin BOOL: assume latin1 codepage (default: 'no')\n");
        printf("  -sound BOOL: play sound (default: 'yes')\n");
        printf(" Appearance options:\n");
        printf("  -width NUM: manual screen width (if not specified, will try to measure)\n");
        printf("  -height NUM: manual screen height (ditto)\n");
        printf("  -border BOOL: draw borders between windows (default: 'no')\n");
        printf("  -normbg BOOL: match background to \"normal\" style (default: 'yes')\n");
        printf("  -fg NUM: default foreground color (default: 7)\n");
        printf("  -bg NUM: default background color (default: 1)\n");
        printf("  -normal/emphasis/preform/header/subheader/alert/note/blockquote/\n");
        printf("   user1/user2/link NUM:\n");
        printf("    individual style appearances (see config menu for defaults)\n");
        printf("  -revgrid BOOL: reverse text in grid (status) windows (default 'no')\n");
        printf("  -gl BOOL: show links in grid windows (default: 'no')\n");
        printf("  -bl BOOL: show links in buffer windows (default: 'yes')\n");

        printf(" Pager options:\n");
        printf("  -more BOOL: display \"[MORE]\" pager (default: 'yes')\n");
        printf("  -fraction BOOL: display fractional pager (default: 'yes')\n");
        printf("  -percent BOOL: display percentage pager (default: 'no')\n");

        if (cscr_version()==CSCR_ALLEGRO) {
        printf(" Allegro options:\n");
        printf("  -db BOOL: reduce flicker via screen buffering (default: 'yes')\n");
        printf("  -res NUM: set resolution to NUM (default: 2)\n");
        printf("            see documentation for resolutions\n");
        printf("  -vresx NUM: set virtual X-resolution to NUM (default: 0=use screen size)\n");
        printf("  -vresy NUM: set virtual Y-resolution to NUM (default: 0=use screen size)\n");
        printf("  -font NUM: use font NUM from the font datafile (default: 0=use builtin font)\n");
        printf("  -cpl NUM: scale (non-builtin) font to NUM characters per line \n");
        printf("            (default: 0=do not rescale)\n");
        printf("  -fl: display a list of all available fonts\n");
        printf("  -dvorak BOOL: Remap for dvorak keyboard (default: 'no')\n");
        }
        printf("\n");
        printf("  -credits: display library credits\n");
        printf("  -version: display Glk library version\n");
        printf("  -help: display this list\n");
        printf("\nNUM values can be any number. BOOL values can be 'yes' or 'no', \nor no value to toggle.\n");
        return 1;
    }
    
    if (glk_preferences[pref_printversion] || glk_preferences[pref_credits])
        { print_glk_credits(glk_preferences[pref_credits]); return 1; }
    /* We now start up curses. From now on, the program must exit through
        glk_exit(), so that endwin() is called. */
    memcpy(style_mask,def_style_mask,style_NUMSTYLES*sizeof(chtype));
    gli_setup_curses();
    
    /* Initialize things. */
    gli_initialize_misc();
    gli_initialize_windows();
    gli_initialize_events();
    
    inittime = TRUE;
    if (!glkunix_startup_code(&startdata)) {
        glk_exit();
    }
    inittime = FALSE;
    /* Call the program main entry point, and then exit. */
    glk_main();
    glk_exit();
    
    /* glk_exit() doesn't return, but the compiler may kvetch if main()
        doesn't seem to return a value. */
    return 0;
}

/* This is my own parsing system for command-line options. It's nothing
    special, but it works. 
   Given argc and argv, check to see if option argnum matches the string
    optname. If so, parse its value according to the type flag. Store the
    result in result if it matches, and return TRUE; return FALSE if it
    doesn't match. argnum is a pointer so that it can be incremented in
    cases like "-width 80". defval is the default value, which is only
    meaningful for boolean options (so that just "-ml" can toggle the
    value of the ml option.) */
static int extract_value(int argc, char *argv[], char *optname, int type,
    int *argnum, int *result, int defval)
{
    int optlen, val;
    char *cx, *origcx, firstch;
    
    optlen = strlen(optname);
    origcx = argv[*argnum];
    cx = origcx;
    
    firstch = *cx;
    cx++;
    
    if (strncmp(cx, optname, optlen))
        return FALSE;
    
    cx += optlen;
    
    switch (type) {
    
        case ex_Void:
            if (*cx)
                return FALSE;
            *result = TRUE;
            return TRUE;
    
        case ex_Int:
            if (*cx == '\0') {
                if ((*argnum)+1 >= argc) {
                    cx = "";
                }
                else {
                    (*argnum) += 1;
                    cx = argv[*argnum];
                }
            }
            val = atoi(cx);
            if (val == 0 && cx[0] != '0') {
                printf("%s: %s must be followed by a number\n", 
                    argv[0], origcx);
                errflag = TRUE;
                return FALSE;
            }
            *result = val;
            return TRUE;

        case ex_Bool:
            if (*cx == '\0') {
                if ((*argnum)+1 >= argc) {
                    val = -1;
                }
                else {
                    char *cx2 = argv[(*argnum)+1];
                    val = string_to_bool(cx2);
                    if (val != -1)
                        (*argnum) += 1;
                }
            }
            else {
                val = string_to_bool(cx);
                if (val == -1) {
                    printf("%s: %s must be followed by a boolean value\n", 
                        argv[0], origcx);
                    errflag = TRUE;
                    return FALSE;
                }
            }
            if (val == -1)
                val = !defval;
            *result = val;
            return TRUE;
            
    }
    
    return FALSE;
}

static int string_to_bool(char *str)
{
    if (!strcmp(str, "y") || !strcmp(str, "yes"))
        return TRUE;
    if (!strcmp(str, "n") || !strcmp(str, "no"))
        return FALSE;
    if (!strcmp(str, "on"))
        return TRUE;
    if (!strcmp(str, "off"))
        return FALSE;
    if (!strcmp(str, "+"))
        return TRUE;
    if (!strcmp(str, "-"))
        return FALSE;
        
    return -1;
}

strid_t glkunix_stream_open_pathname(char *pathname, glui32 textmode, 
    glui32 rock)
{
    if (!inittime)
        return 0;
    return gli_stream_open_pathname(pathname, (textmode != 0), rock);
}
