/*
    DosGlk  --  A Glk implementation for MS-DOS
    Copyright (C) 1998  Matt Kimball

    Permission is hereby granted, free of charge, to any person
    obtaining a copy of this software and associated documentation
    files (the "Software"), to deal in the Software without
    restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies
    of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following condition:
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL MATT KIMBALL BE LIABLE FOR ANY
    CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
    CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glk.h"
#include "list.h"
#include "rect.h"
#include "win.h" 
#include "screen.h"

#define trim_Max        125
#define trim_Size       25

typedef struct _wrow wrow;
struct _wrow {
	unsigned size;
	char *content;  
	char *style;
};

static list *winlist = NULL;
#define getwin(id) (lglk_list_index(winlist, (unsigned)(id) - 1))
static winid_t root = 0;
static winid_t focus = 0;

winid_t lglk_get_focus() {
	return focus;
}

int lglk_cycle_focus_win(winid_t id, int set) {
	win *w;
	
	w = getwin(id);
		      
	if(set) {
		if(w == NULL)
			return 1;
			
		if(w->type == wintype_Pair) {
			if(lglk_cycle_focus_win(w->child[0], 1))
				return lglk_cycle_focus_win(w->child[1], 1);
			else
				return 0;
		} else {      
			if(w->evrequest != evrequest_None) {
				focus = id;
				return 0;
			}
		}
			
		return 1;
	} else {
		if(w == NULL)
			return 0;

		if(focus == id)
			return 1;
			
		if(w->type == wintype_Pair) {
			if(lglk_cycle_focus_win(w->child[0], 0))
				return lglk_cycle_focus_win(w->child[1], 1);
			else
				return lglk_cycle_focus_win(w->child[1], 0);
		}
		
		return 0;
	}       
}

void lglk_cycle_focus() {
	if(lglk_cycle_focus_win(root, 0))
		lglk_cycle_focus_win(root, 1);
}

win *lglk_get_window(winid_t id) {
	return getwin(id);
}

winid_t glk_window_get_root(void) {
	return root;
}

void lglk_layout(winid_t split) {
	win *w = getwin(split);     
	int height, width;
				  
	if(split == root) {
		w->bounds.l = w->bounds.t = 0;
		w->bounds.b = 24;
		w->bounds.r = 79;
	}
	
	height = w->bounds.b - w->bounds.t + 1;
	width = w->bounds.r - w->bounds.l + 1;                         
	if(w->type == wintype_Pair) {
		win *a, *b;
		a = getwin(w->child[0]);
		b = getwin(w->child[1]);
		
		a->bounds = w->bounds;
		b->bounds = w->bounds;
		
		if((w->split & winmethod_DivisionMask) == winmethod_Fixed) {
			switch(w->split & winmethod_DirMask) {
				case winmethod_Above:
					a->bounds.b = w->bounds.t + (int)w->split_size - 1;
					b->bounds.t = w->bounds.t + (int)w->split_size;
					break;
				case winmethod_Below:
					b->bounds.t = w->bounds.b - (int)w->split_size + 1;
					a->bounds.b = w->bounds.b - (int)w->split_size;
					break;
				case winmethod_Left:
					a->bounds.r = w->bounds.l + (int)w->split_size - 1;
					b->bounds.l = w->bounds.l + (int)w->split_size;
					break;
				case winmethod_Right:
					b->bounds.l = w->bounds.r - (int)w->split_size + 1;
					a->bounds.r = w->bounds.r - (int)w->split_size;
					break;
			}
		} else {
			switch(w->split & winmethod_DirMask) {
				case winmethod_Above:
					a->bounds.b = w->bounds.t + height * (int)w->split_size / 100 - 1;
					b->bounds.t = w->bounds.t + height * (int)w->split_size / 100;
					break;
				case winmethod_Below:
					b->bounds.t = w->bounds.b - height * (int)w->split_size / 100 + 1;
					a->bounds.b = w->bounds.b - height * (int)w->split_size / 100;
					break;
				case winmethod_Left:
					a->bounds.r = w->bounds.l + width * (int)w->split_size / 100 - 1;
					b->bounds.l = w->bounds.l + width * (int)w->split_size / 100 ;
					break;
				case winmethod_Right:
					b->bounds.l = w->bounds.r - width * (int)w->split_size / 100 + 1;
					a->bounds.r = w->bounds.r - width * (int)w->split_size / 100;
					break;
			}               
		}
	
		lglk_layout(w->child[0]);
		lglk_layout(w->child[1]);
	} else {
		w->need_refresh = 1;
	}
}

void lglk_window_set_colors(win *w, glui32 wintype) {
	int at;
	
	for(at = 0; at < style_NUMSTYLES; at++) {
		if(wintype == wintype_TextBuffer)
			w->stylecolor[at] = 0x17;
		else
			w->stylecolor[at] = 0x70;
	}                            
					 
	if(wintype == wintype_TextBuffer) {                                      
		w->stylecolor[style_Header] = w->stylecolor[style_Subheader] = 
			w->stylecolor[style_Emphasized] = w->stylecolor[style_Input] = 0x1f;
		w->stylecolor[style_Alert] = 0x1c;
	} else {
		w->stylecolor[style_Header] = w->stylecolor[style_Subheader] = 
			w->stylecolor[style_Emphasized] = w->stylecolor[style_Input] = 0x7f;
		w->stylecolor[style_Alert] = 0x74;
	}
}

winid_t glk_window_open(winid_t split, glui32 method, glui32 size, 
						glui32 wintype, glui32 rock) {
	winid_t ret;
	win *w;
						
	if(winlist == NULL)
		winlist = lglk_list_create(sizeof(win));
	if(winlist == NULL)
		return 0;
		
	ret = (winid_t)lglk_list_add(winlist);
	if(ret == LGLK_NULL_INDEX)
		return 0;
	ret++;
	w = getwin(ret);
	w->row = lglk_list_create(sizeof(wrow));
	if(w->row == NULL) {
		lglk_list_remove(winlist, (unsigned)ret - 1);
		return 0;
	}
	lglk_list_add(w->row);
		
	w->type = wintype;
	w->rock = rock;
	lglk_window_set_colors(w, wintype);
	if(!root) { 
		root = ret;
		focus = ret;
		
		w->bounds.r = 79;
		w->bounds.b = 24;
	} else {
		winid_t p_id;
		win *p;
		win *splitwin;
		
		splitwin = getwin(split);
		
		p_id = (winid_t)lglk_list_add(winlist);
		if(p_id == LGLK_NULL_INDEX) {
			glk_window_close(ret, NULL);
			return 0;
		}   
		p_id++;
		p = getwin(p_id);
				       
		p->split = method;
		p->split_size = size;
		p->type = wintype_Pair;
		if((method & winmethod_DirMask) == winmethod_Left 
		   || (method & winmethod_DirMask) == winmethod_Above) { 
		p->child[0] = ret;
		p->child[1] = split;
		} else {
			p->child[0] = split;
			p->child[1] = ret;
		}   
		p->parent = splitwin->parent;
		p->bounds = splitwin->bounds;
		
		if(root == split)
			root = p_id;
			
		if(splitwin->parent) {
			win *old_p;
			
			old_p = getwin(splitwin->parent);
			if(old_p->child[0] == split)
				old_p->child[0] = p_id;
			if(old_p->child[1] == split)
				old_p->child[1] = p_id;
		}
		splitwin->parent = p_id;
		w->parent = p_id;
	
		lglk_layout(p_id);
	}
	
	return ret;
}

void lglk_window_free(winid_t id, stream_result_t *result) {
	win *w;
	
	w = getwin(id);
	if(w == NULL)
		return;
		
	glk_window_clear(id);
	
	lglk_list_delete(w->row);
	if(w->stream) {
		glk_stream_close(w->stream, result);
	} else if(result) {
		result->readcount = 0;
		result->writecount = 0;
	}
	lglk_list_remove(winlist, (unsigned)id - 1);
}

void glk_window_close(winid_t id, stream_result_t *result) {
	win *w;
	int myix, oix, hadfocus;
	
	w = getwin(id);
	if(w == NULL)
		return;
		
	if(focus == id)
		hadfocus = 1;
	else 
		hadfocus = 0;
		
	if(w->type == wintype_Pair) {
		winid_t second;
		
		second = w->child[1];
		glk_window_close(w->child[0], NULL);
		glk_window_close(second, NULL);
		return;
	}
			    
	if(w->parent) {
		win *p, *gp, *o;
		int pix, poix;
		
		p = getwin(w->parent);
		if(p->child[0] == id) { 
			myix = 0;  oix = 1;
		} else {
			myix = 1;  oix = 0;
		}
		o = getwin(p->child[oix]);
		
		if(p->parent) {
			gp = getwin(p->parent);
			
			if(gp->child[0] == w->parent) {
				pix = 0;  poix = 1;
			} else {
				pix = 1;  poix = 0;
			}
			
			gp->child[pix] = p->child[oix];
			o->parent = p->parent;
			
			lglk_layout(p->parent);
			lglk_window_free(w->parent, NULL);
		} else {
			root = p->child[oix];
			o->parent = 0;
			
			lglk_layout(root);
		}
	} else {
		root = 0;
	}                           
	
	lglk_window_free(id, result);
	
	if(hadfocus) {                    
		focus = 0;
		lglk_cycle_focus_win(root, 1);
	}
}

void glk_window_get_size(winid_t id, glui32 *widthptr, 
						 glui32 *heightptr) {
	win *w;
	
	w = getwin(id);
	if(widthptr)
		*widthptr = w->bounds.r - w->bounds.l + 1;
	if(heightptr)
		*heightptr = w->bounds.b - w->bounds.t + 1;
}
						 
void glk_window_set_arrangement(winid_t id, glui32 method,
							glui32 size, winid_t keywin) {
    win *w;
    int key_ix;
    winid_t tmp;
    
    w = getwin(id);
    if(w == NULL)
	return;
	
    w->split = method;
    w->split_size = size;
    
	if((w->split & winmethod_DirMask) == winmethod_Left 
	   || (w->split & winmethod_DirMask) == winmethod_Above)               
	key_ix = 0;
    else
	key_ix = 1;
	
    if(w->child[key_ix] != keywin) {
	tmp = w->child[0];
	w->child[0] = w->child[1];
	w->child[1] = tmp;
    }
    
    lglk_layout(id);
}
							
void glk_window_get_arrangement(winid_t id, glui32 *methodptr,
								glui32 *sizeptr, winid_t *keywinptr) {
    win *w;
    
    w = getwin(id);
    if(w == NULL)
	return;
	
    if(methodptr != NULL)
	*methodptr = w->split;
    if(sizeptr != NULL)
	*sizeptr = w->split_size;
    if(keywinptr != NULL) {
		if((w->split & winmethod_DirMask) == winmethod_Left 
		   || (w->split & winmethod_DirMask) == winmethod_Above)
		*keywinptr = w->child[0];
	else
		*keywinptr = w->child[1];
    }
}

winid_t glk_window_iterate(winid_t id, glui32 *rockptr) {
	win *w;

	while((unsigned)id <= lglk_list_size(winlist)) {
		w = getwin(++id);
		if(w) {
			if(rockptr)
				*rockptr = w->rock;
			return id;
		}
	}
	return 0;
}

glui32 glk_window_get_rock(winid_t id) {
	win *w;
	
	w = getwin(id);
	if(w == NULL)
		return 0;
		
	return w->rock;
}

glui32 glk_window_get_type(winid_t id) {
	win *w;
	
	w = getwin(id);
	if(w == NULL)
		return 0;
		
	return w->type;
}

winid_t glk_window_get_parent(winid_t id) {
	win *w;
	
	w = getwin(id);
	if(w == NULL)
		return 0;
		
	return w->parent;
}

void glk_window_clear(winid_t id) {
	wrow *r;
	win *w;
	unsigned ix;
	   
    w = getwin(id);
    if(w == NULL)
	return;     
    ix = 0;
    do {
		r = lglk_list_index(w->row, ix);
		if(r == NULL)
			break;
			
		if(r->content != NULL) {
			free(r->content);
		}
		
		if(r->style != NULL) {
			free(r->style);
		}                  
		
		lglk_list_remove(w->row, ix++);
	} while(1);

	lglk_list_add(w->row);
	
	w->top_row = 0;
	w->out_row = 0;                   
	w->out_col = 0;
	w->need_refresh = 1;
}

void glk_window_move_cursor(winid_t id, glui32 xpos, glui32 ypos) {
	win *w;
	wrow *r;
	
	w = getwin(id);
	if(w == NULL)
		return;
	
	r = lglk_list_index(w->row, (unsigned)ypos);
	if(r == NULL) {
		unsigned ix;
		
		do {
			ix = lglk_list_add(w->row);
			if(ix == LGLK_NULL_INDEX)
				return;
		} while(ix != ypos);  
		r = lglk_list_index(w->row, (unsigned)ypos);
	}                  
	w->out_row = (unsigned)ypos;
	w->out_col = 0;  
	
	if(xpos) {
		if(!r->content) 
			lglk_window_putch(id, ' ', style_Normal);
			
		w->out_col = strlen(r->content);
		while(strlen(r->content) < xpos)
			lglk_window_putch(id, ' ', style_Normal);
	}
	w->out_col = (unsigned)xpos;
}

void lglk_cursor_set(win *w) {
	unsigned row, col;
	
	row = w->out_row;
	col = w->out_col;

	if(w->evrequest == evrequest_Line) {
		col += (unsigned)w->linebuf_pos;
	}

	if(row < w->top_row) {
		lglk_screen_gotoxy(w->bounds.l, w->bounds.t);
	} else if(row > w->top_row + w->bounds.b - w->bounds.t) {
		lglk_screen_gotoxy(w->bounds.r, w->bounds.b);
	} else if(col < 0) {
		lglk_screen_gotoxy(w->bounds.l, row - w->top_row + w->bounds.t);
	} else if((int)col > w->bounds.r - w->bounds.l) {
		lglk_screen_gotoxy(w->bounds.r, row - w->top_row + w->bounds.t);
	} else {
		lglk_screen_gotoxy(col + w->bounds.l, 
							row - w->top_row + w->bounds.t);
	}
}

void lglk_window_draw(win *w) {
	int r, c, x;
	wrow *row;
	
	for(r = w->bounds.t; r <= w->bounds.b; r++) {
	    row = lglk_list_index(w->row, w->top_row + (r - w->bounds.t));
		   
	    x = 0;
		for(c = w->bounds.l; c <= w->bounds.r; c++) {
			if(!row) {
				lglk_screen_put(c, r, w->stylecolor[style_Normal], ' ');
			} else if(!row->content) {
				lglk_screen_put(c, r, w->stylecolor[style_Normal], ' ');                        
			} else if(row->content[x]) {
				lglk_screen_put(c, r, w->stylecolor[row->style[x]], row->content[x]);
				x++;
			} else {
				lglk_screen_put(c, r, w->stylecolor[style_Normal], ' ');                        
			}
		}
	}
}

void lglk_window_draw_input(win *w) {
	unsigned at;
	    
	if(w->out_row < w->top_row || w->out_row > w->top_row + w->bounds.b - w->bounds.t)
		return;
	
	for(at = 0; at < w->linebuf_cur; at++) {
		if(w->out_col + at < 0 || w->out_col + at > (unsigned)w->bounds.r - w->bounds.l)
			continue;
			
		lglk_screen_put(w->bounds.l + w->out_col + at, 
						w->bounds.t + w->out_row - w->top_row,
						w->stylecolor[style_Input], w->linebuf[at]);
	}
	
	for(at = at + w->out_col + w->bounds.l; at <= (unsigned)w->bounds.r; at++) {
		lglk_screen_put(at, w->bounds.t + w->out_row - w->top_row, w->stylecolor[style_Input], ' ');
	}
}

void lglk_window_update(winid_t id) {
	win *w;
	
	w = getwin(id);
	if(w == NULL)
		return;
		
	if(w->child[0])
		lglk_window_update(w->child[0]);
	if(w->child[1])
		lglk_window_update(w->child[1]);
		
	if(w->need_refresh) {   
		lglk_window_draw(w);
		
		w->need_refresh = 0;
	}
				 
	if(w->evrequest == evrequest_Line) {
		lglk_window_draw_input(w);
	} 
	
	if(id == focus)
		lglk_cursor_set(w);
}

void lglk_window_sync() {
	lglk_window_update(root);
}

void lglk_window_trim(winid_t id) {
	win *w;
	int at;
	
	w = getwin(id);
	if(w == NULL)
		return;

	/*  It's slightly ugly, but if some of the output hasn't been
	    displayed yet, we will output it now and wait for the user
	    to read it.  */
	while(w->top_row < trim_Size) {
		focus = id;
	    lglk_event_page();
	}

	for(at = 0; at < trim_Size; at++) {
		wrow *r;
		
		r = lglk_list_index(w->row, at);
		if(r->content)
			free(r->content);
		if(r->style)
			free(r->style);
		lglk_list_remove(w->row, at);
	}
	lglk_list_pack(w->row);
	
	w->out_row -= trim_Size;
	if((signed)w->out_row < 0)
		w->out_row = 0;
	w->top_row -= trim_Size;
	if((signed)w->top_row < 0)
		w->top_row = 0;  
	w->last_page -= trim_Size;
	if((signed)w->last_page < 0)
		w->last_page = 0;
}

void lglk_window_putch(winid_t id, char c, glui32 style) {
	win *w;
	wrow *r;
	
	w = getwin(id);
	if(w == NULL)
		return;
	if(c >= 0 && c < 32 && c != '\n')
		return;
		
	r = lglk_list_index(w->row, w->out_row);
	if(r->content == NULL) {
		r->size = w->bounds.r - w->bounds.l + 1;
		r->content = (char *)malloc(r->size + 1);
		if(r->content == NULL)
			return;     
			
		r->style = (char *)malloc(r->size + 1);
		if(r->style == NULL) {
			free(r->content);
			r->content = NULL;
			return;
		}          
		
		r->content[0] = 0;
	}
	
	if(c == '\n') {
		if(w->echo) {
			glk_put_string_stream(w->echo, r->content);
			glk_put_char_stream(w->echo, '\n');
		}
		
		w->out_row = lglk_list_add(w->row);
		w->out_col = 0;
		
		if(w->out_row > trim_Max)
			lglk_window_trim(id);
	} else {                       
		if(w->out_col < r->size) {
			if(!r->content[w->out_col])
				r->content[w->out_col + 1] = 0;
				
			r->style[w->out_col] = (unsigned char)style;
			r->content[w->out_col++] = c;
		} else if(w->type == wintype_TextBuffer) {
			int space;
			unsigned at;
			
			space = -1;
			for(at = 0; at < r->size; at++)
				if(r->content[at] == ' ')
					space = at;    
					
			if(space == -1) {
				lglk_window_putch(id, '\n', style_Normal);
			} else {
				r->content[space] = 0;           
				lglk_window_putch(id, '\n', style_Normal);
				for(at = space + 1; r->content[at]; at++) {
					lglk_window_putch(id, r->content[at], r->style[at]);
				}
			}
			lglk_window_putch(id, c, style);
		}
	}
	
	w->need_refresh = 1;    
}

void lglk_window_snap_win(winid_t id) {
	win *w;
	
	w = getwin(id);
	if(w == NULL)
		return;
	
	if(w->type == wintype_Pair) {
		lglk_window_snap_win(w->child[0]);
		lglk_window_snap_win(w->child[1]);
	} else {
		if(w->out_row > w->top_row + w->bounds.b - w->bounds.t) {
			w->top_row = w->out_row - (w->bounds.b - w->bounds.t);
			if(w->top_row > w->last_page)
				w->top_row = w->last_page + 1;
			if(w->out_row <= w->top_row + w->bounds.b - w->bounds.t) 
				w->last_page = w->out_row;
			w->need_refresh = 1;
		} else {
			w->last_page = w->out_row;
		}
	}
}

void lglk_window_page_win(winid_t id) {
	win *w;
	int new_top;
	
	w = getwin(id);
	if(w == NULL)
		return;
	
	if(w->type != wintype_Pair) {
		new_top = w->out_row - (w->bounds.b - w->bounds.t);
		if(new_top > w->top_row + w->bounds.b - w->bounds.t) {
			w->top_row = w->top_row + w->bounds.b - w->bounds.t + 1;
			w->last_page = w->top_row - 1;
		} else {
			w->top_row = new_top;
			w->last_page = w->out_row;
		}
		w->need_refresh = 1;
	}
}

void lglk_window_snap() {
	lglk_window_snap_win(root);
}





