/* parse.c: Floo file parsing.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/floo/index.html
*/

#include "glk.h"
#include "floo.h"
#include <stdlib.h>
#include <string.h>

#define FEOF (-1)

static char *strbuf = NULL;
static int strbuf_size = 0;
static glsi32 ch;
static strid_t curfile = NULL;

/* Set everything up. Returns TRUE for ok. */
int init_parser()
{
    strbuf_size = 80;
    strbuf = (char *)malloc(sizeof(char) * strbuf_size);
    if (!strbuf)
        return FALSE;
        
    curfile = 0;
    ch = '\n';
    
    return TRUE;
}

/* Read one object from the given stream. */
obj_t *parse_object(strid_t file)
{
    int ix, jx;
    obj_t *ob;
    
    if (curfile != file) {
        curfile = file;
        ch = '\n'; /* This doesn't actually work if switching from one file to
            another; but we're only parsing a top-level file right now. */
    }
    
    while (ch != FEOF) {
        /* Eat whitespace. */
        while (ch == ' ' || ch == '\t' || ch == '\n' || ch == '\r')
            ch = glk_get_char_stream(curfile);
        
        if ((ch >= '0' && ch <= '9') || ch == '-') {
            /* An integer. */
            glsi32 val = 0;
            int negflag = FALSE;
            if (ch == '-') {
                negflag = TRUE;
                ch = glk_get_char_stream(curfile);
                if (!(ch >= '0' && ch <= '9')) {
                    floo_err("dash without following digits");
                }
            }
            val = (ch - '0');
            ch = glk_get_char_stream(curfile);
            if (val == 0 && (ch == 'x' || ch == 'X')) {
                ch = glk_get_char_stream(curfile);
                while ((ch >= '0' && ch <= '9') 
                    || (ch >= 'a' && ch <= 'f') || (ch >= 'A' && ch <= 'F')) {
                    glsi32 cval;
                    if (ch >= 'a' && ch <= 'f')
                        cval = 10 + (ch - 'a');
                    else if (ch >= 'A' && ch <= 'F')
                        cval = 10 + (ch - 'A');
                    else
                        cval = (ch - '0');
                    val = (val*16) + cval;
                    ch = glk_get_char_stream(curfile);
                }
            }
            else {
                while (ch >= '0' && ch <= '9') {
                    val = (val*10) + (ch - '0');
                    ch = glk_get_char_stream(curfile);
                }
            }
            if (negflag)
                val = -val;
            ob = new_obj(otyp_Integer);
            ob->u.num = val;
            return ob;
        }
        else if ((ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z') || ch == '_'
            || ch == '/') {
            /* A name. */
            int firstflag, typ;
            if (ch == '/') {
                typ = otyp_ID;
                firstflag = TRUE;
                ch = glk_get_char_stream(curfile);
            }
            else {
                typ = otyp_XID;
                firstflag = FALSE;
            }
            ix = 0;
            while ((ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z') || ch == '_'
                || (!firstflag && ch >= '0' && ch <= '9')) {
                if (ix >= strbuf_size) {
                    strbuf_size *= 2;
                    strbuf = (char *)realloc(strbuf, sizeof(char) * strbuf_size);
                }
                strbuf[ix] = ch;
                ix++;
                ch = glk_get_char_stream(curfile);
                firstflag = FALSE;
            } 
            if (ix == 0)
                floo_err("slash without following identifier");
            ob = new_obj(typ);
            ob->u.name.atom = atomdict_find(strbuf, ix, TRUE);
            return ob;
        }
        else if (ch == '"') {
            /* A string. */
            char *str;
            ch = glk_get_char_stream(curfile);
            ix = 0;
            while (ch != '"') {
                if (ch == FEOF || ch == '\n' || ch == '\r') {
                    floo_err("Unterminated string.");
                }
                if (ch == '\\') {
                    ch = glk_get_char_stream(curfile);
                    if (ch == '\n' || ch == '\r') {
                        while (ch == '\n' || ch == '\r')
                            ch = glk_get_char_stream(curfile);
                        continue; 
                    }
                    if (ch == FEOF)
                        floo_err("Backslash at end of file.");
                    if (ch == 'n')
                        ch = '\n';
                    else if (ch >= '0' && ch <= '7') {
                        int octval;
                        octval = (ch - '0');
                        ch = glk_get_char_stream(curfile);
                        if (!(ch >= '0' && ch <= '7'))
                            floo_err("Backslash octal code without three digits.");
                        octval = 8 * octval + (ch - '0');
                        ch = glk_get_char_stream(curfile);
                        if (!(ch >= '0' && ch <= '7'))
                            floo_err("Backslash octal code without three digits.");
                        octval = 8 * octval + (ch - '0');
                        ch = octval;
                    }
                }
                if (ix >= strbuf_size) {
                    strbuf_size *= 2;
                    strbuf = (char *)realloc(strbuf, sizeof(char) * strbuf_size);
                }
                strbuf[ix] = ch;
                ix++;
                ch = glk_get_char_stream(curfile);
            }
            ch = glk_get_char_stream(curfile);
            str = (char *)malloc(sizeof(char) * (ix+1));
            memcpy(str, strbuf, sizeof(char) * ix);
            str[ix] = '\0';
            ob = new_obj(otyp_String);
            ob->u.str.s = str;
            ob->u.str.len = ix;
            return ob;
        }
        else if (ch == '{') {
            /* A procedure. This is tricky, since we have to recursively read
                objects until we reach a '}'. */
            obj_t *ob2;
            ch = glk_get_char_stream(curfile);
            ob2 = new_obj(otyp_Mark);
            ob2->u.num = 2;
            stack_push(valst, ob2);
            while ((ob2 = parse_object(curfile)) != NULL
                && !(ob2->type == otyp_Mark && ob2->u.num == 1)) {
                stack_push(valst, ob2);
            };
            if (!ob2)
                floo_err("Unterminated procedure.");
            delete_obj(ob2);
            ix = stack_searchmark(valst, 2);
            if (ix < 0)
                floo_err("Unbegun procedure.");
            ob = new_obj_array(ix, TRUE);
            for (jx = ix-1; jx >= 0; jx--) {
                ob2 = stack_pop(valst);
                ob->u.arr.o[jx] = ob2;
            }
            ob2 = stack_pop(valst);
            delete_obj(ob2);
            return ob;
        }
        else if (ch == '}') {
            /* Create a special mark, which will only be used to close a
                '{' procedure. */
            ch = glk_get_char_stream(curfile);
            ob = new_obj(otyp_Mark);
            ob->u.num = 1;
            return ob;
        }
        else if (ch == '[' || ch == ']') {
            /* These are one-character names -- no special handling 
                otherwise. */
            char gotch = ch;
            ch = glk_get_char_stream(curfile);
            ob = new_obj(otyp_XID);
            ob->u.name.atom = atomdict_find(&gotch, 1, TRUE);
            return ob;
        }
        else if (ch == '#') {
            /* Comment; eat the rest of the line. */
            while (!(ch == '\n' || ch == '\r' || ch == FEOF))
                ch = glk_get_char_stream(curfile);
        }
        else if (ch == FEOF) {
            continue;
        }
        else {
            floo_err("Illegal character.");
        }
    }
    
    return NULL;
}

/* Check if a stream starts with the string "#FLOO", and then eat the rest of
    that first line. */
int is_floo_file(strid_t file)
{
    int ix, res;
    res = glk_get_buffer_stream(file, strbuf, 5);
    if (res != 5)
        return FALSE;
    
    if (strncmp(strbuf, "#FLOO", 5))
        return FALSE;
    
    ch = glk_get_char_stream(file);
    while (!(ch == '\n' || ch == '\r' || ch == FEOF))
        ch = glk_get_char_stream(file);
    ch = '\n';
    return TRUE;
}
