/* dispatch.c: Floo interface to the Glk dispatch layer.
    Designed by Andrew Plotkin <erkyrath@netcom.com>
    http://www.eblong.com/zarf/glk/floo/index.html
*/

#include <stdlib.h>
#include "glk.h"
#include "gi_dispa.h"
#include "floo.h"

/* We maintain a hash table for each opaque Glk class. classref_t are the
    nodes of the table, and classtable_t are the tables themselves. */

typedef struct classref_struct classref_t;
struct classref_struct {
    void *obj;
    glui32 id;
    int bucknum;
    classref_t *next;
};

#define CLASSHASH_SIZE (31)
typedef struct classtable_struct {
    glui32 lastid;
    classref_t *bucket[CLASSHASH_SIZE];
} classtable_t;

/* A context structure for the prototype-string decoder. */
typedef struct floo_dispatch_struct {
    int numwanted;
    obj_t *obarr;
    int maxargs;
    gluniversal_t *arglist;
} floo_dispatch_t;

/* The list of hash tables, for the classes. */
static int num_classes = 0;
classtable_t **classes = NULL;

/* A static array for the argument lists. */
static gluniversal_t *floo_arglist = NULL;
static int floo_arglist_size = 0;

static classtable_t *new_classtable(glui32 firstid);
static void *classes_get(int classid, glui32 objid);
static classref_t *classes_put(int classid, void *obj);
static void classes_remove(int classid, void *obj);

static gidispatch_rock_t floo_classtable_register(void *obj, glui32 objclass);
static void floo_classtable_unregister(void *obj, glui32 objclass, 
    gidispatch_rock_t objrock);
static gidispatch_rock_t floo_arraytable_register(void *array, glui32 len, 
    char *typecode);
static void floo_arraytable_unregister(void *array, glui32 len, char *typecode, 
    gidispatch_rock_t objrock);

static void arraytable_add(obj_t *obj, void *array);
static void arraytable_clear(void);

static int floo_prepare_glk_args(char *proto, floo_dispatch_t *splot);
static int floo_parse_glk_args(floo_dispatch_t *splot, 
    char **proto, int depth, obj_t **obarrptr, glui32 *argnum);
static int floo_unparse_glk_args(floo_dispatch_t *splot, 
    char **proto, int depth, obj_t **obarrptr, glui32 *argnum);

int init_dispatch()
{
    int ix;
    
    /* Allocate the class hash tables. */
    num_classes = gidispatch_count_classes();
    classes = (classtable_t **)malloc(num_classes * sizeof(classtable_t *));
    if (!classes)
        return FALSE;
    
    for (ix=0; ix<num_classes; ix++) {
        classes[ix] = new_classtable(ix * 100 + 100);
        if (!classes[ix])
            return FALSE;
    }
    
    /* Set up the four callbacks. */
    gidispatch_set_object_registry(&floo_classtable_register, 
        &floo_classtable_unregister);
    gidispatch_set_retained_registry(&floo_arraytable_register, 
        &floo_arraytable_unregister);
    
    return TRUE;
}

/* This is the big Glk entry point. It does the work of the "glk" operator. */
void floo_dispatch()
{
    obj_t *ob;
    glui32 funcnum;
    
    ob = stack_pop(valst);
    if (!ob)
        floo_err("Stack underflow.");
    if (ob->type != otyp_Integer)
        floo_err("Glk dispatcher must take an integer operand.");
    funcnum = ob->u.num;
    delete_obj(ob);
    
    switch (funcnum) {
        /* To speed life up, we could implement commonly-used Glk functions
            directly -- instead of bothering with the whole prototype mess.
            But I haven't written those shortcuts yet. */
        default: {
            /* Go through the full dispatcher prototype foo. */
            floo_dispatch_t splot;
            char *proto, *cx;
            glui32 argnum, argnumout;
            
            /* Grab the string. */
            proto = gidispatch_prototype(funcnum);
            if (!proto)
                floo_err("Unknown Glk function.");

            /* The work goes in four phases. First, we figure out how many
                arguments we want, and pull them off the stack. Then we go
                through the arguments and load them into the arglist array. 
                Then we call. Then we go through the arguments and unload
                the data back into Floo objects. */
            
            /* Phase 0: get ready. */
            splot.obarr = NULL;
            if (!floo_prepare_glk_args(proto, &splot)) 
                floo_err("Invalid arguments.");
            if (!floo_arglist) {
                floo_arglist_size = splot.maxargs + 16;
                floo_arglist = (gluniversal_t *)malloc(
                    floo_arglist_size * sizeof(gluniversal_t));
            }
            else if (splot.maxargs >= floo_arglist_size) {
                floo_arglist_size = splot.maxargs + 16;
                floo_arglist = (gluniversal_t *)realloc(floo_arglist,
                    floo_arglist_size * sizeof(gluniversal_t));
            }
            splot.arglist = floo_arglist;
            
            /* Phase 1: load 'em up. */
            argnum = 0;
            cx = proto;
            if (!floo_parse_glk_args(&splot, &cx, 0, &splot.obarr, &argnum))
                floo_err("Invalid arguments.");
            
            /* Phase 2: call. */
            gidispatch_call(funcnum, argnum, floo_arglist);

            /* Phase 3: unload. */
            argnumout = 0;
            cx = proto;
            if (!floo_unparse_glk_args(&splot, &cx, 0, &splot.obarr, &argnumout))
                floo_err("Invalid arguments.");
            if (argnum != argnumout)
                floo_err("Unmarshalling argument mismatch.");
            
            /* And some cleanup. */
            delete_obj(splot.obarr);
            arraytable_clear();
            break;
        }
    }
}

/* Read the prefixes of an argument string -- the "<>&+:#!" chars. */
static char *read_prefix(char *cx, int *isref, int *isarray,
    int *passin, int *passout, int *nullok, int *isretained)
{
    *isref = FALSE;
    *passin = FALSE;
    *passout = FALSE;
    *nullok = TRUE;
    *isarray = FALSE;
    *isretained = FALSE;
    while (1) {
        if (*cx == '<') {
            *isref = TRUE;
            *passout = TRUE;
        }
        else if (*cx == '>') {
            *isref = TRUE;
            *passin = TRUE;
        }
        else if (*cx == '&') {
            *isref = TRUE;
            *passout = TRUE;
            *passin = TRUE;
        }
        else if (*cx == '+') {
            *nullok = FALSE;
        }
        else if (*cx == ':') {
            *isref = TRUE;
            *passout = TRUE;
            *nullok = FALSE;
        }
        else if (*cx == '#') {
            *isarray = TRUE;
        }
        else if (*cx == '!') {
            *isretained = TRUE;
        }
        else {
            break;
        }
        cx++;
    }
    return cx;
}

/* This reads through the prototype string, and pulls Floo objects off the
    stack. It also works out the maximal number of gluniversal_t objects
    which could be used by the Glk call in question. 
   This fills in the numwanted, obarr, and maxargs fields in splot. */
static int floo_prepare_glk_args(char *proto, floo_dispatch_t *splot)
{
    int ix;
    obj_t *ob, *obarr;
    char *cx = proto;
    char *bx;
    int maxargs;
    int numwanted;
    
    numwanted = 0;
    while (*cx >= '0' && *cx <= '9') {
        numwanted = 10 * numwanted + (*cx - '0');
        cx++;
    }
    splot->numwanted = numwanted;

    obarr = new_obj_array(numwanted, FALSE);

    maxargs = 0;
    for (ix = 0; ix < numwanted; ix++) {
        int isref, passin, passout, nullok, isarray, isretained;
        cx = read_prefix(cx, &isref, &isarray, &passin, &passout, &nullok,
            &isretained);
        if (isref) {
            maxargs += 2;
        }
        else {
            maxargs += 1;
        }
        
        if (!isref || passin || isarray || (passout && nullok)) {
            /* nothing */
        }
        else {
            obarr->u.arr.o[ix] = new_nullref();
        }

        if (*cx == 'I' || *cx == 'C') {
            cx += 2;
        }
        else if (*cx == 'Q') {
            cx += 2;
        }
        else if (*cx == 'S') {
            cx += 1;
        }
        else if (*cx == '[') {
            int refdepth, nwx;
            cx++;
            nwx = 0;
            while (*cx >= '0' && *cx <= '9') {
                nwx = 10 * nwx + (*cx - '0');
                cx++;
            }
            maxargs += nwx; /* This is *only* correct because all structs contain
                plain values. */
            refdepth = 1;
            while (refdepth > 0) {
                if (*cx == '[')
                    refdepth++;
                else if (*cx == ']')
                    refdepth--;
                cx++;
            }
        }
        else {
            floo_err("Illegal format string.");
        }
    }

    if (*cx != ':' && *cx != '\0')
        floo_err("Illegal format string.");

    for (ix = numwanted-1; ix >= 0; ix--) {
        if (!obarr->u.arr.o[ix]) {
            ob = stack_pop(valst);
            if (!ob)
                floo_err("Stack underflow.");
            obarr->u.arr.o[ix] = ob;
        }
    }
    
    splot->obarr = obarr;
    splot->maxargs = maxargs;
    return TRUE;
}

/* This long and unpleasant function translates a set of Floo objects into
    a gluniversal_t array. It's recursive, too, to deal with structures. */
static int floo_parse_glk_args(floo_dispatch_t *splot,
    char **proto, int depth, obj_t **obarrptr, glui32 *argnumptr)
{
    int ix;
    char *cx;
    obj_t *ob, *obarr;
    int numwanted; 
    glui32 argnum, val;
    gluniversal_t *arglist;
    void *opref;
    
    arglist = splot->arglist;
    obarr = *obarrptr;
    argnum = *argnumptr;
    cx = *proto;

    numwanted = 0;
    while (*cx >= '0' && *cx <= '9') {
        numwanted = 10 * numwanted + (*cx - '0');
        cx++;
    }

    if (!obarr || obarr->u.arr.len != numwanted) {
        floo_err("Glk type error (struct wrong size).");
    }

    for (ix = 0; ix < numwanted; ix++) {
        char typeclass;
        int isref, passin, passout, nullok, isarray, isretained;
        cx = read_prefix(cx, &isref, &isarray, &passin, &passout, &nullok,
            &isretained);
        
        typeclass = *cx;
        cx++;

        if (!isref || passin || isarray) {
            /* Or if depth > 0, really, but as that currently implies isref
                we'll leave it as it is. */
            int skipval = FALSE;
            ob = obarr->u.arr.o[ix]; /* nonref */
            if (isref) {
                if (nullok && ob->type == otyp_Null) {
                    arglist[argnum].ptrflag = FALSE;
                    argnum++;
                    skipval = TRUE;
                }
                else {
                    arglist[argnum].ptrflag = TRUE;
                    argnum++;
                }
            }
            if (!skipval) {
                if (!isarray) {
                    switch (typeclass) {
                        case '[':
                            if (ob->type != otyp_Array)
                                floo_err("Glk type error (struct).");
                            if (!floo_parse_glk_args(splot, &cx, depth+1, &ob, 
                                &argnum))
                                floo_err("Invalid arguments.");
                            break;
                        case 'I':
                            if (ob->type != otyp_Integer)
                                floo_err("Glk type error (integer).");
                            if (*cx == 'u')
                                arglist[argnum].uint = (glui32)(ob->u.num);
                            else if (*cx == 's')
                                arglist[argnum].sint = (glsi32)(ob->u.num);
                            else
                                floo_err("Illegal format string.");
                            argnum++;
                            break;
                        case 'Q':
                            if (ob->type != otyp_Integer)
                                floo_err("Glk type error (integer id).");
                            if (ob->u.num)
                                opref = classes_get(*cx-'a', ob->u.num);
                            else
                                opref = NULL;
                            arglist[argnum].opaqueref = opref;
                            argnum++;
                            break;
                        case 'C':
                            if (ob->type != otyp_Integer)
                                floo_err("Glk type error (char).");
                            if (*cx == 'u')
                                arglist[argnum].uch = (unsigned char)(ob->u.num);
                            else if (*cx == 's')
                                arglist[argnum].sch = (signed char)(ob->u.num);
                            else if (*cx == 'n')
                                arglist[argnum].ch = (char)(ob->u.num);
                            else
                                floo_err("Illegal format string.");
                            argnum++;
                            break;
                        case 'S':
                            if (ob->type != otyp_String)
                                floo_err("Glk type error (string).");
                            arglist[argnum].charstr = ob->u.str.s;
                            argnum++;
                            break;
                        default:
                            floo_err("Illegal format string.");
                            break;
                    }
                }
                else {
                    switch (typeclass) {
                        case 'C':
                            if (ob->type != otyp_String)
                                floo_err("Glk type error (char array).");
                            arglist[argnum].array = ob->u.str.s;
                            argnum++;
                            arglist[argnum].uint = ob->u.str.len;
                            argnum++;
                            if (isretained)
                                arraytable_add(ob, ob->u.str.s);
                            break;
                        default:
                            floo_err("Illegal format string.");
                            break;
                    }
                }
            }
            else {
                if (typeclass == '[') {
                    int numsubwanted, refdepth;
                    numsubwanted = 0;
                    while (*cx >= '0' && *cx <= '9') {
                        numsubwanted = 10 * numsubwanted + (*cx - '0');
                        cx++;
                    }
                    refdepth = 1;
                    while (refdepth > 0) {
                        if (*cx == '[')
                            refdepth++;
                        else if (*cx == ']')
                            refdepth--;
                        cx++;
                    }
                }
            }
        }
        else if (passout) {
            int jx, numsubwanted, refdepth;
            int skipval = FALSE;
            if (nullok) {
                ob = obarr->u.arr.o[ix]; /* nonref */
                if (ob->type != otyp_Boolean)
                    floo_err("Glk type error (boolean).");
                if (!ob->u.num) {
                    arglist[argnum].ptrflag = FALSE;
                    argnum++;
                    skipval = TRUE;
                }
                else {
                    arglist[argnum].ptrflag = TRUE;
                    argnum++;
                }
            }
            else {
                arglist[argnum].ptrflag = TRUE;
                argnum++;
            }
            if (typeclass == '[') {
                numsubwanted = 0;
                while (*cx >= '0' && *cx <= '9') {
                    numsubwanted = 10 * numsubwanted + (*cx - '0');
                    cx++;
                }
                refdepth = 1;
                while (refdepth > 0) {
                    if (*cx == '[')
                        refdepth++;
                    else if (*cx == ']')
                        refdepth--;
                    cx++;
                }
            }
            else {
                numsubwanted = 1;
            }
            if (!skipval) {
                for (jx=0; jx<numsubwanted; jx++) {
                    arglist[argnum].uint = 0;
                    argnum++;
                }
            }
        }
        if (typeclass != 'S' && typeclass != '[')
            cx++;
    }

    if (depth > 0) {
        if (*cx != ']')
            floo_err("Illegal format string.");
        cx++;
    }
    else {
        if (*cx != ':' && *cx != '\0')
            floo_err("Illegal format string.");
    }
    
    *proto = cx;
    *argnumptr = argnum;
    *obarrptr = obarr;
    
    return TRUE;
}

/* This is about the reverse of floo_parse_glk_args(). */
static int floo_unparse_glk_args(floo_dispatch_t *splot,
    char **proto, int depth, obj_t **obarrptr, glui32 *argnumptr)
{
    int ix;
    char *cx;
    obj_t *ob, *obarr;
    int numwanted; 
    glui32 argnum, val;
    gluniversal_t *arglist;
    void *opref;
    
    arglist = splot->arglist;
    obarr = *obarrptr;
    argnum = *argnumptr;
    cx = *proto;

    numwanted = 0;
    while (*cx >= '0' && *cx <= '9') {
        numwanted = 10 * numwanted + (*cx - '0');
        cx++;
    }

    if (!obarr || obarr->type != otyp_Array) {
        if (obarr)
            delete_obj(obarr);
        *obarrptr = NULL;
        obarr = new_obj_array(numwanted, FALSE);
    }
    else if (obarr->u.arr.len != numwanted) {
        floo_err("Glk type error (struct wrong size).");
    }

    for (ix = 0; ix < numwanted; ix++) {
        char typeclass;
        int isref, passin, passout, nullok, isarray, isretained;
        cx = read_prefix(cx, &isref, &isarray, &passin, &passout, &nullok,
            &isretained);
        
        typeclass = *cx;
        cx++;

        if ((passout && !isretained) || depth > 0) {
            int skipval = FALSE;
            if (isref) {
                if (!arglist[argnum].ptrflag)
                    skipval = TRUE;
                argnum++;
            }
            if (!skipval) {
                if (!isarray) {
                    switch (typeclass) {
                        case '[':
                            ob = obj_newref(obarr->u.arr.o[ix]); 
                            if (!floo_unparse_glk_args(splot, &cx, depth+1, &ob, 
                                &argnum))
                                floo_err("Invalid arguments.");
                            break;
                        case 'I':
                            ob = new_obj(otyp_Integer);
                            if (*cx == 'u')
                                ob->u.num = (glsi32)(arglist[argnum].uint);
                            else if (*cx == 's')
                                ob->u.num = (glsi32)(arglist[argnum].sint);
                            else
                                floo_err("Illegal format string.");
                            argnum++;
                            break;
                        case 'Q': 
                            ob = new_obj(otyp_Integer);
                            opref = arglist[argnum].opaqueref;
                            if (!opref)
                                ob->u.num = 0;
                            else {
                                gidispatch_rock_t objrock = 
                                    gidispatch_get_objrock(opref, *cx-'a');
                                ob->u.num = ((classref_t *)objrock.ptr)->id;
                            }
                            argnum++;
                            break;
                        case 'C':
                            ob = new_obj(otyp_Integer);
                            if (*cx == 'u')
                                ob->u.num = (glsi32)(glui32)(arglist[argnum].uch);
                            else if (*cx == 's')
                                ob->u.num = (glsi32)(arglist[argnum].sch);
                            else if (*cx == 'n')
                                ob->u.num = (glsi32)(arglist[argnum].ch);
                            else
                                floo_err("Illegal format string.");
                            argnum++;
                            break;
                        case 'S':
                            floo_err("Glk type error (cannot return string).");
                            break;
                        default:
                            floo_err("Illegal format string.");
                            break;
                    }
                }
                else {
                    switch (typeclass) {
                        case 'C':
                            ob = obj_newref(obarr->u.arr.o[ix]); 
                            if (ob->type != otyp_String)
                                floo_err("Glk type error (char array).");
                            argnum += 2;
                            break;
                        default:
                            floo_err("Illegal format string.");
                            break;
                    }
                }
                if (depth == 0) {
                    stack_push(valst, ob); 
                }
                else {
                    if (ob != obarr->u.arr.o[ix]) {
                        if (obarr->u.arr.o[ix])
                            delete_obj(obarr->u.arr.o[ix]);
                        obarr->u.arr.o[ix] = ob;
                    }
                }
            }
            else {
                if (typeclass == '[') {
                    int numsubwanted, refdepth;
                    numsubwanted = 0;
                    while (*cx >= '0' && *cx <= '9') {
                        numsubwanted = 10 * numsubwanted + (*cx - '0');
                        cx++;
                    }
                    refdepth = 1;
                    while (refdepth > 0) {
                        if (*cx == '[')
                            refdepth++;
                        else if (*cx == ']')
                            refdepth--;
                        cx++;
                    }
                }
            }
        }
        else if (!isref || passin) {
            int jx, numsubwanted, refdepth;
            int skipval = FALSE;
            if (isref) {
                if (!arglist[argnum].ptrflag)
                    skipval = TRUE;
                argnum++;
            }
            if (typeclass == '[') {
                numsubwanted = 0;
                while (*cx >= '0' && *cx <= '9') {
                    numsubwanted = 10 * numsubwanted + (*cx - '0');
                    cx++;
                }
                refdepth = 1;
                while (refdepth > 0) {
                    if (*cx == '[')
                        refdepth++;
                    else if (*cx == ']')
                        refdepth--;
                    cx++;
                }
            }
            else {
                numsubwanted = 1;
                if (isarray)
                    numsubwanted++;
            }
            if (!skipval) {
                for (jx=0; jx<numsubwanted; jx++) {
                    arglist[argnum].uint = 0;
                    argnum++;
                }
            }
        }
        if (typeclass != 'S' && typeclass != '[')
            cx++;
    }

    if (depth > 0) {
        if (*cx != ']')
            floo_err("Illegal format string.");
        cx++;
    }
    else {
        if (*cx != ':' && *cx != '\0')
            floo_err("Illegal format string.");
    }
    
    *proto = cx;
    *argnumptr = argnum;
    *obarrptr = obarr;
    
    return TRUE;
}

/* Build a hash table to hold a set of Glk objects. */
static classtable_t *new_classtable(glui32 firstid)
{
    int ix;
    classtable_t *ctab = (classtable_t *)malloc(sizeof(classtable_t));
    if (!ctab)
        return NULL;
    
    for (ix=0; ix<CLASSHASH_SIZE; ix++)
        ctab->bucket[ix] = NULL;
    
    ctab->lastid = firstid;
    
    return ctab;
}

/* Find a Glk object in the appropriate hash table. */
static void *classes_get(int classid, glui32 objid)
{
    classtable_t *ctab;
    classref_t *cref;
    if (classid < 0 || classid >= num_classes)
        return NULL;
    ctab = classes[classid];
    cref = ctab->bucket[objid % CLASSHASH_SIZE];
    for (; cref; cref = cref->next) {
        if (cref->id == objid)
            return cref->obj;
    }
    return NULL;
}

/* Put a Glk object in the appropriate hash table. */
static classref_t *classes_put(int classid, void *obj)
{
    int bucknum;
    classtable_t *ctab;
    classref_t *cref;
    if (classid < 0 || classid >= num_classes)
        return NULL;
    ctab = classes[classid];
    cref = (classref_t *)malloc(sizeof(classref_t));
    if (!cref)
        return NULL;
    cref->obj = obj;
    cref->id = ctab->lastid;
    ctab->lastid++;
    bucknum = cref->id % CLASSHASH_SIZE;
    cref->bucknum = bucknum;
    cref->next = ctab->bucket[bucknum];
    ctab->bucket[bucknum] = cref;
    return cref;
}

/* Delete a Glk object from the appropriate hash table. */
static void classes_remove(int classid, void *obj)
{
    classtable_t *ctab;
    classref_t *cref;
    classref_t **crefp;
    gidispatch_rock_t objrock;
    if (classid < 0 || classid >= num_classes)
        return;
    ctab = classes[classid];
    objrock = gidispatch_get_objrock(obj, classid);
    cref = objrock.ptr;
    if (!cref)
        return;
    crefp = &(ctab->bucket[cref->bucknum]);
    for (; *crefp; crefp = &((*crefp)->next)) {
        if ((*crefp) == cref) {
            *crefp = cref->next;
            cref->obj = NULL;
            cref->id = 0;
            cref->next = NULL;
            free(cref);
            return;
        }
    }
    return;
}

/* The object registration/unregistration callbacks that the library calls
    to keep the hash tables up to date. */
    
static gidispatch_rock_t floo_classtable_register(void *obj, glui32 objclass)
{
    classref_t *cref;
    gidispatch_rock_t objrock;
    cref = classes_put(objclass, obj);
    objrock.ptr = cref;
    return objrock;
}

static void floo_classtable_unregister(void *obj, glui32 objclass, 
    gidispatch_rock_t objrock)
{
    classes_remove(objclass, obj);
}

/* The structure to deal with retained arrays is sort of a pain in the ass.
    In every floo_dispatch() call, we keep a linked list of arrays that
    were passed to "!" arguments -- this is temparraytable. At the end of
    floo_dispatch(), we go through and clear this out. Normally, this means
    just deleting the reference. However, if the Glk call invoked
    floo_arraytable_register() (our retained-registry callback),
    the appropriate entry in temparraytable was marked as a keeper. And then
    the clearout procedure moves the reference to keeparraytable, where it
    stays until floo_arraytable_unregister() is called to erase it.
    
    Even grosser, it's probably legal for a single array to be retained
    several times by the library. I think. Anyway, this code allows that.
*/

/* An entry. Has useful pointers to the Glk array object and the actual
    array in memory, and a reference count. */  
typedef struct floo_arraytable_struct {
    obj_t *obj;
    void *array;
    int keeps;
    struct floo_arraytable_struct *next;
} floo_arraytable_t;

/* The two linked lists. */
static floo_arraytable_t *temparraytable = NULL;
static floo_arraytable_t *keeparraytable = NULL;

static void arraytable_add(obj_t *obj, void *array)
{
    floo_arraytable_t *art = (floo_arraytable_t *)malloc(sizeof(floo_arraytable_t));
    if (!art)
        floo_err("Out of memory.");
    art->obj = obj_newref(obj);
    art->array = array;
    art->keeps = 0;
    art->next = temparraytable;
    temparraytable = art;
}

static void arraytable_clear(void)
{
    floo_arraytable_t *art;
    while (temparraytable) {
        art = temparraytable;
        temparraytable = art->next;
        if (art->keeps > 0) {
            art->next = keeparraytable;
            keeparraytable = art;
        }
        else {
            delete_obj(art->obj);
            art->obj = NULL;
            free(art);
        }
    }
}

static gidispatch_rock_t floo_arraytable_register(void *array, glui32 len, 
    char *typecode)
{
    floo_arraytable_t *art;
    gidispatch_rock_t arrrock;
    for (art = temparraytable; art; art = art->next) {
        if (art->array == array) {
            art->keeps++;
            arrrock.ptr = art;
            return arrrock;
        }
    }
    floo_err("arraytable_register: unable to find array.");
    arrrock.ptr = NULL;
    return arrrock;
}

static void floo_arraytable_unregister(void *array, glui32 len, char *typecode, 
    gidispatch_rock_t objrock)
{
    floo_arraytable_t *art, **artp;
    for (artp = &keeparraytable; *artp; artp = &((*artp)->next)) {
        if ((*artp)->array == array) {
            break;
        }
    }
    if (!*artp) {
        for (artp = &temparraytable; *artp; artp = &((*artp)->next)) {
            if ((*artp)->array == array) {
                break;
            }
        }
    }
    
    if (*artp) {
        art = *artp;
        art->keeps--;
        if (art->keeps <= 0) {
            *artp = art->next;
            delete_obj(art->obj);
            art->obj = NULL;
            free(art);
        }
    }
}

