/* Scan.c - a lexical scanner for the compiler */
/*
	Copyright (c) 1993, by David Michael Betz
	All rights reserved
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "Drool.h"
#include "Compiler.h"
#include "Streams.h"

/* useful definitions */
#define maplower(ch)	(isupper(ch) ? tolower(ch) : ch)

/* global variables */
int linenumber;		/* line number */
int errcount=0;		/* error count */
int t_value;		/* numeric value */
char t_token[TKNSIZE+1];/* token string */
char *t_names[] = {
	"<UNDEF>",
	"(",
	")",
	"'",
	"STRING",
	"IDENTIFIER",
	"NUMBER",
	"EOF"
};

/* local variables */
static StreamHandle inputStream;	/* current input stream */
static int savetkn;			/* look ahead token */
static int savech;			/* look ahead character */
static char line[200];			/* current input line */
static char *lptr;			/* line pointer */
static int ieof;			/* input end of file flag */

/* prototypes */
static int rtoken(void);
static int getstring(void);
static int getid(int ch);
static int isnumber(char *str,int *pval);
static int skipspaces(void);
static int isidchar(int ch);
static int getch(void);
static int osgetc(void);

/* InitScan - initialize the scanner */
void InitScan(StreamHandle s)
{
    /* remember the input stream */
    inputStream = s;
    
    /* setup the line buffer */
    lptr = line; *lptr = '\0';
    linenumber = 0;

    /* no lookahead yet */
    savetkn = 0;
    savech = 0;

    /* not eof yet */
    ieof = FALSE;
}

/* token - get the next token */
int token(void)
{
    int tkn;
    if (tkn = savetkn)
	savetkn = 0;
    else
	tkn = rtoken();
    return (tkn);
}

/* stoken - save a token */
void stoken(int tkn)
{
    savetkn = tkn;
}

/* rtoken - read the next token */
static int rtoken(void)
{
    int ch;

    /* check the next character */
    for (;;)
	switch (ch = skipspaces()) {
	case StreamEOF:	return (T_EOF);
	case '(':	strcpy(t_token,"("); return (T_OPEN);
	case ')':	strcpy(t_token,")"); return (T_CLOSE);
	case '\'':	strcpy(t_token,"'"); return (T_QUOTE);
	case '"':	return (getstring());
	case ';':	while (getch() != '\n'); break;
	default:	return (getid(ch));
	}
}

/* getstring - get a string */
static int getstring(void)
{
    char *p;
    int ch;
    p = t_token;
    while ((ch = getch()) != StreamEOF && ch != '"') {
	if (ch == '\\')
	    switch (ch = getch()) {
	    case 'n':  ch = '\n'; break;
	    case 't':  ch = '\t'; break;
	    }
	*p++ = ch;
    }
    *p = '\0';
    return (T_STRING);
}

/* getid - get an identifier */
static int getid(int ch)
{
    char *p;

    p = t_token; *p++ = maplower(ch);
    while ((ch = getch()) != StreamEOF && isidchar(ch))
	*p++ = maplower(ch);
    *p = '\0';
    savech = ch;
    return (isnumber(t_token,&t_value) ? T_NUMBER : T_IDENTIFIER);
}

/* isnumber - check if this string is a number */
static int isnumber(char *str,int *pval)
{
    int digits;
    char *p;

    /* initialize */
    p = str; digits = 0;

    /* check for a sign */
    if (*p == '+' || *p == '-')
	p++;

    /* check for a string of digits */
    while (isdigit(*p))
	p++, digits++;

    /* make sure there was at least one digit and this is the end */
    if (digits == 0 || *p)
	return (FALSE);

    /* convert the string to an integer and return successfully */
    if (*str == '+') ++str;
    *pval = atoi(str);
    return (TRUE);
}

/* skipspaces - skip leading spaces */
static int skipspaces(void)
{
    int ch;
    while ((ch = getch()) && isspace(ch))
	;
    return (ch);
}

/* isidchar - is this an identifier character */
static int isidchar(int ch)
{
    return (!isspace(ch) && ch != '(' && ch != ')' && ch != '"');
}

/* getch - get the next character */
static int getch(void)
{
    int ch;

    /* check for a lookahead character */
    if (ch = savech) {
	savech = 0;
	return (ch);
    }

    /* check for a buffered character */
    while (*lptr == '\0') {

	/* check for end of file */
	if (ieof)
	    return (StreamEOF);

	/* read another line */
	else {

	    /* read the line */
	    for (lptr = line; (ch = osgetc()) != StreamEOF && ch != '\n'; )
		*lptr++ = ch;
	    *lptr++ = '\n'; *lptr = '\0';
	    lptr = line;
	    ++linenumber;

	    /* check for end of file */
	    if (ch < 0)
	 	ieof = TRUE;
	 }
    }

    /* return the current character */
    return (*lptr++);
}

static int osgetc(void)
{
    int ch;
    if ((ch = StreamGetC(inputStream)) == '\r')
        ch = '\n';
    return ch;
}
