/* Objects.h - definitions for Objects.c */
/*
	Copyright (c) 1993, by David Michael Betz
	All rights reserved
*/

#ifndef __OBJECTS__
#define __OBJECTS__

#include "Streams.h"

typedef void *ObjectPtr;


/* TYPE DISPATCH */

typedef struct {
  long typeTag;
  void (*print)(ObjectPtr obj,Str255 buf);
  long (*size)(ObjectPtr obj);
  ObjectPtr (*copy)(ObjectPtr obj);
  void (*scan)(ObjectPtr obj);
  int (*write)(ObjectPtr obj,StreamHandle fh);
  int (*read)(ObjectPtr obj,StreamHandle fh);
  long (*inspectorCount)(ObjectPtr obj);
  void (*inspectorPrint)(ObjectPtr obj,long data,Str255 buf);
  ObjectPtr (*inspectorOpen)(ObjectPtr obj,long data);
} TypeDispatch,*TypeDispatchPtr;

#define BrokenHeartTypeTag	0
#define NumberTypeTag		1
#define ObjectTypeTag		2
#define ConsTypeTag		3
#define PropertyTypeTag		4
#define SymbolTypeTag		5
#define StringTypeTag		6
#define VectorTypeTag		7
#define MethodTypeTag		8
#define PackageTypeTag		9
#define CMethodTypeTag		10

/* OBJECT HEADER */

typedef struct {
  TypeDispatchPtr dispatch;
} ObjectHdr;

#define PointerP(o)			(((long)o & 1) == 0)
#define ObjectDispatch(o)		(NumberP(o) ? &NumberDispatch : QuickDispatch(o))
#define QuickDispatch(o)		(((ObjectHdr *)o)->dispatch)
#define SetObjectDispatch(o,v)		(((ObjectHdr *)o)->dispatch = (v))
#define TypeCheck(o,t)			(PointerP(o) && ObjectDispatch(o) == t)


/* NUMBER */

#define NumberP(o)			(((long)o & 1) != 0)
#define BoxNumber(n)			((ObjectPtr)(((n) << 1) | 1))
#define UnboxNumber(o)			((long)o >> 1)

extern TypeDispatch NumberDispatch;


/* OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  ObjectPtr classList;
  ObjectPtr searchList;
  ObjectPtr properties;
  ObjectPtr sharedProperties;
} Object;

#define ObjectP(o)			TypeCheck(o,&ObjectDispatch)
#define ObjectClassList(o)		(((Object *)o)->classList)
#define SetObjectClassList(o,v)		(((Object *)o)->classList = (v))
#define ObjectSearchList(o)		(((Object *)o)->searchList)
#define SetObjectSearchList(o,v)	(((Object *)o)->searchList = (v))
#define ObjectProperties(o)		(((Object *)o)->properties)
#define SetObjectProperties(o,v)	(((Object *)o)->properties = (v))
#define ObjectSharedProperties(o)	(((Object *)o)->sharedProperties)
#define SetObjectSharedProperties(o,v)	(((Object *)o)->sharedProperties = (v))

extern TypeDispatch ObjectDispatch;
ObjectPtr NewObject(void);
ObjectPtr CloneObject(ObjectPtr obj);


/* CONS OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  ObjectPtr car;
  ObjectPtr cdr;
} ConsObject;

#define ConsP(o)			TypeCheck(o,&ConsDispatch)
#define Car(o)				(((ConsObject *)o)->car)  
#define SetCar(o,v)			(((ConsObject *)o)->car = (v))
#define Cdr(o)				(((ConsObject *)o)->cdr)  
#define SetCdr(o,v)			(((ConsObject *)o)->cdr = (v))

extern TypeDispatch ConsDispatch;
ObjectPtr Cons(ObjectPtr car,ObjectPtr cdr);


/* PROPERTY OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  ObjectPtr tag;
  ObjectPtr value;
} PropertyObject;

#define PropertyP(o)			TypeCheck(o,&PropertyDispatch)
#define PropertyTag(o)			(((PropertyObject *)o)->tag)  
#define SetPropertyTag(o,v)		(((PropertyObject *)o)->tag = (v))
#define PropertyValue(o)		(((PropertyObject *)o)->value)  
#define SetPropertyValue(o,v)		(((PropertyObject *)o)->value = (v))

extern TypeDispatch PropertyDispatch;
ObjectPtr NewPropertyObject(ObjectPtr tag,ObjectPtr value);


/* SYMBOL OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  ObjectPtr package;
  ObjectPtr printName;
  ObjectPtr value;
} SymbolObject;

#define SymbolP(o)			TypeCheck(o,&SymbolDispatch)
#define SymbolPackage(o)		(((SymbolObject *)o)->package)  
#define SetSymbolPackage(o,v)		(((SymbolObject *)o)->package = (v))  
#define SymbolPrintName(o)		(((SymbolObject *)o)->printName)  
#define SetSymbolPrintName(o,v)		(((SymbolObject *)o)->printName = (v))
#define SymbolValue(o)			(((SymbolObject *)o)->value)  
#define SetSymbolValue(o,v)		(((SymbolObject *)o)->value = (v))

extern TypeDispatch SymbolDispatch;
ObjectPtr NewSymbolObject(ObjectPtr printName);


/* STRING OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  long size;
/*unsigned char data[0];*/
} StringObject;

#define StringP(o)			TypeCheck(o,&StringDispatch)
#define StringSize(o)			(((StringObject *)o)->size)  
#define SetStringSize(o,v)		(((StringObject *)o)->size = (v))
#define StringDataAddress(o)		((unsigned char *)o + sizeof(StringObject))
#define StringElement(o,i)		(StringDataAddress(o)[i])  
#define SetStringElement(o,i,v)		(StringDataAddress(o)[i] = (v))  

extern TypeDispatch StringDispatch;
ObjectPtr NewStringObject(unsigned char *data,long size);
ObjectPtr NewCStringObject(char *str);


/* VECTOR OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  long size;
/*ObjectPtr data[0];*/
} VectorObject;

#define VectorP(o)			TypeCheck(o,&VectorDispatch)
#define VectorSize(o)			(((VectorObject *)o)->size)  
#define SetVectorSize(o,v)		(((VectorObject *)o)->size = (v))
#define VectorDataAddress(o)		((ObjectPtr *)((unsigned char *)o + sizeof(VectorObject))) 
#define VectorElement(o,i)		(VectorDataAddress(o)[i])
#define SetVectorElement(o,i,v)		(VectorDataAddress(o)[i] = (v))

extern TypeDispatch VectorDispatch;
ObjectPtr NewVectorObject(ObjectPtr *data,long size);


/* METHOD OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  ObjectPtr methodClass;
  ObjectPtr literals;
  ObjectPtr code;
} MethodObject;

#define MethodP(o)			TypeCheck(o,&MethodDispatch)
#define MethodClass(o)			(((MethodObject *)o)->methodClass)  
#define SetMethodClass(o,v)		(((MethodObject *)o)->methodClass = (v))
#define MethodLiterals(o)		(((MethodObject *)o)->literals)  
#define SetMethodLiterals(o,v)		(((MethodObject *)o)->literals = (v))
#define MethodCode(o)			(((MethodObject *)o)->code)  
#define SetMethodCode(o,v)		(((MethodObject *)o)->code = (v))

extern TypeDispatch MethodDispatch;
ObjectPtr NewMethodObject(ObjectPtr literals,ObjectPtr code);


/* PACKAGE OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  ObjectPtr name;
  ObjectPtr hashTable;
} PackageObject;

#define PackageP(o)			TypeCheck(o,&PackageDispatch)
#define PackageName(o)			(((PackageObject *)o)->name)  
#define SetPackageName(o,v)		(((PackageObject *)o)->name = (v))  
#define PackageHashTable(o)		(((PackageObject *)o)->hashTable)  
#define SetPackageHashTable(o,v)	(((PackageObject *)o)->hashTable = (v))

extern TypeDispatch PackageDispatch;
ObjectPtr NewPackageObject(ObjectPtr printName);


/* C METHOD OBJECT */

typedef struct {
  TypeDispatchPtr dispatch;
  char *name;
  void (*handler)(void);
} CMethodObject;

#define CMethodP(o)			TypeCheck(o,&CMethodDispatch)
#define CMethodName(o)			(((CMethodObject *)o)->name)  
#define SetCMethodName(o,v)		(((CMethodObject *)o)->name = (v))  
#define CMethodHandler(o)		(((CMethodObject *)o)->handler)  
#define SetCMethodHandler(o,v)		(((CMethodObject *)o)->handler = (v))

extern TypeDispatch CMethodDispatch;
ObjectPtr NewCMethodObject(char *name,void (*handler)(void));


/* BROKEN HEART */

typedef struct {
  TypeDispatchPtr dispatch;
  ObjectPtr forwardingAddress;
} BrokenHeart;

#define BrokenHeartP(o)			TypeCheck(o,&BrokenHeartDispatch)
#define ForwardingAddress(o)		(((BrokenHeart *)o)->forwardingAddress)  
#define SetForwardingAddress(o,v)	(((BrokenHeart *)o)->forwardingAddress = (v))  

extern TypeDispatch BrokenHeartDispatch;


/* definitions for the stop and copy collector */

typedef struct {
  unsigned char *base;
  unsigned char *free;
  unsigned char *top;
} MemorySpace;

/* stack manipulation macros */
#define check(n)	do { if (sp - (n) < &stack[0]) StackOverflow(); } while (0)
#define cpush(v)	do { if (sp > &stack[0]) push(v); else StackOverflow(); } while (0)
#define push(v)		(*--sp = (v))
#define pop()		(*sp++)
#define drop(n)		(sp += (n))

/* function table entry structure */
typedef struct {
  char *name;
  void (*handler)(void);
} FunctionTableEntry;

/* global variables */
extern ObjectPtr nilObject;
extern ObjectPtr trueObject;
extern ObjectPtr falseObject;
extern ObjectPtr symbolPackage;
extern ObjectPtr wordPackage;
extern ObjectPtr *stack,*sp,*fp,*stackTop,code;
extern unsigned char *cbase,*pc;

/* prototypes */
void InitObjectMemory(long size,long stackSize);
void CollectGarbage(void);
void CopyRootObjects(void);
void GarbageCollectionDone(void);
ObjectPtr CopyObject(ObjectPtr obj);
void ObjectPrint(ObjectPtr obj,Str255 buf);
long ObjectSize(ObjectPtr obj);
long ObjectInspectorCount(ObjectPtr obj);
void ObjectInspectorPrint(ObjectPtr obj,long refCon,Str255 buf);
ObjectPtr ObjectInspectorOpen(ObjectPtr obj,long data);
ObjectPtr GetProperty(ObjectPtr obj,ObjectPtr tag);
ObjectPtr GetSharedProperty(ObjectPtr obj,ObjectPtr tag);
ObjectPtr GetInheritedProperty(ObjectPtr obj,ObjectPtr tag);
long ListLength(ObjectPtr list);
ObjectPtr ListElement(ObjectPtr list,long n);
ObjectPtr InternCString(ObjectPtr package,char *str);
ObjectPtr InternSymbol(ObjectPtr package,ObjectPtr printName);
void StackOverflow(void);
void ResetStack(void);
int SaveWorkspace(StreamHandle fh);
int RestoreWorkspace(StreamHandle fh);

#endif