"""
t3Image.py

Provides a class representing a t3 image file.
"""

VERSION = "1.0.1"

import struct, time

class FutureVersionError:
	pass
	
class MalformedFileError:
	pass

T3VERSION = 1
REFERENCESIG = "T3-image\015\012\032"					

class T3Image:
	"""
	A t3 image file.
	"""
	
	def __init__(self, file, options={"ALL": True}):
		self.filestream = open(file, "rb")
		self.filestreamidx = 0
		self.blocks = []
		self.options = options
		self.lastcpdf = None
		self.mcld = None
		self.functionList = []
#		try:
		self.read()
#		except struct.error:
#			raise MalformedFileError
		self.build_pools()
		self.build_obj_dict()
		
							
	def get_block(self, blocktype):
		"""
		Returns one of the block classes or None if we're not
		currently interested in that type of block. The advantage of doing it
		this way is that we only have to bother reading or importing blocks
		that we do care about. For example, if we were simply running through a
		t3 file in order to rip any resources it contained, we could put "MRES"
		in self.allowed_blocktypes and ignore all others.
		"""
		if "ALL" not in self.options.keys():
			if blocktype not in self.options.keys():
				return None
		exec("import t3" + blocktype)
		return eval("t3" + blocktype + "." + blocktype + "_Block")
								

	def read(self):
		self.filestream.seek(0)
		self.read_header()
		while True:
			blockattrs = self.read_block_header()
			# get the appropriate class for the block
			blockclass = self.get_block(blockattrs["s_blocktype"])
			# add an instance of it to self.blocks
			if blockclass == None:
				self.filestream.seek(self.filestream.tell() + 
											blockattrs["i_size"])
			else:
				self.blocks.append(blockclass(self.filestream,
											self, blockattrs))
				if blockattrs["s_blocktype"] == "CPDF":
					self.lastcpdf = self.blocks[-1]
				# if the block we just added was the last one, we exit
			if blockattrs["s_blocktype"] == "EOF":
				break
		# if saveafterEOF is a key in the options passed to us, we save all the
		# data to a file with name of the value of options[key].
		fn = self.options.get("saveafterEOF", False)
		if fn:
			f = open(fn, "wb")
			f.write(self.filestream.read())
			f.close()
		
	def read_header(self):
		"""
		Reads the header of a t3 image file, saving various data
		to the instance.
		"""
		sigbytes = self.filestream.read(11)
		self.b_sigOK = (sigbytes == REFERENCESIG)
		
		uint2bytes = self.filestream.read(2)
		self.i_version = struct.unpack("<H", uint2bytes)[0]
		
		if self.i_version > T3VERSION:
			raise FutureVersionError
		
		reserved = self.filestream.read(32)
		self.s_reservedFutureBytes = repr(reserved[:28])
		self.s_reservedCompilerBytes = repr(reserved[28:])
		self.b_reservedZeroBytesAreZero = (reserved[:28] == "\000" * 28)
		
		self.s_timestamp = self.filestream.read(24)
		self.struct_timestamp = time.strptime(self.s_timestamp)
		self.filestreamidx += 69
		
	def read_block_header(self):
		"""
		Read the header of a block. This initialises various data that is
		common to all blocktypes. This should be called before examining
		the block's actual contents.
		"""
		headerdict = {}
		headerdict["s_blocktype"] = self.filestream.read(4).strip()
		headerdict["i_size"] = struct.unpack("<I", self.filestream.read(4))[0]
		flags = struct.unpack("<H", self.filestream.read(2))[0]
		headerdict["l_flags"] = [bool(flags & (2**i)) for i in xrange(16)]
		headerdict["b_mandatoryFlag"] = headerdict["l_flags"][0]
		self.filestreamidx += 10
		return headerdict
		
	def get_mcld(self):
		if self.mcld == None:
			for block in self.blocks:
				if block.s_blocktype == "MCLD":
					self.mcld = block
					break
		return self.mcld
		
	def build_pools(self):
		"""
		Scans through the blocks, building dictionaries for both constant data
		and byte code pools.
		"""
		constant = {}
		code = {}
		for block in self.blocks:
			if block.s_blocktype == "CPPG":
				if block.data["b_bytecodePool"]:
					code[block.data["i_pageIndex"]] = block
				else:
					constant[block.data["i_pageIndex"]] = block
		self.constantPool = constant
		self.codePool = code
		
	def build_obj_dict(self):
		objdict = {}
		for block in self.blocks:
			if block.s_blocktype == "OBJS":
				for obj in block.data["l_objects"]:
					objdict[obj.i_id] = obj
		self.objectDict = objdict

	def report_header(self):
		sl = []
		sl.append("Signature Present: ")
		sl.append(str(self.b_sigOK) + "\n")
		sl.append("Version: ")
		sl.append(str(self.i_version) + "\n")
		sl.append("Reserved Bytes are Zero: ")
		sl.append(str(self.b_reservedZeroBytesAreZero) + "\n")
		sl.append("Reserved Compiler Bytes: ")
		sl.append(self.s_reservedCompilerBytes + "\n")
		sl.append("Timestamp: ")
		sl.append(self.s_timestamp + "\n")
		return "".join(sl)
		
	def report_data_blocks(self):
		s = []
		for block in self.blocks:
			# ignore pool block
			if block.s_blocktype != "CPPG":
				s.append(block.report())
		return ("\n" + ("*" * 79) + "\n\n").join(s)
		
	def report_data_blocks_nodebug(self):
		s = []
		for block in self.blocks:
			# ignore pool blocks and data blocks
			if block.s_blocktype not in ["CPPG", "SRCF", "GSYM", "MHLS",
										"MACR"]:
				s.append(block.report())
		return ("\n" + ("*" * 79) + "\n\n").join(s)
		
	def report_functions(self):
		s = []
		for function in self.functionList:
			s.append(function.report())
		return ("\n" + ("#" * 20) + "\n").join(s)