"""
t3GlobalSymbolTable.py

This module provides classes for all of the data types that can be included in
a Global Symbol Table block in a t3 image file, as well as an abstract class
representing the block itself.
"""

import struct
from t3Block import *

class GSYM_DataType:
	def __repr__(self):
		self.__str__()
	
class GSYM_Function(GSYM_DataType):
	def __init__(self, s):
		offset, argc, var, hasret = struct.unpack("<IH2B", s[:8])
		self.offset = offset
		self.argc = argc
		self.takesVariableArguments = bool(var)
		self.hasReturnValue = bool(hasret)
		
	def __str__(self):
		s = "Function(offset: " + str(self.offset) + ", argc: " + str(self.argc)
		if self.takesVariableArguments:
			s = s + ", takes variable number of arguments"
		if self.hasReturnValue:
			s = s + ", has a return value"
		s = s + ")"
		return s
		
class GSYM_Object(GSYM_DataType):
	def __init__(self, s):
		self.objectID, self.modifyingObjectID = struct.unpack("<2I", s[:8])
		
	def __str__(self):
		s = "Object(object ID: " + str(self.objectID)
		if self.modifyingObjectID > 0:
			s = s + ", modifying object ID: " + str(self.modifyingObjectID)
		s = s + ")"
		return s
		
class GSYM_Property(GSYM_DataType):
	def __init__(self, s):
		self.propertyID = struct.unpack("<H", s[:2])[0]
				
	def __str__(self):
		s = "Property(property id: " + str(self.propertyID) + ")"
		return s
		
class GSYM_IntrinsicFunction(GSYM_DataType):
	def __init__(self, s):
		idx, depidx, hasret, minarg, maxarg, var = struct.unpack("<2HB2HB", s[:10])
		self.functionSetIndex = idx
		self.functionSetDependencyIndex = depidx
		self.hasReturnValue = bool(hasret)
		self.minimumArguments = minarg
		self.maximumArguments = maxarg
		self.takesVariableArguments = var
				
	def __str__(self):
		s = "IntrinsicFunction(function set index: " + str(self.functionSetIndex) 
		s = s + ", function set dependency index: " + str(self.functionSetDependencyIndex)
		s = s + ", minimum arguments: " + str(self.minimumArguments)
		s = s + ", maximum arguments: " + str(self.maximumArguments)
		if self.takesVariableArguments:
			s = s + ", takes variable number of arguments"
		if self.hasReturnValue:
			s = s + ", has a return value"
		s = s + ")"
		return s

class GSYM_IntrinsicClass(GSYM_DataType):
	def __init__(self, 	s):
		idx, objid = struct.unpack("<HI", s[:6])
		self.metaclassDependencyTableIdx = idx
		self.objectIDofMetaclassInstance = objid
			
	def __str__(self):
		s = "InstrinsicClass(metaclass dependency table index: " + str(self.metaclassDependencyTableIdx)
		s = s + ", object id of metaclass instance: " + str(self.objectIDofMetaclassInstance)
		return s

class GSYM_Enumerator(GSYM_DataType):
	def __init__(self, 	s):
		self.internalEnumeratorID = struct.unpack("<I", s[:4])[0]
			
	def __str__(self):
		s = "Enumerator(internal enumerator ID: " + str(self.internalEnumeratorID) + ")"
		return s

GSYMTypedict = {1: GSYM_Function,
				2: GSYM_Object,
				3: GSYM_Property,
				6: GSYM_IntrinsicFunction,
				9: GSYM_IntrinsicClass,
				10: GSYM_Enumerator}


class GSYM_Block(Block):
	"""
	A global symbol table block. Normally this will only be included in an
	image file if it was compiled for debugging.
	"""
	name = "Global Symbol Table Block"
	debug = True

	def __init__(self, datastream, image, attribs):
		Block.__init__(self, datastream, image, attribs)
		self.read_block()
		
	def read_block(self):
		"""
		Reads the data included in the block.
		"""
		n = struct.unpack("<I", self.datastream.read(4))[0]
		entries = []
		for i in xrange(n):
			entrydict = {}
			lenname, lenextradata, typecode = struct.unpack("<HHH",
													self.datastream.read(6))
			entrydict["s_name"] = self.datastream.read(lenname)
			entrydict["i_typecode"] = typecode
			if GSYMTypedict.has_key(typecode):
				entrydict["o_entry"] = GSYMTypedict[typecode](
											self.datastream.read(lenextradata))
			else:
				entrydict["o_entry"] = "UNKNOWN"
			entries.append(entrydict)
		self.data["l_entries"] = entries

	def report_data(self):
		sl = []
		for entry in self.data["l_entries"]:
			sl.append(entry["s_name"] + ": ")
			sl.append(str(entry["o_entry"]) + "\n")
		return "".join(sl)