#charset "us-ascii"

/*

RAP 1.2 Goto Copyright (c) by Steve Breslin
(email: versim@hotmail.com)

License:

You can use this material however you want, but if you decide to
publicize anything that uses, #include's or otherwise borrows from
this material, you have to make your improvements publicly available,
and advertise them to the IF community.

That way, you will help keep this up to date for everybody else, and
everybody else will help keep it up to date for you.

You may redistribute this verbatim or in modified form only if you
keep the copyrights and license intact.

Feel encouraged to release your source code along with your game,
though this isn't a requirement.

 * Instructions / caveats:
 * 
 * To use this module, you must provide all relevant Room objects with
 * suitable vocabWords. Then just include this file (along with the
 * base RAP kernel and planbase) in your project window in workbench,
 * or otherwise include it in your build directives.
 *
 * If you want the player to be able to order NPC's to "go to," also
 * include the Order module in your build.
 *
 * The included test-game implements "goto" and includes the Order
 * module, so you can experiment with everything there.
 *
 * There's a major problem with the mechanism at present: it doesn't
 * just allow the player to move between rooms, but allows all other
 * relevant plans, such as key searching, unlocking, opening, etc.
 *
 * In a future version of the kernel, I intend to add an option to
 * retrieve not only an action, but also a full path generating
 * that action. This could then be searched for offending plans,
 * and if it contains one, the goto travel could then be aborted.
 *
 * As it is now, the goto action is aborted when a non-travel action
 * is returned by rFind. This means that a player might move towards
 * a key, but will not do any actions other than travel actions, so
 * the player won't pick up the key for example. Still, this will
 * effectively lead your player to the key.
 */
#include <adv3.h>
#include <en_us.h>

/* We define two verbs, rGoto and rContinue. */

DefineTAction(rGoto);

modify rGotoAction
    objInScope(obj) { return true; }
;

VerbRule(rGoto)
    'goto' singleDobj | 'go' 'to' singleDobj
    : rGotoAction
    verbPhrase = 'go/going to (where)'
;

DefineIAction(rContinue)
    execAction() {
        if (gPlayerChar.rGotoDest)

            /* rContinue just remaps to rGoto, given the player's
             * goto-travel destination.
             */
            replaceAction(rGoto, gPlayerChar.rGotoDest);
        else
            "There is no current goto-travel pending. ";
    }
;

VerbRule(rContinue)
    'c' | 'cont' | 'continue'
    : rContinueAction
    verbPhrase = 'continue/continuing'
;

/* We modify thing, so it can verify and process the goto action. */

modify Thing
    dobjFor(rGoto) {
        verify() {
            if (!ofKind(BasicLocation))

                /* By default we do not allow the player to go to
                 * game objects. You may wish to replace this verify
                 * method for certain landmark objects.
                 */
                illogical('Please try: go to (room).');

            if (gActor != gPlayerChar &&
                                 !gActor.propDefined(&rObeyOrder))

                /* The Order module defines Rapper.rObeyOrder. If
                 * rObeyOrder is undefined, we disallow "go to"
                 * commands directed at NPC's.
                 */
                illogical('Only the player character can go to.');
        }
        check() {}
        action() {

            /* We let the actor handle goto processing. */
            gActor.rGoto(self);
        }
    }
;

modify Rapper

    /* rGoto is the entry point for goto processing.
     *
     * If the actor is the PC, we get the rPCGoto method to handle
     * the processing. Otherwise, we treat this as any other order,
     * passing process to the rObeyOrder() method, which will by
     * default instantiate an rAnimateDaemon for carrying out the
     * order.
     */
    rGoto(dest) {
        if (self == gPlayerChar)
            rPCGoto(dest);
        else if (propDefined(&rObeyOrder))
            rObeyOrder(rIn, dest);
    }

    /* rPCGoto() handles goto commands for the PC. We check if the
     * destination is achieved, and if not, try to find a path to the
     * destination. We allow travel to proceed if the action returned
     * is an rIn action. Finally, if we have not arrived, we print
     * some instructions for the player, and record the final
     * destination in the rGotoDest property.
     */
    rPCGoto(dest) {
        local rstep, param, x;
        if (location == dest) {
            "\n[Destination achieved.] ";
            exit;
        }
        x = rFind(rIn, dest);
        if (dataType(x) != TypeList) {
            "Sorry, no path found. ";
            exit;
        }
        rstep = x[1];
        param = x[2];
        if (rstep != rIn) {
            "Sorry, no path found. ";
            exit;
        }
        "\n[You travel to <<param.theName>>";
        if (param != dest)
            ", on the way towards <<dest.theName>>";
        ".]\n";
        rstep.rAction(self, param);
        if (dest != location) {
            "\b[Press [enter] or type 'continue' to continue
            goto-travel.]\b";
            rGotoDest = dest;
        }
        else {
            rGotoDest = nil;
            "\n[Destination achieved.]\n";
        }
    }

    /* This records the desired destination, so we can refer back to
     * it with the rGoto action processing.
     */
    rGotoDest = nil
;

/* We add a new StringPreParser to allow the player to enter an empty
 * line for a 'continue' command. If rGotoDest is not currently
 * defined, we just return the string. If it is defined, but the player
 * is doing something other than continuing the goto travel, we reset
 * the rGotoDest to nil.
 */
GotoPreParser: StringPreParser
    doParsing(str, which) {
        if (gPlayerChar.rGotoDest) {
            if (str == '')
                str = 'continue';
            if (str not in
                    ('continue', 'c', 'cont', 'contin', 'continu') )
                gPlayerChar.rGotoDest = nil;
        }
        return str;
    }
;

