#charset "us-ascii"

/* 
 *  Copyright (c) 2004 - 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 London Cabs Library Extension
 *
 *  cabbie.t
 *
 *  Defines the london cabbie actor and associated state
 *  and topic classes. 
 */

#include <adv3.h>
#include <en_us.h>

/*
 *  Defines the destination travel topic class used
 *  for indicating which locations our cabs can travel to.
 */
class DestTravelTopic: SpecialTopic
{
    location = cabbieConvNode

    timesToSuggest = 0  
        
    topicResponse()
    {
        local veh;

        /* get our actor's cab */
        veh = getActor().location;

        /* tell the cab to travel to the topic's destination */
        veh.travelToLoc(dest);
    }
}

/*
 *  Common Cabbie Travel Conv Actor State. This state
 *  encapsulates attributes and methods common to both
 *  growler and hansom cabbies when they are in "travel
 *  mode".
 */
class CabbieTravelConv: InConversationState
{
    attentionSpan = nil
    takeTurn()
    {
        if (gPlayerChar.location == getActor().location)
            getActor().initiateConversation(nil, 'where-to'); 
        else
        {
            getActor().endConversation();
            getActor().location.leaveLoc();
        }
    }
    beforeTravel(traveler, connector)
    {
        local pAction, t;
        
        t = getActor().getTraveler(getActor());
        pAction = gAction.parentAction;

        if (t == traveler
            && pAction 
            && pAction.ofKind(TravelAction))
        {
            local obj = getActor().location;

            mainReport(&exitCabMsg, obj);
            exitAction;
        }
        else
            inherited(traveler, connector);
    }
}

class WaitingBoardingActorState: ActorState
{
    waitingCnt = 0

    takeTurn()
    {
        local actor, veh, loc;
        
        actor = getActor();
        veh = actor.location;
        loc = veh.location;
        if (loc)
        {
            if (waitingCnt > 4)
            {
                veh.driveOff();
                waitingCnt = 0;
            }
            
            ++waitingCnt;
        }
    }
}

/*
 *  Our London Cabbie is both a SoundObserver and a Person
 *  which means he can react to sound events, such as the sound
 *  of the whistle when he's in the same location, and possesses
 *  various states to govern his behavior.
 *
 *  In this implementation, our cabbie will become either a
 *  hansom or growler cabbie, depending on the number of consecutive
 *  whistle blows.
 */
londonCabbie: SoundObserver, Person
{
    vocabWords  = (cabLanguageGlobals.londonCabbieVocab)
    name        = (cabLanguageGlobals.londonCabbieName)

    whistleCnt  = 0
    fuse        = nil
    dest        = nil
    cab         = nil

    /*
     *  If our cabbie is within earshot then
     *  we don't need to request a cab!
     */
    notifySoundEvent(event, source, info)
    {
        reportFailure(&callCabFailure);
        exitAction;
    }

    requestCab(loc)
    {
        whistleCnt++;
        dest = loc;

        if (fuse == nil)
            fuse = new Fuse(self,
                &dispatchCab, 1);
    }

    dispatchCab()
    {
        local prop = &adjective;

        // We only allow one cab in the game at a time
        if (cab)
            cab.moveInto(nil);

        // remove cabbie adjectives
        cmdDict.removeWord(self, self.(prop), prop);

        // choose the cab based on the whistleCnt
        switch(whistleCnt)
        {
            case 1:
                cab = growlerCab;
                name = cabLanguageGlobals.growlerCabbieName;
                adjective += cabLanguageGlobals.growlerCabbieAdj;
                setCurState(growlerActorState);
                break;

            case 2:
                cab = hansomCab;
                name = cabLanguageGlobals.hansomCabbieName;
                adjective += cabLanguageGlobals.hansomCabbieAdj;
                setCurState(hansomActorState);
                break;
                
            default:
                cab = growlerCab;
                name = cabLanguageGlobals.growlerCabbieName;
                adjective += cabLanguageGlobals.growlerCabbieAdj;
                setCurState(growlerActorState);
        }

        // add cabbie adjectives
        cmdDict.addWord(self, self.(prop), prop);

        // move the dispatcher to the cabbie
        moveInto(cab);

        // let the cab handle the travel
        cab.calledToLoc(dest);

    
        // initialize the fuse, dest, and whistleCnt values
        whistleCnt  = 0;
        fuse        = nil;
        dest        = nil;
    }
}

/*------------------------------------------------------------
 *  HANSOM CAB STATES & TOPICS
 *------------------------------------------------------------
 */

+ hansomActorState: WaitingBoardingActorState
{
    takeTurn()
    {
        if (gPlayerChar.location == getActor().location)
            getActor().initiateConversation(hansomTravelConv, 
                cabLanguageGlobals.whereToTag); 
        else
            inherited();
    }
}

/*
 *  The default any topic handles our cabbie's response to 
 *  any query or request to engage his attention.
 */
++ DefaultAnyTopic
    "<<mainReport(&hansomDefault)>>"
;

+ hansomTravelConv: CabbieTravelConv
{
    stateDesc   = "<<mainReport(&hansomStateDesc)>>"
    specialDesc = "<<mainReport(&hansomSpecialDesc)>>"
}

/*
 *  The default any topic handles our cabbie's response to 
 *  any query or request to engage his attention.
 */
++ DefaultAnyTopic
    "<<mainReport(&hansomDefault)>>"
;

/*------------------------------------------------------------
 *  GROWLER CAB STATES & TOPICS
 *------------------------------------------------------------
 */

+ growlerActorState: WaitingBoardingActorState
{
    takeTurn()
    {
        if (gPlayerChar.location == getActor().location)
            getActor().initiateConversation(growlerTravelConv, 
                cabLanguageGlobals.whereToTag); 
        else
            inherited();
    }
}

/*
 *  The default any topic handles our cabbie's response to 
 *  any query or request to engage his attention.
 */
++ DefaultAnyTopic
    "<<mainReport(&growlerDefault)>>"
;

+ growlerTravelConv: CabbieTravelConv
{
    stateDesc   = "<<mainReport(&growlerStateDesc)>>"
    specialDesc = "<<mainReport(&growlerSpecialDesc)>>"
}

/*
 *  The default any topic handles our cabbie's response to 
 *  any query or request to engage his attention.
 */
++ DefaultAnyTopic
    "<<mainReport(&growlerDefault)>>"
;

/*------------------------------------------------------------
 *  CONVERSATION NODES AND DEFAULT TRAVEL TOPICS
 *------------------------------------------------------------
 */

/* 
 *  The conversation node handles our cabbie's one request.
 */
+ cabbieConvNode: ConvNode
{
    name = (cabLanguageGlobals.whereToTag)

    npcGreetingMsg = "<<mainReport(&whereToQuery)>>"
    npcContinueMsg = "<<mainReport(&whereToQuery)>>"
}