#charset "us-ascii"
#include <adv3.h>

/*

Double-Breadth (Bidirectional) Path Finder

Steve Breslin, 2004
steve.breslin@gmail.com

This module provides a class that implements a more sophisticated "dual
breadth-first" search algorithm for finding the shortest path between
two nodes in an undirected, unweigheted graph. The basic DoubleBreadth
class makes no assumptions about the nature of the underlying graph,
except that it is undirected and unweighted; subclasses must provide the
concrete details about the graph being traversed.

Everyone is granted permission to use and modify this file for any
purpose, provided that the original author is credited.

*/

class DoubleBreadth: object

    /* Find the best path from 'fromNode' to 'toNode', which are nodes in
     * the underlying graph.  We'll return a vector consisting of graph
     * nodes, in order, giving the shortest path between the nodes.  Note
     * that 'fromNode' and 'toNode' are included in the returned vector.
     *
     * If the two nodes 'fromNode' and 'toNode' aren't connected in the
     * graph, we'll simply return nil.  
     */
    findPath(fromNode, toNode) {
        
        /* nodeVectorForward is a vector of nodes, beginning with the
         * fromNode and expanding outward along connections from the
         * fromNode to other nodes.
         *
         * We initialize the vector with the fromNode at index 1.
         */
        local nodeVectorForward = new Vector(16, [fromNode]);

        /* nodeVectorForwardPIdx is a vector of "parent indexes",
         * corresponding with nodeVectorForward. E.g.: if
         * nodeVectorForwardPIdx[5] equals 3, that means that
         * nodeVectorForward[5]'s parent is nodeVectorForward[3].
         */
        local nodeVectorForwardPIdx = new Vector(16, [nil]);

        /* We are also building a path backwards from the toNode, so
         * we have corresponding vectors for keeping track of that
         * information:
         */
        local nodeVectorBack = new Vector(16, [toNode]);
        local nodeVectorBackPIdx = new Vector(16, [nil]);

        /* until we have found an intersection of nodeVectorForward
         * and nodeVectorBack, we have not succeeded in finding the
         * path.
         */
        local success = nil;

        /* We wrap the main search in a try, so we can exit immediately
         * once we have found the path.
         */
        try {

            /* We check if either nodeVectorBack or nodeVectorForward
             * is smaller than 'i'. If either or both are, fromNode and
             * toNode are not connected: the vector has completed its
             * expansion into the full graph connected to that node.)
             */
            for(local i = 1 ; i <= nodeVectorForward.length()
                           && i <= nodeVectorBack.length() ; i++)  {

                /* add each adjacent item to nodeVectorForward */
                forEachAdjacent(nodeVectorForward[i], new function(adj, dist) {

                    /* add the adjacent node and its parent only if it's
                     * not already in the node vector.
                     */
                    if(!nodeVectorForward.indexOf(adj)) {
                        nodeVectorForward.append(adj);
                        nodeVectorForwardPIdx.append(i);
                        if(nodeVectorBack.indexOf(adj)) {
                            success = true;
                            exit;
                        }
                    }
                });

                /* add each adjacent item to nodeVectorBack */
                forEachAdjacent(nodeVectorBack[i], new function(adj, dist) {

                    /* add the adjacent node and its parent only if it's
                     * not already in the node vector.
                     */
                    if(!nodeVectorBack.indexOf(adj)) {
                        nodeVectorBack.append(adj);
                        nodeVectorBackPIdx.append(i);
                        if(nodeVectorForward.indexOf(adj)) {
                            success = true;
                            exit;
                        }
                    }
                });
            }
        }
        finally {
            if(success)
                return calcPath(nodeVectorForward,
                                nodeVectorForwardPIdx,
                                nodeVectorBack,
                                nodeVectorBackPIdx);
            return nil;
        }
    }

    calcPath(nodeVectorForward,
             nodeVectorForwardPIdx,
             nodeVectorBack,
             nodeVectorBackPIdx) {

        local ret = new Vector(8);

        /* We could have arrived here in one of two ways: either the
         * last element of nodeVectorForward was found in
         * nodeVectorBack, or the last element of nodeVectorBack was
         * found in nodeVectorForward. If it's not one, it's the other.
         * (If fromNode and toNode were directly connected, it's both,
         * but in this case we can construct the path either way.)
         *
         * We guess what it is, then correct the guess if necessary.
         */
        local intersect = nodeVectorForward[nodeVectorForward.length];
        if(!nodeVectorBack.indexOf(intersect))
            intersect = nodeVectorBack[nodeVectorBack.length];

        ret.append(intersect); // begin with the intersect node.

        /* append to ret the back path towards the intersect. */
        getPath(nil, nodeVectorBack, nodeVectorBackPIdx, ret, intersect);

        /* prepend the forward vector to the intersect (intersection) */
        getPath(true, nodeVectorForward, nodeVectorForwardPIdx, ret,
                intersect);

        return ret; // return the vector
    }

    getPath(forwards, nodeVec, PIdxVec, ret, intersect) {
        local pIdx = PIdxVec[nodeVec.indexOf(intersect)];
        local curNode;
        while(pIdx) {
            curNode = nodeVec[pIdx];
            if(forwards)
                ret.prepend(curNode);
            else
                ret.append(curNode);
            pIdx = PIdxVec[nodeVec.indexOf(curNode)];
        }
        return ret;
    }

    findDistance(fromNode, toNode) {
        return (findPath(fromNode, toNode).length() - 1);
    }

    /* 
     *   Iterate over everything adjacent to the given node in the
     *   underlying graph.  This routine must simply invoke the given
     *   callback function once for each graph node adjacent to the given
     *   graph node.
     *   
     *   The callback takes two arguments: the adjacent node, and the
     *   distance from 'node' to the adjacent node.  Note that the distance
     *   must be a positive number, as Dijkstra's algorithm depends on
     *   positive distances.  If the graph isn't weighted by distance,
     *   simply use 1 for all distances.
     *   
     *   This method isn't implemented in the base class, since we don't
     *   make any assumptions about the underlying graph.  Subclasses must
     *   provide concrete implementations of this routine to define the
     *   underlying graph.  
     */
    forEachAdjacent(node, func) { /* subclasses must override */ }
;
