/* Copyright (c) 1998 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  plantstd.t - standard definitions
Function
  Provides definitions for the game-defined functions and objects that
  are required by the system and adv.t.
Notes
  
Modified
  06/27/98 MJRoberts  - Creation
*/

/*
 *   Pre-declare all functions, so the compiler knows they are functions.
 *   (This is only really necessary when a function will be referenced
 *   as a daemon or fuse before it is defined; however, it doesn't hurt
 *   anything to pre-declare all of them.)
 */
die: function;
scoreRank: function;
init: function;
initRestore: function;
terminate: function;
pardon: function;
darkTravel: function;
commonInit: function;
initRestore: function;
lightningDaemon: function;
fullScoreStatus: function;
showCredits: function;


/*
 *   Preparse - do some initial preprocessing on the command string.  We
 *   have one unusual need: we want to allow the player to enter numbers
 *   containing decimal points.  The normal number format doesn't allow
 *   decimal points, so we need to modify the command slightly to allow
 *   them.  To handle this, add quotes around any number containing a
 *   decimal point.
 */
preparse: function(cmd)
{
    local cur;
    local len;
    local first_dig;
    local in_quote;
    local sp;
    local have_dot;

    /* make a quick check - if there's no '.', we can skip preprocessing */
    if (find(cmd, '.') = nil)
        return true;

    /* scan the string */
    len := length(cmd);
    for (cur := 1, first_dig := nil, in_quote := nil, sp := true,
         have_dot := nil ;
         cur <= len ; ++cur)
    {
        local ch := substr(cmd, cur, 1);

        /* check to see if we're inside quotes */        
        if (in_quote != nil)
        {
            /* 
             *   check to see if this is the closing quote - if so, we're
             *   no longer inside quotes 
             */
            if (ch = in_quote)
                in_quote := nil;

            /* skip this character */
            continue;
        }

        /* check for special characters */
        switch(ch)
        {
        case '"':
            in_quote := ch;
            break;

        case '\'':
            if (sp)
                in_quote := ch;
            break;
        }

        /* note whether this is a word character or separator character */
        sp := !((ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z')
                || (ch >= '0' && ch <= '9') || ch = '-');

        /* if it's a digit, so note */
        if (ch >= '0' && ch <= '9')
        {
            /* if we're not inside a number, this is the first digit */
            if (first_dig = nil)
            {
                first_dig := cur;
                have_dot := nil;
            }
        }
        else if (ch = '.' && first_dig != nil)
        {
            /* note that we have a dot within a number */
            have_dot := true;
        }
        else
        {
            /* 
             *   this isn't a number - if we're within a number, and we
             *   have a dot, we need to quote the number 
             */
            if (first_dig != nil && have_dot)
            {
                /* add the quotes */
                cmd := substr(cmd, 1, first_dig - 1) + '"'
                       + substr(cmd, first_dig, cur - first_dig)
                       + '"' + substr(cmd, cur, length(cmd));

                /* advance past the extra characters */
                cur += 2;
            }

            /* we're no longer in a number */
            first_dig := nil;
        }
    }

    /* if it ended with a number with a dot, quote it */
    if (first_dig != nil && have_dot)
        cmd := substr(cmd, 1, first_dig - 1) + '"'
               + substr(cmd, first_dig, cur - first_dig)
               + '"' + substr(cmd, cur, length(cmd));

    /* return the modified string */
    return cmd;
}

/*
 *   pre-parse a command line after tokenization 
 */
preparseCmd: function(lst)
{
    local i, len;
    local mod;
    local num;

    /* presume we're not going to change anything */
    mod := nil;
    
    /*
     *   Run through the command.  If we find any phrases "to X", where X
     *   is a single digit, replace X with 0X (for example: change "turn
     *   dial to 3" to "turn dial to 03").
     *   
     *   This is a horrible hack to work around an irritating problem in
     *   the TADS parser with some versions involving numeric adjectives.
     *   The problem is that the parser gives precedence to an object with
     *   a numeric adjective, completely bypassing numObj, when a number
     *   matches the numeric adjective of an in-scope object.  We can work
     *   around this by putting in the equivalent number with a leading
     *   zero so that it doesn't look like a number defined as a
     *   vocabulary word anywhere.
     *   
     *   This hack is specific to this game, because we assume that the
     *   only numeric adjectives defined are single-digit numbers.  Other
     *   games may have to handle other numbers as well.  We also assume
     *   that the only place we care about numObj is in "to X" phrases,
     *   which is true of this game but doesn't generalize very well.
     *   
     *   THIS HACK IS NOT NECESSARY IN TADS 2.2.7 AND LATER.  However,
     *   we're leaving it in for the time being so that people with older
     *   versions won't encounter the bug.  
     */
    for (i := 1, len := length(lst) ; i <= len - 1 ; ++i)
    {
        /* 
         *   if we're looking at "to X", where X is a digit, replace it
         *   with "to 0X" 
         */
        if (lst[i] = 'to' && length(lst[i+1]) = 1
            && find('0123456789', lst[i+1]))
        {
            /* add a leading zero in front of the digit */
            lst[i+1] := '0' + lst[i+1];

            /* note that we've made a change */
            mod := true;
        }
    }

    /*
     *   Look for a few alternative phrasings that people might try for
     *   "look up <number> in cards".  Some players have reported
     *   difficulty finding the standard phrasing.  If the cards are
     *   visible to the player, check for alternative phrasings and
     *   suggest the correct format.
     *   
     *   Presume we won't find an alternative format; if we do, we'll set
     *   'num' to the number we find.  
     */
    num := nil;

    /* check for "find <number>" */
    if (length(lst) = 2
        && lst[1] = 'find'
        && substr(lst[2], 1, 1) = '"' && find(lst[2], '.'))
        num := lst[2];

    /* check for "find/read/get/take/examine/inspect/x card <number>" */
    if (length(lst) = 3
        && (lst[1] = 'find' || lst[1] = 'read' || lst[1] = 'get'
            || lst[1] = 'take' || lst[1] = 'x' || lst[1] = 'examine'
            || lst[1] = 'examine')
        && lst[2] = 'card'
        && substr(lst[3], 1, 1) = '"' && find(lst[3], '.'))
        num := lst[3];

    /* check for "look at/for card <number>" */
    if (length(lst) = 4
        && (lst[1] = 'look' || lst[1] = 'l')
        && (lst[2] = 'at' || lst[2] = 'for')
        && lst[3] = 'card'
        && substr(lst[4], 1, 1) = '"' && find(lst[4], '.'))
        num := lst[4];

    /* if we found an alternative phrasing, suggest the preferred phrasing */
    if (num != nil && warehouse_deck.isseen)
    {
        "If you want to look up a number in the cards, just say
        so&mdash;try LOOK UP <<substr(num, 2, length(num)-2)>> IN CARDS.";
        
        /* we've suggested the alternative; abort the command */
        return nil;
    }

    /* 
     *   if we made any changes, return the modified list; otherwise, just
     *   return true to indicate that we should proceed with the original
     *   command 
     */
    if (mod)
        return lst;
    else
        return true;
}


/*
 *   commonInit() - this is a helper function where we put common
 *   initialization code in a single place.  Both init() and initRestore()
 *   call this.  Since the system may call either init() or initRestore()
 *   during game startup, but not both, it's desirable to have a single
 *   function that both of those functions call for code that must be
 *   performed at startup regardless of how we're starting.  
 */
commonInit: function
{
    /* run in HTML mode */
    "\H+";

    /* make a note if we're running with a pre-2.2.5 run-time */
    if (systemInfo(__SYSINFO_SYSINFO) != true
        || systemInfo(__SYSINFO_VERSION) < '2.2.5')
    {
        "\b\b\(Warning:\) This game requires version 2.2.5 or higher
        of the TADS run-time.  You appear to be using an earlier
        version of TADS.  The game may not function correctly with
        this version.  If possible, you should get the latest TADS
        run-time (try \(ftp://ftp.ifarchive.org/if-archive/\)) before
        playing this game.  If you choose to proceed with this version,
        you may encounter problems.
        \b\bPlease press space to continue, or \(Q\) to quit...";
        switch(inputkey())
        {
        case 'q':
        case 'Q':
            quit();
            break;
        }
    }

    /* show the title */
    "<title>The Plant</title>";

    /* randomize */
    randomize();
}

/*
 *   initRestore() - the system calls this function automatically at game
 *   startup if the player specifies a saved game to restore on the
 *   run-time command line (or through another appropriate
 *   system-dependent mechanism).  We'll simply restore the game in the
 *   normal way.
 */
initRestore: function(fname)
{
    /* perform common initializations */
    commonInit();

    /* tell the player we're restoring a game */
    "\b[Restoring saved position...]\b";

    /* go restore the game in the normal way */
    mainRestore(fname);
}

/*
 *   init - this is the main game start-up function 
 */
init: function
{
    /* perform common initializations */
    commonInit();

    /* start the turn counter */
    setdaemon(turncount, nil);

    /* show some administrative information before we get going */
    "\b\b[If you haven't played this game before, please type
    <a href=Help>HELP</a>.]";

    /* display introductory text */
    "<br height=4>You're just starting to doze off when a jerking motion
    brings you back to alertness.  You look over to see your boss,
    Mr.\ Teeterwaller, struggling to steer the car onto the shoulder
    as the engine dies.  You can see that all of the dashboard lights
    are on as the car jerks to a stop.
    \bThis is turning into a fine business trip. First Mr.\ Teeterwaller
    insists on making the five-hour car trip in the middle of the night
    so the company won't have to pay for a hotel, then you spend an
    hour stuck behind a convoy of slow trucks on Teeterwaller's two-lane
    supposed short-cut, and now his aging bargain-basement car strands
    you out in the middle of nowhere.
    \bTeeterwaller turns off the headlights and turns on
    the hazard lights. <q>I just had this thing in the shop,</q>
    he mumbles.
    <br height=2>";

    /* display title */
    version.sdesc;

    /* move to the starting location */
    parserGetMe().location := inCar;
    inCar.lookAround(true);
    inCar.isseen := true;

    /* show player's inventory if appropriate */
    if (itemcnt(parserGetMe().contents) != 0)
        "\bYou're carrying <<listcont(parserGetMe())>>. ";

    /* initialize the score display */
    scoreStatus(0, 0);

    /* start the car daemon */
    notify(inCar_car, &carDaemon, 0);

    /* start the elevator daemon */
    notify(inElevator, &notifier, 0);

    /* set up teeterwaller's running action daemon */
    notify(teeterwaller, &actorDaemon, 0);

    /* start the lightning daemon */
    setdaemon(lightningDaemon, nil);
}


/*
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */
die: function
{
    "\b*** You have died ***\b";
    exitOptions(nil);
}

/*
 *   Show the end of game options and let the player tell us what to do
 *   next.  This function can be used when the game ends for any reason,
 *   such as the player character getting killed, winning the game, or
 *   reaching any other type of conclusion to the game.  This function
 *   will always take some kind of sweeping action, such as restarting the
 *   game, restoring a saved game, undoing a turn, or quitting.  
 */
exitOptions: function(offerExtras)
{
    /* show the current score */
    scoreRank();

    /* offer the options */
    "\bYou can ";

    if (offerExtras)
        "show the CREDITS, view the SPECIAL edition menu, ";

    "show your FULL score, RESTORE a saved game, RESTART the
    game from the beginning, QUIT the game, or UNDO the last
    command.\n>";

    /* keep going until we get a valid option */
    while (true)
    {
        local resp;

        resp := upper(input());
        if (resp = 'RESTORE')
        {
            resp := askfile('File to restore');
            if (resp = nil)
                "Restore failed. ";
            else if (restore(resp))
                "Restore failed. ";
            else
            {
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
        }
        else if (resp = 'RESTART')
        {
            scoreStatus(0, 0);
            restart();
        }
        else if (resp = 'QUIT' || resp = 'Q')
        {
            terminate();
            quit();
            abort;
        }
        else if (resp = 'UNDO')
        {
            if (undo())
            {
                "(Undoing one command)\b";
                parserGetMe().location.lookAround(true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
            else
                "Sorry, no undo information is available. ";
        }
        else if (resp = 'FULLSCORE' || resp = 'FULL' || resp = 'FULL SCORE')
        {
            fullScoreStatus();
            "\b";
        }
        else if (resp = 'CREDITS' || resp = 'CREDIT')
        {
            showCredits();
            "\b";
        }
        else if (resp = 'SPECIAL')
        {
            /* warn if necessary */
            if (specialVerb.showWarning)
                specialFeatures();
        }

        "\bPlease enter ";
        if (offerExtras)
            "CREDITS, SPECIAL, ";
        "FULL, RESTORE, RESTART, QUIT, or UNDO:\n>";
    }
}

/*
 *   The scoreRank() function displays how well the player is doing.
 *   This default definition doesn't do anything aside from displaying
 *   the current and maximum scores.  Some game designers like to
 *   provide a ranking that goes with various scores ("Novice Adventurer,"
 *   "Expert," and so forth); this is the place to do so if desired.
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game; change the property in the "global" object
 *   if necessary.
 */
scoreRank_internal: function(end_sentence)
{
    local ranks := [[0   'needing lots of management attention']
                    [5   'a promising new hire']
                    [10  'an up-and-coming go-getter']
                    [40  'next in line for a promotion']
                    [60  'the next junior executive']
                    [100 'ready for the executive suite']];
    local i, tot;

    "Your score is <<global.score>> of a possible <<global.maxscore>>,
    in <<global.turnsofar>> moves.  If Mr.\ Teeterwaller were to evaluate
    your performance thus far, he would rank you as ";

    for (i := length(ranks) ; i >= 1 ; --i)
    {
        if (global.score >= ranks[i][1])
        {
            say(ranks[i][2]);
            break;
        }
    }

    if (end_sentence)
        ".\n";
}

scoreRank: function
{
    scoreRank_internal(true);
}


/*
 *   preinit() is called after compiling the game, before it is written
 *   to the binary game file.  It performs all the initialization that can
 *   be done statically before storing the game in the file, which speeds
 *   loading the game, since all this work has been done ahead of time.
 *
 *   This routine puts all lamp objects (those objects with islamp = true) into
 *   the list global.lamplist.  This list is consulted when determining whether
 *   a dark room contains any light sources.
 */
preinit: function
{
    local o;
    
    global.lamplist := [];
    o := firstobj();
    while(o <> nil)
    {
        if (o.islamp)
            global.lamplist := global.lamplist + o;
        o := nextobj(o);
    }
    initSearch();
    writeGameInfo(getGameInfo());
}

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  Note that this function is
 *   called only when the game is about to exit, NOT after dying, before a
 *   restart, or anywhere else.  
 */
terminate: function
{
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  
 */
pardon: function
{
    "I beg your pardon? ";
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj
    location = { return parserGetMe().location; }
    verIoChangeTo(actor) = { }
    ioChangeTo(actor, dobj) = { dobj.doChangeTo(actor, self); }
    ioSynonym('ChangeTo') = 'SetTo' 'MoveTo' 'SlideTo' 'SwitchTo'

    verDoLookupIn(actor, iobj) = { }
    doSynonym('LookupIn') = 'FindIn' 'SearchforIn'
    verIoSearchFor(actor) = { }
    ioSearchFor(actor, dobj) = { dobj.doSearchFor(actor, self); }
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj     // use default definition from adv.t
    verDoLookupIn(actor, iobj) = { }
    doSynonym('LookupIn') = 'FindIn' 'SearchforIn'
    verIoSearchFor(actor) = { }
    ioSearchFor(actor, dobj) = { dobj.doSearchFor(actor, self); }

    doSay(actor) =
    {
        local blottRoots := ['azdrup' 'yijtg' 'lloyr' 'chdral' 'wurtr'
                             'shchrud' 'pozhbr' 'oxv'];
        local blottDeepverbs := [blottSitVerb blottStandVerb blottSpeakVerb
                                 blottQuietVerb blottAttackVerb
                                 blottBefriendVerb blottnianNullVerbs
                                 blottnianNullVerbs];
        local blottSuffixes := ['iq' 'oq' 'onsh' 'ioo' 'im' 'as' 'azim' 'u'];
        local i, cnt;

        /* check each blottnian verb root */
        for (i := 1, cnt := length(blottRoots) ; i <= cnt ; ++i)
        {
            /* check if this root matches */
            if (substr(self.value, 1, length(blottRoots[i])) = blottRoots[i])
            {
                local j, cnt2;
                
                /* check for a valid suffix */
                for (j := 1, cnt2 := length(blottSuffixes) ; j <= cnt2 ; ++j)
                {
                    /* see if the suffix matches */
                    if (substr(self.value, length(blottRoots[i]) + 1, 1000)
                        = blottSuffixes[j])
                    {
                        /* 
                         *   if the suffix is anything but 'u', process it
                         *   as a null verb; otherwise, process it using
                         *   the specialized verb handler for the
                         *   particular verb root they used 
                         */
                        if (blottSuffixes[j] != 'u')
                            blottnianNullVerbs.action(actor);
                        else
                            blottDeepverbs[i].action(actor);

                        /* we're done processing this command */
                        return;
                    }
                }
            }
        }

        /* take special action for certain verbs */
        if (self.value = 'aus')
            ausVerb.action(actor);
        else if (self.value = 'yes')
            yesVerb.action(actor);
        else if (self.value = 'no')
            noVerb.action(actor);
        else
            inherited.doSay(actor);
    }
;

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.
 */
global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    oldscore = 0                                   // score as of previous turn
    maxscore = 100                                    // maximum possible score
    verbose = nil                             // we are currently in TERSE mode
    awakeTime = 0               // time that has elapsed since the player slept
    sleepTime = 400     // interval between sleeping times (longest time awake)
    lastMealTime = 0              // time that has elapsed since the player ate
    eatTime = 200         // interval between meals (longest time without food)
    lamplist = []              // list of all known light providers in the game
    scoreNotify = true              // notify the player when the score changes
    fullScoreList = []         // full score list - list of [score 'why'] lists
    mjr_email =
    {
        if (systemInfo(__SYSINFO_SYSINFO) = true
            && systemInfo(__SYSINFO_LINKS_MAILTO))
        {
            "<a href='mailto:mjr_@hotmail.com'>
            <tt>mjr_@hotmail.com</tt></a>";
        }
        else
        {
            /* just show the address as text */
            "<tt>mjr_@hotmail.com</tt>";
        }
    }
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    sdesc = "<h2><i><<self.namedesc>></i></h2>
             \n<i>TADS Interactive Fiction by Michael J.\ Roberts</i>
             \bRelease <<self.vsnnum>>/Special Edition
             \nCopyright &copy;1998 Michael J.\ Roberts.\b"
    namedesc = "The Plant"
    vsnnum = '2'
;

/*
 *   "Me" is the initial player's actor; the parser automatically uses the
 *   object named "Me" as the player character object at the beginning of
 *   the game.  We'll provide a default definition simply by creating an
 *   object that inherits from the basic player object, basicMe, defined
 *   in "adv.t".
 *   
 *   Note that you can change the player character object at any time
 *   during the game by calling parserSetMe(newMe).  You can also create
 *   additional player character objects, if you want to let the player
 *   take the role of different characters in the course of the game, by
 *   creating additional objects that inherit from basicMe.  (Inheriting
 *   from basicMe isn't required for player character objects -- you can
 *   define your own objects from scratch -- but it makes it a lot easier,
 *   since basicMe has a lot of code pre-defined for you.)  
 */
Me: basicMe
    doHello =
    {
        "I'm not sure to whom you're talking. ";
    }
    conversingWith = nil
    actorAction(verb, dobj, prep, iobj) = 
    {
        /* if we're conversing with someone, let them limit our actions */
        if (self.conversingWith != nil)
            self.conversingWith.converseAction(verb, dobj, prep, iobj);
    }
    answerYes =
    {
        if (self.conversingWith != nil)
            self.conversingWith.answerYes;
        else
            "It's good that you're feeling so positive. ";
    }
    answerNo =
    {
        if (self.conversingWith != nil)
            self.conversingWith.answerNo;
        else
            "Sorry that you're feeling so negative. ";
    }
    travelTo(rm) =
    {
        /*
         *   If I'm carrying the fire hose end, notify the room we're
         *   departing - if it says not to proceed (by returning nil), do
         *   not proceed 
         */
        if (fireHoseEnd.isIn(self))
        {
            if (!self.location.fireHosePreTravelTo(rm))
                return;
        }
        
        /* 
         *   Have Mr. Teeterwaller say what he's doing if appropriate.
         *   Note that we only do this if we're traveling somewhere valid,
         *   and the destination is not an obstacle - in the case of
         *   obstacles, we'll invoke travelTo recursively for the actual
         *   location, so we don't want to add any extra messages while
         *   traversing the obstacles. 
         */
        if (rm != nil && !rm.isobstacle)
            teeterwaller.preFollowMe(self, rm);

        /* do the normal work */
        inherited.travelTo(rm);

        /* do the post-following work if appropriate */
        if (rm != nil && !rm.isobstacle)
            teeterwaller.postFollowMe(self, rm);
    }
;

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function
{
    "***\bYou wake up some time later, feeling refreshed. ";
    global.awakeTime := 0;
}

/* ------------------------------------------------------------------------ */
/*
 *   Modify the way the list of visible items in a particular location is
 *   figured, to make it more configurable at the level of a room.  Rather
 *   than using a single function to compute visibility, we'll move
 *   visibility into the room, so that it can be overridden.  The default
 *   behavior will work the same way as usual, but this provides more
 *   flexibility for overriding.  
 */
modify deepverb
    validDoList(actor, prep, iobj) =
    {
        /* 
         *   if we have an actor, let it compute the visible list, and add
         *   in the list of floating items 
         */
        if (actor != nil && isclass(actor, movableActor))
            return actor.getVisibleList + global.floatingList;

        /* otherwise, use the default method */
        return inherited.validDoList(actor, prep, iobj);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   make some changes to the base 'thing' class 
 */
modify thing
    /* 
     *   by default, allow unlimited inventory by making everything have
     *   zero bulk 
     */
    bulk = 0
    
    verDoCut(actor) = { "That's not something %you% can cut. "; }
    verDoClimbover(actor) = { "%You% can't climb over <<self.thedesc>>. "; }
    verDoJumpin(actor) = { "That's not something %you% can jump into. "; }
    verDoSwimin(actor) = { "That's not something %you% can swim in. "; }
    doSynonym('Take') = 'Lift'
    verIoShineOn(actor) = { }
    ioShineOn(actor, dobj) =
    {
        "That makes <<self.thedesc>> a little brighter, but you don't
        notice anything new. ";
    }
    verDoShineOn(actor, iobj) =
    {
        "%You% can't shine <<self.thedesc>> on anything. ";
    }
    verIoShineThru(actor) = { }
    ioShineThru(actor, dobj) =
    {
        "You see nothing new. ";
    }

    verDoSprayAt(actor, iobj) = { "It's not clear how to do that. "; }
    verIoSprayAt(actor) = { }
    ioSprayAt(actor, dobj) = { dobj.doSprayAt(actor, self); }
    doSprayAt(actor, iobj) = { "It's not clear how to do that. "; }
    
    verDoPointAt(actor, iobj) = { "It's not polite to point. "; }
    verIoPointAt(actor) = { }
    ioPointAt(actor, dobj) = { dobj.doPointAt(actor, self); }
    doPointAt(actor, iobj) = { "It's not polite to point. "; }

    replace adesc =
    {
        if (self.isThem)
            "some ";
        else
            "a ";
        self.sdesc;
    }
    replace ldesc = "You see nothing unusual about << self.thedesc >>. "
    replace actorAction(v, d, p, i) =
    {
        "You can't address <<self.thedesc>>. ";
        exit;
    }
    doSynonym('PutIn') = 'ThrowIn'
    ioSynonym('PutIn') = 'ThrowIn'
    verDoReadWith(actor, iobj) =
    {
        "\^<<iobj.thedesc>> doesn't help you read <<self.thedesc>>. ";
    }
    verIoReadWith(actor) =
    {
        "\^<<self.thedesc>> doesn't help you read anything. ";
    }
    verDoScanWith(actor, iobj) = { }
    verIoScanWith(actor) =
    {
        "%You% can't use <<self.thedesc>> to scan anything. ";
    }

    verDoFindIn(actor, iobj) = { }
    verDoLookupIn(actor, iobj) = { }
    verDoSearchforIn(actor, iobj) = { }

    verIoSearchFor(actor) = { }
    verDoSearchFor(actor, iobj) =
    {
        "%You% %do%n't need to be so specific when searching. ";
    }
    ioSearchFor(actor, dobj) = { dobj.doSearchFor(actor, self); }
    doSearchFor(actor, iobj) = { iobj.ioSearchforIn(actor, self); }

    verIoFindIn(actor) =
    {
        "%You% %do%n't need to be so specific when searching. ";
    }
    verIoSearchforIn(actor) = { self.verIoFindIn(actor); }
    ioSearchforIn(actor, dobj) =
    {
        "%You% %do%n't need to be so specific when searching. ";
    }
    verIoLookupIn(actor) =
    {
        "You can't look up anything in <<self.thedesc>>. ";
    }
    verDoClimbup(actor) = { "You see no way to do that. "; }
    verDoClimbdown(actor) = { "You see no way to do that. "; }
    verDoPutThru(actor, iobj) = { self.verDoPutIn(actor, iobj); }
    doPutThru(actor, iobj) =
    {
        "%You% can't put that through <<iobj.sdesc>>. ";
    }
    verDoPet(actor) = { "You're anthropomorphizing. "; }

    verDoThrow(actor) = { }
    doThrow(actor) =
    {
        "It would be simpler to just drop it. ";
    }

    verDoInstall(actor) =
    {
        "You'll have to be more specific about where you want
        to put <<self.thedesc>>. ";
    }

    verDoInstallIn(actor, iobj) = { self.verDoPutIn(actor, iobj); }
    verIoInstallIn(actor) =
    {
        "You'll have to be more specific about how you want to
        do that. ";
    }
;

/*
 *   allow carrying lots of stuff by default 
 */
modify item
    bulk = 0
;

/*
 *   make some modifications to surfaces 
 */
modify surface
    /*
     *   Optional contents list - if there is a contents list, describe
     *   it, otherwise say nothing.  This can be used in an overriding
     *   ldesc to append the contents list to the description, but leave
     *   the description alone when nothing is on the surface. 
     */
    optionalListContents =
    {
        switch(itemcnt(self.contents))
        {
        case 0:
            /* say nothing */
            break;

        case 1:
            /* a single object */
            "On <<self.thedesc>> is <<listcont(self)>>. ";
            break;

        case 2:
            /* multiple objects */
            "On <<self.thedesc>> are <<listcont(self)>>. ";
            break;
        }
    }
;

/* 
 *   marker class - inherit (along with room or other appropriate base
 *   classes) to mark a location as not having the default 'theFloor'
 *   object present 
 */
class noFloorRoom: object
;

/*
 *   Don't include the default floor object in certain rooms 
 */
modify theFloor
    location =
    {
        /* if we're in a noFloorRoom location, I'm not here */
        if (isclass(parserGetMe().location, noFloorRoom))
            return nil;
        else
            return inherited.location;
    }
;



/* ------------------------------------------------------------------------ */
#ifdef never
/*
 *   make some formatting changes to the standard room display style 
 */
modify room
    replace dispParagraph = "\b"
    replace dispBeginLdesc = "\n"
    replace dispBeginSdesc = "<b>"
    replace dispEndSdesc = "</b>"
;
#endif


/* ------------------------------------------------------------------------ */
/*
 *   show the prompt 
 */
commandPrompt: function(code)
{
    /* if the score has changed since the last prompt, notify the player */
    if (code = 0 && global.score != global.oldscore && global.scoreNotify)
    {
        local amount := global.score - global.oldscore;
        
        "\b[Your score has increased by <<spellOutNumber(amount)>> point<<
        amount = 1 ? "" : "s">>.]";
        global.oldscore := global.score;
    }
    
    "\b&gt;<font face='tads-input'>";
}

commandAfterRead: function(code)
{
    "</font>";
}

/* ------------------------------------------------------------------------ */
/*
 *   Add some functionality to the basic 'room' class 
 */
modify room
    /*
     *   Use roughly the standard HTML status line code from adv.t, but
     *   make some small modifications to add some controls to the status
     *   line.  
     */
    statusLine =
    {
        /*
         *   Check the system to see if this is an HTML-enabled run-time.
         *   If so, generate an HTML-style banner; otherwise, generate an
         *   old-style banner. 
         */
        if (systemInfo(__SYSINFO_SYSINFO) = true
            and systemInfo(__SYSINFO_HTML) = 1)
        {
            "<banner id=StatusLine height=previous border>
            <body bgcolor=statusbg text=statustext><b>
            <a plain href='look'>";

            self.statusRoot;
            
            "</a></b>
            <tab align=right><i><a plain href='Full Score'>";

            say(global.score); "/";
            say(global.turnsofar);

            "</a></i>
            &nbsp;&nbsp;&nbsp;&nbsp;
            <a plain href='Help' title='Help'><img src='help.jpg'></a>
            </banner>";
        }
        else
        {
            self.statusRoot;
            self.dispParagraph;
        }
    }

    /*
     *   Find an active light source that's present.  Returns a list of
     *   all of the active and present light sources, or an empty list if
     *   no active light sources are present.  
     */
    getLightSources =
    {
        local rem, cur, tot, i;
        local ret;

        /* start off with an empty list */
        ret := [];

        /* scan the set of all light sources in the whole game */
        rem := global.lamplist;
        tot := length(rem);
        i := 1;
        while (i <= tot)
        {
            /* if this item is present and lit, add it to the list */
            cur := rem[i];
            if (cur.isIn(self) and cur.islit)
                ret += cur;

            /* move on to the next item */
            i := i + 1;
        }

        /* return the result list we constructed */
        return ret;
    }

    /*
     *   Determine if a light source is present in the room.  This is
     *   essentially the same test that a dark room performs to determine
     *   if it's lit; this can be used in any room to determine if some
     *   supplemental light source is present.  We simply run through the
     *   global list of light-source items (objects whose 'islamp'
     *   property is true) to see if any are present and providing light
     *   (the 'islit' property is true).  
     */
    isLightPresent =
    {
        local rem, cur, tot, i;

        rem := global.lamplist;
        tot := length(rem);
        i := 1;
        while (i <= tot)
        {
            cur := rem[i];
            if (cur.isIn(self) and cur.islit)
                return true;
            i := i + 1;
        }
        return nil;
    }

    /* 
     *   handle the actor typing 'swim' while in the room -- for most
     *   rooms, this will simply indicate that there's nothing good for
     *   swimming here
     */
    roomSwim(actor) =
    {
        "There's no place to swim here. ";
    }

    /*
     *   Get the list of objects visible to an actor within this room.  
     */
    getVisibleList(actor) =
    {
        /* 
         *   if I'm nested inside another location, use my location's
         *   visibility list 
         */
        if (self.location != nil)
            return self.location.getVisibleList(actor);

        /* 
         *   return the list of objects directly and indirectly visible
         *   from this location 
         */
        return visibleList(self, actor);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Add 'turn off' handling to lightsource, to complement the 'turn on'
 *   handling.  
 */
modify lightsource
    doTurnoff(actor) =
    {
        local waslit := actor.location.islit;

        /* turn it off */
        self.islit := nil;
        self.isActive := nil;

        /* 
         *   report on the change; if the room's lighting status changed,
         *   note that as well 
         */
        "You switch off <<self.thedesc>>";
        if (waslit && !actor.location.islit)
            ", rendering the area pitch black";
        ". ";
    }
    verDoShineOn(actor, iobj) =
    {
        if (!self.islit)
            "It won't have much effect to shine <<self.thedesc>> on
            anything unless it's lit. ";
    }
    doShineOn(actor, iobj) =
    {
        /*
         *   If we get here, it means that the indirect object's ioShineOn
         *   or ioPointAt routine passed the command to us, so it doesn't
         *   do anything special when a light source shines upon it.
         *   Provide a suitable default message. 
         */
        "That makes <<iobj.thedesc>> a little brighter, but you don't
        notice anything new. ";
    }
    verDoShineThru(actor, iobj) = { self.verDoShineOn(actor, iobj); }
    doShineThru(actor, iobj) =
    {
        /*
         *   If we got here, it means that the indirect object is pointing
         *   back at us, so they don't want to do anything special. 
         */
        "You see nothing new. ";
    }

    doSynonym('ShineOn') = 'PointAt'

    doSynonym('Turnon') = 'Light'
    doSynonym('Turnoff') = 'Extinguish'
;


/* ------------------------------------------------------------------------ */
/*
 *   Add a few verbs that aren't part of the adv.t repetiore 
 */

/*
 *   make 'kick' equivalent to 'attack'; also, allow it to be used without
 *   an indirect object 
 */
modify attackVerb
    verb = 'kick'
    doAction = 'Attack'
;

/*
 *   add a 'cut' verb 
 */
cutVerb: deepverb
    sdesc = "cut"
    verb = 'cut' 'chop' 'slice'
    doAction = 'Cut'
;

/* 
 *   add a 'pet' verb 
 */
petVerb: deepverb
    sdesc = "pet"
    verb = 'pet' 'pat'
    doAction = 'Pet'
;

/*
 *   Add 'inflate' and 'deflate' verbs 
 */
inflateVerb: deepverb
    sdesc = "inflate"
    verb = 'inflate' 'blow up' 'pump' 'pump up'
    doAction = 'Inflate'
    ioAction(withPrep) = 'InflateWith'
;

deflateVerb: deepverb
    sdesc = "deflate"
    verb = 'deflate' 'empty'
    doAction = 'Deflate'
;

/*
 *   Add 'tie' and 'untie' 
 */
tieVerb: deepverb
    sdesc = "tie"
    verb = 'tie'
    doAction = 'Tie'
;
untieVerb: deepverb
    sdesc = "untie"
    verb = 'untie'
    doAction = 'Untie'
;

/*
 *   add a 'climb over' verb 
 */
climboverVerb: deepverb
    sdesc = "climb over"
    verb = 'climb over'
    doAction = 'Climbover'
;

/*
 *   add a 'throw' action with no indirect object
 */
modify throwVerb
    doAction = 'Throw'
;

/*
 *   add some synonyms for 'attach' and 'detach' that fit the context of
 *   this game 
 */
modify attachVerb
    verb = 'hook' 'clip'
;
modify detachVerb
    verb = 'unhook' 'unclip'
;

/*
 *   for this game, "fasten" and "unfasten" will be the same as "attach"
 *   and "detach" 
 */
modify fastenVerb
    ioAction(toPrep) = 'AttachTo'
;
modify unfastenVerb
    ioAction(fromPrep) = 'DetachFrom'
;

/* 
 *   add 'climb up' and 'climb down' 
 */
climbupVerb: deepverb
    sdesc = "climb up"
    verb = 'climb up'
    doAction = 'Climbup'
;

climbdownVerb: deepverb
    sdesc = "climb down"
    verb = 'climb down'
    doAction = 'Climbdown'
;

/*
 *   make 'go up' and 'go down' synonymous with 'climb up' and 'climb
 *   down' when used transitively 
 */
modify uVerb
    doAction = 'Climbup'
;
modify dVerb
    doAction = 'Climbdown'
;

/*
 *   add a 'jump into' verb 
 */
jumpinVerb: deepverb
    sdesc = "jump in"
    verb = 'jump in' 'jump into'
    doAction = 'Jumpin'
;

/*
 *   'swim in' verb
 */
swiminVerb: deepverb
    sdesc = "swim in"
    verb = 'swim in'
    doAction = 'Swimin'
;

/*
 *   just plain 'swim' verb 
 */
swimVerb: deepverb
    sdesc = "swim"
    verb = 'swim'
    action(actor) =
    {
        /* let the room handle it */
        actor.location.roomSwim(actor);
    }
;

/*
 *   add a 'lift' verb 
 */
liftVerb: deepverb
    sdesc = "lift"
    verb = 'lift' 'raise'
    doAction = 'Lift'
;

/*
 *   add 'shine dobj on iobj' 
 */
shineVerb: deepverb
    verb = 'shine'
    sdesc = "shine"
    ioAction(onPrep) = 'ShineOn'
    ioAction(atPrep) = 'ShineOn'
    ioAction(thruPrep) = 'ShineThru'
;

/*
 *   add 'point dobj at iobj' 
 */
pointVerb: deepverb
    verb = 'point'
    sdesc = "point"
    ioAction(atPrep) = 'PointAt'
;

/* 'spray dobj at iobj' */
sprayVerb: deepverb
    verb = 'spray' 'squirt'
    sdesc = "spray"
    ioAction(atPrep) = 'SprayAt'
;

/*
 *   add 'light obj' for turning on flashlights and lighting matches and
 *   other flammables 
 */
lightVerb: deepverb
    verb = 'light'
    sdesc = "light"
    doAction = 'Light'
;

/*
 *   add 'extinguish obj' for turning off flashlights and putting out
 *   fires 
 */
extinguishVerb: deepverb
    verb = 'extinguish' 'put out'
    sdesc = "extinguish"
    doAction = 'Extinguish'
;

/*
 *   add 'throw dobj into iobj' to the 'throw' verb 
 */
modify throwVerb
    ioAction(inPrep) = 'ThrowIn'
;

/*
 *   change the way 'hello' works a bit 
 */
modify helloVerb
    action(actor) =
    {
        actor.doHello;
    }
;

/*
 *   let us ask someone about something as long as the someone is a valid
 *   actor 
 */
modify askVerb
    validDo(actor, obj, seqno) =
    {
        /* 
         *   if the direct object is an actor, and the actor is valid
         *   (i.e., its validActor returns true), allow it; otherwise, use
         *   the default handling 
         */
        if (actor = parserGetMe() && isclass(obj, movableActor)
            && obj.validActor)
            return true;
        else
            return inherited.validDo(actor, obj, seqno);
    }
    validDoList(actor, prep, iobj) =
    {
        /* 
         *   always include teeterwaller in the possibly-valid direct
         *   object list, even if he's not present; we'll let validDo sort
         *   things out for us more precisely
         */
        return inherited.validDoList(actor, prep, iobj) + teeterwaller;
    }
;

/*
 *   change the default object list for 'get out'
 */
modify getOutVerb
    replace doDefault(actor, prep, io) =
    {
        if (actor.location and actor.location.location)
            return ([] + actor.location);
        else
            return inherited.doDefault(actor, prep, io);
    }
;

modify outVerb
    doAction = 'Unboard'
;

/*
 *   make 'look dobj' work the same as 'look at dobj'
 */
modify lookVerb
    doAction = 'Inspect'
;

/*
 *   add 'read x with y' to the 'read' verb
 */
modify readVerb
    ioAction(withPrep) = 'ReadWith'
;

/*
 *   add a 'put x through y' verb 
 */
modify putVerb
    ioAction(thruPrep) = 'PutThru'
;

/*
 *   add a 'scan' verb 
 */
scanVerb: deepverb
    verb = 'scan'
    sdesc = "scan"
    prepDefault = withPrep
    ioAction(withPrep) = 'ScanWith'
;

/*
 *   add some verbs for finding things in collections: find x in y, look
 *   up x in y, search for x in y, search y for x 
 */

findVerb: deepverb
    verb = 'find'
    sdesc = "find"
    prepDefault = inPrep
    ioAction(inPrep) = 'FindIn'
;

lookupVerb: deepverb
    verb = 'look up'
    sdesc = "look up"
    prepDefault = inPrep
    ioAction(inPrep) = 'LookupIn'
;

forPrep: Prep
    preposition = 'for'
    sdesc = "for"
;

modify searchVerb
    ioAction(forPrep) = 'SearchFor'
;

searchforVerb: deepverb
    sdesc = "search for"
    verb = 'search for' 'look for'
    prepDefault = inPrep
    ioAction(inPrep) = 'SearchforIn'
;

/*
 *   On the hilltop, Teeterwaller suggests that we should hide ("get
 *   down," he says).  We'll add a verb for it, just for a sensical
 *   response, even though we won't do anything.  
 */
hideVerb: deepverb
    verb = 'hide'
    action(actor) =
    {
        "You don't need to do that. ";
    }
;

getdownVerb: deepverb
    verb = 'get down'
    action(actor) =
    {
        "Get down, get funky, get back up again. ";
    }
;

/*
 *   add a verb for installing things 
 */
installVerb: deepverb
    verb = 'install'
    sdesc = "install"
    doAction = 'Install'
    ioAction(inPrep) = 'InstallIn'
;

/*
 *   add some verbs for setting switches - set, change, slide 
 */
setVerb: deepverb
    verb = 'set'
    sdesc = "set"
    doAction = 'Set'
    ioAction(toPrep) = 'SetTo'
;

changeVerb: deepverb
    verb = 'change'
    sdesc = "change"
    doAction = 'Change'
    ioAction(toPrep) = 'ChangeTo'
;

slideVerb: deepverb
    verb = 'slide'
    sdesc = "slide"
    doAction = 'Slide'
    ioAction(toPrep) = 'SlideTo'
;

modify switchVerb
    ioAction(toPrep) = 'SwitchTo'
;

/*
 *   Fix 'unwear' (take off clothing) so that it only works for the actor
 *   actually wearing the clothing. 
 */
modify clothingItem
    doUnwear(actor) =
    {
        if (self.isworn)
        {
            local cont;
            
            /* 
             *   look up the containment chain for an actor; if we find
             *   one, and the command's actor is not the innermost
             *   containing actor, the command actor isn't the one wearing
             *   the object, hence should not be allowed to remove it 
             */
            for (cont := self.location ; cont != nil ; cont := cont.location)
            {
                /* 
                 *   if this is an actor, check to see if it's the command
                 *   actor 
                 */
                if (isclass(cont, movableActor))
                {
                    if (cont = actor)
                    {
                        /* it's the command's actor - do the normal work */
                        inherited.doUnwear(actor);
                    }
                    else
                    {
                        /* someone else is wearing it - don't allow it */
                        "%You%'ll have to ask <<cont.thedesc>> to do that. ";
                        break;
                    }
                }
            }
        }
        else
            inherited.doUnwear(actor);
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   There are some special locations where we can't stand up, exactly 
 */
modify standVerb
    action(actor) =
    {
        switch(actor.location)
        {
        case inCar:
            "It's too cramped to stand up inside the car. ";
            break;

        default:
            /* for any other locations, inherit the default behavior */
            inherited.action(actor);
        }
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   gratuitous magic words
 */
xyzzyVerb: deepverb
    verb = 'xyzzy'
    action(actor) =
    {
        "Poof!  In a cloud of orange smoke, you're magically
        transported to...\b";
        actor.location.lookAround(true);
        "\b(Okay, it wasn't <i>that</i> magical.)";
    }
;

/* ------------------------------------------------------------------------ */
/* 
 *   yes/no answers 
 */
compoundWord 'say' 'yes' 'say_yes';
yesVerb: deepverb
    verb = 'yes' 'say_yes'
    action(actor) =
    {
        actor.answerYes;
    }
;

compoundWord 'say' 'no' 'say_no';
noVerb: deepverb
    verb = 'no' 'say_no'
    action(actor) =
    {
        actor.answerNo;
    }
;

/*
 *   Add default yes/no processing to actors 
 */
modify movableActor
    answerYes =
    {
        "\^<<self.thedesc>> seems confused by your statement. ";
    }
    answerNo =
    {
        "\^<<self.thedesc>> seems confused by your statement. ";
    }
    verDoTurnon(actor) = { "This isn't that kind of game. "; }
    verDoPet(actor) =
    {
        "\^<<self.thedesc>> seems confused by your gesture. ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Make some small changes to the basic actor class
 */
modify movableActor
    disavow = "<q>I don't know much about that.</q> "
    verGrab(item) =
    {
        "\^<< self.thedesc >> is << item.isworn ? "wearing" : "carrying"
        >> << item.thedesc >> and won't let %youm% have it. ";
    }

    /*
     *   Generate the list of visible objects 
     */
    getVisibleList =
    {
        local ret;
        
        /* get the list of objects directly visible to me */
        ret := visibleList(self, self);

        /* add in the list of objects visible from my location */
        if (self.location != nil)
            ret += self.location.getVisibleList(self);

        /* return the reuslt */
        return ret;
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Increase the score only if the given object's pointsAwarded property
 *   is not true.  This can be used so that points are awarded only once
 *   for a particular action.  Returns true if we awarded points, nil if
 *   the points had already been awarded.  
 */
maybeIncscore: function(amount, desc, obj)
{
    /* check to see if we've awarded these points before already */
    if (obj.pointsAwarded)
        return nil;

    /* award the points */
    incscore(amount, desc);

    /* 
     *   set a flag in the object to indicate that the points have been
     *   awarded, so that we don't try to award them again 
     */
    obj.pointsAwarded := true;

    /* return true to indicate that we awarded the points */
    return true;
}

/* ------------------------------------------------------------------------ */
/*
 *   Spell out a number in words.  Returns a string representing the
 *   number in words.  
 */
spellOutNumber: function(num)
{
    local digits := ['one' 'two' 'three' 'four' 'five'
                     'six' 'seven' 'eight' 'nine'];
    local teens := ['ten' 'eleven' 'twelve' 'thirteen' 'fourteen' 'fifteen'
                    'sixteen' 'seventeen' 'eighteen' 'nineteen'];
    local tens := ['twenty' 'thirty' 'forty' 'fifty' 'sixty'
                   'seventy' 'eighty' 'ninety'];
    local str := '';
    local cur;
    local powers := [
                     [1000000000 'billion']
                     [1000000    'million']
                     [1000       'thousand']
                     [100        'hundred']
                    ];
    local i;
                    
    /* if it's negative, start with "minus" */
    if (num < 0)
    {
        str := 'minus ';
        num := -num;
    }

    /* if it's zero, just say so */
    if (num = 0)
        return 'zero';

    /* 
     *   Go through the power groups (thousands, millions, billions).
     *   Start out at the highest power and work our way down, because
     *   that's how numbers are spelled out. 
     */
    for (i := 1 ; i <= length(powers) ; ++i)
    {
        /* check to see if we have any of this power group */
        if (num >= powers[i][1])
        {
            /* figure out how many of this power group */
            cur := num / powers[i][1];

            /* 
             *   spell out the number of this power and add the name of
             *   the power group ('one hundred fifty million', for
             *   example) 
             */
            str += spellOutNumber(cur) + ' ' + powers[i][2];

            /* take out this power group amount */
            num -= cur * powers[i][1];

            /* if there will be more to add after this, add a space */
            if (num != 0)
                str += ' ';
        }
    }

    /*
     *   We're down under a hundred now.  Check to see if we're in the
     *   regular tens groups from 20 to 99 (the teens are irregular).  
     */
    if (num >= 20)
    {
        /* 
         *   we're in the regular tens groups - add the tens group -
         *   divide by ten to get the number of tens, and subtract one
         *   because tens[1] is 'twenty' 
         */
        str += tens[(num / 10) - 1];

        /* remove the tens from the number */
        num -= (num / 10) * 10;

        /* if there's more, add a dash ("twenty-five") */
        if (num != 0)
            str += '-';
    }

    /*
     *   Check to see if we're in the teens (anything from 10 to 19 is a
     *   teen, as far as we're concerned).  
     */
    if (num >= 10)
    {
        /* 
         *   We're in the teens - add the appropriate teen name.  Since
         *   teens[1] is 'ten', subtract 9 from the number to get the name
         *   index. 
         */
        str += teens[num - 9];

        /* this fully names the number, so we need nothing more */
        num := 0;
    }

    /*
     *   Check to see if we have any units remaining, and add the unit
     *   name if so. 
     */
    if (num != 0)
        str += digits[num];

    /* return the result */
    return str;
}

/* ------------------------------------------------------------------------ */
/*
 *   define our own incscore - this version notifies the player when the
 *   score changes.  The 'desc' is a string giving the reason for the
 *   score change 
 */
replace incscore: function(amount, desc)
{
    local i;
    local tot;
    local found;
    
    /* adjust the score */
    global.score := global.score + amount;

    /*
     *   Search the list of score detail items to see if we can find an
     *   existing entry with the exact same description; if so, do not
     *   create a new item, but simply combine the points from this new
     *   item into the existing item -- this way, if the game wants to
     *   score a few points here and there for something like "finding
     *   sundry items," it'll just show up once with the combined number
     *   of points. 
     */
    for (found := nil, i := 1, tot := length(global.fullScoreList) ;
         i <= tot ; ++i)
    {
        /* if this item matches, combine the points */
        if (global.fullScoreList[i][2] = desc)
        {
            /* add the new points into the existing item */
            global.fullScoreList[i][1] += amount;
            
            /* note that we've found it */
            found := true;

            /* no need to look any further */
            break;
        }
    }
    
    /* 
     *   If we didn't find an existing item with the same description, add
     *   the detailed description to the full score list.  Each entry in
     *   the full score list is a two-element list: the first element is
     *   the number of points awarded by this action, and the second
     *   element is a string describing the action.
     *   
     *   Note the funny notation with the double brackets: If we add a
     *   list to a list, we'd normally just add the *elements* of the
     *   second list to the first; in this case, we want to add a new
     *   sublist to the list, so we need to add a list containing a list,
     *   which will add a single element - the sublist - to the original
     *   list.  
     */
    if (!found)
        global.fullScoreList += [[amount desc]];

    /* update the status line */
    scoreStatus(global.score, global.turnsofar);
}

/*
 *   turn score notification on and off 
 */
notifyVerb: sysverb
    verb = 'notify'
    action(actor) =
    {
        global.scoreNotify := not global.scoreNotify;
        if (global.scoreNotify)
            "You will now be notified of score changes.";
        else
            "You will no longer be notified of score changes.
            If you change your mind and want to be notified
            again, just type NOTIFY again.";
        abort;
    }
;

/*
 *   make 'score' tell the player about 'fullscore' the first time they
 *   use it 
 */
modify scoreVerb
    usedOnce = nil
    replace action(actor) =
    {
        self.showScore(actor);
        if (!self.usedOnce)
        {
            "\b[Use the FULL SCORE command to display a detailed list of
            the points you've earned.]\n";
            self.usedOnce := true;
        }
        abort;
    }
;

fullScoreStatus: function
{
    scoreRank_internal(true);
    if (global.score != 0)
    {
        local i;
        
        "You scored:\n";
        for (i := 1 ; i <= length(global.fullScoreList) ; ++i)
        {
            local cur;
            
            cur := global.fullScoreList[i];
            "\t<< cur[1] >> point<< cur[1] = 1 ? "" : "s"
                                             >> for << cur[2] >>";
            if (i != length(global.fullScoreList))
                ";";
            else
                ".";
            "\n";
        }
    }
}

/*
 *   show full score information
 */
compoundWord 'full' 'score' 'fullscore';
fullscoreVerb: sysverb
    verb = 'fullscore' 'full'
    action(actor) =
    {
        fullScoreStatus();
        abort;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Some additional system verbs 
 */

/*
 *   show game credits 
 */
showCredits: function
{
    "\b
    <center>This game was designed, written, and implemented by
    Michael J.\ Roberts, using the Text Adventure Development System.

    \bOriginally released for the 4th Annual Interactive Fiction
    Competition, October 1998.
    
    \bThe author would like to express his sincere appreciation to
    Steve McAdams, Dennis Buchheim, and Sean Trabosh for their patience
    and thoroughness testing the early versions of the game and for
    their many ideas for improving it.  Thanks also to
    everyone who sent comments after playing the game during
    the competition, and to the contest judges who took the time to
    write reviews of the game; your comments and reviews are deeply
    appreciated.  Finally, I'd like to thank David Dyte and the rest
    of this year's competition organizers for making the
    event enjoyable and memorable.
    
    \bThis game is a work of fiction.  Any resemblance to actual
    persons, organizations, places, or events is purely coincidental.
    All scenes involving animals were produced under the supervision
    of the Interactive Fiction Society for the Prevention of Cruelty
    to Animals.
    
    \bPresented In
    \n<font face='TADS-Sans'><b>S U P E R\ \ \ T A D S - O - R A M A</b>
    </font>
    \nOn Selected Computers
    
    \b<img src='htmltads.jpg'>

    \bIf you have any comments or suggestions, or would like to
    report a bug, the author would be pleased to hear from you.
    Please send email to <<global.mjr_email>> (yes, that <i>is</i>
    an underscore in the user name).
    </center> ";
}

/*
 *   CREDITS verb - show the game credits 
 */
creditsVerb: sysverb
    verb = 'credits'
    action(actor) =
    {
        showCredits();
        abort;
    }
;

/*
 *   show copyright information 
 */
copyrightVerb: sysverb
    verb = 'copyright'
    action(actor) =
    {
        version.namedesc;
        "\nCopyright &copy;1998 by Michael J.\ Roberts.
        \bThis game is protected by copyright.  The author hereby grants
        permission to everyone to make copies of this game, provided that
        all copies are complete and unmodified and retain the original
        copyright notices.  Please contact the author at
        <<global.mjr_email>> if you have any questions. ";

        abort;
    }
;

/*
 *   Help
 */
helpVerb: sysverb
    verb = 'help'
    action(actor) =
    {
        "Welcome to <i><<version.namedesc>></i>!
        \bBelow is a brief list of some of the special commands commands
        that this game recognizes.  If you're new to interactive fiction,
        please type <a href='instructions'>INSTRUCTIONS</a> for a detailed
        introduction to text adventures.
        \b
        \n\t<a href='save'>SAVE</a> - Save the current game position
        \n\t<a href='restore'>RESTORE</a> - Restore a saved game position
        \n\t<a href='undo'>UNDO</a> - Take back the previous turn
        \n\t<a href='quit'>QUIT</a> - End the game
        \b
        \n\t<a href='score'>SCORE</a> - Show your progress in the game
        \n\t<a href='full score'>FULL SCORE</a> - Show a full account of
           what you've done so far
        \n\t<a href='credits'>CREDITS</a> - Show the game's credits
        \n\t<a href='copyright'>COPYRIGHT</a> - Copyright information
        \n\t<a href='about'>ABOUT</a> - Show some background information
           about the game
        \n\t<a href='version'>VERSION</a> - Show the current version of the
           game
        \n\t<a href='special'>SPECIAL</a> - Show the <q>Special Features</q>
           menu
        \b
        \n\t<a href='script'>SCRIPT</a> - Start recording a transcript of
           your moves to a file
        \n\t<a href='unscript'>UNSCRIPT</a> - Stop recording the
           current transcript
        \b";

        if (systemInfo(__SYSINFO_SYSINFO) = true
            and systemInfo(__SYSINFO_HTML) = 1)
            "Since you're playing this game with HTML TADS, you can use a
            couple of additional shortcuts.  You can get a description of
            the current location by clicking on the location name in the
            status line, and you can get a full accounting of your score
            by clicking on the score in the status line.\b";

        abort;
    }
;

/*
 *   About - show some background information about the game 
 */
aboutVerb: sysverb
    verb = 'about'
    action(actor) =
    {
        "<b>About <i>The Plant</i></b>
        \b\b";
        "One of the main reasons I developed TADS in the first place is
        that I very much enjoy writing adventure games.  So, once the
        HTML TADS project started approaching some semblance of completion,
        I started to work on a new game.  After several false starts, the
        present game started to take shape. ";

        "\bThis game follows a design principle that I used in
        <i>Perdition's Flames</i>: the player character should never
        die, and the game should never become unsolvable.  I know that this
        can sometimes strain the illusion of reality, but I think the
        illusion is destroyed utterly if you constantly have to use
        <q>save</q> and <q>restore</q> to repair past mistakes.  More
        importantly, though, I have a lot more fun playing a game when
        I don't need to worry that any misstep could be punished with the
        loss of hours of hard work; if I want that I can just fire
        up <i>Word</i>. ";
        
        "\bOne might wonder why, after going to so much trouble to build a
        new adventure game run-time with graphics and sound capabilities,
        I didn't overload this game with multimedia special effects.  The
        simple answer is that I wanted to write a new text game, and I'm
        not much of a graphic artist.
        I put all of the whizzy media features into HTML TADS because I knew
        that it wouldn't be long before someone would want them, even if
        I'm not going to need them immediately.  I have, however, used
        numerous tiny HTML features here and there, and I hope they
        enhance the game. ";
        
        "\b<i>The Plant</i> was originally written as an entry in the
        Fourth Annual Interactive Fiction Competition held in October,
        1998.  This revised <q>Special Edition</q> corrects a number of bugs
        in the original competition edition, and has several minor changes
        to the story that, I think, clarify a few matters and correct
        some inconsistencies.  In addition, I added the
        <q><a href='special'>Special Features</a></q> section,
        which provides some background on the
        design and implementation of the game; I hope that others with
        an interest in adventure game design will find it useful. ";
        
        "\b<i>
            \t\t&mdash;<tab id=afterdash>Mike Roberts
            \n\t\t<tab to=afterdash>Palo Alto, California
            \n\t\t<tab to=afterdash>December, 1998
        </i>
        \b";

        abort;
    }
;

#include <mru_dis.t>

modify pushVerb
    disambigGenObj(actor, objlist) =
    {
        local ret;
        local i;
        
        /* 
         *   'push' is usually used for buttons in this game; try to find
         *   a button that the player isn't carrying around with them,
         *   since the portable buttons tend to be useful only where there
         *   aren't other buttons, and hence should be selected against
         *   when a non-portable button is present.  
         */
        ret := [];
        for (i := 1 ; i <= length(objlist) ; ++i)
        {
            if (!objlist[i].isIn(actor))
                ret += objlist[i];
        }

        /* if we found one object, use it */
        if (length(ret) = 1)
        {
            "(<<ret[1].thedesc>>)\n";
            return ret;
        }

        /* no luck - use the default form */
        return inherited.disambigGenObj(objlist);
    }
;

/*
 *   For wearing things, we'd certainly prefer to wear something we're
 *   carrying over something that we're not carrying. 
 */
modify wearVerb
    disambigGenObj(actor, objlist) =
    {
        local ret;
        local i, len;

        /* make a list of the objects that we're carrying */
        for (i := 1, len := length(objlist), ret := [] ; i <= len ; ++i)
        {
            /* 
             *   if the actor is carrying this and not already wearing it,
             *   add it to our list 
             */
            if (objlist[i].isIn(actor) && !objlist[i].isworn)
                ret += objlist[i];
        }

        /* if the list has just one element, use it */
        if (length(ret) = 1)
        {
            "(<<ret[1].thedesc>>)\n";
            return ret;
        }

        /* we couldn't add any value here - use the default handling */
        return inherited.disambigGenObj(objlist);
    }
;

/* 
 *   allow 'insert x into y' as a synonym for 'put x in y' 
 */
insertVerb: deepverb
    verb = 'insert'
    sdesc = "insert"
    ioAction(inPrep) = 'PutIn'
    prepDefault = inPrep
;

/*
 *   verbs 'free' and 'release' 
 */
freeVerb: deepverb
    verb = 'free'
    sdesc = "free"
    doAction = 'Free'
;

releaseVerb: deepverb
    verb = 'release'
    sdesc = "release"
    doAction = 'Release'
;

