#include <adv3.h>


modify Thing
    /* 
     *   If this property is true, default status reports will be included in 
     *   descriptions. By default it is true, for backwards compatibility.
     */
    statusReportable = true
    
    /* Only show default status reports if allowed. */
    examineStatus() {
        if (statusReportable)
            inherited();
    }
    
    /*
     *   If you want specialized lister behavior when reporting the 
     *   contents, you can call this reporter with a specific lister as an 
     *   argument.
     */
    reportContentsWith(lister) {
        examineListContentsWith(lister);
    }
    
    /*
     *   The TADS library reports contents with a special description 
     *   separately. So let's also have a separate opt-in "reporting 
     *   method" for including special contents in descriptions.
     */
    reportSpecialContents {
        examineSpecialContents();
    }
    
    /* 
     *   If you want to report all contents (regular and special) let's make 
     *   that easy too.
     */
    reportAllContents {
        reportContents;
        reportSpecialContents;
    }
    
    /* 
     *   Only show special descriptions for contents if statusReportable and 
     *   specialContentsListedInExamine are both true.
     */
    examineSpecialContents() {
        if (statusReportable && specialContentsListedInExamine)
            inherited;
    }
    
    /* 
     *   specialContentsListedInExamine works analogously to 
     *   contentsListedInExamine, except that if it is nil it is the listing 
     *   of this object's special contents (i.e. any contained object that 
     *   uses a specialDesc or initSpecialDesc, including actors) that will 
     *   be suppressed.
     *
     *   By default it is true, for backwards compatibility.
     */
    specialContentsListedInExamine = true
    
    /* 
     *   Handling for room descriptions. Theoretically we could use 
     *   statusReportable for this too, but that could cause nasty bugs with 
     *   nested rooms. So we'll add a new property to remove default room 
     *   description status reports: roomStatusReportable. By default it is 
     *   true, for backwards compatibility.
     */
    roomStatusReportable = true
    lookAroundWithin(actor, pov, verbose) {
        local illum;
        local infoTab;
        local info;
        local specialList;
        local specialBefore, specialAfter;

        /* check for the special 'true' and 'nil' settings for 'verbose' */
        if (verbose == true) {
            /* true -> show the standard verbose description */
            verbose = (LookRoomName | LookRoomDesc);
        } else if (verbose == nil) {
            /* nil -> show the standard terse description */
            verbose = (LookRoomName);
        }
        
        /* check for roomStatusReportable */
        if (roomStatusReportable) {
            verbose |= (LookListSpecials | LookListPortables);
        }

        /* 
         *   if we've never seen the room before, include the room
         *   description, even if it wasn't specifically requested 
         */
        if (!actor.hasSeen(self))
            verbose |= LookRoomDesc;

        /* 
         *   get a table of all of the objects that the viewer can sense
         *   in the location using sight-like senses 
         */
        infoTab = actor.visibleInfoTableFromPov(pov);

        /* get the ambient illumination at the point of view */
        info = infoTab[pov];
        if (info != nil) {
            /* get the ambient illumination from the info list item */
            illum = info.ambient;
        } else {
            /* the actor's not in the list, so it must be completely dark */
            illum = 0;
        }

        /* 
         *   adjust the list of described items (usually, this removes the
         *   point-of-view object from the list, to ensure that we don't
         *   list it among the room's contents) 
         */
        adjustLookAroundTable(infoTab, pov, actor);
        
        /* if desired, display the room name */
        if ((verbose & LookRoomName) != 0) {
            "<.roomname>";
            lookAroundWithinName(actor, illum);
            "<./roomname>";
        }

        /* if we're in verbose mode, display the full description */
        if ((verbose & LookRoomDesc) != 0) {
            /* display the full room description */
            "<.roomdesc>";
            lookAroundWithinDesc(actor, illum);
            "<./roomdesc>";
        }

        /* show the initial special descriptions, if desired */
        if ((verbose & LookListSpecials) != 0) {
            local plst;

            /* 
             *   Display any special description messages for visible
             *   objects, other than those carried by the actor.  These
             *   messages are part of the verbose description rather than
             *   the portable item listing, because these messages are
             *   meant to look like part of the room's full description
             *   and thus should not be included in a non-verbose listing.
             *   
             *   Note that we only want to show objects that are currently
             *   using special descriptions AND which aren't contained in
             *   the actor doing the looking, so subset the list
             *   accordingly.  
             */
            specialList = specialDescList(
                infoTab,
                {obj: obj.useSpecialDescInRoom(self) && !obj.isIn(actor)});

            /* 
             *   partition the special list into the parts to be shown
             *   before and after the portable contents list 
             */
            plst = partitionList(specialList,
                                 {obj: obj.specialDescBeforeContents});
            specialBefore = plst[1];
            specialAfter = plst[2];

            /* 
             *   at this point, describe only the items that are to be
             *   listed specially before the room's portable contents list 
             */
            specialContentsLister.showList(pov, nil, specialBefore,
                                           0, 0, infoTab, nil);
        }

        /* show the portable items, if desired */
        if ((verbose & LookListPortables) != 0) {
            /* 
             *   Describe each visible object directly contained in the
             *   room that wants to be listed - generally, we list any
             *   mobile objects that don't have special descriptions.  We
             *   list items in all room descriptions, verbose or not,
             *   because the portable item list is more of a status display
             *   than it is a part of the full description.
             *   
             *   Note that the infoTab has sense data that will ensure that
             *   we don't show anything we shouldn't if the room is dark.  
             */
            lookAroundWithinContents(actor, illum, infoTab);
        }

        /*
         *   If we're showing special descriptions, show the special descs
         *   for any items that want to be listed after the room's portable
         *   contents.  
         */
        if ((verbose & LookListSpecials) != 0) {
            /* show the subset that's listed after the room contents */
            specialContentsLister.showList(pov, nil, specialAfter,
                                           0, 0, infoTab, nil);
        }
        
        /* check for roomStatusReportable again */
        if (roomStatusReportable) {
            /* show all of the sounds we can hear */
            lookAroundWithinSense(actor, pov, sound, roomListenLister);

            /* show all of the odors we can smell */
            lookAroundWithinSense(actor, pov, smell, roomSmellLister);
        }

        /* show the exits, if desired */
        lookAroundWithinShowExits(actor, illum);

        /* 
         *   If we're not in the dark, note that we've been seen.  Don't
         *   count this as having seen the location if we're in the dark,
         *   since we won't normally describe any detail in the dark.  
         */
        if (illum > 1)
            actor.setHasSeen(self);
    }
;

/* 
 *   Change the examineStatus method on all other classes so that it checks 
 *   whether it's allowed to show default status reports.
 */
modify ComplexContainer
    examineStatus() {
        if (statusReportable)
            inherited();
    }
;

modify Container
    examineStatus() {
        if (statusReportable)
            inherited();
    }
;

modify Attachable
    examineStatus() {
        if (statusReportable)
            inherited();
    }
    
    /* A handy method to list items attached to this object. */
    reportAttachments {
        local tab;
        
        /* get the actor's visual sense table */
        tab = gActor.visibleInfoTable();
        
        /* add our list of attachments */
        attachmentLister.showList(gActor, self, attachedObjects,
                                  0, 0, tab, nil);
        
        /* add our list of major attachments */
        majorAttachmentLister.showList(gActor, self, attachedObjects,
                                       0, 0, tab, nil);
    }
;

modify RoomPart
    examineStatus() {
        if (statusReportable)
            inherited();
    }
    
    /* Handy method for reporting our contents. */
    reportContents {        
        examinePartContents(&descContentsLister);
    }
;

modify ContainerDoor
    examineStatus() {
        if (statusReportable)
            inherited();
    }
;

modify Openable
    /* 
     *   openStatusReportable works on analogy with lockStatusReportable on 
     *   Lockable. It allows us to suppress the 'it's open' and 'it's 
     *   closed' messages that follow the description of an Openable object 
     *   (by setting this property to nil). By default this property is 
     *   true, giving the standard library behaviour.
     *
     *   It may also be useful to set this property to the value of an 
     *   expression, e.g.:
     *
     *   openStatusReportable = (isOpen)
     *
     *   would result in the 'it's open' message being appended when we're 
     *   open, but nothing being appended when we're closed.
     */
    openStatusReportable = (canInherit() ? inherited() : true)
    
    /* "It's open. " */
    reportOpenStatus {
        say ('\^' + openStatus + '. ');
    }
    
    /* "It's currently open. " */
    reportCurrentlyOpenStatus {
        say(isOpen ? gLibMessages.currentlyOpen :
            gLibMessages.currentlyClosed);
    }
    
    /* "The object is open. " */
    reportObjOpenStatus {
        gMessageParams(self);
        "{The self/he} is <<openDesc>>. ";
    }
;

modify openableContentsLister
    showListEmpty(pov, parent) {
        if(parent.openStatusReportable)
           "\^<<parent.openStatus>>. ";
    }
    
    showListPrefixWide(itemCount, pov, parent) {
        if(parent.openStatusReportable)
           "\^<<parent.openStatus>>, and";
        else
           "\^<<parent.theName>>";
        " contain<<parent.verbEndingSEd>> ";
    }
;

modify Lockable
     /*   
      *  Default reporting of lock status respects the lockStatusObvious and
      *  lockStatusReportable flags.
      */
    reportLockedStatus {
        if (lockStatusObvious && lockStatusReportable)
            say ( '\^' + itIsContraction + ' ' + lockedDesc + '. ');
    }
    
    reportCurrentlyLockedStatus {
        if (lockStatusObvious && lockStatusReportable)
            say(isLocked ? gLibMessages.currentlyLocked
                :
                gLibMessages.currentlyUnlocked);
    }
        
    reportDobjLockedStatus {
        if (lockStatusObvious && lockStatusReportable)
            "{The dobj/he} is <<lockedDesc>>. ";
    }
;

/* 
 *   The Actor modifications are a little more complex than other classes. 
 *   Rather than define statusReportable directly, we'll grab it from the 
 *   current state. Also, we need to show the stateDesc regardless of 
 *   whether statusReportable returns true.
 *
 *   We'll also add a postureReportable property to control whether the 
 *   Actor's posture should be reported in default status reports.
 */
modify Actor
    replace examineStatus() {
        /* 
         *   If I'm an NPC, show where I'm sitting/standing/etc.  (If I'm 
         *   the PC, we don't usually want to show this explicitly to avoid 
         *   redundancy.  The player is usually sufficiently aware of the 
         *   PC's posture by virtue of being in control of the actor, and 
         *   the information also tends to show up often enough in other 
         *   places, such as on the status line and in the room 
         *   description.)  
         *
         *   Also, we only show this if the postureReportable property is 
         *   true (by default, this checks with the current state). This 
         *   allows redundant posture descriptions to be suppressed if the 
         *   current ActorState's stateDesc already mentions them.
         */
        if (!isPlayerChar() && statusReportable && postureReportable)
            postureDesc;
        
        /* show the status from our state object */
        curState.stateDesc;
               
        /* inherit the default handling to show our contents */
        if (statusReportable)
           inherited();
    }
    
    
    /* 
     *   By default, we decide whether to give default status reports by 
     *   checking with the Actor's current state. This means that we can 
     *   override this on a case-by-case (state-by-state?) basis or once for 
     *   all on the Actor.
     */
    statusReportable = (curState.statusReportable)
    postureReportable = (curState.postureReportable)
    
    /*
     *   Reporters for actors
     *
     *   The default examineStatus provides a posture description, and a 
     *   state description. Let's provide some default reporters.
     */
    reportPosture {
        postureDesc;
    }
    
    reportState {
        curState.stateDesc;
    }
    
    /* 
     *   Combine all the actor status pieces into one default report, if 
     *   desired.
     */
    reportActorStatus {
        reportPosture;
        reportState;
        reportContents;
    }
;

modify ActorState
    /* By default, ActorStates allow the Actor to give status reports. */
    statusReportable = true
    
    /* 
     *   By default the actor's postureDesc() is displayed between its desc 
     *   and the stateDesc on the current ActorState. If the stateDesc 
     *   already mentions the actor's posture this may be redundant; changing
     *   postureReportable to nil suppresses the display of the 
     *   postureDesc().
     */
    postureReportable = true
;
//------------------------------------------------------------------------------

/* 
 *   The following modification to ThingState is designed to make it easier to
 *   customize state-dependent messages such as '(providing light)' or 
 *   '(being worn)'. The mechanism should be readily extensible to 
 *   author-defined ThingStates. 
 */

modify LightSource
    /* 
     *   The message that's appended to our name in inventory listings and the
     *   like when we're lit; the default in the English language library is 
     *   '(providing light)'. To suppress the message altogether, set this 
     *   property to nil. To replace it with another message, omit the 
     *   parentheses, e.g. on a flashlight you might define:
     *
     *   providingLightMsg = 'currently switched on'
     */
    providingLightMsg = lightSourceStateOn.listName_
    notProvidingLightMsg = lightSourceStateOff.listName_
;

modify Matchstick
    /* 
     *   The message that's appended to our name in inventory listings and the
     *   like when we're lit; the default in the English language library is 
     *   '(lit)'. To suppress the message altogether, set this 
     *   property to nil. To replace it with another message, omit the 
     *   parentheses, e.g. :
     *
     *   providingLightMsg = 'now ablaze'
     */
    providingLightMsg = matchStateLit.listName_
    notProvidingLightMsg = matchStateUnlit.listName_
;

modify Wearable
    /*
     *   The messages that's appended to our name in listing when we're worn 
     *   or unworn. By default this will be '(worn)' or nothing; but the 
     *   wornMsg won't be used if we're appearing in a list of items 
     *   introduced as being worn (e.g. "You are wearing...").
     */
    wornMsg = wornState.listName_
    unwornMsg = unwornState.listName_
;


modify ThingState
    listName(lst)
    {
        /* 
         *   if we have a msgProp and there's one item in our list, use the 
         *   msgProp of that item to provide our state-dependent text.
         */
        if(msgProp != nil && lst.length() == 1)
            return (lst[1]).(msgProp);
        
        /*  Otherwise, use our listName_ (as in the standard library) */
        return listName_;
    }
    
    /* 
     *   The property on the object being described that will provide the 
     *   text for any state-dependent extra description, such as 'providing 
     *   light'. If this is defined, it should be set to a property pointer, 
     *   e.g. &providingLightMsg. If it is left at nil ThingState will behave
     *   just as it does in the standard library.
     */
    msgProp = nil
;

modify lightSourceStateOn    
    msgProp = &providingLightMsg
;

modify matchStateLit
    msgProp = &providingLightMsg
;

modify wornState
    msgProp = &wornMsg
;

/* 
 *   Although the following three ThingStates don't display any state-related 
 *   text in the standard library, equivalent modifications are added here 
 *   to make it easy to add object-specific state information (e.g. 
 *   'currently switched off' on a flashlight) if so desired). This does not 
 *   affect the standard library behaviour unless a game author chooses to 
 *   override the notProvidingLightMsg_ property anywhere. These two 
 *   ThingStates can also be given a listName_ property if a globally 
 *   applicable message is required (e.g. 'unlit' for every unlit matchstick)
 */

modify lightSourceStateOff
    msgProp = &notProvidingLightMsg
;

modify matchStateUnlit
    msgProp = &notProvidingLightMsg
;

modify unwornState
    msgProp = &unwornMsg
;


/* 
 *   Note that the same coding pattern can be used if desired on custom 
 *   ThingStates. E.g.
 *
 *.   class MyCustomClass: Thing
 *.      allStates = [sillyState, sensibleState]
 *.      getState = sillyState
 *.      sillyMsg = sillyState.listName_
 *.      sensibleMsg = sensibleState.listName_
 *.   ;
 *.
 *.   sillyState: ThingState 'silly'
 *.     stateTokens = ['silly']
 *.     msgProp = &sillyMsg
 *.   ;
 *.
 *.   sensibleState: ThingState 'sensible'
 *.     stateTokens = ['sensible']
 *.     msgProp = &sensibleMsg
 *.   ;
 *
 *   Note, however, that there is no need to define the msgProps if the 
 *   ThingStates are being created for a single object, or if all the 
 *   objects they will apply to are to use the same state-specific messages; 
 *   in these cases ThingStates may be defined and used just as they are in 
 *   the standard library. For example, if we never wanted to vary the 
 *   'silly' and 'sensible' messages on different objects, there would be no 
 *   need (and not point) to define sillyMsg and sensibleMsg on 
 *   MyCustomClass, and no need to define msgProp on sillySrate and 
 *   sensibleState.
 *
 */


//------------------------------------------------------------------------------
