#include <adv.t>
#include <std.t>


/* This program is Copyright (c) 1994 David Allen.  It may be freely
   distributed as long as you leave my name and copyright notice on it.
   I'd really like your comments and feedback; send e-mail to
   allen@viewlogic.com, or send us-mail to David Allen, 10 O'Moore Ave,
   Maynard, MA 01754. */


/* This file contains a really simple Eliza implementation and an even
   simpler sample adventure.  It contains only the one room, and no
   objects.  It is intended to show the string substitution and reply
   mechanism of Eliza type programs.

   To give credit where credit is due, I started from a five page BASIC
   implementation, apparently in Apple ][+ "Applesoft", from Wes Modes
   (modes@earwig.mport.com).  This may have come from a 1979 Creative
   Computing article, and the program contained comments crediting Patricia
   Danielson and Paul Hashfield.

   In this version, I have not attempted to make the verbs work well.
   You enter the command SAY "SOME TEXT".  The "say" verb passes the
   string argument to a function which produces an Eliza style response.
   A much better implementation, which I have not yet figured out how
   to do, would require an actor to be in the room, so you would enter
   SAY "SOME TEXT" TO BILL.  Or, you could enter SAY "SOME TEXT" and
   the program would figure out that you meant Bill.  This is the way
   "ask about" presently functions, but I haven't maneuvered my way
   through the do and io routines which would be needed.  */
   

startroom: room
   sdesc = "The Room"
   ldesc = "You're standing in the only room in this game.  Say things.";


/* Takes three strings.  If the second string is found inside the first,
   substitute the third string for the second and return the new string.
   If not, return the old string. */
elizaFixone: function (target, source, dest) {
   local n;
   if ((n := find (target, source)) = nil) return (target);
   return (substr (target, 1, n-1) + dest +
      substr (target, n + length (source), length (target))); }


/* Takes a string s and a list p.  P contains an even number of elements;
   it contains pairs where the first is either nil or a string, and the 
   second is a string.  The function randomly picks one of the pairs.  If 
   the first of the pair is nil, the corresponding string is simply printed.
   If the first is not nil, the input string is "fixed" with strings like
   "you" replaced by "me", then the input string is printed after the 
   corresponding string, and finally the first of the pair is printed as
   the punctuation to end the sentence. */
elizaSub: function (p, s) {
   local i, len := length (p) / 2;
   i := rand (len) * 2 - 1;
   "\""; say (p[i+1]); 
   if (p[i] <> nil) {
      s := ' ' + s + ' ';
      s := elizaFixone (s, ' yourself ',' myself ');
      s := elizaFixone (s, ' myself ',  ' yourself ');
      s := elizaFixone (s, ' mine ',    ' yours ');
      s := elizaFixone (s, ' yours ',   ' mine ');
      s := elizaFixone (s, ' my ',      ' your ');
      s := elizaFixone (s, ' your ',    ' my ');
      s := elizaFixone (s, ' are ',     ' am ');
      s := elizaFixone (s, ' am ',      ' are ');
      s := elizaFixone (s, ' was ',     ' were ');
      s := elizaFixone (s, ' were ',    ' was ');
      s := elizaFixone (s, ' i ',       ' you ');
      s := elizaFixone (s, ' me ',      ' you ');
      s := elizaFixone (s, ' you ',     ' me ');
      s := elizaFixone (s, ' i\'ve ',   ' you\'ve ');
      s := elizaFixone (s, ' you\'ve ', ' I\'ve ');
      s := elizaFixone (s, ' i\'m ',    ' you\'re ');
      s := elizaFixone (s, ' you\'re ', ' I\'m ');
      /* Print result with trailing punctuation */
      say (s); say (p[i]); }
   "\""; }


replace strObj: basicStrObj
   /* This simple implementation assumes that any string spoken is to be
      handled by the Eliza routine.  It loops over all "elizaKey" objects.
      Each one has a list of matching keywords.  This function takes the
      first elizaKey containing a keyword found in the sentence, and hands
      it to elizasub for further work.  If no keywords are found, the
      special elizaNokey object is handed to elizaSub. */
   doSay (actor) = {
      local e, i, n, s := self.value;
      for (e := firstobj (elizaKey); e <> nil; e := nextobj (e, elizaKey))
         for (i := 1; i <= length (e.keys); i++)
            if ((n := find (s, e.keys[i])) <> nil) {
               n += length (e.keys[i]);
               return (elizaSub (e.pairs, substr (s, n, length (s)))); }
      return (elizaSub (elizaNokey.pairs, s)); };


/* I first tried defining a single list containing all 100+ strings.
   However, I had to boost up the parse node size in the compiler (-mp)
   to about 120000 and the stack size in the runtime (-mh) to 60000
   to get it to compile and run.  This approach requires no special
   compiler or runtime switches. */

class elizaKey: object;

t001: elizaKey keys = ['can you'] pairs =
      ['?',  'Don\'t you believe that I can',
       nil,  'Perhaps you would like to be like me',
       '?',  'You want me to be able to'];
t002: elizaKey keys = ['can i'] pairs =
      ['.',  'Perhaps you don\'t want to',
       '?',  'Do you want to be able to'];
t003: elizaKey keys = ['you\'re' 'you are'] pairs =
      ['?',  'What makes you think I am',
       '?',  'Does it please you to believe I am',
       '.',  'Perhaps you would like to be',
       '?',  'Do you sometimes wish you were'];
t004: elizaKey keys = ['i don\'t'] pairs =
      ['?',  'Don\'t you really',
       '?',  'Why don\'t you',
       '?',  'Do you wish to be able to',
       '?',  'Does that trouble you'];
t005: elizaKey keys = ['i feel'] pairs =
      ['?',  'Do you often feel',
       '?',  'Do you often feel',
       '?',  'Do you enjoy feeling'];
t006: elizaKey keys = ['why don\'t you'] pairs =
      ['?',  'Do you really believe I don\'t',
       '.',  'Perhaps in good time I will',
       '?',  'Do you want me to'];
t007: elizaKey keys = ['why can\'t i'] pairs =
      ['?',  'Do you think you should be able to',
       '?',  'Why can\'t you'];
t008: elizaKey keys = ['are you'] pairs =
      ['?',  'Why are you interested in whether or not I am',
       '?',  'Would you prefer if I were not',
       '.',  'Perhaps in your fantasies I am'];
t009: elizaKey keys = ['i can\'t'] pairs =
      ['?',  'How do you know you can\'t',
       nil,  'Have you tried?',
       '.',  'Perhaps you can now'];
t010: elizaKey keys = ['i\'m' 'i am'] pairs =
      ['?',  'Did you come to me because you are',
       '?',  'How long have you been',
       '?',  'Do you believe it is normal to be',
       '?',  'Do you enjoy being'];
t011: elizaKey keys = ['you'] pairs =
      [nil,  'We were discussing you -- not me.',
       '?',  'Oh, I',
       nil,  'You\'re not really talking about me, are you?'];
t012: elizaKey keys = ['i want'] pairs =
      ['?',  'What would it mean to you if you got',
       '?',  'Why do you want',
       '.',  'Suppose you soon got',
       '?',  'What if you never got',
       '.',  'I sometimes also want'];
t013: elizaKey keys = ['who' 'what' 'where' 'when' 'why' 'how'] pairs =
      [nil,  'Why do you ask?',
       nil,  'Does that question interest you?',
       nil,  'What answer would please you the most?',
       nil,  'What do you think?',
       nil,  'Are such questions on your mind often?',
       nil,  'What is it that you really want to know?',
       nil,  'have you asked anyone else?',
       nil,  'have you asked such questions before?',
       nil,  'What else comes to mind when you ask that?'];
t014: elizaKey keys = ['name'] pairs =
      [nil,  'names don\'t interest me.',
       nil,  'I don\'t care about names -- please go on.'];
t015: elizaKey keys = ['cause'] pairs =
      [nil,  'Is that the real reason?',
       nil,  'Don\'t any other reasons come to mind?',
       nil,  'Does that reason explain anything else?',
       nil,  'What other reasons might there be?'];
t016: elizaKey keys = ['sorry'] pairs =
      [nil,  'Please don\'t apologize!',
       nil,  'Apologies are not necessary.',
       nil,  'What feelings do you have when you apologize?',
       nil,  'Don\'t be so defensive!'];
t017: elizaKey keys = ['dream'] pairs =
      [nil,  'What does that dream suggest to you?',
       nil,  'Do you dream often?',
       nil,  'What persons appear in your dreams?',
       nil,  'Are you disturbed by your dreams?'];
t018: elizaKey keys = ['hello' 'hi'] pairs =
      [nil,  'How do you do ... please state your problem.'];
t019: elizaKey keys = ['maybe'] pairs =
      [nil,  'You don\'t seem quite certain.',
       nil,  'Why the uncertain tone?',
       nil,  'Can\'t you be more positive?',
       nil,  'You aren\'t sure?',
       nil,  'Don\'t you know?'];
t020: elizaKey keys = ['no'] pairs =
      [nil,  'Are you saying no just to be negative?',
       nil,  'You are being a bit negative.',
       nil,  'Why not?',
       nil,  'Are you sure?',
       nil,  'Why no?'];
t021: elizaKey keys = ['your'] pairs =
      ['?',  'Why are you concerned about my',
       '?',  'What about your own'];
t022: elizaKey keys = ['always'] pairs =
      [nil,  'Can you think of a specific example?',
       nil,  'When?',
       nil,  'What are you thinking of?',
       nil,  'Really, always?'];
t023: elizaKey keys = ['think'] pairs =
      [nil,  'Do you really think so?',
       '?',  'But you are not sure you',
       '?',  'Do you doubt you'];
t024: elizaKey keys = ['alike'] pairs =
      [nil,  'In what way?',
       nil,  'What resemblance do you see?',
       nil,  'What does the similarity suggest to you?',
       nil,  'What other connections do you see?',
       nil,  'Could there really be some connection?',
       nil,  'How?',
       nil,  'You seem quite positive.'];
t025: elizaKey keys = ['yes'] pairs =
      [nil,  'Are you sure?',
       nil,  'I see.',
       nil,  'I understand.'];
t026: elizaKey keys = ['friend'] pairs =
      [nil,  'Why do you bring up the topic of friends?',
       nil,  'Do your friends worry you?',
       nil,  'Do your friends pick on you?',
       nil,  'Are you sure you have any friends?',
       nil,  'Do you impose on your friends?',
       nil,  'Perhaps your love for friends worries you.'];
t027: elizaKey keys = ['computer'] pairs =
      [nil,  'Do computers worry you?',
       nil,  'Are you talking about me in particular?',
       nil,  'Are you frightened by machines?',
       nil,  'Why do you mention computers?',
       nil,  'What do you think machines have to do with your problem?',
       nil,  'Don\'t you think computers can help people?',
       nil,  'What is it about machines that worries you?'];
elizaNokey: elizaKey keys = [] pairs =
      [nil,  'Say, do you have any psychological problems?',
       nil,  'What does that suggest to you?',
       nil,  'I see.',
       nil,  'I\'m not sure I understand you fully.',
       nil,  'Come, come, elucidate your thoughts.',
       nil,  'Can you elaborate on that?',
       nil,  'That is quite interesting.'];
