/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  DVEHICLE.T				
 *  version 1.0
 *
 *   	This file defines the dvehicle (directed-vehicle) class which allows for 
 *		the implentation of vehicles responsive to travelverb commands, such as 
 *		'north' or 'up' or 'in'.
 *
 *		Unlike the TADS normal vehicle class, which responds with 'You must get 
 *		out of the vehicle first.', the directed vehicle will attempt to move
 *		in the direction indicated.
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.2.6 or later
 *      + Should be #included after ADV.T and STD.T     
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      None.
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		99-03-10 Creation.
 */
 
#define __DVEHICLE_MODULE_
 
/*
 *	class dvehicle: vehicle
 *
 *	This vehicle responds to travelVerb commands, such as 'north', 'south',
 *	'up', 'out'. The vehicle will first check to see that it is active. If 
 *	not active then the inactiveMessage is printed. 
 *
 *	Next it checks to see if the direction indicated is a valid room direction, 
 *	indicated by the return of an object in the room's direction property. The 
 *	object is then checked to determine whether it is a passable obstacle and if 
 *	so, whether the obstacle returns an object. If it does not then the
 *	cantgoMessage is displayed.
 *
 *	Finally a check is made to determine if the direction is valid for the
 *	vehicle. Authors may wish to customize this code to limit the vehicle's
 *	movement to specific classes of rooms, lit rooms, or directions. If the
 *	direction is not valid for the vehicle then the cantgoMessage is displayed.
 *
 *	If these checks are passed then the vehicle is moved to its travelloc and
 *	the goMessage is displayed. If parserGetMe() is in the vehicle then we note
 *	the new room location.
 */
class dvehicle: vehicle
	isdvehicle = true
	isActive = nil
	traveldir = nil
	travelloc = nil
	/*
	 *	Only travelVerbs are handled. We do all our travel validation from here.
	 */
	roomAction( a, v, d, p, i ) =
	{
		if ( isclass( v, travelVerb ) )
		{
			if ( self.isActive )
			{
				if ( not ( self.validRoomDir( v ) 
				and self.validVehDir( v, self.travelloc ) ) )
				{
					self.cantgoMessage( self.traveldir, self.travelloc );
					exit;
				}
			}
			else
			{
				self.inactiveMessage;
				exit;
			}
		}
	}
	/*
	 *	This method checks whether the room the vehicle is in allows movement in
	 *	this direction. If the proptype of the room direction is an object we
	 *	then check to see whether it is an obstacle and whether the obstacle
	 *	will let us pass. If a valid room direction the method returns true; 
	 *	otherwise it returns nil.
	 */
	validRoomDir( v ) =
	{
		switch( v )
		{
			case nVerb:
				self.traveldir := 'north';
				if ( proptype( self.location, &north ) = 2 )
				{
					return self.checkPassable( self.location.north );
				}
				break;
			case sVerb:
				self.traveldir := 'south';
				if ( proptype( self.location, &south ) = 2 )
				{
					return self.checkPassable( self.location.south );
				}
				break;
			case eVerb:
				self.traveldir := 'east';
				if ( proptype( self.location, &east ) = 2 )
				{
					return self.checkPassable( self.location.east );
				}
				break;
			case wVerb:
				self.traveldir := 'west';
				if ( proptype( self.location, &west ) = 2 )
				{
					return self.checkPassable( self.location.west );
				}
				break;
			case neVerb:
				self.traveldir := 'northeast';
				if ( proptype( self.location, &ne ) = 2 )
				{
					return self.checkPassable( self.location.ne );
				}
				break;
			case nwVerb:
				self.traveldir := 'northwest';
				if ( proptype( self.location, &nw ) = 2 )
				{
					return self.checkPassable( self.location.nw );
				}
				break;
			case seVerb:
				self.traveldir := 'southeast';
				if ( proptype( self.location, &se ) = 2 )
				{
					return self.checkPassable( self.location.se );
				}
				break;
			case swVerb:
				self.traveldir := 'southwest';
				if ( proptype( self.location, &sw ) = 2 )
				{
					return self.checkPassable( self.location.sw );
				}
				break;
			case uVerb:
				self.traveldir := 'up';
				if ( proptype( self.location, &up ) = 2 )
				{
					return self.checkPassable( self.location.up );
				}
				break;
			case dVerb:
				self.traveldir := 'down';
				if ( proptype( self.location, &down ) = 2 )
				{
					return self.checkPassable( self.location.down );
				}
				break;
			case inVerb:
				self.traveldir := 'in';
				if ( proptype( self.location, &in ) = 2 )
				{
					return self.checkPassable( self.location.in );
				}
				break;
			case outVerb:
				self.traveldir := 'out';
				if ( proptype( self.location, &out ) = 2 )
				{
					return self.checkPassable( self.location.out );
				}
				break;
		}
		return nil;
	}
	/*
	 *	This method checks to see if the object pointed to by the room direction
	 *	is an obstacle. In this case we really want the doordest, not the door. 
	 *	We also check to see if the 'door' is open, as most vehicles aren't
	 *	driven through closed doors. This method returns true if there is no
	 *	obstacle, or if the obstacle is open and returns an object as its
	 *	doordest; otherwise it returns nil.
	 */
	checkPassable( o ) =
	{
		if ( isclass( o, obstacle ) )
		{	
			if ( o.isopen and proptype( o, &doordest ) = 2 )
			{
				self.travelloc := o.doordest;
				return true;
			}
			else
			{
				self.travelloc := nil;
				return nil;
			}
		}
		else
		{
			self.travelloc := o;
			return true;
		}
	}
	/*
	 *	This method is used to limit the movement of the vehicle itself. It is 
	 *	passed the travelVerb and the travelloc. By default it returns true. The
	 *	author should override this method to limit vehicle movement. For
	 *	instance, the author may want to limit a car so that it may not move up 
	 *	or down. If the direction is allowed then the method should return true;
	 *	otherwise it should return nil.
	 */
	validVehDir( v, l ) = { return true; }
	inactiveMessage =
	{
		caps(); "<<self.thedesc>> is not active. ";
	}
	goMessage( d, l ) = 
	{
		caps(); "<<self.thedesc>> heads "; say( d ); ". ";
	}
	cantgoMessage( d, l ) =
	{
		caps(); "<<self.thedesc>> can't go "; say( d ); ". ";
	}
	/* 
	 *	This method displays the goMessage, and moves the vehicle to the 
	 *	new location. It returns nil, because the movement is handled here and
	 *	not in the direction properties, which expect either a location or nil.
	 */
	moveVehicle =
	{
		self.goMessage( self.traveldir, self.travelloc ); "\b";
		self.moveInto( self.travelloc );
		if ( parserGetMe().isIn( self ) )
		{
			parserGetMe().location.lookAround( true );
			parserGetMe().location.isseen := true;
		}
		return nil;
	}
    north = { self.moveVehicle; }
    south = { self.moveVehicle; }
    east = 	{ self.moveVehicle; }
    west = 	{ self.moveVehicle; }
    ne = 	{ self.moveVehicle; }
    nw = 	{ self.moveVehicle; }
    se = 	{ self.moveVehicle; }
    sw = 	{ self.moveVehicle; }
    up = 	{ self.moveVehicle; }
    down = 	{ self.moveVehicle; }
    in = 	{ self.moveVehicle; }
    out = 	{ self.moveVehicle; }
;
