#ifdef RCSID
static char RCSid[] =
"$Header$";
#endif

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
Name
  vmarray.cpp - T3 array metaclass
Function
  
Notes
  
Modified
  03/07/00 MJRoberts  - Creation
*/

#include <stdlib.h>

#include "t3std.h"
#include "vmtype.h"
#include "vmmcreg.h"
#include "vmmeta.h"
#include "vmglob.h"
#include "vmobj.h"
#include "vmarray.h"
#include "vmlst.h"
#include "vmerr.h"
#include "vmerrnum.h"
#include "vmfile.h"
#include "vmstack.h"
#include "vmbif.h"
#include "vmundo.h"
#include "vmrun.h"
#include "vmiter.h"
#include "vmsort.h"


/* ------------------------------------------------------------------------ */
/*
 *   statics 
 */

/* metaclass registration object */
static CVmMetaclassArray metaclass_reg_obj;
CVmMetaclass *CVmObjArray::metaclass_reg_ = &metaclass_reg_obj;

/* function table */
int (CVmObjArray::
     *CVmObjArray::func_table_[])(VMG_ vm_obj_id_t self,
                                  vm_val_t *retval, uint *argc) =
{
    &CVmObjArray::getp_undef,
    &CVmObjArray::getp_to_list,
    &CVmObjArray::getp_get_size,
    &CVmObjArray::getp_copy_from,
    &CVmObjArray::getp_fill_val,
    &CVmObjArray::getp_subset,
    &CVmObjArray::getp_apply_all,
    &CVmObjArray::getp_index_which,
    &CVmObjArray::getp_for_each,
    &CVmObjArray::getp_map_all,
    &CVmObjArray::getp_index_of,
    &CVmObjArray::getp_val_which,
    &CVmObjArray::getp_last_index_of,
    &CVmObjArray::getp_last_index_which,
    &CVmObjArray::getp_last_val_which,
    &CVmObjArray::getp_count_of,
    &CVmObjArray::getp_count_which,
    &CVmObjArray::getp_get_unique,
    &CVmObjArray::getp_append_unique,
    &CVmObjArray::getp_sort
};


/* ------------------------------------------------------------------------ */
/*
 *   Create from stack arguments 
 */
vm_obj_id_t CVmObjArray::create_from_stack(VMG_ const uchar **pc_ptr,
                                           uint argc)
{
    vm_obj_id_t id;
    CVmObjArray *arr;
    size_t idx;
    size_t cnt;
    vm_val_t *arg1;
    const char *lst;

    /* check our arguments */
    if (argc < 1)
        err_throw(VMERR_WRONG_NUM_OF_ARGS);

    /* see what we have for the first argument */
    arg1 = G_stk->get(0);
    if (arg1->typ == VM_INT)
    {
        vm_val_t nil_val;
        
        /* 
         *   it's a simple count argument - make sure we only have one
         *   argument 
         */
        if (argc != 1)
            err_throw(VMERR_WRONG_NUM_OF_ARGS);

        /* get the number of elements */
        cnt = (size_t)arg1->val.intval;

        /* create the array with the given number of elements */
        id = vm_new_id(vmg_ FALSE);
        arr = new (vmg_ id) CVmObjArray(vmg_ cnt);

        /* set each element to nil */
        for (idx = 0, nil_val.set_nil() ; idx < cnt ; ++idx)
            arr->set_element(idx, &nil_val);
    }
    else if ((lst = arg1->get_as_list(vmg0_)) != 0
             || (arg1->typ == VM_OBJ
                 && (vm_objp(vmg_ arg1->val.obj)->get_metaclass_reg()
                     == metaclass_reg_)))
    {
        size_t src_idx;
        size_t src_cnt;
        int is_list;
        CVmObjArray *src_arr;

        /* get the size of the source list */
        if (lst != 0)
        {
            /* get the count from the list */
            src_cnt = vmb_get_len(lst);

            /* it's a list */
            is_list = TRUE;

            /* there's no source array */
            src_arr = 0;
        }
        else
        {
            /* remember the source array */
            src_arr = (CVmObjArray *)vm_objp(vmg_ arg1->val.obj);

            /* get the count from the array */
            src_cnt = src_arr->get_element_count();

            /* note that it's not a list */
            is_list = FALSE;
        }
        
        /* 
         *   we have a list - check for the optional actual element count,
         *   and the optional starting index 
         */
        if (argc == 1)
        {
            /* no size specified - use the same size as the original list */
            cnt = src_cnt;

            /* start at the first element */
            src_idx = 0;
        }
        else
        {
            /* make sure it's an integer */
            if (G_stk->get(1)->typ != VM_INT)
                err_throw(VMERR_INT_VAL_REQD);

            /* use the specified starting index */
            src_idx = (size_t)G_stk->get(1)->val.intval;

            /* if the index is below 1, force it to 1 */
            if (src_idx < 1)
                src_idx = 1;
            
            /* adjust to a zero-based index */
            --src_idx;
            
            /* check for the starting element argument */
            if (argc >= 3)
            {
                /* make sure it's an integer */
                if (G_stk->get(2)->typ != VM_INT)
                    err_throw(VMERR_INT_VAL_REQD);
                
                /* use the specified element count */
                cnt = (size_t)G_stk->get(2)->val.intval;

                /* make sure we don't have any extra arguments */
                if (argc > 3)
                    err_throw(VMERR_WRONG_NUM_OF_ARGS);
            }
            else
            {
                /* 
                 *   no count specified - use the number of elements in
                 *   the original remaining after the starting index 
                 */
                if (src_idx >= src_cnt)
                    cnt = 0;
                else
                    cnt = src_cnt - src_idx;
            }
        }

        /* create the new array */
        id = vm_new_id(vmg_ FALSE);
        arr = new (vmg_ id) CVmObjArray(vmg_ cnt);

        /* Copy the source list or array values */
        for (idx = 0 ; idx < cnt ; ++idx, ++src_idx)
        {
            vm_val_t val;

            /* 
             *   get this element of the original list or array; if we're
             *   past the end of the source, use nil 
             */
            if (src_idx < src_cnt)
            {
                /* get the list or array value, as appropriate */
                if (is_list)
                    CVmObjList::index_list(vmg_ &val, lst, src_idx + 1);
                else
                    src_arr->get_element(src_idx, &val);
            }
            else
            {
                /* we're out of source elements - use nil */
                val.set_nil();
            }
            
            /* set this element in our new list */
            arr->set_element(idx, &val);
        }
    }
    else
    {
        /* invalid argument */
        err_throw(VMERR_BAD_TYPE_BIF);
    }

    /* discard arguments */
    G_stk->discard(argc);

    /* return the new object */
    return id;
}

/* ------------------------------------------------------------------------ */
/*
 *   create 
 */
vm_obj_id_t CVmObjArray::create(VMG_ int in_root_set)
{
    vm_obj_id_t id = vm_new_id(vmg_ in_root_set);
    new (vmg_ id) CVmObjArray();
    return id;
}

/* ------------------------------------------------------------------------ */
/*
 *   create with known element count 
 */
vm_obj_id_t CVmObjArray::create(VMG_ int in_root_set, size_t element_count)
{
    vm_obj_id_t id = vm_new_id(vmg_ in_root_set);
    new (vmg_ id) CVmObjArray(vmg_ element_count);
    return id;
}

/* ------------------------------------------------------------------------ */
/*
 *   instantiate 
 */
CVmObjArray::CVmObjArray(VMG_ size_t element_count)
{
    /* allocate space */
    alloc_array(vmg_ element_count);
}

/* ------------------------------------------------------------------------ */
/*
 *   allocate space 
 */
void CVmObjArray::alloc_array(VMG_ size_t cnt)
{
    /* allocate space for the given number of elements */
    ext_ = (char *)G_mem->get_var_heap()->alloc_mem(calc_alloc(cnt), this);

    /* set the element count and allocated count */
    set_allocated_count(cnt);
    set_element_count(cnt);

    /* clear the undo bits */
    clear_undo_bits();
}

/* ------------------------------------------------------------------------ */
/*
 *   notify of deletion 
 */
void CVmObjArray::notify_delete(VMG_ int)
{
    /* free our extension */
    if (ext_ != 0)
        G_mem->get_var_heap()->free_mem(ext_);
}

/* ------------------------------------------------------------------------ */
/* 
 *   set a property 
 */
void CVmObjArray::set_prop(VMG_ class CVmUndo *,
                           vm_obj_id_t, vm_prop_id_t,
                           const vm_val_t *)
{
    err_throw(VMERR_INVALID_SETPROP);
}

/* ------------------------------------------------------------------------ */
/*
 *   Create an iterator 
 */
void CVmObjArray::new_iterator(VMG_ vm_val_t *retval,
                               const vm_val_t *self_val)
{
    vm_val_t copy_val;
    
    /* 
     *   Create a copy of the array - since an array can change, we must
     *   always create an iterator based on a copy, so that the iterator
     *   has a consistent view of the original array as of the time the
     *   iterator was created.  To create the copy, call the stack-based
     *   constructor with a self-reference as the argument.  
     */
    G_stk->push(self_val);
    copy_val.set_obj(create_from_stack(vmg_ 0, 1));

    /* 
     *   Set up a new indexed iterator object.  The first valid index for
     *   an array is always 1, and the last valid index is the same as the
     *   number of elements.  
     */
    retval->set_obj(CVmObjIterIdx::create_for_coll(vmg_ &copy_val,
        1, get_element_count()));
}

/*
 *   Create a live iterator
 */
void CVmObjArray::new_live_iterator(VMG_ vm_val_t *retval,
                                    const vm_val_t *self_val)
{
    /* 
     *   Set up a new indexed iterator object.  The first valid index for
     *   an array is always 1, and the last valid index is the same as the
     *   number of elements.  Since we want a "live" iterator, create the
     *   iterator with a direct reference to the array.  
     */
    retval->set_obj(CVmObjIterIdx::create_for_coll(vmg_ self_val,
        1, get_element_count()));
}

/* ------------------------------------------------------------------------ */
/* 
 *   get a property 
 */
int CVmObjArray::get_prop(VMG_ vm_prop_id_t prop, vm_val_t *retval,
                          vm_obj_id_t self, vm_obj_id_t *source_obj,
                          uint *argc)
{
    ushort func_idx;
    
    /* translate the property index to an index into our function table */
    func_idx = G_meta_table
               ->prop_to_vector_idx(metaclass_reg_->get_reg_idx(), prop);

    /* call the appropriate function */
    if ((this->*func_table_[func_idx])(vmg_ self, retval, argc))
    {
        *source_obj = metaclass_reg_->get_class_obj(vmg0_);
        return TRUE;
    }

    /* inherit default handling */
    return CVmObjCollection::get_prop(vmg_ prop, retval, self,
                                      source_obj, argc);
}

/* ------------------------------------------------------------------------ */
/* 
 *   notify of new undo savepoint 
 */
void CVmObjArray::notify_new_savept()
{
    /* 
     *   clear the undo bits - we obviously have no undo for the new
     *   savepoint yet 
     */
    clear_undo_bits();
}

/* ------------------------------------------------------------------------ */
/*
 *   apply undo 
 */
void CVmObjArray::apply_undo(VMG_ CVmUndoRecord *rec)
{
    /* set the element from the undo data */
    set_element((size_t)rec->id.intval, &rec->oldval);
}

/* ------------------------------------------------------------------------ */
/*
 *   mark references from an undo record 
 */
void CVmObjArray::mark_undo_ref(VMG_ CVmUndoRecord *rec)
{
    /* if the undo record refers to an object, mark the object */
    if (rec->oldval.typ == VM_OBJ)
        G_obj_table->mark_all_refs(rec->oldval.val.obj, VMOBJ_REACHABLE);
}

/* ------------------------------------------------------------------------ */
/* 
 *   mark references 
 */
void CVmObjArray::mark_refs(VMG_ uint state)
{
    size_t cnt;
    char *p;

    /* get my element count */
    cnt = get_element_count();

    /* mark as referenced each object in our array */
    for (p = get_element_ptr(0) ; cnt != 0 ; --cnt, inc_element_ptr(&p))
    {
        /* 
         *   if this is an object, mark it as referenced, and mark its
         *   references as referenced 
         */
        if (vmb_get_dh_type(p) == VM_OBJ)
            G_obj_table->mark_all_refs(vmb_get_dh_obj(p), state);
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   load from an image file 
 */
void CVmObjArray::load_from_image(VMG_ vm_obj_id_t self,
                                  const char *ptr, size_t siz)
{
    /* load from the image data */
    load_image_data(vmg_ ptr, siz);

    /* 
     *   save our image data pointer in the object table, so that we can
     *   access it (without storing it ourselves) during a reload
     */
    G_obj_table->save_image_pointer(self, ptr, siz);
}

/*
 *   reload the object from image data 
 */
void CVmObjArray::reload_from_image(VMG_ vm_obj_id_t,
                                    const char *ptr, size_t siz)
{
    /* load the image data */
    load_image_data(vmg_ ptr, siz);
}

/*
 *   Load from image data 
 */
void CVmObjArray::load_image_data(VMG_ const char *ptr, size_t siz)
{
    size_t cnt;

    /* if we already have memory allocated, free it */
    if (ext_ != 0)
    {
        G_mem->get_var_heap()->free_mem(ext_);
        ext_ = 0;
    }

    /* get the new array size */
    cnt = vmb_get_len(ptr);

    /* make sure the size isn't larger than we'd expect */
    if (siz > VMB_LEN + (VMB_DATAHOLDER * cnt))
        siz = VMB_LEN + (VMB_DATAHOLDER * cnt);

    /* allocate memory at the new size as indicated in the image data */
    alloc_array(vmg_ cnt);

    /* if the size is smaller than we'd expect, set extra elements to nil */
    if (siz < VMB_LEN + (VMB_DATAHOLDER * cnt))
    {
        size_t i;
        vm_val_t nil_val;
        
        /* set all elements to nil */
        nil_val.set_nil();
        for (i = 0 ; i < cnt ; ++i)
            set_element(i, &nil_val);
    }

    /* copy the elements from the image data */
    memcpy(get_element_ptr(0), ptr + VMB_LEN, siz - VMB_LEN);

    /* we're resetting to initial state, so forget all undo */
    clear_undo_bits();
}

/* ------------------------------------------------------------------------ */
/*
 *   save 
 */
void CVmObjArray::save_to_file(VMG_ CVmFile *fp)
{
    size_t cnt;

    /* get our element count */
    cnt = get_element_count();

    /* write the count and the elements */
    fp->write_int2(cnt);
    fp->write_bytes(get_element_ptr(0), calc_alloc_ele(cnt));
}

/* ------------------------------------------------------------------------ */
/*
 *   restore
 */
void CVmObjArray::restore_from_file(VMG_ vm_obj_id_t,
                                    CVmFile *fp, CVmObjFixup *fixups)
{
    size_t cnt;

    /* read the element count */
    cnt = fp->read_uint2();

    /* free any existing extension */
    if (ext_ != 0)
    {
        G_mem->get_var_heap()->free_mem(ext_);
        ext_ = 0;
    }

    /* allocate the space */
    alloc_array(vmg_ cnt);

    /* read the contents */
    fp->read_bytes(get_element_ptr(0), calc_alloc_ele(cnt));

    /* fix up the contents */
    fixups->fix_dh_array(vmg_ get_element_ptr(0), cnt);
}

/* ------------------------------------------------------------------------ */
/* 
 *   index the array 
 */
void CVmObjArray::index_val(VMG_ vm_val_t *result, vm_obj_id_t,
                            const vm_val_t *index_val)
{
    uint32 idx;

    /* get the index value as an integer */
    idx = index_val->num_to_int();

    /* make sure it's in range - 1 to our element count, inclusive */
    if (idx < 1 || idx > get_element_count())
        err_throw(VMERR_INDEX_OUT_OF_RANGE);

    /* 
     *   get the indexed element and store it in the result, adjusting the
     *   index to the C-style 0-based range 
     */
    get_element(idx - 1, result);
}

/* ------------------------------------------------------------------------ */
/* 
 *   set an indexed element of the array 
 */
void CVmObjArray::set_index_val(VMG_ vm_val_t *new_container,
                                vm_obj_id_t self,
                                const vm_val_t *index_val,
                                const vm_val_t *new_val)
{
    uint32 idx;

    /* get the index value as an integer */
    idx = index_val->num_to_int();

    /* make sure it's in range - 1 to our element count, inclusive */
    if (idx < 1 || idx > get_element_count())
        err_throw(VMERR_INDEX_OUT_OF_RANGE);

    /* adjust to a zero-based index */
    --idx;

    /* set the element and record undo */
    set_element_undo(vmg_ self, (size_t)idx, new_val);

    /* the result is the original array value */
    new_container->set_obj(self);
}

/* ------------------------------------------------------------------------ */
/*
 *   Set an element and record undo for the change
 */
void CVmObjArray::set_element_undo(VMG_ vm_obj_id_t self,
                                   size_t idx, const vm_val_t *new_val)
{
    /* if we don't have undo for this element already, save undo now */
    if (G_undo != 0 && !get_undo_bit(idx))
    {
        vm_val_t old_val;

        /* get the pre-modification value of this element */
        get_element(idx, &old_val);

        /* add the undo record */
        G_undo->add_new_record_int_key(vmg_ self, idx, &old_val);

        /* 
         *   set the undo bit to indicate that we now have undo for this
         *   element 
         */
        set_undo_bit(idx, TRUE);
    }

    /* 
     *   get the indexed element and store it in the result, adjusting the
     *   index to the C-style 0-based range 
     */
    set_element(idx, new_val);
}

/* ------------------------------------------------------------------------ */
/* 
 *   Check a value for equality.  We will match any list or array with the
 *   same number of elements and the same value for each element.  
 */
int CVmObjArray::equals(VMG_ vm_obj_id_t self, const vm_val_t *val) const
{
    const char *p;
    int p_is_list;
    CVmObjArray *arr2;
    size_t cnt;
    size_t cnt2;
    size_t idx;

    /* if the other value is a reference to myself, we certainly match */
    if (val->typ == VM_OBJ && val->val.obj == self)
    {
        /* no need to look at the contents if this is a reference to me */
        return TRUE;
    }

    /* try it as a list and as an array (or array subclass) */
    if ((p = val->get_as_list(vmg0_)) != 0)
    {
        /* it's a list */
        p_is_list = TRUE;

        /* get the number of elements in the list */
        cnt2 = vmb_get_len(p);
    }
    else
    {
        /* it's not a list */
        p_is_list = FALSE;

        /* if it's an array, get its extension data */
        if (val->typ == VM_OBJ
            && (vm_objp(vmg_ val->val.obj)->is_of_metaclass(metaclass_reg_)))
        {
            /* 
             *   get the object's extension, which uses the same format as
             *   lists 
             */
            arr2 = (CVmObjArray *)vm_objp(vmg_ val->val.obj);

            /* get the number of elements */
            cnt2 = arr2->get_element_count();
        }
        else
        {
            /* it's some other type, so it can't be equal */
            return FALSE;
        }
    }

    /* if the sizes don't match, the values are not equal */
    cnt = get_element_count();
    if (cnt != cnt2)
        return FALSE;

    /* compare element by element */
    for (idx = 0 ; idx < cnt ; ++idx)
    {
        vm_val_t val1;
        vm_val_t val2;
        
        /* get this element of my array */
        get_element(idx, &val1);

        /* get this element of the other value */
        if (p_is_list)
            CVmObjList::index_list(vmg_ &val2, p, idx + 1);
        else
            arr2->get_element(idx, &val2);

        /* if these elements aren't equal, our values aren't equal */
        if (!val1.equals(vmg_ &val2))
            return FALSE;

        /* 
         *   if the other value is a list, re-translate it in case we did
         *   any constant data swapping 
         */
        if (p_is_list)
            p = val->get_as_list(vmg0_);
    }

    /* we didn't find any differences, so the values are equal */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   Hash value calculation 
 */
uint CVmObjArray::calc_hash(VMG_ vm_obj_id_t self) const
{
    uint hash;
    uint i;
    
    /* combine the hash values of the members of the array */
    for (hash = 0, i = 0 ; i < get_element_count() ; ++i)
    {
        vm_val_t ele;
        
        /* get this element */
        get_element(i, &ele);

        /* calculate its hash value and add it into the combined hash */
        hash += ele.calc_hash(vmg0_);
    }

    /* return the combined hash */
    return hash;
}

/* ------------------------------------------------------------------------ */
/* 
 *   add a value to the array 
 */
void CVmObjArray::add_val(VMG_ vm_val_t *result,
                          vm_obj_id_t self, const vm_val_t *val)
{
    size_t idx;
    size_t rhs_cnt;
    size_t i;
    CVmObjArray *new_arr;

    /* push the value to append, for gc protection */
    G_stk->push(val);

    /* 
     *   remember the index of the first new element - this is simply one
     *   higher than the last valid current index 
     */
    idx = get_element_count();

    /* get the number of elements to add */
    rhs_cnt = val->get_coll_addsub_rhs_ele_cnt(vmg0_);

    /* allocate a new array (or appropriate subclass) for the return value */
    result->set_obj(create_for_append(vmg_ self, idx + rhs_cnt));

    /* push the result for gc protection */
    G_stk->push(result);

    /* get the return value as an array */
    new_arr = (CVmObjArray *)vm_objp(vmg_ result->val.obj);

    /* copy myself into the result, if returning a new object */
    if (new_arr != this)
        memcpy(new_arr->get_element_ptr(0), get_element_ptr(0),
               calc_alloc_ele(idx));

    /* add the new elements */
    for (i = 0 ; i < rhs_cnt ; ++i, ++idx)
    {
        vm_val_t ele;

        /* get this element from the right-hand side */
        val->get_coll_addsub_rhs_ele(vmg_ i + 1, &ele);

        /* store the element in the result */
        new_arr->set_element(idx, &ele);
    }

    /* discard the gc protect */
    G_stk->discard(2);
}

/* ------------------------------------------------------------------------ */
/* 
 *   subtract a value from the array
 */
void CVmObjArray::sub_val(VMG_ vm_val_t *result,
                          vm_obj_id_t self, const vm_val_t *val)
{
    size_t ele_cnt;
    size_t rhs_cnt;
    size_t i;
    CVmObjArray *new_arr;

    /* push the value to append, for gc protection */
    G_stk->push(val);

    /* note the initial element count */
    ele_cnt = get_element_count();

    /* get the number of elements to remove */
    rhs_cnt = val->get_coll_addsub_rhs_ele_cnt(vmg0_);

    /* 
     *   allocate a new array (or appropriate subclass) for the return
     *   value - the new object will be no larger than the current object,
     *   so allocate at the same size if we're allocating a new object
     */
    result->set_obj(create_for_append(vmg_ self, ele_cnt));

    /* push the result for gc protection */
    G_stk->push(result);

    /* get the return value as an array */
    new_arr = (CVmObjArray *)vm_objp(vmg_ result->val.obj);

    /* copy myself into the result, if returning a new object */
    if (new_arr != this)
        memcpy(new_arr->get_element_ptr(0), get_element_ptr(0),
               calc_alloc_ele(ele_cnt));

    /* subtract each element of the right-hand side */
    for (i = 0 ; i < rhs_cnt ; ++i)
    {
        vm_val_t ele_val;

        /* get this right-hand side element */
        val->get_coll_addsub_rhs_ele(vmg_ i + 1, &ele_val);

        /* subtract it */
        ele_cnt = new_arr->sub_one_val(vmg_ self, &ele_val, ele_cnt);
    }

    /* update the stored element count */
    new_arr->set_element_count_for_sub(vmg_ self, ele_cnt);

    /* discard the gc protection */
    G_stk->discard(2);
}

/*
 *   Subtract an element from the values.  This doesn't use or change the
 *   stored element count, but rather uses the given element count and
 *   returns the new element count; this allows this routine to be called
 *   repeatedly to subtract a series of values from the array without
 *   creating individual undo records for each subtraction.  The caller is
 *   responsible for storing the final element count when done with the
 *   entire bundle of subtractions.  
 */
size_t CVmObjArray::sub_one_val(VMG_ vm_obj_id_t self,
                                const vm_val_t *val, size_t ele_cnt)
{
    size_t src;
    size_t dst;
    size_t idx;
    vm_val_t nil_val;

    /* search the array for the given value, and remove each occurrence */
    for (src = dst = 0 ; src < ele_cnt ; ++src)
    {
        vm_val_t ele;

        /* get this element */
        get_element(src, &ele);

        /* check for a match to the value to be subtracted */
        if (!ele.equals(vmg_ val))
        {
            /* 
             *   this element must be copied to the output - if the
             *   destination and source pointers aren't the same, copy
             *   this source value to its new position (note that we
             *   specifically do not perform the copy if it the pointers
             *   are the same so that we can avoid creating unnecessary
             *   undo records) 
             */
            if (dst != src)
                set_element_undo(vmg_ self, dst, &ele);

            /* adjust the destination index for the added value */
            ++dst;
        }
    }

    /*
     *   Fill out the rest of the elements (the ones that will be
     *   eliminated after we set the new array size) with nil.  This will
     *   ensure that we create an undo record for each of the element
     *   slots we're dropping, so that they'll be restored to their
     *   pre-shrinkage values if we undo. 
     */
    nil_val.set_nil();
    for (idx = dst ; idx < ele_cnt ; ++idx)
        set_element_undo(vmg_ self, idx, &nil_val);

    /* return the number of elements we retained */
    return dst;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - convert to a list 
 */
int CVmObjArray::getp_to_list(VMG_ vm_obj_id_t self, vm_val_t *retval,
                              uint *argc)
{
    size_t src_cnt;
    size_t dst_cnt;
    size_t start_idx;
    vm_val_t self_val;
    CVmObjList *lst;
    size_t idx;
    uint orig_argc = (argc != 0 ? *argc : 0);
    static CVmNativeCodeDesc desc(0, 2);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* note our size */
    src_cnt = get_element_count();

    /* 
     *   if there's a starting index, retrieve it; otherwise, start at our
     *   first element 
     */
    if (orig_argc >= 1)
        start_idx = (size_t)CVmBif::pop_int_val(vmg0_);
    else
        start_idx = 1;

    /* if the starting index is below 1, force it to 1 */
    if (start_idx < 1)
        start_idx = 1;

    /* adjust the starting index to a 0-based index */
    --start_idx;

    /* 
     *   if there's a size argument, retrieve it; if it's not specified,
     *   use our actual size for the output size 
     */
    if (orig_argc >= 2)
        dst_cnt = (size_t)CVmBif::pop_int_val(vmg0_);
    else
        dst_cnt = src_cnt;

    /* 
     *   in no case will the result list have more elements than we can
     *   actually supply 
     */
    if (start_idx >= src_cnt)
    {
        /* we're starting past our last element - we can't supply anything */
        dst_cnt = 0;
    }
    else if (src_cnt - start_idx < dst_cnt)
    {
        /* we can't supply as many values as requested - lower the size */
        dst_cnt = src_cnt - start_idx;
    }

    /* push a self-reference for garbage collection protection */
    self_val.set_obj(self);
    G_stk->push(&self_val);

    /* create the new list */
    retval->set_obj(CVmObjList::create(vmg_ FALSE, dst_cnt));
    lst = (CVmObjList *)vm_objp(vmg_ retval->val.obj);

    /* set the list elements */
    for (idx = 0 ; idx < dst_cnt ; ++idx)
    {
        vm_val_t val;

        /* get my element at this index */
        get_element(idx + start_idx, &val);

        /* store the element in the list */
        lst->cons_set_element(idx, &val);
    }

    /* discard the self-reference */
    G_stk->discard();

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - copy from another array or list 
 */
int CVmObjArray::getp_copy_from(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                uint *argc)
{
    vm_val_t src_val;
    size_t src_cnt;
    size_t dst_cnt;
    size_t src_start;
    size_t dst_start;
    size_t copy_cnt;
    const char *src_lst;
    int src_is_list;
    CVmObjArray *src_arr;
    size_t i;
    static CVmNativeCodeDesc desc(4);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* retrieve the source value */
    G_stk->pop(&src_val);

    /* get the source starting index */
    src_start = (size_t)CVmBif::pop_int_val(vmg0_);

    /* get the destination starting index */
    dst_start = (size_t)CVmBif::pop_int_val(vmg0_);

    /* get the number of elements to copy */
    copy_cnt = (size_t)CVmBif::pop_int_val(vmg0_);

    /* if either starting index is below 1, force it to 1 */
    if (src_start < 1)
        src_start = 1;
    if (dst_start < 1)
        dst_start = 1;

    /* adjust the starting indices to 0-based values */
    --src_start;
    --dst_start;

    /* note the destination length (i.e., my element count) */
    dst_cnt = get_element_count();

    /* get the source value */
    if ((src_lst = src_val.get_as_list(vmg0_)) != 0)
    {
        /* note that it's a list */
        src_is_list = TRUE;

        /* there's no source array */
        src_arr = 0;

        /* note the source size */
        src_cnt = vmb_get_len(src_lst);
    }
    else if (src_val.typ == VM_OBJ
             && (vm_objp(vmg_ src_val.val.obj)->get_metaclass_reg()
                 == metaclass_reg_))
    {
        /* it's an array */
        src_arr = (CVmObjArray *)vm_objp(vmg_ src_val.val.obj);

        /* it's not a list */
        src_is_list = FALSE;

        /* note the soure size */
        src_cnt = src_arr->get_element_count();
    }
    else
        err_throw(VMERR_BAD_TYPE_BIF);

    /* limit our copying to the remaining elements of the source */
    if (src_start >= src_cnt)
        copy_cnt = 0;
    else if (src_start + copy_cnt >= src_cnt)
        copy_cnt = src_cnt - src_start;

    /* limit our copying to the remaining elements of the destination */
    if (dst_start >= dst_cnt)
        copy_cnt = 0;
    else if (dst_start + copy_cnt >= dst_cnt)
        copy_cnt = dst_cnt - dst_start;

    /* set the list elements */
    for (i = 0 ; i < copy_cnt ; ++i)
    {
        vm_val_t val;

        /* get my element at this index */
        if (src_is_list)
            CVmObjList::index_list(vmg_ &val, src_lst, i + src_start + 1);
        else
            src_arr->get_element(i + src_start, &val);

        /* set my element at this index, recording undo for the change */
        set_element_undo(vmg_ self, i + dst_start, &val);
    }

    /* the return value is 'self' */
    retval->set_obj(self);

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - fill with a value 
 */
int CVmObjArray::getp_fill_val(VMG_ vm_obj_id_t self, vm_val_t *retval,
                               uint *argc)
{
    vm_val_t fill_val;
    size_t cnt;
    size_t start_idx;
    size_t end_idx;
    size_t idx;
    uint orig_argc = (argc != 0 ? *argc : 0);
    static CVmNativeCodeDesc desc(1, 2);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* note our size */
    cnt = get_element_count();

    /* get the fill value */
    G_stk->pop(&fill_val);

    /* 
     *   if there's a starting index, retrieve it; otherwise, start at our
     *   first element 
     */
    if (orig_argc >= 2)
        start_idx = (size_t)CVmBif::pop_int_val(vmg0_);
    else
        start_idx = 1;

    /* if the starting index is below 1, force it to 1 */
    if (start_idx < 1)
        start_idx = 1;

    /* adjust the starting index to a 0-based index */
    --start_idx;

    /* 
     *   if there's a count argument, retrieve it; if it's not specified,
     *   use our actual size for the count 
     */
    if (orig_argc >= 3)
        end_idx = start_idx + (size_t)CVmBif::pop_int_val(vmg0_);
    else
        end_idx = cnt;

    /* limit the copy count to our available elements */
    if (end_idx > cnt)
        end_idx = cnt;

    /* set the elements to the fill value */
    for (idx = start_idx ; idx < end_idx ; ++idx)
    {
        /* set the element to the fill value, recording undo for the change */
        set_element_undo(vmg_ self, idx, &fill_val);
    }

    /* the return value is 'self' */
    retval->set_obj(self);

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - get the number of elements 
 */
int CVmObjArray::getp_get_size(VMG_ vm_obj_id_t, vm_val_t *retval,
                               uint *argc)
{
    static CVmNativeCodeDesc desc(0);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* return my element count */
    retval->set_int(get_element_count());

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - select a subset
 */
int CVmObjArray::getp_subset(VMG_ vm_obj_id_t self, vm_val_t *retval,
                             uint *argc)
{
    const vm_val_t *func_val;
    size_t src;
    size_t dst;
    size_t cnt;
    CVmObjArray *new_arr;
    static CVmNativeCodeDesc desc(1);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* get the function pointer argument, but leave it on the stack */
    func_val = G_stk->get(0);

    /* push a self-reference while allocating to protect from gc */
    G_interpreter->push_obj(vmg_ self);

    /* 
     *   allocate the new array (or appropriate subclass) that we'll
     *   return - all we know at this point is that the new array won't be
     *   larger than the current array, so allocate at the current size 
     */
    cnt = get_element_count();
    retval->set_obj(create_for_getp(vmg_ cnt));

    /* get the return value as an array */
    new_arr = (CVmObjArray *)vm_objp(vmg_ retval->val.obj);

    /* 
     *   push a reference to the new list to protect it from the garbage
     *   collector, which could be invoked in the course of executing the
     *   user callback 
     */
    G_stk->push(retval);

    /*
     *   Go through each element of our list, and invoke the callback on
     *   each element.  If the element passes, write it to the current
     *   output location in the list; otherwise, just skip it.
     *   
     *   Note that we're using the same list as source and destination,
     *   which is easy because the list will either shrink or stay the
     *   same - we'll never need to insert new elements.  
     */
    for (src = dst = 0 ; src < cnt ; ++src)
    {
        vm_val_t ele;
        const vm_val_t *val;

        /* get this element from the source array */
        get_element(src, &ele);

        /* push the element as the callback's argument */
        G_stk->push(&ele);

        /* invoke the callback */
        G_interpreter->call_func_ptr(vmg_ func_val, 1,
                                     TRUE, "array.subset", 0);

        /* get the result from R0 */
        val = G_interpreter->get_r0();

        /* 
         *   if the callback returned non-nil and non-zero, include this
         *   element in the result 
         */
        if (val->typ == VM_NIL
            || (val->typ == VM_INT && val->val.intval == 0))
        {
            /* it's nil or zero - don't include it in the result */
        }
        else
        {
            /* 
             *   include this element in the result (there's no need to
             *   save undo, since the whole array is new) 
             */
            new_arr->set_element(dst, &ele);

            /* advance the output index */
            ++dst;
        }
    }

    /* set the actual number of elements in the result */
    new_arr->set_element_count(dst);

    /* discard our gc protection (self, return value) and our arguments */
    G_stk->discard(3);

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - apply a callback to each element
 */
int CVmObjArray::getp_apply_all(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                uint *argc)
{
    const vm_val_t *func_val;
    size_t idx;
    static CVmNativeCodeDesc desc(1);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* get the function pointer argument, but leave it on the stack */
    func_val = G_stk->get(0);

    /* push a self-reference while working to protect from gc */
    G_interpreter->push_obj(vmg_ self);

    /* 
     *   we're going to return the 'self' object, since we update the
     *   array in-place 
     */
    retval->set_obj(self);

    /*
     *   Go through each element of our array, and invoke the callback on
     *   each element.  Replace each element with the result of the
     *   callback.  Note that we intentionally re-check the element count on
     *   each iteration, in case the callback changes the number of
     *   elements.  
     */
    for (idx = 0 ; idx < get_element_count() ; ++idx)
    {
        vm_val_t ele;

        /* get this element */
        get_element(idx, &ele);

        /* push the element as the callback's argument */
        G_stk->push(&ele);

        /* invoke the callback */
        G_interpreter->call_func_ptr(vmg_ func_val, 1,
                                     TRUE, "array.applyAll", 0);

        /* replace this element with the result */
        set_element_undo(vmg_ self, idx, G_interpreter->get_r0());
    }

    /* discard our gc protection (self) and our arguments */
    G_stk->discard(2);

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - find the first element matching a condition
 */
int CVmObjArray::getp_index_which(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                  uint *argc)
{
    /* use our general handler, working forwards */
    return gen_index_which(vmg_ self, retval, argc, TRUE, FALSE);
}

/*
 *   property evaluator - lastIndexWhich 
 */
int CVmObjArray::getp_last_index_which(VMG_ vm_obj_id_t self,
                                       vm_val_t *retval, uint *argc)
{
    /* use our general handler, working backwards */
    return gen_index_which(vmg_ self, retval, argc, FALSE, FALSE);
}

/*
 *   general handler for indexWhich and lastIndexWhich 
 */
int CVmObjArray::gen_index_which(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                 uint *argc, int forward, int count_only)
{
    const vm_val_t *func_val;
    size_t cnt;
    size_t idx;
    int val_cnt;
    static CVmNativeCodeDesc desc(1);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* get the function pointer argument, but leave it on the stack */
    func_val = G_stk->get(0);

    /* push a self-reference while working to protect from gc */
    G_interpreter->push_obj(vmg_ self);

    /* presume we won't find a matching element */
    retval->set_nil();
    val_cnt = 0;

    /* get the number of elements to visit */
    cnt = get_element_count();

    /* start at the first or last element, depending on direction */
    idx = (forward ? 0 : cnt);

    /*
     *   Go through each element of our array, and invoke the callback on
     *   each element, looking for the first one that returns true.  
     */
    for (;;)
    {
        vm_val_t ele;

        /* if we've reached the last element, we can stop looking */
        if (forward ? idx >= cnt : idx == 0)
            break;

        /* if we're going backwards, decrement the index */
        if (!forward)
            --idx;

        /* get this element */
        get_element(idx, &ele);

        /* push the element as the callback's argument */
        G_stk->push(&ele);

        /* invoke the callback */
        G_interpreter->call_func_ptr(vmg_ func_val, 1,
                                     TRUE, "array.indexWhich", 0);

        /* 
         *   if the callback returned true, we've found the element we're
         *   looking for 
         */
        if (G_interpreter->get_r0()->typ == VM_NIL
            || (G_interpreter->get_r0()->typ == VM_INT
                && G_interpreter->get_r0()->val.intval == 0))
        {
            /* nil or zero - this one failed the test, so keep looking */
        }
        else
        {
            /* it passed the test - check what we're doing */
            if (count_only)
            {
                /* we only want the count */
                ++val_cnt;
            }
            else
            {
                /* we want the (1-based) index - return it */
                retval->set_int(idx + 1);
            
                /* found it - no need to keep looking */
                break;
            }
        }

        /* if we're going forwards, increment the index */
        if (forward)
            ++idx;
    }
    
    /* discard our gc protection (self) and our arguments */
    G_stk->discard(2);

    /* return the count if desired */
    if (count_only)
        retval->set_int(val_cnt);
    
    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - lastValWhich 
 */
int CVmObjArray::getp_last_val_which(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                     uint *argc)
{
    /* get the index of the value using lastIndexWhich */
    getp_last_index_which(vmg_ self, retval, argc);

    /* if the return value is a valid index, get the value at the index */
    if (retval->typ == VM_INT)
    {
        int idx;
        
        /* get the element as the return value */
        idx = (int)retval->val.intval;
        get_element(idx - 1, retval);
    }

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - valWhich 
 */
int CVmObjArray::getp_val_which(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                uint *argc)
{
    /* get the index of the value using indexWhich */
    getp_index_which(vmg_ self, retval, argc);

    /* if the return value is a valid index, get the value at the index */
    if (retval->typ == VM_INT)
    {
        int idx;

        /* get the element as the return value */
        idx = (int)retval->val.intval;
        get_element(idx - 1, retval);
    }

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - call a callback for each element
 */
int CVmObjArray::getp_for_each(VMG_ vm_obj_id_t self, vm_val_t *retval,
                               uint *argc)
{
    const vm_val_t *func_val;
    size_t idx;
    static CVmNativeCodeDesc desc(1);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* get the function pointer argument, but leave it on the stack */
    func_val = G_stk->get(0);

    /* push a self-reference while working to protect from gc */
    G_interpreter->push_obj(vmg_ self);

    /* no return value */
    retval->set_nil();

    /* 
     *   Invoke the callback on each element.  Note that we intentionally do
     *   not cache the element count, since it is possible that a program
     *   could change the array size (especially for a Vector subclass) in
     *   the course of an iteration; if we cached the size, and the actual
     *   size were reduced during the iteration, we would visit invalid
     *   elements past the new end of the array.  To avoid this possibility,
     *   we re-check the current element count on each iteration to make
     *   sure we haven't run off the end of the array.  
     */
    for (idx = 0 ; idx < get_element_count() ; ++idx)
    {
        vm_val_t ele;

        /* get this element */
        get_element(idx, &ele);

        /* push the element as the callback's argument */
        G_stk->push(&ele);

        /* invoke the callback */
        G_interpreter->call_func_ptr(vmg_ func_val, 1,
                                     TRUE, "array.forEach", 0);
    }

    /* discard our gc protection (self) and our arguments */
    G_stk->discard(2);
    
    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - mapAll
 */
int CVmObjArray::getp_map_all(VMG_ vm_obj_id_t self, vm_val_t *retval,
                              uint *argc)
{
    const vm_val_t *func_val;
    size_t idx;
    CVmObjArray *new_arr;
    static CVmNativeCodeDesc desc(1);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* get the function pointer argument, but leave it on the stack */
    func_val = G_stk->get(0);

    /* push a self-reference while working to protect from gc */
    G_interpreter->push_obj(vmg_ self);

    /* 
     *   allocate a new array for the return value - the new array will
     *   have the same size as the original, since we're mapping each
     *   element of the old array to the corresponding element of the new
     *   array 
     */
    retval->set_obj(create_for_getp(vmg_ get_element_count()));

    /* get the return value as an array */
    new_arr = (CVmObjArray *)vm_objp(vmg_ retval->val.obj);

    /* set the result element count the same as my own */
    new_arr->set_element_count(get_element_count());

    /* 
     *   push a reference to the new list to protect it from the garbage
     *   collector, which could be invoked in the course of executing the
     *   user callback 
     */
    G_stk->push(retval);

    /*
     *   Go through each element of our array, and invoke the callback on
     *   each element, storing the result in the corresponding element of
     *   the new array.  Note that we re-check the element count on each
     *   iteration, in case the callback changes it on us.  
     */
    for (idx = 0 ; idx < get_element_count() ; ++idx)
    {
        vm_val_t ele;

        /* get this element */
        get_element(idx, &ele);

        /* push the element as the callback's argument */
        G_stk->push(&ele);

        /* invoke the callback */
        G_interpreter->call_func_ptr(vmg_ func_val, 1,
                                     TRUE, "array.mapAll", 0);

        /* 
         *   replace this element with the result (there's no need to save
         *   undo, since the whole array is new) 
         */
        new_arr->set_element(idx, G_interpreter->get_r0());
    }

    /* discard our gc protection (self, new array) and our arguments */
    G_stk->discard(3);

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - indexOf 
 */
int CVmObjArray::getp_index_of(VMG_ vm_obj_id_t self, vm_val_t *retval,
                               uint *argc)
{
    /* use our general handler, going forwards */
    return gen_index_of(vmg_ self, retval, argc, TRUE, FALSE);
}

/*
 *   property evaluator - lastIndexOf 
 */
int CVmObjArray::getp_last_index_of(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                    uint *argc)
{
    /* use our general handler, going backwards */
    return gen_index_of(vmg_ self, retval, argc, FALSE, FALSE);
}

/*
 *   general handler for indexOf and lastIndexOf - searches the array
 *   either forwards of backwards for a given value 
 */
int CVmObjArray::gen_index_of(VMG_ vm_obj_id_t self, vm_val_t *retval,
                              uint *argc, int forward, int count_only)
{
    const vm_val_t *val;
    size_t cnt;
    size_t idx;
    int val_cnt;
    static CVmNativeCodeDesc desc(1);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* get the value, but leave it on the stack */
    val = G_stk->get(0);

    /* push a self-reference while working to protect from gc */
    G_interpreter->push_obj(vmg_ self);

    /* presume we won't find a matching element */
    retval->set_nil();
    val_cnt = 0;

    /* get the number of elements to visit */
    cnt = get_element_count();

    /* start at the first or last element, depending on the direction */
    idx = (forward ? 0 : cnt);

    /* scan the list, looking for the element */
    for (;;)
    {
        vm_val_t ele;

        /* if we've reached the last element, stop looking */
        if (forward ? idx >= cnt : idx == 0)
            break;

        /* if we're going backwards, move to the next element position */
        if (!forward)
            --idx;

        /* get this element */
        get_element(idx, &ele);

        /* if the element matches the search value, return its index */
        if (ele.equals(vmg_ val))
        {
            /* it matches - see what we're doing */
            if (count_only)
            {
                /* they only want the count */
                ++val_cnt;
            }
            else
            {
                /* this is the one - return its 1-based index */
                retval->set_int(idx + 1);
                
                /* foind it - no need to keep searching */
                break;
            }
        }

        /* if we're going forwards, move to the next element */
        if (forward)
            ++idx;
    }

    /* discard our gc protection (self) and our arguments */
    G_stk->discard(2);

    /* return the count if desired */
    if (count_only)
        retval->set_int(val_cnt);
    
    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - countOf
 */
int CVmObjArray::getp_count_of(VMG_ vm_obj_id_t self, vm_val_t *retval,
                               uint *argc)
{
    /* use our general handler to obtain the count */
    return gen_index_of(vmg_ self, retval, argc, TRUE, TRUE);
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - countWhich
 */
int CVmObjArray::getp_count_which(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                  uint *argc)
{
    /* use our general handler to obtain the count */
    return gen_index_which(vmg_ self, retval, argc, TRUE, TRUE);
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - getUnique 
 */
int CVmObjArray::getp_get_unique(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                 uint *argc)
{
    CVmObjArray *new_arr;
    size_t cnt;
    static CVmNativeCodeDesc desc(0);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* put myself on the stack for GC protection */
    G_interpreter->push_obj(vmg_ self);

    /* 
     *   allocate a new array for the return value - the new array will
     *   never be any larger than the original 
     */
    cnt = get_element_count();
    retval->set_obj(create_for_getp(vmg_ cnt));

    /* push a reference to the new list for gc protection */
    G_stk->push(retval);

    /* get the return value as an array */
    new_arr = (CVmObjArray *)vm_objp(vmg_ retval->val.obj);

    /* start out with the result element count the same as my own */
    new_arr->set_element_count(cnt);

    /* copy my elements to the new array */
    memcpy(new_arr->get_element_ptr(0), get_element_ptr(0),
           calc_alloc_ele(cnt));

    /* uniquify the result */
    new_arr->cons_uniquify(vmg0_);

    /* discard the gc protection */
    G_stk->discard(2);

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   property evaluator - appendUnique 
 */
int CVmObjArray::getp_append_unique(VMG_ vm_obj_id_t self, vm_val_t *retval,
                                    uint *argc)
{
    vm_val_t *val2;
    const char *lst2;
    CVmObjArray *arr2;
    size_t cnt2;
    CVmObjArray *new_arr;
    size_t cnt;
    static CVmNativeCodeDesc desc(1);

    /* check arguments */
    if (get_prop_check_argc(retval, argc, &desc))
        return TRUE;

    /* get the other value, but leave it on the stack */
    val2 = G_stk->get(0);

    /* check the type of the other argument - it must be an array or list */
    if ((lst2 = val2->get_as_list(vmg0_)) != 0)
    {
        /* it's a list, so it's not an array */
        arr2 = 0;

        /* note the length */
        cnt2 = vmb_get_len(lst2);
    }
    else
    {
        /* it's not a list - see if it's an array */
        if (val2->typ == VM_OBJ
            && (vm_objp(vmg_ val2->val.obj)->get_metaclass_reg()
                == metaclass_reg_))
        {
            /* it's an array */
            arr2 = (CVmObjArray *)vm_objp(vmg_ val2->val.obj);

            /* note the number of elements */
            cnt2 = arr2->get_element_count();
        }
        else
        {
            /* it's not an array - we can't deal with other types */
            err_throw(VMERR_BAD_TYPE_BIF);
        }
    }

    /* put myself on the stack for GC protection */
    G_interpreter->push_obj(vmg_ self);

    /* 
     *   allocate a new array for the return value - the new array will
     *   never be larger than the sum of the two input lengths 
     */
    cnt = get_element_count();
    retval->set_obj(create_for_append(vmg_ self, cnt + cnt2));

    /* get the return value as an array */
    new_arr = (CVmObjArray *)vm_objp(vmg_ retval->val.obj);

    /* push a reference to the new object for gc protection */
    G_stk->push(retval);

    /* copy myself into the result, if returning a new object */
    if (new_arr != this)
        memcpy(new_arr->get_element_ptr(0), get_element_ptr(0),
               calc_alloc_ele(cnt));

    /* copy the second list/array into the result */
    if (lst2 != 0)
    {
        /* append the list's contents */
        memcpy(new_arr->get_element_ptr(cnt), lst2 + VMB_LEN,
               calc_alloc_ele(cnt2));
    }
    else
    {
        /* append the array's contents */
        memcpy(new_arr->get_element_ptr(cnt), arr2->get_element_ptr(0),
               calc_alloc_ele(cnt2));
    }

    /* the result element count is initially the sum of the two sources */
    new_arr->set_element_count(cnt + cnt2);

    /* uniquify the result */
    new_arr->uniquify_for_append(vmg_ self);

    /* discard the gc protection and arguments */
    G_stk->discard(3);

    /* handled */
    return TRUE;
}

/* ------------------------------------------------------------------------ */
/*
 *   For construction, eliminate repeated elements of the array, leaving
 *   only the unique elements.  Reduces the size of the array to the size
 *   required to accomodate the unique elements. 
 */
void CVmObjArray::cons_uniquify(VMG0_)
{
    size_t cnt;
    size_t src, dst;

    /* get the number of elements */
    cnt = get_element_count();

    /* loop through the list and look for repeated values */
    for (src = dst = 0 ; src < cnt ; ++src)
    {
        size_t idx;
        vm_val_t src_val;
        int found;

        /* 
         *   look for a copy of this source value already in the output
         *   list 
         */
        get_element(src, &src_val);
        for (idx = 0, found = FALSE ; idx < dst ; ++idx)
        {
            vm_val_t idx_val;

            /* get this value */
            get_element(idx, &idx_val);

            /* if it's equal to the current source value, note it */
            if (src_val.equals(vmg_ &idx_val))
            {
                /* note that we found it */
                found = TRUE;

                /* no need to look any further */
                break;
            }
        }

        /* if we didn't find the value, copy it to the output list */
        if (!found)
        {
            /* 
             *   add it to the output array - since this is a
             *   construction-only method, there's no need to save undo
             *   (if we apply undo, we'll undo the entire construction of
             *   the array, hence there's no need to track changes made
             *   since creation) 
             */
            set_element(dst, &src_val);

            /* count it in the output */
            ++dst;
        }
    }

    /* adjust the size of the result list */
    set_element_count(dst);
}

/* ------------------------------------------------------------------------ */
/*
 *   sorter for array
 */
class CVmQSortArray: public CVmQSortVal
{
public:
    CVmQSortArray()
    {
        arr_ = 0;
        self_ = VM_INVALID_OBJ;
    }

    /* get an element from the array */
    void get_ele(VMG_ size_t idx, vm_val_t *val)
        { arr_->get_element(idx, val); }

    /* store an element */
    void set_ele(VMG_ size_t idx, const vm_val_t *val)
        { arr_->set_element_undo(vmg_ self_, idx, val); }

    /* our array object */
    CVmObjArray *arr_;
    vm_obj_id_t self_;
};

/*
 *   property evaluator - sort 
 */
int CVmObjArray::getp_sort(VMG_ vm_obj_id_t self, vm_val_t *retval,
                           uint *in_argc)
{
    size_t len;
    uint argc = (in_argc == 0 ? 0 : *in_argc);
    CVmQSortArray sorter;    
    static CVmNativeCodeDesc desc(0, 2);

    /* check arguments */
    if (get_prop_check_argc(retval, in_argc, &desc))
        return TRUE;

    /* remember the length of my list */
    len = get_element_count();

    /* set the array object in the sorter */
    sorter.arr_ = this;
    sorter.self_ = self;

    /* if we have an 'descending' flag, note it */
    if (argc >= 1)
        sorter.descending_ = (G_stk->get(0)->typ != VM_NIL);

    /* 
     *   if we have a comparison function, note it, but leave it on the
     *   stack for gc protection 
     */
    if (argc >= 2)
        sorter.compare_fn_ = *G_stk->get(1);

    /* put myself on the stack for GC protection */
    G_interpreter->push_obj(vmg_ self);

    /* sort the array, if we have any elements */
    if (len != 0)
        sorter.sort(vmg_ 0, len - 1);

    /* discard the gc protection and arguments */
    G_stk->discard(1 + argc);

    /* return myself */
    retval->set_obj(self);

    /* handled */
    return TRUE;
}

