/*
	HDWIN.C

	Win32 wrapper for the Hugo Debugger

	Copyright (c) 1995-2006 by Kent Tessman

	This file contains Windows API interfacing; Hugo Debugger interfacing
	is performed by HDMSVC.C.

	NOTE:  The trick is that the debugger (i.e., the non-portable
	core code), Windows, and this Win32 interface layer all have 
	different "windows" that they deal with.  The debugger's are
	internal lists of data, Windows' are actual window handles,
	and the interface layer's are sets of data relating the first
	to the second.

	The debugger uses 'struct window_structure window[n]' where
	n is a non-sequentially numbered VIEW_... constant defined
	in hdinter.h.  Windows uses HWND window handles.  The Win32
	interface layer uses 'debug_childwindow_struct
	debug_childwindow[n]' where n is a sequentially numbered
	window index; it is this array of structures that the
	interface deals with when translating information from the
	debugger core to the Win32 API.

	WhichWindow(VIEW_...) and WhichHWND(HWND) both return window
	indexes; and VIEW_... and HWND values are both available from
	the debug_childwindow structure via the .view and .hwnd
	members.  For example, to access an internal debugger window
	structure starting with a Win32 windows handle (hwnd):

	    window[debug_childwindow[WhichHWND(hwnd)].view].<member>
*/

#define HD_PROGRAM_NAME		"hdwin.exe"

#include <windows.h>
#undef GetProp			// from win.h

#include <zmouse.h>

#include <commctrl.h>

#undef TextOut
#define TextOut(hdc, x, y, str, len)  ExtTextOut(hdc, x, y, 0, NULL, str, len, NULL)

#include "heheader.h"
#include "hdheader.h"
#include "hdwinext.h"

#define APSTUDIO_INVOKED	// to get _APS_NEXT_... values
#include "hdwin.h"
#undef APSTUDIO_INVOKED
#include "hewin.h"


int CreateDebugFont(void);
void InitDebuggerChildWindows(void);
void UpdateDebuggerChildHeight(HWND hwnd);
HWND CreateMDIChild(int i);
void RecordVisibleWindows(void);
void SetScrollBars(HWND hwnd);
void CheckDebugMenuItems(void);
int ViewObjectTree(HWND hwnd, char *t, char allow_move_objects);
char *ViewAllSymbols(HWND hwnd);
void LoadDebuggerRegistry(void);
void SaveDebuggerRegistry(void);

/* from hewin.c */
int WINAPI he_WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
		char *szCmdLine, int iCmdShow);
BOOL CALLBACK AboutDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK SetupDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
void PushKeypress(int k);
extern char scrollback_active;

/* from hdupdate.c */
void UpdateOtherWindow(int v);

/* from hdtools.c */
void SetupColors(void);

/* For Windows: */
LRESULT CALLBACK DebugWndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK DebugChildWndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK HelpWndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);


/* Windows API management info: */
HINSTANCE AppInstance;
HWND wndDebug;
HWND wndStatus;
HWND wndMDIClient;
HWND wndHelpText;
WNDCLASS wndclass;
/* This has been set by hand, so there's a potential for mismatches
   if anything changes: */
#define IDWINDOWMENU 5	// position of "Window" on the menubar
HMENU menuDebug;
HACCEL accelDebug;
HFONT fontDebug = NULL;
LOGFONT lfDebug;

debug_childwindow_struct debug_childwindow[DEBUG_CHILD_WINDOWS];

/* This window index needs updating; if it's -1, nothing does */
int current_windex_update = -1;

char help_text_buffer[MAX_HELP_TEXT];
int help_text_len = 0;

/* from hewin.c: */
extern HWND wndMain;
extern char during_player_input;

/* Registry info: */
#define DEF_DEBUG_FONT_FACE "Courier New"
#define DEF_DEBUG_FONT_SIZE 11
char debug_FixedFont[LF_FACESIZE] = DEF_DEBUG_FONT_FACE;
int debug_FixedPoint = DEF_DEBUG_FONT_SIZE;
/* Main debugger window: */
int debug_FullScreen = false;
int debug_xPos = CW_USEDEFAULT,
	debug_yPos = CW_USEDEFAULT,
	debug_Width = CW_USEDEFAULT,
	debug_Height = CW_USEDEFAULT;


/*
 * Hugo Debugger initialization:
 *
 */

/* WinMain */

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
		char *szCmdLine, int iCmdShow)
{
	static char szAppName[MAXPATH + 32];
	CLIENTCREATESTRUCT ccs;
	int i;


	InitCommonControls();

	/* Get some more global info */
	AppInstance = hInstance;

	sprintf(szAppName, "Hugo Debugger");

	InitDebuggerChildWindows();	/* i.e., just basic info */
	LoadDebuggerRegistry();

	/* The application window class */
	wndclass.style          = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc    = DebugWndProc;
	wndclass.cbClsExtra     = 0;
	wndclass.cbWndExtra     = 0;
	wndclass.hInstance      = hInstance;
	wndclass.hIcon          = LoadIcon(hInstance, MAKEINTRESOURCE(PROGRAM_ICON));
	wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground  = (HBRUSH)NULL;
	wndclass.lpszMenuName   = NULL;
	wndclass.lpszClassName  = szAppName;
	RegisterClass(&wndclass);

	wndDebug = CreateWindowEx(
		0,
		szAppName,		// window class name
		szAppName,              // window caption
		WS_OVERLAPPEDWINDOW |	// window style
			WS_CLIPCHILDREN | WS_CLIPSIBLINGS |
			WS_CAPTION,
		debug_xPos,		// initial x position
		debug_yPos,		// initial y position
		debug_Width,
		debug_Height,
		NULL,                   // parent window handle
		NULL,                   // window menu handle
		hInstance,              // program instance handle
		NULL);                  // creation parameters

	/* Set up main menubar and accelerator keys */
	menuDebug = LoadMenu(AppInstance, MAKEINTRESOURCE(HDWIN_MENUBAR));
	SetMenu(wndDebug, menuDebug);
	accelDebug = LoadAccelerators(AppInstance,
			MAKEINTRESOURCE(HDWIN_MENUBAR_ACCELERATORS));

	/* Get a font handle for the default debugger display font */
	CreateDebugFont();

	/* Create the MDI client window */
        ccs.hWindowMenu = GetSubMenu(menuDebug, IDWINDOWMENU); 
        ccs.idFirstChild = _APS_NEXT_COMMAND_VALUE;

	wndMDIClient = CreateWindowEx(
		0,
		"MDICLIENT",
		NULL,			// window caption
		WS_CHILD | WS_CLIPCHILDREN |
			WS_VSCROLL | WS_HSCROLL, 
		0, 0, 0, 0,
		wndDebug,		// parent window handle
		NULL,			// window menu handle
		hInstance,		// program instance handle
		(LPTSTR)&ccs);
 
        ShowWindow(wndMDIClient, SW_SHOW); 

	/* Now create the all the MDI child windows, reusing and 
	   modifying the wndclass window class
	*/
	wndclass.lpfnWndProc    = DebugChildWndProc;
	wndclass.lpszClassName  = "HDChildWindow";
	wndclass.hbrBackground  = (HBRUSH)GetStockObject(WHITE_BRUSH);
	wndclass.style |= CS_DBLCLKS | CS_PARENTDC;
	RegisterClass(&wndclass);

	/* Note that wndclass must be persistent from this point
	   forward (i.e., unchanged), since it is used repeatedly
	   by CreateMDIChild()
	*/

	/* Create the status window */
	wndStatus = CreateWindowEx(
		0,			// no extended styles
		STATUSCLASSNAME,	// name of status window class
		(LPCTSTR)NULL,		// no text when first created
		SBARS_SIZEGRIP |	// includes a sizing grip
			WS_BORDER | WS_VISIBLE |
			WS_CHILD,	// creates a child window
		0, 0, 0, 0,		// ignores size and position
		wndDebug,		// handle to parent window
		(HMENU)HD_STATUS_WINDOW,// ID from hdwin.h
		hInstance,		// handle to application instance
		NULL);			// no window creation data

	/* Now actually bring the main debugger window up */
	ShowWindow(wndDebug, debug_FullScreen?SW_MAXIMIZE:iCmdShow);
	if (debug_FullScreen)
		SendMessage(wndDebug, WM_MOVE, 0, 0);
	UpdateWindow(wndDebug);

	/* Now that wndDebug is shown and its coordinates are known,
	   open any initially visible child windows
	*/
	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		int v;

		if (debug_childwindow[i].visible)
			CreateMDIChild(i);

		debug_clearview(v = debug_childwindow[i].view);
		window[v].changed = true;
	}

	/* Set the focus to the first-opened window */
	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		if (IsWindow(debug_childwindow[i].hwnd))
		{
			SetFocus(debug_childwindow[i].hwnd);
			break;
		}
	}

	/* Set up SaveDebuggerRegistry() to be called on normal termination */
	atexit(SaveDebuggerRegistry);

	/* Finally, launch the engine */
	return he_WinMain(hInstance, hPrevInstance,
		szCmdLine, iCmdShow);
}


/*
 * Main window callback:
 *
 */

/* DebugWndProc

	DebugWndProc is the callback for the main window, responsible for
	processing all messages related to window updating, resizing,
        movement, etc.  As well, we have to use key messages and
	PushKeypress() to record keypresses so that hugo_getkey() can
	restore them to simulate getch().
*/

LRESULT CALLBACK DebugWndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
		case WM_CREATE:
			return 0;

		case WM_SETFOCUS:
			/* If switching from the game screen */
			if (((HWND)wParam==wndMain || active_screen==GAME)
				&& !scrollback_active)
			{
				int temp_active;

				/* Make sure the Code window is updated */
				if (active_window!=CODE_WINDOW)
				{
					temp_active = active_view;
					active_view = CODE_WINDOW;
					RedrawDebuggerChild(0);
					active_view = temp_active;
				}

				if (!during_player_input)
				{
					debugger_interrupt = true;
					return 0;
				}
				else
				{
					PushKeypress(9);
				}
				break;
			}

		case WM_SIZE:
			/* Have to resize the status bar and MDI
			   client area */
			if (wndStatus) SendMessage(wndStatus, iMsg, wParam, lParam);
			if (wParam!=SIZE_MINIMIZED && wndStatus && wndMDIClient)
			{
				RECT rc, rcClient;

				GetClientRect(hwnd, &rcClient);
				GetWindowRect(wndStatus, &rc);
				ScreenToClient(hwnd, (LPPOINT)&rc.left);
				rcClient.bottom = rc.top;
				MoveWindow(wndMDIClient,
					rcClient.left,
					rcClient.top,
					rcClient.right-rcClient.left,
					rcClient.bottom-rcClient.top,
					TRUE);
			}

			if (wParam==SIZE_MAXIMIZED)
				debug_FullScreen = true;
			else
				debug_FullScreen = false;

			/* Only save values for normal move/size */
			if (wParam==SIZE_MINIMIZED || wParam==SIZE_MAXIMIZED)
				return 0;

			/* Otherwise fall through */

		case WM_MOVE:
		{
			RECT rect;

			GetWindowRect(hwnd, &rect);
			debug_xPos = rect.left;
			debug_yPos = rect.top;
			debug_Width = rect.right - rect.left;
			debug_Height = rect.bottom - rect.top;

			return 0;
		}

		case WM_INITMENU:
			CheckDebugMenuItems();
			return 0;

		case WM_SYSCOMMAND:
			switch (wParam)
			{
				case SC_CLOSE:
					RecordVisibleWindows();
					break;
			}
			break;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{

			/* FILE */
				case HD_FILE_RESTART:
					event.object = MENU_FILE + FILE_RESTART;
ReturnSelect:
					event.action = SELECT;
					return 0;

				case HD_FILE_PRINT:
					MessageBox(wndDebug,
						"Output to printer is currently not available.",
						"Print", MB_ICONEXCLAMATION);
					return 0;

				case HD_FILE_EXIT:
					RecordVisibleWindows();
					if (MessageBox(hwnd,
						"Abandon current debugging session?",
						"Exit Debugger",
						MB_ICONEXCLAMATION | MB_YESNO)==IDYES)
					{
						SaveSetupFile();

						PostQuitMessage(0);
					}
					return 0;

			/* EDIT */
				case HD_EDIT_COPY:
				{
					char *copy;
					int i;
					if (!window[active_window].count || window[active_window].selected==-1)
						break;
					copy = (char *)GlobalAlloc(0, CHILD_TEXT_WIDTH+1);
					if (!copy) return 0;	// GlobalAlloc failed
					// Copy the relevant textgrid line into the copy string
					for (i=0; i<CHILD_TEXT_WIDTH; i++)
					{
						copy[i] = debug_childwindow[active_window].textgrid[i][window[active_window].selected-window[active_window].first].chr;
						copy[i+1] = 0;
					}
					// Right-trim the copy string
					i--;
					while ((i>=0) && (copy[i]==' ' || copy[i]==-1))
						copy[i--] = 0;
					if (!OpenClipboard(hwnd) || !EmptyClipboard() || !SetClipboardData(CF_TEXT, copy))
					{
						GlobalFree(copy);
					}
					CloseClipboard();
					return 0;
				}
				case HD_EDIT_DELETE:
					event.action = DELETE;
					return 0;
				case HD_EDIT_SELECT:
					if (active_window==0)	// Code window
					{
						EnterBreakpoint();
						return 0;
					}
					event.action = SELECT;
					event.object = 0;
					current_windex_update = WhichWindow(active_window);
					return 0;

			/* VIEW */
				case HD_VIEW_CODE:
				case HD_VIEW_WATCH:
				case HD_VIEW_CALLS:
				case HD_VIEW_BREAKPOINTS:
				case HD_VIEW_LOCALS:
				case HD_VIEW_ALIASES:
				case HD_VIEW_HELP:
				{
					int i;
					HWND wnd = NULL;

					for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
					{
						if (debug_childwindow[i].menuitem==(int)wParam)
						{
							wnd = debug_childwindow[i].hwnd;
							break;
						}
					}

					if (i==DEBUG_CHILD_WINDOWS)
						break;

					if (!IsWindow(wnd))
					{
						CreateMDIChild(i);
					}
					else
					{
						SetFocus(debug_childwindow[i].hwnd);
					}

					/* If not the code window, force an update in case
					   something is supposed to be in the window */
					if (i!=0)
					{
						active_window = active_view = debug_childwindow[i].view;
						window[active_view].changed = true;
						UpdateOtherWindow(active_view);
					}

					return 0;
				}

				case HD_VIEW_OUTPUT:
					SetForegroundWindow(wndMain);
					if (IsIconic(wndMain))
						ShowWindow(wndMain, SW_RESTORE);
					return 0;

			/* RUN */
				case HD_RUN_GO:
					event.object = MENU_RUN + RUN_GO;
					goto ReturnSelect;
				case HD_RUN_FINISH:
					event.object = MENU_RUN + RUN_FINISH;
					goto ReturnSelect;
				case HD_RUN_STEP:
					event.object = MENU_RUN + RUN_STEP;
					goto ReturnSelect;
				case HD_RUN_STEPOVER:
					event.object = MENU_RUN + RUN_STEPOVER;
					goto ReturnSelect;
				case HD_RUN_SKIP:
					event.object = MENU_RUN + RUN_SKIP;
					goto ReturnSelect;
				case HD_RUN_STEPBACK:
					event.object = MENU_RUN + RUN_STEPBACK;
					goto ReturnSelect;

			/* DEBUG */
				case HD_DEBUG_SEARCH:
					event.object = MENU_DEBUG + DEBUG_SEARCH;
					goto ReturnSelect;
				case HD_DEBUG_WATCH:
					event.object = MENU_DEBUG + DEBUG_WATCH;
					goto ReturnSelect;
				case HD_DEBUG_SET:
					event.object = MENU_DEBUG + DEBUG_SET;
					goto ReturnSelect;
				case HD_DEBUG_BREAKPOINT:
					event.object = MENU_DEBUG + DEBUG_BREAKPOINT;
					goto ReturnSelect;

				case HD_DEBUG_VIEWLOCALSAS:
					local_view_type = SelectWatchType();
					window[VIEW_LOCALS].changed = true;
					PrintScreenBorders();
					return 0;

				case HD_DEBUG_VIEWSYMBOLS:
					ViewAllSymbols(wndDebug);
					return 0;

				case HD_DEBUG_OBJTREE:
					ViewObjectTree(wndDebug, "Object Tree", 1);
					return 0;

				case HD_DEBUG_MOVEOBJ:
				{
					int obj, parent;

					obj = ViewObjectTree(wndDebug, "Select Object to Move", 0);
					if (obj==-1) return 0;

					sprintf(debug_line, "Select New Parent for \"%s\"", objectname[obj]);
					parent = ViewObjectTree(wndDebug, debug_line, 0);
					if (parent==-1) return 0;

					MoveObj(obj, parent);
					return 0;
				}

				case HD_DEBUG_WARNINGS:
					runtime_warnings = !runtime_warnings;
					return 0;
				case HD_DEBUG_NESTING:
					format_nesting = !format_nesting;
					return 0;

			/* TOOLS */
				case HD_TOOLS_SETUP:
					DialogBox(AppInstance,
						MAKEINTRESOURCE(HE_SETUP_DIALOG),
						wndMain, SetupDialog);
					return 0;
				case HD_TOOLS_SETUPCOLORS:
					SetupColors();
					return 0;
					
			/* WINDOW */
				case HD_WINDOW_TILEHORIZONTALLY:
					SendMessage(wndMDIClient, WM_MDITILE, MDITILE_HORIZONTAL, 0);
					return 0;
				case HD_WINDOW_TILEVERTICALLY:
					SendMessage(wndMDIClient, WM_MDITILE, MDITILE_VERTICAL, 0);
					return 0;
				case HD_WINDOW_CASCADE:
					SendMessage(wndMDIClient, WM_MDICASCADE, 0, 0);
					return 0;
				case HD_WINDOW_ARRANGEICONS:
					SendMessage(wndMDIClient, WM_MDIICONARRANGE, 0, 0);
					return 0;
				case HD_WINDOW_CLOSEALL:
				{
					int i;
					for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
						SendMessage(wndMDIClient, WM_MDIDESTROY, 
							(WPARAM)debug_childwindow[i].hwnd, 0);
					break;
				}

			/* HELP */
				case HD_HELP_CONTEXT:
				case HD_HELP_TOPIC:
					if (!IsWindow(debug_childwindow[WhichWindow(VIEW_HELP)].hwnd))
						SendMessage(hwnd, WM_COMMAND, HD_VIEW_HELP, 0);
					SendMessage(wndMDIClient, WM_MDIACTIVATE, (WPARAM)debug_childwindow[WhichWindow(VIEW_HELP)].hwnd, 0);

					if (LOWORD(wParam)==HD_HELP_CONTEXT)
					{
						SearchHelp(context_help);
						event.action = 0;
						return 0;
					}
					else
					{
						event.object = MENU_HELP + HELP_TOPIC;
						goto ReturnSelect;
					}

				case HD_HELP_ABOUT:
					/* Use the engine's "About" box */
					DialogBox(AppInstance,
						MAKEINTRESOURCE(HD_ABOUT_DIALOG),
					        wndDebug, AboutDialog);
					break;
			}

			/* Break to DefFrameProc() so the Window
			   menu can activate MDI children
			*/
			break;

		case WM_DESTROY:
			DeleteObject(fontDebug);
			PostQuitMessage(0);
			return 0;

	/*
	 * These messages are passed from an MDI child window:
	 */
		case WM_KEYDOWN:
			/* Since it's possible for wndDebug to get
			   the keyboard focus directly, and therefore
			   WM_KEYDOWN messages:
			*/
			if (current_windex_update==-1)
				current_windex_update = WhichWindow(active_window);

			/* Send Help window key messages to wndHelpText */
			if (active_window==VIEW_HELP)
			{
				SendMessage(wndHelpText, iMsg, wParam, lParam);
				return 0;
			}

		/* VK_KEY messages re: selection movement can for the
		   most part be sent as scrollbar messages to the
		   debugger child window
		*/
			switch (wParam)
			{
				case VK_UP:
					event.object = UP;
ReturnMove:
					event.action = MOVE;
					return 0;
				case VK_DOWN:
					event.object = DOWN;
					goto ReturnMove;

				/* Cursor/selection movement handled 
				   by the scrollbar, where applicable
				*/
				case VK_RIGHT:
					SendMessage(debug_childwindow[WhichWindow(active_window)].hwnd, WM_HSCROLL, SB_LINERIGHT, 0);
					return 0;
				case VK_LEFT:
					SendMessage(debug_childwindow[WhichWindow(active_window)].hwnd, WM_HSCROLL, SB_LINELEFT, 0);
					return 0;

				case VK_PRIOR:
					SendMessage(debug_childwindow[WhichWindow(active_window)].hwnd, WM_VSCROLL, SB_PAGEUP, 0);
					return 0;
				case VK_NEXT:
					SendMessage(debug_childwindow[WhichWindow(active_window)].hwnd, WM_VSCROLL, SB_PAGEDOWN, 0);
					return 0;
				
				/* HOME and END */
				case VK_HOME:
				case VK_END:
				{
					/* CTRL+HOME and CTRL+END */
					if (GetKeyState(VK_CONTROL)&0x8000000)
					{
						if (wParam==VK_HOME)
							event.object = START;
						else
							event.object = FINISH;
						goto ReturnMove;
					}

					/* Non-CTRL HOME and END */
					else
					{
						if (wParam==VK_HOME)
							SendMessage(debug_childwindow[WhichWindow(active_window)].hwnd, WM_HSCROLL, SB_PAGELEFT, 0);
						else
							SendMessage(debug_childwindow[WhichWindow(active_window)].hwnd, WM_HSCROLL, SB_PAGERIGHT, 0);
					}

					/* Don't need non-control HOME/END;
					   horizontal movement is handled by
					   scrollbar */
				}
				break;

				case VK_DELETE:
					event.action = DELETE;
					return 0;
			}
			return 0;

		case WM_CHAR:
			switch (wParam)
			{
				/* Tab to Engine/Output by simulating menu choice */
				case VK_TAB:
					SendMessage(hwnd, WM_COMMAND, HD_VIEW_OUTPUT, 0);
					return 0;

				case VK_RETURN:		/* Enter */
					event.object = 0;
					goto ReturnSelect;
			}
			return 0;

		case WM_MOUSEWHEEL:
			// Send this on to where it should be
			SendMessage(debug_childwindow[WhichWindow(active_window)].hwnd, iMsg, wParam, lParam);
			return 0;
		
		/* For mouse button messages, event.object is set
		   by the child window proc, below
		*/
		case WM_LBUTTONDOWN:
			event.action = SINGLECLICK;
			return 0;
		case WM_LBUTTONDBLCLK:
			event.action = DOUBLECLICK;
			return 0;
	}

	return DefFrameProc(hwnd, wndMDIClient, iMsg, wParam, lParam);
}


/*
 * Child window callback:
 *
 */

/* DebugChildWndProc

	The basic callback shared by the debugger child windows.
*/

LRESULT CALLBACK DebugChildWndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
		case WM_CREATE:
			return 0;

		case WM_MDIACTIVATE:
		case WM_SETFOCUS:
		{
			HWND oldhwnd;

			/* If windows and the debugger itself aren't
			   properly initialized yet, skip this
			*/
			if (WhichHWND(hwnd)==-1 || mem==NULL) break;

			/* Otherwise, update any changed windows 
			   (including this one) */
			
			/* Erase the selection highlight if in a different window */
			if (debug_childwindow[WhichHWND(hwnd)].view!=active_window)
				HighlightCurrent(0);
			InvalidateRect(oldhwnd = debug_childwindow[WhichWindow(active_window)].hwnd, NULL, 1);

			window[active_window].changed = true;
			window[active_view].changed = true;
			if (WhichHWND(hwnd)!=-1)
				window[active_window = active_view = debug_childwindow[WhichHWND(hwnd)].view].changed = true;

			HighlightCurrent(1);

			ForceRedraw(oldhwnd);
			ForceRedraw(hwnd);

			break;
		}

		case WM_MOVE:
		case WM_SIZE:
		{
			RECT rect;
			POINT p1, p2;

			/* Don't save coords. for minimized or maximized windows */
			if (IsIconic(hwnd) || IsZoomed(hwnd))
			{
				/* Make sure the window is properly
				   aligned re: its scrollbars
				*/
				if (WhichHWND(hwnd)!=-1)
				{
					SendMessage(hwnd, WM_HSCROLL, -1, 0);
					SendMessage(hwnd, WM_VSCROLL, -1, 0);
				}
				break;
			}

			GetWindowRect(hwnd, &rect);
			p1.x = rect.left;
			p1.y = rect.top;
			p2.x = rect.right;
			p2.y = rect.bottom;
			ScreenToClient(wndDebug, &p1);
			ScreenToClient(wndDebug, &p2);

			/* If debug_childwindow[].hwnd has been initialized */
			if (WhichHWND(hwnd)!=-1)
			{
				debug_childwindow[WhichHWND(hwnd)].xpos = p1.x;
				debug_childwindow[WhichHWND(hwnd)].ypos = p1.y;
				debug_childwindow[WhichHWND(hwnd)].width = p2.x-p1.x;
				debug_childwindow[WhichHWND(hwnd)].height = p2.y-p1.y;

				UpdateDebuggerChildHeight(hwnd);
			}

			/* Resize the Help window's text (edit) box */
			if (WhichHWND(hwnd)==WhichWindow(VIEW_HELP))
			{
				GetClientRect(hwnd, &rect);
				MoveWindow(wndHelpText, 0, 0, rect.right, rect.bottom, TRUE);
				ShowWindow(wndHelpText, SW_SHOW);
			}

			SetScrollBars(hwnd);

			/* Break to DefMDIChildProc() */
			break;
		}

		case WM_PAINT:
		{
			PAINTSTRUCT ps;

			if (WhichHWND(hwnd)==WhichWindow(VIEW_HELP))
				break;

			InvalidateRect(hwnd, NULL, 0);
			BeginPaint(hwnd, &ps);
			if (WhichHWND(hwnd)!=-1)
				RedrawDebuggerChild(WhichHWND(hwnd));
			EndPaint(hwnd, &ps);

			return 0;
		}

		/* A right-button press will bring up the child window
		   popup menu
		*/
		case WM_RBUTTONDOWN:
		{
			POINT rightclick;

			CheckDebugMenuItems();

			rightclick.x = LOWORD(lParam);
			rightclick.y = HIWORD(lParam);
			ClientToScreen(hwnd, &rightclick);
			TrackPopupMenu(GetSubMenu(menuDebug, 1),
				0,		/* horizontal alignment */
				rightclick.x, rightclick.y,
				0, wndDebug, NULL);

			return 0;
		}

		/* Scrollbar processing: */
		case WM_HSCROLL:
		case WM_VSCROLL:
		{
			int nPos;
			struct window_structure *win;
			int width, height;	// in char, of visible window
			HDC hdc;
			HFONT oldFont;
			TEXTMETRIC tm;
			RECT rect;
			int windex;

			win = &window[windex = debug_childwindow[WhichHWND(hwnd)].view];
			nPos = HIWORD(wParam);	// scrollbox position, if needed
			hdc = GetDC(hwnd);
			oldFont = SelectObject(hdc, fontDebug);
			DeleteObject(oldFont);
			GetTextMetrics(hdc, &tm);
			ReleaseDC(hwnd, hdc);
			GetClientRect(hwnd, &rect);
			width = rect.right/tm.tmAveCharWidth;
			height = rect.bottom/tm.tmHeight;

		/* Horizontal */
			if (iMsg==WM_HSCROLL)
			{
				switch (LOWORD(wParam))
				{
					case SB_LINERIGHT:
						debug_childwindow[windex].xoffset++;
						break;
					case SB_LINELEFT:
						debug_childwindow[windex].xoffset--;
						break;
					case SB_PAGERIGHT:
						debug_childwindow[windex].xoffset+=width;
						break;
					case SB_PAGELEFT:
						debug_childwindow[windex].xoffset-=width;
						break;
					case SB_THUMBTRACK:
						debug_childwindow[windex].xoffset = nPos;
						break;
				}

				if (debug_childwindow[windex].xoffset > win->width - width - 2)
					debug_childwindow[windex].xoffset = win->width - width - 2;
				if (debug_childwindow[windex].xoffset < 0)
					debug_childwindow[windex].xoffset = 0;
			}

		/* Vertical */
			else if (iMsg==WM_VSCROLL)
			{
				HighlightCurrent(0);

				switch (LOWORD(wParam))
				{
					/* Use the keystroke equivalents for these: */
					case SB_LINEUP:
						//SendMessage(hwnd, WM_KEYDOWN, VK_UP, 0);
						//return 0;
						win->selected--;
						break;
					case SB_LINEDOWN:
						//SendMessage(hwnd, WM_KEYDOWN, VK_DOWN, 0);
						//return 0;
						win->selected++;
						break;
					
					/* But these are actually called by their
					   keyboard equivalents: */
					case SB_PAGEUP:
						win->selected-=height;
						break;
					case SB_PAGEDOWN:
						win->selected+=height;
						break;

					case SB_THUMBTRACK:
						win->selected = nPos;
						if ((unsigned)nPos >= win->count-height-1)
							nPos = win->count-1;
						break;
				}

				if ((signed)win->selected >= (signed)win->count)
					win->selected = win->count-1;
				if ((signed)win->selected < 0)
					win->selected = 0;
			}

			if (win->selected >= win->first)
				HighlightCurrent(1);

			win->changed = true;
			RedrawDebuggerChild(windex);
		}
		return 0;

		/* Mousewheel processing */
		case WM_MOUSEWHEEL:
		{
			int zDelta;
			UINT i, num;
			//struct window_structure *win;
			
			if (WhichHWND(hwnd)==WhichWindow(VIEW_HELP))
				break;

			zDelta = (short)HIWORD(wParam)/WHEEL_DELTA;
			// Constrain zDelta for more controlled wheel-spinning
			if (zDelta > 0)
				zDelta = 1;
			else
				zDelta = -1;
			if (!SystemParametersInfo(104, 0, &num, 0))	// #define SPI_GETWHEELSCROLLLINES   104
				num = 3;

/* Auto-scroll to top/bottom of window currently commented out:
			win = &window[debug_childwindow[WhichHWND(hwnd)].view];
			if (zDelta>=0)
				win->selected = win->first;
			else
			{
				win->selected = win->first+win->height-1;				
				if (win->selected >= win->count)
					win->selected = win->count-1;
			}
*/
			for (i=0; i<abs(zDelta)*num; i++)
			{
				SendMessage(hwnd, WM_VSCROLL, (zDelta>=0)?SB_LINEUP:SB_LINEDOWN, 0);
			}
			return 0;
		}

		case WM_DESTROY:
			/* Highlight the new selection in whatever
			   other window
			*/
			HighlightCurrent(1);
			return 0;

	/* Pass these messages to the main debug window */
		case WM_LBUTTONDOWN:
		case WM_LBUTTONDBLCLK:
		{
			HFONT oldFont;
			TEXTMETRIC tm;
			HDC dc;

			dc = GetDC(hwnd);
			oldFont = SelectObject(dc, fontDebug);
			DeleteObject(oldFont);
			GetTextMetrics(dc, &tm);
			ReleaseDC(hwnd, dc);
			event.object = window[debug_childwindow[WhichHWND(hwnd)].view].first + HIWORD(lParam)/tm.tmHeight;

			SetForegroundWindow(hwnd);

			/* fall through */
		}
		case WM_CHAR:
		case WM_KEYDOWN:
			if (WhichHWND(hwnd)!=-1)
				current_windex_update = WhichHWND(hwnd);
			SendMessage(wndDebug, iMsg, wParam, lParam);

			return 0;
	}
	return DefMDIChildProc(hwnd, iMsg, wParam, lParam);
}


/*
 * General Win32 debugger functions:
 *
 */

/* CreateDebugFont */

int CreateDebugFont(void)
{
	LOGFONT lf;
	HDC dc;

	if (fontDebug!=NULL) return true;

	lf.lfWidth = lf.lfEscapement = lf.lfOrientation = 0;

	lf.lfWeight = FW_NORMAL;
	lf.lfItalic = 0;
	lf.lfUnderline = 0;
	lf.lfCharSet = DEFAULT_CHARSET;
	lf.lfStrikeOut = 0;
	lf.lfOutPrecision = OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
	lf.lfQuality = DEFAULT_QUALITY;
	lf.lfPitchAndFamily = FIXED_PITCH;
	strcpy(lf.lfFaceName, debug_FixedFont);

	/* Convert point to logical height */
	lf.lfHeight = -MulDiv(debug_FixedPoint,
		GetDeviceCaps(dc = GetDC(wndDebug), LOGPIXELSY), 72);
	ReleaseDC(wndDebug, dc);

	fontDebug = CreateFontIndirect(&lf);
	if (fontDebug==NULL)
	{
		MessageBox(wndMain, "Unable to load font(s)", "Hugo Engine", MB_ICONWARNING);
		return false;
	}
	else
		/* Whatever we requested, figure out what we got */
		GetObject(fontDebug, sizeof(LOGFONT), &lfDebug);

	return true;
}


/* PrintFatalDebuggerError */

void PrintFatalDebuggerError(char *a)
{
	if (a[strlen(a)-1]=='\n') a[strlen(a)-1] = '\0';
	if (a[0]=='\n') a = &a[1];
	MessageBox(wndDebug, a, "Hugo Debugger Fatal Error", MB_ICONSTOP);
}


/* ForceRedraw

	Forces a window to redraw by "moving" it to the same position.
*/

void ForceRedraw(HWND hwnd)
{
	WINDOWPLACEMENT wp;

	wp.length = sizeof(WINDOWPLACEMENT);
	GetWindowPlacement(hwnd, &wp);

	/* Normal show state */
	if (wp.showCmd==SW_SHOW)
	{
		MoveWindow(hwnd,
			wp.rcNormalPosition.left,
			wp.rcNormalPosition.top,
			wp.rcNormalPosition.right-wp.rcNormalPosition.left,
			wp.rcNormalPosition.bottom-wp.rcNormalPosition.top,
			TRUE);
	}
}


/*
 * Win32 child-window management functions:
 *
 */

/* InitDebuggerChildWindows */

void InitDebuggerChildWindows(void)
{
	int i;

	debug_childwindow[0].name = "Code";
	debug_childwindow[0].menuitem = HD_VIEW_CODE;
	debug_childwindow[0].view = CODE_WINDOW;

	debug_childwindow[1].name = "Watch";
	debug_childwindow[1].menuitem = HD_VIEW_WATCH;
	debug_childwindow[1].view = VIEW_WATCH;

	debug_childwindow[2].name = "Calls";
	debug_childwindow[2].menuitem = HD_VIEW_CALLS;
	debug_childwindow[2].view = VIEW_CALLS;

	debug_childwindow[3].name = "Breakpoints";
	debug_childwindow[3].menuitem = HD_VIEW_BREAKPOINTS;
	debug_childwindow[3].view = VIEW_BREAKPOINTS;

	debug_childwindow[4].name = "Local Variables";
	debug_childwindow[4].menuitem = HD_VIEW_LOCALS;
	debug_childwindow[4].view = VIEW_LOCALS;

	debug_childwindow[5].name = "Property/Attribute Aliases";
	debug_childwindow[5].menuitem = HD_VIEW_ALIASES;
	debug_childwindow[5].view = VIEW_ALIASES;

	debug_childwindow[6].name = "Help";
	debug_childwindow[6].menuitem = HD_VIEW_HELP;
	debug_childwindow[6].view = VIEW_HELP;

	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		debug_childwindow[i].height = CW_USEDEFAULT;
		debug_childwindow[i].width = CW_USEDEFAULT;
		debug_childwindow[i].xpos = CW_USEDEFAULT;
		debug_childwindow[i].ypos = CW_USEDEFAULT;
		debug_childwindow[i].xoffset = 0;
		debug_childwindow[i].visible = 0;
	}
}


/* UpdateDebuggerChildHeight */

void UpdateDebuggerChildHeight(HWND hwnd)
{
	HDC hdc;
	HFONT oldFont;
	RECT rect;
	TEXTMETRIC tm;
	struct window_structure *win;

	hdc = GetDC(hwnd);
	oldFont = SelectObject(hdc, fontDebug);
	DeleteObject(oldFont);
	GetTextMetrics(hdc, &tm);
	GetClientRect(hwnd, &rect);

	win = &window[debug_childwindow[WhichHWND(hwnd)].view];
	win->height = (rect.bottom-rect.top-1)/tm.tmHeight;
	if (win->height > CHILD_TEXT_HEIGHT)
		win->height = CHILD_TEXT_HEIGHT;
}


/* CreateMDIChild

	Note that wndclass must leave WinMain() unscathed since
	it gets used repeatedly here.
*/

HWND CreateMDIChild(int i)
{
	HWND hwnd;

	/* Help window */
	if (i==WhichWindow(VIEW_HELP))
	{
		WNDCLASS wc;
		RECT rect;

		if (!GetClassInfo(AppInstance, "HDHelpClass", &wc));
		{
			GetClassInfo(AppInstance, wndclass.lpszClassName, &wc);
			wc.lpszClassName = "HDHelpClass";
			wc.hIcon = LoadIcon(NULL, IDI_QUESTION);
			wc.hbrBackground = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
			RegisterClass(&wc);
		}
		
		hwnd = debug_childwindow[i].hwnd =

			CreateWindowEx(
				WS_EX_MDICHILD,
				wc.lpszClassName,
				debug_childwindow[i].name,
				WS_OVERLAPPEDWINDOW |
					WS_CAPTION,
				debug_childwindow[i].xpos,
				debug_childwindow[i].ypos,
				debug_childwindow[i].width,
				debug_childwindow[i].height,
				wndMDIClient, NULL, AppInstance, NULL);

		/* The help text is a separate child edit control */
		GetClientRect(hwnd, &rect);
		wndHelpText = CreateWindowEx(
			0, "EDIT", help_text_buffer,
			WS_CHILD |
				WS_VISIBLE | ES_LEFT |
				ES_READONLY | ES_MULTILINE |
				WS_HSCROLL | WS_VSCROLL |
				WS_DLGFRAME,
			0, 0, rect.right, rect.bottom,
			hwnd, NULL, AppInstance, NULL);

		SendMessage(wndHelpText, WM_SETFONT, (WPARAM)GetStockObject(ANSI_FIXED_FONT), 0);
		SetWindowText(wndHelpText, help_text_buffer);
	}

	/* All other windows */
	else
	{

		hwnd = debug_childwindow[i].hwnd =

			CreateWindowEx(
				WS_EX_MDICHILD,
				wndclass.lpszClassName,
				debug_childwindow[i].name,
				WS_OVERLAPPEDWINDOW |
					WS_CAPTION | WS_CLIPSIBLINGS |
					WS_VSCROLL | WS_HSCROLL |
					WS_CHILD,
				debug_childwindow[i].xpos,
				debug_childwindow[i].ypos,
				debug_childwindow[i].width,
				debug_childwindow[i].height,
				wndMDIClient, NULL, AppInstance, NULL);

		if (hwnd)
		{
			UpdateDebuggerChildHeight(hwnd);
		}
	}

	if (hwnd==NULL)
	{
		MessageBox(NULL,
"Unable to create window.  (Exiting this application is recommended.)",
			"Hugo Debugger",
			MB_ICONEXCLAMATION);
		return NULL;
	}

	/* Correct the highlight, setting focus to this window */
	HighlightCurrent(0);
	SendMessage(hwnd, WM_SETFOCUS, (WPARAM)debug_childwindow[WhichWindow(active_window)].hwnd, 0);

	active_window = active_view = debug_childwindow[i].view;
	SetWindowTitle(i);

	return hwnd;
}


/* RedrawDebuggerChild

	Checks to make sure the selection bar is onscreen (if
	applicable), then redraws the window as per the textgrid.
*/

void RedrawDebuggerChild(int windex)
{
	HWND hwnd;
	HDC hdc;
	HFONT oldFont;
	TEXTMETRIC tm;
	RECT rect;
	int x, y;
	char a[CHILD_TEXT_WIDTH];
	int alen, ax;
	struct window_structure *win;
	int height;

	if (windex==WhichWindow(VIEW_HELP)) return;

	win = &window[debug_childwindow[windex].view];

	hwnd = debug_childwindow[windex].hwnd;
	hdc = GetDC(hwnd);
	oldFont = SelectObject(hdc, fontDebug);
	DeleteObject(oldFont);
	GetTextMetrics(hdc, &tm);
	GetClientRect(hwnd, &rect);

	currently_updating = debug_childwindow[windex].view;

	/* Check to make sure the selection bar is onscreen.  We
	   can set active_view before calling this because, although
	   normally the interface layer doesn't watch active_view,
	   it may be necessary to update the Code window without
	   making it the active_window (to avoid highlighting).
	*/
	if ((windex==WhichWindow(active_window) || windex==WhichWindow(active_view))
		&& win->count && win->first!=win->selected)
	{
		height = rect.bottom/tm.tmHeight;
		if ((signed)win->selected >= (signed)win->first+height)
		{
			if ((signed)win->first < 0)
				win->first = 0;
			win->first = (signed)win->selected-height+1;
			if (active_window==CODE_WINDOW || active_view==CODE_WINDOW)
				UpdateFullCodeWindow(win->first, win->horiz, win->width);
			else
				UpdateOtherWindow(active_window);
		}
		if (win->selected < win->first)
		{
			win->first = win->selected;
			if (active_window==0)
				UpdateFullCodeWindow(win->first, win->horiz, win->width);
			else
				UpdateOtherWindow(active_window);
		}
	}

	/* Print this child window's text grid */
	for (y=0; y<CHILD_TEXT_HEIGHT; y++)
	{
		COLORREF last_fcolor = 0;
		COLORREF last_bgcolor = 0;
		SetTextColor(hdc, last_fcolor);
		SetTextColor(hdc, last_bgcolor);

		strcpy(a, "");
		alen = 0;
		ax = 0;
		
		for (x=0; x<CHILD_TEXT_WIDTH; x++)
		{
			int offset;

			offset = debug_childwindow[windex].xoffset;

			if ((COLORREF)debug_childwindow[windex].textgrid[x + offset][y].fcolor != last_fcolor)
			{
				if (alen)
				{
					TextOut(hdc, ax * tm.tmAveCharWidth, y * tm.tmHeight, a, alen);
					alen = 0;
					ax = x;
				}

				SetTextColor(hdc, (COLORREF)debug_childwindow[windex].textgrid[x + offset][y].fcolor);
				last_fcolor = (COLORREF)debug_childwindow[windex].textgrid[x + offset][y].fcolor;
			}
			if ((COLORREF)debug_childwindow[windex].textgrid[x + offset][y].bgcolor != last_bgcolor)
			{
				if (alen)
				{
					TextOut(hdc, ax * tm.tmAveCharWidth, y * tm.tmHeight, a, alen);
					alen = 0;
					ax = x;
				}

				SetBkColor(hdc, (COLORREF)debug_childwindow[windex].textgrid[x + offset][y].bgcolor);
				last_bgcolor = (COLORREF)debug_childwindow[windex].textgrid[x + offset][y].bgcolor;
			}

			a[alen] = debug_childwindow[windex].textgrid[x + offset][y].chr;
			if (a[alen]==0)
				a[alen] = ' ';
			alen++;

			if (x*tm.tmAveCharWidth > rect.right)
				break;
		}

		if (alen)
			TextOut(hdc, ax * tm.tmAveCharWidth, y * tm.tmHeight, a, alen);

		if (y*tm.tmHeight > rect.bottom || (unsigned)y>=window[debug_childwindow[windex].view].count)
			break;
	}

	ReleaseDC(hwnd, hdc);

	/* In case the debugger added info to the window which changes
	   the scrolling parameters
	*/
	SetScrollBars(hwnd);
}


/* RecordVisibleWindows */

void RecordVisibleWindows(void)
{
	int i;

	/* Check the visibility flag of each child window */
	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		if (IsWindow(debug_childwindow[i].hwnd))
			debug_childwindow[i].visible = true;
		else
			debug_childwindow[i].visible = false;
	}
}


/* SetScrollBars */

void SetScrollBars(HWND hwnd)
{
	RECT rect;
	HDC dc;
	HFONT oldFont;
	TEXTMETRIC tm;
	SCROLLINFO si;
	struct window_structure *win;
	int width, height;
	int windex;

	if ((windex = WhichHWND(hwnd))==-1) return;

	win = &window[debug_childwindow[WhichHWND(hwnd)].view];

	/* Get the size of the window in characters */
	GetClientRect(hwnd, &rect);
	dc = GetDC(hwnd);
	oldFont = SelectObject(dc, fontDebug);
	DeleteObject(oldFont);
	GetTextMetrics(dc, &tm);
	ReleaseDC(hwnd, dc);

	width = (rect.right-rect.left)/tm.tmAveCharWidth + 2;
	height = (rect.bottom-rect.top)/tm.tmHeight;
	
	si.cbSize = sizeof(SCROLLINFO);
	/* Set pos, page size, and range; disable if no scroll necessary */
	si.fMask = SIF_ALL | SIF_DISABLENOSCROLL;

	/* Vertical based on win->selected: */

	si.nMin = 0;
	si.nMax = win->count-2+height;
	if (si.nMax<0 || si.nMax<height) si.nMax = 0;	// disable
	if (win->count-1 < (unsigned)height) si.nMax = 0;
	si.nPage = height;
	si.nPos = win->selected;
	/* Set and redraw */
	SetScrollInfo(hwnd, SB_VERT, &si, TRUE);

	/* Horizontal based on debug_childwindow[windex].xoffset: */

	si.nMin = 0;
	si.nMax = win->width-1;
	if (si.nMax < 0) si.nMax = 0;			// disable
	if (win->count==0) si.nMax = 0;
	si.nPage = width;
	si.nPos = debug_childwindow[windex].xoffset;
	/* Set and redraw */
	SetScrollInfo(hwnd, SB_HORZ, &si, TRUE);
}


/* CheckDebugMenuItems */

void CheckDebugMenuItems(void)
{
	CheckMenuItem(menuDebug, HD_DEBUG_NESTING,
		format_nesting ? MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem(menuDebug, HD_DEBUG_WARNINGS,
		runtime_warnings ? MF_CHECKED:MF_UNCHECKED);

	if (window[active_window].count && window[active_window].selected!=-1)
	{
		EnableMenuItem(menuDebug, HD_EDIT_COPY,   MF_ENABLED);
		EnableMenuItem(menuDebug, HD_EDIT_DELETE, MF_ENABLED);
		EnableMenuItem(menuDebug, HD_EDIT_SELECT, MF_ENABLED);
	}
	/* Delete is only valid for Watch and Breakpoint windows */
	if ((active_window!=VIEW_WATCH && active_window!=VIEW_BREAKPOINTS))
		EnableMenuItem(menuDebug, HD_EDIT_DELETE, MF_GRAYED);
	else
		EnableMenuItem(menuDebug, HD_EDIT_DELETE, MF_ENABLED);
	/* Select is only valid for Code and Watch windows */
	if (active_window!=CODE_WINDOW && active_window!=VIEW_WATCH)
		EnableMenuItem(menuDebug, HD_EDIT_SELECT, MF_GRAYED);
	else
		EnableMenuItem(menuDebug, HD_EDIT_SELECT, MF_ENABLED);
	if (!window[active_window].count || window[active_window].selected==-1)
	{
		EnableMenuItem(menuDebug, HD_EDIT_COPY,   MF_GRAYED);
		EnableMenuItem(menuDebug, HD_EDIT_DELETE, MF_GRAYED);
		EnableMenuItem(menuDebug, HD_EDIT_SELECT, MF_GRAYED);
	}

}


/* SetWindowTitle */

void SetWindowTitle(int windex)
{
	static char code_window_title[72];  // Maximum "Code:  obj.prop" = 
					    // 8 + 32 + 32 + 2 = 72
	static char local_window_title[96]; // "Local variables: ..."
	char *wintitle;

	if (windex==0)
		wintitle = &code_window_title[0];
	else if (windex==WhichWindow(VIEW_LOCALS))
		wintitle = &local_window_title[0];
	else
		return;

	if (RoutineName(currentroutine)[0]=='<')
		sprintf(wintitle, debug_childwindow[windex].name);
	else
		sprintf(wintitle, "%s: %s", debug_childwindow[windex].name, RoutineName(currentroutine));
	SetWindowText(debug_childwindow[windex].hwnd, wintitle);
}


/* WhichWindow

	Returns a debug_childwindow index based on an active_window
	value (i.e., VIEW(something) from hdinter.h).
*/

int WhichWindow(int win)
{
	switch (win)
	{
		case CODE_WINDOW:	return 0;
		case VIEW_WATCH:	return 1;
		case VIEW_CALLS:	return 2;
		case VIEW_BREAKPOINTS:	return 3;
		case VIEW_LOCALS:	return 4;
		case VIEW_ALIASES:	return 5;
		case VIEW_HELP:		return 6;
	}
	return -1;
}


/* WhichHWND

	Returns a debug_childwindow index based on the supplied
	hwnd handle.
*/

int WhichHWND(HWND hwnd)
{
	int i;

	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		if (hwnd==debug_childwindow[i].hwnd)
			return i;
	}
	return -1;
}


/*
 * Tree-view functions:
 *
 */

HTREEITEM AddTreeItem(char *name, HTREEITEM parent, HTREEITEM after, int state, int image);
HWND hwndTree;	// the tree-view control
char *tree_caption;
void (*tree_init)(void);
char *tree_selection;
char moving_objects;


/* TreeDialog

	Callback for the tree-view dialog.
*/

BOOL CALLBACK TreeDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_CREATE:

		case WM_INITDIALOG:
		{
			hwndTree = GetDlgItem(hwndDlg, HD_TREEVIEW_TREE);
			SetWindowText(hwndDlg, tree_caption);

			if (!moving_objects)
			{
				ShowWindow(GetDlgItem(hwndDlg, HD_TREEVIEW_MOVE), SW_HIDE);
			}
  	
			/* Call the tree_init function */
			(tree_init)();

			SetFocus(GetDlgItem(hwndDlg, HD_TREEVIEW_TREE));
			return 0;       /* since focus has been set */
		}

		case WM_NOTIFY:
			/* Get the new selection--using about the
			   stupidest method I've ever encountered:
			*/
			if (wParam==HD_TREEVIEW_TREE)
			{
				NMHDR *nmhdr;

				nmhdr = (NMHDR *)lParam;
				if (nmhdr->code==TVN_SELCHANGED)
				{
					NM_TREEVIEW *nm;
					TV_ITEM *tvi;

					nm = (NM_TREEVIEW *)lParam;
					tvi = &nm->itemNew;
					tree_selection = (char *)tvi->lParam;
				}

//				else if (nmhdr->code==NM_RCLICK)
//				{}

				return 0;
			}
			break;

		case WM_COMMAND:
			switch LOWORD(wParam)
			{
				case IDOK:
					EndDialog(hwndDlg, 1);
					break;

				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					break;

				case HD_TREEVIEW_MOVE:
				{
					int i, obj = -1, parent = -1;

					/* Find out the currently selected object */
					for (i=0; i<objects; i++)
					{
						if (!strcmp(objectname[i], tree_selection))
							obj = i;
					}
					if (obj==-1) return 0;

					sprintf(debug_line, "Select New Parent For \"%s\"", objectname[obj]);
					parent = ViewObjectTree(hwndDlg, debug_line, 0);
					if (parent==-1)
					{
						RedrawWindow(hwndDlg, NULL, NULL, RDW_INVALIDATE | RDW_UPDATENOW);
						return 0;
					}

					MoveObj(obj, parent);
					moving_objects = -1;
					EndDialog(hwndDlg, 1);
					return 0;
				}
			}
			return 0;
	}

	return 0;
}


/* ViewObjectTree

	Returns the object number of the selected object, or -1
	if cancelled.
*/

void AddObjectsUnder(int obj, HTREEITEM parent)
{
	HTREEITEM this_obj;
	int i;

	this_obj = AddTreeItem(objectname[obj], parent, TVI_LAST, 0, 0);

	for (i=Child(obj); i; i=Sibling(i))
	{
		AddObjectsUnder(i, this_obj);
	}

	TreeView_Expand(hwndTree, this_obj, TVE_EXPAND);
}


void AddAllObjectstoTree(void)
{
	HTREEITEM nothing;
	int i;

	if (objects==0) return;
	
	nothing = AddTreeItem(objectname[0], NULL, TVI_FIRST, 0, 0);

	for (i=1; i<objects; i++)
	{
		if (Parent(i)==0) AddObjectsUnder(i, nothing);
	}

	TreeView_Expand(hwndTree, nothing, TVE_EXPAND);
}

int ViewObjectTree(HWND hwnd, char *t, char allow_move_objects)
{
RefreshObjectTree:

	/* allow_move_objects is false for the explicit Move Object
	   dialog call(s); it is true for the simple Object Tree
	   dialog
	*/
	moving_objects = allow_move_objects;

	tree_caption = t;
	tree_init = AddAllObjectstoTree;

	if (DialogBox(AppInstance, MAKEINTRESOURCE(HD_TREEVIEW), 
		hwnd, TreeDialog))
	{
		int i;

		/* If an object has been moved: */
		if (moving_objects==-1) goto RefreshObjectTree;

		for (i=0; i<objects; i++)
			if (!strcmp(objectname[i], tree_selection))
				return i;
		return -1;
	}
	else
		return -1;
}


/* ViewAllSymbols

	Returns the string name of the selected symbol, or NULL
	if cancelled.
*/

void AddAllSymbolstoTree(void)
{
	HTREEITEM ht;
	int i;

 	ht = AddTreeItem("Objects ", NULL, TVI_FIRST, TVIS_BOLD, 0);
	for (i=0; i<objects; i++)
		AddTreeItem(objectname[i], ht, TVI_LAST, 0, 0);
	TreeView_SortChildren(hwndTree, ht, 0);

	ht = AddTreeItem("Properties ", NULL, TVI_FIRST, TVIS_BOLD, 0);
	for (i=0; i<properties; i++)
		AddTreeItem(propertyname[i], ht, TVI_LAST, 0, 0);
	TreeView_SortChildren(hwndTree, ht, 0);

	ht = AddTreeItem("Attributes ", NULL, TVI_FIRST, TVIS_BOLD, 0);
	for (i=0; i<attributes; i++)
		AddTreeItem(attributename[i], ht, TVI_LAST, 0, 0);
	TreeView_SortChildren(hwndTree, ht, 0);

	ht = AddTreeItem("Routines ", NULL, TVI_FIRST, TVIS_BOLD, 0);
	for (i=0; i<routines; i++)
		AddTreeItem(routinename[i], ht, TVI_LAST, 0, 0);
	TreeView_SortChildren(hwndTree, ht, 0);

	ht = AddTreeItem("Global Variables ", NULL, TVI_FIRST, TVIS_BOLD, 0);
	for (i=0; i<globals; i++)
		AddTreeItem(globalname[i], ht, TVI_LAST, 0, 0);
	TreeView_SortChildren(hwndTree, ht, 0);

	ht = AddTreeItem("Arrays ", NULL, TVI_FIRST, TVIS_BOLD, 0);
	for (i=0; i<arrays; i++)
		AddTreeItem(arrayname[i], ht, TVI_LAST, 0, 0);
	TreeView_SortChildren(hwndTree, ht, 0);

	/* Sort the root items */
	TreeView_SortChildren(hwndTree, NULL, 0);
}

char *ViewAllSymbols(HWND hwnd)
{
	tree_caption = "All Public Symbols";
	tree_init = AddAllSymbolstoTree;

	if (DialogBox(AppInstance, MAKEINTRESOURCE(HD_TREEVIEW), 
		hwnd, TreeDialog))
	{
		/* If it ends in a ' ', it's one of the headers */
		if (tree_selection[strlen(tree_selection)-1]==' ')
			return NULL;
		else
			return tree_selection;
	}
	else
		return NULL;
}


/* AddTreeItem

	The adding function called by ViewObjectTree or
	ViewAllSymbols.  The 'image' parameter, specifying an 
	image list index, is not currently used.
*/

HTREEITEM AddTreeItem(char *name, HTREEITEM parent, HTREEITEM after, int state, int image)
{
	static TV_ITEM tvi;
	TV_INSERTSTRUCT tvis;

	tvi.mask = TVIF_TEXT | TVIF_STATE | TVIF_PARAM;
	tvi.state = state;
	tvi.stateMask = state;
	tvi.iImage = image-1;
	tvi.iSelectedImage = image;
	tvi.pszText = name;
	tvi.cchTextMax = strlen(name);
	tvi.lParam = (long)name;

	tvis.item = tvi;
	tvis.hParent = parent;
	tvis.hInsertAfter = after;

	return TreeView_InsertItem(hwndTree, &tvis);
}


/*
 * Win32 registry management functions:
 *
 */

/* LoadDebuggerRegistry */

void LoadDebuggerRegistry(void)
{
	HKEY hkey;
	DWORD type;
	int i;
	size_t size;

	if (RegOpenKey(HKEY_CURRENT_USER,
		"Software\\General Coffee Co.\\Hugo\\Debugger\\Debugger",
		&hkey)==ERROR_SUCCESS)
	{
		/* REG_DWORD values */
		type = REG_DWORD;
		size = sizeof(int);
		RegQueryValueEx(hkey, "FixedPoint", 0, &type, (LPBYTE)&debug_FixedPoint, &size);
		RegQueryValueEx(hkey, "FullScreen", 0, &type, (LPBYTE)&debug_FullScreen, &size);
		RegQueryValueEx(hkey, "xPos", 0, &type, (LPBYTE)&debug_xPos, &size);
		RegQueryValueEx(hkey, "yPos", 0, &type, (LPBYTE)&debug_yPos, &size);
		RegQueryValueEx(hkey, "Width", 0, &type, (LPBYTE)&debug_Width, &size);
		RegQueryValueEx(hkey, "Height", 0, &type, (LPBYTE)&debug_Height, &size);

		/* REG_SZ values */
		type = REG_SZ;
		size = sizeof(int);
		RegQueryValueEx(hkey, "FixedFont", 0, &type, (LPBYTE)&debug_FixedFont, &size);

		/* REG_BIN values */
		type = REG_BINARY;
		size = sizeof(int);
		RegQueryValueEx(hkey, "UserColors", 0, &type, (LPBYTE)&color, &size);

		RegCloseKey(hkey);
	}

	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		struct dbc
		{
			int xpos, ypos, width, height;
			char visible;
		} dbc;

		type = REG_BINARY;
		size = sizeof(dbc);

		if (RegOpenKey(HKEY_CURRENT_USER, 
			"Software\\General Coffee Co.\\Hugo\\Debugger\\Windows",
			&hkey)==ERROR_SUCCESS)
		{
			RegQueryValueEx(hkey, debug_childwindow[i].name, 0, &type, (LPBYTE)&dbc, &size);

			debug_childwindow[i].xpos = dbc.xpos;
			debug_childwindow[i].ypos = dbc.ypos;
			debug_childwindow[i].width = dbc.width;
			debug_childwindow[i].height = dbc.height;
			debug_childwindow[i].visible = dbc.visible;

			RegCloseKey(hkey);
		}
	}
}


/* SaveDebuggerRegistry */

void SaveDebuggerRegistry(void)
{
	int i;
	HKEY hkey;

	if (RegCreateKey(HKEY_CURRENT_USER,
		"Software\\General Coffee Co.\\Hugo\\Debugger\\Debugger",
		&hkey)==ERROR_SUCCESS)
	{
		RegSetValueEx(hkey, "FixedFont", 0, REG_SZ,
			(LPBYTE)&debug_FixedFont, strlen(debug_FixedFont));
		RegSetValueEx(hkey, "FixedPoint", 0, REG_DWORD,
			(LPBYTE)&debug_FixedPoint, sizeof(debug_FixedPoint));

		RegSetValueEx(hkey, "FullScreen", 0, REG_DWORD,
			(LPBYTE)&debug_FullScreen, sizeof(debug_FullScreen));
		RegSetValueEx(hkey, "xPos", 0, REG_DWORD,
			(LPBYTE)&debug_xPos, sizeof(debug_xPos));
		RegSetValueEx(hkey, "yPos", 0, REG_DWORD,
			(LPBYTE)&debug_yPos, sizeof(debug_yPos));
		RegSetValueEx(hkey, "Width", 0, REG_DWORD,
			(LPBYTE)&debug_Width, sizeof(debug_Width));
		RegSetValueEx(hkey, "Height", 0, REG_DWORD,
			(LPBYTE)&debug_Height, sizeof(debug_Height));

		RegSetValueEx(hkey, "UserColors", 0, REG_BINARY,
			(LPBYTE)&color, 17*sizeof(int));

		RegCloseKey(hkey);
	}

	for (i=0; i<DEBUG_CHILD_WINDOWS; i++)
	{
		struct dbc
		{
			int xpos, ypos, width, height;
			char visible;
		} dbc;

		if (RegCreateKey(HKEY_CURRENT_USER, 
			"Software\\General Coffee Co.\\Hugo\\Debugger\\Windows",
			&hkey)==ERROR_SUCCESS)
		{
			dbc.xpos = debug_childwindow[i].xpos;
			dbc.ypos = debug_childwindow[i].ypos;
			dbc.width = debug_childwindow[i].width;
			dbc.height = debug_childwindow[i].height;
			dbc.visible = debug_childwindow[i].visible;

			RegSetValueEx(hkey, debug_childwindow[i].name,
				0, REG_BINARY, (LPBYTE)&dbc, sizeof(dbc));

			RegCloseKey(hkey);
		}
	}
}
