!-------------------------------------------------------------------------------
!  dynmem.h - an Inform library for dynamic memory use
!  Copyright (C) 1999 by John Cater
!  Send comments or suggestions to: katre@rice.edu
!-------------------------------------------------------------------------------
#Ifndef dynmem_h;
Constant dynmem_h;
Message "[Including <dynmem>]";
System_file;

!! Please read the README file that should have been included with this
!! library for directions on its use.

#ifdef dynmem_h;  ! remove "Constant declared but not used" warnings
#endif;

#Include "debuglib.h";

#Ifdef TARGET_GLULX;
Constant ARRAY_GROWTH 256;
#Endif;

! Constant definitions
#Ifndef NULL;
Constant NULL -1;
#Endif;

! Header size constants
Constant _dynmem_Offset_VL_Status	0;
Constant _dynmem_Offset_VL_Magic	1;
Constant _dynmem_Offset_VL_Length	2;
Constant _dynmem_Offset_VL_Data		2 + WORDSIZE;
Constant _dynmem_Header_VL_length	2 + WORDSIZE;

Constant _dynmem_Offset_FL_Status	0;
Constant _dynmem_Offset_FL_Magic	1;
Constant _dynmem_Offset_FL_Length	0;
Constant _dynmem_Offset_FL_Data		2;
Constant _dynmem_Header_FL_length	2;

Constant _dynmem_End_Header_length	2;
Constant _dynmem_End_Block_length	2 + WORDSIZE;

! Status Constants
Constant _dynmem_Status_Unorg		0;
Constant _dynmem_Status_Ready		1;
Constant _dynmem_Status_InUse		2;
Constant _dynmem_Status_Freed		3;
Constant _dynmem_Status_End		4;

Constant __DYNMEM_MIN_BLOCKS		10;

! Global variables
#ifndef DYNMEM_SIZE;
Constant DYNMEM_SIZE 5120; ! 5 * 1024
#Endif;
Array  __dynmem_base_array -> DYNMEM_SIZE;
Global __dynmem_base_array_init = false;

Class MemController
  with
    ! Variables first
    block_size 0,	! The length of blocks to be allocated, or 0 if they
			!  will be variable
    gc_percent 75,	! The percentage of memory use to be reached before
			!  freed memory is recollected
    mem_array NULL,	! The array of memory to use
    array_size 0,	! The size of the array
  private
    ! Variables first
    initialized false,	! Wether the object has been initialized
    bytes_consumed 0,	! The number of bytes in use or freed and not
			!  collected or used as headers
    next_available_array NULL,	! The address of the array of the next block
    next_available_offset 0,	! The offset of the next block of memory
    
    magic_number 0,		! A magic number to identify the object

    header_offset_status	0,	! header information
    header_offset_magic		0,
    header_offset_length	0,
    header_offset_data		0,
    header_length		0,
    end_block_length		0,
    end_header_length		0,

  with
    ! Now, the functions
    alloc_ptr		! A function to allocate memory
    [ length tmp_ptr;
      Print_Trace (self, "Allocating memory.");

      if (~~(self.check_init())) rfalse;
      self.check_collect();
      
      if (self.block_size == 0 && length == 0)
      {
	Print_Warn (self, "Attempting to allocate a block with 0 length.");
	return NULL;
      }

      if (self.block_size == 0)
      {
	 tmp_ptr = self.__alloc_var (length);
      } else ! fixed length
      {
	 tmp_ptr = self.__alloc_fixed ();
      }

      ! Increase bytes_consumed by the amount allocated
      if (self.block_size == 0)
	self.bytes_consumed = self.bytes_consumed + length;
      else
	self.bytes_consumed = self.bytes_consumed + self.block_size;

      ! return block
      if (tmp_ptr == NULL) return NULL;

      return tmp_ptr + self.header_length;
    ],
    realloc_ptr		! A function to reallocate memory
    [ptr new_length old_length real_ptr tmp_ptr extra_length length_needed i;
      Print_Trace (self, "Reallocating memory.");

      if (~~(self.check_init())) rfalse;
      self.check_collect();
      if (~~(self.check_ptr(ptr)))
      {
	Print_Warn (self, "Attempting to reallocate a foriegn pointer.");
	return NULL;
      }
      
      if (ptr == 0 or NULL)
      {
	 tmp_ptr = self.alloc_ptr (new_length);
	 return tmp_ptr;
      }
      if (new_length == 0)
      {
	 self.free_ptr (ptr);
	 return NULL;
      }

      if (self.block_size ~= 0)
      {
	 Print_Warn (self, "Attempting to reallocate a fixed size block.");
	 return NULL;
      }

      ! Find the start of the block
      real_ptr = ptr - self.header_length;
      old_length = (real_ptr + self.header_offset_length)-->0;
      
      ! If new_length is equal to block, return
      if (new_length == old_length)
	 return ptr;

      ! If new_length is smaller than block, shrink it.
      if (new_length < (old_length - self.header_length))
      {
	 (real_ptr + self.header_offset_length)-->0 = new_length +
		 self.header_length;
	 tmp_ptr = real_ptr + new_length + self.header_length;
	 (tmp_ptr + self.header_offset_status)->0 = _dynmem_Status_Freed;
	 (tmp_ptr + self.header_offset_magic)->0 = self.magic_number;
	 (tmp_ptr + self.header_offset_length)-->0 = old_length - new_length -
		 self.header_length;
	 
	 return ptr;
      }
      
      ! If new_length is larger than block, check for space beyond the end.
      if (new_length > (old_length - self.header_length))
      {
	 ! Check next block
	 if ((real_ptr + old_length + self.header_offset_status)->0 ==
			 _dynmem_Status_Ready)
	 {
	    ! Check block's length
	    extra_length = (real_ptr + old_length +
			    self.header_offset_length)-->0;
	    length_needed = new_length - old_length + 2 * self.header_length;
	    if (extra_length > length_needed)
	    {
	       ! Expand into new space, leaving rest as ready, initializing
	       !       over old header
	       (real_ptr + self.header_offset_length)-->0 = new_length +
	       		self.header_length;
	       for (i = 0: i < self.header_length: i++)
	          (real_ptr + old_length)->i = 0;
	       tmp_ptr = real_ptr + new_length + self.header_length;
	       (tmp_ptr + self.header_offset_status)->0 = _dynmem_Status_Ready;
	       (tmp_ptr + self.header_offset_magic)->0 = self.magic_number;
	       (tmp_ptr + self.header_offset_length)-->0 = extra_length -
		       length_needed + self.header_length;
      
	       ! Set bytes_consumed and next_available
	       self.bytes_consumed = self.bytes_consumed + length_needed;
	       self.next_available_offset = tmp_ptr;
	       return ptr;
	    }
	    ! Not enough length
	 }
	 ! Next block not right
	 ! Allocate new block of correct size, copy data, free old block
	 tmp_ptr = self.alloc_ptr (new_length);
	 if (tmp_ptr == NULL)
	 {
	    ! alloc failed
            Print_Warn (self, "realloc_ptr: alloc failed");
	    return NULL;
	 }

	 for (i = self.header_length: i < old_length: i++)
	    tmp_ptr->i = ptr->i;
	 self.free_ptr (ptr);

	 return tmp_ptr;
      }
      
      return ptr;
    ],
    free_ptr		! A function to free memory
    [ptr real_ptr;
      Print_Trace (self, "Freeing memory.");

      if (ptr == NULL)
      {
	 Print_Warn (self, "Attempting to free a NULL pointer.");
	 rfalse;
      }

      if (~~(self.check_init())) rfalse;
      if (~~(self.check_ptr(ptr)))
      {
	 Print_Warn (self, "Attempting to free a foreign pointer.");
	 rfalse;
      }

      ! Find the start of the block
      real_ptr = ptr - self.header_length;
      
      ! Set its status to freed
      (real_ptr + self.header_offset_status)->0 = _dynmem_Status_Freed;
      (real_ptr + self.header_offset_magic)->0 = self.magic_number;

      self.check_collect();
      
      rtrue;
    ],
    force_collect	! A function to force memory collection
    [;
      Print_Trace (self, "Dynmem: Forcing garbage collection.");
      return self.collect();
    ],
    profile		! A function to profile memory usage
    [;
      if (~~(debugging_level >= DEBUG_CRITICAL)) rfalse;

      Print_Trace (self, "Profiling memory usage.");
      ! Report bytes_consumed, gc_percent, block_size, array_size.
      print "MemController object ", (name) self, " has used ",
      	self.bytes_consumed, " of ", self.array_size, " total bytes.^";
      print "This object will collect free memory when it is ",
      	self.gc_percent, " percent full.^";
      if (self.block_size == 0)
	print "This object provides variable sized blocks of memory.^";
      else
        print "This object allocates blocks of size ", self.block_size, ".^";
      print "self.next_available_array = ", self.next_available_array, "^";
      print "self.next_available_offset = ", self.next_available_offset, "^";
      rtrue;
    ],
    check_mem
    [ tmp status length magic cont;
      if (~~(debugging_level >= DEBUG_CRITICAL)) rfalse;

      Print_Trace (self, "Checking memory state.");
      if (~~(self.check_init())) rfalse;

      ! Go over mem_array, giving details on each block found.
      tmp = self.mem_array;
      
      cont = true;
      while (cont)
      {
        status = (tmp + self.header_offset_status)->0;
	magic = (tmp + self.header_offset_magic)->0;

        if (magic ~= self.magic_number)
	{
	   Print_Crit (self, "Memory is corrupted.");
	   quit;
	}

	if (self.block_size == 0)
	{
           length = (tmp + self.header_offset_length)-->0;
	} else
	{
           length = self.block_size;
	}

        print "Found a block at address ", tmp, " of type ";
        switch (status)
        {
	  _dynmem_Status_Unorg:
	    length = (tmp + self.header_offset_data)-->0;
	    print "Unorganized, with length ", length;
          _dynmem_Status_Ready:
	    print "Ready, with length ", length;
	  _dynmem_Status_InUse:
	    print "InUse, with length ", length;
	  _dynmem_Status_Freed:
	    print "Freed, with length ", length;
	  _dynmem_Status_End:
	    print "End";
	    if ((tmp + self.end_header_length)-->0 ~= 0)
	    {
	       tmp = (tmp + self.end_header_length)-->0;
	       print ", moving to new array at address ", tmp, ".^";
	       continue;
	    } else
	    {
	       cont = false;
	    }
	  default:
	    print "Unknown";
        }
        print ", with magic ", magic, ".^";
	tmp = tmp + length;
      }
      rtrue;
    ],
    check_ptr		! A function to check if a ptr is from this object
    [ptr tmp_ptr;
      Print_Trace (self, "Checking pointer correctness.");

      if (ptr == NULL)
	 rfalse;

      tmp_ptr = ptr - self.header_length;
      if ((tmp_ptr + self.header_offset_magic)->0 == self.magic_number)
	rtrue;
      else
        rfalse;
    ],

  private
    ! Now, the functions
    check_init		! A function to check the initialization state
    [;
      ! First, check if the base array is properly initialized
      if (__dynmem_base_array_init ~= true)
      {
	 if (~~(__dynmem_base_initialize ()))
	    rfalse;
      }

      ! Now, check if this object is initialized
      if (self.initialized ~= true)
      {
	 if (~~(self.initialize()))
	    rfalse;
	 self.initialized = true;
      }
      rtrue;
    ],
    initialize		! A function to initialize the object
    [;
      Print_Trace (self, "Initializing.");
      
      ! Check (or set) properties
      ! mem_array
      if (self.mem_array == NULL)
      {
	 ! There's an error
	 Print_Crit (self, "mem_array == NULL");
	 quit;
      }

      ! array_size
      if (self.array_size == 0)
      {
	 Print_Crit (self, "array_size == 0");
	 quit;
      }

      ! Set up the magic number
      self.magic_number = self % 256; ! Must reduce to one byte, though

      ! Set up info about header size, etc.
      if (self.block_size == 0)
      {
        self.header_offset_status	= _dynmem_Offset_VL_Status;
	self.header_offset_magic	= _dynmem_Offset_VL_Magic;
        self.header_offset_length	= _dynmem_Offset_VL_Length;
        self.header_offset_data		= _dynmem_Offset_VL_Data;
        self.header_length		= _dynmem_Header_VL_length;
      }
      else
      {
        self.header_offset_status	= _dynmem_Offset_FL_Status;
	self.header_offset_magic	= _dynmem_Offset_FL_Magic;
	self.header_offset_length	= _dynmem_Offset_FL_Length;
        self.header_offset_data		= _dynmem_Offset_FL_Data;
        self.header_length		= _dynmem_Header_FL_length;
	
      }
                        
      self.end_block_length		= _dynmem_End_Block_length;
      self.end_header_length		= _dynmem_End_Header_length;

      ! Now, ititialize the array
      self.init_mem (self.mem_array, self.array_size);
      
      ! Set the last fields
      self.next_available_array = self.mem_array;
      self.next_available_offset = 0;

      rtrue;
    ],
    init_mem		! A function to initialize a block of memory
    [ptr length r_length;
      Print_Trace (self, "Initializing memory blocks.");
      r_length = length - self.end_block_length;
      if (r_length <= 0)
      {
         Print_Warn (self, "mem_array is not large enough.");
         quit;
      }

      if (self.block_size == 0)
      {
         ! Set up two blocks.  One ready, and one End.
	 ! Set up block header
	 (ptr + self.header_offset_status)->0 = _dynmem_Status_Ready;
	 (ptr + self.header_offset_magic)->0 = self.magic_number;
	 (ptr + self.header_offset_length)-->0 = r_length;

      }
      else
      {
	 ! Set up a few blocks, leave the rest unorganized
	 if (r_length % self.block_size ~= 0)
	 {
	    r_length = (r_length / self.block_size) * self.block_size;
	 }
	 self.__init_fl_mem (ptr, r_length);

      }

      ! Set up End block
      (ptr + r_length + self.header_offset_status)->0 =
		 _dynmem_Status_End;
      (ptr + r_length + self.header_offset_magic)->0 = self.magic_number;
      (ptr + r_length + self.end_header_length)-->0 = 0;
	    
      rtrue;
    ],
    check_collect	! A function to determine if collection is necessary
    [;
      if ((self.bytes_consumed * 100 / self.array_size) >= self.gc_percent)
	return self.collect();
	
      rtrue;
    ],
    collect		! A function to collect freed chunks
    [;
      Print_Trace (self, "Collecting freed memory.");
      
      if (~~self.collect_free())
	rfalse;
      
      self.next_available_array = self.mem_array;
      self.next_available_offset = 0;

      if (self.block_size ~= 0)
	rtrue;

      Print_Trace (self, "Concatenating adjacent free blocks.");
      return self.collect_concat();
    ],
    collect_free	! A function to ready free blocks
    [tmp cont status length i;
      ! Loop over every block.  If the block is Freed, initialize it to 0, and 
      !		set it Ready
      tmp = self.mem_array;
      cont = true;
      while (cont)
      {
        status = (tmp + self.header_offset_status)->0;
        if ((tmp + self.header_offset_magic)->0 ~= self.magic_number)
        {
           Print_Crit (self, "Memory is corrupted.");
           quit;
        }

	if (self.block_size == 0)
	{
           length = (tmp + self.header_offset_length)-->0;
	} else
	{
           length = self.block_size;
	}
	
        if (status == _dynmem_Status_End)
	{
	   if ((tmp + self.end_header_length)-->0 ~= 0)
	   {
	      tmp = (tmp + self.end_header_length)-->0 - length;
	   } else
	      cont = false;
        }
	if (status == _dynmem_Status_Freed)
	{
	   ! Initialize block
	   (tmp + self.header_offset_status)->0 = _dynmem_Status_Ready;
	   (tmp + self.header_offset_magic)->0 = self.magic_number;
	   for (i = 0: i < length - self.header_length: i++)
	      (tmp + self.header_length)->i = 0;
	}

	tmp = tmp + length;
      }
      
      rtrue;
    ],
    collect_concat	! A function to concatenate adjacent ready blocks
    [tmp old cont status old_status length old_length i;
      ! Loop again, this time, concatenating adjacent Ready blocks
      tmp = self.mem_array;
      cont = true;
      while (cont)
      {
        status = (tmp + self.header_offset_status)->0;
        if ((tmp + self.header_offset_magic)->0 ~= self.magic_number)
        {
           Print_Crit (self, "Memory is corrupted.");
           quit;
        }

        length = (tmp + self.header_offset_length)-->0;

        if (status == _dynmem_Status_End)
	{
	   if ((tmp + self.end_header_length)-->0 ~= 0)
	   {
	      tmp = (tmp + self.end_header_length)-->0 - length;
	   } else
	      cont = false;
	   cont = false;
        }
	if (old == 0)
	{
	   old = tmp;
	   tmp = tmp + length;
	   continue;
	}

        old_status = (old + self.header_offset_status)->0;
        old_length = (old + self.header_offset_length)-->0;
	
	if ((status == _dynmem_Status_Ready) && (old_status ==
				_dynmem_Status_Ready))
	{
	   ! concatenate
	   (old + self.header_offset_length)-->0 = old_length + length;
	   for (i = 0: i < self.header_length: i++)
	      tmp->i = 0;

	   tmp = tmp + length;
	   continue;
	}

	old = tmp;
	tmp = tmp + length;
      }

      rtrue;
    ],
    __grow_array
    [ ptr old_len new_len result;
#Ifdef TARGET_GLULX;	! Turn into new prop, this gets called a lot.
      Print_Trace (self, "Attempting to grow a new array.");
      ! Try to allocate another array
      @getmemsize old_len;
      new_len = old_len + ARRAY_GROWTH;
      @setmemsize new_len result;
      if (result == 0)
      {
         ! Set up new array
         self.init_mem (old_len + 1, ARRAY_GROWTH);
         
         (ptr + self.end_header_length)-->0 = old_len + 1;
         self.next_available_array = old_len + 1;
         self.next_available_offset = 0;
      }
#Endif;
      return result;
    ],
    __alloc_var
    [length tmp_ptr cont tmp_stat tmp_len;
      Print_Trace (self, "Allocating a variable amount of memory.");
      ! Find the first available block with enough space
      tmp_ptr = self.next_available_array + self.next_available_offset;
      cont = true;
      
      while (cont)
      {
         tmp_stat = (tmp_ptr + self.header_offset_status)->0;
         tmp_len = (tmp_ptr + self.header_offset_length)-->0 -
                 self.header_length;

         if ((tmp_ptr + self.header_offset_magic)->0 ~= self.magic_number) 
         {
            Print_Crit (self, "Memory is corrupted.");
            quit;
         }

         if (tmp_stat == _dynmem_Status_Ready)
         {
            if (tmp_len >= length)
            {
               ! Use this block, but set up the rest as another block
               (tmp_ptr + self.header_offset_status)->0 =
            		_dynmem_Status_InUse;
               (tmp_ptr + self.header_offset_magic)->0 = self.magic_number;
               (tmp_ptr + self.header_offset_length)-->0 = length +
         		self.header_length;
               if (tmp_len > length)
               {
                  (tmp_ptr + self.header_length + length +
          		self.header_offset_status)->0 = _dynmem_Status_Ready;
                  (tmp_ptr + self.header_length + length +
                     self.header_offset_magic)->0 = self.magic_number;
                  (tmp_ptr + self.header_length + length +
          		self.header_offset_length)-->0 = tmp_len - length;
               }
               
               ! set next_available
               self.next_available_offset = (tmp_ptr - 
             		self.next_available_array) + (tmp_ptr + 
             		self.header_offset_length)-->0;
               
               cont = false;
            }
            else !if (tmp_len - self.header_length < length)
	    {
               tmp_ptr = tmp_ptr + tmp_len + self.header_length;
	    }
         }
         else if (tmp_stat == _dynmem_Status_End)
         {
            if ((tmp_ptr + self.end_header_length)-->0 ~= 0)
            {
               self.next_available_array = (tmp_ptr +
             		  self.end_header_length)-->0;
               tmp_ptr = self.next_available_array;
               continue;
            }
            else
            {
               if (self.__grow_array (tmp_ptr))
               {
                  tmp_ptr = self.next_available_array;
                  continue;
               } else
               {
                  ! No new array
                  cont = false;
                  tmp_ptr = NULL;
                }
            }
         }
         else tmp_ptr = tmp_ptr + tmp_len;

      }
      
      return tmp_ptr;
    ],
    __alloc_fixed
    [tmp_ptr cont tmp_stat tmp_len len_left;
      Print_Trace (self, "Allocating a fixed amount of memory.");
      ! Find the first available block.
      tmp_ptr = self.next_available_array + self.next_available_offset;
      cont = true;
 
      while (cont)
      {
         tmp_stat = (tmp_ptr + self.header_offset_status)->0;
         tmp_len = self.block_size + self.header_length;
      
         if ((tmp_ptr + self.header_offset_magic)->0 ~= self.magic_number)
         {
            Print_Crit (self, "Memory is corrupted.");
            quit;
         }

         if (tmp_stat == _dynmem_Status_Unorg)
         {
	    ! Organize some more memory
            len_left = (tmp_ptr + self.header_offset_data)-->0;
	    self.__init_fl_mem (tmp_ptr, len_left);
            ! Restart with the same tmp_ptr
            continue;
      
         }
         if (tmp_stat == _dynmem_Status_Ready)
         {
            (tmp_ptr + self.header_offset_status)->0 = _dynmem_Status_InUse;
            (tmp_ptr + self.header_offset_magic)->0 = self.magic_number;
            cont = false;
         }
         else if (tmp_stat == _dynmem_Status_End)
         {
            if ((tmp_ptr + self.end_header_length)-->0 ~= 0)
            {
               self.next_available_array = (tmp_ptr +
      		  self.end_header_length)-->0;
               tmp_ptr = self.next_available_array;
               continue;
            }
            else
            {
               if (self.__grow_array (tmp_ptr))
               {
                  tmp_ptr = self.next_available_array;
                  continue;
               } else
               {
                  ! No new array
                  cont = false;
                  tmp_ptr = NULL;
                }
            }
	 }
         else tmp_ptr = tmp_ptr + tmp_len;
      }      
 
      ! Set next_available
      self.next_available_offset = (tmp_ptr - self.next_available_array) +
       		self.block_size;
 
      return tmp_ptr;
    ],
    __init_fl_mem
    [ ptr len tmp i;
      Print_Trace (self, "Initializing some fixed length blocks.");
      ! We need to initialize a few more blocks
      ! First, determine how much space is left
      if (len / self.block_size <= __DYNMEM_MIN_BLOCKS)
      {
         ! ok, organize them all
         for (tmp = 0: tmp < len: tmp = tmp + self.block_size)
         {
            (ptr + tmp + self.header_offset_status)->0 =
          	  _dynmem_Status_Ready;
            (ptr + tmp + self.header_offset_magic)->0 =
          	  self.magic_number;
         }
      } else
      {
         for (i = 0, tmp = 0: i < __DYNMEM_MIN_BLOCKS: i++, tmp = tmp +
      		self.block_size)
         {
            (ptr + tmp + self.header_offset_status)->0 =
          	  _dynmem_Status_Ready;
            (ptr + tmp + self.header_offset_magic)->0 = 
          	  self.magic_number;
         }
         (ptr + self.block_size * __DYNMEM_MIN_BLOCKS +
          	self.header_offset_status)->0 = _dynmem_Status_Unorg;
         (ptr + self.block_size * __DYNMEM_MIN_BLOCKS +
          	self.header_offset_magic)->0 = self.magic_number;
         (ptr + self.block_size * __DYNMEM_MIN_BLOCKS +
          	self.header_offset_data)-->0 = len - 
                  self.block_size * __DYNMEM_MIN_BLOCKS;
      }
      
      rtrue;
    ],
  has
;
 
! Functions
[ __dynmem_base_initialize
    obj num size;
  
  Print_Trace (nothing, "__dynmem_base_initialize: beginning.");

  ! First, count the number of MemController objects
  num = 0;
  objectloop (obj ofclass MemController)
  {
     num++;
  }

  ! Then, divide up __dynmem_base_array properly
  size = DYNMEM_SIZE / num;
  num = 0;
  objectloop (obj ofclass MemController)
  {
     obj.mem_array = __dynmem_base_array + num*size;
     obj.array_size = size;
     num++;
  }
  
  ! set __dynmem_base_array_init
  __dynmem_base_array_init = true;

  rtrue;
];

#Endif;

