# model.c: Model program for Glk API, version 0.5.
#    Designed by Andrew Plotkin <erkyrath@netcom.com>
#    http://www.eblong.com/zarf/glk/index.html
#    This program is in the public domain.

# This is a simple model of a text adventure which uses the Glk API.
#    It shows how to input a line of text, display results, maintain a
#    status window, write to a transcript file, and so on.

# This Python port of the original C program is by Joe Mason, 2002.  It's
# still in the public domain.

import glk

TRUE = 1
FALSE = 0

mainwin = glk.window_open(None, 0, 0, glk.wintype_TextBuffer, 1)
if mainwin == None:
	glk.exit()

glk.set_window(mainwin)
 
statuswin = glk.window_open(mainwin, glk.winmethod_Above | glk.winmethod_Fixed, 
	3, glk.wintype_TextGrid, 0)

quotewin = None

current_room = 0
need_look = TRUE 
scriptstr = None
scriptref = None

def draw_statuswin():

	global mainwin
	global statuswin
	global current_room

	if not statuswin: 
		return

	if current_room == 0:
		roomname = "The Room"
	else:
		roomname = "A Different Room"

	glk.set_window(statuswin)
	glk.window_clear(statuswin)

	(width, height) = glk.window_get_size(statuswin)

	glk.window_move_cursor(statuswin, (width - len(roomname)) / 2, 1)
	glk.put_string(roomname)

	glk.window_move_cursor(statuswin, width - 3, 0)
	glk.put_string("\\|/")
	glk.window_move_cursor(statuswin, width - 3, 1)
	glk.put_string("-*-")
	glk.window_move_cursor(statuswin, width - 3, 2)
	glk.put_string("/|\\")

	glk.set_window(mainwin)

def yes_or_no():

	global mainwin

	draw_statuswin()

	while TRUE:
		commandbuf = glk.buffer_t(None, 255)
		glk.request_line_event(mainwin, commandbuf, 0)

		gotline = FALSE 
		while not gotline:

			ev = glk.select()

			if ev.type == glk.evtype_LineInput:
				if ev.win == mainwin:
					gotline = TRUE
			if ev.type == glk.evtype_Arrange:
				draw_statuswin()

		cmd = commandbuf.snapshot(ev.val1)
		if cmd[0] == 'y' or cmd[0] == 'Y':
			return TRUE
		if cmd[0] == 'n' or cmd[0] == 'N':
			return FALSE

		glk.put_string("Please enter \"yes\" or \"no\": ")

def verb_help():
	glk.put_string("This model only understands the following commands:\n")
	glk.put_string("HELP: Display this list.\n")
	glk.put_string("JUMP: A verb which just prints some text.\n")
	glk.put_string("YADA: A verb which prints a very long stream of text.\n")
	glk.put_string("MOVE: A verb which prints some text, and also changes the status line display.\n")
	glk.put_string("QUOTE: A verb which displays a block quote in a temporary third window.\n")
	glk.put_string("SCRIPT: Turn on transcripting, so that output will be echoed to a text file.\n")
	glk.put_string("UNSCRIPT: Turn off transcripting.\n")
	glk.put_string("SAVE: Write fake data to a save file.\n")
	glk.put_string("RESTORE: Read it back in.\n")
	glk.put_string("QUIT: Quit and exit.\n")

def verb_jump():
	glk.put_string("You jump on the fruit, spotlessly.\n")

def verb_yada():
	numwords = 13
	wordcaplist = [ "Ga", "Bo", "Wa", "Mu", "Bi", "Fo", "Za", "Mo", "Ra", "Po",
			"Ha", "Ni", "Na" ]
	wordlist = [ "figgle", "wob", "shim", "fleb", "moobosh", "fonk", "wabble",
			"gazoon", "ting", "floo", "zonk", "loof", "lob" ]

	wcount1 = 0
	wcount2 = 0
	wstep = 1
	jx = 0
	first = TRUE 
	
	for ix in range(0, 85):
		if ix > 0:
			glk.put_string(" ")

		if first:
			glk.put_string(wordcaplist[(ix / 17) % numwords])
			first = FALSE 

		glk.put_string(wordlist[jx])
		jx = (jx + wstep) % numwords
		wcount1 += 1
		if wcount1 >= numwords:
			wcount1 = 0
			wstep += 1
			wcount2 += 1
			if wcount2 >= numwords - 2:
				wcount2 = 0
				wstep = 1

		if (ix % 17) == 16:
			glk.put_string(".")
			first = TRUE 

	glk.put_char('\n')

def verb_quote():

	global quotewin

	glk.put_string("Someone quotes some poetry.\n")

	if not quotewin:
		quotewin = glk.window_open(mainwin, glk.winmethod_Above | glk.winmethod_Fixed, 
			5, glk.wintype_TextBuffer, 0)
		if not quotewin:
			return
	else:
		glk.window_clear(quotewin)

	glk.set_window(quotewin)
	glk.set_style(glk.style_BlockQuote)
	glk.put_string("Tomorrow probably never rose or set\n"
		"Or went out and bought cheese, or anything like that\n"
		"And anyway, what light through yonder quote box breaks\n"
		"Handle to my hand?\n")
	glk.put_string("              -- Fred\n")
	
	glk.set_window(mainwin)

def verb_move():

	global current_room

	current_room = (current_room+1) % 2
	need_look = TRUE 

	glk.put_string("You walk for a while.\n")

def verb_quit():
	glk.put_string("Are you sure you want to quit? ")
	if yes_or_no():
		glk.put_string("Thanks for playing.\n")
		glk.exit()

def verb_script():

	global mainwin
	global scriptstr
	global scriptref

	if scriptstr:
		glk.put_string("Scripting is already on.\n")
	
	if not scriptref:
		scriptref = glk.fileref_create_by_prompt(
			glk.fileusage_Transcript | glk.fileusage_TextMode, 
			glk.filemode_WriteAppend, 0)
		if not scriptref:
			glk.put_string("Unable to place script file.\n")
			return
	
	scriptstr = glk.stream_open_file(scriptref, glk.filemode_WriteAppend, 0)
	if not scriptstr:
		glk.put_string("Unable to write to script file.\n")
		return

	glk.put_string("Scripting on.\n")
	glk.window_set_echo_stream(mainwin, scriptstr)
	glk.put_string_stream(scriptstr, 
		"This is the beginning of a transcript.\n")

def verb_unscript():

	global mainwin
	global scriptstr
	global scriptref

	if not scriptstr:
		glk.put_string("Scripting is already off.\n")
		return

	glk.put_string_stream(scriptstr, "This is the end of a transcript.\n\n")
	glk.stream_close(scriptstr)
	glk.put_string("Scripting off.\n")
	scriptstr = None 

def verb_save():
	saveref = glk.fileref_create_by_prompt(
		glk.fileusage_SavedGame | glk.fileusage_BinaryMode, 
		glk.filemode_Write, 0)
	if not saveref:
		glk.put_string("Unable to place save file.\n")

	savestr = glk.stream_open_file(saveref, glk.filemode_Write, 0)
	if not savestr:
		glk.put_string("Unable to write to save file.\n")
		glk.fileref_destroy(saveref)
		return

	glk.fileref_destroy(saveref)
	
	for ix in range(0, 256):
		glk.put_char_stream(savestr, chr(ix))
	
	glk.stream_close(savestr)
	
	glk.put_string("Game saved.\n")

def verb_restore():
	saveref = glk.fileref_create_by_prompt(
		glk.fileusage_SavedGame | glk.fileusage_BinaryMode, 
		glk.filemode_Read, 0)
	if not saveref:
		glk.put_string("Unable to find save file.\n")
		return

	savestr = glk.stream_open_file(saveref, glk.filemode_Read, 0)
	if not savestr:
		glk.put_string("Unable to read from save file.\n")
		glk.fileref_destroy(saveref)
		return

	glk.fileref_destroy(saveref)

	err = FALSE 
	
	for ix in range(0, 256):
		ch = glk.get_char_stream(savestr)
		if ch == -1:
			glk.put_string("Unexpected end of file.\n")
			err = TRUE
			break
		if ch != chr(ix):
			glk.put_string("This does not appear to be a valid saved game.\n")
 			err = TRUE
 			break

	glk.stream_close(savestr)

	if err: 
		glk.put_string("Failed.\n")
		return
	
	glk.put_string("Game restored.\n")

glk.put_string("Model Glk Program\nAn Interactive Model Glk Program\n")
glk.put_string("By Andrew Plotkin.\nRelease 7.\n")
glk.put_string("Type \"help\" for a list of commands.\n")

while TRUE:
 
	draw_statuswin()
 
	if need_look:
		need_look = FALSE 
		glk.put_string("\n")
		glk.set_style(glk.style_Subheader)
		if current_room == 0:
			glk.put_string("The Room\n")
		else:
			glk.put_string("A Different Room\n")
		glk.set_style(glk.style_Normal)
		glk.put_string("You're in a room of some sort.\n")
 
	glk.put_string("\n>")
	commandbuf = glk.buffer_t(None, 255)
	glk.request_line_event(mainwin, commandbuf, 0)

	gotline = FALSE 
	while not gotline:

		ev = glk.select()

		if ev.type == glk.evtype_LineInput:
			if ev.win == mainwin:
				gotline = TRUE 
		elif ev.type == glk.evtype_Arrange:
				draw_statuswin()

	if quotewin:
		glk.window_close(quotewin)
		quotewin = None 

	cmd = ""
	for cx in commandbuf.snapshot(ev.val1).rstrip().lstrip():
		cmd += glk.char_to_lower(cx)

	if cmd == "":
		glk.put_string("Excuse me?\n")
	elif cmd == "help":
		verb_help()
	elif cmd == "move":
		verb_move()
	elif cmd == "jump":
		verb_jump()
	elif cmd == "yada":
		verb_yada()
	elif cmd == "quote":
		verb_quote()
	elif cmd == "quit":
		verb_quit()
	elif cmd == "save":
		verb_save()
	elif cmd == "restore":
		verb_restore()
	elif cmd == "script":
		verb_script()
	elif cmd == "unscript":
		verb_unscript()
	else: 
		glk.put_string("I don't understand the command \"")
		glk.put_string(cmd)
		glk.put_string("\".\n")
