%module glkc
%{
#include "glkc.h"
#include "ref_llhash.h"
#include "buf_llhash.h"

/* save buffer_t objects so they can be used by the Glk lib */

REF_LLHTABLE gBufferRefs;
BUF_LLHTABLE gSavedBuffers;

void refBuffer(char *buf)
{
	if (buf != NULL) {
		ref_hash_struct *h;
		h = ref_llhlookup(gBufferRefs, (int) buf);
		if (NULL == h) {
			ref_hash_struct newh;
			newh.buf = buf;
			newh.refcount = 1;
			ref_llhinsert(gBufferRefs, newh);
		} else {
			h->refcount++;
		}
	}
}

void unrefBuffer(char *buf)
{
	if (buf != NULL) {
		ref_hash_struct *h;
		h = ref_llhlookup(gBufferRefs, (int) buf);
		if (NULL == h) {
			free(buf);
		} else {
			h->refcount--;
			if (0 == h->refcount) {
				free(h->buf);
				ref_llhremove(gBufferRefs, (int) buf);
			}
		}
	}
}

void saveBuffer(char *buf, void *key)
{
	if (buf != NULL && key != NULL) {
		buf_hash_struct h;
		h.glkobj = key;
		h.buf = buf;
		buf_llhinsert(gSavedBuffers, h);
		refBuffer(buf);
	}
}

void releaseBuffer(void *key)
{
	if (key != NULL) {
		buf_hash_struct *h;
		h = buf_llhlookup(gSavedBuffers, (int) key);
		if (h != NULL) {
			unrefBuffer(h->buf);
			buf_llhremove(gSavedBuffers, (int) key);
		}
	}
}

/* event_t manipulators */

event_t *new_event_t(char *str) {
	event_t *ptr;
	if (SWIG_GetPtr(str, (void **) &ptr, "_event_t_p") == NULL) {
		return ptr;
	} else {
		/* FIXME: error message? */
		return NULL;
	}
}

void delete_event_t(event_t *ptr) {
	if (ptr != NULL) {
		free(ptr);
	}
}

/* window_t manipulators */

window_t *new_window_t(char *str) {
	window_t *ptr;
	if (SWIG_GetPtr(str, (void **) &ptr, "_window_t_p") == NULL) {
		return ptr;
	} else {
		/* FIXME: error message? */
		return NULL;
	}
}

void delete_window_t(window_t *ptr) {
}

/* stream_t manipulators */

stream_t *new_stream_t(char *str) {
	stream_t *ptr;
	if (SWIG_GetPtr(str, (void **) &ptr, "_stream_t_p") == NULL) {
		return ptr;
	} else {
		/* FIXME: error message? */
		return NULL;
	}
}

void delete_stream_t(stream_t *ptr) {
}

/* fileref_t manipulators */

fileref_t *new_fileref_t(char *str) {
	fileref_t *ptr;
	if (SWIG_GetPtr(str, (void **) &ptr, "_fileref_t_p") == NULL) {
		return ptr;
	} else {
		/* FIXME: error message? */
		return NULL;
	}
}

void delete_fileref_t(fileref_t *ptr) {
}

/* schannel_t manipulators */

schannel_t *new_schannel_t(char *str) {
	schannel_t *ptr;
	if (SWIG_GetPtr(str, (void **) &ptr, "_schannel_t_p") == NULL) {
		return ptr;
	} else {
		/* FIXME: error message? */
		return NULL;
	}
}

void delete_schannel_t(schannel_t *ptr) {
}

/* glk_exit implementation */

extern void glk_exit(void);

static PyObject *glkc_exit(PyObject *self, PyObject *args) {
	PyObject *result;

	if (!PyArg_ParseTuple(args, ":glk_exit")) {
		return NULL;
	}
	cleanup_pyglk();
	glk_exit();
	Py_INCREF(Py_None);
	result = Py_None;
	return result;
}

/* glk_gestalt_ext implementation */

extern glui32 glk_gestalt_ext(glui32 sel, glui32 val, glui32 *arr,
	glui32 arrlen);

static PyObject *glkc_gestalt_ext(PyObject *self, PyObject *args) {
	glui32 sel;
	glui32 val;
	glui32 arrlen;
	glui32 *arr;
	glui32 result;
	PyObject *resultList;
	PyObject *resultTuple;
	unsigned short i;

	if (!PyArg_ParseTuple(args, "lll:glk_gestalt_ext", &sel, &val, &arrlen)) {
		return NULL;
	}
	arr = (glui32 *) malloc(arrlen * sizeof(glui32));
	memset(arr, 0, arrlen * sizeof(glui32));
	result = glk_gestalt_ext(sel, val, arr, arrlen);
	resultTuple = PyTuple_New(2);
	PyTuple_SetItem(resultTuple, 0, Py_BuildValue("l", result));
	resultList = PyList_New(0);
	for (i = 0; i < arrlen; i++) {
		PyList_Append(resultList, Py_BuildValue("l", arr[i]));
	}
	free(arr);
	PyTuple_SetItem(resultTuple, 1, resultList);
	return resultTuple;
}

/* glk_window_close implementation */
extern void glk_window_close(window_t *win, stream_result_t *result);

static PyObject *glkc_window_close(PyObject *self, PyObject *args) {
	winid_t win;
	stream_result_t strcount;
	strid_t str;
	PyObject *winObj;
	PyObject *result;

	if (!PyArg_ParseTuple(args, "O:glk_window_close", &winObj)) {
		return NULL;
	}
	if (winObj) {
		if (winObj == Py_None) {
			win = NULL;
		} else if (SWIG_GetPtrObj(winObj, (void **) &win, "_window_t_p")) {
			PyErr_SetString(PyExc_TypeError,
				"Type error in argument 1 of glk_window_close. " \
				"Expected _window_t_p.");
			return NULL;
		}
	}

	/* unref the buffers for this window or its stream */
	str = glk_window_get_stream(win);
	releaseBuffer(win);
	releaseBuffer(str);

	glk_window_close(win, &strcount);

	result = PyTuple_New(2);
	PyTuple_SetItem(result, 0, Py_BuildValue("l", strcount.readcount));
	PyTuple_SetItem(result, 1, Py_BuildValue("l", strcount.writecount));
	return result;
}

/* glk_stream_close implementation */

extern void glk_stream_close(stream_t *str, stream_result_t *result); 

static PyObject *glkc_stream_close(PyObject *self, PyObject *args) {
	strid_t str;
	stream_result_t strcount;
	PyObject *strObj;
	PyObject *result;

	if (!PyArg_ParseTuple(args, "O:glk_stream_close", &strObj)) {
		return NULL;
	}
	if (strObj) {
		if (strObj == Py_None) {
			str = NULL;
		} else if (SWIG_GetPtrObj(strObj, (void **) &str, "_stream_t_p")) {
			PyErr_SetString(PyExc_TypeError,
				"Type error in argument 1 of glk_stream_close. " \
				"Expected _stream_t_p.");
			return NULL;
		}
	}

	/* unref the buffer for this stream */
	releaseBuffer(str);

	glk_stream_close(str, &strcount);

	result = PyTuple_New(2);
	PyTuple_SetItem(result, 0, Py_BuildValue("l", strcount.readcount));
	PyTuple_SetItem(result, 1, Py_BuildValue("l", strcount.writecount));
	return result;
}


/* glk_get_line_stream implementation */

extern glui32 glk_get_line_stream(stream_t *str, char *out, glui32 len);

static PyObject *glkc_get_line_stream(PyObject *self, PyObject *args) {
	PyObject *strObj;
	strid_t str;
	char *buf;
	glui32 len;
	glui32 numread;
	char pname[128];
	PyObject *result;

	if (!PyArg_ParseTuple(args, "Ol:glk_get_line_stream", &strObj, &len)) {
		return NULL;
	}
	if (strObj) {
		if (strObj == Py_None) {
			str = NULL;
		} else if (SWIG_GetPtrObj(strObj, (void **) &str, "_stream_t_p")) {
			PyErr_SetString(PyExc_TypeError,
				"Type error in argument 1 of glk_get_line_stream. " \
				"Expected _stream_t_p.");
			return NULL;
		}
	}
	buf = malloc(len * sizeof(char *));
	/* glk_get_line_stream only reads len-1 chars */ 
	numread = glk_get_line_stream(str, buf, len);
	result = Py_BuildValue("s", buf);
	if (buf != NULL) {
		free(buf);
	}
	return result;
}

/* glk_get_buffer_stream implementation */

extern glui32 glk_get_buffer_stream(stream_t *str, char *buf, glui32 len);

static PyObject *glkc_get_buffer_stream(PyObject *self, PyObject *args) {
	PyObject *strObj;
	strid_t str;
	char *buf;
	glui32 len;
	glui32 numread;
	char pname[128];
	PyObject *result;

	if (!PyArg_ParseTuple(args, "Ol:glk_get_buffer_stream", &strObj, &len)) {
		return NULL;
	}
	if (strObj) {
		if (strObj == Py_None) {
			str = NULL;
		} else if (SWIG_GetPtrObj(strObj, (void **) &str, "_stream_t_p")) {
			PyErr_SetString(PyExc_TypeError,
				"Type error in argument 1 of glk_get_buffer_stream. " \
				"Expected _stream_t_p.");
			return NULL;
		}
	}
	buf = malloc((len+1) * sizeof(char *));
	numread = glk_get_buffer_stream(str, buf, len);
	if (numread <= len) {
		buf[numread] = '\0';
	}
	result = Py_BuildValue("s", buf);
	if (buf != NULL) {
		free(buf);
	}
	return result;
}

/* buffer_t wraps a C buffer for glk_request_line and glk_stream_open_memory */

typedef struct glkc_buffer_struct {
	char * buf;
	unsigned long maxlen;
} buffer_t;

buffer_t *new_buffer_t(char *str, unsigned long maxlen) {
	buffer_t *buffer = (buffer_t *) malloc(sizeof(buffer_t));
	buffer->maxlen = maxlen;
	buffer->buf = (char *) malloc((maxlen+1) * sizeof(char));
	if (str == NULL) {
		buffer->buf[0] = '\0';
	} else {
		strcpy(buffer->buf, str);
	}
	refBuffer(buffer->buf);
	return buffer;
}

char *snapshot_whole_buffer(buffer_t *buffer) {
	buffer->buf[buffer->maxlen] = '\0';
	return buffer->buf;
}

char *snapshot_buffer(buffer_t *buffer, unsigned int len) {
	if (len <= buffer->maxlen) {
		buffer->buf[len] = '\0';
	}
	return buffer->buf;
}

glui32 get_buffer_maxlen(buffer_t *buffer) {
	return buffer->maxlen;
}

void delete_buffer_t(buffer_t *buffer) {
	if (buffer != NULL) {
		if (buffer->buf != NULL) {
			unrefBuffer(buffer->buf);
			buffer->buf = NULL;
		}
		
		free(buffer);
		buffer = NULL;
	}
}

/* glk_stream_open_memory implementation */

extern stream_t *glk_stream_open_memory(char *buf, glui32 buflen, glui32 fmode, glui32 rock);

static PyObject *glkc_stream_open_memory(PyObject *self, PyObject *args) {
	PyObject *bufferObj;
	buffer_t *buffer;
	char *buf;
	glui32 buflen;
	glui32 fmode;
	glui32 rock;
	strid_t stream;
	char pname[128];
	PyObject *result;

	if (!PyArg_ParseTuple(args, "Oll:glk_stream_open_memory", &bufferObj,
		&fmode, &rock))
	{
		return NULL;
	}
	if (bufferObj) {
		if (bufferObj == Py_None) {
			buffer = NULL;
		} else if (SWIG_GetPtrObj(bufferObj, (void **) &buffer,
			"_buffer_t_p"))
		{
			PyErr_SetString(PyExc_TypeError,
				"Type error in argument 1 of glk_stream_open_memory. " \
				"Expected _buffer_t_p.");
			return NULL;
		}
	}
	if (buffer) {
		buf = buffer->buf;
		buflen = buffer->maxlen;
	} else {
		buf = NULL;
		buflen = 0;
	}
	stream = glk_stream_open_memory(buf, buflen, fmode, rock);
	SWIG_MakePtr(pname, (char *) stream, "_stream_t_p");
	result = Py_BuildValue("s", pname);
	if (buf && stream) {
		/* hold a reference to the buffer until the stream is closed */
		saveBuffer(buf, stream);
	}
	return result;
}

/* glk_request_line_event implementation */

extern void glk_request_line_event(winid_t win, char *buf, glui32 maxlen,
	glui32 initlen);

static PyObject *glkc_request_line_event(PyObject *self, PyObject *args) {
	PyObject *winObj;
	PyObject *bufferObj;
	buffer_t *buffer;
	char *buf;
	winid_t win;
	glui32 maxlen;
	glui32 initlen;
	PyObject *result;

	if (!PyArg_ParseTuple(args, "OOl:glk_request_line_event", &winObj,
		&bufferObj, &initlen))
	{
		return NULL;
	}
	if (winObj) {
		if (winObj == Py_None) {
			win = NULL;
		} else if (SWIG_GetPtrObj(winObj, (void **) &win, "_window_t_p")) {
			PyErr_SetString(PyExc_TypeError,
				"Type error in argument 1 of glk_request_line_event. " \
				"Expected _window_t_p.");
			return NULL;
		}
	}
	if (bufferObj) {
		if (bufferObj == Py_None) {
			buffer = NULL;
		} else if (SWIG_GetPtrObj(bufferObj, (void **) &buffer, \
			"_buffer_t_p"))
		{
			PyErr_SetString(PyExc_TypeError,
				"Type error in argument 2 of glk_request_line_event. " \
				"Expected _buffer_t_p.");
			return NULL;
		}
	}
	if (buffer == NULL) {
		buf = NULL;
		maxlen = 0;
	} else {
		buf = buffer->buf;
		maxlen = buffer->maxlen;
	}
	glk_request_line_event(win, buf, maxlen, initlen);
	if (buf && win) {
		/* hold a reference to the buffer until the event is returned */
		saveBuffer(buf, win);
	}
	Py_INCREF(Py_None);
	result = Py_None;
	return result;
}

/* call the Python interrupt handler */

void callInterruptHandler(void) {
	/* FIXME: should probably use PyEval_CallObject */
	PyRun_SimpleString("glk.callInterruptHandler()");
	cleanup_pyglk();
}

%}

/* Types */

/* event_t */
event_t *new_event_t(char *str);
void delete_event_t(event_t *ptr);

/* window_t */
window_t *new_window_t(char *str);
void delete_window_t(window_t *ptr);

/* stream_t */
stream_t *new_stream_t(char *str);
void delete_stream_t(stream_t *ptr);

/* fileref_t */
fileref_t *new_fileref_t(char *str);
void delete_fileref_t(fileref_t *ptr);

/* schannel_t */
schannel_t *new_schannel_t(char *str);
void delete_schannel_t(schannel_t *ptr);

/* Functions */

/* glk_exit */
%native(glk_exit) glkc_exit;

/* glk_gestalt_ext */
%native(glk_gestalt_ext) glkc_gestalt_ext;

/* glk_select, glk_select_poll, glk_cancel_line_event */
%typemap(python, ignore) event_t *event (event_t *outval) {
	$target = (event_t *) malloc(sizeof(event_t));
}
%typemap(python, argout) event_t *event {
	char pname[128];
	PyObject *result;

	/* unref the buffer if this is a line input event */
	if ($source->type == evtype_LineInput) {
		releaseBuffer($source->win);
	}

	SWIG_MakePtr(pname, (char *) $source, "_event_t_p");
	result = Py_BuildValue("s", pname);
	if ((!$target) || ($target == Py_None)) {
		$target = result;
	} else {
		if (!PyList_Check($target)) {
			PyObject *temp = $target;
			$target = PyList_New(0);
			PyList_Append($target, temp);
			Py_XDECREF(temp);
		}
		PyList_Append($target, result);
		Py_XDECREF(result);
	}
}

/* glk_window_close, glk_stream_close */
%native(glk_window_close) glkc_window_close;
%native(glk_stream_close) glkc_stream_close;

/* anything that returns width, height, rock, etc. */
%typemap(python, ignore) glui32 * (glui32 outval) {
	$target = &outval;
}
%typemap(python, argout) glui32 * {
	PyObject *result;
	
	result = Py_BuildValue("l", *($source));
	if ((!$target) || ($target == Py_None)) {
		$target = result;
	} else {
		if (!PyList_Check($target)) {
			PyObject *temp = $target;
			$target = PyList_New(0);
			PyList_Append($target, temp);
			Py_XDECREF(temp);
		}
		PyList_Append($target, result);
		Py_XDECREF(result);
	}
}

/* glk_window_get_arrangement */
%typemap(python, ignore) window_t ** (window_t *outval) {
	$target = &outval;
}
%typemap(python, argout) window_t ** {
	char pname[128];
	PyObject *result;
	
	SWIG_MakePtr(pname, (char *) $source, "_window_t_p");
	result = Py_BuildValue("s", pname);
	if ((!$target) || ($target == Py_None)) {
		$target = result;
	} else {
		if (!PyList_Check($target)) {
			PyObject *temp = $target;
			$target = PyList_New(0);
			PyList_Append($target, temp);
			Py_XDECREF(temp);
		}
		PyList_Append($target, result);
		Py_XDECREF(result);
	}
}

/* glk_get_line_stream */
%native(glk_get_line_stream) glkc_get_line_stream;

/* glk_get_buffer_stream */
%native(glk_get_buffer_stream) glkc_get_buffer_stream;

/* buffer_t */
buffer_t *new_buffer_t(char *str, unsigned long maxlen);
char *snapshot_whole_buffer(buffer_t *buffer);
char *snapshot_buffer(buffer_t *buffer, unsigned long len);
glui32 get_buffer_maxlen(buffer_t *buffer); 
void delete_buffer_t(buffer_t *buffer);

/* glk_stream_open_memory */
%native(glk_stream_open_memory) glkc_stream_open_memory;

/* glk_request_line_event */
%native(glk_request_line_event) glkc_request_line_event;

%include "glkc.h"

%init %{
	ref_llhinit(gBufferRefs);
	buf_llhinit(gSavedBuffers);
	glk_set_interrupt_handler(&callInterruptHandler);
%}
