#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/stat.h>

#include <signal.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "xglk.h"
#include "xg_internal.h"
#include "gi_blorb.h"

#define giblorb_ID_FORM      (giblorb_make_id('F', 'O', 'R', 'M'))
#define giblorb_ID_MOD       (giblorb_make_id('M', 'O', 'D', ' '))



#ifdef GLK_MODULE_SOUND

#define BLEN 1024
static unsigned char block[BLEN];

/* GLK sound channel is an opaque type */

static schanid_t schan_list = NULL;


/*
 *  Create sound handler
 */
static int create_handler( schanid_t chan, char *cmdname ) {
  int pid, stat, i, p2c[2], c2p[2];


  if (pipe(p2c)) {
    return -1;
  }
  fcntl(p2c[1], F_SETFD, FD_CLOEXEC);
  if (pipe(c2p)) {
    return -1;
  }

  pid = fork();
  if (pid>0) {
    /* parent */
    chan->fhand = fdopen(p2c[1],"w");
    close(p2c[0]);
    chan->fnotify = fdopen(c2p[0],"r");
    close(c2p[1]);
  }
  else if (pid==0) {
    /* child */
    if (dup2(p2c[0],0) == -1) {
      fprintf(stderr,"child dup2 stdin\n");
      exit(-1);
    }
    if (dup2(c2p[1],1) == -1) {
      fprintf(stderr,"child dup2 stdout\n");
      exit(-1);
    }
    execlp( cmdname, cmdname, NULL );
    fprintf( stderr, "Failed to exec %s\n",cmdname);
    exit(-1);
  }
  else {
    if (p2c[0]>2) close(p2c[0]);
    if (p2c[1]>2) close(p2c[1]);
    if (c2p[0]>2) close(c2p[0]);
    if (c2p[1]>2) close(c2p[1]);
    return -2;
  }

  chan->handout = p2c[1];
  chan->handin = c2p[0];

  chan->hpid = pid;
  return 0;
}



/*
 * Reap sound handler child process
 */
static void reap_handler(schanid_t chan) {
  int i;

  /* reap child */
  if (chan->hpid) {
    if (!kill(chan->hpid,SIGHUP)) {
      wait(&i);
    }
    else {
      while( waitpid(-1, &i, WNOHANG)>0 ) {
	/* reap all zombie children */
      }
    }
  }
  if (chan->fhand) fclose(chan->fhand);
  if (chan->fnotify) fclose(chan->fnotify);

  if (chan->handout>0) close(chan->handout);
  if (chan->handin>0) close(chan->handin);

  chan->fhand = NULL;
  chan->fnotify = NULL;
  chan->handout = 0;
  chan->handin = 0;
  chan->hpid = 0;
}

/*
 * Check if handler has died prematurely
 */
static int handler_alive( schanid_t chan ) {
  int i;
  if (chan->fhand && (waitpid( chan->hpid, &i, WNOHANG)>0 )) {
    fclose(chan->fhand);
    chan->fhand = NULL;
  }
  return chan->fhand ? 1 : 0;
}

/*
 * Lookup sound resource and invoke appropriate sound/music handler
 */

static glui32 sound_resource(schanid_t chan, unsigned long id, glui32 repeats,
			     glui32 notify)
{
  FILE *fl;
  glui32 chunktype;
  long len;
  int close_after;

  fprintf(stderr, "n %d\n", notify);

  /* First, make sure the data exists. */
  close_after = 1;
  if (!xres_is_resource_map()) {
    char filename[PATH_MAX];
    unsigned char buf[8];
    struct stat filestat;


    sprintf(filename, "SND%ld", id); 
    /* Could check an environment variable or preference for a directory,
       if we were clever. */

    fl = fopen(filename, "r");
    if (!fl) {
      return 0;
    }

    if (fstat( fileno(fl), &filestat)!=0) {
      return 0;
    }
    len = filestat.st_size;

    if (fread(buf, 1, 12, fl) != 12) {
      /* Can't read the first few bytes. Forget it. */
      fclose(fl);
      return 0;
    }

    if (buf[8] == 'A' && buf[9] == 'I' && buf[10] == 'F') {
      chunktype = giblorb_ID_FORM;
    }
    else {
      /* Assume it's a MOD file */
      chunktype = giblorb_ID_MOD;
    }

    fseek(fl, 0, 0);

  }
  else {
    long pos;
    xres_get_resource(giblorb_ID_Snd, id, &fl, &pos, &len, &chunktype);
    if (!fl)
      return 0;
    fseek(fl, pos, 0);

    close_after = 0;
  }

  reap_handler(chan);

  if (chunktype == giblorb_ID_MOD) {
    if (create_handler(chan, "glkmodplay")<0)
      return 0;
  } else if (create_handler(chan, "glksndplay")<0)
      return 0;

  if (handler_alive(chan))
    fprintf( chan->fhand, "VOL %ld\n", chan->volume);
  if (handler_alive(chan))
    fprintf( chan->fhand, "REP %ld\n", repeats );

  if (handler_alive(chan))
    fprintf( chan->fhand,"READ %ld\n", len);

  while(len>0 && handler_alive(chan)) {
    int i = fread(block, 1, len>BLEN ? BLEN : len, fl);
    int j = fwrite(block, 1, i, chan->fhand);
    len -= i;
    if ((i==0) || (j<i)) break;
  }
  fflush(chan->fhand);

  if (close_after)
    fclose(fl);

  chan->resource_id = id;
  chan->notify = notify;

  return chunktype;

}




schanid_t glk_schannel_create(glui32 rock)
{

  schanid_t new_schan = (schanid_t) malloc(sizeof(struct glk_schannel_struct));

  new_schan->rock = rock;
  new_schan->hpid = 0;
  new_schan->volume = 0x10000;
  new_schan->notify = 0;
  new_schan->fhand = NULL;
  new_schan->fnotify = NULL;
  new_schan->handin = 0;
  new_schan->handout = 0;
  new_schan->response = 0;

  /* doubly linked circular list */
  if (schan_list) {
    new_schan->next = schan_list;
    new_schan->prev = schan_list->prev;
    schan_list->prev->next = new_schan;
    schan_list->prev = new_schan;
  }
  else {
    new_schan->next = new_schan;
    new_schan->prev = new_schan;
    schan_list = new_schan;
  }

  /* register with dispatch */
  if (gli_register_obj)
    new_schan->disprock = (*gli_register_obj)(new_schan, gidisp_Class_Schannel);
  else
    new_schan->disprock.ptr = NULL;

  return new_schan;
}



void glk_schannel_destroy(schanid_t chan)
{
  schanid_t c0,c1;

  reap_handler(chan);

  /* unlink doubly linked list */
  if (schan_list == chan) {
    schan_list = chan->next;
    if (schan_list == chan) schan_list = NULL;
  }
  c0 = chan->next;
  c1 = chan->prev;
  
  c0->prev = chan->prev;
  c1->next = chan->next;

  free(chan);
}

schanid_t glk_schannel_iterate(schanid_t chan, glui32 *rockptr)
{
  if (!chan) chan = schan_list;
  else if (chan->next == schan_list) {
    chan = NULL;
  }
  else chan = chan->next;

  if (rockptr) {
    *rockptr = 0;
    if (chan)
      *rockptr = chan->rock;
  }

  return chan;
}

glui32 glk_schannel_get_rock(schanid_t chan)
{
  if (!chan) {
    gli_strict_warning("schannel_get_rock: invalid id.");
    return 0;
  }
  return chan->rock;
}

glui32 glk_schannel_play(schanid_t chan, glui32 snd)
{
  if (!chan) {
    gli_strict_warning("schannel_play: invalid id.");
    return 0;
  }

  if (sound_resource( chan, snd, 1, 0))
    return 1;

  return 0;
}

glui32 glk_schannel_play_ext(schanid_t chan, glui32 snd, glui32 repeats,
			     glui32 notify)
{
  if (!chan) {
    gli_strict_warning("schannel_play_ext: invalid id.");
    return 0;
  }


  if (sound_resource( chan, snd, repeats, notify))
    return 1;

  return 0;
}

void glk_schannel_stop(schanid_t chan)
{
  if (!chan) {
    gli_strict_warning("schannel_stop: invalid id.");
    return;
  }
  reap_handler(chan);
}

void glk_schannel_set_volume(schanid_t chan, glui32 vol)
{
  if (!chan) {
    gli_strict_warning("schannel_set_volume: invalid id.");
    return;
  }
  chan->volume = vol;

  if (handler_alive(chan)) {
    fprintf( chan->fhand, "VOL %ld\n", chan->volume);
    fflush( chan->fhand );
  }
}

void glk_sound_load_hint(glui32 snd, glui32 flag)
{
  gli_strict_warning("schannel_sound_load_hint: invalid id.");
}

glui32 xg_schannel_response( schanid_t chan ) {
  int c = fgetc(chan->fnotify);
  if (c<0) {
    reap_handler(chan);
    return 0;
  }
  chan->response = ((chan->response & 0xffffff)<<8) + c;
  /* notify if response is "FIN\n" */
  return (chan->notify && chan->response==0x46494e0a);
}

#endif /* GLK_MODULE_SOUND */
